; Copyright (c)  NV5 Geospatial Solutions, Inc. All
;       rights reserved. Unauthorized reproduction is prohibited.
;
;+
; CLASS_NAME:
;    YAML_Node
;
; PURPOSE:
;    The YAML_Node class contains YAML properties such as ALIAS, ANCHOR and TAG.
;    This class is designed to be mixed in with other YAML classes.
;
; CATEGORY:
;    Datatypes
;
;-

;---------------------------------------------------------------------------
function YAML_Node::_overloadHelp, varname
  compile_opt IDL2, HIDDEN
  on_error, 2

  if (obj_isa(self, 'Collection')) then begin
    result = self.Collection::_overloadHelp(varname)
  endif else begin
    result = varname + ('               ').substring(varname.strlen()) + $
      ` ${obj_class(self)}  <ID=${obj_valid(self, /get_heap_id)}>`
  endelse

  extra = ''
  if (self.alias) then extra += ` ALIAS='${self.alias}'`
  if (self.anchor) then extra += ` ANCHOR='${self.anchor}'`
  if (self.tag) then extra += ` TAG='${self.tag}'`
  if (self.value) then extra += ` VALUE='${self.value}'`
  ng = total(self.global_tags ne '', /int)
  if (ng gt 0) then extra += ` GLOBAL_TAGS=${ng}`
  if (extra) then result += ' ' + extra

  return, result
end

;---------------------------------------------------------------------------
function YAML_Node::_overloadPrint
  compile_opt IDL2, HIDDEN
  on_error, 2
  result = yaml_serialize(self)
  return, result
end

;---------------------------------------------------------------------------
function YAML_Node::_overloadImpliedPrint, varname
  compile_opt IDL2, HIDDEN
  on_error, 2
  return, yaml_serialize(self)
end

;---------------------------------------------------------------------------
function YAML_Node::toString, varname
  compile_opt IDL2, HIDDEN
  on_error, 2
  return, yaml_serialize(self)
end

;---------------------------------------------------------------------------
pro YAML_Node::GetProperty, ALIAS=alias, ANCHOR=anchor, COMMENTS=comments, $
  STYLE=style, GLOBAL_TAGS=global_tags, TAG=tag, VALUE=value, $
  _REF_EXTRA=ex
  compile_opt IDL2, HIDDEN
  on_error, 2

  if (arg_present(alias)) then alias = self.alias
  if (arg_present(anchor)) then anchor = self.anchor
  if (arg_present(tag)) then tag = self.tag
  if (arg_present(value)) then value = self.value

  if (arg_present(comments)) then begin
    comments = ptr_valid(self.comments) ? *self.comments : ''
  endif

  if (arg_present(style)) then begin
    style = (['', 'flow', 'block'])[self.style]
  endif

  if (arg_present(global_tags)) then begin
    ; Return a scalar if only one tag, otherwise only return the exact # of tags
    good = where(self.global_tags ne '') > 0
    global_tags = self.global_tags[good.length eq 1 ? 0 : good]
  endif

  if (isa(ex)) then begin
    self->IDL_Object::GetProperty, _EXTRA=ex
  endif
end

;---------------------------------------------------------------------------
pro YAML_Node::SetProperty, ALIAS=alias, ANCHOR=anchor, COMMENTS=comments, $
  STYLE=style, GLOBAL_TAGS=global_tags, TAG=tag, VALUE=value
  compile_opt IDL2, HIDDEN
  on_error, 2

  if (isa(alias)) then self.alias = strtrim(alias, 2)
  if (isa(anchor)) then self.anchor = strtrim(anchor, 2)
  if (isa(tag)) then self.tag = strtrim(tag, 2)
  if (isa(comments)) then self.comments = ptr_new(comments)
  if (isa(value)) then self.value = strtrim(value, 2)

  if (isa(global_tags)) then begin
    ; rapidyaml allows a max of 4 global tags
    n = (global_tags.length - 1) < 3
    self.global_tags[0: n] = strtrim(global_tags[0: n], 2)
  endif

  if (isa(style)) then begin
    case ((style.toString()).toLower()) of
      '1': self.style = 1
      'flow': self.style = 1
      '2': self.style = 2
      'block': self.style = 2
      else: self.style = 0
    endcase
  endif
end

;---------------------------------------------------------------------------
pro YAML_Node__DEFINE
  compile_opt IDL2, HIDDEN
  void = {YAML_Node, $
    anchor: '', $
    alias: '', $
    value: '', $
    tag: '', $
    global_tags: strarr(4), $  ; rapidyaml allows a max of 4 global tags
    comments: ptr_new(), $
    style: 0b $
    }
end
