;+
; :Description:
;   This class provides a list of "handlers" for converting a url
;   into a filename, or for converting a filename into a url.
;   
;   The "handlers" are invoked in order until a handler derives
;   a result. In other words, not all of them may be invoked.
;   
;   If a listed handler is not found, it will silently be skipped
;   and the next handler in the list will be invoked to try
;   deriving a result.
;   
;   The "handlers" are identified via a list of "pro code function names"
;   defined in the init method of this class.  This code could be changed to
;   obtain the list of pro code function names from a configuration file, and
;   then this routine would never need to be modified.
;   
;-



;------------------------------------------------------
; Private utility function used by this class.
;------------------------------------------------------
FUNCTION _uh_makeUrlPretty, url
  COMPILE_OPT idl2, hidden
  ON_ERROR,2

  RETURN, STRJOIN(strsplit(url,'\',/extract),'/')
END


;------------------------------------------------------
; Private utility function used by this class.
;------------------------------------------------------
FUNCTION _uh_makePathPretty, path
  COMPILE_OPT idl2, hidden
  ON_ERROR,2

  ret = STRARR(N_ELEMENTS(path))
  FOR i=0,N_ELEMENTS(path)-1 DO BEGIN
    isunix=0
    isunc=0
    IF STRPOS(path[i],'\\') EQ 0 || STRPOS(path[i],'//') EQ 0 THEN BEGIN
      isunc=1
      path[i]=STRMID(path[i],2)
    ENDIF
    IF STRPOS(path[i],'/') EQ 0 THEN isunix=1

    ret[i] = STRJOIN(strsplit(path[i],'/',/extract),PATH_SEP())
    ret[i] = STRJOIN(strsplit(ret[i],'\',/extract),PATH_SEP())

    IF isunix THEN ret[i] = PATH_SEP() + ret[i]
    IF isunc THEN ret[i] = '\\' + ret[i]
  ENDFOR

  RETURN,ret
END



;+
;  :Description:
;     Handle the case where the url is a jpip:// scheme.
;-
FUNCTION _url_handler_jpip, OP=op, URL=url, FILE=file, PREFIX_FILE=prefix, PREFIX_URL=prefix_url
  compile_opt idl2, hidden
 
   if (strcmp(op,'urlToFile',/Fold_Case)) then begin 
    
     if (isa(url, 'STRING') && strcmp(url, 'jpip://', 7, /FOLD_CASE)) then begin
       return, url
     endif
     
   endif
   
   ;; Note: At this time, we don't yet register a file with a JPIP service
   ;; so we don't handle the 'filetoUrl' operation.
   
   return, !NULL
END

;+
;  :Description:
;     Handle the case where the url is a file:// scheme.
;-
FUNCTION _url_handler_file, OP=op, URL=url, FILE=file, PREFIX_FILE=prefix, PREFIX_URL=prefix_url
  compile_opt idl2, hidden


  if (strcmp(op,'urlToFile',/Fold_Case)) then begin
    if (STRCMP(url, 'file:', 5, /Fold_Case)) then begin

      ; remove file: from the beginning
      url = STRMID(url, 5)

      ; now if there are 3 slashes, it is a local path
      IF (STRCMP(url, '///', 3)) then begin
        ; strip off the first 2 slashes
        url = STRMID(url,2)
        ; handle local file paths on linux
        ; if the path now starts like this, /C:, where C is any letter,
        ; it must be a windows path, so strip off the initial /
        ; leaving just the drive letter at the beginning
        ; for linux paths starting with /, we are already done
        if (STREGEX(url, '\/[A-Z]:', /FOLD_CASE) eq 0) then begin
          url = STRMID(url,1)
        endif

        ; if it starts with //, then it is a UNC path
      endif else if (STRCMP(url, '//', 2)) then begin
        ; leave it as is
      endif

      ; makepathpretty always returns an array
      val = _uh_makepathpretty(url)
      return, val[0]

    endif
  endif

  ; we could not derive a result
  return, !NULL

END



;-------------------------------------------------------------------------------
FUNCTION _url_handler_ese, OP=op, URL=inurl, FILE=infile, PREFIX_FILE=prefix, PREFIX_URL=prefix_url
  compile_opt idl2, hidden
  
  !SERVER.GetProperty, HOSTNAME=hostname, DOCROOT=docroot, ROOTURL=rooturl

  ;**********************
  ; urlToFile
  ; 
  ; In this handler, convert an ese url into a local filename on disk.
  ; An ese url is a link to a resource contained within ESE's docroot.
  ;***********************************************
  if (STRCMP(op,'urlToFile',/Fold_Case) && $
     (STRCMP(inurl, 'http://', 7, /Fold_Case) || STRCMP(inurl, 'https://', 8, /Fold_Case))) then begin
    
    ; Locate the offset position of our hostname in the url string
    pos = STRPOS(STRLOWCASE(inurl),STRLOWCASE(hostname))
    if (pos eq -1) then begin
      ; At this point, our hostname is not in the url, so look for the special
      ; hostname = 'localhost'
      hostname='localhost'
      pos = STRPOS(STRLOWCASE(inurl),hostname)
    endif
    
    file = !NULL 
    
    IF pos GT -1 THEN BEGIN
      ; At this point, the url does belong to a resource on this machine.
      ; Return the filename to the resource on disk.
      fspos = STRPOS(STRLOWCASE(inurl),'/',pos)
      IF fspos GT -1 THEN BEGIN
        file = docroot + PATH_SEP() + STRJOIN(strsplit(STRMID(inurl,fspos),'/',/EXTRACT),PATH_SEP())
      ENDIF ELSE BEGIN
        file = docroot + PATH_SEP() + STRJOIN(strsplit(STRMID(inurl,pos+STRLEN(hostname)),'/',/EXTRACT),PATH_SEP())
      ENDELSE
    ENDIF ELSE BEGIN
      ;If hostname not in url then do not append docroot and leave as is
      file = inurl
    ENDELSE 
    
    return, file      
  endif

  ;**********************
  ; fileToUrl
  ; 
  ; In this handler, convert a filename on disk into an ESE url.
  ;***************************************************** 
  if (strcmp(op,'fileToUrl',/Fold_Case)) then begin
    file = infile
       
    ; Is our docroot the prefix of the file name?
    ;   If so, remove it from the filename.
    pos = STRPOS(_uh_makepathpretty(STRUPCASE(file)),_uh_makepathpretty(STRUPCASE(docroot)))
    IF pos GE 0 THEN BEGIN
      file = STRMID(file,pos+STRLEN(docroot))
      url=rooturl
    ENDIF ELSE BEGIN
      url=prefix_url
    ENDELSE

    ; Now, construct the url to the file
    r = url + '/' + ((STRPOS(file,'/') NE 0 && STRPOS(file,'\') NE 0) ? file : STRMID(file,1))
    r = _uh_makeurlpretty(r)

    return, r
 
  endif
  
  ; At this point, a result could not be derived.
  return, !NULL
end




;+
; :Description:
;    Setup a list of url handlers.  We create a list
;    of IDL function names that provides url-to-file
;    conversions.
;
;-
FUNCTION UrlHandlers::Init

    ; !! THE ORDER OF THIS LIST IS IMPORTANT. !!
    ; HANDLERS ARE CALLED IN ORDER UNTIL A RESULT CAN BE DERIVED.
    ; 
    ; Notes: 
    ;     1) The handlers that begin with "_" are actually defined
    ;       in this file.
    ;     2) The url_handler_geoserver resides in ESE/lib
    
    self.__handlers = list( '_url_handler_jpip',       $ ; "jpip://"
                            '_url_handler_file',       $ ; "file://" 
                         ;-------------
                         ; NOTE:
                         ;   Uncomment this line if you wish to register output products with GeoServer
                         ;------------
                         ;   'url_handler_geoserver',   $ ; "http://...ogc.../"
                            '_url_handler_ese'         $ ; probably should be the last resort, when all above fail.
                           ) 
                              
    return, 1 
      
END



;+
; :Description:
;    Loop through our url handlers until one of them can derive a result
;    for the desired operation.
;
;
; :Keywords:
;    OP
;    URL
;    FILE
;    PREFIX
;
;-
function UrlHandlers::__invokeHandlers, OP=op, URL=url, FILE=file, PREFIX_FILE=prefix, PREFIX_URL=prefix_url
  compile_opt idl2, hidden

  result = !NULL
  index = 0

  num_handlers = n_elements(self.__handlers)

  ; Setup an error handler. If a url handler doesn't exist, or throws
  ; an exception, continue on with the next url handler in the list
  ; and let it try to derive a result.
  catch, err_status
  if (err_status ne 0) then begin
    ;This is usually because we invoked a non-existant handler.
    ;Increment the handler index and try the next handler.
    ;print, !error_state.msg
    result=!NULL
    index++
  endif

  ; Loop through our defined error handlers until a result is derived.
  while ( index lt num_handlers and result eq !NULL )do begin
    ;print, '*** Invoking url handler: ',self.__handlers[index]
    result = call_function( self.__handlers[index], OP=op, URL=url, FILE=file, PREFIX_FILE=prefix, PREFIX_URL=prefix_url )
    index++
  endwhile

  ; Return the derived result
  return, result

end

;-------------------------------------------------------------------------------
FUNCTION UrlHandlers::makeUrlPretty,url
  COMPILE_OPT idl2
  ON_ERROR,2

  RETURN, _uh_makeUrlPretty(url)
END

;-------------------------------------------------------------------------------
FUNCTION UrlHandlers::makePathPretty,path
  COMPILE_OPT idl2

  RETURN,  _uh_makepathpretty(path)
END

;+
; :Description:
;    Convert a file into a url. This is used to register a file
;    with, say GeoServer, so we can return an ogc url.
;
; :Keywords:
;    PREFIX_FILE
;    FILE
;    PREFIX_URL
;
; :Returns:
;    return an empty list, if we can't derive a url from a file
;    
;-
function UrlHandlers::fileToUrl, PREFIX_FILE=prefix, FILE=file, PREFIX_URL=prefix_url
  compile_opt idl2
  
  r = self.__invokeHandlers(OP='fileToUrl', PREFIX_FILE=prefix, FILE=file, PREFIX_URL=prefix_url)
  if ( isa(r,'string') ) then begin
    result = list()
    result.add, HASH('url',r)
  endif else if ( isa(r,'hash') ) then begin
    result = list()
    result.add, r
  endif else begin
    result = r
  endelse
  
  return, result
  
end

;-------------------------------------------------------------------------------
function UrlHandlers::urlToFile, url

  compile_opt idl2

  return, self.__invokeHandlers(OP='urlToFile', URL=url)

end

;-------------------------------------------------------------------------------
FUNCTION UrlHandlers::deleteUrl, url
   compile_opt idl2
   return, self.__invokeHandlers(OP='deleteUrl', URL=url)
   
END


;-------------------------------------------------------------------------------
PRO UrlHandlers__define

   COMPILE_OPT idl2, hidden
   
   struct = {UrlHandlers, INHERITS IDL_OBJECT, $
      __handlers : obj_new() $
   }
     
END
