; Copyright (c)  NV5 Geospatial Solutions, Inc. All
;       rights reserved. Unauthorized reproduction is prohibited.
;+
;-

;-----------------------------------------------------------------------------
;
function cw_itpropertytoolbar_get_offset, wId, OFFSET=offset, TLB=tlb
  compile_opt idl2, hidden

  if (N_Elements(offset) eq 0) then begin
    offset = [0,0]
  endif

  if (~Widget_Info(wId, /VALID_ID)) then begin
    return, 1
  endif

  geometry = Widget_Info(wId, /GEOMETRY)
  offset[0] += geometry.XOFFSET
  offset[1] += geometry.YOFFSET
  if (~Widget_Info(Widget_Info(wId, /PARENT), /VALID_ID)) then begin
    ; We are at the top base, let's add the size of the window border.
    tlb = wId
    offset[0] += (geometry.SCR_XSIZE-geometry.XSIZE)
    offset[1] += (geometry.SCR_YSIZE-geometry.YSIZE)
  endif

  return, cw_itpropertytoolbar_get_offset(Widget_Info(wId, /PARENT), $
                                          OFFSET=offset, TLB=tlb)
end

;-------------------------------------------------------------------------
pro cw_itpropertytoolbar_build_dialog_close, event
  compile_opt idl2, hidden

  Widget_Control, event.top, /DESTROY
end

;-------------------------------------------------------------------------
pro cw_itpropertytoolbar_build_dialog_event, event
    compile_opt idl2, hidden

   if (Tag_Names(event, /STRUCTURE) eq 'WIDGET_SLIDER') then begin
     Widget_Control, widget_info(event.top, find_by_uname='timer_base'), TIMER=-1
     ; No return in this case.
   endif

   if ((Tag_Names(event, /STRUCTURE) eq 'WIDGET_TRACKING') && $
       (Widget_Info(event.id, /TYPE) eq 8)) then begin
     Widget_Control, widget_info(event.top, find_by_uname='timer_base'), TIMER=-1
     return
   endif

   if (event.id eq event.top) then begin
     if ((Tag_Names(event, /STRUCTURE) eq 'WIDGET_KBRD_FOCUS') && $
         (event.enter eq 0)) then begin
       ; In Unix, widget_sliders produce kbrd_focus events that mess up this 
       ; mechanism. Use a timer and if we don't get any slider events, then we'll destroy
       ; the dialog.
       Widget_Control, widget_info(event.top, find_by_uname='timer_base'), TIMER=0.5       
     endif
     return
   endif

   widget_control, event.top, get_uvalue=state
   uname = widget_info(event.id, /UNAME)
   unameContainer = widget_info(event.top, /UNAME)
   widget_control, event.id, get_uvalue=value

   oSelected = state.oTool.GetSelectedItems(COUNT=nSelected)
   if (nSelected eq 0) then begin
     return    
   endif

   doCommit = 1
   forceCommit = 0
   property = uname.ToUpper()
   case uname of
     'more_colors': begin
       value = dialog_colorpicker(WCALLER=state.parent, $
                                  DIALOG_PARENT=state.parent)
       if N_Elements(value) eq 0 then return
       property = 'COLOR'       
     end
     'thick': begin
       widget_control, event.id, get_value=value, get_uvalue=options
       wLabel = widget_info(widget_info(event.id, /parent), /all_children)
       widget_control, wLabel[-1], SET_VALUE=(options[value]).ToString() + ' pt'
       value = options[value]
     end
     'font_name': begin
       if (event.index eq 0) then return
       widget_control, event.id, get_uvalue=options
       value = options[event.index-1]
     end
     'font_size': begin
       widget_control, event.id, get_value=value, get_uvalue=options
       wLabel = widget_info(widget_info(event.id, /parent), /all_children)
       widget_control, wLabel[-1], SET_VALUE=(options[value]).ToString()
       value = options[value]
     end
     'symbol': begin
       if (event.index eq 0) then return
       value = event.index-1
     end
     'sym_size': begin
       widget_control, event.id, get_value=value, get_uvalue=options
       wLabel = widget_info(widget_info(event.id, /parent), /all_children)
       widget_control, wLabel[-1], SET_VALUE=String(options[value], format='(F3.1)')
       value = options[value]       
     end
     'sym_filled': begin
       value = event.select
     end
     'transparency': begin
       widget_control, event.id, get_value=value
       if (unameContainer eq 'background') then begin
         property = 'FILL_TRANSPARENCY'
       endif
     end
     'linestyle': begin
       if (event.index eq 0) then return
       value = event.index -1       
     end
     'arrow_style': begin
       if (event.index eq 0) then return
       value = event.index
     end
     'arrow_thick': begin
       widget_control, event.id, get_value=value, get_uvalue=options
       value = options[value]
       property = 'THICK'
     end
     'head_angle': begin
       widget_control, event.id, get_value=value
     end
     'head_indent': begin
       widget_control, event.id, get_value=value
       value = (value-50)/50.0
     end
     'head_size': begin
       widget_control, event.id, get_value=value
       value = value/10.0
     end  
     else:   
   endcase

   if (property eq 'COLOR') then begin
     if (unameContainer eq 'background') then begin
       property = 'FILL_COLOR'
     endif
     if (unameContainer eq 'symbol') then begin
       property = 'SYM_COLOR'
     endif
   endif

   isSlider = Tag_Names(event, /STRUCTURE) eq 'WIDGET_SLIDER'
   if (isSlider) then begin
     doCommit = event.drag eq 0
     ; The action will fail when we release the mouse since the objects already
     ; have that value, but that's exactly when we want to commit the event.
     forceCommit = event.drag eq 0
   endif

   if (~isSlider && (uname ne 'sym_filled') && widget_info(event.top, /valid_id)) then begin
     widget_Control, event.top, /destroy
   endif

   success = 0
   foreach oItem, oSelected do begin
     if (property eq 'FILL_COLOR') then begin
       !null = state.oTool->DoSetProperty(oItem->GetFullIdentifier(), 'FILL_BACKGROUND', 1)
     endif
     ok = state.oTool->DoSetProperty(oItem->GetFullIdentifier(), property, value)
     success = success || ok
   endforeach

   if ((success && doCommit) || forceCommit) then begin
     state.oTool->CommitActions
   endif
   state.oTool->RefreshCurrentWindow
end

;-------------------------------------------------------------------------
pro cw_itpropertytoolbar_build_dialog, wID, oTool, COLOR=color, FONT_NAME=fontName, $
  LINE_SIZE=lineSize, TRANSPARENCY=transparency, WEIGHT=weight, LINE_STYLE=lineStyle, $
  SYMBOL_TYPE=symbolType, SYMBOL_SIZE=symbolSize, FONT_SIZE=fontSize, NO_FILL=noFill, $
  ARROW_STYLE=arrowStyle, ARROW_THICK=arrowThick, HEAD_ANGLE=headAngle, $
  HEAD_INDENT=headIndent, HEAD_SIZE=headSize, SYMBOL_FILL=symbolFill
  compile_opt idl2, hidden

  oSelected = oTool.GetSelectedItems(COUNT=nSelected)
  if (nSelected eq 0) then begin
    ok = dialog_message('Please, select a graphical element first.')
    return
  endif

  isWindows = (!version.os_family eq 'Windows')
  parent = widget_info(wID, /PARENT)
  offset = [0,0]
  offsetAdjust = (isWindows) ? 6 : 0  
  !null = cw_itpropertytoolbar_get_offset(parent, OFFSET=offset)
  geom = Widget_Info(parent, /GEOMETRY)
  xoffset = offset[0]-offsetAdjust
  yoffset = offset[1]+geom.SCR_YSIZE-offsetAdjust
  unameContainer = widget_info(wID, /UNAME)

  ; NOTE: Unix shows a bar on this base... must have a title. 
  wBase = Widget_Base(/COL, XPAD=2, YPAD=2, SPACE=0, $
                      /KBRD_FOCUS_EVENTS, $
                      TLB_FRAME_ATTR=(isWindows ? 31 : 17) , $
                      UNAME=unameContainer, $
                      TITLE='Properties', $
                      XOFFSET=xoffset, $
                      YOFFSET=yoffset, $
                      GROUP_LEADER=parent)
  
  ; This base will hold the timer and handle destroy.
  !null = Widget_Base(wBase, XSIZE=0, YSIZE=0, XPAD=0, YPAD=0, $
                      UNAME='timer_base', $
                      EVENT_PRO='cw_itpropertytoolbar_build_dialog_close')

  if (Keyword_Set(color)) then begin
    case unameContainer of
      'text': str = 'Text Colors'
      'background': str = 'Fill Colors'
      'line': str = 'Line Colors'
      'symbol': str = 'Symbol Colors'
      else: str = ''
    endcase
    wCont = Widget_Base(wBase, /ROW)
    !null = Widget_Label(wCont, VALUE=str, FONT='*14*bold')

    colors = [[[213,253,218], [253,245,213], [255,221,217], [247,221,255], [221,223,255], [255,255,255]], $
              [[159,249,170], [250,232,204], [254,182,174], [236,176,255], [179,183,255], [200,200,200]], $
              [[118,248,133], [249,223,117], [254,156,146], [223,130,254], [153, 158, 255], [175,175,175]], $
              [[65,243,86], [247,215,85], [253,104,89], [210,84,252], [112, 119, 254], [150,150,150]], $
              [[33,241,57], [244,202,36], [253,61,41], [196,29,251], [70,79,252], [125,125,125]], $
              [[15,221,39], [215,174,11], [246,27,4], [171,4,226], [18,29,252], [100,100,100]], $
              [[13,189,34], [187,152,11], [209,22,3], [140,3,185], [4,14,222], [75,75,75]], $
              [[12,156,29], [159,129,9], [177,21,5], [118,2,156], [3,12,189], [50,50,50]], $
              [[9,125,23], [127,103,7], [130,16,4], [97,2,128], [2,9,142], [25,25,25]], $
              [[5,77,13], [91,74,5], [89,11,3], [69,1,91], [0,0,90], [0,0,0]]]
    size = Size(colors, /DIMENSIONS)
    boxSize = 16
    img = Bytarr(boxSize,boxSize,4)
    img[*,*,3] = 255b
    for i=0,size[1]-1 do begin
      !null = Widget_Base(wBase, /ROW, xpad=0, ypad=0, space=0, ysize=1)
      wRow = Widget_Base(wBase, /ROW, xpad=0, ypad=0, space=0)
      for j=0,size[2]-1 do begin      
        img[*,*,0] = colors[0,i,j]
        img[*,*,1] = colors[1,i,j]
        img[*,*,2] = colors[2,i,j]
        !null = Widget_Base(wRow, /ROW, xpad=0, ypad=0, space=0, xsize=1)
        !null = Widget_Button(wRow, VALUE=img, $
                                    UVALUE=Reform(colors[*,i,j]), $
                                    SCR_XSIZE=boxSize, $
                                    SCR_YSIZE=boxSize, $
                                    UNAME='color', $
                                    /BITMAP, $
                                    /FLAT)      
      endfor    
    endfor
    !null = Widget_Button(wBase, $
                          VALUE='More Colors...', $
                          UNAME='more_colors')
    !null = Widget_Base(wBase, xpad=0, ypad=0, ysize=2)
  endif

  if (Keyword_Set(lineStyle)) then begin
    options = ['Solid', 'Dots', 'Dashes', 'Dash-Dot', 'Dash-Dot-Dot', 'Long Dash', 'None']
    wCont = Widget_Base(wBase, /ROW, xpad=0)
    !null = Widget_Droplist(wCont, $
                            VALUE=['Style', options], $
                            UVALUE=options, $
                            /TRACKING_EVENTS, $
                            UNAME='linestyle')
    !null = Widget_Base(wBase, xpad=0, ypad=0, ysize=2) 
  endif

  if (Keyword_Set(lineSize)) then begin
    value = 0
    options = indgen(10)+1
    wCont = Widget_Base(wBase, /ROW, xpad=0, ypad=0)
    if (nSelected gt 0) then begin
      oSelected[0].GetProperty, THICK=thick
      if (n_elements(thick) ne 0) then begin
        pos = where(options eq thick, /null)
        if (pos ne !null) then begin
          value = pos
        endif
      endif
    endif
    !null = Widget_Slider(wCont, MINIMUM=0, MAXIMUM=n_elements(options)-1, TITLE='Weight', $
                            /SUPPRESS_VALUE, UNAME='thick', /DRAG, UVALUE=options, VALUE=value)
    !null = Widget_Base(wCont, xpad=0, ypad=0, xsize=5)                        
    !null = Widget_Label(wCont, VALUE=(options[value]).ToString() + ' pt', /DYNAMIC_RESIZE)
    !null = Widget_Base(wBase, xpad=0, ypad=0, ysize=2) 
  endif

  if (Keyword_Set(noFill)) then begin
    if (nSelected gt 0) then begin
      oSelected[0].GetProperty, FILL_BACKGROUND=fillBackground
      if (Keyword_Set(fillBackground)) then begin
        wCont = Widget_Base(wBase, /ROW, XPAD=0)
        !null = Widget_Button(wCont, $
                              VALUE='No Fill', $
                              UNAME='fill_background', $
                              UVALUE=0)
      endif
    endif
  endif

  if (Keyword_Set(fontName)) then begin
    oBuffer = Obj_New('IDLgrBuffer', DIMENSIONS=[2,2])
    fontNames = oBuffer->GetFontNames('*')
    Obj_Destroy, oBuffer
    title = 'Font'
    if (nSelected gt 0) then begin
      oSelected[0].GetProperty, FONT_NAME=fontName
      if (Isa(fontName, 'String')) then begin
        title = fontName
      endif
    endif
    wCont = Widget_Base(wBase, /ROW, xpad=0)
    !null = widget_droplist(wCont, $
                            VALUE=[title, fontNames], $
                            scr_xsize=160, $
                            UVALUE=fontNames, $
                            /TRACKING_EVENTS, $
                            UNAME='font_name')    
  endif

  if (Keyword_Set(fontSize)) then begin
    value = 0
    wCont = Widget_Base(wBase, /ROW, xpad=0)
    options = [8, 9, 10, 11, 12, 14, 16, 18, 20, 24, 28, 32, 36, 40, 44, 48, 54, 60, 66, 72, 80, 88]
    if (nSelected gt 0) then begin
      oSelected[0].GetProperty, FONT_SIZE=fontSize
      if (n_elements(fontSize) ne 0) then begin
        pos = where(options eq fontSize, /null)
        if (pos ne !null) then begin
          value = pos
        endif
      endif
    endif
    !null = Widget_Slider(wCont, MINIMUM=0, MAXIMUM=n_elements(options)-1, TITLE='Font Size', $
                          /SUPPRESS_VALUE, UNAME='font_size', /DRAG, UVALUE=options, VALUE=value)
    !null = Widget_Base(wCont, xpad=0, ypad=0, xsize=5)
    !null = Widget_Label(wCont, VALUE=(options[value]).ToString(), /DYNAMIC_RESIZE)
  endif

  if (Keyword_Set(symbolType)) then begin
    options = ['None', 'Plus', 'Asterisk', 'Dot', 'Diamond', 'Triangle', 'Square', 'X', '>', '<', $
               'Triangle Down', 'Triangle Left', 'Triangle Right', 'Tri Up', 'Tri Down', 'Tri Left', $
               'Tri Right', 'Thin Diamond', 'Pentagon', 'Hexagon 1', 'Hexagon 2', 'Vertical Line', 'Horizontal Line', 'Star', 'Circle', 'Pin', 'Pushpin']    
    wCont = Widget_Base(wBase, /ROW, xpad=0)
    !null = Widget_Droplist(wCont, $
                            VALUE=['Symbols', options], $
                            UVALUE=options, $
                            /TRACKING_EVENTS, $
                            UNAME='symbol')
    !null = Widget_Base(wBase, xpad=0, ypad=0, ysize=2)
  endif

  if (Keyword_Set(symbolSize)) then begin
    value = 0
    options = [0.2, 0.5, 1.0, 1.5, 2.0, 2.5, 3.0, 3.5, 4.0, 4.5, 5.0]
    wCont = Widget_Base(wBase, /ROW, xpad=0, ypad=0)
    if (nSelected gt 0) then begin
      oSelected[0].GetProperty, SYM_SIZE=symSize
      if (n_elements(symSize) ne 0) then begin
        pos = where(options eq symSize, /null)
        if (pos ne !null) then begin
          value = pos
        endif
      endif
    endif
    !null = Widget_Slider(wCont, MINIMUM=0, MAXIMUM=n_elements(options)-1, TITLE='Size', $
                          /SUPPRESS_VALUE, UNAME='sym_size', /DRAG, UVALUE=options, VALUE=value)
    !null = Widget_Base(wCont, xpad=0, ypad=0, xsize=5)
    !null = Widget_Label(wCont, VALUE=String(options[value], format='(F3.1)'), /DYNAMIC_RESIZE)
    !null = Widget_Base(wBase, xpad=0, ypad=0, ysize=2)
  endif
  
  if (Keyword_Set(symbolFill)) then begin
    wCont = Widget_Base(wBase, /ROW, /NONEXCLUSIVE)
    wButton = Widget_Button(wCont, VALUE='Fill Symbols', UNAME='sym_filled')
    if (nSelected gt 0) then begin
      oSelected[0].GetProperty, SYM_FILLED=symFilled
      if ((n_elements(symFilled) ne 0) && symFilled) then begin
        Widget_Control, wButton, /SET_BUTTON
      endif
    endif    
  endif

  if (Keyword_Set(arrowStyle)) then begin
    options = ['------->', '<-------', '<------>', '>------>', '<------<']
    wCont = Widget_Base(wBase, /ROW, xpad=0, ypad=0)
    !null = Widget_Droplist(wCont, $
                            VALUE=['Arrow Style', options], $
                            UVALUE=options, $
                            /TRACKING_EVENTS, $
                            UNAME='arrow_style')
    !null = Widget_Base(wBase, xpad=0, ypad=0, ysize=2)
  endif

  if (Keyword_Set(arrowThick)) then begin
    value = 0
    wCont = Widget_Base(wBase, /COL, xpad=0)
    options = indgen(10)+1
    if (nSelected gt 0) then begin
      oSelected[0].GetProperty, THICK=arrowThick
      if (n_elements(arrowThick) ne 0) then begin
        pos = where(options eq arrowThick, /null)
        if (pos ne !null) then begin
          value = pos
        endif
      endif
    endif    
    !null = Widget_Slider(wCont, MINIMUM=0, MAXIMUM=n_elements(options)-1, TITLE='Thickness', $
                          /SUPPRESS_VALUE, UNAME='arrow_thick', /DRAG, UVALUE=options, VALUE=value)
  endif

  if (Keyword_Set(headAngle)) then begin
    value = 0    
    if (nSelected gt 0) then begin
      oSelected[0].GetProperty, HEAD_ANGLE=value
      !null = Widget_Base(wBase, xpad=0, ypad=0, ysize=10)
      !null = Widget_Slider(wBase, MINIMUM=0, MAXIMUM=90, TITLE='Head Angle', $
                            /SUPPRESS_VALUE, UNAME='head_angle', VALUE=value, /DRAG)
    endif
  endif

  if (Keyword_Set(headIndent)) then begin
    value = 1    
    if (nSelected gt 0) then begin
      oSelected[0].GetProperty, HEAD_INDENT=value
      if (N_Elements(value) ne 0) then begin
        value = value*50+50
      endif
      !null = Widget_Base(wBase, xpad=0, ypad=0, ysize=10)
      !null = Widget_Slider(wBase, MINIMUM=0, MAXIMUM=100, TITLE='Head Indent', $
                            /SUPPRESS_VALUE, UNAME='head_indent', VALUE=value, /DRAG)
    endif
  endif

  if (Keyword_Set(headSize)) then begin
    if (nSelected gt 0) then begin
      oSelected[0].GetProperty, HEAD_SIZE=value
      if (N_Elements(value) ne 0) then begin
        value = 0 > value*10 < 100
      endif
      !null = Widget_Base(wBase, xpad=0, ypad=0, ysize=10)
      !null = Widget_Slider(wBase, MINIMUM=1, MAXIMUM=100, TITLE='Head Size', $
                            /SUPPRESS_VALUE, UNAME='head_size', VALUE=value, /DRAG)
    endif
  endif

  if (Keyword_Set(transparency)) then begin
    value = 0
    if (nSelected gt 0) then begin
      if (unameContainer eq 'background') then begin
        oSelected[0].GetProperty, FILL_TRANSPARENCY=value
      endif else begin
        oSelected[0].GetProperty, TRANSPARENCY=value
      endelse
    endif
    !null = Widget_Base(wBase, xpad=0, ypad=0, ysize=10)
    !null = Widget_Slider(wBase, MINIMUM=0, MAXIMUM=100, TITLE='Transparency', $
                          /SUPPRESS_VALUE, UNAME='transparency', VALUE=value, /DRAG)
  endif

  state = {oTool: oTool, $
           parent: parent}
  Widget_Control, wBase, SET_UVALUE=state

  Widget_Control, wBase, /REALIZE
  XManager, 'cw_itpropertytoolbar_build_dialog', wBase, /NO_BLOCK
end

;-------------------------------------------------------------------------
pro cw_itpropertytoolbar_event, event
    compile_opt idl2, hidden

   widget_control, widget_info(widget_info(widget_info(event.id, /parent), /parent), /parent), $
                   GET_UVALUE=state   
   case widget_info(event.id, /UNAME) of
     'text': begin
        cw_itpropertytoolbar_build_dialog, event.id, state.oTool, $
           /COLOR, /FONT_NAME, /FONT_SIZE, /TRANSPARENCY
     end
     'background': begin
        cw_itpropertytoolbar_build_dialog, event.id, state.oTool, $
           /COLOR, /TRANSPARENCY, /NO_FILL
     end
     'line': begin
        cw_itpropertytoolbar_build_dialog, event.id, state.oTool, $
           /COLOR, /WEIGHT, /LINE_SIZE, /LINE_STYLE, /TRANSPARENCY
     end   
     'symbol': begin
        cw_itpropertytoolbar_build_dialog, event.id, state.oTool, $
           /COLOR, /SYMBOL_TYPE, /SYMBOL_SIZE, /SYMBOL_FILL, /TRANSPARENCY
     end
     'arrow': begin
        cw_itpropertytoolbar_build_dialog, event.id, state.oTool, $
           /ARROW_STYLE, /ARROW_THICK, /HEAD_ANGLE, /HEAD_INDENT, /HEAD_SIZE
     end
     else:
   endcase
end

;-------------------------------------------------------------------------
function cw_itpropertytoolbar, parent, GRAPHIC=isGraphic
  compile_opt idl2, hidden
  
  wBase = widget_Base(parent, /ROW, xpad=0, ypad=0)
  wContainer = widget_Base(wBase, /ROW, xpad=0, ypad=0)
  !null = widget_Button(wContainer, $
    VALUE=cw_ittoolbar_getbitmapfile('font.svg', ALLOW_LARGE=isGraphic), $
    TOOLTIP='Text Properties', $
    /FLAT, UNAME='text', EVENT_PRO='cw_itpropertytoolbar_event')
  wContainer = widget_Base(wBase, /ROW, xpad=0, ypad=0)
  !null = widget_Button(wContainer, $
    VALUE=cw_ittoolbar_getbitmapfile('fill-drip.svg', ALLOW_LARGE=isGraphic), $
    TOOLTIP='Fill Properties', $
    /FLAT, UNAME='background', EVENT_PRO='cw_itpropertytoolbar_event')
  wContainer = widget_Base(wBase, /ROW, xpad=0, ypad=0)
  !null = widget_Button(wContainer, $
    VALUE=cw_ittoolbar_getbitmapfile('chart-line.svg', ALLOW_LARGE=isGraphic), $
    TOOLTIP='Line Properties', $
    /FLAT, UNAME='line', EVENT_PRO='cw_itpropertytoolbar_event')
  wContainer = widget_Base(wBase, /ROW, xpad=0, ypad=0)
  !null = widget_Button(wContainer, $
    TOOLTIP='Symbol Properties', $
    VALUE=cw_ittoolbar_getbitmapfile('asterisk.svg', ALLOW_LARGE=isGraphic), $
    /FLAT, UNAME='symbol', EVENT_PRO='cw_itpropertytoolbar_event')   
  wContainer = widget_Base(wBase, /ROW, xpad=0, ypad=0)
  !null = widget_Button(wContainer, $
    TOOLTIP='Arrow Properties', $
    VALUE=cw_ittoolbar_getbitmapfile('up-long.svg', ALLOW_LARGE=isGraphic), $
    /FLAT, UNAME='arrow', EVENT_PRO='cw_itpropertytoolbar_event')

  return, wBase
end