; Copyright (c)  NV5 Geospatial Solutions, Inc. All
;       rights reserved. Unauthorized reproduction is prohibited.
;
;----------------------------------------------------------------------------
;+
; :Description:
;   Retrieve the next data block from the file as an anonymous structure.
;
; :Returns:
;   Structure<any>
;
; :Arguments:
;   lun: in, required, Integer
;     Unit number for the open file
;
;-
function asdf_parse_read_block, lun
  compile_opt idl2, hidden
  on_ioerror, doReturn

  block = 0
  block_magic_token = bytarr(4)
  readu, lun, block_magic_token
  if (~array_equal(block_magic_token, [0xd3, 0x42, 0x4c, 0x4b])) then begin
    return, 0
  endif

  header_size = 0us
  readu, lun, header_size
  swap_endian_inplace, header_size, /swap_if_little_endian
  header = { $
    flags: 0uL, $
    compression: bytarr(4), $
    alloc_size: 0uLL, $
    used_size: 0uLL, $
    data_size: 0uLL, $
    checksum: bytarr(16) $
  }
  readu, lun, header
  swap_endian_inplace, header, /swap_if_little_endian
  if (header_size gt 48) then begin
    xtra = bytarr(header_size - 48)
    readu, lun, xtra
  endif
  if (header.flags) then begin
    message, 'Unable to read streaming data'
  endif
  point_lun, -lun, dataStart
  nextblock = dataStart + header.alloc_size
  point_lun, lun, nextblock
  block = {compression: string(header.compression), $
    dataStart: dataStart, dataSize: header.used_size}
  doReturn:
  message, /reset
  return, block
end

;----------------------------------------------------------------------------
;+
; :Description:
;   Retrieve all of the binary data blocks as a list of block structures
;
; :Returns:
;   List<Structure<any>>
;
; :Arguments:
;   lun: in, required, Integer
;     Unit number for the open file
;
;-
function asdf_parse_get_blocks, lun
  compile_opt idl2, hidden
  on_ioerror, doReturn

  blocks = list()

  ; Find the beginning of the data blocks (if any)
  ; Everything below may fail with an eof - if that happens, it's fine, just
  ; return the blocks we've read so far.
  x = 0b
  while (x ne 0xD3) do begin
    readu, lun, x
  endwhile

  ; Back up one byte
  point_lun, -lun, pos
  point_lun, lun, pos - 1
  while (!true) do begin
    block = asdf_parse_read_block(lun)
    if isa(block, /scalar) then break
    blocks.Add, block
  endwhile

  doReturn:
  message, /reset
  return, blocks
end

;----------------------------------------------------------------------------
;+
; :Description:
;   Parse the ASDF header into an IDL YAML data structure
;
; :Returns:
;   YAML_Map
;
; :Arguments:
;   filename: in, required, String
;     The filename to parse
;   lun: out, optional, Integer
;     Unit number for the open file
;
;-
function asdf_parse_get_metadata, filename, lun
  compile_opt idl2, hidden

  docEnd = 0ll
  s = ''
  openr, lun, filename, /get_lun
  while (s ne '...' && not eof(lun)) do begin
    readf, lun, s
    docEnd++
  endwhile
  if (docEnd eq 0) then begin
    message, 'Invalid ASDF file: Unable to locate end of metadata.'
  endif
  metadata = strarr(docEnd)
  point_lun, lun, 0
  readf, lun, metadata
  metadata = metadata.join(`\n`)
  yaml = yaml_parse(metadata)

  if (~isa(yaml, 'YAML_Stream_Map')) then begin
    message, `Unexpected top-level class from yaml_parse: '${obj_class(yaml)}'`
  endif

  return, yaml
end

;----------------------------------------------------------------------------
;+
; :Description:
;   Convert the YAML_Map header object into an ASDF_File object.
;
; :Returns:
;   ASDF_File
;
; :Arguments:
;   yaml: in, required, YAML_Map
;     A YAML_Map object containing the ASDF header.
;
;-
function asdf_parse_convert_header, yaml
  compile_opt idl2, hidden

  ; We want to get the headers from our newly-read file, rather than our
  ; predefined header. That way if the headers change in the future,
  ; we'll have the latest version.
  asdf = ASDF_File(/skip_header)
  asdf.SetProperty, comments=yaml.comments, $
    global_tags=yaml.global_tags, tag=yaml.tag
  foreach tag, yaml, key do begin
    asdf[key] = tag
  endforeach
  catch, err
  if (err eq 0) then begin
    ; ASDF outputs these software tags in "flow" { } format.
    (asdf['asdf_library']).style = 'flow'
    (asdf['history', 'extensions', 0, 'software']).style = 'flow'
  endif
  catch, /cancel

  return, asdf
end

;----------------------------------------------------------------------------
;+
; :Description:
;   Convert a YAML_Map data block node into an ASDF_NDArray object,
;   and replace the original item in the hierarchy.
;
; :Arguments:
;   item: in, required, YAML_Map
;     a YAML_Map item to convert
;   node: in, required, YAML_Ma | YAML_Sequence
;     the parent container (either a YAML_Map or Sequence)
;   key: in, required, String
;     key used to store the item in the node
;   blocks: in, required, List<Structure<any>>
;     a list of internal data block structures
;   file: in, required, String
;     the input filename
;
;-
pro asdf_parse_create_datanode, item, node, key, blocks, file
  compile_opt idl2, hidden

  newitem = ASDF_NDArray()
  newitem.tag = item.tag
  newitem.alias = item.alias
  newitem.anchor = item.anchor
  newitem.value = item.value
  foreach value, item, subkey do begin
    newitem[idl_validname(subkey, /convert_all)] = value
  endforeach

  ; Replace the old data item
  node[key] = newitem

  if (~newitem.hasKey('source')) then return

  source = newitem.source

  ; Retrieve data block from either our own file or an external file.
  if (isa(source, /integer)) then begin
    n = blocks.length
    if (source ge -n && source lt n) then begin
      newitem._filename = file
      block = blocks[source]
    endif
  endif else if (isa(source, /string)) then begin
    externalFile = file_dirname(file, /mark_directory) + source
    newitem._filename = externalFile
    catch, err
    if (err eq 0) then begin
      !null = asdf_parse_get_metadata(externalFile, lun)
      block = (asdf_parse_get_blocks(lun))[0]
    endif
    catch, /cancel
    free_lun, lun, /force
  endif

  if (isa(block)) then begin
    if (block.compression ne '') then begin
      newitem.compression = block.compression
    endif
    newitem._dataStart = block.dataStart
    newitem._dataSize = block.dataSize
  endif
end

;----------------------------------------------------------------------------
;+
; :Description:
;   Convert all of our YAML_Map ndarrays to ASDF_NDArray data nodes
;
; :Arguments:
;   node: in, required, YAML_Ma | YAML_Sequence
;     the parent container (either a YAML_Map or Sequence)
;   blocks: in, required, List<Structure<any>>
;     a list of internal data block structures
;   file: in, required, String
;     the input filename
;
;-
pro asdf_parse_walk_metadata, node, blocks, file
  compile_opt idl2, hidden

  if (isa(node, 'list') || isa(node, 'hash')) then begin
    foreach item, node, key do begin
      if (isa(item, 'YAML_Map') && item.tag eq '!core/ndarray-1.0.0') then begin
        asdf_parse_create_datanode, item, node, key, blocks, file
      endif else begin
        asdf_parse_walk_metadata, item, blocks, file
      endelse
    endforeach
  endif
end

;----------------------------------------------------------------------------
;+
; :Description:
;   The asdf_parse function takes an ASDF filename and converts it
;   into an IDL variable.
;
; :Returns:
;   any
;
; :Arguments:
;   filename: in, required, String
;     *Filename* must be a path to the name of a valid ASDF file .
;
; :Keywords:
;   debug: in, optional, Boolean
;     Set to enable debug mode. Undocumented.
;
; :Author:
;   CT, March 2023.
;
;-
function asdf_parse, filename, DEBUG=debug

  compile_opt idl2, hidden
  lun = 0
  if (~keyword_set(debug)) then begin
    on_error, 2
    catch, ierr
    if (ierr ne 0) then begin
      catch, /cancel
      if (lun ne 0) then free_lun, lun, /force
      msg = (!error_state.msg).Replace('YAML_PARSE_INTERNAL: ', '')
      message, msg
    endif
  endif

  result = asdf_parse_get_metadata(filename, lun)
  blocks = asdf_parse_get_blocks(lun)
  free_lun, lun, /force
  asdf_parse_walk_metadata, result, blocks, filename
  result = asdf_parse_convert_header(result)

  return, result
end