; $Id$
;###############################################################################
;+
; NAME:
;   cw_dataSlice
;
; PURPOSE:
;   A compound widget that forms part of the UI service for the 
;   Data Slice operation.
;
; CATEGORY:
;   DAVE Widgets
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Jan 2009
;-
;###############################################################################




;===============================================================================
; cw_dataSlice_callback
; 
; PURPOSE:
;   Callback routine executed when a notification message is received
;
; PARAMETERS
;   wBase     - widget ID of the widget registered as an observer
;
;   strID     - string identifier of the source of the message
;
;   MessageIn - string that uniquely identifies the message being sent
;
;   valuedata - Data associated with the message being sent
;
; KEYWORDS:
;
; RETURN VALUE:
;
pro cw_dataSlice_callback, wBase, strID, messageIn, valueData
compile_opt idl2

if (~widget_info(wBase,/valid)) then return

; Only expecting messages from the operation when any of it's properties changes
; So respond by refreshing the graphic window to reflect the current cut settings

wChild = widget_info(wBase,/child)
widget_control, wChild, get_uvalue=sPtr

; draw the dataspace
cw_dataSlice_drawData, sPtr

; retrieve current cut parameters from the operation
cw_dataSlice_updateFromOp, sPtr

; draw the cut parameters
cw_dataSlice_drawSlice, sPtr


end


;===============================================================================
; cw_dataSlice_updateToOp
; 
; PURPOSE:
;   Update operation cut parameters (center and width) from graphics
;
; PARAMETERS
;   sPtr (in) - the state structure.
;
; KEYWORDS:
;
; RETURN VALUE:
;
pro cw_dataSlice_updateToOp, sPtr
compile_opt idl2

oOp = (*sPtr).oOp
oOp->SetProperty, sliceCenter=(*sPtr).center, sliceWidth=(*sPtr).width
oOp->DoOnNotify, oOp->GetFullIdentifier(), 'SETPROPERTY', ''

end


;===============================================================================
; cw_dataSlice_updateFromOp
; 
; PURPOSE:
;   Retrieve current cut parameters (axis, center and width) from the operation
;
; PARAMETERS
;   sPtr (in) - the state structure.
;
; KEYWORDS:
;
; RETURN VALUE:
;
pro cw_dataSlice_updateFromOp, sPtr
compile_opt idl2

oOp = (*sPtr).oOp
oOp->GetProperty, axis=axis, sliceCenter=center, sliceWidth=width $
                   ,rangemin=rangemin, rangemax=rangemax
(*sPtr).axis = axis
(*sPtr).center = center
(*sPtr).width = width
(*sPtr).range = [rangemin,rangemax]

end

;===============================================================================
; cw_dataSlice_drawSlice
; 
; PURPOSE:
;   Update the plot using the current cut parameters (axis, center and width)
;
; PARAMETERS
;   sPtr (in) - the state structure.
;
; KEYWORDS:
;
; RETURN VALUE:
;
pro cw_dataSlice_drawSlice, sPtr
compile_opt idl2

; Copy bkgd dataspace plot from the pixmap to vis window
wset, (*sPtr).winVis
device, copy=[0,0,!d.x_size,!d.y_size,0,0,(*sPtr).winPix]

; Now overplot an outline of the slice
;range = (axis eq 0)? (!x.crange)[1]-(!x.crange)[0] : (!y.crange)[1]-(!y.crange)[0]

axis = (*sPtr).axis
range = (*sPtr).range
wid = (*sPtr).width
cen = (*sPtr).center
rge = range[1] - range[0]

if (wid lt 0.01*rge) then begin
   if (axis eq 0) then $
      plots, [cen,cen], !y.crange, /data $ ;!y.crange, /data $
   else $
      plots, !x.crange, [cen,cen], /data ;!x.crange, [cen,cen], /data
endif else begin
   if (axis eq 0) then begin
      xlo = cen - 0.5 * wid
      xhi = cen + 0.5 * wid
      ylo = (!y.crange)[0]
      yhi = (!y.crange)[1]
   endif else begin
      ylo = cen - 0.5 * wid
      yhi = cen + 0.5 * wid
      xlo = (!x.crange)[0]
      xhi = (!x.crange)[1]
   endelse
   xbox = [xlo,xhi,xhi,xlo,xlo]
   ybox = [ylo,ylo,yhi,yhi,ylo]
   polyfill, xbox, ybox, /data
endelse

end


;===============================================================================
; cw_dataSlice_drawData
; 
; PURPOSE:
;   Update the pixmap window with the data
;
; PARAMETERS
;   sPtr (in) - the state structure.
;
; KEYWORDS:
;
; RETURN VALUE:
;
pro cw_dataSlice_drawData, sPtr
compile_opt idl2

; Retrieve data to be displayed from the operation
oOp = (*sPtr).oOp
oOp->GetProperty, xValue=x, yValue=y, xtitle=xtitle, ytitle=ytitle, dataValue=data

wset, (*sPtr).winPix

device, get_decomposed=old_dc, decomposed=0
loadct, 39, /silent
;plot, x, y, /nodata, xtitle=xtitle, ytitle=ytitle, xstyle=1, ystyle=1
;data = bytscl(temporary(data))
nx = n_elements(x)
ny = n_elements(y)
zdims = size(data,/dimensions)
zx = zdims[0]
zy = zdims[1]
if (ny eq zy+1) then begin
   ; y is in histogram so convert to points
   index = lindgen(zy)
   y = 0.5*(y[index] + y[index+1])
endif
if (nx eq zx+1) then begin
   ; x is in histogram so convert to points
   index = lindgen(zx)
   x = 0.5*(x[index] + x[index+1])
endif
contour, alog10(data),x,y, xtitle=xtitle, ytitle=ytitle, xstyle=1, ystyle=1 $
         ,nlevels=20, /fill

device, decomposed=old_dc

end


;===============================================================================
; cw_dataSlice_drawEvents
; 
; PURPOSE:
;   Main event handler
;
; PARAMETERS
;   event (in) - the event structure.
;
; KEYWORDS:
;
; RETURN VALUE:
;
pro cw_dataSlice_drawEvents, event
compile_opt idl2

wChild = widget_info(event.handler,/child)
widget_control, wChild, get_uvalue=sPtr

dataRange = (*sPtr).range
maxWidth = abs(dataRange[1] - dataRange[0])

possibleEventTypes = ['DOWN','UP','MOTION','SCROLL','EXPOSE']
eventType = possibleEventTypes[event.type]

refresh = 0
switch eventType of
   'DOWN': (*sPtr).mouse = event.press
   'UP': (*sPtr).mouse = event.press
   'MOTION': begin   ; up, down or motion are treated the same
      
      coords = (convert_coord(event.x,event.y,/device,/to_data))    

      ; If the mouse click falls outside the dataregion then only include
      ; values upto the data min,max
      if ((*sPtr).axis eq 0) then begin
         coords[0] = coords[0] < dataRange[1]
         coords[0] = coords[0] > dataRange[0]
      endif else begin
         coords[1] = coords[1] < dataRange[1]
         coords[1] = coords[1] > dataRange[0]
      endelse
      
      if ((*sPtr).mouse eq 4) then begin   ; right click so change slice width
         ; determine width depending on cut axis
         (*sPtr).width = ((*sPtr).axis eq 0)? $
                            2.0*abs( (*sPtr).center - coords[0]) : $
                            2.0*abs( (*sPtr).center - coords[1])
         refresh = 1
      endif

      if ((*sPtr).mouse eq 1) then begin   ; left click so change slice 
         (*sPtr).center = ((*sPtr).axis eq 0)? coords[0] : coords[1]
         refresh = 1
      endif
      
      if (refresh) then begin
         ; update graphics
         cw_dataslice_drawSlice, sPtr
         
         ; update operation
         cw_dataslice_updateToOp, sPtr
      endif
      
      break
   end
   
   else:   ; nothing to do
   
endswitch


end


;===============================================================================
; cw_dataSlice_events
; 
; PURPOSE:
;   Main event handler
;
; PARAMETERS
;   event (in) - the event structure.
;
; KEYWORDS:
;
; RETURN VALUE:
;
pro cw_dataSlice_events, event
compile_opt idl2

; basic error handler
catch, iError
if (iError ne 0) then begin
   catch, cancel   
   wChild = widget_info(event.handler,/child)
   widget_control, wChild, get_uvalue=sPtr
   oTool = ((*sPtr).oUI)->GetTool()
   oTool->ErrorMessage, !ERROR_STATE.MSG, severity=2 
   return
endif

sname = tag_names(event,/STRUCTURE_NAME)

if (strcmp(sname,'WIDGET_DRAW')) then cw_dataSlice_drawEvents, event

end


;===============================================================================
; cw_dataSlice_realize
; 
; PURPOSE:
;   Event handler. Called when compound widget is realized
;
; PARAMETERS
;   wBase (in) - id of the base widget.
;
; KEYWORDS:
;
; RETURN VALUE:
;
pro cw_dataSlice_realize, wBase
compile_opt idl2

wChild = widget_info(wBase,/child)
widget_control, wChild, get_uvalue=sPtr

; Get the IDL window id of the visible draw window
widget_control, (*sPtr).wDraw, get_value=winVis
(*sPtr).winVis = winVis

; draw the data axes
cw_dataSlice_drawData, sPtr


; retrieve current cut parameters from the operation
cw_dataSlice_updateFromOp, sPtr

; draw the cut parameters
cw_dataSlice_drawSlice, sPtr

end

;===============================================================================
; cw_dataSlice_killNotify
; 
; PURPOSE:
;   Event handler. Called when registered widget receives killnotify event
;
; PARAMETERS
;   wChild (in) - id of the widget registered to receive the killnotify message
;
; KEYWORDS:
;
; RETURN VALUE:
;
pro cw_dataSlice_killNotify, wChild
compile_opt idl2

widget_control, wChild, get_uvalue=sPtr

; cleanup
wDelete, (*sPtr).winPix
ptr_free, sPtr

end



;===============================================================================
; cw_dataSlice
; 
; PURPOSE:
;   Construct a compound widget that functions as part of the UI service used by
;   the dataSlice operation.
;
; PARAMETERS
;   wParent (in) - id of the parent widget. The parent widget embeds the this one.
;
;   oUI (in)     - object reference of the tool UI
;
; KEYWORDS:
;   uName (in)   - the uname to be assigned to this widget
;
;   xSize (in)   - the width of this widget
;
;   ySize (in)   - the height of this widget
;
;   value (in)   - object reference of the opertion requesting the UI services which
;                  contains this compound widget.
;
; RETURN VALUE:
;   the id of the base widget of the compound widget hierarchy.
;
function cw_dataSlice, wParent, oUI, uname=uName,xsize=xSize,ysize=ySize $
           ,value=oOperation, _REF_EXTRA=etc
compile_opt idl2

; Required parameters / keywords
if (n_params() ne 2) then Message, 'Expect number of arguments to be 2'
if (~widget_info(wParent,/valid)) then Message, 'Invalid parent widget'
if (~obj_valid(oUI)) then Message, 'Invalid UI object reference'
if (~obj_valid(oOperation)) then Message, 'Operation object reference is not specified'

; basic error handler
catch, iError
if (iError ne 0) then begin
   catch, /cancel
   Message, !error_state.msg
endif

;oTool = oUI->GetTool()
;
;oOperation->GetProperty, xValues=xVals, yValues=yVals, xTitle=xTitle, yTitle=yTitle

if (n_elements(uname) eq 0) then uname='TLB_CW_DATASLICE'
wBase = widget_base(wParent $
                        ,/align_center $
                        ,event_pro='cw_dataSlice_events' $
                        ,notify_realize='cw_dataSlice_realize' $
                        ,/row $
                        ,uname=uname $
                        ,_EXTRA=etc $
                        )


; Register this widget with the UI object
idSelf = oUI->RegisterWidget(wBase,'DataSlice','cw_dataSlice_callback')

; Register for notification messages from the operation component
oUI->AddOnNotifyObserver, idSelf, oOperation->GetFullIdentifier()

; add labels
wCB = widget_base(wBase, /col, /frame)
text = 'Cut Center'
text = [text, 'Click and drag']
text = [text, 'on draw window']
text = [text, 'with left mouse.']
text = [text, 'Release to set center.']
text = [text, ' ']
text = [text, 'Cut Width']
text = [text, 'Click and drag']
text = [text, 'on draw window']
text = [text, 'with right mouse.']
text = [text, 'Release to set width.']
;For i=0,n_elements(text)-1 do void = widget_label(wCB,value=text[i])

for i=0,n_elements(text)-1 do begin
    fgcolor = (i eq 0 || i eq 6)? [255,0,0] : [0,0,255]
    void = cw_coloredlabel(wCB,value=text[i],foreground_color=fgcolor,/align_center)
    ;void = widget_label(CB1,/align_center,value=strout[i])
endfor

; draw window
geom = widget_info(wCB,/geometry)
xsize = (n_elements(xsize) gt 0)? xsize : 200
ysize = (n_elements(ysize) gt 0)? ysize : geom.ysize ;200
wDraw = widget_draw(wBase,/button_events,/motion_events,xsize=xsize,ysize=ysize,uname='DRAW_CW_DATASLICE')
winVis = 0L
window,/free,xsize=xsize,ysize=ysize,/pixmap
winPix = !d.window

sPtr = ptr_new({oUI:oUI $
                  ,oOp:oOperation $
                  ,wDraw:wDraw $
                  ,winVis:winVis $
                  ,winPix:winPix $
                  ,axis:0 $            ; 0=>x, 1=>y
                  ,center:0.0 $        ; data units
                  ,width:0.0 $         ; data units
                  ,range:[0.0,0.0] $   ; data range on cut axis
                  ,mouse:0 $           ; which mouse? left or right
                 })


; store state var in first child of wBase since wBase is passed outside
; this compound widget and hence can be manipulated in an unpredictable way.
wChild = widget_info(wBase, /child)
widget_control, wChild, set_uvalue=sPtr, kill_notify='cw_dataSlice_killNotify'


return, wBase
end