; $Id$
;###############################################################################
;+
; NAME:
;   GenQFPreferences
;
; PURPOSE:
;   Retrieve or save user preferences for GenQF tool
;
; CATEGORY:
;   GenQF tool

; PARAMETERS:
;
; KEYWORDS:
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; April 2014
;-
;###############################################################################




;===============================================================================
;+
; NAME:
;   GenQFPreferences
;
; PURPOSE:
;   Main function
;
; PARAMETERS
;   preferences [in|out] - the prefrences structure to be save or updated if passed in or
;                          that was created or updated on exit.
; KEYWORDS:
;   save [in] - force the input preferences structure to be saved to disk
;   
;   reset [in] - force the preferences structure to be reset to their default values and then 
;                saved to disk
;   
;   updateVersion [in] - force a check of the preferenceVersion field of the input preferences
;                        structure. If it is not up to date, then update preferences structure
;                        to the current definition.
;
; RETURN:
;   Return values are 1 if successful or 0 otherwise
;-
function GenQFPreferences, preferences, save=save, reset=reset, updateVersion=updateVersion
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        print,'GenQFPreferences: Error Encountered'
        print,!ERROR_STATE.MSG
        catch, /cancel
        return, 0
    endif
endif

; ** Version number **
; ** Increment by 1 whenever the preferences structure content changes **
; ** Version 2:
;    new additions: AACompFlag, overplotFlag
versionNumber = 2

; Retrieve the app directory
appDir = getDAVEConfigDir()
if (n_elements(appDir) le 0) then return, 0
if (~file_test(appDir,/write,/read)) then return, 0
prefFile = appDir+path_sep()+'genqfpreferences.txt'

preferenceIsDefined = strcmp(size(preferences,/tname),'STRUCT')

; this updateVersion variable supercedes the updateVersion keyword but the keyword is retain for clarity of use!
updateVersion = 0
if (preferenceIsDefined) then begin
   ; old preferences structures may be missing the preferenceVersion field!
   oldTags = tag_names(preferences)
   void = where(strcmp(oldTags,'preferenceVersion',/fold),prefVersionIsDefined)
   oldVersion = (prefVersionIsDefined)? fix(preferences.preferenceVersion) : 0
   updateVersion = (versionNumber ne oldVersion)?  1 : 0
endif

if (keyword_set(save) && preferenceIsDefined) then begin
   ; save and exit
   ntags = n_tags(preferences)
   tags = tag_names(preferences)
   if (ntags lt 1) then return, 0

   openw, lun, prefFile, /get_lun
   ;printf, lun, 'PreferenceVersion : ',strtrim(string(versionNumber),2)
   for i=0,ntags-1 do printf, lun, tags[i] + ' : '+preferences.(i)
   
   close, lun, /force
   
   return, 1
endif

; Check version number
fileVersionNumber = 0
if (file_test(prefFile,/read)) then begin
   buffer = ''
   openr, lun, prefFile, /get_lun
   readf, lun, buffer
   close, lun, /force

   toks = strtrim(strsplit(buffer[0],':',count=ntok,/extract), 2)
   if (ntok eq 2) then fileVersionNumber = fix(toks[1])
endif
if (versionNumber ne fileVersionNumber) then reset = 1   ; force a reset of the preference file

if (keyword_set(reset) || ~file_test(prefFile,/read) || keyword_set(updateVersion)) then begin
   ; Set preferences to their default values, save to file and return to caller
   ; Also update structure if required and return to caller
   defPreferences = {PreferenceVersion:     strtrim(string(versionNumber),2) $
                  ,funcFlag:            '2' $
                  ,AACompFlag:          '0' $
                  ,overplotFlag:        '0' $
                  ,sf:                  '1.0' $
                  ,QQ:                  '25.0' $
                  ,mass:                '4.0026' $
                  ,kc:                  '0.5' $
                  ,n0:                  '0.07' $
                  ,Temp:                '1.0' $
                  ,alpha2:              '0.897' $
                  ,abar3:               '2.43' $
                  ,abar4:               '0.0' $
                  ,alpha4:              '0.46' $
                  ,abar52:              '2500' $
                  ,abar64:              '220' $
                  ,alpha6:              '0.38' $
                  ,u2:                  '0.9' $
                  ,u3:                  '0.2' $
                  ,u4:                  '0.4' $
                  ,u5:                  '0.1' $
                  ,u6:                  '0.4' $
                  ,smin:                '0.0' $
                  ,smax:                '6.0' $
                  ,sint:                '0.1' $
                  ,kmin:                '0.0' $
                  ,kmax:                '4.0' $
                  ,kint:                '0.1' $
                  ,ymin:                '-7.0' $
                  ,ymax:                '7.0' $
                  ,yint:                '0.1' $
                  ,errorbar_capsize:    '0.1' $
                  ,transparency:        '50' $
                  ,antialias:           '3' $
                  ,color:               '0,0,255' $
                  ,errorbar_color:      '0,0,255' $
                  ,use_default_color:   '1'   $
                  ,sym_increment:       '1'   $
                  ,sym_index:           '0'   $
                  ,sym_size:            '1.25'   $          ; 
                  ,sym_color:           '0,0,255' $   ;
                  ,sym_thick:           '1.0' $            ;
                  ,y_errorbars:         '0' $              ; don't show y errors
                  ,linestyle:           '0' $              ;
                  ,nsum:                '1' $              ; no point everaging
                  ,thick:               '3.0' $            ; line thickness
                  ,sym_filled:          '1'   $            ; use filled symbols
                  ,sym_fill_color:      '0,0,255' $   ; symbol fill color
                  ,font_name:           'Helvetica'  $   ;
                  ,font_size:           '13' $             ;
                  ,text_color:          '0,0,0'  $    ;
                 }

   ntags = n_tags(defPreferences)
   tags = tag_names(defPreferences)

   if (keyword_set(reset) || ~file_test(prefFile,/read)) then begin
      openw, lun, prefFile, /get_lun
      ;printf, lun, 'PreferenceVersion : ',strtrim(string(versionNumber),2)
      for i=0,ntags-1 do printf, lun, tags[i] + ' : '+defPreferences.(i)
      
      close, lun, /force
   endif
   if (keyword_set(updateVersion)) then begin
      ; if the version number of the input preference structure is not current, then
      ; we need to update the input structure to match the current definition by taking the
      ; default preferences structure and updating any matching fields with values from the input structure
      oTags = tag_names(preferences)
      oNtags = n_tags(preferences)
      for i = 0,oNtags-1 do begin   
         if (strcmp(oTags[i],'PreferenceVersion',/fold)) then continue ; ignore the old preferenceVersion field
         index = where(strcmp(tags,oTags[i]), found)
         if (found) then defPreferences.(index) = preferences.(i)
      endfor
   endif
   preferences = defPreferences
endif else begin
   ; Retrieve the current preferences from file and return to caller
   lines = file_lines(prefFile)
   buffer = strarr(lines)
   openr, lun, prefFile, /get_lun
   readf, lun, buffer
   close, lun, /force
   
   for i=0,lines-1 do begin
      toks = strtrim(strsplit(buffer[i],':',count=ntok,/extract), 2)
      ;toks = strsplit(buffer[i],':',count=ntok,/extract)
      if (ntok gt 1) then begin
         value = (ntok eq 2)? toks[1] : ''
         preferences = (i eq 0)? create_struct(toks[0],value) : $
            create_struct(preferences,toks[0],value)
      endif
   endfor
endelse

return, 1

end

