; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEtool
;
; PURPOSE:
;   This file implements the non-interactive core of the DAVE application
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitTool
;
; SUBCLASSES:
;
; CREATION:
;   See DAVEtool::Init
;
; METHODS:
;   AddDavePtrToDataManager
;   DelDAVEDataManagerContents
;   DelSystemDataManagerContents
;   DelVisTools
;   GetDaveDataManagerContents
;   GetProperty
;   GetStatePtr
;   GetSelectedData
;   GetSelectedDataNameTag
;   GetSystem
;   GetSystemDataManagerContents
;   GetUI
;   GetVisToolObjRefs
;   SetProperty
;   SetVisToolObjRefs
;   SetVisIDCount
;   GetVisIDCount
;   MenuLayout
;   ModifyComponents
;   SetPreferences
;
; INTERFACES:
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; May 2004
;-
;###############################################################################

;===============================================================================
; DAVEtool::Cleanup
; 
; PURPOSE:
;   Cleanup method for objects of DAVEtool class
;
; PARAMETERS:
;
; KEYWORDS:
;
pro DAVEtool::Cleanup
compile_opt idl2

; Cleanup superclass
self->IDLitTool::Cleanup

end

;===============================================================================
; DAVEtool::GetSystem
; 
; PURPOSE:
;   Return the IDLitSystem object
;
; PARAMETERS:
;
; KEYWORDS:
;
function DAVEtool::GetSystem
compile_opt idl2

return, self->_GetSystem()

end


;===============================================================================
; DAVEtool::GetUI
; 
; PURPOSE:
;   Return the IDLitUI object for this tool
;
; PARAMETERS:
;
; KEYWORDS:
;
function DAVEtool::GetUI
compile_opt idl2

return, self._oUIConnection

end


;===============================================================================
; DAVEtool::ModifyComponents
; 
; PURPOSE:
;   Modify/remove components that were added or registered by the
;   base class (IDLitTool) or System object.
;
; PARAMETERS
;
; KEYWORDS:
;
function DAVEtool::ModifyComponents
compile_opt idl2

; The main interface does not have a graphics window, so there is no
; need for the 'probe' segment of the statusbar. In addition, rescale
; the 'message' segment to occupy the full width of the dialog.
self->UnRegisterStatusBarSegment, 'PROBE'
oSegment = self->GetStatusBarSegments(identifier='MESSAGE', count=cnt)
if (cnt && obj_valid(oSegment)) then oSegment->SetProperty, normalized_width=1.0
 

return, 1
end


;===============================================================================
; DAVEtool::_restoreSettings
; 
; PURPOSE:
; Restore preferences from file. Simply call the system object's method
;
; PARAMETERS
;
; KEYWORDS:
;
function DAVETool::_restoreSettings
compile_opt idl2

; TODO: Completely override IDLitystem::_restoreSettings method so as to avoid restoring from the 
;       default 'itools_prefs.sav' file that is used by all iTool-based applications.


; Retrieve the preference file, if present. Extract the structure names currently defined in the 
; file. If DAVEGENERALSETTINGS structure is not present, then don't restore this file since it implies it
; was not created by DAVE but another iTool program
if (~IDLitGetResource('', strName, /USERDIR)) then $
    return, 0
strPrefs = file_expand_path(filepath("itools_prefs.sav", root_dir=strName))

if (~file_test(strPrefs)) then return, 0

oSaveFile = OBJ_NEW('IDL_Savefile', strPrefs)
structs = oSaveFile->Names(COUNT=nstruct, /STRUCTURE_DEFINITION)
obj_destroy, oSavefile
void = where(structs eq 'DAVEGENERALSETTINGS', exist)
if (~exist) then return, 0

oSystem = self->GetSystem()

; Call system object's restore method to actually restore contents of
; saved settings/preferences file.
if (~oSystem->_restoreSettings()) then return, 0

if (self.reqReadWriteReg) then begin
    ;; Unregister ASCII text reader/writer registered in call to
    ;; oSystem::_ReigisterReaderWriters as they are replaced by
    ;; DAVE-specific ascii readers/writers
    self->UnRegisterFileReader, 'ASCII text'
    self->UnRegisterFileWriter, 'ASCII text'

    ;; Register DAVE-specific readers/writers
    xmlfile = file_which(!DAVE_AUXILIARY_DIR,'components.xml')
    xmlDoc = OBJ_NEW('IDLffXMLDOMDocument', filename=xmlfile)
    element = xmldoc->GetElementsByTagName("dave2_component")
    if (~obj_valid(element) || (element->GetLength() lt 1)) then return, 0

    void = registerReaderWriters(oSystem, xmlDoc)

    obj_destroy, xmlDoc
    self.reqReadWriteReg = 0    ; set flag to disable re-registration of DAVE specific readers/writers
    
endif


; Syncronize directories with tools that are currently active
; also synchronize the ftkwwFlag
oGeneral = oSystem->GetByIdentifier('/REGISTRY/SETTINGS/GENERAL_SETTINGS')
if (OBJ_VALID(oGeneral)) then BEGIN
    oGeneral->getProperty, working_directory=wDir, data_directory=dDir, ftkwwFlag=ftkwwFlag
    oGeneral->_UpdateCurrentTools,/update_tool,working_directory=wDir, data_directory=dDir
    !ftkwwFlag = ftkwwFlag
ENDIF

return, 1

end


;===============================================================================
; DAVEtool::_saveSettings
; 
; PURPOSE:
; Save current preferences to file. Simply call the system object's method
;
; PARAMETERS
;
; KEYWORDS:
;
pro DAVETool::_saveSettings
compile_opt idl2

;oSystem = self->GetSystem()
;oSystem->_SetCurrentTool, Self, /no_notify
;oSystem->_saveSettings


;** Copy verbatim from idlitsystem::_savesettings **

;; Save our settings
;; Make sure we can write to this directory
if (~Idlitgetresource('', strName, /USERDIR, /WRITE)) then begin
  if (Keyword_set(shutdown)) then Return
  self->Errormessage, Idlitlangcatquery('Error:CannotWritePrefsDir:Text') +strName, $
    severity=1
  Return
endif

strPrefs = File_expand_path(Filepath("itools_prefs.sav", root_dir=strName))


;; Get the settings folder and just save it off.
oSettings = self->Getbyidentifier("/REGISTRY/SETTINGS")

;; Set up a catch block to handle any errors save might throw at us.
@idlit_catch
if(iErr ne 0)then begin
  Catch,/cancel
  if (Keyword_set(shutdown)) then Return
  self->Errormessage, Idlitlangcatquery('Error:CannotWritePrefs:Text') +strPrefs, $
    severity=1
  Return
endif
@idlitconfig.pro
PREFERENCES_VERSION =  ITOOLS_STRING_VERSION
Save, file=strPrefs, oSettings, PREFERENCES_VERSION, /compress


end


;===============================================================================
; DAVEtool::initPreferences
; 
; PURPOSE:
; Initialize application settings/preferences.
; Method is called after this tool is fully created. Objective is to (re)initialize the application preferences
;
; PARAMETERS
;
; KEYWORDS:
;
pro DAVETool::initPreferences
compile_opt idl2

self->_redefineSettings
void = self->_restoreSettings()

end



;===============================================================================
; DAVEtool::_resetSettings
; 
; PURPOSE:
;   Reset the global preferences to their default values. Use DAVE specific code that
;   extends the standard iTools version, IDLitSystem::_ResetSettings()
;
; PARAMETERS
;
; KEYWORDS:
;
pro DAVETool::_resetSettings
compile_opt idl2

; Re-create the global preference object
self->_redefineSettings

;; Do we have a saved reg file? Delete it.
if (IDLitGetResource('', strName, /USERDIR)) then begin
    strPrefs = file_expand_path(filepath("itools_prefs.sav", root_dir=strName))
    file_delete, strPrefs, /allow_nonexistent, /quiet
    if(file_test(strPrefs))then begin
        self->ErrorMessage, $
          [IDLitLangCatQuery('Error:ResetSettings:Text1'), $
           strPrefs, IDLitLangCatQuery('Error:ResetSettings:Text2')], $
          title=IDLitLangCatQuery('Error:ResetSettings:Title'), severity=1
    endif
endif

end


;===============================================================================
; DAVEtool::_redefineSettings
; 
; PURPOSE:
;   Redefine the global preferences using DAVE specific code that
;   extends the standard iTools version, IDLitSystem::_ResetSettings()
;
; PARAMETERS
;
; KEYWORDS:
;
pro DAVETool::_redefineSettings
compile_opt idl2

oSystem = self->GetSystem()

;; delete prefrences objects from the registry
obj_destroy, oSystem->RemoveByIdentifier("/REGISTRY/SETTINGS")
self.reqReadWriteReg = 1        ; set flag to enable re-registration of DAVE specific readers/writers

;; Create our folders
oSystem->CreateFolders, [ $
  "Registry/Settings/File Readers", $
  "Registry/Settings/File Writers"]

;; Okay, create the settings objects
self->_RegisterPreferences

;; update styles and available languages
oGeneral = oSystem->GetByIdentifier('/REGISTRY/SETTINGS/GENERAL_SETTINGS')
if (OBJ_VALID(oGeneral)) then oGeneral->VerifySettings

end


;===============================================================================
; DAVEtool::_RegisterPreferences
; 
; PURPOSE:
;   Register global system preferences using  DAVE specific class.
;   Based on IDLitSystem::_RegisterPreferences
;
; PARAMETERS
;
; KEYWORDS:
;
pro DAVETool::_RegisterPreferences
compile_opt  idl2

oSystem = self->GetSystem()

;; Add our general settings
oCurrent = oSystem->GetByIdentifier('/REGISTRY/SETTINGS/GENERAL_SETTINGS')

oGeneral = OBJ_NEW('DAVEGeneralSettings', TOOL=oSystem $
                   ,name='General DAVE Settings' $
                   ,description='General Application-wide Settings' $
                   ,IDENTIFIER="GENERAL_SETTINGS")
oSystem->AddSetting, oGeneral, POSITION=0

; Register system define readers/writers
oSystem->_RegisterReaderWriters

if (self.reqReadWriteReg) then begin
    ;; Unregister ASCII text reader/writer registered in call to
    ;; oSystem::_ReigisterReaderWriters as they are replaced by
    ;; DAVE-specific ascii readers/writers
    self->UnRegisterFileReader, 'ASCII text'
    self->UnRegisterFileWriter, 'ASCII text'

    ;; Register DAVE-specific readers/writers
    xmlfile = file_which(!DAVE_AUXILIARY_DIR,'components.xml')
    xmlDoc = OBJ_NEW('IDLffXMLDOMDocument', filename=xmlfile)
    element = xmldoc->GetElementsByTagName("dave2_component")
    if (~obj_valid(element) || (element->GetLength() lt 1)) then return

    void = registerReaderWriters(oSystem, xmlDoc)

    obj_destroy, xmlDoc
;    self.reqReadWriteReg = 0    ; set flag to disable re-registration of DAVE specific readers/writers
    
endif

end



;===============================================================================
; DAVEtool::SetPreferences
; 
; PURPOSE:
;   Perform various tasks which essentially customize the system to
;   our needs. The DAVEtool class inherits a lot of stuff which need
;   to be customize.
; PARAMETERS
;   oSystem - the iTool system object
;
; KEYWORDS:
;
;function DAVEtool::SetPreferences
;compile_opt idl2
;
;; Customize various components
;if (~self->ModifyComponents()) then return, 0
;
;return, 1
;
;end



;===============================================================================
;
function DAVETool::GetSelectedDataNameTag, count
compile_opt idl2

selItem = Self->GetSelectedData(count=cnt)
if (cnt eq 0) then return, ''

For i = 0,cnt-1 do begin
   oItem = selItem[i]
   if (~obj_valid(oItem)) then continue
   oItem->GetProperty, name=name
   oTop = getmaindataobjectparent(oItem)
   if (obj_valid(oTop) && obj_isa(oTop, 'GENERICDATASET')) then $
      oTop->GetProperty, name=name
   nameTag = (n_elements(nameTag) gt 0)? [nameTag,name] : name
endfor
count = n_elements(nameTag)
return, (n_elements(nameTag) gt 0)? nameTag : ''

end



;===============================================================================
; Convert the specified davePtr data structure into a dataset object and
; add it to the Data Manager
pro DAVETool::AddDavePtrToDataManager, davePtr, nameTag
compile_opt idl2

if ((n_elements(nameTag) le 0) || (strtrim(nameTag,2) eq '')) then nameTag='Untitled'
oData = obj_new('DAVEDataset',davePtr=davePtr, nameTag=nameTag)

if (obj_valid(oData)) then begin
    Self->AddDatasetToDataManager, oData
endif

end


;===============================================================================
; Add the specified dataset object to the Data Manager
pro DAVETool::AddDatasetToDataManager, oData
compile_opt idl2

n = n_elements(oData)
if (n lt 1) then return

for i=0,n-1 do begin
   if (obj_valid(oData[i])) then begin
      ; iTools does not handle '_[0-9]' characters in identifiers well eg _0, _1, etc
      ; replace all '_' with '-'

      oData[i]->GetProperty, identifier=idd
      hasChanged = 0
      while ((pos = stregex(idd,'_[0-9]')) ne -1) do begin
         strput, idd,'-',pos
         hasChanged = 1
      endwhile
      if (strmatch(idd,'[0-9]*')) then begin   ; if first char is a number
         idd = 'id'+idd
         hasChanged = 1
      endif
      if (hasChanged) then oData[i]->SetProperty,identifier=idd 
   
       self->AddByIdentifier, "DAVE Data Manager", oData[i]
   endif
endfor

end


;===============================================================================
; Return the dataset objects currently in the Data Manager folder
function DAVETool::GetDAVEDataManagerContents, count=count
compile_opt idl2

count = 0

; Get the DAVE Data Manager Folder
dmid = self->GetFullIdentifier()+'/DAVE DATA MANAGER'
oDM = self->GetByIdentifier(dmid)
if (~obj_valid(oDM) || ~obj_isa(oDM,'IDLitDataManagerFolder')) then return, obj_new()

return, oDM->Get(/all,count=count)

end


;===============================================================================
; Return the contents of the System's Data Manager folder
function DAVETool::GetSystemDataManagerContents, count=count
compile_opt idl2

count = 0

; Get the Data Manager Folder
oSystem = self->GetSystem()
oDM = oSystem->GetByIdentifier('/Data Manager')
if (~obj_valid(oDM) || ~obj_isa(oDM,'IDLitDataManagerFolder')) then return, obj_new()

return, oDM->Get(/all,count=count)

end


;===============================================================================
; Delete all dataset objects from the Data Manager folder
pro DAVETool::DelDAVEDataManagerContents
compile_opt idl2

; Get the datasets
oDataSets = self->GetDAVEDataManagerContents(count=count)
if (count lt 1) then return

; Get the DAVE Data Manager Service
oDM = self->GetService('DAVEDATA_MANAGER')
if (~obj_valid(oDM) || ~obj_isa(oDM,'IDLitsrvDataManager')) then return

; Delete them
while (count-- gt 0) do begin
    id = oDatasets[count]->GetFullIdentifier()
    if (strtrim(id,2) ne '') then status = oDM->DeleteData(id)
endwhile

end


;===============================================================================
; Delete the contents of the System's data manager folder
pro DAVETool::DelSystemDataManagerContents
compile_opt idl2

; Get the datasets
oDataSets = self->GetSystemDataManagerContents(count=count)
if (count lt 1) then return

; Get the Data Manager Service
oDM = self->GetService('DATA_MANAGER')
if (~obj_valid(oDM) || ~obj_isa(oDM,'IDLitsrvDataManager')) then return

; Delete them
while (count-- gt 0) do begin
    id = oDatasets[count]->GetFullIdentifier()
    if (strtrim(id,2) ne '') then status = oDM->DeleteData(id)
endwhile

end


;===============================================================================
; Delete all existing visualization tools (windows)
pro DAVETool::DelVisTools
compile_opt idl2

; Get the Vis Tools
oVisTools = self->GetVisToolObjRefs(count=count)
if (count lt 1) then return

; Delete them
while (count-- gt 0) do begin
    ;; retrieve the shutdown service
    oShutDown = oVisTools[count]->GetService('SHUTDOWN')
    if (~obj_valid(oShutdown)) then continue

    ;; and kill the tool
    oUI = oVisTools[count]->GetUI()
    oVisTools[count]->_SetDirty, 0 ; so it appears unmodified -> no prompting!
    status = oUI->DoAction(oShutdown->GetFullIdentifier())
endwhile

end


;===============================================================================
; Get the state pointer associated with the main application's top level base
function DAVETool::GetStatePtr
compile_opt idl2

; get the TLB
oUI = self->GetUI()
oUI->GetProperty, group_leader=wTLB
widget_control, widget_info(wTLB,/child), get_uvalue=sPtr
return, sPtr

end


;===============================================================================
; Return all items that are currently selected in the Data Browser tree widget
function DAVETool::GetSelectedData, count=count
compile_opt idl2

sPtr = self->GetStatePtr()
widget_control, (*sPtr).wDM, get_uvalue=dm_sPtr
DataIDs = (*(*dm_sPtr).pDataIDs)
for i = 0,n_elements(DataIDs)-1 do begin
    oTmp = self->getbyidentifier(DataIDs[i])
    oData = (n_elements(odata) eq 0)? oTmp : [oData,oTmp] 
endfor

count = n_elements(oData)
if (count gt 0) then $
  return, oData $
else $
  return, obj_new()

end


;===============================================================================
; Convert the selected dataset(s) into davePtr structure format.
; Will only be successfull if the selected item is the main
; dataset parent and is derived from the GenericDataset class.
;
; NB: Potential memory leak - the DAVEPtr created here is owned by the caller and
;     hence is the caller's respnsibility to delete it.
function DAVETool::GetSelectedDataAsDAVEPtr, count=count
compile_opt idl2

oDataArr = self->GetSelectedData(count=count)
if (count eq 0) then return, ptr_new()

errMsg = ''
for i=0,count-1 do begin
   oData = oDataArr[i]
   if (obj_valid(oData) && obj_isa(oData,'GENERICDATASET')) then begin
      davePtr = ptr_new()  ; ensure an empty davePtr is passed to toDavePtr()
      status = oData->ToDavePtr(davePtr, errMsg)
      if (status) then  $
         davePtrArr = (n_elements(davePtrArr) gt 0)? [davePtrArr,davePtr] : davePtr $
         else Self->Statusmessage, errMsg
   endif
endfor

count = n_elements(davePtrArr)

if (count gt 0) then $
  return, davePtrArr $
else $
  return, ptr_new()

end


;===============================================================================
; Convert all datasets in the Data Manager into davePtr structure format.
; Only datasets derived from the GenericDataset class can be converted by this method
;
; NB: Potential memory leak - the DAVEPtr created here is owned by the caller and
;     hence is the caller's respnsibility to delete it.
function DAVETool::GetDAVEDataManagerContentsAsDAVEPtr, count=count
compile_opt idl2

oDataArr = self->GetDAVEDataManagerContents(count=count)

for i=0,count-1 do begin
   oData = oDataArr[i]
   if (obj_valid(oData) && obj_isa(oData,'GENERICDATASET')) then begin
      davePtr = ptr_new()  ; ensure an empty davePtr is passed to toDavePtr()
      status = oData->ToDavePtr(davePtr)
      if (status) then  $
         davePtrArr = (n_elements(davePtrArr) gt 0)? [davePtrArr,davePtr] : davePtr
   endif
endfor

count = n_elements(davePtrArr)

if (count gt 0) then $
  return, davePtrArr $
else $
  return, ptr_new()

end


;===============================================================================
;
pro DAVETool::SetVisToolObjRefs, visToolObjRefs
compile_opt idl2

index = where(obj_valid(visToolObjRefs),cnt)
if (cnt eq 0) then return

sPtr = self->GetStatePtr()
(*(*sPtr).visToolArrPtr) = visToolObjRefs[index]

end


;===============================================================================
;
function DAVETool::GetVisToolObjRefs, count=cnt
compile_opt idl2

sPtr = self->GetStatePtr()
oArr = (*(*sPtr).visToolArrPtr)
void = where(obj_valid(oArr), cnt)
return, (oArr)

end


;===============================================================================
;
pro DAVETool::SetVisIDCount, visIDCount
compile_opt idl2

sPtr = self->GetStatePtr()
(*(*sPtr).visIDCountPtr) = visIDCount

end


;===============================================================================
;
function DAVETool::GetVisIDCount
compile_opt idl2

sPtr = self->GetStatePtr()
return, (*(*sPtr).visIDCountPtr)

end



;===============================================================================
; DAVEopFileOpen::GetProperty
; 
; PURPOSE:
;   Accessor method
;
; PARAMETERS:
;
; KEYWORDS:
;   data_directory [out] - Default location for searching raw data files
;
; RETURN VALUE:
;
pro DAVETool::GetProperty, data_directory=datadir $
  ,dave_major_version=major_ver $
  ,dave_minor_version=minor_ver $
  ,_REF_EXTRA=etc
compile_opt idl2

if (arg_present(major_ver)) then begin
    major_ver =  self.dave_major_version
endif
if (arg_present(minor_ver)) then begin
    minor_ver =  self.dave_minor_version
endif
if (arg_present(dataDir)) then begin
    dataDir =  self.dataDir
endif

if(n_elements(etc) gt 0) then $
  self->IDLitTool::GetProperty, _EXTRA=etc

end


;===============================================================================
; DAVETool::SetProperty
; 
; PURPOSE:
;   Accessor method
;
; PARAMETERS:
;
; KEYWORDS:
;   data_directory [in] - The names of readers that should be opened by
;                       an operation created from this class
;
; RETURN VALUE:
;
pro DAVETool::SetProperty, data_directory=dataDir $
  ,dave_major_version=major_ver $
  ,dave_minor_version=minor_ver $
  , _EXTRA=etc
compile_opt idl2

if (n_elements(major_ver) gt 0) then begin
      self.dave_major_version = major_ver
endif
if (n_elements(minor_ver) gt 0) then begin
      self.dave_minor_version = minor_ver
endif
if (n_elements(dataDir) gt 0) then begin
      self.dataDir = dataDir
endif

if(n_elements(etc) gt 0) then $
  self->IDLitTool::SetProperty, _EXTRA=etc 

end


;===============================================================================
; DAVEtool::Init
; 
; PURPOSE:
;   Initialization method for objects of DAVEtool class
;
; PARAMETERS
;
; KEYWORDS:
;
function DAVEtool::Init, _REF_EXTRA=etc
compile_opt idl2

catch, theError
if (theError ne 0) then begin
    catch, /cancel
    message, /reissue_last
    return, 0
endif

; call superclass
if (~self->IDLitTool::init(_EXTRA=etc $
                           ,help='DAVETOOL_HELP' $ ; help keyword for this class
                          )) then return, 0


; Add a 'DAVEDATA_MANAGER' service
self->AddService, OBJ_NEW("DAVEsrvDatamanager", name="DAVEDATA_MANAGER")


; Create the dave data manager folder
oDMFolder = Obj_New("IDLitDataManagerFolder", identifier='DAVE Data Manager' $
                      ,name='Data', description='DAVE Data Manager Container',tool=self)
self->Add, oDMFolder


; Create the dave visualization manager folder
;self->createFolders,'DAVE Visualization Manager'
;oVMFolder = self->getbyidentifier('DAVE Visualization Manager')
;oVMFolder->SetProperty,name='Visualizations',description='DAVE Visualization Manager Container'


; Do any tool customization now
if (~self->ModifyComponents()) then return, 0


; Define required menu containers and menu items for this tool
; by linking in known or expected registered components
if (~self->MenuLayout()) then return, 0


; set flag to disable re-registration of DAVE specific readers/writers
self.reqReadWriteReg = 0 


; return success
return, 1

end



;===============================================================================
; Class structure definition
pro DAVEtool__Define
compile_opt idl2

void = {DAVEtool, inherits IDLitTool $
        ,dave_major_version:0 $      ; vesion info
        ,dave_minor_version:0 $      ; vesion info
        ,reqReadWriteReg:0B    $ ; 
        ,dataDir:'' $
       }

end
