; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEopFileOpen
;
; PURPOSE:
;   A general-purpose operations class for initiating a file open
;   action within DAVE. A menu button which opens a file would normally be
;   associated with this operation.
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOpFileOpen
;
; METHODS:
;   DoAction
;   GetfilterList
;   GetProperty
;   SetProperty
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; July 2004
;-
;###############################################################################


;===============================================================================
; DAVEopFileOpen::GetProperty
; 
; PURPOSE:
;   Accessor method
;
; PARAMETERS:
;
; KEYWORDS:
;   readerNames [out] - The names of readers that should be opened by
;                       an operation created from this class
;
; RETURN VALUE:
;
pro DAVEopFileOpen::GetProperty, readerNames=readerNames, _REF_EXTRA=etc
compile_opt idl2

if (arg_present(readerNames)) then begin
    readerNames =  (ptr_valid(self.readerNames))? (*self.readerNames) : ''
endif

if(n_elements(etc) gt 0) then $
  self->IDLitopFileOpen::GetProperty, _EXTRA=etc

end


;===============================================================================
; DAVEopFileOpen::SetProperty
; 
; PURPOSE:
;   Accessor method
;
; PARAMETERS:
;
; KEYWORDS:
;   readerNames [in] - The names of readers that should be opened by
;                       an operation created from this class
;
; RETURN VALUE:
;
pro DAVEopFileOpen::SetProperty, readerNames=readerNames, _EXTRA=etc
compile_opt idl2

if (n_elements(readerNames) gt 0) then begin
    if (ptr_valid(self.readerNames)) then $
      (*self.readerNames) = readerNames $
    else $
      self.readerNames = ptr_new(readerNames)
endif

if(n_elements(etc) gt 0) then $
  self->IDLitopFileOpen::SetProperty, _EXTRA=etc 

end


;===============================================================================
; DAVEopFileOpen::GetFilterList
; 
; PURPOSE:
;   Provide a list of file filters (extensions) that should be handled
;   by the current operation. Method is intended to be called by the
;   UI Service which launches the file selection dialog.
;   In DAVE, this would be the 'FileOpen' UI Service
;
; PARAMETERS:
;
; KEYWORDS:
;   count [out] - The number of filter groups (one group per reader
;                 which may actually contain more than one filter)
;
; RETURN VALUE:
;    If successful, an array of reader name/filter(s) pairs
;    If unsuccessful, an empty string.
;
function DAVEopFileOpen::GetFilterList, COUNT=count
compile_opt idl2

oTool = self->GetTool()
oReadFile = oTool->GetService("READ_FILE")
if(not obj_valid(oReadFile))then begin
    count = 0
    return,''
endif

self->GetProperty,readerNames=readerNames

return, oReadFile->GetFilterListByReaderName(readerNames,COUNT=count)

end


;===============================================================================
; DAVEopFileOpen::DoAction
; 
; PURPOSE:
;   Implements the main function for this operation. The main task is
;   to read data from a file (using the appropriate file reader) and
;   place the result in DAVE's public data folder.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an array of reader name/filter(s) pairs
;    If unsuccessful, an empty string.
;
function DAVEopFileOpen::DoAction, oTool

compile_opt idl2

; Get the File Reader service - provides various useful functionality
;                               for reading data from a file.
oReadFile = oTool->GetService('READ_FILE')
if(not obj_valid(oReadFile))then begin
    errorMsg = 'Unable to access data reading service.'
    errorTitle = 'Internal Error'
    self->ErrorMessage, errorMsg,title=errorTitle, severity=2
    return, obj_new()
endif

; By default, the WORKING_DIRECTORY property of the Tool is used as the
; directory from while files are to be loaded from. This works well for all 
; readers except for RAWOSIRIS which requires the DATA_DIRECTORY instead.
; As a fix, simply assign the data directory to the working directory property
; temporarily if the reader is RAWOSIRIS
isRawOsiris = 0
if (n_elements(Self.readerNames) gt 0) then begin
   void = where(strupcase(*Self.readerNames) eq 'RAWOSIRIS', isRAWOSIRIS)
   if (isRawOsiris gt 0) then begin
      oTool->GetProperty, WORKING_DIRECTORY=workingDirectory, DATA_DIRECTORY=dataDirectory
      oTool->SetProperty, WORKING_DIRECTORY=dataDirectory
   endif
endif


; Present a file selection dialog to the user to choose the file to
; be read.
success = oTool->DoUIService('FileOpen', self)
if (isRawOsiris gt 0) then oTool->SetProperty, working_directory=workingdirectory   ; reset prev working directory
if (success eq 0) then $
  return, obj_new()

; Check to make sure at least one file was indeed selected from the
; preceeding step.
nFiles = N_ELEMENTS(*self._fileNames)
if(nFiles eq 0)then $
  return, obj_new()

; Loop through each selected file, read its contents are place them in
; the DAVE's data folder.
nData =0
for i=0, nFiles-1 do begin
    status = oReadFile->ReadFile((*self._fileNames)[i], oData)
    if (status lt 0) then continue ; user hit cancel
    if (status eq 1) then begin
        nData += n_elements(oData)
        oItems = (n_elements(oItems) gt 0)? [oItems,oData] : oData 
    endif else begin
        self->ErrorMessage, /USE_LAST_ERROR, $
                            title="File Error", severity=2, $
                            ["Error Reading from file:", $
                             (*self._fileNames)[i]]
    endelse
endfor
if (nData gt 0) then begin
   ;; Add data to the data folder
   oTool->AddDatasetToDataManager, oItems
   oTool->_SetDirty, 1     ; indicate tool has been modified
endif

; We never received any data. Either all of the files were bad,
; or the user hit cancel on some dialog, or we were restoring
; a save state file.
if (nData eq 0) then $
  return, obj_new()

; Usually, the doAction() method will also create a commandset object
; that will contain undo/redo data. The initial state of the (a
; selected) object is saved, the object is modified as requested and
; the final state saved again. Finally the commandset object is
; returned. 
; For now, no commandset object is constructed implying that this
; operation is not subject to the undo/redo functionality.
;
; **** Try adding undo/redo!!!!!!!!!!!!!!!!
return, obj_new()

end


;===============================================================================
; DAVEopFileOpen::Cleanup
; 
; PURPOSE:
;   DAVEopFileOpen class cleanup
;
pro DAVEopFileOpen::Cleanup

compile_opt idl2

ptr_free, self.readerNames

; call base class cleanup
self->IDLitopFileOpen::Cleanup

end


;===============================================================================
; DAVEopFileOpen__define
; 
; PURPOSE:
;   DAVEopFileOpen class structure definition
;
pro DAVEopFileOpen__define

compile_opt idl2

struc = {DAVEopFileOpen, $
         readerNames:ptr_new(''), $ ; names of readers that should be opened
         inherits IDLitopFileOpen }

end
