
; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DaveopDataCombine
;
; PURPOSE:
;   A simple operation for summing two or more datasets together. The
;   datasets must of the same size (dimensions and elements in each dimension).
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOperation
;
; METHODS:
;   DoAction
;   DoExecuteUI
;   GetProperty
;   RecordInitialValues
;   RecordFinalValues
;   RedoOperation
;   ResetOpAttributes
;   SetProperty
;   Sum
;   UndoOperation
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Mar 2005
;-
;###############################################################################


;===============================================================================
; DaveopDataCombine::GetProperty
; 
; PURPOSE:
;   Accessor
;
; PARAMETERS:
;
; KEYWORDS:
;   dataset*   - Eg dataset1, dataset2, ...
;                These are the names of the datasets that the user has selected to be
;                included in the sum. Because the number of datasets are not known a priori,
;                it would be very cumbersome to itemise these properties as keywords to this 
;                procedure. Instead they are received via the _REF_EXTRA keyword and 
;                additional processing is applied with the use of the Scope_Varfetch() to
;                determine when these properties are requested. The actual values for the
;                properties are maintained in the class variables Self.propertyNames and
;                Self.datasetsIndex
;
;   ndata      - The number of datasets to be summed
;   
;   resultname - The name of the resultant dataset. The user has the option to specify
;                this although a standard default is always set otherwise.
;
;   algorithm  - The algorithm to be used: simple sum or weighted average; default is 
;                simple sum
;
; RETURN VALUE:
;
pro DaveopDataCombine::GetProperty ,ndata=ndata,resultname=resultname,algorithm=algorithm $
                     , combVarName=combVarName, combVarLabel=combVarLabel, combVarUnits=combVarUnits $
                     ,_REF_EXTRA=etc
compile_opt idl2

;ndata
if (arg_present(ndata)) then ndata=self.ndata
;name of dataset to store result
if (arg_present(resultName)) then resultName=self.resultName
; Algorithm
if (arg_present(algorithm)) then algorithm=self.algorithm
if (arg_present(combVarName)) then combVarName=self.combVarName
if (arg_present(combVarLabel)) then combVarLabel=self.combVarLabel
if (arg_present(combVarUnits)) then combVarUnits=self.combVarUnits

; dataset* (dataset1, dataset2, etc). These are the names of the datasets that are to 
; be included in the sum
nEtc = n_elements(etc)
if (nEtc gt 0) then begin
   propNames = Self.propertyNames
   for i=0, nEtc-1 do begin
      index = where(etc[i] eq strupcase(propNames), found)
      if (found) then (Scope_Varfetch(etc[i],/REF_EXTRA)) = Self.combVarValue[index]
   endfor
endif

; also call base class accessor
if(nEtc gt 0) then $
  self->IDLitOperation::GetProperty, _EXTRA=etc

end


;===============================================================================
; DaveopDataCombine::SetProperty
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;
; KEYWORDS:
;   dataset*   - Eg dataset1, dataset2, ...
;                These are the names of the datasets that the user has selected to be
;                included in the sum. Because the number of datasets are not known a priori,
;                it would be very cumbersome to itemise these properties as keywords to this 
;                procedure. Instead they are received via the _EXTRA keyword and 
;                additional processing is applied with the use of the Scope_Varfetch() to
;                determine when these properties are being set. The actual values for the
;                properties are maintained in the class variables Self.propertyNames and
;                Self.datasetsIndex
;
;   ndata      - The number of datasets to be summed
;
;   nmax       - Maximum nos of datasets available (ie. that is present in data browser)
;
;   resultname - The name of the resultant dataset. The user has the option to specify
;                this although a standard default is always set otherwise.
;
;   algorithm  - The algorithm to be used: simple sum or weighted average; default is 
;                simple sum
;
; RETURN VALUE:
;
pro DaveopDataCombine::SetProperty,ndata=ndata,nmax=nmax,resultname=resultname,algorithm=algorithm $
                      , combVarName=combVarName, combVarLabel=combVarLabel, combVarUnits=combVarUnits $
                      ,_EXTRA=etc
compile_opt idl2

; Nmax
if (n_elements(nmax)) then self.nmax = nmax > 2 ; must be >=2

if (n_elements(combVarname)) then Self.combVarName = combVarName

if (n_elements(combVarLabel)) then Self.combVarLabel = combVarLabel

if (n_elements(combVarUnits)) then Self.combVarUnits = combVarUnits

; Ndata
if (n_elements(ndata)) then begin
    self.ndata= ndata
    for i=0,self.nmax-1 do begin
        hide = (i ge self.ndata)
        propName = Self.propertyNames[i]
        self->SetPropertyAttribute, propName, hide=hide
    endfor
endif

; Name of dataset to store results
if (n_elements(resultName)) then begin
    self.resultName=resultName
    self.namechanged = 1
endif

; Algorithm
if (n_elements(algorithm)) then self.algorithm=algorithm

; dataset* properties
nEtc = n_elements(etc)
if (nEtc gt 0) then begin
   propNames = Self.propertyNames
   etcTags = tag_names(etc)
   for i=0, nEtc-1 do begin
      index = where(etcTags[i] eq strupcase(propNames), found)
      ;if (found) then  Self.datasetsIndex[index] = etc.(i) ;Scope_Varfetch(etc.(i))
      if (found) then  Self.combVarValue[index] = etc.(i) ;Scope_Varfetch(etc.(i))
   endfor
endif

; Also Call base class mutator
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::SetProperty, _EXTRA=etc 

end


;===============================================================================
; DaveopDataCombine::RecordInitialValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case). Only included here to maintain the API
;                  of this method
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DaveopDataCombine::RecordInitialValues, oCmdSet, oTarget, idProp
compile_opt idl2

; create a command object to store useful info. The target for the
; command is the operation itself. For the sum operation, only
; one sum will be performed per command set object created! If
; this were not the case, then it would be better to make a dataset
; object as the target.
oCmd = obj_new('IDLitCommand',target_identifier=oTarget->getfullidentifier())
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, resultName=resultName
void = oCmd->AddItem('OLD_RESULTNAME',resultName)

; Add the command to command set
oCmdSet->Add, oCmd

return, 1

end


;===============================================================================
; DaveopDataCombine::RecordFinalValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case). Only included here to maintain the API
;                  of this method
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DaveopDataCombine::RecordFinalValues, oCmdSet, oTarget, idProp
compile_opt idl2

; Retrieve the first command object from the command set
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
oTarget->GetProperty, resultName=resultName;,algorithm=algorithm
void = oCmd->AddItem('NEW_RESULTNAME',resultName)
;void = oCmd->AddItem('NEW_ALGORITHM',algorithm)

return, 1

end


;===============================================================================
; DaveopDataCombine::DoExecuteUI
; 
; PURPOSE:
;   Launch the UI dialog to collect appropriate user information for
;   this operation.
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DaveopDataCombine::DoExecuteUI
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Use the build-in 'PropertySheet' UI service to let the user
; customize the operation's properties.
return, oTool->DoUIService('PropertySheet',self)

end


;===============================================================================
; DaveopDataCombine::UndoOperation
; 
; PURPOSE:
;   Provides the 'undo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DaveopDataCombine::UndoOperation, oCmdSet
compile_opt idl2

; Retrieve the command object.
oCmd = oCmdSet->Get(/all,count=nCmds)
if (nCmds ne 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Restore operation attributes
void = oCmd->GetItem('OLD_RESULTNAME',resultName)
self->SetProperty, resultName=resultName

; Undo previous sum by deleting the previously generated output from the Data Manager
void = oCmd->GetItem('FULLOUTPUTID',idDataset)
oDataset = oTool->GetByIdentifier(idDataset)
oDM = oTool->GetService("DAVEDATA_MANAGER")	
status = oDM->DeleteData(oDataset->GetFullIdentifier())

; Indicate that the tool has been modified and force it to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1

end


;===============================================================================
; DaveopDataCombine::RedoOperation
; 
; PURPOSE:
;   Provides the 'redo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DaveopDataCombine::RedoOperation, oCmdSet
compile_opt idl2

; Retrieve the first and only command object from the command set
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get the target (eventhough it should be self in this case) for the
; first command
oCmd->GetProperty, target_identifier=idCmdTarget
oCmdTarget = oTool->GetByIdentifier(idCmdTarget)

; Restore operation attributes
void = oCmd->GetItem('NEW_RESULTNAME',resultName)
self->SetProperty, resultName=resultName

; Restrieve davePtr and re-create a DaveDataset object from it
void = oCmd->GetItem('DAVEPTR',davePtr)
void = oCmd->GetItem('OUTPUTNAME',resName)
void = oCmd->GetItem('OUTPUTID',resID)

if (~ptr_valid(davePtr)) then return, 0
oRes = obj_new('DAVEDataset',davePtr=davePtr)
oRes->SetProperty, name=resName, identifier=resID
oTool->AddDatasetToDataManager, oRes ; And add it to the Data Manager

; Update 'OUTPUTREF' entry
void = oCmd->AddItem('FULLOUTPUTID',oRes->GetFullIdentifier(),/overwrite)

; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1

end


;===============================================================================
; DaveopDataCombine::ResetOpAttributes
; 
; PURPOSE:
;   Reset attributes for this operation to reflect the currently
;   selected datasets
;
; PARAMETERS:
;
; KEYWORDS:
;
;
function DaveopDataCombine::ResetOpAttributes, selDatasets, selNames, SelValues ;,combVarNames, combVarValues
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get the all datasets in Data Manager
oDatasets = oTool->GetDAVEDataManagerContents(count=nMax)
if (nMax lt 2) then return, 0

; Retrieve ids and names of datasets
nSel = n_elements(selDatasets)
self->GetProperty, types=validTypes ; should be ['DAVE1DATASET','ASCIISPE','ASCIICOL','ASCIIGRP']
nValidTypes = n_elements(validTypes)
nValid = 0
selIndex = []
therestIndex = []
names = []
validDatasets = []
varValues = []
varName = Self.CombVarName
varValue = 0.0
For i=0,nmax-1 do begin
    if (~obj_valid(oDatasets[i])) then continue

    ;; Does dataset contain valid target data type?
    ;j = 0
    ;repeat begin
    ;    oRes = oDatasets[i]->GetByType(validTypes[j],count=found)
    ;endrep until (found || ++j ge nValidTypes)
    ;if (~Obj_valid(oRes)) then continue
    oRes = oDatasets[i]->GetByType(validTypes,count=found)
    if (found le 0) then continue


    ; Can't combine 2D data, selection must contain 1D data.    
    oDatasets[i]->GetProperty, name=name, nDimensions=nDim
    if (nDim eq 1) then begin
      if (obj_isa(oDatasets[i],'DaveDataset')) then begin
         void = oDatasets[i]->_GetDescriPtrStr(dscStr)
         varName = dscStr.name
         varValue = dscStr.qty
      endif ;else continue
      if (~strcmp(varName,Self.CombVarName,/fold)) then continue
      varValues = [varValues,varValue]
      names = [names,name]
      validDatasets = [validDatasets,oDatasets[i]]
      nValid++
      ; if current dataset is selected, record it in 'selIndex', otherwise record it in 'theRestIndex'
      index = where(strcmp(selNames,name),cnt)
      if (cnt eq 1) then selIndex = [selIndex,i] $
      else theRestIndex = [theRestIndex,i]
    endif
endfor

; rearrange so that selected datasets are at the beginning of list
if (n_elements(selIndex) gt 0 && n_elements(theRestIndex) gt 0) then begin
  index = [selIndex,theRestIndex]
  validDatasets = validDatasets[index]
  names = names[index]
  varValues = varValues[index]
endif

nValid = n_elements(validDatasets)
if (nValid lt 2) then return, 0

; number of valid datasets should not exceed hardnmax
if (nValid gt Self.hardNmax) then begin
   msg = "Maximum datasets that can be combined is "+string(Self.hardNmax,format='(I-3)')
   msg = [msg,'Only the first '+ string(Self.hardNmax,format='(I-3)') + ' datasets will be used']
;   msg = [msg,'before launching the "Combine" operation']
   Self->ErrorMessage, msg, severity=0
;   return, 0
endif

self->SetProperty, nmax = (nValid < Self.hardNmax); the nos of datasets with valid targets
if (ptr_valid(self.validDatasetsPtr)) then ptr_free, self.validDatasetsPtr
self.validDatasetsPtr = ptr_new(validDatasets) ; the object refs of valid datasets found from the datamanager


; Set -
; ndata to number of selected dataset
; range of ndata to between 2 and nos of valid datasets
; a default name to be used to store the result
self->SetPropertyattribute, 'ndata', valid_range=[2,self.nmax]
self->SetProperty, ndata=self.selCount

guess1 = self.resultName
guess2 = IDLitGetUniqueName(names, guess1) 
if (guess2 ne guess1) then begin
    pos = strpos(guess2,'_',/reverse_search)
    if (pos ne -1) then begin
        guess1  = strmid(guess2,0,pos) + ' '
        len = strlen(guess2) - (pos+1)
        guess1 += strmid(guess2,pos+1,len)
    endif
endif

self.namechanged = 0            ; indicate using default result name
self.resultName = guess2        ; may contain an underscore
self.resultID = guess1          ; the underscore replaced by space

; the variable value for the selected datasets
; if the values are not unique, it implies they are not set. In this case, replace them with indices
index = uniq(varValues[selIndex],sort(varValues[selIndex])) ; for selected dataset (not all loaded data)
nmax = Self.nmax
if (n_elements(index) ne nSel) then begin
  ; simple use index as the combining vary
  Self.combVarName = 'yaxis'
  Self.combVarLabel = 'Index Nos'
  Self.combVarUnits = ''
  varValues = findgen(nmax) + 1
endif 
combVarValue = Self.combVarValue
combVarValue[0:nmax-1] = varValues
Self.combVarValue = combVarValue

; Assign the valid datasets (upto nmax) to the dataset* (registered) properties of the operation
if (ptr_valid(self.namesPtr)) then ptr_free, self.namesPtr
self.namesPtr = ptr_new(names)  ; the string names associated with the valid datasets found in the DM
for i = 0, nmax-1 do begin
    propName = Self.propertyNames[i]
    Self->SetPropertyAttribute, propName, name=names[i], hide= (i ge nSel)
    Self->SetPropertyByIdentifier, propName, varValues[i]
;    index = i
;    if (i lt self.selCount) then begin
;        ;; assign selected datasets as defaults
;        index = where(names eq selNames[i], cnt)
;        index = (cnt eq 1)? index[0] : i
;    endif
;self.datasetsIndex[i] = index
endfor

return, 1

end

;===============================================================================
; DaveopDataCombine::DoAction
; 
; PURPOSE:
;   Implements the main function for this operation. Sum the specified datasets 
;   and store the results in a new dataset stored in the Data Browser.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an IDLitCommandSet object
;    If unsuccessful, a NULL object.
;
function DaveopDataCombine::DoAction, oTool
compile_opt idl2

; oTool should be valid and the DAVE Main Tool
if (~obj_valid(oTool) || ~obj_isa(oTool,'DAVETOOL')) then return, obj_new()

; Get the selected dataset(s)
oSelections = oTool->GetSelectedData()
void = where(obj_valid(oSelections),cnt)
if (cnt lt 2) then begin
    oTool->StatusMessage, 'Please, select at least two top-level datasets from the Data Browser tree.'
    return, obj_new()
endif

; Locate datasets containing experiment data
; Only top-level entries in the Data Manager Browser can be valid datasets
self->GetProperty, types=validTypes ; should be ['DAVE1DATASET','ASCIISPE','ASCIICOL','ASCIIGRP']
nValid = n_elements(validTypes)
validName = 'DAVE Data Manager'
first = 1
oFinalSel = []
selNames = []
SelValues = []
for i = 0,n_elements(oSelections)-1 do begin
    ;; Only top-level dataset selections from the data browser can be processed
    ;; ie the this operation is design to deal with complete datasets
    oTop = getmaindataobjectparent(oSelections[i])
    if (~obj_valid(oTop)) then continue
    if (oTop ne oSelections[i]) then continue ; ie selection must be a main entry in the tree

    ;; Search for one of the valid data types from this selection.
    ;; Essentially, the valid types refer to the experimental or plottable data containing
    ;; an independent, dependent and error component!
    j = 0
    repeat begin
        oRes = oSelections[i]->GetByType(validTypes[j],count=found)
    endrep until (found || ++j ge nValid)


    if (~obj_valid(oRes)) then continue
    
    ; Can't combine 2D data, selection must contain 1D data.    
    oSelections[i]->GetProperty, name=selName, nDimensions=nDim
    if (nDim eq 1) then begin
      varName = Self.CombVarName
      varValue = 0.0
      varUnits = ''
      varLabel = ''
      if (obj_isa(oSelections[i],'DaveDataset')) then begin
         void = oSelections[i]->_GetDescriPtrStr(dscStr)
         varName = dscStr.name
         varValue = dscStr.qty
         varUnits = dscStr.units
         varLabel = dscStr.legend
      endif
      if (first eq 1) then begin
         combVarName = varname
         combVarUnits = varUnits
         combVarLabel = varLabel
      endif
      if (~strcmp(varName,combVarName,/fold)) then continue

      first = 0
      oFinalSel = [oFinalSel,oSelections[i]]
      selNames = [selNames,selName]
      selValues = [selValues,varValue]
    endif
endfor

if (n_elements(selNames) eq 0) then begin
    oTool->StatusMessage, 'Cannot combine selected datasets!'
    return, obj_new()
endif

Self.selCount = n_elements(oFinalSel)
Self.combVarName = combVarName
Self.combVarUnits = combVarUnits
Self.combVarLabel = combVarLabel

; Create a command set obj by calling the base class DoAction
oCmdSet = self->IDLitOperation::DoAction(oTool)

;; Set operation properties based on the selected data
if (~self->ResetOpAttributes(oFinalSel, selNames, selValues)) then begin
    obj_destroy, oCmdSet
    return, obj_new()
endif

; Is some UI needed prior to execution?
self->GetProperty, show_execution_ui=doUI
;hasPropSet = 0b
if (doUI) then begin
    ;; Record initial properties for this operation.
    if (~Self->RecordInitialValues(oCmdSet,self,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; Perform our UI.
;    if (~Self->DoExecuteUI()) then begin
;        obj_destroy, oCmdSet
;        return, obj_new()
;    endif
;
    yValuesNotUnique = 1
    while (yValuesNotUnique) do begin
      ; repeat until the values for the new axis entered by the user are unique
      if (~Self->DoExecuteUI()) then begin
        obj_destroy, oCmdSet
        return, obj_new()        
      endif
      nData = Self.nData
      ;oDatasets = (*Self.validDatasetsPtr)[0:nData-1]
      ;names = (*Self.namesPtr)[0:nData-1]
      varValues = (Self.combVarValue)[0:nData-1]

      index = uniq(varValues, sort(varValues))
      yValuesNotUnique = n_elements(index) ne n_elements(varValues)
      if (yValuesNotUnique) then begin
        msg = ["Please review and fix the duplcate entries ","in the new axis values for the datasets to be combined!"]
        oTool->ErrorMessage, msg, severity=1
      endif
    endwhile

    if (~Self->RecordFinalValues(oCmdSet,self,'')) then begin
      obj_destroy, oCmdSet
      return, obj_new()      
    endif
    ;; The hourglass will have been cleared by the dialog.
    ;; So turn it back on.
    ;void = oTool->DoUIService("HourGlassCursor", self)
endif

;; Perform sum and record changes in undo/redo buffer
if (~Self->Combine(oCmdSet)) then begin
    obj_destroy, oCmdSet
    return, obj_new()
endif

; return the command set obj
return, oCmdSet

end


;===============================================================================
; DaveopDataCombine::Combine
; 
; PURPOSE:
;   Perform difference b/n two datasets and record changes in the
;   operation's undo/redo buffer.
;
; PARAMETERS:
;   oCmdSet [in|out] - an IDLitCommandSet object which stores the
;                      undo/redo buffer info.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, 1
;    If unsuccessful, 0
;
function DaveopDataCombine::Combine, oCmdSet
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Retrieve the first command object from the command set
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Operation settings
;resName = IDLitGetUniqueName((*self.namesPtr),self.resultName)
resName = Self.resultName
resID = Self.resultID
nData = Self.nData
oDatasets = (*Self.validDatasetsPtr)[0:nData-1]
names = (*Self.namesPtr)[0:nData-1]
varValues = (Self.combVarValue)[0:nData-1]

; sort wrt to varValues
index = sort(varValues)
varValues = varValues[index]
names = names[index]
oDatasets = oDatasets[index]

oDatasets[0]->GetProperty, axis1Value=xv
nx = n_elements(xv)
ind1 = indgen(nx-1) + 1
ptol = 0.001*abs(mean(xv[ind1] - xv[ind1-1], /double)) ; 0.1% of the mean of the difference
mtol = -1.0*ptol                        ;-0.001

; Determine the agregate first ind axis range that overlaps all datasets
for i = 0,nData-1 do begin
  oDataset = oDatasets[i]
  if (~obj_valid(oDataset)) then continue
  oDataset->GetProperty, axis1Value=xv, axis1Distribution=xType
  xv =  xv[sort(xv)]  ; ensure x is in ascending order
  if (strcmp(xType,'HIST',4,/fold)) then begin  ; if histogram, convert to point mode
    n = n_elements(xv)
    xv = 0.5*(xv[0:n-2]+xv[1:n-1])
  endif
  if (i eq 0) then x = xv
  nx = n_elements(x)
  index1 = where((xv-x[0]) lt mtol,cnt1) ;where(xv lt x[0], cnt1)
  index2 = where((xv-x[nx-1]) gt ptol,cnt2) ;where(xv gt x[nx-1], cnt2)
  if (cnt1 gt 0) then x = [xv[index1],x]
  if (cnt2 gt 0) then x = [x,xv[index2]]
endfor

; Now put the dataset together
nx = n_elements(x)
ny = nData
newTrmt = ['==================================================' $
  ,'Timestamp: '+systime()]
newTrmt = [newTrmt,'Sum operation performed']
idDatasets = strarr(nData)

for i = 0,ny-1 do begin
  oDataset = oDatasets[i]
  if (~obj_valid(oDataset)) then continue
  idDatasets[i] = oDataset->GetFullIdentifier()
  oDataset->GetProperty, dataRef=oDat, errorRef=oErr, axis1Ref=oX   $
    ,axis1Value=xnew, axis1Distribution=xDist, dataValue=znew, errorValue=dznew, name=nameTag
  if (~obj_valid(oDat) || ~obj_valid(oErr) || ~obj_valid(oX)) then begin
    msg = 'Error retrieving data from '+nameTag+'!'
    Self->ErrorMessage, msg, severity = 2
    return, 0
  endif
  xSortIndex = sort(xnew) ; ensure xnew is in ascending order
  xnew =  xnew[xSortIndex]
  nxnew = n_elements(xnew)
  znew = znew[xSortIndex]
  dznew = dznew[xSortIndex]
  sfnew = znew*0.0 + 1.0
  if i eq 0 then begin
    z = dblarr(nx,ny)
    dz = dblarr(nx,ny)
    y = dblarr(ny)
    ;mon = fltarr(nx,ny)
    sf = fltarr(nx,ny) + 1.0   ;<-- records where data consists of overlapping data points
    ;monScale = mnew[0]
  endif
  y[i] = varValues[i]
  
  void = where(abs(xnew-x) gt ptol,notTheSame)
  if (notTheSame eq 0) then notTheSame = nx ne nxnew
  if (notTheSame gt 0) then begin ; interpolate
    x0_le_xnew0 = ((x[0] - xnew[0]) lt mtol) || (x[0] eq xnew[0])
    xn_ge_xnewn = ((x[nx-1] - xnew[nxnew-1]) gt ptol) || (x[nx - 1] eq xnew[nxnew - 1])
    if (x0_le_xnew0 or xn_ge_xnewn) then begin
      ;; avoid any extrapolation by running interpol() only for
      ;; values of x that lie within xnew
      x_ge_xnew0 = ((x - xnew[0]) gt mtol) or ((x - xnew[0]) eq 0)
      x_le_xnewn = ((x - xnew[nxnew-1]) lt ptol) or ((x - xnew[nxnew-1]) eq 0)
      inIndex = where(x_ge_xnew0 and x_le_xnewn, inCnt $
        ,complement=outIndex,ncomplement=outCnt)
      if (inCnt gt 0) then begin
        xValid = x[inIndex]
        zinterp = interpol(znew,xnew,xValid)
        dzinterp = interpol(dznew,xnew,xValid)
        ;minterp = interpol(mnew,xnew,xValid)
        z[inIndex,i] = zinterp
        dz[inIndex,i] = dzinterp
        ;mon[inIndex,i] = minterp
      endif
      if (outCnt gt 0) then begin
        ; value is undefined so use a NaN
        z[outIndex,i] = !values.D_NAN
        dz[outIndex,i] = !values.D_NAN
        ;mon[outIndex,i] = 0.0 ;monScale
        sf[outIndex,i] = 0.0
      endif
    endif
  endif else begin
    z[*,i] = znew
    dz[*,i] = dznew
    ;mon[*,i] = mnew
  endelse  
   newTrmt = [newTrmt,nameTag]
endfor

;; Rescale monitor to that of the first data point of the first dataset
;mon = mon/monScale

dims = size(z,/dimensions)
n_x = dims[0]
n_y = dims[1]
i = 0
while (i lt n_y-1) do begin
  index = i
  for j = i, n_y-2 do begin
    diff = abs(y[i] - y[j+1])
    if (diff le ptol) then begin
      index = [index,j+1]
    endif else begin
      break
    endelse
  endfor
  i = j + 1
  ymean = (moment(y[index],maxmoment=1))[0]
  nIndex = n_elements(index)
  if (nIndex gt 1) then begin
    c1 = total(z[*,index],2,/NAN)
    e1 = sqrt(total(dz[*,index]^2,2,/NAN))
    ;m1 = total(mon[*,index], 2, /NAN)
    s1 = total(sf[*,index],2, /NAN)
  endif else begin
    c1 = z[*,index]
    e1 = dz[*,index]
    ;m1 = mon[*,index]
    s1 = sf[*,index]
  endelse
  if (size(c1,/n_dimensions) eq 1) then begin
    c1 = reform(c1,n_x,1)
    e1 = reform(e1,n_x,1)
  endif
  if (n_elements(newy) eq 0) then begin
    newy = ymean
    newc = c1
    newe = e1
    ;newm = m1
    news = s1
  endif else begin
    newy = [newy,ymean]
    newc = transpose([transpose(newc),transpose(c1)])
    newe = transpose([transpose(newe),transpose(e1)])
    ;newm = transpose([transpose(newm),transpose(m1)])
    news = transpose([transpose(news),transpose(s1)])
  endelse
endwhile
lastIndex = index[n_elements(index)-1]
if (lastIndex eq n_y-2) then begin
  ; the last group was not processed so include it.
  newy = [newy,y[n_y-1]]
  c1 = reform(z[*,n_y-1],n_x,1)
  e1 = reform(dz[*,n_y-1],n_x,1)
  ;m1 = reform(mon[*,n_y-1],n_x,1)
  s1 = reform(sf[*,n_y-1],n_x,1)
  newc = transpose([transpose(newc),transpose(c1)])
  newe = transpose([transpose(newe),transpose(e1)])
  ;newm = transpose([transpose(newm),transpose(m1)])
  news = transpose([transpose(news),transpose(s1)])
endif
y = newy
z = newc
dz = newe
;mon = newm
sf = news

index = where(sf le 0.0, cnt)
if (cnt gt 0) then sf[index] = 1.0
z = z/sf
dz = dz/sf

newTrmt = [newTrmt,'_________________________________________' $
  ,'The files listed above were included']

idDatasets = strarr(nData)

; Store useful details
void = oCmd->AddItem('IDDATASETS',idDatasets)
void = oCmd->AddItem('DATASETNAMES',names)
void = oCmd->AddItem('DATASETVALUES',varValues)

resID = (self.namechanged)? resName : self.resultID
void = oCmd->AddItem('OUTPUTNAME',resName)
void = oCmd->AddItem('OUTPUTID',resID)

; retrieve various basic info from first dataset
status = oDataset[0]->ToDavePtr(davePtr)
oDataset[0]->GetProperty, treatment=trmt
; Create a davePtr to hold the combined results
status = modify_dave_pointer(davePtr $
  ,instrument = inst   $
  ,qty = z $
;  ,qtunits = qtyUnits $
;  ,qtlabel = qtyLabel $
  ,err = dz $
  ,xvals = x $
  ,xtype = 'POINTS' $
;  ,xunits = xunits $
;  ,xlabel = xlabel $
  ,yvals = y $
  ,ytype = 'POINTS' $
  ,yunits = Self.combVarUnits $
  ,ylabel = Self.combVarLabel $
;  ,specificstr = specificstr $
  ,treatment = [trmt,newTrmt] $
;  ,dname = dptrName $
;  ,dunits = dptrUnits $
;  ,dlegend = dptrDesc $
;  ,dqty = dptrQty $
;  ,derr = dptrErr $
  ,ermsg = errMsg $
)

if (~status) then begin
  oTool->StatusMessage, errMsg
  return, 0
endif

; Create a DaveDataset object
oRes = obj_new('DAVEDataset',davePtr=davePtr)
oRes->SetProperty, name=resName, identifier=resID
oTool->AddDatasetToDataManager, oRes ; And add it to the Data Manager

void = oCmd->AddItem('FULLOUTPUTID',oRes->GetFullIdentifier())
void = oCmd->AddItem('DAVEPTR',davePtr)


; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1
end


;===============================================================================
; DaveopDataCombine::Cleanup
; 
; PURPOSE:
;   DaveopDataCombine class cleanup
;
pro DaveopDataCombine::Cleanup
compile_opt idl2

if (ptr_valid(self.validDatasetsPtr)) then ptr_free, self.validDatasetsPtr
if (ptr_valid(self.namesPtr)) then ptr_free, self.namesPtr

; call base class cleanup
self->IDLitOperation::Cleanup

end
;-------------------------------------------------------------------------------


;===============================================================================
; DaveopDataCombine::Init
; 
; PURPOSE:
;   Initialize an object of this class
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - if successful
;    0 - otherwise
;
function DaveopDataCombine::Init, _REF_EXTRA=etc
compile_opt idl2

; call superclass init
; This operation is reversible
; This operation is not expensive
if (~self->IDLitOperation::Init(NAME='Data Sum' $
                                ,reversible_operation=1,expensive_computation=1 $
                                ,_EXTRA=etc)) then return, 0
;types=['DAVE1DATASET','ASCIISPE','ASCIICOL','ASCIIGRP']

; Unhide the SHOW_EXECUTION_UI property
self->SetPropertyAttribute, 'SHOW_EXECUTION_UI', hide=0

; Register properties for this operation
;self.algorithms = ['SIMPLE SUM','WEIGHTED AVERAGE']
;self.algorithm = 0
;self->RegisterProperty, 'algorithm', description='The algorithm to be used for sum' $
;  ,name='Algorithm to use',sensitive=1,enumlist=self.algorithms

Self.RegisterProperty, 'CombVarName', /string, description='Name of  combining variable to create' $
   ,name='Combining Variable Name (keyword)'

Self.RegisterProperty, 'CombVarLabel', /string, description='Label to be used with combining variable to create' $
   ,name='Combining Variable Label'

Self.RegisterProperty, 'CombVarUnit', /string, description='Units of  combining variable to create' $
   ,name='Combining Variable Units'

self->RegisterProperty, 'resultName',/string, description='Name of dataset to store result' $
   ,name='Output dataset name',sensitive=1
self->RegisterProperty, 'ndata', /integer, description='Nos of datasets to combine' $
   ,name='Nos of datasets to combine',sensitive=1,valid_range=[2,2]

Self.hardNmax = 99
for i = 0,Self.hardNmax-1 do begin
    self.datasetsIndex[i] = i
    NicePropName = 'Dataset '+strtrim(string(i+1),2)
    prop = 'dataset'+strtrim(string(i+1, format='(I02)'),2)
    self->RegisterProperty, prop, /float, description=NicePropName $
      ,name=NicePropName, sensitive=1,hide=1
    Self.propertyNames[i] = prop
endfor

self.resultName = 'combineddata'
self.combVarName = 'Temp'
self.combVarUnits = 'K'
; return success
return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; DaveopDataCombine__define
; 
; PURPOSE:
;   DaveopDataCombine class structure definition
;
pro DaveopDataCombine__define

compile_opt idl2

struc = {DaveopDataCombine $
         ,inherits IDLitOperation $
         ,hardNmax:99 $
         ,nmax:0B $             ; maximum nos of datasets available (present in data browser)
         ,datasetsIndex:bindgen(99) $ ; indices (of valid datasets) to be included in sum. 
                                             ;An absolute limit of 99 is imposed! Therefore nmax must be <= 99
         ,propertyNames:strarr(99) $   ; property names created dynamically to represent datasets
         ,combVarValue:FLTARR(99) $    ; combining variable values for datasets
         ,combVarName:'' $              ; combining variable name eg Temp
         ,combVarLabel:'' $             ; combining variable name eg 'Sample Temperature'
         ,combVarUnits:'' $             ; combining variable units eg K
         ,namechanged:0B $       ; use to indicate when resultName field has been modified
         ,resultName:'' $       ; name of dataset to store the sum
         ,resultID:'' $         ; base id of dataset to store the sum
;         ,algorithms:['',''] $  ; available algorithms to perform sum
;         ,algorithm:0B  $       ; the one used
         ,ndata:0B $            ; nos of datasets to be included in sum
         ,selCount:0B $         ; nos of datasets currently selected
         ,validDatasetsPtr:ptr_new() $ ; ptr to ids of valid datasets
         ,namesPtr:ptr_new() $  ; ptr to names of valid datasets
        }

end
