; $Id$
;
;===============================================================================
;+
; NAME:
;   Gen_nk
;
; PURPOSE:
;   Generate the single particle momentum distribution function for 
;   liquid/solid Helium 4.
;   
;   The model momentum distribution can be represented as
;     n(k) = n0[1 + f(k)] + n*(k)
;   where n0 is the condensate fraction
;         f(k) is the condensate induced peaking term
;         n*(k) represents the momentum distribution of the uncondensed atoms
;         k is the 3-D atomic momentom
;   
;   f(k) is given by
;     f(k) = B/k*Coth(D*k)*Exp(-k^2/kc) where
;   where B, D and kc are constants, kc is cut-off value of k (usually 0.5) included so
;   that the term Exp(-k^2/kc) restricts f(k) to small values of k only.
;   
;   However, n*(k) is determined as a Fourier Transform of n*(s), in which
;     n(s) = n0[1+f(s)] + A*n*(s)
;     s = distance travelled by the struck particle.
;   with the terms having the same meaning as in the k-space representation above.
;   s and k are FT conjugates.
;   - f(s) is implemented as Fofs(s) in another file
;   - n*(s) is approximated by the cumulant expansion as
;     n*(s) = exp[-a2/2!*s^2 + a4/4!*s^4 - a6/6!*s^6] where a2, a4 and a6 are parameters
;
; PARAMETERS
;   k (in) - the input momentum or k values where the distribution is to be evaluated.
;                  k can be either scaler or a vector.
;
; KEYWORDS:
;   outFile (in) - a string specifying an output file to contain the results
;   
;   aValues (in) - a vector containing the a parameters of the model. Default values
;                  are [0.897,0.46,0.38] for a2, a4 and a6.
;   
;   kc (in)      - a scaler specifying the value of kc in inverse angstroms - default is 0.5
;   
;   T (in)    - the Terature in K. The default is 1 K.
;   
;   n0 (in)      - the value of the Bose Condensate as a fraction. Default is 0.07
;
; RETURNS the value of the momentum distribution evaluated at the specified k.
;
; CATEGORY:
;   Quantum Fluids Analysis
;-
;===============================================================================
function Gen_nk, k, outFile=outFile, aValues=aValues, T=T, kc=kc, n0=n0, mass=mass
compile_opt idl2
if (n_elements(k) eq 0) then return, 0.0

; Setup default values for various parameters
if (n_elements(T) eq 0) then T = 1.0
if (n_elements(kc) eq 0) then kc=0.5
if (n_elements(aValues) ne 3) then begin
  aValues = [0.897,0.46,0.38]
endif
muPtr = ptr_new({a2:aValues[0],a4:aValues[1],a6:aValues[2]})
if (n_elements(n0) eq 0) then n0=0.07
if (n_elements(B) eq 0 || n_elements(D) eq 0 || n_elements(F) eq 0) then begin
  if (~isa(mass)) then mass = 4.0026           ; rel a. mass of He4
  hbar = 1.0545919        ; *10**(-34) Js
  ec   = 1.6021917        ; *10**(-19) C
  amu  = 1.660531         ; *10**(-27) Kg
  kB   = 1.380622         ; *10**(-23) J/K
  rho  = 0.6022169/27.0   ; 1/A**3         He4 at SVP
  Vs   = 240              ; m/s            He4 at SVP
  B = mass*amu*Vs/(2.0*hbar*(2.0*!pi)^3*rho)/1000.0  ; A**2
  D = hbar*Vs/(2.0*kb*T)/10.0                    ; A
  F = 2.0*kc^2                                     ; (1/A)**2
endif


I0 = Fofs(0.0, nOrder=1, kc=kc, T=T, B=B, D=D, F=F)  ; 1st moment of fofs i.e. when nOrder=1
I2 = Fofs(0.0, nOrder=3, kc=kc, T=T, B=B, D=D, F=F)  ; 2nd moment of fofs
A = 1.0 - n0*(1 + I0)

etc = create_struct('k',k[0],'muPtr',muPtr)

n = n_elements(k)
amp = A/(2*!pi*!pi)
if (n eq 1) then begin
  nk = amp * qpint1d('Nstarks',0.0D,!values.d_infinity,functargs=etc,status=status,nfev=nfev)
endif else begin
  nk = dblarr(n)
  for i=0,n-1 do begin
    etc.k = k[i]
    nk[i] = amp * qpint1d('Nstarks',0.0D,!values.d_infinity,functargs=etc,status=status,nfev=nfev)
  endfor
endelse

;
if (n_elements(outFile) gt 0) then begin
  openw, lun, outfile, /get_lun
  printf, lun, k, nk, 0.05*nk, format='(3(G11.4,1X))'
  close, lun, /force
endif

return, nk

end


;##############################################################################
pro Test_gen_nk, k=k, aValues=aValues, T=T, kc=kc, n0=n0, clac=calc, plot=plot
compile_opt idl2

if (n_elements(T) eq 0) then T = 1.0
if (n_elements(kc) eq 0) then kc=0.5
if (n_elements(n0) eq 0) then n0=0.07
if (n_elements(aValues) ne 3) then begin
  aValues = [0.897,0.46,0.38]
endif
if (n_elements(k) eq 0) then k = dindgen(41)*0.1
  
calc = Gen_nk(k, aValues=aValues, T=T, kc=kc, n0=n0)
str = 'n$_0$ = '+string(n0,format='(F4.2)')
str += ', $\alpha_2$ = '+string(aValues[0],format='(F5.3)')
str += ', $\alpha_4$ = '+string(aValues[1],format='(F5.3)')
str += ', $\alpha_6$ = '+string(aValues[2],format='(F5.3)')

if keyword_set(plot) then begin
  p = plot(k, calc, title=['Liquid $^4$He','Uncondensed Momentum Distribution',str])
;  prompt='Press enter to exit: '
;  answer=''
;  read,answer,prompt=prompt
;  p.Close
endif
  
end