; $Id$
;###############################################################################
;
; NAME:
;  PAN_ThreeFoldTunneling_2D
;
; PURPOSE:
;  Implementation of the 3-fold rotational tunneling model for use in fitting
;  data in PAN.
;
; CATEGORY:
;  DAVE, Data Analysis, surface fitting
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
;  Robert M. Dimeo, Ph.D.
;  NIST Center for Neutron Research
;  Phone: (301) 975-8135
;  E-mail: robert.dimeo@nist.gov
;  http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function PAN_ThreeFoldTunneling_2D,x,parms,         $
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

nx = isa(x)? n_elements(x) : n_elements(xvals)
nq = n_elements(qvals)
ux = 1+bytarr(nx)
uq = 1+bytarr(nq)

canDraw = 0
twoDimFlag = 1
resolutionRequiredFlag = 1   ; does function require a resolution function to be defined 0=>No, 1=>Yes

extConvolFlag = 0            ; does function require external convolution? 0=>No, 1=>Yes
                             ; if set to No, it implies a convolution is performed within this function when a resolution function is present
                             ; Also set to No, if you DO NOT wish any external convolution to be performed at all

if (n_params() eq 0) or (n_params() eq 1) then begin
  parmnames = ['Radius','Tunnel energy',$
    'Intensity'+Strtrim(String(Indgen(nq)),2), $
    'Center'+Strtrim(String(Indgen(nq)),2)]
   single_parmnames = parmnames[0:1]
  multi_parmnames = ['Intensity','Center']


	if arg_present(initParms) then begin
	  intensity = Max(yVals,dimension = 1,index)
	  index = index - Indgen(nq)*nx ; index contains the 1D subscripts so need to reform
	  center = xvals[index]

    ; assuming there are 3 peaks present, estimate the peak pos for these
    ; at each Q and then record the the one at the highest energy. 
    ; Et is taken as mean of these
    npeaks = 3
    highest_peak = []
    for i = 0,nq-1 do begin
      xpeaks = Get_peak_pos(xVals,yVals[*,i],npeaks)
      highest_peak = [highest_peak,max(xpeaks)]
    endfor
    Et = mean(highest_peak)
    
    radius = 1.0      ; Ang units
    
		initParms = [radius,Et,intensity,center]
	endif


	return,-1
endif

if (~Isa(resPtr,'Pointer')) then Return, Fltarr(nx,nq)*0.0 ; can't proceed without a pointer to the resolution function
yout = Fltarr(nx,nq)
yres = (*resPtr).data
if (yres.ndim ne 2) then Return, yout   ; bail if resolution is not 2D

radius = abs(parms[0])
Et = (parms[1])

index = indgen(nq) + 2
intensity = ux#(parms[index])
center = parms[index+nq] 

eisf = (5.0+4.0*sph_bessel(qVals*radius*sqrt(3.0),0))/9.0

p_delta3 = (p_delta2=(p_delta1 = Fltarr(2,nq)))
p_delta1[0,*] = eisf
p_delta1[1,*] = center

p_delta2[0,*] = 0.5*(1 - eisf)
p_delta2[1,*] = center + Et

p_delta3[0,*] = 0.5*(1 - eisf)
p_delta3[1,*] = center - Et


yout = intensity* (Pan_convolute(x,resPtr,pseudo_delta = p_delta1) + $
                   Pan_convolute(x,resPtr,pseudo_delta = p_delta2) + $
                   Pan_convolute(x,resPtr,pseudo_delta = p_delta3) )
return,yout

end
