; $Id$
;###############################################################################
;
; NAME:
;  PAN_IsotropicRotDiffusion_2D
;
; PURPOSE:
;  Implementation of Isotropic Rotational Diffusion model for use in fitting
;  data in PAN.
;
; CATEGORY:
;  DAVE, Data Analysis, surface fitting
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
;  Robert M. Dimeo, Ph.D.
;  NIST Center for Neutron Research
;  Phone: (301) 975-8135
;  E-mail: robert.dimeo@nist.gov
;  http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function PAN_IsotropicRotDiffusion_2D,x,parms,         $
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

nx = isa(x)? n_elements(x) : n_elements(xvals)
nq = n_elements(qvals)
ux = 1+bytarr(nx)
uq = 1+bytarr(nq)

canDraw = 0
twoDimFlag = 1
resolutionRequiredFlag = 1   ; does function require a resolution function to be defined 0=>No, 1=>Yes

extConvolFlag = 0            ; does function require external convolution? 0=>No, 1=>Yes
                             ; if set to No, it implies a convolution is performed within this function when a resolution function is present
                             ; Also set to No, if you DO NOT wish any external convolution to be performed at all

if (n_params() eq 0) or (n_params() eq 1) then begin
  ; Define hbar according to the energy units
  eUnits = xunits.Toupper()
  eUnitsKnown = eUnits.Equals('MEV') || eUnits.Equals('UEV')
  if (eUnitsKnown) then begin
    if (eUnits.Equals('UEV')) then hbar = 658.21        ; ueV.ps
    if (eUnits.Equals('MEV')) then hbar = 0.65821       ; meV.ns
  endif else begin
    ; Unknown energy units so let user specify
    msg = 'The unit of energy for the data could not be determined!'
    msg = [msg,'Is the unit of the energy axis in ueV or meV?',' ']
    msg = [msg,'Click Yes if energy unit is ueV',' ']
    msg = [msg,'Or No if energy unit is meV']
    title = 'Diffussion in A Sphere 2D -- Specify Energy Unit'
    if (N_elements(wTLB) eq 0) then wTLB=0L
    decision = Dialog_message(msg,title=title,/question,dialog_parent=wTLB)
    xunits = (decision.Equals('Yes'))? 'ueV' : 'meV'
    hbar = (decision.Equals('Yes'))? 658.21 : 0.65821
    eUnits = xunits.Toupper()
  endelse
  ;if strupcase(xunits) eq 'MEV' then hbar = hbar*1.d4 else hbar = hbar*1.d7
  if (Isa(func_dataHash,'hash')) then func_dataHash['hbar'] = hbar

  tUnits = eUnits.Equals('UEV')? 'ps' : 'ns'     ; time unit depends on energy unit of data

  parmnames = ['Radius (Ang)','Diffusivity (Ang^2/'+tUnits+')', $
    'Intensity'+Strtrim(String(Indgen(nq)),2), $
    'Center'+Strtrim(String(Indgen(nq)),2)]

  single_parmnames = parmnames[0:1]
  multi_parmnames = ['Intensity','Center']

	if arg_present(initParms) then begin
    intensity = max(yVals,dimension = 1,index)
    index = index - indgen(nq)*nx ; index contains the 1D subscripts so need to reform
    center = xvals[index]

    ; assuming a single main peak then estimate HWHM for the _largest_ Q value
    hm = 0.5*intensity[nq-1]
    res = Where(yVals[*,nq-1] ge hm, found)
    if (found gt 0) then begin
      Gamma = 0.5*xvals[res[found-1]] - xvals[res[0]]
    endif

		D = 0.5*Gamma/hbar                ; estimate D from Gamma; D in units of Ang^2/ps or Ang^2/ns
		radius = 1.0                      ; in Ang units
		initParms = [radius,D,intensity,center]
	endif
  return,-1
endif

if (~Isa(resPtr,'Pointer')) then Return, Fltarr(nx,nq)*0.0 ; can't proceed without a pointer to the resolution function
yout = Fltarr(nx,nq)
yres = (*resPtr).data
if (yres.ndim ne 2) then Return, yout   ; bail if resolution is not 2D

hbar = 658.0  ; ueV.ps; assumes energy units in microvolts! Should have been defined already
if (Isa(func_dataHash,'hash')) then begin
  hbar = func_dataHash['hbar']
endif

radius = parms[0]
D = parms[1]

index = indgen(nq) + 2
intensity = ux#(parms[index])
center = ux#(parms[index+nq])

; Now calculate the EISF and delta term
A0 = Sph_bessel(qVals*radius,0)^2
pseudo_delta = Fltarr(2,nq)             ; delta-function parameters
pseudo_delta[0,*] = A0                  ; delta area
pseudo_delta[1,*] = parms[index+nq]     ; delta position

; Now determine the second term which is an accumulated sum
; over i of the product of the Ai form factor and a Lorentzian
nterms = 40           ; truncate the sum to nterms=40
sumALor = dblarr(nx,nq)
xuq = x#uq
uxuq = ux#uq
for i = 1,nterms-1 do begin
	Ai = ux#( (2.0*i+1.0) * sph_bessel(qVals*radius,i)^2)
	tau_inv = i*(i+1.0)*D
	Gamma = hbar*tau_inv
	Li = Gamma*uxuq/(Gamma^2*uxuq+(xuq-center)^2)
	sumALor = sumALor + Ai*Li
endfor
sumALor = sumALor/!dpi

; Perform convolution with resolution
; both the first (delta) and 2nd terms are convoluted and summed
yout = intensity*Pan_convolute(x,resPtr,sumALor,pseudo_delta = pseudo_delta)

return,yout
end
