;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; NAME:
;  PAN_HydrogenRotRecoil2D
;
; PURPOSE:
;  PAN model function to determine hydrogen recoil scattering including 
;  rotational transitional terms for use in fitting data in the impulse 
;  approximation.
;
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; AUTHOR:
;  Timothy Prisk
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.


;+
; ARCS_ResolutionWidth_2D
;
; PURPOSE:
;   Evaluates the Gaussian instrumental resolution width (FWHM) on the
;   ARCS direct geometry time-of-flight spectrometer at SNS, given the
;   at any energy transfer given an incident energy.
;
;   The method used to estimate the width is the Gaussian approximation outlined in
;   appendix A of Prisk et al, Journal of Alloys and Compounds vol 818 pg 152832 (2020)
;   https://doi.org/10.1016/j.jallcom.2019.152832
;
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; PARAMETERS:
;   E:   the neutron energy transfer at which the resolution width is required
;        Default is 0.0 meV
;
; KEYWORDS:
;   Ei:  the incident neutron energy used.
;        Default is 500 meV.
;
; RETURN VALUE:
;   the Gaussian instrumental resolution width (FWHM)
;-
function Arcs_resolutionwidth_2d, E, Ei=Ei
  compile_opt idl2

  ; ARCS instrument parameters.
  if (N_elements(Ei) eq 0) then Ei = 500  ; Default incident energy is 500 meV
  m_neutron = 1.674e-5 / 1.602            ; mass of the neutron (meV)
  L0 = 11.61                              ; moderator to Fermi chopper distance (m)
  L1 = 1.99                               ; Fermi chopper to sample distance (m)
  L2 = 3.5                                ; sample to detector distance (m)
  W_DET = 0.0254                          ; detector width (m)
  DTM = 3.095e-6                          ; moderator pulse width (s)
  DTC = 3.19e-6                           ; chopper pulse width (s)

  if N_params() eq 0 then E = 0.0 ; default energy transfer to 0.0 if not specified

  ; fudge to ensure Et is never bigger than Ei otherwise the returned with is a NaN
  index = where(E ge Ei, found)  ; E is scalar or vector
  if (found gt 0) then E[index] = 0.999*Ei

  Ef = Ei - E
  vi = Sqrt(2*Ei/m_neutron) ; incident neutron speed
  vf = Sqrt(2*Ef/m_neutron) ; final neutron speed
  dtd = w_Det/vf            ; neutron tof within detector
  mod_term = (vi^3 / L0 + vf^3 * L1 / (L0 * L2)) * DTM
  chop_term = (vi^3 / L0 + vf^3 * (L0 + L1) / (L0 * L2)) * DTC
  det_term = (vf^3 / L2) * dtd
  Return, m_neutron * Sqrt(mod_term^2 + chop_term^2 + det_term^2)

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;+
; ARCSResWidth_2D
;
; PURPOSE:
;   Evaluates the instrumental resolution width at any energy transfer
;   These are semi-emperical values determined for the ARCS instrument 
;   and then fitted to a polynomial of degree 4
;   W = b0 + b1*Et + b2*Et^2 + b3*Et^3 + b4*Et^4; 
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; PARAMETERS:
;   eTransfer:  the neutron energy transfer at which the resolution width is required
;
; RETURN VALUE:
;   the instrumental resolution width
;-
function ARCSResWidth_2D, eTransfer, _Extra=extra
compile_opt idl2

if (n_elements(eTransfer) eq 0) then eTransfer=0.0

; parameters for ARCS instrument at Ei = 500 meV
b0 =  15.53747
b1 = -0.03454
b2 =  1.96388E-5
b3 = -2.38325E-8
b4 =  5.60666E-11

Return, b0 + b1*eTransfer + b2*eTransfer^2 + b3*eTransfer^3 + b4*eTransfer^4

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;+
; ARCSResWidth1
;
; PURPOSE:
;   Evaluates the instrumental resolution width at the peak centers of
;   the first 5 rotational peaks for para-hydrogen.
;   These are semi-emperical values determined for the ARCS instrument
;   and then fitted to a polynomial to obtain coeeficients that can be used 
;   to estimate the widths at other Q values for the first 5 transitions.
;   W = b0 + b1*Et + b2*Et^2 + b3*Et^3 + b4*Et^4;
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; PARAMETERS:
;   eTransfer:  the neutron energy transfer at which the resolution width is required
;
; RETURN VALUE:
;   the instrumental resolution width
;-
function Arcsreswidth1_2D, Qgroup, _Extra=extra
compile_opt idl2

if (N_elements(Qgroup) eq 0) then Qgroup=1.0

; Calculate resolution at peak center.
R1 = 15.08155D0 - 0.02166D0*Qgroup - 0.03074D0*Qgroup^2 - 4.12848D-4*Qgroup^3 + 3.12092D-5*Qgroup^4
R2 = 14.11106D0 - 0.02666D0*Qgroup - 0.02881D0*Qgroup^2 - 5.03550D-4*Qgroup^3 + 3.49062D-5*Qgroup^4
R3 = 12.72221D0 - 0.03764D0*Qgroup - 0.02517D0*Qgroup^2 - 7.01154D-4*Qgroup^3 + 4.26839D-5*Qgroup^4
R4 = 11.00586D0 - 0.06322D0*Qgroup - 0.01802D0*Qgroup^2 - 0.00116D0*Qgroup^3 + 5.99657D-5*Qgroup^4
R5 = 9.1162D0  - 0.12902D0*Qgroup - 0.00205D0*Qgroup^2 - 0.00233D0*Qgroup^3 + 1.02931D-4*Qgroup^4

Return, [R1,R2,R3,R4,R5]

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;+
; Pan_hydrogenRecoilAA
;
; PURPOSE:
;   Standard PAN built-in fit model function.
;   Hydrogen Recoil scattering function. :
;   - the Impulse Approximation is not valid
;   - the Additive Approach model for the momentum distribution is used
;   - the momentum distribution is a Gaussian + a small correction
;   - Final State Effects (FSE) taken into account through an additive first order correction to the scattering
;   - The relative peak intensities of the peaks modified by the rotational transitions are determined by the
;     Young-Kopel theory. The expressions for the intensities are those specified by eqns (20) to (24) in 
;     Dawidowski et al, PRB vol 73, Pg 144203 (2006) DOI: ht tps://doi.org/10.1103/PhysRevB.73.144203
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; PARAMETERS:
;
;
; RETURN VALUE:
;
;-
function Pan_HydrogenRecoilAAYK_2D,x,parms,$
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

compile_opt idl2

canDraw = 0
twoDimFlag = 1
resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

extConvolFlag = 0            ; does function require external convolution? 0=>No, 1=>Yes
; if set to No, it implies a convolution is performed within this function when a resolution function is present
; Also set to No, if you DO NOT wish any external convolution to be performed at all

nx = Isa(x)? N_elements(x) : N_elements(xvals)
nq = N_elements(qvals)
ux = 1+Bytarr(nx)
uq = 1+Bytarr(nq)

if N_params() eq 0 then begin
    
  ; parameter names
  parmnames = []
  aname = 'shift'
  multi_parmnames = ['shift','scaleFac']
  single_parmnames = ['KE [K]','delta','a3bar [1/(A^2 ps)]','a4bar [1/(A^2 ps^2)]','H2 Radius [A]']
  parmnames = single_parmnames
  for i=0,n_elements(multi_parmnames)-1 do begin
    name = multi_parmnames[i]
    names = name+Strtrim(String(Indgen(nq)),2)
    parmnames = [parmnames,names]
  endfor

  ; initialial parameter guesses since canDraw is 0
  initParms = [60.0,0.0,20.0,0.0,0.37] ; init values for the single params
  initParms = [initParms,Fltarr(nq),Fltarr(nq)+0.001]   ; include init for 'shift' and 'scaleFac' for all groups

  Return,-1
endif

; Parameter definition and initalization.
KE = parms[0]
delta = parms[1]        ; ratio of alpha4/alpha2^2
a3bar = parms[2]        ; 1/(A^2 ps)
a4bar = parms[3]        ; 1/(A^2 ps^2)
h2_radius = parms[4]    ; h2 radius in A
index = Indgen(nq)
shift = ux#parms[5+index]     ; group-dependent arb energy shift to correct for any error in incident energy
sf    = ux#parms[5+nq+index]   ; group-dependent arb scale factor to scale model to data

; Define constants.
hbarsquared = 4.18016 ; meV amu A^2
m  = 1.007825*2.0     ; amu
B0 = 7.3567           ; meV
D0 = 0.00569          ; meV
H0 = 6.4e-6           ; meV

hbar    = 6.35078D0         ; amu A^2/ps
hbar2   = 4.18016D0         ; meV amu A^2
m       = 1.007825D0*2.0D0  ; amu
B0      = 7.3567D0          ; meV
D0      = 0.00569D0         ; meV
H0      = 6.4D-6            ; meV
KpermeV = 11.60451812D0     ; NIST CODATA value.
scaleFac = 9.64853          ; 1 meV = 9.64853 amu A^2/ps^2

; Convert kinetic energy to peak width.
KE_meV = KE/KpermeV

; Calculate cumulants.
alpha2 = (2.0D0/3.0D0)*(m/hbarsquared)*(KE_meV)                         ; A^{-2}
alpha4 = delta*alpha2^2                                                 ; A^{-4}
mu2    = ux # (hbarsquared*(hbarsquared*qVals^2/m^2)*alpha2)            ; meV^2
mu3    = ux # ((hbar^5*qVals^2/m^2)*a3bar*(1.0D0/scaleFac)^3)           ; meV^3
mu4    = (hbar^6)*((qVals/m)^2)*a4bar*(1.0D0/scaleFac)^4                ;
mu4    = ux # (mu4 + (hbar^8)*((qVals/m)^4)*alpha4*(1.0D0/scaleFac)^4)  ; meV^4


; Determine peak positions.
Recoil = ux # (0.5D0*(hbarsquared/m)*qVals^2)
Rot1 = B0*1.0D0*(1.0D0 + 1.0D0) - D0*(1.0D0*(1.0D0 + 1.0D0))^2 + H0*(1.0D0*(1.0D0 + 1.0D0))^3
Rot2 = B0*2.0D0*(2.0D0 + 1.0D0) - D0*(2.0D0*(2.0D0 + 1.0D0))^2 + H0*(2.0D0*(2.0D0 + 1.0D0))^3
Rot3 = B0*3.0D0*(3.0D0 + 1.0D0) - D0*(3.0D0*(3.0D0 + 1.0D0))^2 + H0*(3.0D0*(3.0D0 + 1.0D0))^3
Rot4 = B0*4.0D0*(4.0D0 + 1.0D0) - D0*(4.0D0*(4.0D0 + 1.0D0))^2 + H0*(4.0D0*(4.0D0 + 1.0D0))^3
Rot5 = B0*5.0D0*(5.0D0 + 1.0D0) - D0*(5.0D0*(5.0D0 + 1.0D0))^2 + H0*(5.0D0*(5.0D0 + 1.0D0))^3
C1 = Recoil + Rot1 - shift
C2 = Recoil + Rot2 - shift
C3 = Recoil + Rot3 - shift
C4 = Recoil + Rot4 - shift
C5 = Recoil + Rot5 - shift

; Observed width is intrinsic width and res width added in quadrature .
StoW = Sqrt(8.0*Alog(2.0))
mu2_p1 = mu2 + (Arcs_resolutionwidth_2d(C1)/StoW)^2
mu2_p2 = mu2 + (Arcs_resolutionwidth_2d(C2)/StoW)^2
mu2_p3 = mu2 + (Arcs_resolutionwidth_2d(C3)/StoW)^2
mu2_p4 = mu2 + (Arcs_resolutionwidth_2d(C4)/StoW)^2
mu2_p5 = mu2 + (Arcs_resolutionwidth_2d(C5)/StoW)^2

; The model is a sum of five peaks.
; Consisting of a recoil peak (additive approach model with final state corrections included)
; The recoil peaks are then modulated by the the rotational transitions of molecular para-hydrogen


inv_sqrt2pi = 1.0D0/Sqrt(2.0D0*!PI)   ; Precompute Gaussian normalization constant.

; Calculate AA final state effects to order 1/Q.
xq = x # uq
IApeak1 = inv_sqrt2pi/Sqrt(mu2_p1)*Exp(-0.5D0*(xq - C1)^2/mu2_p1)
IApeak2 = inv_sqrt2pi/Sqrt(mu2_p2)*Exp(-0.5D0*(xq - C2)^2/mu2_p2)
IApeak3 = inv_sqrt2pi/Sqrt(mu2_p3)*Exp(-0.5D0*(xq - C3)^2/mu2_p3)
IApeak4 = inv_sqrt2pi/Sqrt(mu2_p4)*Exp(-0.5D0*(xq - C4)^2/mu2_p4)
IApeak5 = inv_sqrt2pi/Sqrt(mu2_p5)*Exp(-0.5D0*(xq - C5)^2/mu2_p5)

z1 = (xq - C1)
z2 = (xq - C2)
z3 = (xq - C3)
z4 = (xq - C4)
z5 = (xq - C5)

odd1 = -(0.5D0*mu3/mu2_p1^2)*z1*(1.0D0 - 0.5D0*z1^2/mu2_p1)
odd2 = -(0.5D0*mu3/mu2_p2^2)*z2*(1.0D0 - 0.5D0*z2^2/mu2_p2)
odd3 = -(0.5D0*mu3/mu2_p3^2)*z3*(1.0D0 - 0.5D0*z3^2/mu2_p3)
odd4 = -(0.5D0*mu3/mu2_p4^2)*z4*(1.0D0 - 0.5D0*z4^2/mu2_p4)
odd5 = -(0.5D0*mu3/mu2_p5^2)*z5*(1.0D0 - 0.5D0*z5^2/mu2_p5)

even1 = (mu4/(8.0D0*mu2_p1^2))*(1.0D0 - 2.0D0*z1^2/mu2_p1 + z1^4/(3.0D0*mu2_p1^2))
even2 = (mu4/(8.0D0*mu2_p2^2))*(1.0D0 - 2.0D0*z2^2/mu2_p2 + z2^4/(3.0D0*mu2_p2^2))
even3 = (mu4/(8.0D0*mu2_p3^2))*(1.0D0 - 2.0D0*z3^2/mu2_p3 + z3^4/(3.0D0*mu2_p3^2))
even4 = (mu4/(8.0D0*mu2_p4^2))*(1.0D0 - 2.0D0*z4^2/mu2_p4 + z4^4/(3.0D0*mu2_p4^2))
even5 = (mu4/(8.0D0*mu2_p5^2))*(1.0D0 - 2.0D0*z5^2/mu2_p5 + z5^4/(3.0D0*mu2_p5^2))

FSE1 = odd1 + even1
FSE2 = odd2 + even2
FSE3 = odd3 + even3
FSE4 = odd4 + even4
FSE5 = odd5 + even5

; Peak intensities based on Young-Kopel theory
; from Eqns (20) through (24) in Dawidowski et al, PRB vol 73, Pg 144203 (2006)
; DOI: https://doi.org/10.1103/PhysRevB.73.144203
; Only the para-contribution is used.
; Ortho contribution and coherent scattering evaluated and found to be small (Tim Prisk)
CROSS_SECTION_INC = 80.26    ; barns for odd j
CROSS_SECTION_COH = 1.7568   ; barns for even j
ScatCrossSec = [CROSS_SECTION_COH,CROSS_SECTION_INC]

Ak = fltarr(nq,5)

; Scattering cross section 
for k=1,5 do Ak[*,k-1] = ScatCrossSec[k mod 2] 
; times Form factor  
;h2_radius = 0.37  ; in Angstrom
arg = qVals*h2_radius
for k=1,5 do Ak[*,k-1] = Ak[*,k-1] * (2*k + 1)*(Sph_bessel(arg,k))^2    ; ak * A_p = (2k + 1)*(j_k(Q*a))^2; k goes from 1 to 5
                                                                    ; ak is the scattering cross section from above
; The model is a sum of five peaks.
peak1 = (ux # Ak[*,0])*IApeak1*(1.0D0 + FSE1)
peak2 = (ux # Ak[*,1])*IApeak2*(1.0D0 + FSE2)
peak3 = (ux # Ak[*,2])*IApeak3*(1.0D0 + FSE3)
peak4 = (ux # Ak[*,3])*IApeak4*(1.0D0 + FSE4)
peak5 = (ux # Ak[*,4])*IApeak5*(1.0D0 + FSE5)

Return, sf*(peak1 + peak2 + peak3 + peak4 + peak5)

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;+
; Pan_hydrogenRecoilAA
;
; PURPOSE:
;   Standard PAN built-in fit model function.
;   Hydrogen Recoil scattering function. :
;   - the Impulse Approximation is not
;   - the Additive Approach model for the momentum distribution is used
;   - the momentum distribution is a Gaussian + a small correction
;   - Final State Effects (FSE) taken into account thorugh a an additive first order correction to the scattering
;   - there are no restrictions in the relative peak intensities of the peaks modified by the rotational transitions
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; PARAMETERS:
;
;
; RETURN VALUE:
;
;-
function Pan_HydrogenRecoilAA_2D,x,parms,$
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

compile_opt idl2

canDraw = 0
twoDimFlag = 1
resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

extConvolFlag = 0            ; does function require external convolution? 0=>No, 1=>Yes
; if set to No, it implies a convolution is performed within this function when a resolution function is present
; Also set to No, if you DO NOT wish any external convolution to be performed at all

nx = Isa(x)? N_elements(x) : N_elements(xvals)
nq = N_elements(qvals)
ux = 1+Bytarr(nx)
uq = 1+Bytarr(nq)

if N_params() eq 0 then begin
    
  ; parameter names
  parmnames = []
  aname = 'shift'
  multi_parmnames = ['A1', 'A2', 'A3', 'A4', 'A5','shift']
  single_parmnames = ['KE [K]','delta','a3bar [1/(A^2 ps)]','a4bar [1/(A^2 ps^2)]']
  parmnames = single_parmnames
  for i=0,n_elements(multi_parmnames)-1 do begin
    name = multi_parmnames[i]
    names = name+Strtrim(String(Indgen(nq)),2)
    parmnames = [parmnames,names]
  endfor

  ; initialial parameter guesses aince canDraw is 0
  initParms = [60.0,0.0,20.0,0.0] ; init values for the single params
  
  ; init A1,A3,A5 to 1.0; A2,A4 to 0.0 for all groups
  initParms = [initParms, fltarr(nq)+0.006]
  initParms = [initParms, Fltarr(nq)+0.0]
  initParms = [initParms, Fltarr(nq)+0.004]
  initParms = [initParms, Fltarr(nq)+0.0]
  initParms = [initParms, Fltarr(nq)+0.002]
  ; init shift to 0.0 and scaleFac to 1.0 for all groups
  initParms = [initParms, Fltarr(nq)+0.0]

  Return,-1
endif

;; Parameter definition and initalization.

; group-independent parameters
KE = parms[0]           ; kinetic ebergy
delta = parms[1]        ; ratio of alpha4/alpha2^2
a3bar = parms[2]        ; 1/(A^2 ps)
a4bar = parms[3]        ; 1/(A^2 ps^2)

; group dependent paramters
index = Indgen(nq)
A1 = ux # parms[4 + 0*nq + index]       ; relative peak intensity for first rotational peak
A2 = ux # parms[4 + 1*nq + index]       ; same for 2nd peak
A3 = ux # parms[4 + 2*nq + index]       ; etc
A4 = ux # parms[4 + 3*nq + index]
A5 = ux # parms[4 + 4*nq + index]

shift = ux # parms[4 + 5*nq + index]    ; group-dependent arb energy shift to correct for any error in incident energy

; Define constants.
hbarsquared = 4.18016 ; meV amu A^2
m  = 1.007825*2.0     ; amu
B0 = 7.3567           ; meV
D0 = 0.00569          ; meV
H0 = 6.4e-6           ; meV

hbar    = 6.35078D0         ; amu A^2/ps
hbar2   = 4.18016D0         ; meV amu A^2
m       = 1.007825D0*2.0D0  ; amu
B0      = 7.3567D0          ; meV
D0      = 0.00569D0         ; meV
H0      = 6.4D-6            ; meV
KpermeV = 11.60451812D0     ; NIST CODATA value.
scaleFac = 9.64853          ; 1 meV = 9.64853 amu A^2/ps^2

; Convert kinetic energy to peak width.
KE_meV = KE/KpermeV
;sigmaK = Sqrt(KE_meV*(2.0*m)/(3.0*hbarsquared))
;sigmaS = (hbarsquared/m)*Qgroup*sigmaK

; Calculate cumulants.
alpha2 = (2.0D0/3.0D0)*(m/hbarsquared)*(KE_meV)                         ; A^{-2}
alpha4 = delta*alpha2^2                                                 ; A^{-4}
mu2    = ux # (hbarsquared*(hbarsquared*qVals^2/m^2)*alpha2)            ; meV^2
mu3    = ux # ((hbar^5*qVals^2/m^2)*a3bar*(1.0D0/scaleFac)^3)           ; meV^3
mu4    = (hbar^6)*((qVals/m)^2)*a4bar*(1.0D0/scaleFac)^4                ;
mu4    = ux # (mu4 + (hbar^8)*((qVals/m)^4)*alpha4*(1.0D0/scaleFac)^4)  ; meV^4

; Determine peak positions.
Recoil = ux # (0.5D0*(hbarsquared/m)*qVals^2)
Rot1 = B0*1.0D0*(1.0D0 + 1.0D0) - D0*(1.0D0*(1.0D0 + 1.0D0))^2 + H0*(1.0D0*(1.0D0 + 1.0D0))^3
Rot2 = B0*2.0D0*(2.0D0 + 1.0D0) - D0*(2.0D0*(2.0D0 + 1.0D0))^2 + H0*(2.0D0*(2.0D0 + 1.0D0))^3
Rot3 = B0*3.0D0*(3.0D0 + 1.0D0) - D0*(3.0D0*(3.0D0 + 1.0D0))^2 + H0*(3.0D0*(3.0D0 + 1.0D0))^3
Rot4 = B0*4.0D0*(4.0D0 + 1.0D0) - D0*(4.0D0*(4.0D0 + 1.0D0))^2 + H0*(4.0D0*(4.0D0 + 1.0D0))^3
Rot5 = B0*5.0D0*(5.0D0 + 1.0D0) - D0*(5.0D0*(5.0D0 + 1.0D0))^2 + H0*(5.0D0*(5.0D0 + 1.0D0))^3
C1 = Recoil + Rot1 - shift
C2 = Recoil + Rot2 - shift
C3 = Recoil + Rot3 - shift
C4 = Recoil + Rot4 - shift
C5 = Recoil + Rot5 - shift

; Observed width is intrinsic width and res width added in quadrature .
StoW = Sqrt(8.0*Alog(2.0))
mu2_p1 = mu2 + (Arcs_resolutionwidth_2d(C1)/StoW)^2
mu2_p2 = mu2 + (Arcs_resolutionwidth_2d(C2)/StoW)^2
mu2_p3 = mu2 + (Arcs_resolutionwidth_2d(C3)/StoW)^2
mu2_p4 = mu2 + (Arcs_resolutionwidth_2d(C4)/StoW)^2
mu2_p5 = mu2 + (Arcs_resolutionwidth_2d(C5)/StoW)^2

; The model is a sum of five peaks.
; Consisting of a recoil peak (additive approach model with final state corrections included)
; The recoil peaks are then modulated by the the rotational transitions of molecular para-hydrogen
inv_sqrt2pi = 1.0D0/Sqrt(2.0D0*!PI)   ; Precompute Gaussian normalization constant.

; Calculate AA final state effects to order 1/Q.
xq = x # uq
IApeak1 = A1*inv_sqrt2pi/Sqrt(mu2_p1)*Exp(-0.5D0*(xq - C1)^2/mu2_p1)
IApeak2 = A2*inv_sqrt2pi/Sqrt(mu2_p2)*Exp(-0.5D0*(xq - C2)^2/mu2_p2)
IApeak3 = A3*inv_sqrt2pi/Sqrt(mu2_p3)*Exp(-0.5D0*(xq - C3)^2/mu2_p3)
IApeak4 = A4*inv_sqrt2pi/Sqrt(mu2_p4)*Exp(-0.5D0*(xq - C4)^2/mu2_p4)
IApeak5 = A5*inv_sqrt2pi/Sqrt(mu2_p5)*Exp(-0.5D0*(xq - C5)^2/mu2_p5)

z1 = (xq - C1)
z2 = (xq - C2)
z3 = (xq - C3)
z4 = (xq - C4)
z5 = (xq - C5)

odd1 = -(0.5D0*mu3/mu2_p1^2)*z1*(1.0D0 - 0.5D0*z1^2/mu2_p1)
odd2 = -(0.5D0*mu3/mu2_p2^2)*z2*(1.0D0 - 0.5D0*z2^2/mu2_p2)
odd3 = -(0.5D0*mu3/mu2_p3^2)*z3*(1.0D0 - 0.5D0*z3^2/mu2_p3)
odd4 = -(0.5D0*mu3/mu2_p4^2)*z4*(1.0D0 - 0.5D0*z4^2/mu2_p4)
odd5 = -(0.5D0*mu3/mu2_p5^2)*z5*(1.0D0 - 0.5D0*z5^2/mu2_p5)

even1 = (mu4/(8.0D0*mu2_p1^2))*(1.0D0 - 2.0D0*z1^2/mu2_p1 + z1^4/(3.0D0*mu2_p1^2))
even2 = (mu4/(8.0D0*mu2_p2^2))*(1.0D0 - 2.0D0*z2^2/mu2_p2 + z2^4/(3.0D0*mu2_p2^2))
even3 = (mu4/(8.0D0*mu2_p3^2))*(1.0D0 - 2.0D0*z3^2/mu2_p3 + z3^4/(3.0D0*mu2_p3^2))
even4 = (mu4/(8.0D0*mu2_p4^2))*(1.0D0 - 2.0D0*z4^2/mu2_p4 + z4^4/(3.0D0*mu2_p4^2))
even5 = (mu4/(8.0D0*mu2_p5^2))*(1.0D0 - 2.0D0*z5^2/mu2_p5 + z5^4/(3.0D0*mu2_p5^2))

FSE1 = odd1 + even1
FSE2 = odd2 + even2
FSE3 = odd3 + even3
FSE4 = odd4 + even4
FSE5 = odd5 + even5

; The model is a sum of five peaks.
peak1 = IApeak1*(1.0D0 + FSE1)
peak2 = IApeak2*(1.0D0 + FSE2)
peak3 = IApeak3*(1.0D0 + FSE3)
peak4 = IApeak4*(1.0D0 + FSE4)
peak5 = IApeak5*(1.0D0 + FSE5)

Return, peak1 + peak2 + peak3 + peak4 + peak5

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;+
; Pan_hydrogenRecoilIA
;
; PURPOSE:
;   Standard PAN built-in fit model function.
;   Hydrogen Recoil scattering function. Assumes:
;   - the momentum distribution is Gaussian
;   - the Impulse Approximation is valid
;   - there are no restrictions in the relative peak intensities of the peaks modified by the rotational transitions
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; PARAMETERS:
;   
;
; RETURN VALUE:
;   
;-
function Pan_hydrogenRecoilIA_2D,x,parms,$
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

compile_opt idl2

canDraw = 0
twoDimFlag = 1
resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

extConvolFlag = 0            ; does function require external convolution? 0=>No, 1=>Yes
; if set to No, it implies a convolution is performed within this function when a resolution function is present
; Also set to No, if you DO NOT wish any external convolution to be performed at all

nx = Isa(x)? N_elements(x) : N_elements(xvals)
nq = N_elements(qvals)
ux = 1+Bytarr(nx)
uq = 1+Bytarr(nq)

if N_params() eq 0 then begin
    
  ; parameter names
  parmnames = []
  aname = 'shift'
  multi_parmnames = ['A1', 'A2', 'A3', 'A4', 'A5', 'A6','shift']
  single_parmnames = ['KE [K]']
  parmnames = single_parmnames
  for i=0,n_elements(multi_parmnames)-1 do begin
    name = multi_parmnames[i]
    names = name+Strtrim(String(Indgen(nq)),2)
    parmnames = [parmnames,names]
  endfor

  ; initialial parameter guesses aince canDraw is 0
  initParms = [60.0] ; init values for the single params
  
  ; init A1,A3,A5 to 1.0; A2,A4,A6 to 0.0 for all groups
  initParms = [initParms, fltarr(nq)+0.006]
  initParms = [initParms, Fltarr(nq)+0.0]
  initParms = [initParms, Fltarr(nq)+0.004]
  initParms = [initParms, Fltarr(nq)+0.0]
  initParms = [initParms, Fltarr(nq)+0.002]
  initParms = [initParms, Fltarr(nq)+0.0]

  initParms = [initParms, Fltarr(nq)+0.0]

  Return,-1
endif

;; Parameter definition and initalization.

; group-independent parameters
KE = parms[0]           ; kinetic ebergy

; group dependent paramters
index = Indgen(nq)
A1 = ux # parms[1 + 0*nq + index]       ; relative peak intensity for first rotational peak
A2 = ux # parms[1 + 1*nq + index]       ; same for 2nd peak
A3 = ux # parms[1 + 2*nq + index]       ; etc
A4 = ux # parms[1 + 3*nq + index]
A5 = ux # parms[1 + 4*nq + index]
A6 = ux # parms[1 + 5*nq + index]

shift = parms[1 + 6*nq + index]    ; group-dependent arb energy shift to correct for any error in incident energy

; Define constants.
hbarsquared = 4.18016 ; meV amu A^2
m  = 1.007825*2.0     ; amu
B0 = 7.3567           ; meV
D0 = 0.00569          ; meV
H0 = 6.4e-6           ; meV

hbar    = 6.35078D0         ; amu A^2/ps
hbar2   = 4.18016D0         ; meV amu A^2
m       = 1.007825D0*2.0D0  ; amu
B0      = 7.3567D0          ; meV
D0      = 0.00569D0         ; meV
H0      = 6.4D-6            ; meV
KpermeV = 11.60451812D0     ; NIST CODATA value.
scaleFac = 9.64853          ; 1 meV = 9.64853 amu A^2/ps^2
hbarsquared = 4.18016 ; meV amu A^2

; Convert kinetic energy to intrinsic peak width as a function of Q.
KE_meV = KE/KpermeV

sigmaK = Sqrt(KE_meV*(2.0*m)/(3.0*hbarsquared))
sigmaS = (hbarsquared/m)*qVals*sigmaK

; Determine peak positions for all Qs
Recoil = 0.5*(hbarsquared/m)*qVals^2.0

; Rotational shifts for first 6 peaks
Rot1 = B0*1.0*(1.0 + 1.0) - D0*(1.0*(1.0 + 1.0))^2.0 + H0*(1.0*(1.0 + 1.0))^3.0
Rot2 = B0*2.0*(2.0 + 1.0) - D0*(2.0*(2.0 + 1.0))^2.0 + H0*(2.0*(2.0 + 1.0))^3.0
Rot3 = B0*3.0*(3.0 + 1.0) - D0*(3.0*(3.0 + 1.0))^2.0 + H0*(3.0*(3.0 + 1.0))^3.0
Rot4 = B0*4.0*(4.0 + 1.0) - D0*(4.0*(4.0 + 1.0))^2.0 + H0*(4.0*(4.0 + 1.0))^3.0
Rot5 = B0*5.0*(5.0 + 1.0) - D0*(5.0*(5.0 + 1.0))^2.0 + H0*(5.0*(5.0 + 1.0))^3.0
Rot6 = B0*6.0*(6.0 + 1.0) - D0*(6.0*(6.0 + 1.0))^2.0 + H0*(6.0*(6.0 + 1.0))^3.0

; adjusted peak positions as a function of Q
C1 = Recoil + Rot1 - shift
C2 = Recoil + Rot2 - shift
C3 = Recoil + Rot3 - shift
C4 = Recoil + Rot4 - shift
C5 = Recoil + Rot5 - shift
C6 = Recoil + Rot6 - shift

; Observed with as function of Q is intrinsic width and res width added in quadrature .
; followed by vectorizing along x
StoW = Sqrt(8.0*Alog(2.0))    ; Sigma to FWHM factor
sigmaObs_1 = ux # (Sqrt((Arcs_resolutionwidth_2d(C1)/StoW)^2 + sigmaS^2))
sigmaObs_2 = ux # (Sqrt((Arcs_resolutionwidth_2d(C2)/StoW)^2 + sigmaS^2))
sigmaObs_3 = ux # (Sqrt((Arcs_resolutionwidth_2d(C3)/StoW)^2 + sigmaS^2))
sigmaObs_4 = ux # (Sqrt((Arcs_resolutionwidth_2d(C4)/StoW)^2 + sigmaS^2))
sigmaObs_5 = ux # (Sqrt((Arcs_resolutionwidth_2d(C5)/StoW)^2 + sigmaS^2))
sigmaObs_6 = ux # (Sqrt((Arcs_resolutionwidth_2d(C6)/StoW)^2 + sigmaS^2))

; Evaluate the normalized Gaussian widths of the 6 peaks from the above sigma
; Fully vectorized
InvSqrt2pi = 1/sqrt(2.0*!dpi)
xq = x # uq
peak1 = A1*InvSqrt2pi/sigmaObs_1*Exp(-0.5*((xq-(ux # C1))/sigmaObs_1)^2)
peak2 = A2*InvSqrt2pi/sigmaObs_2*Exp(-0.5*((xq-(ux # C2))/sigmaObs_2)^2)
peak3 = A3*InvSqrt2pi/sigmaObs_3*Exp(-0.5*((xq-(ux # C3))/sigmaObs_3)^2)
peak4 = A4*InvSqrt2pi/sigmaObs_4*Exp(-0.5*((xq-(ux # C4))/sigmaObs_4)^2)
peak5 = A5*InvSqrt2pi/sigmaObs_5*Exp(-0.5*((xq-(ux # C5))/sigmaObs_5)^2)
peak6 = A6*InvSqrt2pi/sigmaObs_6*Exp(-0.5*((xq-(ux # C6))/sigmaObs_6)^2)

Return, peak1 + peak2 + peak3 + peak4 + peak5 + peak6

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


pro Test_Pan_HydrogenRecoil_2D, func=func_choice, qVal=qVal, yout=yout, plot=plot
  compile_opt idl2

  func_choice = (n_elements(func_choice) eq 0)? 0 : func_choice
  if (~Isa(qVal)) then qVal = 5.0
  x = Findgen(301)*2.0 - 50.0
  
  case func_choice of
    0: begin
      parms = [60.0,1.0,1.0,1.0,1.0,1.0,1.0,0.0,0.001]
      void = Pan_HydrogenRecoilIA_2D(parmnames=pnames)
      yout = Pan_HydrogenRecoilIA_2D(x,parms,qgroup=qVal)
      if (Keyword_set(plot)) then begin
        title = 'Hydrogen Recoil Peaks IA'
        p = Plot(x,yout,title=title,symbol=0,linestyle=0)
      endif

      end  
        
    1: begin
      parms = [60.0,0.0,1.0,0.5,1.0,1.0,1.0,1.0,1.0,0.0,0.001]
      void = Pan_HydrogenRecoilAA_2D(parmnames=pnames)
      yout = Pan_HydrogenRecoilAA_2D(x,parms,qgroup=qVal)
      if (Keyword_set(plot)) then begin
        title = 'Hydrogen Recoil Peaks AA'
        p = Plot(x,yout,title=title,symbol=0,linestyle=0)
      endif

      end

      2: begin
        parms = [60.0,0.0,1.0,0.5,0.0,0.001]
        void = Pan_hydrogenrecoilaayk_2D(parmnames=pnames)
        yout = Pan_hydrogenrecoilaayk_2D(x,parms,qgroup=qVal)
        if (Keyword_set(plot)) then begin
          title = 'Hydrogen Recoil Peaks AA+YK'
          p = Plot(x,yout,title=title,symbol=0,linestyle=0)
        endif

      end
          
    else:
  endcase



end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
