; $Id$
;###############################################################################
;
; NAME:
;  pan_gaussian_2D
;
; PURPOSE:
;  Implementation of a 2D Gaussian distribution for use in fitting
;  data in PAN.
;
; CATEGORY:
;  DAVE, Data Analysis, PAN, surface fitting
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or anq other
;  characteristic. The use of certain trade names or commercial
;  products does not imply anq endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function pan_gaussGlobalAreaOnly_2D,x,parms,                $
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

nx = isa(x)? n_elements(x) : n_elements(xvals)
nq = n_elements(qvals)
ux = 1+bytarr(nx)
uq = 1+bytarr(nq)

canDraw = 0
twoDimFlag = 1
resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

extConvolFlag = 1            ; does function require external convolution? 0=>No, 1=>Yes
                             ; if set to No, it implies a convolution is performed within this function when a resolution function is present
                             ; Also set to No, if you DO NOT wish any external convolution to be performed at all

if (n_params() eq 0) or (n_params() eq 1) then begin
  single_parmnames = 'Area'
  multi_parmnames = ['Center','FWHM']
  
  fwhmstr = 'FWHM'+Strtrim(String(Indgen(nq)),2)
  centerstr = 'Center'+Strtrim(String(Indgen(nq)),2)
  parmnames = [single_parmnames,centerstr,fwhmstr]

  step = dblarr(3*nq)
  if arg_present(initParms) then begin
    amp = max(yVals,dimension = 1,index)
    yVals_size = size(yVals)
    nq = yVals_size[2]
    nx = yVals_size[1]
    index = index - indgen(nq)*nx ; index contains the 1D subscripts so need to reform
    center = xvals[index]
    
    ; assuming a single main peak then estimate FWHM
    fwhm = fltarr(nq)
    for i=0,nq-1 do begin
      hm = 0.5*amp[i]
      res = where(yVals[*,i] ge hm, found)
      if (found gt 0) then begin
        fwhm[i] = xvals[res[found-1]] - xvals[res[0]]
      endif
    endfor
    sig = fwhm/2.354
    initParms = [sqrt(2.0*!dpi)*sig[0]*amp[0],center,fwhm]  ; area is global so use the area of first group 
  endif

  return,-1
endif

area = parms[0]
index = Indgen(nq)
center = parms[index+1]
sig = parms[index+nq+1]/2.354

yout = (ux#(area/(sig*sqrt(2.0*!dpi))))*exp(-0.5*((x#uq-ux#center)/(ux#sig))^2)
;;yout = scat_convolve(x,y,gauss,*res_ptr) 
;;yout = Scat_convolute(x,y,Gauss,*res_ptr) ;RTA - scat_convolve->scat_convolute
;if ((N_elements(resPtr) gt 0) && Ptr_valid(resPtr) && (N_elements(*resPtr) gt 0)) then begin
;  yout = pan_convolute(x,yout,resPtr)
;endif
return,yout
end


function Pan_gaussGlobalAreaWidth_2d,x,parms,                $
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

  nx = Isa(x)? N_elements(x) : N_elements(xvals)
  nq = N_elements(qvals)
  ux = 1+Bytarr(nx)
  uq = 1+Bytarr(nq)

  canDraw = 0
  twoDimFlag = 1
  resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

  extConvolFlag = 1            ; does function require external convolution? 0=>No, 1=>Yes
  ; if set to No, it implies a convolution is performed within this function when a resolution function is present
  ; Also set to No, if you DO NOT wish any external convolution to be performed at all

  if (N_params() eq 0) or (N_params() eq 1) then begin
    single_parmnames = ['Area','FWHM']
    multi_parmnames = 'Center'
    
    centerstr = 'Center'+Strtrim(String(Indgen(nq)),2)
    parmnames = single_parmnames
    parmnames = [parmnames,centerstr]
    
    step = Dblarr(3*nq)
    if Arg_present(initParms) then begin
      amp = Max(yVals,dimension = 1,index)
      yVals_size = Size(yVals)
      nq = yVals_size[2]
      nx = yVals_size[1]
      index = index - Indgen(nq)*nx ; index contains the 1D subscripts so need to reform
      center = xvals[index]

      ; assuming a single main peak then estimate FWHM
      fwhm = Fltarr(nq)
      for i=0,nq-1 do begin
        hm = 0.5*amp[i]
        res = Where(yVals[*,i] ge hm, found)
        if (found gt 0) then begin
          fwhm[i] = xvals[res[found-1]] - xvals[res[0]]
        endif
      endfor
      sig = fwhm/2.354
      initParms = [Sqrt(2.0*!dpi)*sig[0]*amp[0],fwhm[0],center]   ; area and fwhm are global so use first group only
    endif

    Return,-1
  endif

  area = uq*parms[0]
  sig = uq*parms[1]/2.354
  index = Indgen(nq)
  center = parms[index+2]

  yout = (ux#(area/(sig*Sqrt(2.0*!dpi))))*Exp(-0.5*((x#uq-ux#center)/(ux#sig))^2)
  ;;yout = scat_convolve(x,y,gauss,*res_ptr)
  ;;yout = Scat_convolute(x,y,Gauss,*res_ptr) ;RTA - scat_convolve->scat_convolute
  ;if ((N_elements(resPtr) gt 0) && Ptr_valid(resPtr) && (N_elements(*resPtr) gt 0)) then begin
  ;  yout = pan_convolute(x,yout,resPtr)
  ;endif
  Return,yout
end


function Pan_gaussGlobalAreaWidthCenter_2d,x,parms,                $
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

  nx = Isa(x)? N_elements(x) : N_elements(xvals)
  nq = N_elements(qvals)
  ux = 1+Bytarr(nx)
  uq = 1+Bytarr(nq)

  canDraw = 0
  twoDimFlag = 1
  resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

  extConvolFlag = 1            ; does function require external convolution? 0=>No, 1=>Yes
  ; if set to No, it implies a convolution is performed within this function when a resolution function is present
  ; Also set to No, if you DO NOT wish any external convolution to be performed at all

  if (N_params() eq 0) or (N_params() eq 1) then begin
    ;areastr = 'Area'+Strtrim(String(Indgen(nq)),2)
    ;centerstr = 'Aenter'+Strtrim(String(Indgen(nq)),2)
    ;fwhmstr = 'FWHM'+Strtrim(String(Indgen(nq)),2)
    single_parmnames = ['Area','Center','FWHM']
    multi_parmnames = ''
    parmnames = single_parmnames
    step = Dblarr(3*nq)
    if Arg_present(initParms) then begin
      amp = Max(yVals,dimension = 1,index)
      yVals_size = Size(yVals)
      nq = yVals_size[2]
      nx = yVals_size[1]
      index = index - Indgen(nq)*nx ; index contains the 1D subscripts so need to reform
      center = xvals[index]

      ; assuming a single main peak then estimate FWHM
      fwhm = Fltarr(nq)
      for i=0,nq-1 do begin
        hm = 0.5*amp[i]
        res = Where(yVals[*,i] ge hm, found)
        if (found gt 0) then begin
          fwhm[i] = xvals[res[found-1]] - xvals[res[0]]
        endif
      endfor
      sig = fwhm/2.354
      initParms = [Sqrt(2.0*!dpi)*sig[0]*amp[0],center[0],fwhm[0]]    ; use first group value only since area, cen and fwhm are fixed
    endif

    Return,-1
  endif

  area = ux#uq*parms[0]
  cen  = ux#uq*parms[1]
  sig  = ux#uq*parms[2]/2.354
  xq   = x#uq


  yout = area/(sig*Sqrt(2.0*!dpi))*Exp(-0.5*((xq-cen)/sig)^2)
  ;;yout = scat_convolve(x,y,gauss,*res_ptr)
  ;;yout = Scat_convolute(x,y,Gauss,*res_ptr) ;RTA - scat_convolve->scat_convolute
  ;if ((N_elements(resPtr) gt 0) && Ptr_valid(resPtr) && (N_elements(*resPtr) gt 0)) then begin
  ;  yout = pan_convolute(x,yout,resPtr)
  ;endif
  Return,yout
end


function Pan_gaussGlobalWidthOnly_2d,x,parms,                $
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

  nx = Isa(x)? N_elements(x) : N_elements(xvals)
  nq = N_elements(qvals)
  ux = 1+Bytarr(nx)
  uq = 1+Bytarr(nq)

  canDraw = 0
  twoDimFlag = 1
  resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

  extConvolFlag = 1            ; does function require external convolution? 0=>No, 1=>Yes
  ; if set to No, it implies a convolution is performed within this function when a resolution function is present
  ; Also set to No, if you DO NOT wish any external convolution to be performed at all

  if (N_params() eq 0) or (N_params() eq 1) then begin
    single_parmnames = 'FWHM'
    multi_parmnames = ['Area','Center']
    parmnames = single_parmnames
    areastr = 'Area'+Strtrim(String(Indgen(nq)),2)
    centerstr = 'Center'+Strtrim(String(Indgen(nq)),2)
    parmnames = [parmnames,areastr,centerstr]

    step = Dblarr(3*nq)
    if Arg_present(initParms) then begin
      amp = Max(yVals,dimension = 1,index)
      yVals_size = Size(yVals)
      nq = yVals_size[2]
      nx = yVals_size[1]
      index = index - Indgen(nq)*nx ; index contains the 1D subscripts so need to reform
      center = xvals[index]

      ; assuming a single main peak then estimate FWHM
      fwhm = Fltarr(nq)
      for i=0,nq-1 do begin
        hm = 0.5*amp[i]
        res = Where(yVals[*,i] ge hm, found)
        if (found gt 0) then begin
          fwhm[i] = xvals[res[found-1]] - xvals[res[0]]
        endif
      endfor
      sig = fwhm/2.354
      initParms = [fwhm[0],Sqrt(2.0*!dpi)*sig[0]*amp,center]  ; fwhm is global
    endif

    Return,-1
  endif

  index = Indgen(nq)
  sig = parms[0]/2.354
  area = parms[index+1]
  center = parms[index+nq+1]

  yout = (ux#(area/(sig*Sqrt(2.0*!dpi))))*Exp(-0.5*((x#uq-ux#center)/(ux#sig))^2)
  ;;yout = scat_convolve(x,y,gauss,*res_ptr)
  ;;yout = Scat_convolute(x,y,Gauss,*res_ptr) ;RTA - scat_convolve->scat_convolute
  ;if ((N_elements(resPtr) gt 0) && Ptr_valid(resPtr) && (N_elements(*resPtr) gt 0)) then begin
  ;  yout = pan_convolute(x,yout,resPtr)
  ;endif
  Return,yout
end


function Pan_gaussGlobalWidthQ2Dep_2d,x,parms,                $
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

  nx = Isa(x)? N_elements(x) : N_elements(xvals)
  nq = N_elements(qvals)
  ux = 1+Bytarr(nx)
  uq = 1+Bytarr(nq)

  canDraw = 0
  twoDimFlag = 1
  resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

  extConvolFlag = 1            ; does function require external convolution? 0=>No, 1=>Yes
  ; if set to No, it implies a convolution is performed within this function when a resolution function is present
  ; Also set to No, if you DO NOT wish any external convolution to be performed at all

  if (N_params() eq 0) or (N_params() eq 1) then begin
    single_parmnames = 'FWHM'
    multi_parmnames = ['Area','Center']
    parmnames = single_parmnames
    areastr = 'Area'+Strtrim(String(Indgen(nq)),2)
    centerstr = 'Center'+Strtrim(String(Indgen(nq)),2)
    parmnames = [parmnames,areastr,centerstr]

    step = Dblarr(3*nq)
    if Arg_present(initParms) then begin
      amp = Max(yVals,dimension = 1,index)
      yVals_size = Size(yVals)
      nq = yVals_size[2]
      nx = yVals_size[1]
      index = index - Indgen(nq)*nx ; index contains the 1D subscripts so need to reform
      center = xvals[index]

      ; assuming a single main peak then estimate FWHM
      fwhm = Fltarr(nq)
      for i=0,nq-1 do begin
        hm = 0.5*amp[i]
        res = Where(yVals[*,i] ge hm, found)
        if (found gt 0) then begin
          fwhm[i] = xvals[res[found-1]] - xvals[res[0]]
        endif
      endfor
      qVal   = qVals[groupNumber-1]
      sig = fwhm/2.354
      ;initParms = [Sqrt(2.0*!dpi)*sig*amp,center,fwhm]
      initParms = [fwhm[0]/qVal^2,Sqrt(2.0*!dpi)*sig[0]*amp,center]  ; fwhm is global as well as varies as Q^2
    endif

    Return,-1
  endif

  index = Indgen(nq)
  sig = parms[0]*(qVals^2)/2.354      ; enforce a linear Q^2 dependence for width
  area = parms[index+1]
  center = parms[index+nq+1]

  yout = (ux#(area/(sig*Sqrt(2.0*!dpi))))*Exp(-0.5*((x#uq-ux#center)/(ux#sig))^2)
  ;;yout = scat_convolve(x,y,gauss,*res_ptr)
  ;;yout = Scat_convolute(x,y,Gauss,*res_ptr) ;RTA - scat_convolve->scat_convolute
  ;if ((N_elements(resPtr) gt 0) && Ptr_valid(resPtr) && (N_elements(*resPtr) gt 0)) then begin
  ;  yout = pan_convolute(x,yout,resPtr)
  ;endif
  Return,yout
end


function Pan_gaussGlobalWidthQDep_2d,x,parms,                $
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

  nx = Isa(x)? N_elements(x) : N_elements(xvals)
  nq = N_elements(qvals)
  ux = 1+Bytarr(nx)
  uq = 1+Bytarr(nq)

  canDraw = 0
  twoDimFlag = 1
  resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

  extConvolFlag = 1            ; does function require external convolution? 0=>No, 1=>Yes
  ; if set to No, it implies a convolution is performed within this function when a resolution function is present
  ; Also set to No, if you DO NOT wish any external convolution to be performed at all

  if (N_params() eq 0) or (N_params() eq 1) then begin
    single_parmnames = 'FWHM'
    multi_parmnames = ['Area','Center']
    parmnames = single_parmnames
    areastr = 'Area'+Strtrim(String(Indgen(nq)),2)
    centerstr = 'Center'+Strtrim(String(Indgen(nq)),2)
    parmnames = [parmnames,areastr,centerstr]

    step = Dblarr(3*nq)
    if Arg_present(initParms) then begin
      amp = Max(yVals,dimension = 1,index)
      yVals_size = Size(yVals)
      nq = yVals_size[2]
      nx = yVals_size[1]
      index = index - Indgen(nq)*nx ; index contains the 1D subscripts so need to reform
      center = xvals[index]

      ; assuming a single main peak then estimate FWHM
      fwhm = Fltarr(nq)
      for i=0,nq-1 do begin
        hm = 0.5*amp[i]
        res = Where(yVals[*,i] ge hm, found)
        if (found gt 0) then begin
          fwhm[i] = xvals[res[found-1]] - xvals[res[0]]
        endif
      endfor
      qVal   = qVals[groupNumber-1]
      sig = fwhm/2.354
      ;initParms = [Sqrt(2.0*!dpi)*sig*amp,center,fwhm]
      initParms = [fwhm[0]/qVal,Sqrt(2.0*!dpi)*sig[0]*amp,center]  ; fwhm is global as well as varies as Q
    endif

    Return,-1
  endif

  index = Indgen(nq)
  sig = parms[0]*qVals/2.354      ; enforce a linear Q dependence for width
  area = parms[index+1]
  center = parms[index+nq+1]

  yout = (ux#(area/(sig*Sqrt(2.0*!dpi))))*Exp(-0.5*((x#uq-ux#center)/(ux#sig))^2)
  ;;yout = scat_convolve(x,y,gauss,*res_ptr)
  ;;yout = Scat_convolute(x,y,Gauss,*res_ptr) ;RTA - scat_convolve->scat_convolute
  ;if ((N_elements(resPtr) gt 0) && Ptr_valid(resPtr) && (N_elements(*resPtr) gt 0)) then begin
  ;  yout = pan_convolute(x,yout,resPtr)
  ;endif
  Return,yout
end


function Pan_gaussIndepAreaWidthCenter_2d,x,parms,                $
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

  nx = Isa(x)? N_elements(x) : N_elements(xvals)
  nq = N_elements(qvals)
  ux = 1+Bytarr(nx)
  uq = 1+Bytarr(nq)

  canDraw = 0
  twoDimFlag = 1
  resolutionRequiredFlag = 0   ; does function require a resolution function to be defined 0=>No, 1=>Yes

  extConvolFlag = 1            ; does function require external convolution? 0=>No, 1=>Yes
  ; if set to No, it implies a convolution is performed within this function when a resolution function is present
  ; Also set to No, if you DO NOT wish any external convolution to be performed at all

  if (N_params() eq 0) or (N_params() eq 1) then begin
    areastr = 'Area'+Strtrim(String(Indgen(nq)),2)
    centerstr = 'Center'+Strtrim(String(Indgen(nq)),2)
    fwhmstr = 'FWHM'+Strtrim(String(Indgen(nq)),2)
    parmnames = [areastr,centerstr,fwhmstr]
    single_parmnames = ''
    multi_parmnames = ['Area','Center','FWHM']
    step = Dblarr(3*nq)
    if Arg_present(initParms) then begin
      amp = Max(yVals,dimension = 1,index)
      yVals_size = Size(yVals)
      nq = yVals_size[2]
      nx = yVals_size[1]
      index = index - Indgen(nq)*nx ; index contains the 1D subscripts so need to reform
      center = xvals[index]

      ; assuming a single main peak then estimate FWHM
      fwhm = Fltarr(nq)
      for i=0,nq-1 do begin
        hm = 0.5*amp[i]
        res = Where(yVals[*,i] ge hm, found)
        if (found gt 0) then begin
          fwhm[i] = xvals[res[found-1]] - xvals[res[0]]
        endif
      endfor
      sig = fwhm/2.354
      initParms = [Sqrt(2.0*!dpi)*sig*amp,center,fwhm]
    endif

    Return,-1
  endif

  index = Indgen(nq)
  area = parms[index]
  center = parms[index+nq]
  sig = (parms[index+2*nq])/2.354

  yout = (ux#(area/(sig*Sqrt(2.0*!dpi))))*Exp(-0.5*((x#uq-ux#center)/(ux#sig))^2)
  ;;yout = scat_convolve(x,y,gauss,*res_ptr)
  ;;yout = Scat_convolute(x,y,Gauss,*res_ptr) ;RTA - scat_convolve->scat_convolute
  ;if ((N_elements(resPtr) gt 0) && Ptr_valid(resPtr) && (N_elements(*resPtr) gt 0)) then begin
  ;  yout = pan_convolute(x,yout,resPtr)
  ;endif
  Return,yout
end
