; $Id$
;###############################################################################
;
; NAME:
;  PAN_BoundedJumpDiffusion_2D
;
; PURPOSE:
;  Implementation of the bounded jump diffusion model for use in fitting
;  data in PAN.
;
; CATEGORY:
;  DAVE, Data Analysis, surface fitting
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
;  Robert M. Dimeo, Ph.D.
;  NIST Center for Neutron Research
;  Phone: (301) 975-8135
;  E-mail: robert.dimeo@nist.gov
;  http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function PAN_BoundedJumpDiffusion_2D,x,parms,				$
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

nx = isa(x)? n_elements(x) : n_elements(xvals)
nq = n_elements(qvals)
ux = 1+bytarr(nx)
uq = 1+bytarr(nq)

canDraw = 0
twoDimFlag = 1
resolutionRequiredFlag = 1   ; does function require a resolution function to be defined 0=>No, 1=>Yes

extConvolFlag = 0           ; does function require external convolution? 0=>No, 1=>Yes
                            ; if set to No, it implies a convolution is performed within this function when a resolution function is present
                            ; Also set to No, if you DO NOT wish any external convolution to be performed at all

if (n_params() eq 0) or (n_params() eq 1) then begin
  parmnames = ['Confinement distance (A)','Jump distance (A)', $
    'HWHM',$
    'Intensity'+Strtrim(String(Indgen(nq)),2), $
    'Center'+Strtrim(String(Indgen(nq)),2)]

  single_parmnames = parmnames[0:2]
  multi_parmnames = ['Intensity','Center']

	if arg_present(initParms) then begin
    intensity = Max(yVals,dimension = 1,index)
    index = index - Indgen(nq)*nx ; index contains the 1D subscripts so need to reform
    center = xvals[index]

    ; assuming a single main peak then estimate HWHM for the _largest_ Q value
    hm = 0.5*intensity[nq-1]
    res = Where(yVals[*,nq-1] ge hm, found)
    if (found gt 0) then begin
      HWHM = 0.5*xvals[res[found-1]] - xvals[res[0]]
    endif


		ro = 1.0    ; in angstroms
		L =  2.0    ; in angstroms
		initParms = [L,ro,HWHM,intensity,center]
	endif
  return,-1
endif
if (~Isa(resPtr,'Pointer')) then Return, Fltarr(nx,nq)*0.0 ; can't proceed without a pointer to the resolution function
yres = (*resPtr).data
if (yres.ndim ne 2) then Return, yout   ; bail if resolution is not 2D

L = parms[0]
ro = parms[1]
HWHM = parms[2]

index = indgen(nq) + 3
intensity = ux#(parms[index])
center = ux#(parms[index+nq])

; Delta component
j0 = sph_bessel(0.5*qVals*L,0)
A0 = j0^2
pseudo_delta = Fltarr(2,nq)             ; delta-function parameters
pseudo_delta[0,*] = A0                  ; delta area
pseudo_delta[1,*] = parms[index+nq]     ; delta position

; Lor component
nterms = 40                             ; nos of terms in the sum of Lorentzians (n=1 to infinity)
AnLor = dblarr(nx,nq)
for n = 1,nterms-1 do begin
	An = ux#(((2.0*qVals*L)^2/((qVals*L)^2-(n*!dpi)^2)^2)*(1.0-((-1.0)^n)*cos(qVals*L)))
	GAMn = ux#(HWHM*(1.0-exp(-0.5*(n*!dpi*ro/L)^2)))#uq
	Lor = GAMn/(GAMn^2+(x#uq-center)^2)
	AnLor = AnLor + (An/!dpi)*Lor
endfor

; Perform convolution with resolution
; the lor is convoluted seperatedly as well as the delta and 
; a sum of the two is returned by pan_convolute()
yout = intensity*pan_convolute(x,resPtr,AnLor,pseudo_delta = pseudo_delta)

return,yout
end
