;===============================================================================
;+
; PAN_EPSDE::Plot
;
; PURPOSE:
;   Plot current results
;
; PARAMETERS:
;
; KEYWORDS:
; 
;   
;-
pro PAN_EPSDE::Plot, _REF_EXTRA=etc
compile_opt idl2

Self->GetProperty, xvalues=x, yvalues=y, evalues=e, initparam=xp0,bestparam=xp1
status = Self->Model(xp0,calc=c0)
status = Self->Model(xp1,calc=c1)

p = errorplot(x,y,e,sym_color="green",symbol="square",linestyle=6)
p0 = plot(x,c0,"r-2",/overplot)
p1 = plot(x,c1,"b-3",/overplot)

Self->GetProperty, bestfvalue=bestfValue, generationCnt=genCnt, idealparam=xpi
print,'Initial parameters: ',xp0
print,'Best fit parameters: ',xp1
print,'Ideal fit parameters: ',xpi
print,'%tage dfference: ', 100.0*abs((xp1-xpi)/xpi)
print,'Nof of Generations reached = ',genCnt,'   Chisq = ',bestfValue
end
;-------------------------------------------------------------------------------


;===============================================================================
;+
; PAN_EPSDE::Evaluate 
;
; PURPOSE:
;   Get Property
;
; PARAMETERS:
;
; KEYWORDS:
; 
;   params - function parameters
;   
;   modelFlag  - specify which model to use
;   
;-
function PAN_EPSDE::Model, params, calc=yCalc
compile_opt idl2

if (strcmp(Self.fitFuncName,'')) then return, 0
fitFuncArgs  = ptr_valid(Self.fitFuncArgs)? (*Self.fitFuncArgs) : !NULL


if (n_elements(fitFuncArgs) gt 0) then $
  call_procedure, Self.fitFuncName, (*Self.xDataPtr), params, yCalc, _extra = fitFuncArgs else $
  call_procedure, Self.fitFuncName, (*Self.xDataPtr), params, yCalc

(*Self.calcPtr) = yCalc
Self.nFE++

return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
;+
; PAN_EPSDE::Evaluate 
;
; PURPOSE:
;   Get Property
;
; PARAMETERS:
;
; KEYWORDS:
; 
;   params - function parameters
;   
;   errFuncFlag  - specify which error function calculation to use
;   
;-
function PAN_EPSDE::Evaluate, params, errFuncFlag=errFuncFlag, _REF_EXTRA=etc
compile_opt idl2


if (n_elements(errFuncFlag) eq 0) then errFuncFlag=Self.errFuncFlag

errorFunc = 1.0e4

status = Self->Model(params)

yData = (*Self.yDataPtr)
yError = (*Self.yErrorPtr)
calc = (*Self.calcPtr)
nd = n_elements(yData)
np = n_elements(params)

case errFuncFlag of
   ; mean square error (MSE)
   0: errorFunc = 1.0/(nd-np) * total(((yData - calc)/yError)^2, /NAN)
   
   ; Mean-abosolute error (MAE)
   1: errorFunc = 1.0/(nd-np) * total(abs((yData - calc)/yError), /NAN)
   
   ; Mean-square error of the log transformed data (MSElog)
   2: errorFunc = 1.0/(nd-np) * total(((alog10(yData) - alog10(calc))/alog10(yError))^2, /NAN)
   
   ; Mean-absolute error of the log transformed data (MAElog)
   3: errorFunc = 1.0/(nd-np) * total(abs((alog10(yData) - alog10(calc))/alog10(yError)), /NAN)
   
   else:

endcase


return, errorFunc
end
;-------------------------------------------------------------------------------


;===============================================================================
;+
; PAN_EPSDE::GetProperty 
;
; PURPOSE:
;   Get Property
;
; PARAMETERS:
;
; KEYWORDS:
; 
;   errFuncFlag - the error function choice
;   
;   nPop  - set if the data specified by the dataObject keyword is to be reloaded
;   
;   nMS   - the number of mutation strategies available
;   
;   model - an object reference to the model function to be fitted
;   
;   epsilon  - another terminating condition: Stop fit when change in the objective function is less than epsilon.
;   
;   nFEMax - maximum nos of function evaluations before terminating fit
;-
pro PAN_EPSDE::GetProperty, errFuncFlag=errFuncFlag, nPop=nPop, maxnFE=maxnFE, epsilon=epsilon $
                      , model=model, modelFlag=modelFlag, nMS=nMS $
                      , generationCnt=generationCnt,pTied=pTied $
                      , bestParam=bestParam, initParam=initParam, bestParError=bestParError $
                      , idealParam=idealParam, bestCalc=bestCalc $
                      , nFE=nFE, bestFValue=bestFValue, bestFValThreshold=bestFValThreshold, chisqThreshPercent=chisqThreshPercent $
                      , xvalues=xvalues,yvalues=yvalues, evalues=evalues $
                      , fitFuncName=fitFuncName, fitFuncArgs=fitFuncArgs $
                      , iterProc=iterProc, iterArgs=iterArgs $
                      , enableFlag=enableFlag, wPS=wPS $
                      , _REF_EXTRA=etc
compile_opt idl2

if (arg_present(wPS)) then wPS = Self.wPS
if (arg_present(enableFlag)) then enableFlag = Self.enableFlag
if (arg_present(fitFuncName)) then fitFuncName = Self.fitFuncName
if (arg_present(fitFuncArgs)) then fitFuncArgs = ptr_valid(Self.fitFuncArgs)? (*Self.fitFuncArgs) : !NULL
if (arg_present(iterProc)) then iterProc = Self.iterProc
if (arg_present(iterArgs)) then iterArgs = ptr_valid(Self.iterArgs)? (*Self.iterArgs) : !NULL
if (arg_present(nPop)) then nPop = Self.nPop
if (Arg_present(pTied)) then pTied = (*Self.pTiedPtr)
if (arg_present(maxnFE)) then maxnFE = Self.nFEMax
if (arg_present(epsilon)) then epsilon = Self.epsilon
if (arg_present(model)) then model = Self.model
if (arg_present(nMS)) then nMS = Self.nMS
if (arg_present(errFuncFlag)) then errFuncFlag = Self.errFuncFlag
if (arg_present(modelFlag)) then modelFlag = Self.modelFlag
if (arg_present(generationCnt)) then generationCnt = Self.generationCnt
if (arg_present(nFE)) then nFE = Self.nFE
if (arg_present(bestFValue)) then bestFValue = Self.xpbestFValue
if (arg_present(bestFValThreshold)) then bestFValThreshold = Self.bestFValThreshold
if (Arg_present(chisqThreshPercent)) then chisqThreshPercent = Self.chisqThreshPercent
if (arg_present(bestParam)) then begin
   if (ptr_valid(Self.xpBestPtr)) then bestParam = (*Self.xpBestPtr)
endif
if (Arg_present(bestParError)) then begin
  if (Ptr_valid(Self.xpBestErrorPtr)) then bestParError = (*Self.xpBestErrorPtr)
endif
if (arg_present(InitParam)) then begin
   if (ptr_valid(Self.xpInitPtr)) then initParam = (*Self.xpInitPtr)
endif
if (arg_present(idealParam)) then begin
   if (ptr_valid(Self.pIdealPtr)) then idealParam = (*Self.pIdealPtr)
endif
if (arg_present(xvalues)) then begin
   if (ptr_valid(Self.xDataPtr)) then xvalues = (*Self.xDataPtr)
endif
if (arg_present(yvalues)) then begin
   if (ptr_valid(Self.yDataPtr)) then yvalues = (*Self.yDataPtr)
endif
if (arg_present(evalues)) then begin
   if (ptr_valid(Self.yErrorPtr)) then evalues = (*Self.yErrorPtr)
endif

if (n_elements(etc) gt 0) then Self->IDLitComponent::GetProperty, _EXTRA=etc 

end
;-------------------------------------------------------------------------------


;===============================================================================
;+
; PAN_EPSDE::SetProperty 
;
; PURPOSE:
;   Set Property
;
; PARAMETERS:
;
; KEYWORDS:
; 
;   errFuncFlag - the error function calculation flag
;   
;   nPop  - set if the data specified by the dataObject keyword is to be reloaded
;   
;   nMS   - the number of mutation strategies available
;   
;   model - an object reference to the model function to be fitted
;   
;   epsilon  - another terminating condition: Stop fit when change in the objective function is less than epsilon.
;   
;   maxnFE - maximum nos of function evaluations before terminating fit
;-
pro PAN_EPSDE::SetProperty, errFuncFlag=errFuncFlag, nPop=nPop, maxnFE=maxnFE, epsilon=epsilon $
                      , model=model,modelFlag=modelFlag, nMS=nMS $
                      , startingFlag=startingFlag, bestFValue=bestFValue $
                      , bestFValThreshold=bestFValThreshold,chisqThreshPercent=chisqThreshPercent $
                      , xData=xData, yData=yData, yError=yError, pMin=pMin, pMax=pMax, pTied=pTied $
                      , fitFuncName=fitFuncName, fitFuncArgs=fitFuncArgs $
                      , iterProc=iterProc, iterArgs=iterArgs $
                      , enableFlag=enableFlag, wPS=wPS $
                      , _EXTRA=etc
compile_opt idl2

if (n_elements(pMin) gt 0) then begin
  *Self.pMinPtr = pMin
  Self.nPar = n_elements(pMin)
endif
if (n_elements(pMax) gt 0) then begin
  *Self.pMaxPtr = pMax
  Self.nPar = n_elements(pMax)
endif
if (N_elements(pTied) gt 0) then begin
  *Self.pTiedPtr = pTied
  tiedParFlag = (total(pTied) ge 2)? 1 : 0
  Self.tiedParFlag = tiedParFlag
endif
if (n_elements(xData) gt 0) then *Self.xDataPtr = xData
if (n_elements(ydata) gt 0) then begin
  *Self.yDataPtr = ydata
  *Self.calcPtr = yData
endif
if (n_elements(yError) gt 0) then begin
  index = where(yError le 0.0, cnt)
  if (cnt gt 0) then yError[index] = 1
  *Self.yErrorPtr = yError
endif

if (n_elements(bestFValThreshold) gt 0) then Self.bestFValThreshold = bestFValThreshold
if (N_elements(chisqThreshPercent) gt 0) then Self.chisqThreshPercent = chisqThreshPercent
if (n_elements(wPS) gt 0) then Self.wPS = wPS
if (n_elements(nPop) gt 0) then Self.nPop = nPop
if (n_elements(maxnFE) gt 0) then Self.nFEMax = maxnFE
if (n_elements(epsilon) gt 0) then Self.epsilon = epsilon
if (n_elements(model) gt 0 && obj_valid(model)) then Self.model = model
if (n_elements(nMS) gt 0) then Self.nMS = nMS
if (n_elements(errFuncFlag) gt 0) then Self.errFuncFlag = errFuncFlag
if (n_elements(fitFuncName) gt 0) then Self.fitFuncName = fitFuncName
if (n_elements(iterProc) gt 0) then Self.iterProc = iterProc
if (n_elements(fitFuncArgs) gt 0) then begin
  if (ptr_valid(Self.fitFuncArgs)) then  $
    (*Self.fitFuncArgs) = fitFuncArgs else $
    Self.fitFuncArgs = ptr_new(fitFuncArgs)
endif
if (n_elements(iterArgs) gt 0) then begin
  if (ptr_valid(Self.iterArgs)) then  $
    (*Self.iterArgs) = iterArgs else $
    Self.iterArgs = ptr_new(iterArgs)
endif
if (n_elements(modelFlag) gt 0) then begin
   if (modelFlag le 0) then begin
      print, 'Improper model selection!'
      return
   endif
   Self.modelFlag = modelFlag
   ;Self.startingFlag = 1
   void = Self->Model()
endif

if (n_elements(enableFlag) gt 0) then begin
  Self.enableFlag = enableFlag
  props = ['errFuncFlag','MaxnFE','nPop','epsilon','chisqThreshPercent']
  nProps = n_elements(props)
  for i=0,nProps-1 do Self->SetPropertyAttribute, props[i], sensitive=enableFlag
endif

if (n_elements(etc) gt 0) then Self->IDLitComponent::GetProperty, _EXTRA=etc 

end

;===============================================================================
;+
; PAN_EPSDE::Fit
;
; PURPOSE:
;   Fit a model function using a Differential evolution algorithm with an 
;   Ensemble of Parameters and Mutation Strategies
;
; PARAMETERS:
;
; KEYWORDS:
;  
;-
function PAN_EPSDE::Fit, modelFlag=modelFlag, _EXTRA=etc
compile_opt idl2


; Set and Retrieve info about the model
if (keyword_set(modelFlag)) then Self->SetProperty, modelflag=modelFlag
;if (Self.startingFlag) then void = Self->Model()
nPar = Self.nPar    ; nos of parameters in the model
pMin = (*Self.pMinPtr)    ; 1D array of size nPar containing the minimum range for each parameter
pMax = (*Self.pMaxPtr)    ; 1D array of size nPar containing the maximum range for each parameter
nPop = nPar * Self.nPop     ; the population size; ie each parameter is sampled nPop times in each iteration/generation
tiedParFlag = Self.tiedParFlag  ; are tied parameters present?
varyingParIndex = where(pMin ne pMax, nVaryingPars) ; which paraeters are being fitted ie not fixed

if (tiedParFlag) then begin
  pTied = (*Self.pTiedPtr)    ; 1D array containing 1 | 0 indicating which parameters are tied to each other
  index = where(pTied, nTied)
  firstTiedIndex = index[0]         ; this par will be fitted (allowed to vary)
  otherTiedIndex = index[1:nTied-1] ; these will always be set to the firstTiedIndex
endif

; Create initial population of size (nPar x nPop)
; Made up of nPop vectors, each containing nPar elements
; Each element is randomly generated in the range [pmin,pmax]
uVec = dblarr(nPop) + 1.0
xp = (pMin#uVec) + randomu(systime(/seconds),nPar,nPop)*(pMax#uVec - pMin#uVec)
if (tiedParFlag) then begin
  for i=1,nTied-1 do xp[otherTiedIndex[i-1],*] = xp[firstTiedIndex,*] ; take care of tied parameters
endif

; Randomly assign nPop
; - mutation strategies out of a choice of 3
; - scale factors for scaling the difference vector and
; - cross-overs values: determine how trial vector chosen from a pair of target/mutant vectors 
CR = (*Self.crPtr)
SF = (*Self.sfPtr)
nCR = n_elements(CR)
nSF = n_elements(SF)
scsPool = fltarr(3,nPop)
scsPool[0,*] = fix(3*randomu(systime(/seconds),nPop))+1
scsPool[1,*] = CR[fix(nCR*randomu(systime(/seconds),nPop))]
scsPool[2,*] = SF[fix(nSF*randomu(systime(/seconds),nPop))]


; Find best vector and objective value from initial population
Self.nFE = 0   ; reset to 0
bestIndex = 0
bestfValue = Self->Evaluate(xp[*,0])
fValues = dblarr(nPop)
fValues[0] = bestfValue
for i = 1,nPop-1 do begin
   fValues[i] = Self->Evaluate(xp[*,i])
   if (fValues[i] lt bestfValue) then begin
      bestfValue = fValues[i]
      bestIndex = i
   endif
endfor
bestXp = xp[*,bestIndex]
void = moment(fValues,mean=fValue_mean,sdev=fValue_sdev)
deltafValue = fValue_sdev/fValue_mean
print,' Sdev/mean = ',deltafValue, fValue_sdev, fValue_mean
if (ptr_valid(Self.xpInitPtr)) then $
   (*Self.xpInitPtr) = bestXp else $
   Self.xpInitPtr = ptr_new(bestXp)
if (ptr_valid(Self.xpBestPtr)) then $
   (*Self.xpBestPtr) = bestXp  else $
   Self.xpBestPtr = ptr_new(bestXp)

;if (ptr_valid(Self.xpBest_gen)) then ptr_free, Self.xpBest_gen
;Self.xpBest_gen = ptr_new([])
;if (ptr_valid(Self.xpBestfValue_gen)) then ptr_free, Self.xpBestfValue_gen
;Self.xpBestfValue_gen = ptr_new([])
;if (ptr_valid(Self.xpPool)) then ptr_free, Self.xpPool
;Self.xpPool = ptr_new([])
;if (ptr_valid(Self.xpfValuePool)) then ptr_free, Self.xpfValuePool
;Self.xpfValuePool = ptr_new([])
(*Self.xpBest_gen) = []
(*Self.xpBestfValue_gen) = []
(*Self.xpPool) = []
(*Self.xpfValuePool) = []


Self.generationCnt = 1
scsPoolBreed = []
RR = intarr(nPop) - 1
rate = []
prevGenfValue = bestfValue
;deltafValue = bestfValue
while ((Self.nFE lt Self.nFEMax) && (deltafValue gt Self.epsilon )) do begin  ;|| bestfValue gt Self.bestFValThreshold)
   ;; Note: "bestfValue gt Self.bestFValThreshold" condition prevents premature exiting from the loop b/c "deltafValue lt Self.epsilon"
   ;; Essentially, if bestfvalue (or chsq) is larger than the threshold, carry on until the "Self.nFE ge Self.nFEMax" condition
   nscsPoolBreed = (size(scsPoolBreed))[2]
   if (Self.generationCnt gt 1) then begin
      ; Specify mutation strategies, CR and SF pool for this generation
      
      randk = randomu(systime(/seconds), nPop)
      for k=0,nPop-1 do begin
         if (isa(scsPoolBreed) && (randk[k] le rateMean)) then begin
            ; select a random integer index in range [0:n] where n is the 2nd dim length of scsPoolBreed 
            index = fix(randomu(rrSeed)*nscsPoolBreed)
            RR[k] = index ;fix(randomu(systime(/seconds))*nscsPoolBreed)
            
            ; randomly select from the successful (S,CR,SF) pool and 
            ; replace the current k of the (S,CR,SF) pool
            scsPool[*,k] = scsPoolBreed[*,index]
;;TODO       should the 'index' variable be different rands for RR and scsPool? 
         endif else begin
            RR[k] = -1
            ; make a new (S,CR,SF) set to replace the current k
            scsPool[0,k] = fix(3*randomu(sSeed))+1
            scsPool[1,k] = CR[fix(nCR*randomu(crSeed))]
            scsPool[2,k] = SF[fix(nSF*randomu(sfSeed))]
         endelse
      
      endfor
   endif

   RRR = []
   cntBadTrial = 0
   void = where(RR ge 0, RRisDefined)
   xpOld = xp  ; xpOld is the current pop while xp is the newly 
               ; emerging population for the next generation

   ; loop through the population
   for i=0,nPop-1 do begin
      ; the CR previously allocated randomly
      CRi = scsPool[1,i]    
      
      ; for SF, get the SF previously allocated. Generate a normal
      ; distribution of nPar SF values with a mean set to the given SF and
      ; a std dev of 0.001
      SFdist = randomn(sfdSeed,/normal,nPar)*0.001 + scsPool[2,i]
      
      
      ; Each vector consists of nPar individuals. In generating the trial vector, 
      ; CR controls whether individuals are selected from the current vector or the
      ; mutant vector based on the cross over rules. Individuals are selected based on
      ; the value of a random nos relative to the value of CR.
      ; Generate nPar rand nos between [0,1] and determine those that are less
      ; than or equal to the current CR. Where this is the case, select individuals
      ; from the mutant vector else select from the current vector.
      fromMutant = randomu(mSeed,nPar) le CRi   ; 1's and 0's; 1 ==> mutant individual to be included in next trial
      if (total(fromMutant) eq 0) then begin
         ;==> all the random nos were gt CR (thus the trial will equal the current vector!)
         ; in this case, simply generate a random nos between 1 and nPar and force the
         ; trial individual matching the random nos to come from the mutant
         index = fix(randomu(jSeed)*nPar)
         fromMutant[index] = 1
      endif
      fromXp = fromMutant ne 1   ; a 1 ==> current individual to be included in next trial vector
      
      jrnd = fix(randomu(iSeed,nPop)*nPop) ; randomly select nPop indices from 0 to nPop-1
      case scsPool[0,i] of    ; the mutant strategy
         1: begin ; DE/best/2
            ; Vi = Xbest + SF*(Xr1 - Xr2) + SF*(Xr3 - Xr4)
            ; Ui = Vi if rand[0,1] < CR for j=1,nPar || when randj eq j
            ; Ui = Xi otherwise
            mutantXp = bestXp + SFdist*( xpOld[*,jrnd[0]] - xpOld[*,jrnd[1]] $
                                       + xpOld[*,jrnd[2]] - xpOld[*,jrnd[3]] ) 
            trialXp = xpOld[*,i] * fromXp  +  mutantXp * fromMutant
         end
         
         2: begin ; DE/rand/1
            ; Vi = Xr1 + SF*(Xr2 - Xr3)
            ; Ui = Vi if rand[0,1] < CR for j=1,nPar || when randj eq j
            ; Ui = Xi otherwise
            mutantXp = xpOld[*,jrnd[0]] + SFdist*( xpOld[*,jrnd[1]] - xpOld[*,jrnd[2]])
            trialXp = xpOld[*,i] * fromXp  +  mutantXp * fromMutant
         end
         
         3: begin ; DE/current-to-rand/1
            ; Ui = Xi + K*(Xr1 - Xi) + SF*(Xr2 - Xr3) where k=rand[0,1]
            kk = randomu(kkSeed)
            trialXp = xpOld[*,i] + kk*(xpOld[*,jrnd[0]] - xpOld[*,i]) $
                             + SFdist*(xpOld[*,jrnd[1]] - xpOld[*,jrnd[2]])
         end
         
         else:
      endcase
      
      
      ; Ensure the trial vector is within parameter bounds
      if (total((trialXp lt pMin) or (trialXp gt pMax)) gt 0.0) then begin
         ; => at least one individual is outside bounds 
         ;    so generate a new trial vector using random individuals within bounds
         trialxp = pMin + randomu(pSeed,nPar)*(pMax - pMin)
      endif
      
      ; evaluate trial vector
      if (tiedParFlag) then trialXp[otherTiedIndex] = trialXp[firstTiedIndex] ; take care of tied parameters
      fValue = Self->Evaluate(trialXp)
      if (fValue lt fValues[i]) then begin
         xp[*,i] = trialXp
         fValues[i] = fValue
         if (fValue lt bestfValue) then begin
            bestfValue = fValue
            bestIndex = i
            bestXp = trialXp
         endif
         ; RR contains nPop random indices that were selected whenever an (S,CR,SF) set
         ; was carried over from a previous generation. Whenever a successful trial
         ; vector is found, the current RR index is recorded in RRR
         if (RRisDefined gt 0) then begin
            if (RR[i] ge 0) then RRR = [RRR,RR[i]]
         endif
         ; Store (S,CR,SF) sets that lead to a successful trial vector into scsPoolBreed
         if (isa(scsPoolBreed)) then $
            scsPoolBreed = transpose([transpose(scsPool[*,i]),transpose(scsPoolBreed)]) else $
            scsPoolBreed = scsPool[*,i]
      endif else $
         cntBadTrial++    
   endfor
   
   void = moment(fValues,mean=fValue_mean,sdev=fValue_sdev)
   deltafValue = fValue_sdev/fValue_mean
   print,' Sdev/mean = ',deltafValue, fValue_sdev, fValue_mean

   ; delete entries from scsPoolBreed whose indices match RRR
   if (isa(RRR)) then begin
      scsPoolBreed[0,RRR] = -1
      index = where(scsPoolBreed[0,*] gt 0,cnt)
      if (cnt gt 0) then scsPoolBreed = scsPoolBreed[*,index]
   endif
   
   ; record ratio of unchanged vectors in population in this generation
   rate = [rate,float(cntBadTrial)/nPop]
   ; Calculate mean rate for last 10 records
   n = n_elements(rate)
   rateMean = (n gt 10)? mean(rate[n-10:n-1]) : mean(rate)
   
   (*Self.xpBestPtr) = bestXp
   Self.xpbestfValue = bestfValue
;   if (bestfValue ne prevGenfValue) then  deltafValue = abs(bestfValue - prevGenfValue)
   prevGenfValue = bestfValue
   
   (*Self.xpBest_gen) = [(*Self.xpBest_gen),transpose(bestXp)]
   (*Self.xpBestfValue_gen) = [(*Self.xpBestfValue_gen),bestfValue]
   
;   ;; require a minimum of 20 generations to begin evaluating deltafValue condition
;   if (Self.generationCnt ge 20) then begin
;      ng = Self.generationCnt
;      deltafValue = abs((*Self.xpBestfValue_gen)[ng-20] - (*Self.xpBestfValue_gen)[ng-19])
;      for j = ng-19, ng-2 do $
;         deltafValue = deltafValue > abs((*Self.xpBestfValue_gen)[j] - (*Self.xpBestfValue_gen)[j+1])     
;   endif else begin
;      deltafValue = 1.0
;   endelse

   Self.generationCnt = Self.generationCnt + 1
   
   Self->RefreshPropSheet
   
   ; Check for interrupt request by user
   call_procedure, Self.iterProc, _Extra=(*Self.iterArgs), interrupt=interrupt
   ;print, 'In generation loop: Value of interrupt button is = ',interrupt
   if (interrupt eq 1) then begin
     print,'Exiting because "Interrupt" button was pressed...'
     return, 0
   endif
endwhile

; Estimate error in fitted parameters
; This is a compromise procedure taken from Wormington et al Phil. Trans. R. Soc. Lond. A (1999) vol 357, 2827-2848
; Calculate the the pointwise error for each parameter taken around their best-fit values.
; The pointwise error is defined as the change of the parameter that increases the error 
; function (chisq) by a specified amount, with all other parameters kept constant at their 
; best-fit values. This increase is specified, by default, to be 5% of the best chisq but is
; left as a user-specified value in the Self.chisqThreshPercent variable.
chisqThreshPercent = Self.chisqThreshPercent
fitParameters = bestXp
fitChiSq = bestfValue
chisqThresh = (1 + 0.01*chisqThreshPercent) * bestfValue 
parErrors = bestXp * 0.0
for i = 0,nVaryingPars-1 do begin
  j = varyingParIndex[i]
  if (tiedParFlag && otherTiedIndex.hasValue(j)) then continue ; don't error for parameters that are tied to another
  fitPar = fitParameters[j]

  parValues = (*Self.xpBest_gen)[*,j]
  deltaP = double((max(parValues) - min(parValues)))/100.0    ; an estimate of step-size used for locating the pointwise error
                                                              ; assume the search space is within the range searched previously
                                                              ; for the best fit parameter.
  deltaP = 0.01*double(pmax[j] - pmin[j])
  ; search error for positive deviations of parameter
  found = 0
  trialXp = fitParameters
  parValue = fitPar
  count = 0
  errorPlus = 0.0
  voidPPlus = []
  voidCsqPlus = []
  while (~ found && (count le 500) ) do begin ; use count counter to prevent runaway condition
    count++
    parValue += deltaP      ; +ve chane from best fit value
    trialXp[j] = parValue 
    fValue = Self->Evaluate(trialXp)  ; calculate error function
    voidPPlus = [voidPPlus,parValue]
    voidCsqPlus = [voidCsqPlus,fValue]
    if (fValue gt chisqThresh) then begin
      found = 1
      errorPlus = parValue - fitParameters[j]
    endif
  endwhile

  ; search error for negative deviations of parameter
  found = 0
  trialXp = fitParameters
  parValue = fitPar
  count = 0
  errorMinus = 0.0
  voidPMin = []
  voidCsqMin = []
  while (~ found && (count le 500) ) do begin ; use count counter to prevent runaway condition
    count++
    parValue -= deltaP      ; -ve chane from best fit value
    trialXp[j] = parValue 
    fValue = Self->Evaluate(trialXp)  ; calculate error function
    voidPMin = [voidPMin,parValue]
    voidCsqMin = [voidCsqMin,fValue]
    if (fValue gt chisqThresh) then begin
      found = 1
      errorMinus = fitParameters[j] - parValue
    endif
  endwhile
  voidP = [voidPMin,voidPPlus]
  voidC = [voidCsqMin,voidCsqPlus]
  ;v1 = plot(voidP,voidC)
  parErrors[j] = 0.5*(errorPlus + errorMinus)

endfor

if (tiedParFlag) then parErrors[otherTiedIndex] = parErrors[firstTiedIndex] ; handle any tied parameters
(*Self.xpBestErrorPtr) = parErrors


;            ,xpBest_gen:ptr_new()    $
;            ,xpBestfValue_gen:ptr_new() $
;            ,xpPool:ptr_new()        $
;            ,xpfValuePool:ptr_new()  $


return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
;+
; PAN_EPSDE::Init 
;
; PURPOSE:
;   Initialization method
;
; PARAMETERS:
;
; KEYWORDS:
; 
;   errFuncFlag - the error function calculation to use
;   
;   nPop  - set if the data specified by the dataObject keyword is to be reloaded
;   
;   nMS   - the number of mutation strategies available
;   
;   model - an object reference to the model function to be fitted
;   
;   epsilon  - another terminating condition: Stop fit when change in the objective function is less than epsilon.
;   
;   nFEMax - maximum nos of function evaluations before terminating fit
;   
;   fitFuncName - string specifying procedure name that will be called to evaluate the function to be fitted
;   
;   fitFuncArgs - keyword parameters to be passed to the fitfunc
;   
;   iterProc - (as in mpfit.pro) The name of a procedure to be called upon each NPRINT
;              iteration of the MPFIT routine.  ITERPROC is always
;              called in the final iteration.  It should be declared
;              in the following way:
;
;              PRO ITERPROC, MYFUNCT, p, iter, fnorm, FUNCTARGS=fcnargs, $
;                PARINFO=parinfo, QUIET=quiet, DOF=dof, PFORMAT=pformat, $
;                UNIT=unit, ...
;                ; perform custom iteration update
;              END
;              
;   iterArgs - (as in mpfit.pro) The keyword arguments to be passed to ITERPROC via the
;              _EXTRA mechanism.  This should be a structure, and is
;              similar in operation to FUNCTARGS.
;              Default: no arguments are passed.
;-
function PAN_EPSDE::Init, errFuncFlag=errFuncFlag, nPop=nPop, nFEMax=nFEMax, epsilon=epsilon $
                    , fitFuncName=fitFuncName, fitFuncArgs=fitFuncArgs $
                    , iterArgs=iterArgs, iterProc=iterProc $
                    , model=model, modelFlag=modelFlag, nMS=nMS, _REF_EXTRA=etc
                    
compile_opt idl2

Self.pMinPtr = ptr_new(0)
Self.nPar = n_elements(0)
Self.pMaxPtr = ptr_new(0)
Self.pTiedPtr = Ptr_new(0)
Self.tiedParFlag = 0
Self.xDataPtr = ptr_new(0)
Self.yDataPtr = ptr_new(0)
Self.calcPtr = ptr_new(0)
Self.yErrorPtr = ptr_new(0)

Self.xpBestErrorPtr = ptr_new([])
Self.xpBest_gen = Ptr_new([])
Self.xpBestfValue_gen = Ptr_new([])
Self.xpPool = Ptr_new([])
Self.xpfValuePool = Ptr_new([])


;; Call our super classes
if (~self->IDLitComponent::Init(_EXTRA=etc)) then $
  return, 0
if (~self->IDL_Container::Init()) then $
  return, 0
if (~self->IDL_Object::Init()) then $
  return, 0


if (~keyword_set(errFuncFlag)) then errFuncFlag = 0
Self.errFuncFlag = errFuncFlag

if (~keyword_set(modelFlag)) then modelFlag = 0
Self.modelFlag = modelFlag

if (~keyword_set(nPop)) then nPop = 10
Self.nPop = nPop

if (~keyword_set(nFEMax)) then nFEMax = 10000
Self.nFEMax = nFEMax

if (~keyword_set(epsilon)) then epsilon = 0.03
Self.epsilon = epsilon

if (~keyword_set(model)) then model = obj_new()
Self.model = model

Self.crPtr = ptr_new([0.1,0.2,0.3,0.4,0.5,0.6,0.7,0.9,0.9])

Self.sfPtr = ptr_new([0.4,0.5,0.6,0.7,0.9,0.9])

if (~keyword_set(nMS)) then nMS = 3
Self.nMS = nMS

;Self.bestFValThreshold = 2.0  ; The m

;Self.startingFlag = 1
Self.generationCnt = 0
Self.name = "Differential Evolution Algorithm"
Self->SetPropertyAttribute, 'NAME', /hide ;, name='Differential Evolution Algorithm'
Self->SetPropertyAttribute, 'DESCRIPTION', /hide

Self->SetProperty, fitFuncName=fitFuncName, fitFuncArgs=fitFuncArgs, iterArgs=iterArgs, iterProc=iterProc, _EXTRA=etc

Self.enableFlag = 0
Self->RegisterProperty, 'enableFlag', enumlist=['No','Yes'], name='Enable DE Algorithm?'
Self.errFuncFlag = 0
errfunclist = ['Mean square error (MSE)' $
              ,'Mean-abosolute error (MAE)' $
              ,'MSE of log transformed data' $
              ,'MAE of log transformed data']
Self->RegisterProperty, 'errFuncFlag', enumlist=errfunclist, name='Method of Evaluating ChiSq', sensitive=Self.enableFlag

Self->RegisterProperty, 'nPop', /integer, name='Population factor (x nos params)' $
                      ,description='Population Size', sensitive=Self.enableFlag
Self->RegisterProperty, 'MaxnFE', /float, name='Max function evaluations' $
                      ,description='Max function evaluations', sensitive=Self.enableFlag
Self.chisqThreshPercent = 5.0
Self->RegisterProperty, 'chisqThreshPercent', /float, name='Param Error: ChiSq Threshold (%)' $  ; 'ChiSq Thresh. for param error (%)'
                      ,description='Cost function value', sensitive=0
;Self->Registerproperty, 'bestFValThreshold', /float, name='ChiSq Threshold' $
;  ,description='Cost function value', sensitive=0
Self->RegisterProperty, 'epsilon', /float, name='Terminating ChiSq Tolerance' $
                      ,description='Terminating delta', sensitive=Self.enableFlag
Self->RegisterProperty, 'nFE', /integer, name='Function Evaluation Count' $
                      ,description='Function Evaluation Count', sensitive=0
Self->RegisterProperty, 'generationCnt', /integer, name='Generation Count' $
                      ,description='Generation Count', sensitive=0
Self->RegisterProperty, 'bestFValue', /float, name='Current ChiSq' $
                      ,description='Cost function value', sensitive=0

return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; Handle events generated by the propertysheet widget that displays the
; registered properties of this object
pro PAN_EPSDE::RefreshPropSheet
compile_opt idl2

if (widget_info(Self.wPS, /valid)) then widget_control, Self.wPS, /refresh

end
;-------------------------------------------------------------------------------


;===============================================================================
; Handle events generated by the propertysheet widget that displays the
; registered properties of this object
pro PAN_EPSDE::PropSheet_Events, event
compile_opt idl2

; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'PAN_EPSDE::PropSheet_Eventst: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif

;;******************************
;-- Record change to prop sheet
;-- Act on change to prop sheet
case tag_names(event, /structure_name) of

  'WIDGET_PROPSHEET_CHANGE': begin    
     ; Get the value of the changed property
     if (event.proptype eq 0) then begin
        ;; ==> a userdef property type so handle it appropriately
        ;status = oTool->EditUserDefProperty(oTool, event.identifier)
     endif else begin
        ; Update the property with the tool (event.component)
        value = widget_info(event.id, component=event.component,property_value=event.identifier)
        ; Set the component's property value. 
        event.component->SetPropertyByIdentifier, event.identifier, value 
     endelse
     widget_control, event.id, /refresh_property
  end
  
  else:
endcase

end
;-------------------------------------------------------------------------------


;===============================================================================
pro PAN_EPSDE__define
   struct = {PAN_EPSDE $
            ,inherits IDLitComponent $
            ,inherits IDL_Container  $
            ,inherits IDL_Object $
             
            ,model:obj_new()         $    ; the model function to be fitted
            ,modelFlag:0             $    ; which build-in function to use
            ,errFuncFlag:0           $    ; specify which error function calculation to use
            ,enableFlag:0            $    ; specifies an object of class is to be used or not in PAN
            ,startingFlag:0          $    ; a control flag
            ,nPar:0                  $    ; nos of parameters
            ,pMinPtr:ptr_new()       $    ; parameters bounds - minimum
            ,pMaxPtr:ptr_new()       $    ; parameters bounds - maximum
            ,pTiedPtr:ptr_new()      $    ; list of parameters tied to each other
            ,tiedParFlag:0           $    ; indicate whether there are tied parameters set
            ,pIdealPtr:ptr_new()     $    ; ideal best fir parameters (where available for test model/data)
            ,pIdealErrorPtr:ptr_new()$    ; ideal best fit parameter errors (where available for test model/data)
            ,xDataPtr:ptr_new()      $    ; x data values to be fitted
            ,yDataPtr:ptr_new()      $    ; y data values (dependent data) to be fitted
            ,yErrorPtr:ptr_new()     $    ; error in y values 
            ,calcPtr:ptr_new()       $    ; latest calculated dependent data values
            
            ,wPS:0L                  $    ; widget ID of propertysheet widget that displays props of instance of class
            
            ,nPop:0L                 $    ; population size
            ,nFEMax:0L               $    ; maximum nos of function evaluations before terminating
            ,epsilon:0.0D            $    ; another terminating condition: improvements in the cost function is less than epsilon.
            ,crPtr:ptr_new()         $    ; cross over pool of values
            ,sfPtr:ptr_new()         $    ; scale factor pool of values
            ,nMS:0L                  $    ; nos of mutation strategies in use
            ,msFlagPtr:ptr_new()     $    ; flags indicating mutation strategy pool
            ,generationCnt:0L        $    ; nos of generations to date
            
;            ,xpNewGenPtr:ptr_new()   $    ; the target vectors (new generation population)
;            ,xpOldGenPtr:ptr_new()   $    ; the target vectors (old/current generation population)
;            ,vpPtr:ptr_new()         $    ; the mutation vectors used to determine the trial vectors
;            ,upPtr:ptr_new()         $    ; the trial vectors used to calculate the next generation (xpNewGenPtr)

            ,xpInitPtr:ptr_new()     $    ; the initial vector used at the start
            ,xpBestPtr:ptr_new()     $    ; the best vector found so far
            ,xpBestErrorPtr:Ptr_new()$    ; the error in the best vector (evaluated at the end of the fit)
            ,xpbestfValue:0.0D       $    ; value of cost function evaluated using xpBest, the best parameter
            ,bestFValThreshold:0.0   $    ; minimum required value of cost function before terminating fitting loop
            ,nFE:0L                  $    ; current count of function evaluations
            
            ,chisqThreshPercent:0.0  $    ; Threshold as persentage of chisq that is used to calculate pointwise error
                                          ; for fitted parameters. Ie the error is the delta by which the parameter
                                          ; has to change for the minimum chisq to increase in percentage terms by
                                          ; chisqThreshPercent.
            
            ,xpBest_gen:ptr_new()    $
            ,xpBestfValue_gen:ptr_new() $
            ,xpPool:ptr_new()        $
            ,xpfValuePool:ptr_new()  $
            
            ,fitFuncName:''            $    ; external fit function name
            ,fitFuncArgs:ptr_new()     $    ; keyword args to be used with external fit function
            ,iterProc:''            $    ; external fit function name
            ,iterArgs:ptr_new()     $    ; keyword args to be used with external fit function
           }
end
