; $Id$
;###############################################################################
;
; NAME:
;  DTAS_REDUCE__DEFINE
;
; PURPOSE:
;  Class for TAS reduction.
;
; CATEGORY:
;  DAVE, TAS, Data Reduction
;
; AUTHOR:
;   Robert M. Dimeo, Ph.D.
;   NIST Center for Neutron Research
;   100 Bureau Drive
;   Gaithersburg, MD 20899
;   Phone: (301) 975-8135
;   E-mail: robert.dimeo@nist.gov
;   http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
; +
;
; -
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro dtas_reduce::cleanup
nobj = self.intermediate->count()
if nobj gt 0 then begin
   oall = self.intermediate->get(/all)
   obj_destroy,oall
endif
nsignal = self.signal_container->count()
nempty = self.empty_container->count()
nreduced = self.reduced_container->count()
nint = self.intermediate->count()
nzip = self.zip_container->count()
ncombine = self.combine_container->count()
if ncombine gt 0 then begin
   oall = self.combine_container->get(/all)
   obj_destroy,oall
endif
if nint gt 0 then begin
   oall = self.intermediate->get(/all)
   obj_destroy,oall
endif
if nzip gt 0 then begin
   oall = self.zip_container->get(/all)
   obj_destroy,oall
endif
if nsignal gt 0 then begin
   oall = self.signal_container->get(/all)
   obj_destroy,oall
endif
if nempty gt 0 then begin
   oall = self.empty_container->get(/all)
   obj_destroy,oall
endif
if nreduced gt 0 then begin
   oall = self.reduced_container->get(/all)
   obj_destroy,oall
endif
n_here = self->count()
if n_here gt 0 then begin
   oall = self->get(/all)
   obj_destroy,oall
endif
obj_destroy,[self.intermediate,self.signal_container,self.empty_container, $
   self.reduced_container,self.zip_container,self.combine_container]
self->IDL_CONTAINER::cleanup
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::destroy_element,o
if not obj_valid(o) then return,0
ret = o->get_property(signal_type = signal_type)
case signal_type of
'SIGNAL': $
   begin
      self.signal_container->remove,o
      obj_destroy,o
   end
'INTERMEDIATE': $
   begin
      self.intermediate->remove,o
      obj_destroy,o
   end
'EMPTY': $
   begin
      self.empty_container->remove,o
      obj_destroy,o
   end
'REDUCED': $
   begin
      self.reduced_container->remove,o
      obj_destroy,o
   end
'COMBINED': $
   begin
      self.combine_container->remove,o
      obj_destroy,o
   end
else:
endcase
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::update_plot_structure,oadd,DISPLAY = display,_Extra = extra
if n_params() eq 0 then return,0
if not obj_valid(oadd) then return,0
; Updates the plottable data in the object
; Get out the plot structure
ret = oadd->get_property(plot_str = plot_str,data = data,var_names = var_names)
xvar_name = plot_str.xtitle
x_index = where(var_names eq xvar_name,count_x)
if count_x gt 0 then begin
   *plot_str.x = *data.(x_index[0])
endif
nvars = n_elements(var_names)
*plot_str.y = *data.(nvars-2)
*plot_str.dy = *data.(nvars-1)
ret = oadd->set_property(plot_str = plot_str)
if n_elements(display) ne 0 then $
   ret = oadd->display(psym = 4,_Extra = extra)
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::sum_poisson,msg = msg, signal = signal, empty = empty, o = o,data_objects=data_objects

;if keyword_set(empty) then ocont = self.empty_container
;if keyword_set(signal) then ocont = self.signal_container
;msg = ''
;n_obj = ocont->count()

n_obj=n_elements(data_objects)
if n_obj lt 1 then begin
   msg = 'At least two files are required in the summation'
   return,0
endif
if n_obj eq 1 then begin
   msg = 'Only one data file present'
   oall = ocont->get(/all)
   o = oall[0]->clone()
   oadd = oall[0]->clone()
   return,oadd
endif
; Check if the independent variables match up
oall = data_objects ;ocont->get(/all)
for i = 0,n_obj-1 do begin
   ret = oall[i]->get_property(plot_str = plot_str)
   if i eq 0 then xtitle = plot_str.xtitle else $
      xtitle = [xtitle,plot_str.xtitle]
endfor
if n_elements(uniq(xtitle)) ne 1 then begin
   msg = 'The independent variables do not match up!'
   return,0
endif
ret = oall[0]->get_property(data = data,total_mon = m0,  $
      plot_str = plot_str, var_names = var_names)
if m0 eq 0 then begin
   msg = 'No monitor counts have been defined for this file'
   return,0
endif
baseline_mon = m0
nvars = n_elements(var_names)
var_ptr = ptrarr(nvars,/allocate_heap)

for i = 0,n_elements(oall)-1 do begin
   ret = oall[i]->get_property(data = data,total_mon = m0,  $
         plot_str = plot_str, var_names = var_names,filename = filename)
   if i eq 0 then files_to_sum = filename else $
      files_to_sum = [files_to_sum,filename]
   if i eq 0 then begin
      for j = 0,nvars-1 do begin
         *var_ptr[j] = *data.(j)
      endfor

   endif else begin
      sf = double(baseline_mon)/double(m0)
      for j = 0,nvars-1 do begin
         if (j eq (nvars-1)) or (j eq (nvars-2)) then begin
            ; Must scale the intensity and error bars by the monitor
            ; scale factor
            *var_ptr[j] = [*var_ptr[j],sf*(*data.(j))]
         endif else begin
            *var_ptr[j] = [*var_ptr[j],*data.(j)]
         endelse
      endfor

   endelse
endfor

; Create a data structure consistent with the one in DTAS_DATA class
oadd = oall[0]->clone()
ret = oadd->get_property(data = data,time = time)
for i = 0,nvars-1 do begin
   *data.(i) = *var_ptr[i]
endfor
; Get the independent variable out
xvar_name = plot_str.xtitle
x_index = where(strupcase(var_names) eq strupcase(xvar_name),count_index)
if count_index eq 0 then return,0

ret = oadd->set_property(data = data)
ptr_free,var_ptr

y = reform(*data.(nvars-2))
dy = reform(*data.(nvars-1))

x = reform(*data.(x_index[0]))
; Convert data structure to a data array
darray = oadd->convert_data_to_array()
merge_str = merge_arrays(x,y,dy,uniq_index = uniq_index)
nx = n_elements(merge_str.x)
new_array = dblarr(nx,nvars)
; Fill in what we know first
new_array[*,x_index[0]] = merge_str.x
new_array[*,nvars-2] = merge_str.y
new_array[*,nvars-1] = merge_str.dy

; Now must merge the rest of the data array!
new_array[*,0] = dindgen(nx)
for i = 1,nvars-3 do begin
   if i ne x_index[0] then begin
      new_array[*,i] = darray[uniq_index,i]
   endif
endfor

; Now convert the updated data array to a structure and put it back
; in to the summed object
ret = oadd->convert_array_to_data(new_array)
ret = oadd->get_property(data = new_data)


; Add the summed data to the TAS data container and the reduced container
self->add,oadd
self.reduced_container->add,oadd
ret = self->update_plot_structure(oadd)
nfiles = n_elements(files_to_sum)
for i = 0,nfiles-1 do begin
   new_treatment = 'File added (Poisson): '+files_to_sum[i]
   ret = oadd->update_treatment(new_treatment)
endfor
ret = oadd->update_display_name('_add')
if arg_present(o) then o = oadd->clone()
return,oadd
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::check_same_tags,oarray
; This function checks to see if the tag names in the data structure are
; the same
nobj = n_elements(oarray)
if nobj eq 1 then return,1B
ret = oarray[0]->get_property(var_names = names)
n = n_elements(names)

for i = 1,nobj-1 do begin
   ret = oarray[i]->get_property(var_names = names)
   n1 = n_elements(names)
   if n1 ne n then return,0B
   n = n1
endfor

return,1B
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::remove_mismatch,oarray
; This function removes a single column from the data that cannot be found in
; the other data sets.
nobj = n_elements(oarray)
if nobj eq 1 then return,1B
; First go through each object and get out the variable names
index = -1
for i = 0,nobj-1 do begin
   ret = oarray[i]->get_property(var_names = names)
   if i eq 0 then all_names = names else $
      all_names = [all_names,names]
endfor

; What are the unique names?
uniq_names = all_names[uniq(all_names,sort(all_names))]

; Now find out which objects lack one of the elements in the unique
; name list
for i = 0,nobj-1 do begin
   ret = oarray[i]->get_property(var_names = names,data = this_data)
   help,*this_data.(0)
   nunique_names = n_elements(uniq_names)
   for j = 0,nunique_names-1 do begin
      this_name = uniq_names[j]
      match_index = where(names eq this_name,count_match)
      if count_match eq 0 then bad_var_name = this_name
   endfor
endfor
; Now go through each object and eliminate the data member whose
; tag matches bad_var_name
for i = 0,nobj-1 do begin
   ret = oarray[i]->get_property(data = data,var_names = var_names)
   bad_index = where(var_names eq bad_var_name,count_match)
   if count_match gt 0 then begin
      ; Create a new data structure without the offensive data member
      nvars = n_elements(var_names)
      var_ptr = ptrarr(nvars,/allocate_heap)

      s = create_struct(var_names[0],data.(0))
      for j = 1,nvars-1 do begin
         if j ne bad_index then begin
            s = create_struct(s,var_names[j],data.(j))
         endif
      endfor
      ret = oarray[i]->set_property(data = s)
   endif
endfor
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::sum_straight_old,msg = msg, signal = signal, empty = empty, $
   o = o
; This function performs a straight addition of raw counts and monitor counts
; for overlapping points and a simple merge for non-overlapping points (but
; the monitor values are updated).
if keyword_set(empty) then ocont = self.empty_container
if keyword_set(signal) then ocont = self.signal_container
msg = ''
n_obj = ocont->count()
if n_obj lt 1 then begin
   msg = 'At least two files are required in the summation'
   return,0
endif
if n_obj eq 1 then begin
   msg = 'Only one data file present'
   oall = ocont->get(/all)
   oadd = oall[0]->clone()
   o = oadd->clone()
   self->add,oadd
   return,oadd
endif
; Check if the independent variables match up
oall = ocont->get(/all)

; The following statement checks to see if there are identical
; tag names in the independent variables.
ret = self->check_same_tags(oall)
if ~ret then ret = self->remove_mismatch(oall)

for i = 0,n_obj-1 do begin
   ret = oall[i]->get_property(plot_str = plot_str)
   if i eq 0 then xtitle = plot_str.xtitle else $
      xtitle = [xtitle,plot_str.xtitle]
endfor
if n_elements(uniq(xtitle)) ne 1 then begin
   msg = 'The independent variables do not match up!'
   return,0
endif
ret = oall[0]->get_property(data = data,total_mon = m0,  $
      plot_str = plot_str, var_names = var_names)

if m0 eq 0 then begin
   msg = 'No monitor counts have been defined for this file'
   return,0
endif
baseline_mon = m0
nvars = n_elements(var_names)
var_ptr = ptrarr(nvars,/allocate_heap)

for i = 0,n_elements(oall)-1 do begin
   ret = oall[i]->get_property(data = data,total_mon = m0,  $
         plot_str = plot_str, var_names = var_names,        $
         treatment = treatment, filename = filename)
  ; for j = 0,n_elements(var_names)-1 do help,*data.(j)

;   if i gt 0 then begin
;      if i eq 1 then treatment_array = treatment else $
;         treatment_array = [treatment_array,treatment]
;   endif
   treatment_array = (i eq 0)? treatment : [treatment_array,treatment]

   if i eq 0 then begin
      files_to_sum = filename
      tags = tag_names(data)
      for j = 0,nvars-1 do begin
         *var_ptr[j] = *data.(j)
      endfor

   endif else begin
      files_to_sum = [files_to_sum,filename]
      sf = double(baseline_mon)/double(m0)
      tags = tag_names(data)
      mon_index = where(strupcase(tags) eq 'MON',count_mon)
      if (count_mon eq 0) then $ ; then try looking for MONITOR tag
        mon_index = where(strupcase(tags) eq 'MONITOR',count_mon)

      for j = 0,nvars-1 do begin
         if (j eq (nvars-1)) or (j eq (nvars-2)) or $
            (j eq mon_index) then begin
            *var_ptr[j] = [*var_ptr[j],(*data.(j))]
         endif else begin
            *var_ptr[j] = [*var_ptr[j],*data.(j)]
         endelse
      endfor

   endelse
endfor
; Create a data structure consistent with the one in DTAS_DATA class
oadd = oall[0]->clone()
ret = oadd->update_treatment(treatment_array)
ret = oadd->get_property(data = data,time = time)

for i = 0,nvars-1 do begin
   *data.(i) = *var_ptr[i]
endfor
; Get the independent variable out
xvar_name = plot_str.xtitle
x_index = where(strupcase(var_names) eq strupcase(xvar_name),count_index)
if count_index eq 0 then return,0

ret = oadd->set_property(data = data)
ptr_free,var_ptr
tags = strupcase(tag_names(data))
mon_index = where(tags eq 'MON',count_mon)
if (count_mon eq 0) then $      ; then try looking for MONITOR tag
  mon_index = where(tags eq 'MONITOR',count_mon)
  
time_index = where(tags eq 'MIN',count_time)
if (count_time eq 0) then $ ; then look for TIME tag 
  time_index = where(tags eq 'TIME',count_time)

mon = reform(*data.(mon_index)) & dmon = sqrt(mon)
time = reform(*data.(time_index)) & dtime = sqrt(time)
y = reform(*data.(nvars-2))
dy = reform(*data.(nvars-1))

x = reform(*data.(x_index[0]))
; Convert data structure to a data array
darray = oadd->convert_data_to_array()
; Now be very careful here.  We need the contents of x in the original order
; in which they were stored in the data pointer.  The function called
; MERGE_ARRAYS will change the values.  Must make a clean copy here and use
; it in the function call.
xclean = x
xin = xclean
merge_str = merge_arrays(xin,y,dy,uniq_index = uniq_index,/straight)
xin = xclean
mon_merge = merge_arrays(xin,mon,dmon,uniq_index = mon_uniq,/straight)
xin = xclean
time_merge = merge_arrays(xin,time,dtime,uniq_index = time_uniq,/straight)

nx = n_elements(merge_str.x)
new_array = dblarr(nx,nvars)

; Now must insert the rest of the data array!
new_array[*,0] = dindgen(nx)

;for i = 2,nvars-3 do begin
for i = 1,nvars-3 do begin      ; RTA - don't understand why starting from 2, change to 1
;   if ((i ne x_index[0])) then begin
      new_array[*,i] = darray[uniq_index,i]
;   endif
endfor

; Fill in what we know *after* - RTA
new_array[*,x_index[0]] = merge_str.x
new_array[*,nvars-2] = merge_str.y
new_array[*,nvars-1] = merge_str.dy
new_array[*,mon_index[0]] = mon_merge.y
new_array[*,time_index[0]] = time_merge.y


; Now convert the updated data array to a structure and put it back
; in to the summed object
ret = oadd->convert_array_to_data(new_array)
ret = oadd->get_property(data = new_data)


; Add the summed data to the TAS data container and the reduced container
self->add,oadd
self.reduced_container->add,oadd
ret = self->update_plot_structure(oadd)
nfiles = n_elements(files_to_sum)
if keyword_set(empty) then str = '(empty)'
if keyword_set(signal) then str = '(signal)'
for i = 0,nfiles-1 do begin
   new_treatment = str+'File added (straight): '+files_to_sum[i]
   ret = oadd->update_treatment(new_treatment)
endfor
ret = oadd->update_display_name('_add')
if arg_present(o) then o = oadd->clone()
return,oadd
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::sum_straight,msg = msg, signal = signal, empty = empty, $
   o = o,data_objects=data_objects
; This function performs a straight addition of raw counts and monitor counts
; for overlapping points and a simple merge for non-overlapping points (but
; the monitor values are updated).


;if keyword_set(empty) then ocont = self.empty_container
;if keyword_set(signal) then ocont = self.signal_container
;msg = ''
;n_obj = ocont->count()

n_obj=n_elements(data_objects)
if n_obj lt 1 then begin
   msg = 'At least two files are required in the summation'
   return,0
endif
if n_obj eq 1 then begin
   msg = 'Only one data file present'
   oall = data_objects[0]
   oadd = oall[0]->clone()
   o = oadd->clone()
   self->add,oadd
   return,oadd
endif
; Check if the independent variables match up
oall = data_objects ;ocont->get(/all)

; The following statement checks to see if there are identical
; tag names in the independent variables.
ret = self->check_same_tags(oall)
if ~ret then ret = self->remove_mismatch(oall)

for i = 0,n_obj-1 do begin
   ret = oall[i]->get_property(plot_str = plot_str)
   if i eq 0 then xtitle = plot_str.xtitle else $
      xtitle = [xtitle,plot_str.xtitle]
endfor
if n_elements(uniq(xtitle)) ne 1 then begin
   msg = 'The independent variables do not match up!'
   return,0
endif
ret = oall[0]->get_property(data = data,total_mon = m0,  $
      plot_str = plot_str, var_names = var_names)

if m0 eq 0 then begin
   msg = 'No monitor counts have been defined for this file'
   return,0
endif
baseline_mon = m0
nvars = n_elements(var_names)
var_ptr = ptrarr(nvars,/allocate_heap)

for i = 0,n_elements(oall)-1 do begin
   ret = oall[i]->get_property(data = data,total_mon = m0,  $
         plot_str = plot_str, var_names = var_names,        $
         treatment = treatment, filename = filename)
   treatment_array = (i eq 0)? treatment : [treatment_array,treatment]

   if i eq 0 then begin
      files_to_sum = filename
      tags = tag_names(data)
      for j = 0,nvars-1 do begin
         *var_ptr[j] = *data.(j)
      endfor

   endif else begin
      files_to_sum = [files_to_sum,filename]
      sf = double(baseline_mon)/double(m0)
      tags = tag_names(data)
      mon_index = where(strupcase(tags) eq 'MON',count_mon)
      if (count_mon eq 0) then $ ; then try looking for MONITOR tag
        mon_index = where(strupcase(tags) eq 'MONITOR',count_mon)

      for j = 0,nvars-1 do begin
         if (j eq (nvars-1)) or (j eq (nvars-2)) or $
            (j eq mon_index) then begin
            *var_ptr[j] = [*var_ptr[j],(*data.(j))]
         endif else begin
            *var_ptr[j] = [*var_ptr[j],*data.(j)]
         endelse
      endfor

   endelse
endfor
; Create a data structure consistent with the one in DTAS_DATA class
oadd = oall[0]->clone()
ret = oadd->update_treatment(treatment_array)
ret = oadd->get_property(data = data,time = time)

for i = 0,nvars-1 do begin
   *data.(i) = *var_ptr[i]
endfor
; Get the independent variable out
xvar_name = plot_str.xtitle
x_index = where(strupcase(var_names) eq strupcase(xvar_name),count_index)
if count_index eq 0 then return,0

ret = oadd->set_property(data = data)
ptr_free,var_ptr
tags = strupcase(tag_names(data))
mon_index = where(tags eq 'MON',count_mon)
if (count_mon eq 0) then $      ; then try looking for MONITOR tag
  mon_index = where(tags eq 'MONITOR',count_mon)
  
time_index = where(tags eq 'MIN',count_time)
if (count_time eq 0) then $ ; then look for TIME tag 
  time_index = where(tags eq 'TIME',count_time)

mon = reform(*data.(mon_index)) & dmon = sqrt(mon)
time = reform(*data.(time_index)) & dtime = sqrt(time)
y = reform(*data.(nvars-2))
dy = reform(*data.(nvars-1))

x = reform(*data.(x_index[0]))
; Convert data structure to a data array
darray = oadd->convert_data_to_array()
; Now be very careful here.  We need the contents of x in the original order
; in which they were stored in the data pointer.  The function called
; MERGE_ARRAYS will change the values.  Must make a clean copy here and use
; it in the function call.
xclean = x
xin = xclean
merge_str = merge_arrays(xin,y,dy,uniq_index = uniq_index,/straight)
xin = xclean
mon_merge = merge_arrays(xin,mon,dmon,uniq_index = mon_uniq,/straight)
xin = xclean
time_merge = merge_arrays(xin,time,dtime,uniq_index = time_uniq,/straight)

nx = n_elements(merge_str.x)
new_array = dblarr(nx,nvars)

; Now must insert the rest of the data array!
new_array[*,0] = dindgen(nx)

;for i = 2,nvars-3 do begin
for i = 1,nvars-3 do begin      ; RTA - don't understand why starting from 2, change to 1
;   if ((i ne x_index[0])) then begin
      new_array[*,i] = darray[uniq_index,i]
;   endif
endfor

; Fill in what we know *after* - RTA
new_array[*,x_index[0]] = merge_str.x
new_array[*,nvars-2] = merge_str.y
new_array[*,nvars-1] = merge_str.dy
new_array[*,mon_index[0]] = mon_merge.y
new_array[*,time_index[0]] = time_merge.y


; Now convert the updated data array to a structure and put it back
; in to the summed object
ret = oadd->convert_array_to_data(new_array)
ret = oadd->get_property(data = new_data)


; Add the summed data to the TAS data container and the reduced container
self->add,oadd
self.reduced_container->add,oadd
ret = self->update_plot_structure(oadd)
nfiles = n_elements(files_to_sum)
if keyword_set(empty) then str = '(empty)'
if keyword_set(signal) then str = '(signal)'
for i = 0,nfiles-1 do begin
   new_treatment = str+'File added (straight): '+files_to_sum[i]
   ret = oadd->update_treatment(new_treatment)
endfor
ret = oadd->update_display_name('_add')
if arg_present(o) then o = oadd->clone()
return,oadd
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::subtract_data,oadd,osubtract,msg = msg,o = o
; RTA - Make major changes. The basis of the subtracted results was
;       osubtraction, change this to oadd! It was assummed that the
;       monitor is always at column 2 in the data structure which is
;       not the case.

if n_params() ne 2 then begin
   msg = 'Two data objects must be passed in'
   return,0
endif
msg = ''

; The following statement checks to see if there are identical
; tag names in the independent variables.
ret = self->check_same_tags([oadd,osubtract])
if ~ret then ret = self->remove_mismatch([oadd,osubtract])

; First see if the independent variables match
ret = oadd->get_property(data = add_data,total_mon = m0,  $
   plot_str = plot_str, var_names = var_names)
nvars = n_elements(var_names)

if m0 eq 0 then begin
   msg = 'No monitor counts have been defined for this file'
   return,0
endif

; Get the INTENSITY and ERROR values out
y_baseline = reform(*add_data.(nvars-2))
dy_baseline = reform(*add_data.(nvars-1))
; Get the independent variable out
xvar_name = plot_str.xtitle
x_index = where(strupcase(var_names) eq strupcase(xvar_name),count_index)
if count_index eq 0 then return,0
x_index = x_index[0]

; ************************************************************
; Sort the "add" data out by x_index (added by RMD 05/18/04)
xsort = sort(*add_data.(x_index))
for i = 0,n_tags(add_data)-1 do $
   (*add_data.(i))[*] = (*add_data.(i))[xsort]
y_baseline = y_baseline[xsort]
dy_baseline = dy_baseline[xsort]
; ************************************************************

x_baseline = reform(*add_data.(x_index))

; Make a copy of the first TAS data object for use in the difference.

;-------------START CHANGE - RTA
; ** replace following lines
;mon_baseline = reform(*add_data.(1))
;oresult = osubtract->clone()
;ret = oadd->get_property(treatment = add_treatment,descriptr_str = descriptr_str)
;ret = oresult->set_private(treatment = add_treatment)
;ret = oresult->set_property(descriptr_str = descriptr_str)
;ret = osubtract->get_property(plot_str = plot_str,total_mon = mi, $
;      filename = filename,treatment = treatment,var_names = var_names, $
;      data = sub_data)
;ret = oresult->update_treatment(treatment)
;ret = oresult->get_property(data = result_data)
; * with
mon_baseline = oadd->get_monitor()
oresult = oadd->clone()
ret = osubtract->get_property(total_mon = mi,filename=sub_filename,data = sub_data)
ret = oresult->get_property(data = result_data)
;--------------END CHANGE - RTA


; ************************************************************
; Sort the data on the indexed variable (added by RMD 05/18/04)
x1sort = sort(*result_data.(x_index))
x2sort = sort(*sub_data.(x_index))
for i = 0,n_tags(result_data)-1 do begin
   *result_data.(i) = (*result_data.(i))[x1sort]
   *sub_data.(i) = (*sub_data.(i))[x2sort]
endfor
; ************************************************************

if mi eq 0 then begin
      obj_destroy,oresult
      msg = 'No monitor counts have been defined for this file'
      return,0
endif


xi = reform(*sub_data.(x_index))
; Now get the minimal range based on the intersection of x_baseline and xi
xb_min = min(x_baseline,max = xb_max)
xi_min = min(xi,max = xi_max)
xmin = max([xb_min,xi_min])
xmax = min([xb_max,xi_max])
where_xn_baseline = where((x_baseline ge xmin) and (x_baseline le xmax),count)
if count lt 2 then begin
   obj_destroy,oresult
   msg = 'There is insufficient overlap of the files used in the subtraction'
   return,0
endif
x_baseline = x_baseline[where_xn_baseline]

; *************************BUG FIX*************************************
; The following lines were added 2-10-04 by RMD.  These lines fix a bug
; in the original subtraction algorithm where data ranges were not
; adequately accounted for in the subtraction.
y_baseline = y_baseline[where_xn_baseline]
dy_baseline = dy_baseline[where_xn_baseline]
; ***********************END BUG FIX***********************************

xb_min = min(x_baseline,max = xb_max)

tol = 1e-3
ntags = n_tags(sub_data)
;if (total(xi-x_baseline) ne 0.0) or $
if (ABS(total(xi-x_baseline)) gt tol) or $
   (n_elements(xi) ne n_elements(x_baseline)) then begin
; Interpolate the values contained in osubtract to those of the independent
; variable in oadd:
   *result_data.(x_index) = x_baseline
   for i = 0,ntags-1 do begin
      if i ne x_index[0] then begin
         v = double(reform(*sub_data.(i)))
         x = xi
         u = x_baseline
         ok = where((x ge xb_min) and (x le xb_max))
         v = v[ok]
         x = x[ok]
         case self.interp of
         'BINNING':  $
            begin
               dv = sqrt(v)
               drebin,x,v,dv,u,vi,dvi,/points,/to_points,err = err,emsg = emsg
               ;if i eq (ntags-2) then dvi_out = dvi
            end
         'LINEAR':   $
            begin
               vi = interpol(v,x,u)
            end
         'SPLINE':   $
            begin
               vi = interpol(v,x,u,/spline)
            end
         endcase
         ;if ( (i eq (ntags-1)) and (self.interp eq 'BINNING') ) then begin
         ;   *sub_data.(i) = dvi_out
         ;endif else begin
            *sub_data.(i) = vi
         ;endelse
      endif
   endfor
endif ;else begin
;  ; no interpolation so simply copy data from subtract to result
;  (*result_data.(x_index)) = x_baseline
;  for i = 0,ntags-1 do begin
;    if (i ne x_index[0]) then (*result_data.(i)) = (*sub_data.(i))
;  endfor
;endelse

;-------------START CHANGE - RTA
; ** replace following lines
;moni = reform(*result_data.(1))
;; Update the monitor (RMD 03-17-04)
;mon_new = moni*weight
;ret = oresult->set_monitor(mon_new)
;yi = reform(*result_data.(nvars-2))
;dyi = reform(*result_data.(nvars-1))
; * with
;moni = osubtract->get_monitor()

tags = tag_names(result_data)
mon_index = where(strupcase(tags) eq 'MON',count_mon)
if (count_mon eq 0) then $      ; then try looking for MONITOR tag
  mon_index = where(strupcase(tags) eq 'MONITOR',count_mon)
if count_mon eq 0 then begin
  obj_destroy, oresult
  msg = 'No monitor counts defined for dataset'
  return,0
endif
moni = (*sub_data.(mon_index))


yi = reform(*sub_data.(nvars-2))  ; reform(*sub_data.(nvars-2))
dyi = reform(*sub_data.(nvars-1)) ; reform(*sub_data.(nvars-1))
;--------------END CHANGE - RTA


mon_baseline = mon_baseline[where_xn_baseline]
weight = double(mon_baseline)/double(moni)
y_baseline = y_baseline - weight*yi
dy_baseline = sqrt(dy_baseline^2+(weight*dyi)^2)


;new_treatment = strarr(2)
new_treatment = 'Empty subtracted from signal'
new_treatment = [new_treatment,'Empty file: '+sub_filename]
if self.interp eq 'BINNING' then $
   new_treatment = [new_treatment,self.interp+' used in subtraction'] else $
      new_treatment = [new_treatment,self.interp+' interpolation used in subtraction']

ret = oresult->update_treatment(new_treatment)

*result_data.(nvars-2) = y_baseline
*result_data.(nvars-1) = dy_baseline
*result_data.(mon_index) = mon_baseline ; RTA - update mon to reflect monitor counts of signal data; sub. mon has been included thro weight factor above
; Update the DATA for the object
ret = oresult->set_property(data = result_data)
; Add the summed data to the TAS data container and the reduced container
self->add,oresult
self.reduced_container->add,oresult
; Update the plot structure
ret = self->update_plot_structure(oresult)
;ret = oresult->update_display_name('_subtracted')


; Plot the data
if arg_present(o) then o = oresult->clone()
;-------------START CHANGE - RTA
; Copy the name of the "ADD" file to this one:
;ret = oadd->get_property(display_name = display_name, $
;   filename = filename)
;ret = oresult->set_property(display_name = display_name, $
;   filename = filename)
;ret = oresult->get_property(descriptr_str = d)
;--------------END CHANGE - RTA
ret = oresult->update_display_name('_subtracted')
return,oresult
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::set_property, INTERP = interp,  $
                                    XVAR_NAME = xvar_name
if n_elements(interp) ne 0 then self.interp = interp
if n_elements(xvar_name) ne 0 then begin
   nobj = self->count()
   if nobj ne 0 then begin
      oall = self->get(/all)
      for i = 0,nobj-1 do begin
         ret = oall[i]->set_property(xvar_name = xvar_name)
      endfor
   endif
endif
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::get_property, INTERMEDIATE = intermediate,  $
                                    REDUCED = reduced,            $
                                    SIGNAL = signal,              $
                                    INTERP = interp,              $
                                    COMBINED = combined,          $
                                    EMPTY = empty
combined = self.combine_container
interp = self.interp
reduced = self.reduced_container
signal = self.signal_container
empty = self.empty_container
intermediate = self.intermediate
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::clear_zip
; This function clears the objects from the zip container and destroys them.
nzip = self.zip_container->count()
if nzip gt 0 then begin
   oall = self.zip_container->get(/all)
   obj_destroy,oall
   obj_destroy,self.zip_container
   self.zip_container = obj_new('IDL_CONTAINER')
endif
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::add_zip,oarray
; This function adds the data objects contained in OARRAY to the zip container
; object, sorting them by their descriptive variable first.  Clone the objects
; before adding!
n = n_elements(oarray)
for i = 0,n-1 do begin
   if obj_valid(oarray[i]) then begin
      clone = oarray[i]->clone()
      self.zip_container->add,clone
   endif
endfor
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::check_indep_var_limits
; Check the limits of the independent variable.
;n = self.zip_container->count()
;if n eq 0 then return,0
;oall = self.zip_container->get(/all)
;for i = 0,n-1 do begin
;   ret = oall[i]->get_property(plot_str = plot_str)
;   x = *plot_str.x
;   if i eq 0 then begin
;
;   endif else begin
;   endelse
;endfor
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::zip_data, MSG = msg,DISPLAY = display
; This function "ZIPS" data together based on a descriptive variable such as
; temperature or magnetic field for instance.  The data used in the zipping is
; selected from any of the SIGNAL, EMPTY, or REDUCED containers.
msg = ''
nzip = self.zip_container->count()
if nzip eq 0 then return,0
; Search for the descriptive variable for each and determine if they are the
; same.
oall = self.zip_container->get(/all)

dflag = 1
for i = 0,nzip-1 do begin
   ret = oall[i]->get_property(descriptr_str = descriptr)
   if n_elements(descriptr) eq 0 then dflag = 0
endfor
minXbin = 0.1D
for i = 0,nzip-1 do begin
   ret = oall[i]->get_property(descriptr_str = descriptr_str,plot_str = plot_str,signal_type=stype,display_name=dname)
   xv = (*plot_str.x)
   nx = n_elements(xv)
   if (nx gt 2) then minXbin = minXbin < min(xv[1:nx-1] - xv[0:nx-2])   ; accumulate the smallest x bin width
   ; If there is no descriptr present then fill it out
   if dflag eq 0 then begin
      ret = oall[i]->set_index_descriptr(float(i))
      ret = oall[i]->get_property(descriptr_str = descriptr_str)
   endif
   ;ret = oall[i]->get_property(descriptr_str = descriptr_str)
   if i eq 0 then begin
      name = descriptr_str.name
      qty = descriptr_str.qty
      xtitles = plot_str.xtitle
      notAlreadyNormalized = 1
      if (stype eq 'REDUCED') then begin
        norm2mon = stregex(dname,'norm2mon') ge 0    ; is 'norm2mon' in display name?
        norm2time = stregex(dname,'norm2time') ge 0  ; is 'norm2time' in display name?
        if (norm2mon || norm2time) then notAlreadyNormalized = 0
      endif
   endif else begin
      name = [name,descriptr_str.name]
      qty = [qty,descriptr_str.qty]
      xtitles = [xtitles,plot_str.xtitle]
   endelse
endfor
if n_elements(uniq(name)) gt 1 then begin
   msg = 'Combining variables do not match!'
   return,0
endif
;  See if the first independent variable, in particular the name of x,
;  is the same for all data sets
if n_elements(uniq(xtitles)) gt 1 then begin
   msg = 'Independent variable types do not match!'
   return,0
endif
dsort = sort(qty)
; Perform a check on the independent variables
ret = self->check_indep_var_limits()
if ret eq 0 then return,0
; Reorganize the zip container based on the sorting
for i = 0,nzip-1 do self.zip_container->remove,oall[i]
obj_destroy,self.zip_container
self.zip_container = obj_new('IDL_CONTAINER')
for i = 0,nzip-1 do self.zip_container->add,oall[dsort[i]]
oall = self.zip_container->get(/all)
nzip = self.zip_container->count()

ptol = 0.001*abs(minXbin)  ; default to 0.1% of smallest x bin width ;0.001  
mtol = -0.001*abs(minxbin) ;-0.001

; Determine the agregate first ind axis range that overlaps all datasets
for i=0, nzip-1 do begin
   ret = oall[i]->get_property(plot_str=plot_str)
   xv = (*plot_str.x)
   xv =  xv[sort(xv)]  ; ensure x is in ascending order
   if (i eq 0) then x = xv
   nx = n_elements(x)
   index1 = where((xv-x[0]) lt mtol,cnt1) ;where(xv lt x[0], cnt1)
   index2 = where((xv-x[nx-1]) gt ptol,cnt2) ;where(xv gt x[nx-1], cnt2)
   if (cnt1 gt 0) then x = [xv[index1],x]
   if (cnt2 gt 0) then x = [x,xv[index2]]
endfor

; Now put the final data set together
nx = n_elements(x)
for i = 0,nzip-1 do begin
   ret = oall[i]->get_property(descriptr_str = descriptr_str, $
      plot_str = plot_str,treatment = this_treatment)
   if i eq 0 then treatment = this_treatment else $
      treatment = [treatment,this_treatment]
   ny = nzip

   xnew = *plot_str.x
   xSortIndex = sort(xnew) ; ensure xnew is in ascending order
   xnew =  xnew[xSortIndex]  
   nxnew = n_elements(xnew)

   mnew = oall[i]->get_monitor()
   mnew = mnew[xSortIndex]
   znew = (*plot_str.y)[xSortIndex]
   dznew = (*plot_str.dy)[xSortIndex]
   sfnew = mnew*0.0 + 1.0

   if i eq 0 then begin
      z = dblarr(nx,ny)
      dz = dblarr(nx,ny)
      y = dblarr(ny)
      mon = fltarr(nx,ny)
      sf = fltarr(nx,ny) + 1.0   ;<-- records where data consists of overlapping data points 
      monScale = mnew[0]
   endif

   y[i] = descriptr_str.qty

;   if total(xnew-x) ne 0 then begin ; interpolate
   void = where(abs(xnew-x) gt ptol,notTheSame)
   if (notTheSame eq 0) then notTheSame = nx ne nxnew
   if (notTheSame gt 0) then begin ; interpolate
;      x0_lt_xnew0 = (x[0] - xnew[0]) lt mtol
;      xn_gt_xnewn = (x[nx-1] - xnew[nxnew-1]) gt ptol
      x0_le_xnew0 = ((x[0] - xnew[0]) lt mtol) || (x[0] eq xnew[0])
      xn_ge_xnewn = ((x[nx-1] - xnew[nxnew-1]) gt ptol) || (x[nx - 1] eq xnew[nxnew - 1])
;      if (x[0] lt xnew[0] or x[nx-1] gt xnew[nxnew-1]) then begin
      if (x0_le_xnew0 or xn_ge_xnewn) then begin
          ;; avoid any extrapolation by running interpol() only for
          ;; values of x that lie within xnew 
;          x_ge_xnew0 = (x - xnew[0]) ge mtol
;          x_le_xnewn = (x - xnew[nxnew-1]) le ptol
          x_ge_xnew0 = ((x - xnew[0]) gt mtol) or ((x - xnew[0]) eq 0)
          x_le_xnewn = ((x - xnew[nxnew-1]) lt ptol) or ((x - xnew[nxnew-1]) eq 0)
;          inIndex = where(x ge xnew[0] and x le xnew[nxnew-1], inCnt $
          inIndex = where(x_ge_xnew0 and x_le_xnewn, inCnt $
                        ,complement=outIndex,ncomplement=outCnt)
          if (inCnt gt 0) then begin
              xValid = x[inIndex]
              zinterp = interpol(znew,xnew,xValid)
              dzinterp = interpol(dznew,xnew,xValid)
              minterp = interpol(mnew,xnew,xValid)
              z[inIndex,i] = zinterp
              dz[inIndex,i] = dzinterp
              mon[inIndex,i] = minterp
          endif
          if (outCnt gt 0) then begin
              ; value is undefined so use a NaN
              z[outIndex,i] = !values.D_NAN
              dz[outIndex,i] = !values.D_NAN
              mon[outIndex,i] = 0.0 ;monScale
              sf[outIndex,i] = 0.0
              ;z[outIndex,i] = 0.0
              ;dz[outIndex,i] = 0.0
              ;mon[outIndex,i] = 0.0 ;monScale
          endif
      endif
   endif else begin
      z[*,i] = znew
      dz[*,i] = dznew
      mon[*,i] = mnew
   endelse
endfor

; Rescale monitor to that of the first data point of the first dataset
mon = mon/monScale


; Look for duplicate y values and merge the corresponding groups in z and dz

;** change algorithm - replace commented section below with new code

;i = n_elements(y)
;diff = abs(y[0:i-2] - y[1:i-1])
;dupIndex = where(diff lt ptol, complement=uniqIndex, nDup)
;
;i = 0
;while (i lt nDup) do begin
;   index = [dupIndex[i], dupIndex[i] + 1]   ; each entry in dupIndex implies a dup pair in y
;   for j = i, nDup-1 do begin
;      gap = 2
;      if (j lt nDup-1) then gap = abs(dupIndex[j] - dupIndex[j+1])
;      if (gap eq 1) then $ => more than two consecutive values are the same in y
;         index = [index, dupIndex[j] + 2] $
;      else $
;         break
;   endfor
;   ;print,index
;   i = j + 1
;   
;   ; add up duplicate entries
;   z[*,dupIndex[j]+1] = total(z[*,index], 2, /NAN) 
;   dz[*,dupIndex[j]+1] = sqrt(total(dz[*,index]^2, 2, /NAN)) 
;   mon[*,dupIndex[j]+1] = total(mon[*,index], 2, /NAN)
;   sf[*,dupIndex[j]+1] = total(sf[*,index],2, /NAN)
;   zeroMonIndex = where(mon[*,dupIndex[j]+1] eq 0.0,zeroMonCnt)
;   if (zeroMonCnt gt 0) then begin
;      ;a mon value of 0.0 signifies that the corresponding count should be undefined
;      ;so make sure those values are replaced with NaNs. This step is necessary b/c 
;      ;total(z[],/NAN) will return a 0.0 instead of a NaN if z[] contains all NaNs
;      z[zeroMonIndex,dupIndex[j]+1] = !values.D_NAN
;      dz[zeroMonIndex,dupIndex[j]+1] = !values.D_NAN
;   endif
;endwhile
;
;
;if (nDup gt 0) then begin
;   z = temporary(z[*,uniqIndex])
;   dz = temporary(dz[*,uniqIndex])
;   mon = temporary(mon[*,uniqIndex])
;   sf = temporary(sf[*,uniqIndex])
;   y = y[uniqIndex] 
;endif

dims = size(z,/dimensions)
n_x = dims[0]
n_y = dims[1]
i = 0
while (i lt n_y-1) do begin
   index = i
   for j = i, n_y-2 do begin
      diff = abs(y[i] - y[j+1])
      if (diff le ptol) then begin
         index = [index,j+1]
      endif else begin
         break
      endelse
   endfor
   i = j + 1
   ymean = (moment(y[index],maxmoment=1))[0]
   nIndex = n_elements(index)
   if (nIndex gt 1) then begin
     c1 = total(z[*,index],2,/NAN)
     e1 = sqrt(total(dz[*,index]^2,2,/NAN))
     m1 = total(mon[*,index], 2, /NAN)
     s1 = total(sf[*,index],2, /NAN)
   endif else begin
     c1 = z[*,index]
     e1 = dz[*,index]
     m1 = mon[*,index]
     s1 = sf[*,index]
   endelse
   if (size(c1,/n_dimensions) eq 1) then begin
     c1 = reform(c1,n_x,1)
     e1 = reform(e1,n_x,1)
   endif
   if (n_elements(newy) eq 0) then begin
     newy = ymean
     newc = c1
     newe = e1
     newm = m1
     news = s1
   endif else begin
     newy = [newy,ymean]
     newc = transpose([transpose(newc),transpose(c1)])
     newe = transpose([transpose(newe),transpose(e1)])
     newm = transpose([transpose(newm),transpose(m1)])
     news = transpose([transpose(news),transpose(s1)])
   endelse
;   if (i gt 6189) then begin
;      print,i
;   endif
endwhile
lastIndex = index[n_elements(index)-1]
if (lastIndex eq n_y-2) then begin
  ; the last group was not processed so include it.
   newy = [newy,y[n_y-1]]
   c1 = reform(z[*,n_y-1],n_x,1)
   e1 = reform(dz[*,n_y-1],n_x,1)
   m1 = reform(mon[*,n_y-1],n_x,1)
   s1 = reform(sf[*,n_y-1],n_x,1)
   newc = transpose([transpose(newc),transpose(c1)])
   newe = transpose([transpose(newe),transpose(e1)])
   newm = transpose([transpose(newm),transpose(m1)])
   news = transpose([transpose(news),transpose(s1)])
endif 
y = newy
z = newc
dz = newe
mon = newm
sf = news


monitor = transpose(mon*monscale) ; used below

; Normalize data and error by monitor
if (notAlreadyNormalized) then begin
  ; checking notAlreadynormalized ensures we don't re-normalize reduced data that was previously normalized 
  index = where(mon le 0.0, cnt)
  if (cnt gt 0) then mon[index] = 1.0
  z = z/mon
  dz = dz/mon
endif else begin
  ; the data were previously normalised so all input data are on same monitor
  ; So simply deal with overlapping data now by using sf which essentially contains
  ; a record of the overlaps
  index = where(sf le 0.0, cnt)
  if (cnt gt 0) then sf[index] = 1.0
  z = z/sf
  dz = dz/sf
endelse


treatment = [treatment,'_________________________________________' $
            ,'The files listed above were combined']
; The return value will be a structure
px = ptr_new(y,/no_copy)
py = ptr_new(x,/no_copy)
pz = ptr_new(transpose(z),/no_copy)
bad_index = where(dz eq 0.0,count_bad)
if count_bad gt 0 then dz[bad_index] = 1.0
pdz = ptr_new(transpose(dz),/no_copy)
xtitle = descriptr_str.legend
ytitle = plot_str.xtitle

str = {  px:px,               $
         py:py,               $
         pz:pz,               $
         pdz:pdz,             $
         xtitle:xtitle,       $
         ytitle:ytitle        }

o = obj_new('dtas_zip')
ret = o->set_property(  xval = *str.px,                  $
                        yval = *str.py,                  $
                        z = *str.pz,                     $
                        dz = *str.pdz,                   $
                        mon = monitor, $
                        xtitle = xtitle,                 $
                        display_name = 'COMBINED_DATA',  $
                        treatment = treatment,           $
                        ytitle = ytitle                  )

heap_free,str
self.combine_container->add,o
return,o
end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::zip_non_orth_data, MSG = msg,DISPLAY = display
; This function "ZIPS" data together based on a descriptive variable such as
; temperature or magnetic field for instance.  The data used in the zipping is
; selected from any of the SIGNAL, EMPTY, or REDUCED containers.
msg = ''
nzip = self.zip_container->count()
if nzip eq 0 then return,0
; Search for the descriptive variable for each and determine if they are the
; same.
oall = self.zip_container->get(/all)
first_var = strarr(nzip)
second_var = strarr(nzip)
for i = 0,nzip-1 do begin
   ret = oall[i]->get_property(descriptr_str = descriptr,   $
                               var_names = var_names,data = data,plot_str = plot_str $
                              ,second_var_name=sndVarName)
   ;col_pos = strpos(descriptr.name,':')
   ;index = where(var_names eq strmid(descriptr.name,0,col_pos))
   if i eq 0 then begin
      first_var_name = plot_str.xtitle
      second_var_name = descriptr.name ;var_names[index[0]]
   endif else begin
      first_var_name = [first_var_name,plot_str.xtitle]
      second_var_name = [second_var_name,descriptr.name];[second_var_name,var_names[index[0]]]
   endelse
   if n_elements(uniq(first_var_name)) ne 1  then begin
      msg = 'First independent variable names do not match up'
      return,0
  endif
;   if n_elements(uniq(second_var_name)) ne 1  then begin
;      msg = 'Combining variable names do not match up'
;      return,0
;  endif

endfor

dflag = 1
for i = 0,nzip-1 do begin
   ret = oall[i]->get_property(descriptr_str = descriptr,   $
                               var_names = var_names,data = data,plot_str = plot_str $
                              ,second_var_name=sndVarName)
   if n_elements(descriptr) eq 0 then dflag = 0
   ;col_pos = strpos(descriptr.name,':')
   ;index = where(var_names eq strmid(descriptr.name,0,col_pos))
   first_var_name = plot_str.xtitle
   second_var_name = descriptr.name ;var_names[index[0]]
   ; Get the second independent variable
   if i eq 0 then begin
      y = descriptr.qty ;*data.(index)
      x = *plot_str.x
      z = *plot_str.y
      dz = *plot_str.dy
   endif else begin
      y = [y,descriptr.qty]    ;[y,*data.(index)]
      x = [x,*plot_str.x]
      z = [z,*plot_str.y]
      dz = [dz,*plot_str.dy]
   endelse
endfor

; Now determine the grid
xmin = min(x,max = xmax)
ymin = min(y,max = ymax)
nygrid = n_elements(y)          ;fix(1.0*n_elements(y)/(1.0*nzip))
nxgrid = fix(1.0*n_elements(x)/(1.0*nzip))

dx = (xmax-xmin)/(nxgrid-1.0)
xgrid = xmin+dx*findgen(nxgrid)
dy = (ymax-ymin)/(nygrid-1.0)
ygrid = ymin+dy*findgen(nygrid)
limits = [xmin,ymin,xmax,ymax]
; Now go through and do the interpolation to a regular grid
triangulate, x, y, triangles, boundary
grid = trigrid(x, y, z, triangles,XOUT = xgrid,       $
   YOUT = ygrid,missing = 0.0);,max = max(z))
dgrid = trigrid(x, y, dz, triangles, XOUT = xgrid,    $
   YOUT = ygrid,missing = 0.0)
;grid = trigrid(x, y, z, triangles, limits,XGRID = xgrid, $
;   YGRID = ygrid,missing = 0.0,nx = 10);,max = max(z))
;dgrid = trigrid(x, y, dz, triangles, limits,XGRID = xgrid, $
;   YGRID = ygrid,missing = 0.0,ny = 10)
zout = grid
dzout = dgrid
xout = xgrid
yout = ygrid
;tvlct,r,g,b,/get
;loadct,1,/silent
;;tvimage,bytscl(zout)
;plotimage, bytscl((zout)), imgxrange=[min(xout),max(xout)], imgyrange=[min(yout),max(yout)]
;tvlct,r,g,b
;return,0

;for i = 0,nzip-1 do begin
;   ret = oall[i]->get_property(descriptr_str = descriptr_str,plot_str = plot_str)
;   ; If there is no descriptr present then fill it out
;   if dflag eq 0 then begin
;      ret = oall[i]->set_index_descriptr(float(i))
;      ret = oall[i]->get_property(descriptr_str = descriptr_str)
;   endif
;   ;ret = oall[i]->get_property(descriptr_str = descriptr_str)
;   if i eq 0 then begin
;      name = descriptr_str.name
;      qty = descriptr_str.qty
;      xtitles = plot_str.xtitle
;   endif else begin
;      name = [name,descriptr_str.name]
;      qty = [qty,descriptr_str.qty]
;      xtitles = [xtitles,plot_str.xtitle]
;   endelse
;endfor
;if n_elements(uniq(name)) gt 1 then begin
;   msg = 'Combining variables do not match!'
;   return,0
;endif
;  See if the first independent variable, in particular the name of x,
;  is the same for all data sets
;if n_elements(uniq(xtitles)) gt 1 then begin
;   msg = 'Independent variable types do not match!'
;   return,0
;endif
;dsort = sort(qty)
;; Perform a check on the independent variables
;ret = self->check_indep_var_limits()
;if ret eq 0 then return,0
;; Reorganize the zip container based on the sorting
;for i = 0,nzip-1 do self.zip_container->remove,oall[i]
;obj_destroy,self.zip_container
;self.zip_container = obj_new('IDL_CONTAINER')
;for i = 0,nzip-1 do self.zip_container->add,oall[dsort[i]]
;oall = self.zip_container->get(/all)
;nzip = self.zip_container->count()
;; Now put the final data set together
;for i = 0,nzip-1 do begin
;   ret = oall[i]->get_property(descriptr_str = descriptr_str, $
;      plot_str = plot_str,treatment = this_treatment)
;   if i eq 0 then treatment = this_treatment else $
;      treatment = [treatment,this_treatment]
;   ny = nzip
;   if i eq 0 then begin
;      nx = n_elements(*plot_str.x)
;      x = *plot_str.x
;      z = dblarr(nx,ny)
;      dz = dblarr(nx,ny)
;      y = dblarr(ny)
;   endif
;   xnew = *plot_str.x
;   y[i] = descriptr_str.qty
;   if total(xnew-x) ne 0 then begin ; interpolate
;      znew = *plot_str.y
;      dznew = *plot_str.dy
;      zinterp = interpol(znew,xnew,x)
;      dzinterp = interpol(dznew,xnew,x)
;      z[*,i] = zinterp
;      dz[*,i] = dzinterp
;   endif else begin
;      z[*,i] = *plot_str.y
;      dz[*,i] = *plot_str.dy
;      x = *plot_str.x
;   endelse
;endfor
;treatment = [treatment,'The files listed above were combined']
; The return value will be a structure
px = ptr_new(xout,/no_copy)
py = ptr_new(yout,/no_copy)
pz = ptr_new(transpose(zout),/no_copy)
bad_index = where(dzout eq 0.0,count_bad)
if count_bad gt 0 then dzout[bad_index] = 1.0
pdz = ptr_new(transpose(dzout),/no_copy)
xtitle = first_var_name;descriptr_str.legend
ytitle = second_var_name;plot_str.xtitle

str = {  px:px,               $
         py:py,               $
         pz:pz,               $
         pdz:pdz,             $
         xtitle:xtitle,       $
         ytitle:ytitle        }

o = obj_new('dtas_zip')
ret = o->set_property(  xval = *str.px,                  $
                        yval = *str.py,                  $
                        z = *str.pz,                     $
                        dz = *str.pdz,                   $
                        xtitle = xtitle,                 $
                        display_name = 'COMBINED_DATA',  $
                        treatment = treatment,           $
                        ytitle = ytitle                  )
heap_free,str
self.combine_container->add,o
return,o
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::reduce, signal=oSignal, empty=oEmpty, $
                              SUM_SIGNAL = sum_signal,                        $
                              SUM_BG = sum_bg,                                $
                              STRAIGHT_ADD = straight_add,                    $
                              DETAILED_BALANCE = detailed_balance,            $
                              RESOLUTION_CORRECTION = resolution_correction,  $
                              MONITOR_CORRECTION = monitor_correction,        $
                              FAST_BACKGROUND = fast_background,              $
                              FBG_SET = fbg_set,                              $
                              INTERMEDIATE = intermediate,                    $
                              SUBTRACTION = subtraction,                      $
                              NORMALIZE_TO_MONITOR = normalize_to_monitor,    $
                              NORMALIZE_TO_time = normalize_to_time,    $
                              INTERP = interp,                                $
                              SF_SET = sf_set,                                $
                              SCALE_FACTOR = scale_factor,                    $
                              _EXTRA = extra,                                 $
                              MSG = msg
msg = ''
; The reduction steps are performed in the following order:
; (1).   If the SUM keyword is set then two possible sums are performed: (a). the
;        data from the SIGNAL container are summed and (b). the data from the
;        EMPTY container are summed.  If SUM is not set then all of the data objects
;        in the SIGNAL and EMPTY containers are treated identically.  If the
;        STRAIGHT_ADD keyword is set then both the contents of the SIGNAL container
;        and the EMPTY container are summed as raw counts (with no scaling to
;        monitor).  If STRAIGHT_ADD is not set then the data files are summed
;        in accordance with POISSON statistics.  The default is for straight
;        addition since we are assuming that the input data files are in terms of
;        raw counts.
; (2).   If the SUBTRACTION keyword is set then the sum of the EMPTY container is
;        subtracted from the sum of the SIGNAL container.  If the independent axes
;        do not line up then either LINEAR or (cubic) SPLINE interpolation are used.
;        The INTERP keyword can be set to "LINEAR" or "SPLINE" at the user's discretion
;        but LINEAR is the default.
; (3).   If FAST_BACKGROUND keyword is set equal to a number (assumed in counts/minute)
;        then this subtraction is applied for each value of the dependent variable.
;        Many of the subsequent reduction operations require the FAST_BACKGROUND
;        subtraction.  Such operations are noted explicitly below.
; (4).   If RESOLUTION_CORRECTION is set (Boolean) then the resolution correction
;        is applied.  Requires FAST_BACKGROUND subtraction.
; (5).   If DETAILED_BALANCE is set (Boolean) then the symmetrization of the
;        scattering profile is performed (if the temperature is provided in the
;        raw data file).  Requires FAST_BACKGROUND subtraction.
; (6).   If MONITOR_CORRECTION_FACTOR is applied then the data is corrected for
;        variations in the beam monitor which is provided in an external file
;        provided in this software.  Requires FAST_BACKGROUND subtraction.
; (7).   If NORMALIZE_TO_MONITOR is set then the scattering profile is normalized
;        to the beam monitor.
; (8).   If SCALE_FACTOR is set equal to a number then the dependent variable and
;        its uncertainty get scaled up by the scale factor provided.
;

;RTA - 
;Previously, all the loaded signal and empty files were reduced. So they were simply retrieved
;from their respective containers.
;Now, if a subset of files is selected in the signal or empty container, then only the
;selected files will be reduced. The selected files are passed in using the
;signal and empty keywords.

;signal_cont = self.signal_container & nsignal = signal_cont->count()
;empty_cont = self.empty_container & nempty = empty_cont->count()


; Basic error Handler
if (N_elements(!debug) && (!debug eq 0)) then begin
  Catch, catchError
  if (catchError ne 0) then begin
    ;;print, 'Error handled!'
    eTitle = 'dtas_reducs::reduce: Error encountered'
    eMsg = 'An error or unusual condition was encountered!'
    eMsg = [eMsg,'Please, report the following to the DAVE team:']
    eMsg = [eMsg,!error_state.msg]
    void = Dialog_message(/error,eMsg,title=eTitle)

    Catch, /cancel
    Return, 0
  endif
endif

nsignal = n_elements(oSignal)
nempty = n_elements(oEmpty)
ntotal = nsignal + nempty
if (nempty+nsignal) eq 0 then return,0

if n_elements(interp) ne 0 then self.interp = interp


; If bkgd subtraction is to be performed later, do a fast bkgd subtraction now on all signal + bkgd files
if (keyword_set(fbg_set) && keyword_set(subtraction) && (fast_background ne 0.0)) then begin
  oArray = [oSignal,oEmpty]
  for i=0, ntotal-1 do begin
    obj = oArray[i]->clone()
    ret = obj->fast_background_subtraction(fast_background,o = o1)
    if ((not intermediate) and (obj_valid(o1))) then obj_destroy,o1
    ret = self->update_plot_structure(obj)
    if obj_valid(o1) then begin
       if intermediate then self.intermediate->add,o1
       ret = self->update_plot_structure(obj)
    endif
    oArray[i] = obj
  endfor
  oSignal = oArray[0:nsignal-1]
  if (nEmpty gt 0) then oEmpty = oArray[nsignal:ntotal-1]
endif

; Sum the signal files?
if keyword_set(sum_signal) then begin
   ; If present then sum the SIGNAL files
   if nsignal gt 0 then begin
      if keyword_set(straight_add) then begin
         add_object = self->sum_straight(MSG = msg, o = oadd,data_objects=oSignal,/signal)
      endif else begin
         add_object = self->sum_poisson(MSG = msg, o = oadd,data_objects=oSignal,/signal)
      endelse
      if not obj_valid(add_object) then begin
         msg = msg
         return,0
      endif
      ret = self->update_plot_structure(oadd)
      ret = self->update_plot_structure(add_object)
      if intermediate then self.intermediate->add,oadd else obj_destroy,oadd
      red_object = add_object
   endif
endif

; Sum the background files?
if keyword_set(sum_bg) then begin
   if nempty gt 0 then begin
      if keyword_set(straight_add) then begin
         sub_object = self->sum_straight(MSG = msg, o = osub,/empty,data_objects=oEmpty)
      endif else begin
         sub_object = self->sum_poisson(MSG = msg, o = osub,/empty,data_objects=oEmpty)
      endelse
      if not obj_valid(sub_object) then begin
         msg = msg
         return,0
      endif
      ret = self->update_plot_structure(osub)
      ret = self->update_plot_structure(sub_object)
      if intermediate then self.intermediate->add,osub else obj_destroy,osub
      ;if n_elements(red_object) gt 0 then red_object = [red_object,sub_object] else $
      red_object = sub_object
   endif
endif

; Perform the subtraction?
if keyword_set(subtraction) then begin

   if (nsignal ge 1 and nempty eq 0) then begin
     ;; Subtraction is set even though there are no background files
     if (keyword_set(sum_signal)) then begin
       ;; use the previously added signal object
       red_object = add_object
     endif else begin
       ;; use the individual signals
       red_object = Objarr(nsignal)
       for i = 0,nsignal-1 do begin
         red_object[i] = oSignal[i]->Clone()
       endfor
       ;self->Add,red_object
     endelse
   endif

   if nsignal eq 1 and nempty eq 1 then begin
      ;; Simplest case: 1 signal and 1 bkgd dataset
      ;; so make a direct subtration
      add_object = oSignal[0]->clone() ;(signal_cont->get(position = 0))->clone()
      sub_object = oEmpty[0]->clone()  ;(empty_cont->get(position = 0))->clone()
      ;self->add,add_object
      ;self->add,sub_object
      if obj_valid(add_object) and obj_valid(sub_object) then begin
            subtraction_object = self->subtract_data(add_object,sub_object,   $
            MSG = msg, o = o_sub_object)
            ret = o_sub_object->get_property(plot_str = plot_str,data = data)
            if obj_valid(subtraction_object) then begin
               ret = self->update_plot_structure(o_sub_object)
               if intermediate then self.intermediate->add,o_sub_object else obj_destroy,o_sub_object
               red_object = subtraction_object
               ret = self->update_plot_structure(subtraction_object)
            endif
      endif
   endif
   
   if (nsignal gt 1) and (nempty eq 1) then begin
      sub_object = oEmpty[0]->clone() ;(empty_cont->get(position = 0))->clone()
      self->add,sub_object
      if obj_valid(add_object) then begin
         ; Consider the case where the signal files are summed.
         if obj_valid(add_object) and obj_valid(sub_object) then begin
            subtraction_object = self->subtract_data(add_object,sub_object,   $
            MSG = msg, o = o_sub_object)
            ret = o_sub_object->get_property(plot_str = plot_str,data = data)
            if obj_valid(subtraction_object) then begin
               ret = self->update_plot_structure(o_sub_object)
               if intermediate then self.intermediate->add,o_sub_object else obj_destroy,o_sub_object
               red_object = subtraction_object
               ret = self->update_plot_structure(subtraction_object)
            endif
         endif
      endif else begin
         ; Now consider the case where the one background file is subtracted
         ; from each individual signal file.
         n = nsignal + nempty
         new_red_object = objarr(n-1)
         for i = 0,n-2 do begin
            add_object = oSignal[i]->Clone()  ;(signal_cont->get(position = i))->clone()
            self->add,add_object
            if obj_valid(add_object) and obj_valid(sub_object) then begin
               subtraction_object = self->subtract_data(add_object,sub_object,   $
                  MSG = msg, o = o_sub_object)


               ret = o_sub_object->get_property(plot_str = plot_str,data = data)
               if obj_valid(subtraction_object) then begin
                  ret = self->update_plot_structure(o_sub_object)
                  if intermediate then self.intermediate->add,o_sub_object else obj_destroy,o_sub_object
                  new_red_object[i] = subtraction_object
                  ret = self->update_plot_structure(subtraction_object)
               endif
            endif
         endfor
         red_object = new_red_object
      endelse
   endif

   ;;   if (nsignal gt 1) and (nempty gt 1) then begin
   if (nsignal ge 1) and (nempty gt 1) then begin
       ;; RTA - add option to perform subtraction of equivalent bkgd
       ;;       from signal files when multiple signal/bkgd are loaded
       if (nsignal eq nempty) and ~keyword_set(sum_bg) then begin
           ;; if same nos of signal and bkgd files then do direct
           ;; subtraction of equivalent files in display order
           err_msg = ['Subtracting corresponding background from signal datasets', $
                     'in the order in which they are listed!']
           void = dialog_message(err_msg,/information)


           n = nsignal + 1
           new_red_object = objarr(n-1)
           for i = 0,n-2 do begin
               add_object = oSignal[i]->Clone() ;(signal_cont->get(position = i))->clone()
               self->add,add_object
               sub_object = oEmpty[i]->Clone() ;empty_cont->get(position = i)
               if obj_valid(add_object) and obj_valid(sub_object) then begin
                   subtraction_object = self->subtract_data(add_object,sub_object,   $
                                                            MSG = msg, o = o_sub_object)
                   ret = o_sub_object->get_property(plot_str = plot_str,data = data)
                   if obj_valid(subtraction_object) then begin
                       ret = self->update_plot_structure(o_sub_object)
                       if intermediate then self.intermediate->add,o_sub_object else obj_destroy,o_sub_object
                       new_red_object[i] = subtraction_object
                       ret = self->update_plot_structure(subtraction_object)
                   endif
               endif
           endfor
           red_object = new_red_object
       endif else begin
           
           
           if ~obj_valid(sub_object) then begin
               sub_object = oEmpty[0]->clone()  ;(empty_cont->get(position = 0))->clone()
               self->add,sub_object
               err_msg = 'Using the first file in the background folder only in'+$
                         'the subtraction'
               void = dialog_message(err_msg,/information)
           endif
           
           if obj_valid(add_object) then begin
             ;; Consider the case where signal files are summed and a single bkgd object is subtracted
             subtraction_object = self->subtract_data(add_object,sub_object,   $
                                                      MSG = msg, o = o_sub_object)
             ret = o_sub_object->get_property(plot_str = plot_str,data = data)
             if obj_valid(subtraction_object) then begin
                 ret = self->update_plot_structure(o_sub_object)
                 if intermediate then self.intermediate->add,o_sub_object else obj_destroy,o_sub_object
                 red_object = subtraction_object
                 ret = self->update_plot_structure(subtraction_object)
             endif
           endif else begin
               ;; Now consider the case where a single background object is subtracted
               ;; from each individual signal file.
               n = nsignal + 1
               new_red_object = objarr(n-1)
               for i = 0,n-2 do begin
                   add_object = oSignal[i]->Clone() ;(signal_cont->get(position = i))->clone()
                   self->add,add_object
                   if obj_valid(add_object) and obj_valid(sub_object) then begin
                       subtraction_object = self->subtract_data(add_object,sub_object,   $
                                                                MSG = msg, o = o_sub_object)
                       ret = o_sub_object->get_property(plot_str = plot_str,data = data)
                       if obj_valid(subtraction_object) then begin
                           ret = self->update_plot_structure(o_sub_object)
                           if intermediate then self.intermediate->add,o_sub_object else obj_destroy,o_sub_object
                           new_red_object[i] = subtraction_object
                           ret = self->update_plot_structure(subtraction_object)
                       endif
                   endif
               endfor
               red_object = new_red_object
           endelse
       endelse
   endif
   
endif else begin
  ;; No Subtration

  if obj_valid(add_object) then begin
    ;; Signals summed BUT Bkgds were not summed
    if ~obj_valid(sub_object) and nempty gt 0 then begin
       red_object = objarr(nempty+1)
       red_object[0] = add_object
       for i=1,nempty do red_object[i] = (oEmpty[i-1])->Clone() ;red_object[1:nempty] = (empty_cont->get(/all))->clone()
    endif
  endif
  
  if obj_valid(sub_object) then begin
    ;; Bkgds summed BUT Signals were not summed
    if ~obj_valid(add_object) and nsignal gt 0 then begin
       red_object = objarr(nsignal+1)
       oall = oSignal  ;signal_cont->get(/all)
       for i = 0,n_elements(oall)-1 do red_object[i] = (oall[i])->Clone()
       ;red_object[0:nsignal-1] = (signal_cont->get(/all))->clone()
       red_object[n_elements(oall)] = sub_object
    endif
  endif
  
  if ((~obj_valid(add_object)) and (~obj_valid(sub_object))) then begin
    ;; Signals AND Bkgds were not summed
    red_object = objarr(nsignal+nempty)
    if nsignal gt 0 then begin
       sig_all = oSignal ;signal_cont->get(/all)
       for i = 0,nsignal-1 do begin
          red_object[i] = sig_all[i]->clone()
       endfor
    endif
    if nempty gt 0 then begin
       empty_all = oEmpty  ;empty_cont->get(/all)
       for i = 0,nempty-1 do begin
          red_object[nsignal+i] = empty_all[i]->clone()
       endfor
    endif
    ;self->add,red_object
  endif

  if obj_valid(add_object) and obj_valid(sub_object) then begin
    ;; Sinal AND Bkgd were summed
     red_object = objarr(2)
     red_object[0] = add_object
     red_object[1] = sub_object
    ;self->add,red_object
  endif

endelse ;; No Subtraction

; At this point we must have a reduction object so include it in the container
if (obj_valid(red_object[0])) then Self->add,red_object


;; Now that the bkgd subtraction is made, carry on with rest of reduction steps
n_total = n_elements(red_object)
if keyword_set(fbg_set) then begin
   for i = 0,n_total-1 do begin
   ; For all subsequent corrections the fast background correction is required.
      if (~keyword_set(subtraction)) then begin
        ;; this avoids a repeat fast bkgd sub for bkgd subtraction case since it was already done earlier
        ret = red_object[i]->fast_background_subtraction(fast_background,o = o1)
        if ((not intermediate) and (obj_valid(o1))) then obj_destroy,o1
        ret = self->update_plot_structure(red_object[i])
        if obj_valid(o1) then begin
           if intermediate then self.intermediate->add,o1
           ret = self->update_plot_structure(red_object[i])
        endif
      endif
      if keyword_set(resolution_correction) then begin
         ret = red_object[i]->resolution_correction(o = o2)
         if (not intermediate) and obj_valid(o2) then obj_destroy,o2
         ret = self->Update_plot_structure(red_object[i])
         if Obj_valid(o2) then begin
           if intermediate then self.intermediate->Add,o2
           ret = self->Update_plot_structure(o2)
         endif
      endif
      if keyword_set(detailed_balance) then begin
         ret = red_object[i]->detailed_balance(o = o3)
         if (not intermediate) and obj_valid(o3) then obj_destroy,o3
         ret = self->Update_plot_structure(red_object[i])
         if Obj_valid(o3) then begin
           if intermediate then self.intermediate->Add,o3
           ret = self->Update_plot_structure(o3)
         endif
      endif
      if keyword_set(monitor_correction) then begin
         ret = red_object[i]->monitor_correction_factor(o = o4)
         if ((not intermediate) and (obj_valid(o4))) then obj_destroy,o4
         ret = self->Update_plot_structure(red_object[i])
         if Obj_valid(o4) then begin
           if intermediate then self.intermediate->Add,o4
           ret = self->Update_plot_structure(o4)
         endif
      endif

      if keyword_set(normalize_to_monitor) then begin
         ret = red_object[i]->normalize_to_monitor(o = o5)
         if (not intermediate) and obj_valid(o5) then obj_destroy,o5
         ret = self->Update_plot_structure(red_object[i])
         if Obj_valid(o5) then begin
           if intermediate then self.intermediate->Add,o5
           ret = self->Update_plot_structure(o5)
         endif
     endif else begin
         if (keyword_set(normalize_to_time)) then begin
             ret = red_object[i]->normalize_to_time(o = o5)
             if (not intermediate) and obj_valid(o5) then obj_destroy,o5
         endif
         ret = self->Update_plot_structure(red_object[i])
         if Obj_valid(o5) then begin
           if intermediate then self.intermediate->Add,o5
           ret = self->Update_plot_structure(o5)
         endif
     endelse

   endfor
endif

if keyword_set(sf_set) ne 0 then begin
  ;; Re-scale data by arbitrary scalefactor
  for i = 0,n_total-1 do begin
     ret = red_object[i]->calibration(scale_factor,o = o6)
     ret = self->update_plot_structure(red_object[i])
     if (not intermediate) and obj_valid(o6) then obj_destroy,o6
     if obj_valid(o6) then begin
        if intermediate then self.intermediate->add,o6
           ret = self->update_plot_structure(o6)
     endif
  endfor
endif

if intermediate then nobj = self.intermediate->count()
if intermediate then oall = self.intermediate->get(/all)
ret = red_object[n_total-1]->get_property(treatment = treatment)
reduced = self.reduced_container
reduced_objects = reduced->get(/all)
for i = 0,n_elements(red_object)-1 do begin
   if obj_valid(reduced_objects[0]) then begin
      where_same = where(reduced_objects eq red_object[i],count_same)
      if (count_same eq 0) then $
         self.reduced_container->add,red_object[i]
   endif else begin
      for i = 0,n_total-1 do self.reduced_container->add,red_object[i]
   endelse
endfor

;intermediate = self.intermediate
if n_elements(red_object) ne 0 then begin
   for i = 0,n_elements(red_object)-1 do begin
      if obj_valid(red_object[i]) then begin
         ret = self->update_plot_structure(red_object[i])
         ret = red_object[i]->get_property(data = data)
         ncols = n_tags(data)
         slen = max(strlen(tag_names(data)))
         nx = n_elements(*data.(ncols-1)); n_elements(*data.(0))
         contents = strarr(nx+1)
         fstr = strtrim(string(slen),2)
         str = strtrim(string(tag_names(data),format = fs),2)
         fs = '('+strtrim(string(ncols),2)+'A'+fstr+')';'('+strtrim(string(ncols),2)+'A22)'
         str = strtrim(string(tag_names(data),format = fs),2)
         contents[0] = str
         f = '('+strtrim(string(ncols),2)+'e'+fstr+'.5)';'('+strtrim(string(ncols),2)+'e22.5)'
         for j = 0,nx-1 do begin
            for k = 0,ncols-1 do begin
               if k eq 0 then this_array = (*data.(k))[j] else $
                  this_array = [this_array,(*data.(k))[j]]
            endfor
            str = strtrim(string(this_array,format = f),2)
            contents[j+1] = str
         endfor
         ret = red_object[i]->set_property(signal_type = 'REDUCED',  $
            contents = contents)
      endif
   endfor
endif
if intermediate then begin
   nint = self.intermediate->count()
   oall = self.intermediate->get(/all)
   for i = 0,nint-1 do begin
      if obj_valid(oall[i]) then begin
         ret = self->update_plot_structure(oall[i])
         ret = oall[i]->get_property(data = data)
         ncols = n_tags(data)
         nx = n_elements(*data.(ncols-1)); n_elements(*data.(0))
         contents = strarr(nx+1)
         fstr = strtrim(string(slen),2)
         str = strtrim(string(tag_names(data),format = fs),2)
         fs = '('+strtrim(string(ncols),2)+'A'+fstr+')';'('+strtrim(string(ncols),2)+'A22)'
         str = strtrim(string(tag_names(data),format = fs),2)
         contents[0] = str
         f = '('+strtrim(string(ncols),2)+'e'+fstr+'.5)';'('+strtrim(string(ncols),2)+'e22.5)'
         for j = 0,nx-1 do begin
            for k = 0,ncols-1 do begin
               if k eq 0 then this_array = (*data.(k))[j] else $
                  this_array = [this_array,(*data.(k))[j]]
            endfor
            str = strtrim(string(this_array,format = f),2)
            contents[j+1] = str
         endfor
         ret = oall[i]->set_property(signal_type = 'INTERMEDIATE',  $
            contents = contents)
      endif
   endfor
endif
if n_elements(red_object) eq 0 then red_object = 0
return,red_object
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::reduce_old, SUM_SIGNAL = sum_signal,                        $
                              SUM_BG = sum_bg,                                $
                              STRAIGHT_ADD = straight_add,                    $
                              DETAILED_BALANCE = detailed_balance,            $
                              RESOLUTION_CORRECTION = resolution_correction,  $
                              MONITOR_CORRECTION = monitor_correction,        $
                              FAST_BACKGROUND = fast_background,              $
                              FBG_SET = fbg_set,                              $
                              INTERMEDIATE = intermediate,                    $
                              SUBTRACTION = subtraction,                      $
                              NORMALIZE_TO_MONITOR = normalize_to_monitor,    $
                              NORMALIZE_TO_time = normalize_to_time,    $
                              INTERP = interp,                                $
                              SF_SET = sf_set,                                $
                              SCALE_FACTOR = scale_factor,                    $
                              _EXTRA = extra,                                 $
                              MSG = msg
msg = ''
; The reduction steps are performed in the following order:
; (1).   If the SUM keyword is set then two possible sums are performed: (a). the
;        data from the SIGNAL container are summed and (b). the data from the
;        EMPTY container are summed.  If SUM is not set then all of the data objects
;        in the SIGNAL and EMPTY containers are treated identically.  If the
;        STRAIGHT_ADD keyword is set then both the contents of the SIGNAL container
;        and the EMPTY container are summed as raw counts (with no scaling to
;        monitor).  If STRAIGHT_ADD is not set then the data files are summed
;        in accordance with POISSON statistics.  The default is for straight
;        addition since we are assuming that the input data files are in terms of
;        raw counts.
; (2).   If the SUBTRACTION keyword is set then the sum of the EMPTY container is
;        subtracted from the sum of the SIGNAL container.  If the independent axes
;        do not line up then either LINEAR or (cubic) SPLINE interpolation are used.
;        The INTERP keyword can be set to "LINEAR" or "SPLINE" at the user's discretion
;        but LINEAR is the default.
; (3).   If FAST_BACKGROUND keyword is set equal to a number (assumed in counts/minute)
;        then this subtraction is applied for each value of the dependent variable.
;        Many of the subsequent reduction operations require the FAST_BACKGROUND
;        subtraction.  Such operations are noted explicitly below.
; (4).   If RESOLUTION_CORRECTION is set (Boolean) then the resolution correction
;        is applied.  Requires FAST_BACKGROUND subtraction.
; (5).   If DETAILED_BALANCE is set (Boolean) then the symmetrization of the
;        scattering profile is performed (if the temperature is provided in the
;        raw data file).  Requires FAST_BACKGROUND subtraction.
; (6).   If MONITOR_CORRECTION_FACTOR is applied then the data is corrected for
;        variations in the beam monitor which is provided in an external file
;        provided in this software.  Requires FAST_BACKGROUND subtraction.
; (7).   If NORMALIZE_TO_MONITOR is set then the scattering profile is normalized
;        to the beam monitor.
; (8).   If SCALE_FACTOR is set equal to a number then the dependent variable and
;        its uncertainty get scaled up by the scale factor provided.
;
signal_cont = self.signal_container & nsignal = signal_cont->count()
empty_cont = self.empty_container & nempty = empty_cont->count()

; Determine if there are any ICE files.  If so then provide a warning
; message and indicate that ICE files cannot be reduced yet.
;if nsignal gt 0 then begin
;   osignal = signal_cont->get(/all)
;   for i = 0,nsignal-1 do begin
;      ret = osignal[i]->get_property(filename = filename)
;      if is_ice(filename) then begin
;         msg = 'ICE files cannot be reduced yet.'
;         return,0B
;      endif
;   endfor
;endif
;if nempty gt 0 then begin
;   oempty = empty_cont->get(/all)
;   for i = 0,nempty-1 do begin
;      ret = oempty[i]->get_property(filename = filename)
;      if is_ice(filename) then begin
;         msg = 'ICE files cannot be reduced yet.'
;         return,0B
;      endif
;   endfor
;endif


if (nempty+nsignal) eq 0 then return,0

if n_elements(interp) ne 0 then self.interp = interp

; Sum the signal files?
if keyword_set(sum_signal) then begin
   ; If present then sum the SIGNAL files
   if nsignal gt 0 then begin
      if keyword_set(straight_add) then begin
         add_object = self->sum_straight(MSG = msg, o = oadd,/signal)
      endif else begin
         add_object = self->sum_poisson(MSG = msg, o = oadd,/signal)
      endelse
      if not obj_valid(add_object) then begin
         msg = msg
         return,0
      endif
      ret = self->update_plot_structure(oadd)
      ret = self->update_plot_structure(add_object)
      if intermediate then self.intermediate->add,oadd else obj_destroy,oadd
      red_object = add_object
   endif
endif

; Sum the background files?
if keyword_set(sum_bg) then begin
   if nempty gt 0 then begin
      if keyword_set(straight_add) then begin
         sub_object = self->sum_straight(MSG = msg, o = osub,/empty)
      endif else begin
         sub_object = self->sum_poisson(MSG = msg, o = osub,/empty)
      endelse
      if not obj_valid(sub_object) then begin
         msg = msg
         return,0
      endif
      ret = self->update_plot_structure(osub)
      ret = self->update_plot_structure(sub_object)
      if intermediate then self.intermediate->add,osub else obj_destroy,osub
      ;if n_elements(red_object) gt 0 then red_object = [red_object,sub_object] else $
      red_object = sub_object
   endif
endif

if obj_valid(add_object) then begin
   if ~obj_valid(sub_object) and nempty gt 0 then begin
      red_object = objarr(nempty+1)
      red_object[0] = add_object
      red_object[1:nempty] = (empty_cont->get(/all))->clone()
   endif
endif
if obj_valid(sub_object) then begin
   if ~obj_valid(add_object) and nsignal gt 0 then begin
      red_object = objarr(nsignal+1)
      oall = signal_cont->get(/all)
      for i = 0,n_elements(oall)-1 do red_object[i] = (oall[i])->clone()
      ;red_object[0:nsignal-1] = (signal_cont->get(/all))->clone()
      red_object[n_elements(oall)] = sub_object
   endif
endif

; If we have not summed any of the files then do this...
;if not (obj_valid(red_object))[0] then begin
if ((~obj_valid(add_object)) and (~obj_valid(sub_object))) then begin
   red_object = objarr(nsignal+nempty)
   if nsignal gt 0 then begin
      sig_all = signal_cont->get(/all)
      for i = 0,nsignal-1 do begin
         red_object[i] = sig_all[i]->clone()
      endfor
   endif
   if nempty gt 0 then begin
      empty_all = empty_cont->get(/all)
      for i = 0,nempty-1 do begin
         red_object[nsignal+i] = empty_all[i]->clone()
      endfor
   endif
   self->add,red_object
endif

; Perform the subtraction?
if keyword_set(subtraction) then begin
   if nsignal eq 1 and nempty eq 1 then begin
      add_object = (signal_cont->get(position = 0))->clone()
      sub_object = (empty_cont->get(position = 0))->clone()
      self->add,add_object
      self->add,sub_object
      if obj_valid(add_object) and obj_valid(sub_object) then begin
            subtraction_object = self->subtract_data(add_object,sub_object,   $
            MSG = msg, o = o_sub_object)
            ret = o_sub_object->get_property(plot_str = plot_str,data = data)
            if obj_valid(subtraction_object) then begin
               ret = self->update_plot_structure(o_sub_object)
               if intermediate then self.intermediate->add,o_sub_object else obj_destroy,o_sub_object
               red_object = subtraction_object
               ret = self->update_plot_structure(subtraction_object)
            endif
      endif
   endif
   if (nsignal gt 1) and (nempty eq 1) then begin
   ; Consider the case where we add the signal files first.
      sub_object = (empty_cont->get(position = 0))->clone()
      self->add,sub_object
      if obj_valid(add_object) then begin
         if obj_valid(add_object) and obj_valid(sub_object) then begin
            subtraction_object = self->subtract_data(add_object,sub_object,   $
            MSG = msg, o = o_sub_object)
            ret = o_sub_object->get_property(plot_str = plot_str,data = data)
            if obj_valid(subtraction_object) then begin
               ret = self->update_plot_structure(o_sub_object)
               if intermediate then self.intermediate->add,o_sub_object else obj_destroy,o_sub_object
               red_object = subtraction_object
               ret = self->update_plot_structure(subtraction_object)
            endif
         endif
      endif else begin
   ; Now consider the case where the one background file is subtracted
   ; from each individual signal file.
         n = nsignal + nempty
         new_red_object = objarr(n-1)
         for i = 0,n-2 do begin
            add_object = (signal_cont->get(position = i))->clone()
            self->add,add_object
            if obj_valid(add_object) and obj_valid(sub_object) then begin
               subtraction_object = self->subtract_data(add_object,sub_object,   $
                  MSG = msg, o = o_sub_object)


               ret = o_sub_object->get_property(plot_str = plot_str,data = data)
               if obj_valid(subtraction_object) then begin
                  ret = self->update_plot_structure(o_sub_object)
                  if intermediate then self.intermediate->add,o_sub_object else obj_destroy,o_sub_object
                  new_red_object[i] = subtraction_object
                  ret = self->update_plot_structure(subtraction_object)
               endif
            endif
         endfor
         red_object = new_red_object
      endelse
   endif

;;   if (nsignal gt 1) and (nempty gt 1) then begin
   if (nsignal ge 1) and (nempty gt 1) then begin
       ;; RTA - add option to perform subtraction of equivalent bkgd
       ;;       from signal files when multiple signal/bkgd are loaded
       if (nsignal eq nempty) and ~keyword_set(sum_bg) then begin
           ;; if same nos of signal and bkgd files then do direct
           ;; subtraction of equivalent files in display order
           err_msg = ['Subtracting corresponding background from signal datasets', $
                     'in the order in which they are listed!']
           void = dialog_message(err_msg,/information)


           n = nsignal + 1
           new_red_object = objarr(n-1)
           for i = 0,n-2 do begin
               add_object = (signal_cont->get(position = i))->clone()
               self->add,add_object
               sub_object = empty_cont->get(position = i)
               if obj_valid(add_object) and obj_valid(sub_object) then begin
                   subtraction_object = self->subtract_data(add_object,sub_object,   $
                                                            MSG = msg, o = o_sub_object)
                   ret = o_sub_object->get_property(plot_str = plot_str,data = data)
                   if obj_valid(subtraction_object) then begin
                       ret = self->update_plot_structure(o_sub_object)
                       if intermediate then self.intermediate->add,o_sub_object else obj_destroy,o_sub_object
                       new_red_object[i] = subtraction_object
                       ret = self->update_plot_structure(subtraction_object)
                   endif
               endif
           endfor
           red_object = new_red_object
       endif else begin
           
           
           if ~obj_valid(sub_object) then begin
               sub_object = (empty_cont->get(position = 0))->clone()
               self->add,sub_object
               err_msg = 'Using the first file in the background folder only in'+$
                         'the subtraction'
               void = dialog_message(err_msg,/information)
           endif
           ;; Consider the case where we add the signal files first.
           if obj_valid(add_object) then begin
               subtraction_object = self->subtract_data(add_object,sub_object,   $
                                                        MSG = msg, o = o_sub_object)
               ret = o_sub_object->get_property(plot_str = plot_str,data = data)
               if obj_valid(subtraction_object) then begin
                   ret = self->update_plot_structure(o_sub_object)
                   if intermediate then self.intermediate->add,o_sub_object else obj_destroy,o_sub_object
                   red_object = subtraction_object
                   ret = self->update_plot_structure(subtraction_object)
               endif
           endif else begin
               ;; Now consider the case where the sum of the background files is subtracted
               ;; from each individual signal file.
               n = nsignal + 1
               new_red_object = objarr(n-1)
               for i = 0,n-2 do begin
                   add_object = (signal_cont->get(position = i))->clone()
                   self->add,add_object
                   if obj_valid(add_object) and obj_valid(sub_object) then begin
                       subtraction_object = self->subtract_data(add_object,sub_object,   $
                                                                MSG = msg, o = o_sub_object)
                       ret = o_sub_object->get_property(plot_str = plot_str,data = data)
                       if obj_valid(subtraction_object) then begin
                           ret = self->update_plot_structure(o_sub_object)
                           if intermediate then self.intermediate->add,o_sub_object else obj_destroy,o_sub_object
                           new_red_object[i] = subtraction_object
                           ret = self->update_plot_structure(subtraction_object)
                       endif
                   endif
               endfor
               red_object = new_red_object
           endelse
       endelse
   endif
   
endif else begin

   if obj_valid(add_object) and obj_valid(sub_object) then begin
      red_object = objarr(2)
      red_object[0] = add_object & red_object[1] = sub_object
   endif

endelse


n_total = n_elements(red_object)
if keyword_set(fbg_set) then begin
   for i = 0,n_total-1 do begin
   ; For all subsequent corrections the fast background correction
   ; is essential.
      ret = red_object[i]->fast_background_subtraction(fast_background,o = o1)
      if ((not intermediate) and (obj_valid(o1))) then obj_destroy,o1
      ret = self->update_plot_structure(red_object[i])
      if obj_valid(o1) then begin
         if intermediate then self.intermediate->add,o1
         ret = self->update_plot_structure(red_object[i])
      endif
      if keyword_set(resolution_correction) then begin
         ret = red_object[i]->resolution_correction(o = o2)
         if (not intermediate) and obj_valid(o2) then obj_destroy,o2
      endif
      ret = self->update_plot_structure(red_object[i])
      if obj_valid(o2) then begin
         if intermediate then self.intermediate->add,o2
         ret = self->update_plot_structure(o2)
      endif
      if keyword_set(detailed_balance) then begin
         ret = red_object[i]->detailed_balance(o = o3)
         if (not intermediate) and obj_valid(o3) then obj_destroy,o3
      endif
         ret = self->update_plot_structure(red_object[i])
      if obj_valid(o3) then begin
         if intermediate then self.intermediate->add,o3
         ret = self->update_plot_structure(o3)
      endif
      if keyword_set(monitor_correction) then begin
         ret = red_object[i]->monitor_correction_factor(o = o4)
         if ((not intermediate) and (obj_valid(o4))) then obj_destroy,o4
      endif

      ret = self->update_plot_structure(red_object[i])
      if obj_valid(o4) then begin
         if intermediate then self.intermediate->add,o4
         ret = self->update_plot_structure(o4)
      endif
      if keyword_set(normalize_to_monitor) then begin
         ret = red_object[i]->normalize_to_monitor(o = o5)
         if (not intermediate) and obj_valid(o5) then obj_destroy,o5
     endif else begin
         if (keyword_set(normalize_to_time)) then begin
             ret = red_object[i]->normalize_to_time(o = o5)
             if (not intermediate) and obj_valid(o5) then obj_destroy,o5
         endif
     endelse

     ret = self->update_plot_structure(red_object[i])
     if obj_valid(o5) then begin
         if intermediate then self.intermediate->add,o5
         ret = self->update_plot_structure(o5)
     endif
   endfor
endif

if keyword_set(sf_set) ne 0 then begin
;   i = n_total - 1
   for i = 0,n_total-1 do begin
      ret = red_object[i]->calibration(scale_factor,o = o6)
      ret = self->update_plot_structure(red_object[i])
      if (not intermediate) and obj_valid(o6) then obj_destroy,o6
      if obj_valid(o6) then begin
         if intermediate then self.intermediate->add,o6
            ret = self->update_plot_structure(o6)
      endif
   endfor
endif

if intermediate then nobj = self.intermediate->count()
if intermediate then oall = self.intermediate->get(/all)
ret = red_object[n_total-1]->get_property(treatment = treatment)
reduced = self.reduced_container
reduced_objects = reduced->get(/all)
for i = 0,n_elements(red_object)-1 do begin
   if obj_valid(reduced_objects[0]) then begin
      where_same = where(reduced_objects eq red_object[i],count_same)
      if (count_same eq 0) then $
         self.reduced_container->add,red_object[i]
   endif else begin
      for i = 0,n_total-1 do self.reduced_container->add,red_object[i]
   endelse
endfor

;intermediate = self.intermediate
if n_elements(red_object) ne 0 then begin
   for i = 0,n_elements(red_object)-1 do begin
      if obj_valid(red_object[i]) then begin
         ret = self->update_plot_structure(red_object[i])
         ret = red_object[i]->get_property(data = data)
         ncols = n_tags(data)
         slen = max(strlen(tag_names(data)))
         nx = n_elements(*data.(0))
         contents = strarr(nx+1)
         fstr = strtrim(string(slen),2)
         str = strtrim(string(tag_names(data),format = fs),2)
         fs = '('+strtrim(string(ncols),2)+'A'+fstr+')';'('+strtrim(string(ncols),2)+'A22)'
         str = strtrim(string(tag_names(data),format = fs),2)
         contents[0] = str
         f = '('+strtrim(string(ncols),2)+'e'+fstr+'.5)';'('+strtrim(string(ncols),2)+'e22.5)'
         for j = 0,nx-1 do begin
            for k = 0,ncols-1 do begin
               if k eq 0 then this_array = (*data.(k))[j] else $
                  this_array = [this_array,(*data.(k))[j]]
            endfor
            str = strtrim(string(this_array,format = f),2)
            contents[j+1] = str
         endfor
         ret = red_object[i]->set_property(signal_type = 'REDUCED',  $
            contents = contents)
      endif
   endfor
endif
if intermediate then begin
   nint = self.intermediate->count()
   oall = self.intermediate->get(/all)
   for i = 0,nint-1 do begin
      if obj_valid(oall[i]) then begin
         ret = self->update_plot_structure(oall[i])
         ret = oall[i]->get_property(data = data)
         ncols = n_tags(data)
         nx = n_elements(*data.(0))
         contents = strarr(nx+1)
         fstr = strtrim(string(slen),2)
         str = strtrim(string(tag_names(data),format = fs),2)
         fs = '('+strtrim(string(ncols),2)+'A'+fstr+')';'('+strtrim(string(ncols),2)+'A22)'
         str = strtrim(string(tag_names(data),format = fs),2)
         contents[0] = str
         f = '('+strtrim(string(ncols),2)+'e'+fstr+'.5)';'('+strtrim(string(ncols),2)+'e22.5)'
         for j = 0,nx-1 do begin
            for k = 0,ncols-1 do begin
               if k eq 0 then this_array = (*data.(k))[j] else $
                  this_array = [this_array,(*data.(k))[j]]
            endfor
            str = strtrim(string(this_array,format = f),2)
            contents[j+1] = str
         endfor
         ret = oall[i]->set_property(signal_type = 'INTERMEDIATE',  $
            contents = contents)
      endif
   endfor
endif
if n_elements(red_object) eq 0 then red_object = 0
return,red_object
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::add,oarray
for i = 0,n_elements(oarray)-1 do begin
   if obj_class(oarray[i]) eq 'DTAS_DATA' then $
      self->add,oarray[i]
   ret = oarray[i]->get_property(signal_type = signal_type)
   case signal_type of
   'SIGNAL':  self.signal_container->add,oarray[i]
   'EMPTY':   self.empty_container->add,oarray[i]
   else:
   endcase
endfor
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::check_contents
; This function goes through the contents of the SIGNAL container and the EMPTY
; container and determines if each of the contents' signal_types matches the
; container type (EMPTY or SIGNAL).
oall = self->get(/all)
o_signal = self.signal_container->get(/all)
o_empty = self.empty_container->get(/all)
nsignal = self.signal_container->count()
nempty = self.empty_container->count()
for i = 0,nsignal-1 do begin
   ret = o_signal[i]->get_property(signal_type = signal_type)
   if signal_type eq 'EMPTY' then begin
      self.signal_container->remove,o_signal[i]
      self.empty_container->add,o_signal[i]
   endif
endfor
for i = 0,nempty-1 do begin
   ret = o_empty[i]->get_property(signal_type = signal_type)
   if signal_type eq 'SIGNAL' then begin
      self.empty_container->remove,o_empty[i]
      self.signal_container->add,o_empty[i]
   endif
endfor
return,1
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function dtas_reduce::init,obj_array
if not self->idl_container::init() then return,0
self.signal_container = obj_new('IDL_CONTAINER')
self.empty_container = obj_new('IDL_CONTAINER')
if n_elements(obj_array) ne 0 then begin
; Put the objects into the appropriate container
   nobj = n_elements(obj_array)
   for i = 0,nobj-1 do begin
      ret = obj_array[i]->get_property(signal_type = signal_type)
      case signal_type of
      'SIGNAL':   self.signal_container->add,obj_array[i]
      'EMPTY':    self.empty_container->add,obj_array[i]
      else:
      endcase
   endfor
   self->add,obj_array
endif
self.intermediate = obj_new('IDL_CONTAINER')
self.reduced_container = obj_new('IDL_CONTAINER')
self.zip_container = obj_new('IDL_CONTAINER')
self.combine_container = obj_new('IDL_CONTAINER')
self.interp = 'LINEAR'  ; (default)
return,1 
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro dtas_reduce__define
define = {  dtas_reduce,                  $
            interp:'',                    $
            intermediate:obj_new(),       $
            signal_container:obj_new(),   $
            empty_container:obj_new(),    $
            reduced_container:obj_new(),  $
            zip_container:obj_new(),      $
            combine_container:obj_new(),  $
            inherits IDL_CONTAINER  }
end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
