; $Id$
; ************************************ ;
; NAME:
;  DAVE_PARSE_NICE
;
; PURPOSE:
;  This function parses and NICE data file (ASCII column format only) and
;  returns the data contained therein in a structure
;  accessed via an output parameter DSTRUCT.  The input
;  parameter is DSTRING which is a string array containing
;  the entire contents of the data file.
;
; CATEGORY:
;  DAVE, Data Reduction, TAS.
;
; AUTHOR:
;   Richard Tumanjong Azuah
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
; ************************************ ;
function dave_parse_nice, dstring,dstruct    $
  ,header=header $
  ,headText = headerText $
  ,dattaText = dattaText $
  ,indep_var_pos=indep_var_pos $
  ,dep_var_pos=dep_var_pos $
  ;;,plot_indep_var = plot_indep_var $
  ,lattice=lattice $
  ,lattUnits=latticeUnits $
  ,fixedEType=fixedEType $
  ,fixedEValue=fixedEValue $
  ,dSpaceM=dSpaceM $
  ,dSpaceA=dSpaceA $
  ,orientation=orientation $
  ,sdesc=sdesc $
  ,anaPkgStr=anaPkgStr $
  ,errmsg = errmsg $
  ,DetEffkeyVals=DetEffkeyVals $
  ,PSD_DOI_only=PSD_DOI_only $ ; set to specify that only datasets in which PSD was detector-of-interest should be processed
  ,dep_var = dep_var $
  ,indep_var = indep_var $
  ,coltitles = colTitles $
  ,colUnits = colUnits $
  ,quaternion=quaternion $
  ,dspace_MUnits=dspaceMUnits $
  ,dspace_AUnits=dspaceAUnits $
  ,primaryDetector=primaryDetector 

!QUIET = 1

errmsg = ''
catch,the_error
if the_error ne 0 then begin
   catch,/cancel
   errmsg = !error_state.msg
   !QUIET = 0
   return,0B
endif
nlines = n_elements(dstring)


; Determine the length of the header section
headerLen = 0
header = []
index = (where(stregex(dstring,'# NICE',/bool,/fold_case),found))[0]
if (found) then begin
  toks = strsplit(dstring[index],' ',count=nToks,/extract)
  if (nToks ge 5) then begin
    headerLen = fix(toks[4])
    header = dstring[0:headerLen-1]
  endif else begin
    errmsg = 'NICE file has improper format - cannot determine length of header block'
    return, 0B
  endelse
endif else begin
  errmsg = 'Not a NICE dataset!'
  return, 0B
endelse
headerText = strarr(2,headerLen)
for i = 0,headerLen-1 do begin
  ; done to be compartible with dave_parse_ice() but not sure why it is relevant!
  toks = Strsplit(header[i],' ',/extract,count=nToks)
  headerText[0,i] = toks[0:1]
  headerText[1,i] = (nToks ge 3)? Strjoin(toks[2:nToks-1],' ') : ' '
endfor

; The length of the data block
if (nlines eq headerLen) then begin
  errmsg = 'Data block contains no data'
  Return,0B
endif
numPoints = nlines - headerLen
data_string = dstring[headerLen:nlines-1]


; Go through each element of the string array and get out the first data line
; and the last data line
last_line_boolean = 0B
latt_index = -1

dsmon_index = -1
dsana_index = -1
fEi_index = -1
orient_index = -1
sdesc_index = -1
orientation = [0.0,0.0,0.0,0.0,0.0,0.0]
fixedEType = ''
fixedEValue = 0.0
sdesc = ''
anapkg = 0
doOrient = 0

; Number of data points and columns from header
index = (Where(Stregex(header,'# numPoints ',/bool,/fold_case),found))[0]
if (found) then begin
  toks = Strsplit(header[index],' ',count=nToks,/extract)
  if (nToks eq 3) then begin
    numPoints = numPoints < fix(toks[2])   
    ; use the lesser of the two. First estimated from above (nlines - headerlen)
  endif
endif
void = Strsplit(data_string[0],' ',count=numColumns,/extract) ; how many columns in data block
;numColumns = 0
;index = (Where(Stregex(header,'# numColumns ',/bool,/fold_case),found))[0]
;if (found) then begin
;  toks = Strsplit(header[index],' ',count=nToks,/extract)
;  if (nToks eq 3) then begin
;    numColumns = Fix(toks[2])
;  endif else begin
;    ; if not defined in the header then just count the number of entries on the first line of the data block
;    void = Strsplit(data_string[0],' ',count=numColumns,/extract) ; how many columns in data block
;  endelse
;endif

; Process the column titles and units
strTitle = header[headerLen -1]
if (strTitle.Startswith('##')) then begin
  ; thats is expected.
  ; Strip the ## at the beginning
  strTitle = strTitle.Remove(0,2)
endif else begin
  errmsg = 'Error processing data column titles!'
  Return, 0B
endelse
colTitles = []
colUnits = []
titleToks = Strsplit(strTitle,' ',/extract,count=nToks) ; break it up
colTitles = titleToks
colUnits = Strarr(nToks)
indUnits = Where(titleToks.Matches('[()]'),nUnits,ncomplement=nNoUnits,complement=indNoUnits) ; those containing brackets
for i=0,nUnits-1 do begin
  ; loop through col titles with units and strp the units off
  ; the title and store them in a separate unit array
  toks = Strsplit(titleToks[indUnits[i]],'(',/extract,count=nToks)
  if (nToks eq 2) then begin
    colTitles[indUnits[i]] = toks[0]
    colUnits[indUnits[i]] = (toks[1]).Replace(')','')
  endif
endfor

; What is the Scan or Independent variable(s)
indep_var = 'pt'  ; make the data point number (index) the default
indep_var_pos = (Where(colTitles.Matches(indep_var)))[0]
index = (Where(Stregex(header,'# trajectoryData.xAxis',/bool,/fold_case),found))[0]
if (found) then begin
  toks = Strsplit(header[index],' ',count=nToks,/extract)
  if (nToks ge 3) then tmp_indep_var = toks[2]
endif
index = Where(colTitles.Matches(tmp_indep_var), found)
if (found gt 0) then begin 
  indep_var_pos = index[0]   ; if there are multiple matches, then use the first!
  indep_var = tmp_indep_var
  indep_var = indep_var.Replace('.','_')    ; periods in names do not play well so replace with _
endif

;What is the Detector or Dependent variable(s)
dep_var = 'roi' ; make the roi (region of interest) the feault
dep_var_pos = (Where(colTitles.Matches(dep_var)))[0]
index = (Where(Stregex(header,'# trajectoryData.yAxis',/bool,/fold_case),found))[0]
if (found) then begin
  toks = Strsplit(header[index],' ',count=nToks,/extract)
  if (nToks ge 3) then tmp_dep_var = toks[2]
endif
index = Where(colTitles.Matches(tmp_dep_var), found)
if (found gt 0) then begin
  dep_var_pos = index[0]   ; if there are multiple matches, then use the first!
  dep_var = tmp_dep_var
  dep_var = dep_var.Replace('.','_')    ; periods in names do not play well so replace with _
endif


; Process the data block
data = Dblarr(numColumns,numPoints)
dattaText = Strarr(numColumns,numPoints)

;t0 = systime(/seconds)
for i = 0,numPoints-1 do begin
  tmpBuf = Strsplit(data_string[i],' ',/extract)
  dattaText[*,i] = tmpBuf
  
  ; some entries in the data block are desciptive words and not numbers
  ; Converting these to a double will obviously generate an error.
  ; So use regex to find all valid entries that are numbers (contain char 0 - 9)
  ; Replace those that are invalid with with '0.00' before converting to double, 
  ; thus eliminating any convertion errors
  index = Where(tmpBuf.Matches('[0-9]',/fold_case),ncomplement=invalidNumberCount,complement=invalidNumberIndex)
  if (invalidNumberCount gt 0) then tmpBuf[invalidNumberIndex] = '0.00'
  data[*,i] = Double(tmpBuf)
endfor

; Append the dependent data/error to the last column with the title "Signal"/"ERROR".
; this is for backward compartibility with ICE. Not sure why it was done
signal  = data[dep_var_pos,*]   ; NB, signal will be a column vector, not row
error = signal
index = where(error le 0.0, countZero)
if (countZero gt 0) then error[index] = 1.0
error = sqrt(error)

data = [data,signal,error]    ; this append works because both signal and error are column vectors
                              ; if they were row vetors it will fail!
dattaText = [dattaText,string(signal),string(error)]
colTitles = [colTitles,'Signal','ERROR']
colUnits = [colUnits,'','']
numColumns = numColumns+2
dattaText = transpose([transpose(colTitles),transpose(dattaText)])

; Create a structure for the data with the titles used as the tag names.
titles = colTitles.Replace('.','_') ; cat't have periods in structure tags so replace periods with underscore
                                    ; thus Q.K becomes Q_K
titles = titles.Replace('#','Num')
;TODO:
; Seems like there are sometimes duplicate columns in the dataset! 
; Why????????????
; Loop through and identify any and ignore them in the data struture later
goodCount = 1           ; first element is good by default
goodIndex = 0
goodTitles = titles[0]
for i = 1,numColumns-1 do begin
  ; only record an element if it doesn't already exist in goodTitles!
  void = where(strcmp(goodTitles,titles[i],/fold_case), found)
  if (found eq 0) then begin
    goodTitles = [goodTitles,titles[i]]
    goodIndex = [goodIndex,i]
    goodCount++
  endif
endfor
colTitles = goodTitles
numColumns= goodCount
dstruct = Create_struct(colTitles[0],Ptr_new(Reform(data[0,*])))
for i = 1,numColumns-1 do dstruct = Create_struct(dstruct,colTitles[i],Ptr_new(Reform(data[goodIndex[i],*])))


; Retrieve various sample information
; lattice parameter, orientation, Quaternion, 
lattice = dblarr(6)
latticeUnits = strarr(6)
index = (Where(Stregex(header,'# sample.A ',/bool,/fold_case),found))[0]
if (found) then begin
  toks = Strsplit(header[index],' ',count=nToks,/extract)
  lattice[0] = toks[2]
  latticeUnits[0] = toks[3]
endif
index = (Where(Stregex(header,'# sample.B ',/bool,/fold_case),found))[0]
if (found) then begin
  toks = Strsplit(header[index],' ',count=nToks,/extract)
  lattice[1] = toks[2]
  latticeUnits[1] = toks[3]
endif
index = (Where(Stregex(header,'# sample.C ',/bool,/fold_case),found))[0]
if (found) then begin
  toks = Strsplit(header[index],' ',count=nToks,/extract)
  lattice[2] = toks[2]
  latticeUnits[2] = toks[3]
endif
index = (Where(Stregex(header,'# sample.Alpha ',/bool,/fold_case),found))[0]
if (found) then begin
  toks = Strsplit(header[index],' ',count=nToks,/extract)
  lattice[3] = toks[2]
  latticeUnits[3] = toks[3]
endif
index = (Where(Stregex(header,'# sample.Beta ',/bool,/fold_case),found))[0]
if (found) then begin
  toks = Strsplit(header[index],' ',count=nToks,/extract)
  lattice[4] = toks[2]
  latticeUnits[4] = toks[3]
endif
index = (Where(Stregex(header,'# sample.Gamma ',/bool,/fold_case),found))[0]
if (found) then begin
  toks = Strsplit(header[index],' ',count=nToks,/extract)
  lattice[5] = toks[2]
  latticeUnits[5] = toks[3]
endif
; UQuaternion
quaternion = dblarr(4)
index = (Where(Stregex(header,'# sample.UQuaternion ',/bool,/fold_case),found))[0]
if (found) then begin
  toks = Strsplit(header[index],' ',count=nToks,/extract); tokenize "# sample.UQuaternion [0.62512, -0.62512, 0.330492, 0.330492]"
  if (nToks eq 6) then begin 
    json = strjoin(toks[nToks-4:nToks-1]) ; reconstruct "[0.62512, -0.62512, 0.330492, 0.330492]"
    quaternion = Json_parse(json, /toarray) ; and convert into an array
  endif
endif
; Scattering plane / Orientation vectors
orientation = dblarr(6)
index = (Where(Stregex(header,'# sample.refPlane1 ',/bool,/fold_case),found))[0]
if (found) then begin
  toks = Strsplit(header[index],' ',count=nToks,/extract) ; tokenize "# sample.refPlane1 [1.0, 0.0, 0.0]"
  if (nToks eq 5) then begin
    json = Strjoin(toks[nToks-3:nToks-1])
    orientation[0:2] = Json_parse(json, /toarray)
  endif
endif
index = (Where(Stregex(header,'# sample.refPlane2 ',/bool,/fold_case),found))[0]
if (found) then begin
  toks = Strsplit(header[index],' ',count=nToks,/extract) ; tokenize "# sample.refPlane2 [0.0, 0.0, 1.0]"
  if (nToks eq 5) then begin
    json = Strjoin(toks[nToks-3:nToks-1])
    orientation[3:5] = Json_parse(json, /toarray)
  endif
endif
; Monochromator/Analyzer d-spacing
dspaceM = 0.0
dspaceMUnits = ''
index = (Where(Stregex(header,'# ei.dspacing ',/bool,/fold_case),found))[0]
if (found) then begin
  toks = Strsplit(header[index],' ',count=nToks,/extract) ; tokenize "# sample.refPlane2 [0.0, 0.0, 1.0]"
  if (nToks eq 4) then begin
    dspaceM = float(toks[2])
    dspaceMUnits = toks[3]
  endif
endif
dspaceA = 0.0
dspaceAUnits = ''
index = (Where(Stregex(header,'# ef.dspacing ',/bool,/fold_case),found))[0]
if (found) then begin
  toks = Strsplit(header[index],' ',count=nToks,/extract) ; tokenize "# sample.refPlane2 [0.0, 0.0, 1.0]"
  if (nToks eq 4) then begin
    dspaceA = float(toks[2])
    dspaceAUnits = toks[3]
  endif
endif

; Energy mode: fixedEf or fixedEi
fixedEType = '' ; default is undefined
index = (Where(Stregex(header,'# trajectoryData.et ',/bool,/fold_case),found))[0]
if (found) then begin
  if (header[index].Contains('fixedEnergyMode',/fold)) then begin
    if (header[index].Contains('ef',/fold)) then fixedEType = 'EF'
    if (header[index].Contains('ei',/fold)) then fixedEType = 'EI'
  endif
endif
; fixedEValue: not specified and probably not required since Ei and Ef are recorded in data block

; Retrieve the primary detector
primaryDetector = ''
index = (Where(Stregex(header,'# primaryDetector',/bool,/fold_case),found))[0]
if (found) then begin
  toks = Strsplit(header[index],' ',count=nToks,/extract)
  if (nToks ge 3) then primaryDetector = toks[2]
endif
if (Keyword_set(PSD_DOI_only) && ~Strcmp(primaryDetector,'psDetector',/fold)) then begin
  errmsg = 'PSD Detector is not primary detector for dataset!'
  Return, 0B
endif

;; Retrieve AnalyzerDetector package info and fill the anaPkgStr structure
;; AnalyzerDetector mode: for backward compatibility, convert to previous ICE modes
;; Detector of interest

; Mapping scheme for Analyzer-Detector modes between ICE and NICE datasets
;ICE Mode      ICE Mode Names                 NICE modes
;1             DiffDet                        DD_2AXIS
;2             SDFlat                         SD_3AXIS_FLAT
;3             SDHF (Hor Foc)                 SD_3AXIS_ENERGY
;4             PSDDiff                        PSD_2AXIS
;5             PSDFlat                        PSD_3AXIS_FLAT
;6             PSDFixedEf                     PSD_3AXIS_CONSTANT_EF
;7             Undefined                      SD_2AXIS
;8             PSDSeeThrough                  PSD_3AXIS_ENERGY

NICEModeNames=['DD_2AXIS', $          ; from attribute: DAS_logs->analyzerMode->mode->options
               'SD_3AXIS_FLAT', $
               'SD_3AXIS_ENERGY', $
               'PSD_2AXIS', $
               'PSD_3AXIS_FLAT', $
               'PSD_3AXIS_CONSTANT_EF', $
               'SD_2AXIS', $
               'PSD_3AXIS_ENERGY']    
iceAnaDetModeNames = ['DiffDet', $
                      'SDFlat', $
                      'SDHF', $
                      'PSDDiff', $
                      'PSDFlat', $
                      'PSDFixedEf', $
                      'Undefined', $
                      'PSDSeeThrough']

; What are the primary detectors (as present in the data block)
; equivalent to the ANALYZERDETECTORDEVICESOFINTEREST in ICE datasets
index = where(colTitles.Matches('singleDetector',/fold),found)      ; retrieve actual sing det names from the col title
if (found gt 0) then SDDets = colTitles[index]
index = Where(colTitles.Matches('diffDetector',/fold),found)        ; ditto
if (found gt 0) then DDDets = colTitles[index]
index = Where(colTitles.Matches('doorDetector',/fold),found)        ; ditto
if (found gt 0) then TDDets = colTitles[index]
index = Where(colTitles.Matches('psDetector',/fold),found)          ; ditto
if (found gt 0) then PSDDets = colTitles[index]
;
;
;SDDets = 'singleDetector'+(String(Indgen(3))).Trim()   ; the 3  Single Detector names
;DDDets = 'diffDetector'+(String(Indgen(3))).Trim()     ; the 3  Diffraction Detector names
;TDDets = 'doorDetector'+(String(Indgen(11))).Trim()    ; the 11 Door Detector names
;PSDDets = 'psDetector'+(String(Indgen(48))).Trim()    ; the 48 Door Detector names
case primaryDetector.Toupper() of
  'PSDETECTOR': signalDets = PSDDets
  'SINGLEDETECTOR': signalDets = SDDets
  'DIFFDETECTOR': signalDets = DDDets
  'DOORDETECTOR': signalDets = TDDets
  else: signalDets = ''
endcase

; the analyzerDetector mode for the dataset
NICEmodeName = ''
ICEmode = -1
ICEmodeName = 'Undefined'
index = (Where(Stregex(header,'# analyzerMode ',/bool,/fold_case),found))[0]
if (found) then begin
  toks = Strsplit(header[index],' ',count=nToks,/extract)
  if (nToks ge 3) then NICEmodeName = toks[2]
endif
index = where(NICEmodeNames eq NICEmodeName, found)  ; start by looking in current NICE modes
if (found) then begin
  ICEmode = index[0]+1
  ICEmodeName = iceAnaDetModeNames[index[0]]
endif else begin
  errmsg = 'The AnalyzerDetector Mode for this dataset could not be determined!'
  Return, 0B
endelse

anaPkgStr = {mode:ICEmode $
  ,icemodename:ICEmodeName $      ; new field to register the anadetector mode used in previous ICE format
  ,nicemodename:NICEmodeName $    ; new field to register the anadetector mode used in current NICE format
  ,signaldets:signalDets $
  ,sigdetsinsum:Strjoin(signalDets,';',/single) $
,tddetsinsum:Strjoin(TDDets,';',/single) $
,sddets:SDDets $
  ,dddets:DDDets $
  ,tddets:TDDets $
  ,psddets:PSDDets $
}




;::::::::::::::::::::::::::
;old


;; Skipped or bad data points are identified by a time value of -1.0
;; Remove these points from the data
;iTime = where(strupcase(titles) eq 'TIME',foundTime)
;if (foundTime) then begin
;  time = reform(data[iTime,*])
;  void = where(time lt 0.0,badTimeCnt,COMPLEMENT=goodIndex,ncomplement=goodtimeCnt)
;  if (goodtimeCnt eq 0) then begin
;    errmsg = 'All data points in dataset are tagged as incomplete!'
;    Return, 0B
;  endif
;  if (badTimeCnt gt 0) then begin
;    data = temporary(data[*,goodIndex])
;    dattaText = temporary(dattaText[*,goodIndex])
;    ndata = ndata - badTimeCnt
;  endif
;endif
;
;; Calculate the basis vectors given two orientation vectors and the hkl values.
;; Insert these as additional 'scan' variables in the data immediately after qx, qy, qz and E
;; Note: the new scan variables are only meaningful for Q Scans (Q varying)
;iqx = where(strupcase(titles) eq 'QX',foundx)
;iqy = where(strupcase(titles) eq 'QY',foundy)
;iqz = where(strupcase(titles) eq 'QZ',foundz)
;if (foundx && foundy && foundz && doOrient) then begin
;   qx = reform(data[iqx,*])
;   qy = reform(data[iqy,*])
;   qz = reform(data[iqz,*])
;   orient1 = orientation[0:2]
;   orient2 = orientation[3:5] 
;   basis = fltarr(3,ndata)
;   for i=0,ndata-1 do begin
;      hkl = [qx[i],qy[i],qz[i]]
;      basis[*,i] = HKLtoBasis(orient1,orient2,hkl)
;   endfor
;   ;lab1 = 'ORIENT'+strjoin(string(orient1))
;   ;lab2 = 'ORIENT'+strjoin(string(orient2))
;   ; replace any '-' with '_' in lab1 and lab2
;   while ((pos = stregex(lab1,'-')) ne -1) do strput, lab1,'_',pos
;   while ((pos = stregex(lab2,'-')) ne -1) do strput, lab2,'_',pos
;
;   ; insert into data and title arrays after qx,qy,qz
;   last = max([iqx[0],iqy[0],iqz[0]])
;   ;titles = [titles[0:2],lab1,lab2,titles[3:ncols-1]]
;   ;data = [data[0:2,*],basis[0:1,*],data[3:ncols-1,*]]
;   titles = [titles[0:last],lab1,lab2,titles[last+1:ncols-1]]
;   data = [data[0:last,*],basis[0:1,*],data[last+1:ncols-1,*]]
;   dattaText = [dattaText[0:last,*],string(basis[0:1,*]),dattaText[last+1:ncols-1,*]]
;   
;   ; increment indices,counts by 2 if necessary
;   ncols += 2
;   if (indep_var_pos gt last) then indep_var_pos += 2 
;   if (dep_var_pos gt last) then dep_var_pos += 2
;endif

!QUIET = 0
return,1B
end
; ************************************ ;
pro test_read_nice
;filename = sourcepath()+'blah57.bt7' ;'PrCMO6024.bt7';'fpx6013.bt7' ; Qscan02820.btx
filename = dialog_pickfile(title="Select NICE file",multiple_files=0,/must_exist, dialog_parent=0L $
  ,filter="*.bt7");,path=sourcepath())
if (filename eq '') then return
read_ok = dave_read_filecontents(filename,dstring = dstring,errmsg = errmsg)
if ~read_ok then begin
   print,errmsg
   return
endif
parse_ok = dave_parse_nice(dstring,dstruct,header=header $
                          ,headText = headText $
                          ,indep_var_pos=indep_ind $
                          ,dep_var_pos=dep_ind $
                          ,lattice=lattice $
                          ,fixedEType=fixedEType $
                          ,fixedEValue=fixedEValue $
                          ,dSpaceM=dSpaceM $
                          ,dSpaceA=dSpaceA $
                          ,orientation=orientation $
                          ,sdesc=sdesc $
                          ,anaPkgStr=anaPkgStr $
                          ,errmsg = errmsg)
if ~parse_ok then print,errmsg

if ~parse_ok then return

names = tag_names(dstruct)
x = (*dstruct.(indep_ind))
y = (*dstruct.(dep_ind))
dy = (*dstruct.error)
p1 = errorplot(x,y,dy)

end
