;$Id$
;###############################################################################
;
; NAME:
;  ADD_SCAN
;
; PURPOSE:
;  Adds fixed window scan files together.
;
; CATEGORY:
;  DAVE, HFBS, data reduction
;
; AUTHOR:
;   Robert M. Dimeo, Ph.D.
;   NIST Center for Neutron Research
;   100 Bureau Drive
;   Gaithersburg, MD 20899
;   Phone: (301) 975-8135
;   E-mail: robert.dimeo@nist.gov
;   http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;+
; NAME:
;       ADD_SCAN
;
; PURPOSE:
;
;   Simple function wrapper for the "add" functionality of hscn_addrun.
;   Suitable for invocation at the command line.  The function returns
;   a new object reference containing the data for the summed datasets.
;
; AUTHOR:
;
;       Robert M. Dimeo, Ph.D.
;   NIST Center for Neutron Research
;       100 Bureau Drive
;   Gaithersburg, MD 20899
;       Phone: (301) 975-8135
;       E-mail: robert.dimeo@nist.gov
;       http://www.ncnr.nist.gov/staff/dimeo
;
; CATEGORY:
;
;       Data reduction, DAVE project, HFBS
;
; CALLING SEQUENCE:
;
;   OADD = ADD_SCAN(  OARRAY,            $
;              POISSON = poisson,       $
;              INTERP = interp       )
; PARAMETERS:
;
;   OARRAY:   Array of (at least 2) data objects to be summed.  These are
;         assumed to be of the HSCN_DATA class.
;
; KEYWORDS:
;
;   POISSON:    If set then addition will proceed using the proper
;            Poisson statistics. (optional)
;   INTERPOLATE:   If set then data that do not share the same independent
;            variable values will be interpolated so that they do.
;            (optional)
;
; COMMON BLOCKS:
;
;   None
;
; REQUIREMENTS:
;
;   HSCN_ADDRUN.PRO
;   HSCN_DATA__DEFINE.PRO
;
; MODIFICATION HISTORY:
;
;       Written by Rob Dimeo, July 10, 2003
;-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function add_scan,oarray,poisson = poisson, interp = interp
; Check if correct number of parameters were passed in
if n_params() eq 0 then return,0
; Set default values for the keywords if not specified
if n_elements(poisson) eq 0 then poisson = 0
if n_elements(interp) eq 0 then interp = 0
ndata = n_elements(oarray)
if ndata lt 2 then return,0

; Go through and make sure that the independent variables for each data set
; are set the same way.
ret = oarray[0]->get(indep_var = indep_var)
label = indep_var.label
for i = 1,ndata-1 do begin
   ret = oarray[i]->get(indep_var = indep_var)
   if indep_var.label ne label then begin
     str = 'ret = oarray[i]->set_indep_var(/'+strtrim(label,2)+')'
     r = execute(str,1)
   endif
endfor

for detector = 0,23 do begin
   for i = 0,ndata-1 do begin
     ret = oarray[i]->set(detector = detector)
     s = oarray[i]->create_addrun_data_structure()
     if (i eq 0) then sarray = s else sarray = [sarray,s]
   endfor
   s_sum = hscn_addrun(sarray,/add,poisson = poisson,interp = interp,msg = msg)
   heap_free,sarray
   if msg ne '' then begin
     void = dialog_message(msg,/error)
     return,0
   endif
   if detector eq 0 then s_sum_array = s_sum else $
     s_sum_array = [s_sum_array,s_sum]
endfor
; Now cycle through the s_sum_array, pull out the values and populate
; a new data array and all of the other relevant instance data for the
; HSCN_DATA class.
;
; Get the 1st element out
d = s_sum_array[0]
x = *d.px
y = *d.py
dy = *d.pdy
mon = *d.pmon
dmon = *d.pdmon
heap_free,d
; Clone an existing data object.  ONEW will be the resulting data object
; of the add procedure.
ret = oarray[0]->clone(cloned_object = onew)
ret = onew->get(indep_var = indep_var)
*indep_var.pqty = x & nx = n_elements(x)


; Which independent variable is it?
case indep_var.identity of
'TCONTROL': $
   begin
     tcontrol = (tsample = x)
     time = findgen(n_elements(x))
   end
'TSAMPLE':  $
   begin
     tcontrol = (tsample = x)
     time = findgen(n_elements(x))
   end
'TIME':  $
   begin
     time = x
     tsample = (tcontrol = findgen(n_elements(x)))
   end
else:
endcase

ret = onew->set(indep_var = indep_var)
data = fltarr(24,nx)
error = fltarr(24,nx)
data[0,*] = mon[*]
error[0,*] = dmon[*]

for i = 1,23 do begin
   d = s_sum_array[i]
   x = *d.px & y = *d.py & dy = *d.pdy
   mon = *d.pmon & dmon = *d.pdmon
   data[i,*] = y[*]
   error[i,*] = dy[*]
   heap_free,d
endfor
wbm = data[17,*]
tbm = data[18,*]
;tcontrol = data[5,*]
;tsample = data[6,*]
;time = data[0,*]
livetime = data[1,*]
ret = onew->set(data = data,error = error,fc = mon,indep_var = indep_var, $
         wbm = wbm,tbm = tbm,tcontrol = tcontrol,tsample = tsample, $
         time = time,livetime = livetime)
return,onew
end