;#######################################################################
;
; NAME:
;  dm_write_macs
;
; PURPOSE:
;  This program writes a MACS data file, in ICE ascii column file or 
;  nexus HDF file format.
;  
; CATEGORY:
;  dcs_mslice tools
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  December, 2022
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

;combine a string array input to form a data line of the ICE file
function dm_reform_iceline,ins,header=header,maxlen=maxlen
    if n_elements(maxlen) lt n_elements(ins) then maxlen = strlen(ins)
    if n_elements(header) eq 0 then outs = '' else outs = header
    for i=0,8-strlen(outs) do outs = outs+' '
    for i=0L,n_elements(ins)-1 do begin
        if strlen(ins[i]) gt (13>(maxlen[i])) then outs = outs+' '
        for j=0,(13>(maxlen[i]))-strlen(ins[i]) do outs = outs+' '
        outs = outs+ins[i]
    endfor
    return,outs
end

;write data to a macs file
;parameter:
;   file:         file name of the saved file, a string scalar
;keyword:
;   header:       header info, a string array
;   data:         data, [npoint] string array or [npoint,ncol] double array if nexus keyword is set
;   info:         column id, a string array [ncol]
;   is_histogram: flag for histogram data
;   mesgobj:      message object from calling routine, if error occured, the mesgobj will be destoryed
;   nexus:        if set, save the file as nexus hdf file
;   str_histo:    combined with is_histogram keyword for histogram data
;   trailer:      trailer info, a string array
;   comment,deteff,latticeparm,latticeori,maspacing,t_mcfx see dm_load_macs
pro dm_write_macs,file,data=data,comment=comment,deteff=deteff,header=header,info=info,instrname=instrname,is_histogram=is_histogram,latticeparm=latticeparm,latticeori=latticeori,$
    maspacing=maspacing,mesgobj=mesgobj,nexus=nexus,str_histo=str_histo,trailer=trailer,t_mcfx=t_mcfx
    if keyword_set(nexus) then begin
       ;extrac info from header if missing
       if n_elements(comment) eq 0 then begin
          ind = where(stregex(header,'#Comment',/fold_case,/boolean),cnt)
          if cnt eq 1 then begin
             tmp = stregex(header[ind],'^ *#Comment *(.*)',/subexpr,/fold_case,length=len)
             if len[1] gt 0 then begin
                comment = strmid(header[ind],tmp[1],len[1])
                if (strmid(comment,0,1) eq '"') and (strmid(comment,0,1,/reverse_offset) eq '"') then comment = strmid(comment,1,strlen(comment)-2)
             endif
          endif
       endif
       if n_elements(deteff) ne 40 then begin
          ind = where(stregex(header,'#DetectorEfficiencies',/fold_case,/boolean),cnt)
          if cnt eq 1 then begin
             tmp = stregex(header[ind],'^ *#DetectorEfficiencies *(.*)',/subexpr,/fold_case,length=len)
             if len[1] gt 0 then begin
                deteff = fltarr(2,20)+1.0
                tmp1   = strsplit(strmid(header[ind],tmp[1],len[1]),' '+string(9b),/extract,count=count)
                for j=0,count-1 do begin
                    ind = stregex(tmp1[j],'^spec([0-9]+)=',length=len,/fold_case,/subexpr)
                    if ind[0] ne -1 then deteff[0,fix(strmid(tmp1[j],ind[1],len[1]))-1] = float(strmid(tmp1[j],ind[0]+len[0]))
                    ind = stregex(tmp1[j],'^diff([0-9]+)=',length=len,/fold_case,/subexpr)
                    if ind[0] ne -1 then deteff[1,fix(strmid(tmp1[j],ind[1],len[1]))-1] = float(strmid(tmp1[j],ind[0]+len[0]))
                endfor
             endif
          endif
       endif
       if n_elements(instrname) eq 0 then begin
          ind = where(stregex(header,'#InstrName',/fold_case,/boolean),cnt)
          if cnt eq 1 then begin
             tmp = stregex(header[ind],'^ *#InstrName *(.*)',/subexpr,/fold_case,length=len)
             if len[1] gt 0 then instrname = strmid(header[ind],tmp[1],len[1])
          endif else instrname='MACS'
       endif else instrname='MACS'
       if n_elements(latticeparm) ne 6 then begin
          ind = where(stregex(header,'#Lattice',/fold_case,/boolean),cnt)
          if cnt eq 1 then begin
             tmp = stregex(header[ind],'^ *#Lattice *(.*)',/subexpr,/fold_case,length=len)
             if len[1] gt 0 then begin
                latticeparm = fltarr(6)
                latticeparm[*] = dm_to_number(strsplit(strmid(header[ind],tmp[1],len[1]),' '+string(9b),/extract))
             endif
          endif
       endif
       if n_elements(latticeori) ne 6 then begin
          ind = where(stregex(header,'#Orient',/fold_case,/boolean),cnt)
          if cnt eq 1 then begin
             tmp = stregex(header[ind],'^ *#Orient *(.*)',/subexpr,/fold_case,length=len)
             if len[1] gt 0 then begin
                latticeori = fltarr(6)
                latticeori[*] = dm_to_number(strsplit(strmid(header[ind],tmp[1],len[1]),' '+string(9b),/extract))
             endif
          endif
       endif
       if n_elements(maspacing) ne 2 then begin
          ind1 = where(stregex(header,'#MonoSpacing',/fold_case,/boolean),cnt1)
          ind2 = where(stregex(header,'#AnaSpacing',/fold_case,/boolean),cnt2)
          maspacing = [3.35416,3.35416] ;default graphite
          if cnt1 eq 1 then begin
             tmp = stregex(header[ind1],'^ *#MonoSpacing *(.*)',/subexpr,/fold_case,length=len)
             maspacing[0] = dm_to_number(strmid(header[ind1],tmp[1],len[1]))
          endif
          if cnt2 eq 1 then begin
             tmp = stregex(header[ind2],'^ *#AnaSpacing *(.*)',/subexpr,/fold_case,length=len)
             maspacing[1] = dm_to_number(strmid(header[ind2],tmp[1],len[1]))
          endif
       endif
       if n_elements(t_mcfx) eq 0 then t_mcfx = 300
       
       ; file handle must be a long integer and must be set before calling NXopen()
       handle = 0L
    
       ; Open the hdf file
       ; The file handle will subsequently be used just like a file unit
       if (~NXopen(file,'NXACC_CREATE5',handle)) then begin
          if obj_valid(mesgobj) then obj_destroy,mesgobj
          message,"Can't open "+file+"."
       endif
       entry  = '/entry'
       nxcoll = 'NXcollection'
       status = NXmakegroup(handle,entry,'NXentry')
       status = NXopengroup(handle,entry,'NXentry')
       ;entry group,                              class="NXentry"
       status = NXmakegroup(handle,'DAS_logs',nxcoll)
       status = NXopengroup(handle,'DAS_logs',nxcoll)
       ;    DAS_logs group,                       class="NXcollection"
       ;        CFXBe group,                      class="NXcollection"  
       ;            primaryNode
       ;        CFXHOPG group,                    class="NXcollection"
       ;            primaryNode
       ;        PTAI group,                       class="NXcollection"
       ;            index
       ;        Q group,                          class="NXcollection"
       ;            H
       ;            K
       ;            L
       ;        aColmon group,                    class="NXcollection"
       ;            primaryNode
       ;        anaTheta group,                   class="NXcollection"
       ;            primaryNode
       ;        anaTwoTheta group,                class="NXcollection"
       ;            primaryNode
       ;        anaTwoTheta01 group,              class="NXcollection"
       ;            softPosition
       ;            |
       ;        anaTwoTheta20 group,              class="NXcollection"
       ;            softPosition
       ;        analyzerFilterTemp group,         class="NXcollection"
       ;            primaryNode group,            class="NXlog"
       ;                value
       ;        bColmon group,                    class="NXcollection"
       ;            primaryNode
       ;        beta1Motor group,                 class="NXcollection"
       ;            softPosition
       ;        beta2Motor group,                 class="NXcollection"
       ;            softPosition
       ;        diffDetector group,               class="NXcollection"
       ;            detectorEfficiency
       ;        dmbtMotor group,                  class="NXcollection"
       ;            softPosition
       ;        ef group,                         class="NXcollection"
       ;            dSpacing
       ;            energy
       ;        ei group,                         class="NXcollection"
       ;            dSpacing
       ;            energy
       ;        et group,                         class="NXcollection"
       ;            deltaE
       ;        experiment group,                 class="NXcollection"
       ;            description
       ;            instrument
       ;        hSlitMotor group,                 class="NXcollection"
       ;            softPosition
       ;        kidneyMotor group,                class="NXcollection"
       ;            softPosition
       ;        mag group,                        class="NXcollection"
       ;            primaryNode group,            class="NXlog"
       ;                value
       ;        mbtSlideMotor group,              class="NXcollection"
       ;            softPosition
       ;        mcfxMap group,                    class="NXcollection"
       ;            key
       ;        monoBlade01 group,                class="NXcollection"
       ;            softPosition
       ;           |
       ;        monoBlade21 group,                class="NXcollection"
       ;            softPosition
       ;        monoDTSMotor group,               class="NXcollection"
       ;            softPosition
       ;        monoRotation group,               class="NXcollection"
       ;            softPosition
       ;        monoTheta group,                  class="NXcollection"
       ;            hfocus
       ;            radiusOfCurvature1
       ;            radiusOfCurvature2
       ;            theta
       ;            vfocus
       ;        monoTrans group,                  class="NXcollection"
       ;            softPosition
       ;        monoTwoTheta group,               class="NXcollection"
       ;            a2Ref
       ;            mbtPivot
       ;            primaryNode
       ;        monoVbaHMotor group,              class="NXcollection"
       ;            softPosition
       ;        monoVbaVMotor group,              class="NXcollection"
       ;            softPosition
       ;        sampleElev group,                 class="NXcollection"
       ;            softPosition
       ;        sampleLTilt group,                class="NXcollection"
       ;            softPosition
       ;        sampleState group,                class="NXcollection"
       ;            A
       ;            Alpha
       ;            B
       ;            Beta
       ;            C
       ;            Gamma
       ;            refPlane1
       ;            refPlane2
       ;        sampleTheta group,                class="NXcollection"
       ;            primaryNode
       ;        sampleThetaMotor group,           class="NXcollection"
       ;            softPosition
       ;        sampleTwoTheta group,             class="NXcollection"
       ;            position
       ;        sampleUTilt group,                class="NXcollection"
       ;            softPosition
       ;        sampleXTranslationMotor group,    class="NXcollection"
       ;            softPosition
       ;        sampleYTranslationMotor group,    class="NXcollection"
       ;            softPosition
       ;        spec group,                       class="NXcollection"
       ;            count
       ;        specDetector group,               class="NXcollection"
       ;            detectorEfficiency
       ;        temp group,                       class="NXcollection"    ;way too complicated, needs to be changed
       ;            sensor_A group,               class="NXlog"
       ;                value
       ;            sensor_B group,               class="NXlog"
       ;                value
       ;            sensor_C group,               class="NXlog"
       ;                value
       ;            sensor_D group,               class="NXlog"
       ;                value
       ;            controlLoopSensor_1
       ;            primaryControlLoop (value has to be 1)
       ;            primarySensor
       ;        trajectoryData group,             class="NXcollection"
       ;            description
       ;        vSlitMotor group,                 class="NXcollection"
       ;            softPosition
       status = NXclosegroup(handle)     ; DAS_logs
       status = NXmakegroup(handle,'control','NXmonitor')
       status = NXopengroup(handle,'control','NXmonitor')
       ;    control group,                        class="NXmonitor"
       ;        count_against
       ;        count_end (with start attribute, using entry/start_time)
       ;        count_time
       ;        monitor_counts
       status = NXclosegroup(handle)     ; control
       status = NXmakegroup(handle,'data','NXdata')
       status = NXopengroup(handle,'data','NXdata')
       ;    data group,                           class="NXdata"
       ;        diff
       ;        diffDetector
       ;        spec
       ;        specDetector
       status = NXclosegroup(handle)     ; data
       status = NXmakegroup(handle,'instrument','NXinstrument')
       status = NXopengroup(handle,'instrument','NXinstrument')
       ;    instrument group,                     class="NXinstrument"
       status = NXmakedata(handle,'name','NX_CHAR',1,strlen(instrname))
       status = NXopendata(handle,'name')
       status = NXputdata(handle,instrname)
       status = NXclosedata(handle)
       ;        name
       status = NXclosegroup(handle)     ; instrument
       tmp = stregex(header[0],'^ *#([a-z]+) *(.*)',/subexpr,/fold_case,length=len)
       if len[1] gt 0 then program = strmid(header[0],tmp[1],len[1]) else program = 'NICE'
       if len[2] gt 0 then version = strmid(header[0],tmp[2],len[2]) else version = '0'
       status = NXmakedata(handle,'program_name','NX_CHAR',1,strlen(program))
       status = NXopendata(handle,'program_name')
       status = NXputdata(handle,program)
       status = NXputattr(handle,'version',version)
       status = NXclosedata(handle)
       ;    program_name
       start_time = ' '
       ind = where(stregex(header,'#Date',/fold_case,/boolean),cnt)
       if cnt eq 1 then begin
          tmp = stregex(header[ind],'^ *#Date *(.*)',/subexpr,/fold_case,length=len)
          if len[1] gt 0 then start_time = strmid(header[ind],tmp[1],len[1])
       endif
       status = NXmakedata(handle,'start_time','NX_CHAR',1,strlen(start_time))
       status = NXopendata(handle,'start_time')
       status = NXputdata(handle,start_time)
       status = NXclosedata(handle)
       ;    start_time
       status = NXclosegroup(handle)     ; NXentry
       status = NXclose(handle)          ; Nexus/hdf file
    endif else begin
       maxlen = max(strlen(data),dimension=1)>(strlen(info))
       if strlowcase(info[-1]) eq 'timestamp' then maxlen[-1] = maxlen[-1]+3
       openw,unitw,file,/get_lun,error=error
       if error ne 0 then begin
          if obj_valid(mesgobj) then obj_destroy,mesgobj
          message,!ERROR_STATE.MSG,level=-1,/noprint,/noname
       endif
       for ii=0L,n_elements(header)-1 do printf,unitw,header[ii]
       printf,unitw,dm_reform_iceline(info,header='#Columns',maxlen=maxlen)
       for ii=0L,n_elements(data[*,0])-1 do begin
           printf,unitw,dm_reform_iceline(data[ii,*],maxlen=maxlen)
           if keyword_set(is_histogram) then printf,unitw,str_histo[ii]
       endfor
       for ii=0L,n_elements(trailer)-1 do printf,unitw,trailer[ii]
       free_lun,unitw
    endelse
end

pro test
infile = 'C:\Users\qiuym\Documents\Yiming\data\MACS\nexus\fp_kidney_9.nxs.ng0'
outfile = 'C:\Users\qiuym\Documents\Yiming\data\MACS\nexus\fp_kidney_9.nxs.bt9'
dm_load_macsnexus,infile,data,info,comment=comment,deteff=deteff,header=header,histodata=histodata,instrname=instrname,latticeori=latticeori,latticeparm=latticeparm,$
    maspacing=maspacing,t_mcfx=t_mcfx,error=error
if keyword_set(error) then return
dm_write_macs,outfile,data=data,comment=comment,deteff=deteff,header=header,info=info,instrname=instrname,latticeparm=latticeparm,latticeori=latticeori,$
    maspacing=maspacing,mesgobj=mesgobj,/nexus,t_mcfx=t_mcfx  
end