; $Id$
;#######################################################################
;
; NAME:
;  dm_widget_button
;
; PURPOSE:
;  this program creates a widget_button,checked_menu keyword is set for IDL 5.6 or later
;
; CATEGORY:
;  widget
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-8562
;  United States
;  yiming.qiu@nist.gov
;  January, 2024
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

;read an .ico file
function dm_read_icon,file,red,green,blue,ihdr=ihdr,rgb=rgb,targetsize=targetsize
    openr,unit,file,/get_lun,/swap_if_big_endian
    header = intarr(3) & readu,unit,header
    isize  = bytarr(4,header[2]) & tmp1 = bytarr(4)
    cpbs   = intarr(2,header[2]) & tmp2 = intarr(2)
    dsize  = lonarr(header[2])   & tmp3 = 0L
    doffs  = lonarr(header[2])   & tmp4 = 0L
    for i=0,header[2]-1 do begin
        readu,unit,tmp1 & isize[*,i] = tmp1
        readu,unit,tmp2 & cpbs[*,i] = tmp2
        readu,unit,tmp3 & dsize[i] = tmp3
        readu,unit,tmp4 & doffs[i] = tmp4
    endfor
    tmp = bytarr(8)
    ispng = bytarr(header[2])
    pngheader = ['89'xb,'50'xb,'4e'xb,'47'xb,'0d'xb,'0a'xb,'1a'xb,'0a'xb]
    for i=0,header[2]-1 do begin
        point_lun,unit,doffs[i]
        readu,unit,tmp
        ispng[i] = total(abs([float(tmp)-pngheader])) eq 0
    endfor
    nonpng = where(~ispng,count)
    if count eq 0 then return,0
    if n_elements(targetsize) ne 0 then begin ;find the icon with the desired size
       i_max = where(isize[0,*] eq targetsize and isize[1,*] eq targetsize,count)
       if count eq 0 then tmp = temporary(i_max)
    endif
    if n_elements(i_max) eq 0 then tmp = max(dsize[nonpng],i_max)  ;find the largest non-png image location
    point_lun,unit,doffs[nonpng[i_max[0]]]
    ihdr = { BITMAPINFOHEADER, $
             bisize: 0L, $
             biwidth: 0L, $
             biheight: 0L, $
             biplanes: 0, $
             bibitcount: 0, $
             bicompression: 0L, $
             bisizeimage: 0L, $
             bixpelspermeter: 0L, $
             biypelspermeter: 0L, $
             biclrused: 0L, $
             biclrimportant: 0L $
           }
    readu,unit,ihdr
    big_endian = (byte(1,0,2))[0] eq 0b
    if big_endian then ihdr = swap_endian(ihdr)
    ;Pseudo color?
    if (ihdr.bibitcount lt 16)  then begin
       ncolors = 2L^ihdr.bibitcount
       ; See if we have less than the maximum number of colors.
       if (ihdr.biclrused gt 0 && ihdr.biclrused lt ncolors) then ncolors = ihdr.biclrused
       colors = bytarr(4, ncolors)
       readu, unit, colors               ;Read colors
       red = reform(colors[2, *])        ;Decommutate colors
       green = reform(colors[1, *])
       blue = reform(colors[0, *])
    endif
    nx = ihdr.biwidth       ;Columns
    ny = ihdr.biheight/2    ;Rows
    case ihdr.bibitcount of ;How many bits/pixel?
         4: begin
            data = bytarr(nx, ny, /nozero)
            buff = bytarr(nx/2, /nozero)   ;Line buffer
            even = lindgen(nx/2) * 2
            odd = even + 1
            if nx and 1 then pad = 0B       ;interbyte padding
            i = (n_elements(buff) + n_elements(pad)) and 3  ;bytes we have
            if i ne 0 then pad = bytarr(4-i+n_elements(pad))
            for i=0, ny-1 do begin
                if n_elements(pad) ne 0 then readu, unit, buff, pad $
                else readu, unit, buff
                data[even, i] = ishft(buff, -4)
                data[odd, i] = buff and 15b
                if nx and 1 then data[nx-1, i] = ishft(pad[0], -4) ;Last odd byte?
            endfor
            end
         8: begin
            data = bytarr(nx, ny, /nozero)
            if (nx and 3) eq 0 then readu, unit, data $          ;Slam dunk it
            else begin                      ;Must read line by line...
               pad = bytarr(4 - (nx and 3))
               buff = bytarr(nx, /nozero)
               for i=0, ny-1 do begin       ;Each line
                   readu, unit, buff, pad
                   data[0,i] = buff
               endfor
            endelse
            end
        16: begin     ;16 bits/pixel
            data = intarr(nx * ny, /nozero)  ;Read as a 1D array
            if (nx and 1) eq 0 then readu, unit, data $  ;If even # of cols, no padding
            else begin
               pad = 0     ;Pad 1 16 bit word
               buff = intarr(nx, /nozero)
               for i=0, ny-1 do begin
                   readu, unit, buff, pad
                   data[i * nx] = buff    ;Insert line
               endfor
            endelse
            if big_endian then byteorder, data, /sswap
            r = byte(ishft(data,-7) and 248) ;Shift down 10 and up 3
            g = byte(ishft(data,-2) and 248) ;down 5 and up 3
            data = byte(ishft(data, 3) and 248) ;up 3, really blue
            if keyword_set(RGB) then data = [r,g,data] $  ;Unswitch it?
            else data = [data,g,r]
            data = reform(data, nx * ny, 3, /overwrite) ;Diddle to 3,nx,ny
            data = transpose(temporary(data))   ;Color interleave in 1st dimension
            data = reform(data,3,nx,ny,/overwrite)
            end
        24: begin                    ;24 bits / pixel....
            data = bytarr(3, nx, ny, /nozero)
            ;If even # of cols, no padding
            if ((3 * nx) and 3) eq 0 then readu, unit, data $
            else begin
               pad = bytarr(4 - ((3 * nx) and 3))
               buff = bytarr(3, nx, /nozero)
               for i=0, ny-1 do begin
                   readu, unit, buff, pad
                   data[0,0, i] = buff            ;Insert line
               endfor
            endelse
            if keyword_set(RGB) then begin  ;Unswitch it?
               buff = data[0,*,*]
               data[0,0,0] = data[2,*,*]
               data[2,0,0] = buff
            endif
            end
        32: begin                    ;32 bits / pixel....
            data = bytarr(4, nx, ny, /nozero)
            ;If even # of cols, no padding
            if ((4 * nx) and 3) eq 0 then readu, unit, data $
            else begin
               pad = bytarr(4 - ((3 * nx) and 3))
               buff = bytarr(4, nx, /nozero)
               for i=0, ny-1 do begin
                   readu, unit, buff, pad
                   data[0,0, i] = buff            ;Insert line
               endfor
            endelse
            if keyword_set(RGB) then begin  ;Unswitch it?
               buff = data[0,*,*]
               data[0,0,0] = data[2,*,*]
               data[2,0,0] = buff
            endif
            end
      else:
    endcase
    free_lun,unit
    return, data
end

;resize icon image using cubic interpolation
;parameters:
;   data:       a [3,n,m] byte array, or [n,m] byte array with r,g,b keywords set; or a string of the image file name
;   height:     new_m, default is 16
;keywords:
;   cubic:      cubic interpolation parameter, between -1 and 0, default is -0.5
;   tolerance:  transparency tolerance with respect to the lower left corner pixel, default is 20
;   transpose:  if set transpose the index to [1,2,0]
;   r,g,b:      color table for [n,m] data array
function dm_resizeicon,data,height,cubic=cubic,tolerance=tolerance,transpose=transpose,r=r,g=g,b=b
    if n_elements(data)      eq 0 then return,0
    if n_elements(height)    ne 1 then height=16
    if n_elements(cubic)     ne 1 then cubic=-0.5
    if n_elements(tolerance) ne 1 then tolerance=20
    if size(data,/type) eq 7 then begin     ;data is a file name
       type = strlowcase(strmid(data[0],strpos(data[0],'.',/reverse_search)+1))
       case type of
            'ico': data1 = dm_read_icon(data[0],/rgb,r,g,b,targetsize=height)
            'bmp': data1 = read_bmp(data[0],/rgb,r,g,b)
            'gif': read_gif,data[0],data1,r,g,b
            'png': data1 = read_png(data[0],r,g,b)
            else:  return,0
       endcase
    endif else data1 = data
    oldsize = size(data1,/dimension)
    if n_elements(oldsize) eq 1 then return,data1
    if n_elements(oldsize) eq 2 then begin
       if (n_elements(r) eq 0) or (n_elements(g) eq 0) or (n_elements(b) eq 0) then return,data1  ;no change in size
       data1 = transpose(data1)
       data1 = transpose([[[r[data1]]],[[g[data1]]],[[b[data1]]]])
       oldsize = [3,oldsize]
    endif
    newsize = [round(float(oldsize[1])*height/oldsize[2]),height] ;keep original aspect ratio
    if (newsize[0] eq oldsize[1]) and (newsize[1] eq oldsize[2]) then return, keyword_set(transpose)?transpose(data1,[1,2,0]):data1
    x = (oldsize[1]-1)*((0.5+findgen(newsize[0]))/newsize[0])
    y = (oldsize[2]-1)*((0.5+findgen(newsize[1]))/newsize[1])
    data1 = interpolate(fix(data1),x,y,/grid,cubic=cubic)         ;convert to integer first
    ind = where((abs(data1[0,*,*]-data1[0,0,0]) lt tolerance) and (abs(data1[0,*,*]-data1[1,*,*]+data1[1,0,0]-data1[0,0,0]) le 2) and (abs(data1[1,*,*]-data1[2,*,*]+data1[2,0,0]-data1[1,0,0]) le 2),count)
    if count gt 0 then for i=0,count-1 do data1[0:2,ind[i] mod newsize[0],ind[i]/newsize[0]] = data1[0:2,0,0]    ;for transparency
    data1 = byte(0>(data1)<255)
    return, keyword_set(transpose)?transpose(data1,[1,2,0]):data1
end

; parameter:
;   parent:     the widget ID of the parent for the new button widget
; keyword:
;   accelator:  accelator for buttons, IDL 6.1 or later
;   imagefile:  file name of the image for the button, IDL 8.2 or later and windows
;   set_button: initial check state of the button
;   onstring:   button value when set_button=1 for IDL 5.5 or before
;   offstring:  button value when set_button=0 for IDL 5.5 or before
;   notchecked: not a checked menu button, default is a checked menu button
function dm_widget_button,parent,accelator=accelator,imagefile=imagefile,set_button=set_button,notchecked=notchecked,onstring=onstring,offstring=offstring,_extra=extra
    idl_version = dm_to_number(!version.release)
    if n_elements(extra) eq 0 then extra = {no_copy:1}
    if (idl_version ge 5.6) and ~keyword_set(notchecked) then extra = create_struct(extra,'checked_menu',1)               ;checked_menu keyword
    if (n_elements(accelator) eq 1) and (idl_version ge 6.1) then extra = create_struct(extra,'accelerator',accelerator)  ;accelerator keyword
    if (n_elements(imagefile) eq 1) and (idl_version ge 8.2) and (!version.os_family eq 'Windows') then extra = create_struct(extra,'image',dm_resizeicon(imagefile,/transpose))  ;image keyword
    id = widget_button(parent,_extra=extra)
    if n_elements(set_button) eq 1 then dm_set_button,id,set_button,onstring=onstring,offstring=offstring
    return,id
end