; $Id: $
;#######################################################################
;
; NAME:
;  dm_stepgrid_bin
;
; PURPOSE:
;  bin data according to give xstep and ystep
;
; CATEGORY:
;  general
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  June, 2023
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

; parameters:
;   in_xstep:      xstep
;   in_ystep:      ystep
;   xdat:          input xdat[ne,ndet]    output xdat[xsize]
;   ydat:          input ydat[ne,ndet]    output ydat[ysize]
;   zdat:          input zdat[ne,ndet]    output zdat[xsize,ysize]
; keyword:
;   group_leader:  dialog parent uid 
;   xstart:        binning starting point, if absent, use the min(xdat) as the starting point
;   ystart:        binning starting point, if absent, use the min(ydat) as the starting point
;   checkfinite:   flag for checking finite values of zdat
;   uniq_xval:     if set, then use the given unique values instead of the steps to determine xbin
;   uniq_yval:     if set, then use the given unique values instead of the steps to determine ybin
;   avgsum:        -1:weighted mean 0: arithmetic mean 1:sum, default 0
;   weight:        used as weight for avgsum = -1. if not present, use the error bar   
;   bintr:         binning threshold, bins with fewer data points than bintr*(max number of points in a bin) will be discarded
;   conststep:     0: use the average as the resulted binning values(default) 1: constant binning intervals
;   leave_blank:   if set, leave empty column or rows as they are
;   ewid:          energy width, used when e-integrated intensity option is chose
;                  input  dat=sum(I*ewid)
;                  output dat=sum(I*ewid)/sum(ewid)
;   extlib:        0:not using external library 1:use external library
;   zerr:          input error bar of zdat[ne,ndet], ouput zerr[xsize,ysize]
;   zero_error:    if nonzero, use this value as error bar for zero intensities
;   bin_zeroerror: flag for how to treat zero intenisty error, default 0 (user specify)
;   zoerrestrange: used when bin_zeroerror=1, the zero intensity error bar statistical estimate range: 0-automatic search till non-zero counts are found; n(>=1)-specified search range
;   estapplyall:   used when bin_zeroerror=1, if set, zoerrestrange is applied to all intensity error bar statistical estimate
;   foundrange:    output the range when zoerrestrange=0
pro dm_stepgrid_bin,in_xstep,in_ystep,xdat,ydat,zdat,xstart=xstart,ystart=ystart,checkfinite=checkfinite,uniq_xval=uniq_xval,uniq_yval=uniq_yval,debug=debug,avgsum=avgsum,$
    bintr=bintr,conststep=conststep,ewid=ewid,leave_blank=leave_blank,zerr=zerr,extlib=extlib,weight=weight,zero_error=zero_error,bin_zeroerror=bin_zeroerror,zoerrestrange=zoerrestrange,$
    estapplyall=estapplyall,foundrange=foundrange,group_leader=group_leader

    if n_elements(avgsum) eq 0 then avgsum = 0        ;default using arithmetic mean in binning
    if n_elements(bintr)  eq 0 then bintr = 0.0
    if n_elements(conststep) eq 0 then conststep = 0  ;default is using average as the binned value
    if keyword_set(conststep) then leave_blank = 1
    if n_elements(extlib) eq 0 then extlib = 0        ;default not using external binning library
    if n_elements(zero_error) eq 0 then zero_error = 0.0
    nxdat   = n_elements(xdat)
    ok_eint = (n_elements(ewid) eq nxdat)
    ok_zerr = (n_elements(zerr) eq nxdat)
    ok_wght = (n_elements(weight) eq nxdat)
    x_type  = size(xdat,/type)
    y_type  = size(ydat,/type)
    z_type  = size(zdat,/type)
    if ok_zerr then e_type = size(zerr,/type)
    if ok_wght then w_type = size(weight,/type)
    if ok_eint then avgsum = 1
    if keyword_set(bin_zeroerror) and ok_zerr then begin
       if n_elements(zoerrestrange) eq 0 then zoerrestrange = 0
       if zoerrestrange eq 0 then foundrange = 0
    endif else bin_zeroerror = 0
    if (avgsum eq -1) and (~ok_wght) and (~ok_zerr) then avgsum = 0    ;weighted average requires the weight or corresponding zerr, without it, switching to arithmetic mean
    if keyword_set(debug) then current = systime(/sec)
    if keyword_set(checkfinite) then begin
       ind = where(finite(zdat),count) 
       if (count ne nxdat) and (count ne 0) then begin
          xdat = xdat[ind]
          ydat = ydat[ind]
          zdat = zdat[ind]
          if ok_zerr then zerr = zerr[ind]
          if ok_eint then ewid = ewid[ind]
          if ok_wght then weight = weight[ind]
          ind = 0 & nxdat = count
       endif
    endif
    hstep = float(abs(in_xstep))
    vstep = float(abs(in_ystep))
    xmax  = max(xdat,min=xmin)
    ymax  = max(ydat,min=ymin)
    if n_elements(xstart) eq 1 then begin
       if finite(xstart) then begin
          xmin = xstart
          ind = where(xdat ge xstart,count)
          if count ne nxdat then begin
             xdat = xdat[ind]
             ydat = ydat[ind]
             zdat = zdat[ind]
             if ok_zerr then zerr = zerr[ind]
             if ok_eint then ewid = ewid[ind]
             if ok_wght then weight = weight[ind]
          endif
          ind = 0 & nxdat = count
       endif
    endif
    if n_elements(ystart) eq 1 then begin
       if finite(ystart) then begin
          ymin = ystart
          ind = where(ydat ge ystart,count)
          if count ne nxdat then begin
             xdat = xdat[ind]
             ydat = ydat[ind]
             zdat = zdat[ind]
             if ok_zerr then zerr = zerr[ind]
             if ok_eint then ewid = ewid[ind]
             if ok_wght then weight = weight[ind]
          endif
          ind = 0 & nxdat = count
       endif
    endif
    if (avgsum eq -1) and (~ok_wght) then begin
       weight = zerr
       ind = where(zerr eq 0,count,complement=ind1)
       if count ne 0 then begin
          tmp = min(abs(zerr[ind1]))
          if zero_error ne 0 then tmp = (tmp)<(zero_error)
          weight[ind] = tmp                 ;use zerr as weight, make sure no zerr=0 exists, otherwise replace them with smallest of nonzero value
       endif
       weight = 1./weight^2
       ok_wght = 1b
       ind = 0 & ind1 = 0
    endif    
    if (hstep eq 0) then hstep = ((xmax eq xmin)?(0.1):((xmax-xmin)/100.))
    if (vstep eq 0) then vstep = ((ymax eq ymin)?(0.1):((ymax-ymin)/100.))
    xmax  = xmax+hstep/2.
    xmin  = xmin-hstep/2.
    ymax  = ymax+vstep/2.
    ymin  = ymin-vstep/2.
    xsize = ceil((xmax-xmin)/hstep)
    ysize = ceil((ymax-ymin)/vstep)
    n_unqx = n_elements(uniq_xval)
    n_unqy = n_elements(uniq_yval)
    if n_unqx ne 0  then begin
       uniq_xval = uniq_xval[uniq(uniq_xval,sort(uniq_xval))]
       n_unqx = n_elements(uniq_xval) & xsize = n_unqx
    endif
    if n_unqy ne 0 then begin
       uniq_yval = uniq_yval[uniq(uniq_yval,sort(uniq_yval))]
       n_unqy = n_elements(uniq_yval) & ysize = n_unqy
    endif
    tmp_x = dblarr(xsize) & num_x = lonarr(xsize)
    tmp_y = dblarr(ysize) & num_y = lonarr(ysize)
    tmp_z = dblarr(xsize,ysize) & num_z = lonarr(xsize,ysize)
    if ok_eint then tmp_w = dblarr(xsize,ysize)
    if ok_zerr then tmp_e = dblarr(xsize,ysize)
    if ok_wght then tmp_m = dblarr(xsize,ysize)
    
    defsysv,'!DAVE_AUXILIARY_DIR',exists=exists
    if exists and extlib then begin
       filename = 'rebinnd'+'_'+!version.os+'_'+strtrim(string(!VERSION.MEMORY_BITS),2)
       shlib = (FILE_SEARCH(!DAVE_AUXILIARY_DIR+'rebin_nd'+path_sep()+filename+'*'))[0]
    endif else shlib = ''
    if (strlen(shlib) eq 0) and extlib then begin
       mesg = 'The external binning library is not found. IDL binning code is used instead.'
       if n_elements(group_leader) ne 0 then begin
          if (group_leader eq LookupManagedWidget('dcs_mslice')) or (group_leader eq LookupManagedWidget('mslice')) then begin
             widget_control,group_leader,get_uvalue=obj_mslice
             if obj_valid(obj_mslice) then obj_mslice->setproperty,binextlib=0 $
             else mesg = [mesg,'','Please disable the external library in the menu Option->Binning Method->Use External Binning Library.']
          endif
       endif
       ok = dialog_message(mesg,dialog_parent=group_leader,/center,title='Please note:')
    endif
    
    if ((strlen(shlib) gt 0) and (n_unqx eq 0) and (n_unqy eq 0))then begin
       if (ok_eint and ok_zerr) then begin
          r = CALL_EXTERNAL(shlib, 'rebin2d_zew', long(nxdat),double(xdat),double(ydat),$
                double(zdat),double(zerr),double(ewid),long(xsize),long(ysize),double(xmin),double(hstep),$
                double(ymin),double(vstep), num_x, tmp_x, num_y, tmp_y, num_z, tmp_z, tmp_e, tmp_w, /CDECL) 
       endif       
       if (ok_eint and ~ok_zerr) then begin
          r = CALL_EXTERNAL(shlib, 'rebin2d_zw', long(nxdat),double(xdat),double(ydat),$
                double(zdat),double(ewid),long(xsize),long(ysize),double(xmin),double(hstep),$
                double(ymin),double(vstep), num_x, tmp_x, num_y, tmp_y, num_z, tmp_z, tmp_w, /CDECL) 
       endif       
       if (~ok_eint and ok_zerr) then begin
          r = CALL_EXTERNAL(shlib, 'rebin2d_ze', long(nxdat),double(xdat),double(ydat),$
                double(zdat),double(zerr),long(xsize),long(ysize),double(xmin),double(hstep),$
                double(ymin),double(vstep), num_x, tmp_x, num_y, tmp_y, num_z, tmp_z, tmp_e, /CDECL) 
       endif       
       if (~ok_eint and ~ok_zerr) then begin
          r = CALL_EXTERNAL(shlib, 'rebin2d_z', long(nxdat),double(xdat),double(ydat),$
                double(zdat),long(xsize),long(ysize),double(xmin),double(hstep),$
                double(ymin),double(vstep), num_x, tmp_x, num_y, tmp_y, num_z, tmp_z,  /CDECL) 
       endif 
       if avgsum eq -1 then avgsum = 0  ;weighted mean not implemented yet
    endif else begin  ;use histogram function to do the 2d-binning
       ;first construct a 1-d array for histogram function
       mesg = ''
       if n_unqx ne 0 then begin
          ccnt = 0ll
          tmp_xdat = lonarr(nxdat)
          for i = 0L,n_unqx-1L do begin
              index = where(xdat eq uniq_xval[i],count)
              if count ne 0 then tmp_xdat[index] = i
              ccnt = ccnt + count
              if ccnt eq nxdat then break
          endfor
          if ccnt ne nxdat then mesg = [mesg,'There is a discrepancy between xdat and the unique x binning values.']
          xdat = 0  ;no longer needs it
       endif else begin
          tmp_xdat = long((xdat-xmin)/hstep)
       endelse
       if n_unqy ne 0 then begin
          ccnt = 0ll
          tmp_ydat = lonarr(nxdat)
          for i = 0L,n_unqy-1L do begin
              index = where(ydat eq uniq_yval[i],count)
              if count ne 0 then tmp_ydat[index] = i
              ccnt = ccnt + count
              if ccnt eq nxdat then break
          endfor
          if ccnt ne nxdat then mesg = [mesg,'There is a discrepancy between ydat and the unique y binning values.']
          ydat = 0  ;no longer needs it
       endif else begin
          tmp_ydat = long((ydat-ymin)/vstep)
       endelse
       if n_elements(mesg) gt 1 then ok = dialog_message([mesg[1:*],'This might affect the binning result.'],dialog_parent=group_leader,/center,title='Please note:')
       if (~keyword_set(conststep)) and (n_unqx eq 0) then begin
          num_x = histogram(tmp_xdat,binsize=1l,min=0l,max=xsize-1,reverse_indices=rindx)
          for i=0L,n_elements(num_x)-1L do begin
              if num_x[i] eq 0 then continue
              tmp_x[i] = total(xdat[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
          endfor
          rindx = 0
       endif
       if (~keyword_set(conststep)) and (n_unqy eq 0) then begin
          num_y = histogram(tmp_ydat,binsize=1L,min=0L,max=ysize-1L,reverse_indices=rindx)
          for i=0L,n_elements(num_y)-1L do begin
              if num_y[i] eq 0 then continue
              tmp_y[i] = total(ydat[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
          endfor
          rindx = 0
       endif
       if xsize*ysize lt 0 then begin
          tmp = long64(xsize)*long64(temporary(tmp_ydat))+long64(temporary(tmp_xdat))
          tmpmax = long64(xsize)*long64(ysize)-(1LL) 
       endif else begin
          tmp = long(xsize)*long(temporary(tmp_ydat))+long(temporary(tmp_xdat))
          tmpmax = long(xsize)*long(ysize)-(1L) 
       endelse
       num_z = histogram(temporary(tmp),binsize=1,min=0,max=tmpmax,reverse_indices=rindx)
       num_z = reform(num_z,xsize,ysize,/overwrite)
       if ok_zerr then begin
          zerr = zerr^2
          if (zero_error ne 0) and (avgsum eq -1) then begin
             ind = where(zerr eq 0,cnt)
             if cnt ne 0 then zero_error = zero_error*min(weight[ind])
          endif
       endif
       n_rindx = n_elements(rindx)
       if avgsum eq -1 then begin  ;weighted mean
          if keyword_set(bin_zeroerror) then tmp_zerr = zerr  ;need another copy
          for i=0LL,n_elements(num_z)-1LL do begin
              if num_z[i] eq 0 then continue
              if keyword_set(bin_zeroerror) then begin  ;special care for zero intensities
                 tmp_row = long64(i mod xsize)
                 tmp_col = long64(i/xsize)
                 tmp1    = 0d 
                 tmp2    = 0d
                 if keyword_set(estapplyall) and (zoerrestrange) gt 0 then begin
                    for row=(0LL)>(tmp_row-zoerrestrange),(tmp_row+zoerrestrange)<(xsize-1) do begin
                        for col=(0LL)>(tmp_col-zoerrestrange),(tmp_col+zoerrestrange)<(ysize-1) do begin
                            tmp_in = col*xsize+row
                            if rindx[tmp_in] lt n_rindx then begin
                               if rindx[tmp_in+1] gt rindx[tmp_in] then begin
                                  tmp1 = temporary(tmp1)+total(tmp_zerr[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                                  tmp2 = temporary(tmp2)+total(weight[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                               endif else begin
                                  tmp1 = temporary(tmp1)+tmp_zerr[rindx[rindx[tmp_in]]]
                                  tmp2 = temporary(tmp2)+weight[rindx[rindx[tmp_in]]]
                               endelse
                            endif
                        endfor
                    endfor
                    if tmp2 gt 0 then begin
                       tmp = tmp1/tmp2
                       zerr[rindx[rindx[i]:(rindx[i+1]-1)]] = weight[rindx[rindx[i]:(rindx[i+1]-1)]]*tmp
                    endif
                 endif else begin
                    ind = where(tmp_zerr[rindx[rindx[i]:(rindx[i+1]-1)]] eq 0,cnt)
                    if cnt ne 0 then begin
                       if zoerrestrange eq 0 then begin
                          ;expanding the range gradually until finding non-zero intensities
                          found   = 0b
                          tmp_ran = 0l
                          tmp_max = max([tmp_row,xsize-tmp_row,tmp_col,ysize-tmp_col])
                          while (~found) do begin
                                for row=(0LL)>(tmp_row-tmp_ran),(tmp_row+tmp_ran)<(xsize-1) do begin
                                    for col=(0LL)>(tmp_col-tmp_ran),(tmp_col+tmp_ran)<(ysize-1) do begin
                                        if max(abs([row-tmp_row,col-tmp_col])) eq tmp_ran then begin
                                           tmp_in = col*xsize+row
                                           if rindx[tmp_in] lt n_rindx then begin
                                              if rindx[tmp_in+1] gt rindx[tmp_in] then begin
                                                 tmp1 = temporary(tmp1)+total(tmp_zerr[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                                                 tmp2 = temporary(tmp2)+total(weight[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                                              endif else begin
                                                 tmp1 = temporary(tmp1)+tmp_zerr[rindx[rindx[tmp_in]]]
                                                 tmp2 = temporary(tmp2)+weight[rindx[rindx[tmp_in]]]
                                              endelse
                                           endif
                                        endif
                                    endfor
                                endfor
                                if tmp1 gt 0 then begin
                                   tmp = tmp1/tmp2
                                   if keyword_set(estapplyall) then zerr[rindx[rindx[i]:(rindx[i+1]-1)]] = weight[rindx[rindx[i]:(rindx[i+1]-1)]]*tmp $
                                   else zerr[(rindx[rindx[i]:(rindx[i+1]-1)])[ind]] = weight[(rindx[rindx[i]:(rindx[i+1]-1)])[ind]]*tmp
                                   found = 1b
                                endif
                                tmp_ran = tmp_ran+1
                                if tmp_ran gt tmp_max then found = 1b
                          endwhile
                          foundrange = foundrange>(tmp_ran-1)
                       endif else begin
                          for row=(0LL)>(tmp_row-zoerrestrange),(tmp_row+zoerrestrange)<(xsize-1) do begin
                              for col=(0LL)>(tmp_col-zoerrestrange),(tmp_col+zoerrestrange)<(ysize-1) do begin
                                  tmp_in = col*xsize+row
                                  if rindx[tmp_in] lt n_rindx then begin
                                     if rindx[tmp_in+1] gt rindx[tmp_in] then begin
                                        tmp1 = temporary(tmp1)+total(tmp_zerr[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                                        tmp2 = temporary(tmp2)+total(weight[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                                     endif else begin
                                        tmp1 = temporary(tmp1)+tmp_zerr[rindx[rindx[tmp_in]]]
                                        tmp2 = temporary(tmp2)+weight[rindx[rindx[tmp_in]]]
                                     endelse
                                  endif
                              endfor
                          endfor
                          if tmp2 gt 0 then begin
                             tmp = tmp1/tmp2
                             zerr[(rindx[rindx[i]:(rindx[i+1]-1)])[ind]] = weight[(rindx[rindx[i]:(rindx[i+1]-1)])[ind]]*tmp
                          endif
                       endelse
                    endif
                 endelse
              endif
              tmp_z[i] = total(zdat[rindx[rindx[i]:(rindx[i+1]-1)]]*weight[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
              tmp_m[i] = total(weight[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
              if ok_zerr then tmp_e[i] = total(zerr[rindx[rindx[i]:(rindx[i+1]-1)]]*(weight[rindx[rindx[i]:(rindx[i+1]-1)]])^2,/double)
          endfor
          if keyword_set(bin_zeroerror) then tmp_zerr = 0  ;release the memory
       endif else begin
          for i=0LL,n_elements(num_z)-1LL do begin
              if num_z[i] eq 0 then continue
              tmp_z[i] = total(zdat[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
              if ok_zerr then tmp_e[i] = total(zerr[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
              if ok_eint then tmp_w[i] = total(ewid[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
              if ok_wght then tmp_m[i] = total(weight[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
          endfor
       endelse
       rindx = 0
       if keyword_set(conststep) or (n_unqx ne 0) then num_x = long(total(num_z,2))
       if keyword_set(conststep) or (n_unqy ne 0) then num_y = long(total(num_z,1))
    endelse
    zdat = 0
    if ok_zerr then zerr = 0

    indx = where(num_x ne 0,countx,complement=indx1,ncomplement=countx1)
    indy = where(num_y ne 0,county,complement=indy1,ncomplement=county1)
    if keyword_set(leave_blank) then begin
       if countx1 ne 0 then begin
          num_x[indx1] = 1.0 
          if n_unqx eq 0 then tmp_x[indx1]=xmin+indx1*hstep else tmp_x[indx1]=uniq_xval[indx1]
       endif
       if county1 ne 0 then begin
          num_y[indy1] = 1.0 
          if n_unqy eq 0 then tmp_y[indy1]=ymin+indy1*vstep else tmp_y[indy1]=uniq_yval[indy1] 
       endif
    endif else begin
       if (countx ne 0) and (countx1 ne 0) then begin
          tmp_x = tmp_x[indx]   & num_x = num_x[indx]
          tmp_z = tmp_z[indx,*] & num_z = num_z[indx,*]
          if ok_eint then tmp_w = tmp_w[indx,*]
          if ok_zerr then tmp_e = tmp_e[indx,*]
          if ok_wght then tmp_m = tmp_m[indx,*]
          if n_unqx ne 0 then begin
             uniq_xval = uniq_xval[indx]
             n_unqx = countx & xsize = countx
          endif
       endif
       if (county ne 0) and (county1 ne 0) then begin
          tmp_y = tmp_y[indy]   & num_y = num_y[indy]
          tmp_z = tmp_z[*,indy] & num_z = num_z[*,indy]
          if ok_eint then tmp_w = tmp_w[*,indy]
          if ok_zerr then tmp_e = tmp_e[*,indy]
          if ok_wght then tmp_m = tmp_m[*,indy]
          if n_unqy ne 0 then begin
             uniq_yval = uniq_yval[indy]
             n_unqy = county & ysize = county
          endif
       endif
    endelse
    indx = -1 & indx1 = -1 & indy = -1 & indy1 = -1
    if keyword_set(conststep) or (n_unqx ne 0) then begin
       if n_unqx ne 0 then xdat = uniq_xval $
       else xdat = xmin+(findgen(xsize)+0.5)*hstep
       tmp_z = tmp_z[0:(xsize-1),*]
       num_z = num_z[0:(xsize-1),*]
       if ok_zerr then tmp_e = tmp_e[0:(xsize-1),*]
       if ok_eint then tmp_w = tmp_w[0:(xsize-1),*]
       if ok_wght then tmp_m = tmp_m[0:(xsize-1),*]
    endif else begin
       xdat = temporary(tmp_x)/temporary(num_x)
    endelse
    if keyword_set(conststep) or (n_unqy ne 0) then begin
       if n_unqy ne 0 then ydat = uniq_yval $
       else ydat = ymin+(findgen(ysize)+0.5)*vstep
       tmp_z = tmp_z[*,0:(ysize-1)]
       num_z = num_z[*,0:(ysize-1)]
       if ok_zerr then tmp_e = tmp_e[*,0:(ysize-1)]
       if ok_eint then tmp_w = tmp_w[*,0:(ysize-1)]
       if ok_wght then tmp_m = tmp_m[*,0:(ysize-1)]
    endif else begin
       ydat = temporary(tmp_y)/temporary(num_y)
    endelse
    zdat = temporary(tmp_z)
    if ok_wght then weight = temporary(tmp_m)
    if ok_zerr then begin
       if avgsum ne -1 then begin
          zerr = sqrt(temporary(tmp_e))
          if zero_error ne 0 then begin  
             ind = where(zerr eq 0,count)
             if count ne 0 then zerr[ind] = zero_error
             ind = 0
          endif
       endif else $
          zerr = fltarr(size(zdat,/dimension))
    endif
    index1 = where(num_z ne 0,count1,complement=index0,ncomplement=count0)
    if (count1 ne 0) then begin
       if ok_eint then begin
          zdat[index1] = zdat[index1]/tmp_w[index1]
          if ok_zerr then zerr[index1] = zerr[index1]/tmp_w[index1]
       endif else if avgsum eq 0 then begin   ;arithmetic mean
          zdat[index1] = zdat[index1]/num_z[index1]    
          if ok_zerr then zerr[index1] = zerr[index1]/num_z[index1]
       endif else if avgsum eq -1 then begin  ;weighted mean
          zdat[index1] = zdat[index1]/weight[index1]
          if ok_zerr then begin
             zerr[index1] = sqrt(tmp_e[index1])/weight[index1]
             if zero_error ne 0 then begin
                ind = where(zerr[index1] eq 0,count)
                if count ne 0 then zerr[index1[ind]] = zero_error/weight[index1[ind]]
             endif
             tmp_e = 0  
          endif
       endif
       index1 = 0
       if (bintr gt 0.0) and (bintr lt 1.0) then begin
          min_num_z = bintr*(max(num_z)>1)
          ind = where(num_z lt min_num_z,ct1)
          if ct1 gt 0 then begin
             zdat[ind] = !values.f_nan
             if ok_zerr then zerr[ind] = !values.f_nan
          endif          
       endif
    endif
    if (count0 ne 0) then begin
       zdat[index0] = !values.f_nan
       if ok_zerr then zerr[index0] = !values.f_nan
    endif
    if size(xdat,/type) ne x_type then xdat = fix(xdat,type=x_type)
    if size(ydat,/type) ne y_type then ydat = fix(ydat,type=y_type)
    if size(zdat,/type) ne z_type then zdat = fix(zdat,type=z_type)
    if ok_zerr then begin
       if size(zerr,/type) ne e_type then zerr = fix(zerr,type=e_type)
    endif
    if ok_wght then begin
       if size(weight,/type) ne w_type then weight = fix(weight,type=w_type)
    endif
    if keyword_set(debug) then print,'stepgrid_bin finished in ',systime(/sec)-current,' secs.'
end