; $Id: dm_patch.pro,v 1.9 2017/05/10 14:53:38 ymqiu Exp $
;#######################################################################
;
; NAME:
;  dm_patch
;
; PURPOSE:
;  This program patches the data files for various instrument:
;     DCS:  wavelength info, adapted from John Copley's dcs_patchrawdatafile.pro
;     MACS: flip state and cfx filter, which sometimes are incorrectly recorded
;     
; CATEGORY:
;  file tools
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  June, 2025
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

;clear all objects and widget hierarchy
pro dm_patch_Exit,tlb
    widget_control,tlb,get_uvalue=state,/no_copy
    obj_destroy,[state.open_filesel,state.save_filesel]
    widget_control,tlb,/destroy
end

pro dm_patch_readdcs,state,comment=comment,a2=a2,ss=ss
    state.open_filesel->getproperty,path=open_path,file=open_file,ftpobj=ftpobj,dFiles=dfiles,ftpbufferdir=ftpbufferdir
    state.save_filesel->getproperty,path=save_path
    while(strmid(open_path,0,1,/reverse_offset) eq  ([state.pathsep,'/'])[obj_valid(ftpobj)]) do  open_path = strmid(open_path,0,strlen(open_path)-1)
    while(strmid(save_path,0,1,/reverse_offset) eq  state.pathsep) do  save_path = strmid(save_path,0,strlen(save_path)-1)
    if n_elements(open_file) gt 0 then begin
       davePtr = dm_define_pointer() & omithisto=2 & printinfo=0 & filetype='binary'
       file = open_path+([state.pathsep,'/'])[obj_valid(ftpobj)]+open_file[0]
       if obj_valid(ftpobj) then begin
          ok = ftpobj->GetFileContent(file,localfilename=ftpbufferdir+state.pathsep+'ftptmp'+open_file[0])
          file = ftpbufferdir+state.pathsep+'ftptmp'+open_file[0]
       endif
       dcs_readdatafile,file,omithisto,printinfo,filetype,davePtr=davePtr
       if obj_valid(ftpobj) then file_delete,file,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET
       this = dm_locate_datastrptr(davePtr)
       if arg_present(comment) then comment = strtrim((*(*this).specificPtr).comments,2)
       if arg_present(a2) then a2 = dm_to_string(((*(*this).specificPtr).motor_pos)[1],resolution=3)
       if arg_present(ss) then ss = (*(*this).specificPtr).shutter_stat
       heap_free,davePtr
    endif
end

;set dcs option menu and button sensitivity
pro dm_patch_setdcsoptn,state,title=title,settitle=settitle,setstrtlett=setstrtlett
    for i=0,n_elements(state.optnBut)-1 do dm_set_button,state.optnBut[i],state.dcsoptn[i,0]
    widget_control,state.optnBut[0],sensitive=(total(state.dcsoptn[3:*,0]) eq 0)
    sensitive = (total(state.dcsoptn[1:*,0]) eq 0)
    tmp = [state.mspd,state.wvln,state.resl[1],state.srmode[1],state.denom]
    for i=0,n_elements(tmp)-1 do widget_control,tmp[i],sensitive=sensitive
    tmp = 'lb_'+['mspd','wvln','resl','srmode','denom']
    for i=0,n_elements(tmp)-1 do begin
        wid = widget_info(state.cmdCol,find_by_uname=tmp[i])
        if wid ne 0 then widget_control,wid,sensitive=sensitive
    endfor
    sensitive = (total(state.dcsoptn[2:*,0]) eq 0)
    tmp = state.tsdmin
    for i=0,n_elements(tmp)-1 do widget_control,tmp[i],sensitive=sensitive
    wid = widget_info(state.cmdCol,find_by_uname='lb_tsdmin')
    if wid ne 0 then widget_control,wid,sensitive=sensitive
    ind = where(state.dcsoptn[[0,3,4,5],0],count)
    if count eq 0 then label = '  ' else label = (['Comment',state.optnname[3:*]])[ind]
    widget_control,state.clabel,set_value=label[0]
    ind = where(state.dcsoptn[2:*,0],count)
    if count eq 0 then label = ' file wavelength info' else $
    label = ([' jumpy file',' file single time channel',' file A2 angle',' file '+state.optnname[5],' '+state.optnname[6]+' affected file'])[ind]
    title = 'Patch '+state.ftype+label
    if keyword_set(settitle) then widget_control,state.tlb,tlb_set_title=title
    if keyword_set(setstrtlett) then begin
       tmp = state.p_strtlett
       state.p_strtlett = state.s_strtlett
       state.s_strtlett = tmp
       id  = widget_info(state.strtlett,find_by_uname=state.p_strtlett)
       ind = where(state.lett_ids eq id,cnt)
       if cnt eq 1 then dm_toggle_menubut,uncheck=state.lett_ids[fix(byte(tmp))-fix(byte('A'))],check=id
       widget_control,state.patchBut,set_value='Patch --> ('+state.p_strtlett+')'
    endif
    if (float(!version.release) ge 5.6) and widget_info(state.patchBut,/valid_id) then widget_control,state.patchBut,tooltip=state.tooltip[state.dcsoptn[2,0]+2*state.dcsoptn[6,0]]+state.p_strtlett+'.'
end

;change file type
pro dm_patch_changeftype,state,ftype=ftype,center=center
    if n_elements(ftype) ne 0 then state.ftype = ftype 
    widget_control,state.tlb,update=0
    widget_control,state.cmdCol,map=0  
    kids = widget_info(state.optnmenu,/all_children)
    if kids[0] ne 0 then begin
       for i=n_elements(kids)-1,0,-1 do widget_control,kids[i],/destroy
    endif
    kids = widget_info(state.cmdCol,/all_children)
    if kids[0] ne 0 then begin
       geom1 = widget_info(state.cmdCol,/geometry)
       for i=n_elements(kids)-1,0,-1 do widget_control,kids[i],/destroy
    endif
    widget_control,state.openlabel,set_value=state.ftype+' data file directory'
    case state.ftype of
         'DCS': begin
                dm_toggle_menubut,check=state.dcsBut,uncheck=state.macsBut
                fext            = '.DCS'
                xsize           = 10
                state.optnname  = ['Comment','Tsdmin and Rotate Time Channel','Jumpy File','E (meV)','A2','Shutter State','201905 Histogram Hardware Upgrade']
                menuname        = ['Allow Patching Comments','Patch '+[state.optnname[1:2],'a Single Time Channel',state.optnname[4:*]]+' Only']
                for i=0,n_elements(state.optnname)-1 do state.optnBut[i] = dm_widget_button(state.optnmenu,value=menuname[i])   
                label           = widget_label(state.cmdCol,value='Master Speed',uname='lb_mspd')
                line            = widget_base(state.cmdCol,/row,xpad=0,space=0,ypad=0,/align_center)
                state.mspd[0]   = widget_text(line,value='',xsize=xsize)
                state.mspd[1]   = widget_text(line,value=((state.p_mspd gt 0)?dm_to_string(state.p_mspd):' '),xsize=xsize,/editable,/all_events)
                geom            = widget_info(state.mspd[1],/geometry)
                label           = widget_label(state.cmdCol,value='Wavelength',uname='lb_wvln')
                line            = widget_base(state.cmdCol,/row,xpad=0,space=0,ypad=0,/align_center)
                state.wvln[0]   = widget_text(line,value='',xsize=xsize)
                state.wvln[1]   = widget_text(line,value=dm_to_string(state.p_wvln),xsize=xsize,/editable,/all_events)
                label           = widget_label(state.cmdCol,value='Tsdmin',uname='lb_tsdmin')
                line            = widget_base(state.cmdCol,/row,xpad=0,space=0,ypad=0,/align_center)
                state.tsdmin[0] = widget_text(line,value='',xsize=xsize)
                state.tsdmin[1] = widget_text(line,value=((state.p_tsdmin gt 0)?dm_to_string(state.p_tsdmin):' '),xsize=xsize,/editable,/all_events)
                label           = widget_label(state.cmdCol,value='Resolution',uname='lb_resl')
                line            = widget_base(state.cmdCol,/row,xpad=0,space=0,ypad=0,/align_center)
                state.resl[0]   = dm_widget_droplist(line,value=['low','medium'],xsize=geom.scr_xsize,sensitive=0)
                state.resl[1]   = dm_widget_droplist(line,value=['low','medium'],xsize=geom.scr_xsize,select=state.p_resl-1)
                label           = widget_label(state.cmdCol,value='Speed Ratio Mode',uname='lb_srmode')
                line            = widget_base(state.cmdCol,/row,xpad=0,space=0,ypad=0,/align_center)
                state.srmode[0] = dm_widget_droplist(line,value=['1/m','(m-1)/m'],xsize=geom.scr_xsize,sensitive=0)
                state.srmode[1] = dm_widget_droplist(line,value=['1/m','(m-1)/m'],xsize=geom.scr_xsize,select=state.p_srmode-1)
                label           = widget_label(state.cmdCol,value='Speed Ratio Denominator',uname='lb_denom')
                line            = widget_base(state.cmdCol,/row,xpad=0,space=0,ypad=0,/align_center)
                state.denom[0]  = widget_text(line,value='',xsize=xsize)
                state.denom[1]  = widget_text(line,value=((state.p_denom gt 0)?dm_to_string(state.p_denom):' '),xsize=xsize,/editable,/all_events)
                state.clabel    = widget_label(state.cmdCol,value='  ',/dynamic_resize)
                state.aline     = widget_base(state.cmdCol,/row,xpad=0,space=0,ypad=0,/align_center)
                if state.dcsoptn[0,0] then state.comment  = widget_text(state.aline,value=state.p_comment,/editable,/all_events) $
                else if state.dcsoptn[3,0] then state.comment = widget_text(state.aline,value=dm_to_string(state.p_energy),/editable,/all_events) $
                else if state.dcsoptn[4,0] then begin
                   state.a2[0]  = widget_text(state.aline,value='',xsize=xsize)
                   state.a2[1]  = widget_text(state.aline,value=dm_to_string(state.p_a2),xsize=xsize,/editable,/all_events)
                endif else if state.dcsoptn[5,0] then begin
                   state.a2[0]  = dm_widget_droplist(state.aline,value=['unknown','close','open'],xsize=geom.scr_xsize,sensitive=0)
                   state.a2[1]  = dm_widget_droplist(state.aline,value=['unknown','close','open'],xsize=geom.scr_xsize,select=state.p_ss+1) 
                endif
                dm_patch_setdcsoptn,state,title=title
                end
         'MACS':begin
                dm_toggle_menubut,check=state.macsBut,uncheck=state.dcsBut
                fext            = '.NG0,.BT9'
                title           = 'Patch '+state.ftype+' files'
                xsize           = 80
                label           = widget_label(state.cmdCol,value='Flip State')
                line            = widget_base(state.cmdCol,/row,xpad=0,space=0,ypad=0,/align_center)
                state.flip[0]   = dm_widget_droplist(line,value=['N/A','A','B','A and B','other'],xsize=xsize,sensitive=0)
                state.flip[1]   = dm_widget_droplist(line,value=['No change','A->B, B->A','All to A','All to B'],select=state.p_flip+1,xsize=xsize)
                label           = widget_label(state.cmdCol,value='CFX Filter')
                line            = widget_base(state.cmdCol,/row,xpad=0,space=0,ypad=0,/align_center)
                state.cfx[0]    = dm_widget_droplist(line,value=['N/A','None','Be','HOPG','Be+HOPG','MgF2','Be+MgF2','HOPG+MgF2','Be+HOPG+MgF2'],xsize=xsize,sensitive=0)
                state.cfx[1]    = dm_widget_droplist(line,value=['No change','None','Be','HOPG','MgF2'],select=state.p_cfx+1,xsize=xsize)
                label           = widget_label(state.cmdCol,value='Temperature')
                line            = widget_base(state.cmdCol,/row,xpad=0,space=0,ypad=0,/align_center)
                state.temp[0]   = widget_text(line,value='',scr_xsize=xsize)
                state.temp[1]   = widget_text(line,value=dm_to_string(state.p_temp),scr_xsize=xsize,/editable,/all_events)
                label           = widget_label(state.cmdCol,value='A3')
                line            = widget_base(state.cmdCol,/row,xpad=0,space=0,ypad=0,/align_center)
                state.a3[0]     = widget_text(line,value='',scr_xsize=xsize)
                state.a3[1]     = widget_text(line,value=dm_to_string(state.p_a3),scr_xsize=xsize,/editable,/all_events)
                for i=0,1 do label = widget_label(state.cmdCol,value=' ')
                end
         else:  begin
                fext            = '.*'
                title           = ''
                end
    endcase
    state.ftpBut   = dm_widget_button(state.optnmenu,value='Allow NCNR Data Repository',set_button=state.ncnrftp,separator=(state.ftype eq 'DCS'))
    state.strtlett = widget_button(state.optnmenu,value='Patched File Name Starts With',/separator,/menu)
    
    for i=0,25 do begin
        tmp = string(byte('A')+byte(i))
        state.lett_ids[i] = dm_widget_button(state.strtlett,value=tmp,set_button=(tmp eq state.p_strtlett),uname=tmp)
    endfor
    state.patchBut = widget_button(state.cmdCol,value='Patch --> ('+state.p_strtlett+')')
    geom = widget_info(state.patchBut,/geometry)
    if widget_info(state.comment,/valid_id) then widget_control,state.comment,scr_xsize=geom.scr_xsize
    if state.ftype eq 'MACS' then for i=0,4 do label = widget_label(state.cmdCol,value=' ')
    if float(!version.release) ge 5.6 then widget_control,state.patchBut,tooltip=state.tooltip[0]+state.p_strtlett+'.'
    if keyword_set(center) then dm_center_kid, state.tlb,state.group_leader,/side   ;centering the top level base
    widget_control,state.tlb,tlb_set_title=title
    state.open_filesel->set_filter,fext,path=state.dataDir
    state.save_filesel->set_filter,fext
    widget_control,state.cmdCol,/map 
    widget_control,state.tlb,/update,/realize,/map
    if n_elements(geom1) ne 0 then begin
       geom2 = widget_info(state.cmdCol,/geometry)
       state.geom[0] = state.geom[0]+geom2.scr_xsize-geom1.scr_xsize
       widget_control,state.tlb,scr_xsize=state.geom[0]
    endif
end

;event handler
pro dm_patch_event,event
    compile_opt IDL2    ;,strictarrsubs
                        ;idl2=defint32,strictarr
                        ;strictarr:   [] to index array
                        ;strictarrsubs: error when out-of-range indices,IDL5.6 or after
    WIDGET_CONTROL,/HOURGLASS
    widget_control,event.handler,get_uvalue=state,/no_copy
    state.open_filesel->getproperty,path=open_path,file=open_file,ftpobj=ftpobj,dFiles=dfiles,ftpbufferdir=ftpbufferdir
    state.save_filesel->getproperty,path=save_path

    while(strmid(open_path,0,1,/reverse_offset) eq  ([state.pathsep,'/'])[obj_valid(ftpobj)]) do  open_path = strmid(open_path,0,strlen(open_path)-1)
    while(strmid(save_path,0,1,/reverse_offset) eq  state.pathsep) do  save_path = strmid(save_path,0,strlen(save_path)-1)
    
    ;catch and ignore all errors in this program
    catch, myerror
    if myerror ne 0 then begin
       ok = dialog_message(dialog_parent=state.tlb,!error_state.msg,/error,/center)
       catch,/cancel
       if widget_info(state.cmdCol,/map) eq 0 then widget_control,state.cmdCol,/map
       if widget_info(state.tlb,/update) eq 0 then widget_control,state.tlb,/update
       widget_control,event.handler,set_uvalue=state,/no_copy
       if obj_valid(mesg) then obj_destroy,mesg
       return
    end
    
    tmp = where(state.optnBut eq event.id,event_cnt)
    if event_cnt ne 0 then begin
       widget_control,state.tlb,update=0
       widget_control,state.cmdCol,map=0  
    endif
    if strlowcase(tag_names(event,/structure)) eq 'dm_filesel_select' then event.id = -999
    case (event.id) of
         state.tlb:     widget_control,event.id,scr_xsize=state.geom[0],scr_ysize=state.geom[1]
         state.doneBut: begin
                        widget_control,event.handler,set_uvalue=state,/no_copy
                        widget_control,event.handler,/destroy
                        return
                        end
         state.dcsBut:  if state.ftype ne 'DCS'  then dm_patch_changeftype,state,ftype='DCS'
         state.macsBut: if state.ftype ne 'MACS' then dm_patch_changeftype,state,ftype='MACS'
         state.helpBut: begin
                        if state.ftype eq 'DCS' then begin
                           info = ['To change the wavelength info in a DCS file, select the file in the DCS data file directory first. The original wavelength info will appear in '+$
                                   'the center left column. Enter new values in the center right column. Click Patch button to save the patched file in the patched file directory.','',$
                                   'The comment in a DCS file can also be changed by setting the "Allow Patching Comments" option.','',$
                                   'If only the tsdmin value needs to be changed, set the "Patch Tsdmin and Rotate Time Channel Only" option.','',$
                                   'To patch a jumpy file from .sht files, set the "Patch Jumpy File Only" option. Select the jumpy .dcs.gz file, and click Patch button. '+$
                                   'The .sht files need to be present in the data file directory.','',$
                                   'To patch a single time channel data, set the "Patch a Single Time Channel Only" option. Enter the corresponding energy transfer value for the affected time '+$
                                   'channel. The average of two neighboring time channel values will be used.','',$
                                   'To patch the A2 (sample rotation) angle, set the "Patch A2 Only" option. If multiple files are selected, the same offset value calculated '+$
                                   'from the A2 value in the first file and the entered new A2 value will be applied to the other files.','',$
                                   'To patch the shutter state, set the "Patch Shutter State Only" option. Select the new shutter state in the center right column.','',$
                                   'Select the "Patch 201905 Histogram Hardware Upgrade Only" option to patch the files affected by the hardware upgrade in 201905 cycle. '+$
                                   'The files affected are between 20190523_001 and 20190603_007.']
                        endif else begin
                           info = ['To change the flip state and cfx filter info in a MACS file, select the file in the MACS data file directory first. The original data info will appear '+$
                                   'in the center left column. Set the new values in the center right column. Click Patch button to save the patched file in the patched file directory. ','',$
                                   'The temperature shown in the center left column is the average sample temperature. Leave the temperature field blank if the temperature is not to be changed.','',$
                                   'The A3 shown in the center left column is the first A3 angle in the data file. If multiple files are selected, the same offset value calculated from the first A3 '+$
                                   'value in the first file and the entered new A3 value will be applied to the other files. Leave the A3 field blank if A3 is not to be changed.']
                        endelse
                        info = [info,'','The name of the patched file will start with letter '+state.p_strtlett+' which can be changed in the options menu.']
                        ok = dialog_message(info,/center,/info,dialog_parent=event.handler,title='About Patching '+state.ftype+' Files')
                        end
         state.ftpBut:  begin
                        state.open_fileSel->set_ftpbuffer,(['',dm_define_pointer(/gettempdir)])[1-state.ncnrftp]
                        state.open_fileSel->getproperty,ncnrftp=ncnrftp
                        if state.ncnrftp eq ncnrftp then ok = dialog_message('Cannot establish connection to NCNR Data Repository.',/error,/center,dialog_parent=event.handler) $
                        else begin
                           state.ncnrftp = ncnrftp
                           dm_set_button,event.id,state.ncnrftp
                        endelse
                        end
         ;dcs only events
         state.optnBut[0]:begin   ;allow patching comments
                        state.dcsoptn[0,0] = (~state.dcsoptn[0,0])
                        geom = widget_info(state.patchBut,/geometry)
                        if widget_info(state.comment,/valid_id) then widget_control,state.comment,/destroy
                        if state.dcsoptn[0,0] then begin
                           state.comment = widget_text(state.aline,value='',/editable,/all_events,scr_xsize=geom.scr_xsize)
                           dm_patch_readdcs,state,comment=comment
                           if n_elements(comment) ne 0 then state.p_comment = comment else state.p_comment = ''
                           widget_control,state.comment,set_value=state.p_comment
                        endif
                        dm_patch_setdcsoptn,state
                        end         
         state.optnBut[1]:begin   ;tsdmin only
                        state.dcsoptn[1,0] = (~state.dcsoptn[1,0])
                        state.dcsoptn[2:*,0] = 0b
                        if widget_info(state.a2[0],/valid_id) then for i=0,1 do widget_control,state.a2[i],/destroy
                        if widget_info(state.comment,/valid_id) and (~state.dcsoptn[0,0]) then widget_control,state.comment,/destroy
                        dm_patch_setdcsoptn,state,/settitle,/setstrtlett
                        end  
         state.optnBut[2]:begin   ;jumpy only
                        setstrtlett = state.dcsoptn[1,0]
                        tmp = (~state.dcsoptn[2,0])
                        state.dcsoptn[1:*,0] = 0b
                        state.dcsoptn[2,0] = tmp
                        if widget_info(state.a2[0],/valid_id) then for i=0,1 do widget_control,state.a2[i],/destroy
                        if widget_info(state.comment,/valid_id) and (~state.dcsoptn[0,0]) then widget_control,state.comment,/destroy
                        state.open_filesel->set_filter,(['.dcs','.dcs,.sht'])[state.dcsoptn[2,0]],/sortbyname
                        if obj_valid(ftpobj) then state.open_filesel->set_path,open_path,/ftp
                        dm_patch_setdcsoptn,state,title=title,/settitle,setstrtlett=setstrtlett
                        if state.dcsoptn[2,0] and ~state.dcsoptn[2,1] then begin
                           ok = dialog_message('The .sht files need to be present in the data file directory.',/center,/info,dialog_parent=event.handler)
                           state.dcsoptn[2,1] = 1
                        endif
                        end
         state.optnBut[3]:begin   ;single time channel only
                        setstrtlett = state.dcsoptn[1,0]
                        tmp = (~state.dcsoptn[3,0])
                        state.dcsoptn[*,0] = 0b
                        state.dcsoptn[3,0] = tmp
                        if widget_info(state.comment,/valid_id) then widget_control,state.comment,/destroy
                        if widget_info(state.a2[0],/valid_id) then for i=0,1 do widget_control,state.a2[i],/destroy
                        if state.dcsoptn[3,0] then begin
                           geom = widget_info(state.patchBut,/geometry)
                           state.comment = widget_text(state.aline,value=dm_to_string(state.p_energy),/editable,/all_events,scr_xsize=geom.scr_xsize)
                        endif else state.p_energy = !values.f_nan
                        dm_patch_setdcsoptn,state,title=title,/settitle,setstrtlett=setstrtlett
                        if state.dcsoptn[3,0] and ~state.dcsoptn[3,1] then begin
                           ok = dialog_message('Enter the corresponding energy transfer value for the affected time '+$
                                'channel. The average of two neighboring time channel values will be used.',/center,/info,dialog_parent=event.handler)
                           state.dcsoptn[3,1] = 1     
                        endif
                        end                        
         state.optnBut[4]:begin   ;a2 only
                        setstrtlett = state.dcsoptn[1,0]
                        tmp = (~state.dcsoptn[4,0])
                        state.dcsoptn[*,0] = 0b
                        state.dcsoptn[4,0] = tmp
                        if widget_info(state.comment,/valid_id) then widget_control,state.comment,/destroy
                        if widget_info(state.a2[0],/valid_id) then for i=0,1 do widget_control,state.a2[i],/destroy
                        if state.dcsoptn[4,0] then begin
                           dm_patch_readdcs,state,a2=a2
                           if n_elements(a2) eq 0 then a2 = ''
                           tmp = widget_info(state.wvln[0],/geometry)
                           state.a2[0] = widget_text(state.aline,value=a2,xsize=tmp.xsize)
                           state.a2[1] = widget_text(state.aline,value=dm_to_string(state.p_a2),xsize=tmp.xsize,/editable,/all_events)
                        endif else begin
                           state.a2offset = !values.f_nan
                           state.p_a2 = !values.f_nan
                        endelse
                        dm_patch_setdcsoptn,state,title=title,/settitle,setstrtlett=setstrtlett
                        if state.dcsoptn[4,0] and ~state.dcsoptn[4,1] then begin
                           ok = dialog_message('For patching multiple files, the same offset value calculated '+$
                                'from the A2 value in the first file and the entered new A2 value will be applied to the other files.',/center,/info,dialog_parent=event.handler)
                           state.dcsoptn[4,1] = 1
                        endif
                        end
         state.optnBut[5]:begin   ;shutter state only
                        setstrtlett = state.dcsoptn[1,0]
                        tmp = (~state.dcsoptn[5,0])
                        state.dcsoptn[*,0] = 0b
                        state.dcsoptn[5,0] = tmp
                        if widget_info(state.a2[0],/valid_id) then for i=0,1 do widget_control,state.a2[i],/destroy
                        if widget_info(state.comment,/valid_id) then widget_control,state.comment,/destroy
                        if state.dcsoptn[5,0] then begin
                           dm_patch_readdcs,state,ss=ss
                           if n_elements(ss) eq 0 then ss = -1
                           tmp = widget_info(state.wvln[0],/geometry)
                           state.a2[0]  = dm_widget_droplist(state.aline,value=['unknown','close','open'],xsize=tmp.scr_xsize,select=ss+1,sensitive=0)
                           state.a2[1]  = dm_widget_droplist(state.aline,value=['unknown','close','open'],xsize=tmp.scr_xsize,select=state.p_ss+1)
                        endif else begin
                           state.p_ss = -1
                        endelse
                        dm_patch_setdcsoptn,state,title=title,/settitle,setstrtlett=setstrtlett
                        end
         state.optnBut[6]:begin   ;5/2019 histogram switch only
                        setstrtlett = state.dcsoptn[1,0]
                        tmp = (~state.dcsoptn[6,0])
                        state.dcsoptn[0:*,0] = 0b
                        state.dcsoptn[6,0] = tmp
                        if widget_info(state.a2[0],/valid_id) then for i=0,1 do widget_control,state.a2[i],/destroy
                        if widget_info(state.comment,/valid_id) and (~state.dcsoptn[0,0]) then widget_control,state.comment,/destroy
                        dm_patch_setdcsoptn,state,title=title,/settitle,setstrtlett=setstrtlett
                        if state.dcsoptn[6,0] and ~state.dcsoptn[6,1] then begin
                           ok = dialog_message('Patching files affected by the 201905 cycle histogram hardware upgrade, between 20190523_001 and 20190603_007.',/center,/info,dialog_parent=event.handler)
                           state.dcsoptn[6,1] = 1
                        endif
                        end
         state.mspd[1]: begin
                        widget_control,event.id,get_value=tmp
                        state.p_mspd = dm_to_number(tmp,/int)
                        if state.p_mspd lt 0 then begin
                           ok = dialog_message('The master speed must be a positive integer.',/error,/center,dialog_parent=event.handler)
                           state.p_mspd = 20000
                           widget_control,event.id,set_value='20000'
                        endif
                        end
         state.wvln[1]: begin
                        widget_control,event.id,get_value=tmp
                        state.p_wvln = dm_to_number(tmp,/float)
                        if state.p_wvln lt 0 then begin
                           ok = dialog_message('The wavelength must be a positive number.',/error,/center,dialog_parent=event.handler)
                           state.p_wvln = !values.f_nan
                           widget_control,event.id,set_value=''
                        endif
                        end
         state.tsdmin[1]:begin
                        widget_control,event.id,get_value=tmp
                        state.p_tsdmin = dm_to_number(tmp,/int)
                        if state.p_tsdmin lt 0 then begin
                           ok = dialog_message('The tsdmin must be a positive integer.',/error,/center,dialog_parent=event.handler)
                           state.p_tsdmin = 0
                           widget_control,event.id,set_value=''
                        endif
                        end
         state.resl[1]: begin
                        widget_type = widget_info(event.id,/type)
                        if widget_type eq 8 then tmp_value = widget_info(event.id,/droplist_select) $
                        else if widget_type eq 12 then begin
                           tmp_value = event.index
                           if (['low','medium'])[tmp_value] ne event.str then begin
                              tmp_value = state.p_resl-1
                              dm_set_droplist,event.id,select=tmp_value
                           endif
                        endif else tmp_value=0
                        state.p_resl = tmp_value+1
                        end
         state.srmode[1]:begin
                        widget_type = widget_info(event.id,/type)
                        if widget_type eq 8 then tmp_value = widget_info(event.id,/droplist_select) $
                        else if widget_type eq 12 then begin
                           tmp_value = event.index
                           if (['1/m','(m-1)/m'])[tmp_value] ne event.str then begin
                              tmp_value = state.p_srmode-1
                              dm_set_droplist,event.id,select=tmp_value
                           endif
                        endif else tmp_value = 0
                        state.p_srmode = tmp_value+1
                        end
         state.denom[1]:begin
                        widget_control,event.id,get_value=tmp
                        state.p_denom = dm_to_number(tmp,/int)
                        if state.p_denom lt 0 then begin
                           ok = dialog_message('The speed ratio denominator must be a positive integer.',/error,/center,dialog_parent=event.handler)
                           state.p_denom = 0
                           widget_control,event.id,set_value=''
                        endif
                        end
         state.a2[1]:   begin                        
                        if state.dcsoptn[4,0] then begin
                           widget_control,event.id,get_value=tmp
                           tmp = dm_to_number(tmp,/float)
                           state.p_a2 = tmp
                           widget_control,state.a2[0],get_value=tmp
                           tmp = dm_to_number(tmp,/double)
                           if finite(tmp) and finite(state.p_a2) then state.a2offset=state.p_a2-tmp
                        endif else if state.dcsoptn[5,0] then begin
                           widget_type = widget_info(event.id,/type)
                           if widget_type eq 8 then tmp_value = widget_info(event.id,/droplist_select) $
                           else if widget_type eq 12 then begin
                              tmp_value = event.index
                              if (['unknown','close','open'])[tmp_value] ne event.str then begin
                                 tmp_value = state.p_ss+1
                                 dm_set_droplist,event.id,select=tmp_value
                              endif
                           endif else tmp_value = 0
                           state.p_ss = tmp_value-1
                        endif
                        end
         state.comment: begin
                        widget_control,event.id,get_value=tmp
                        if state.dcsoptn[0,0] then state.p_comment = strtrim(tmp,2) $
                        else if state.dcsoptn[3,0] then state.p_energy = dm_to_number(tmp,/float)
                        end
         ;macs only events
         state.flip[1]: begin
                        widget_type = widget_info(event.id,/type)
                        if widget_type eq 8 then tmp_value = widget_info(event.id,/droplist_select) $
                        else if widget_type eq 12 then begin
                           tmp_value = event.index
                           if (['No change','A->B, B->A','All to A','All to B'])[tmp_value] ne event.str then begin
                              tmp_value = state.p_flip+1
                              dm_set_droplist,event.id,select=tmp_value
                           endif
                        endif else tmp_value=0
                        state.p_flip = tmp_value-1
                        end
         state.cfx[1]:  begin
                        widget_type = widget_info(event.id,/type)
                        if widget_type eq 8 then tmp_value = widget_info(event.id,/droplist_select) $
                        else if widget_type eq 12 then begin
                           tmp_value = event.index
                           if (['No change','None','Be','HOPG','MgF2'])[tmp_value] ne event.str then begin
                              tmp_value = state.p_cfx+1
                              dm_set_droplist,event.id,select=tmp_value
                           endif
                        endif else tmp_value=0
                        state.p_cfx = tmp_value-1
                        end
         state.temp[1]: begin
                        widget_control,event.id,get_value=tmp
                        state.p_temp = dm_to_number(tmp)
                        end
         state.a3[1]:   begin
                        widget_control,event.id,get_value=tmp
                        state.p_a3 = dm_to_number(tmp)
                        end
         state.patchBut:begin
                        n_files = n_elements(open_file)
                        if n_files eq 0 then begin
                           ok = dialog_message('Please select '+state.ftype+' data files first.',/error,/center,dialog_parent=event.handler)
                           break
                        endif
                        if ~(file_test(save_path,/write)) then begin
                           ok = dialog_message('The pathced file directory is unwritable.',/error,/center,dialog_parent=event.handler)
                           break
                        endif
                        if state.ftype eq 'DCS' then begin
                           if state.dcsoptn[6,0] then begin
                              optn6_range = ['20190523_001','20190530_010','20190603_007']  ;after 20190530_010 only swap 756 and 787.                               
                              optn6_indx1 = [386,417,448,479,510,541,572,603,634,665,696,727,758,789,820,851,882,15,46,77,108,139,170,201,232,263,294,325,356,387,418,449,480,511,542,573,$
                                             604,635,666,697,728,759,790,821,852,883,16,47,756]
                              optn6_indx2 = [668,699,730,761,792,823,854,885,18,49,80,111,142,173,204,235,266,297,328,359,390,421,452,483,514,545,576,607,638,669,700,731,762,793,824,855,$
                                             886,19,50,81,112,143,174,205,236,267,298,329,787]              
                           endif else if state.dcsoptn[3,0] then begin
                              if ~finite(state.p_energy) then errmsg = 'Please enter the corresponding time channel energy value.'
                           endif else if state.dcsoptn[2,0] then begin
                              jumpy_file = ['']
                              sht_count  = [0]
                              for i=0,n_files-1 do begin
                                  if stregex(open_file[i],'(\.dcs\.gz|\.dcs)$',/fold_case,/boolean) then begin
                                     jumpy_file=[jumpy_file,open_file[i]]
                                     tmp = strsplit(open_file[i],'.',/extract)
                                     ind = where(stregex(dfiles,'^'+tmp[0]+'_[0-9]{4}\.sht$',/boolean),count)
                                     sht_count = [sht_count,count]
                                  endif             
                              endfor 
                              ind = where(sht_count ne 0,count)
                              if count eq 0 then begin
                                 ok = dialog_message('No .sht files are found for the selected data data file'+(['','s'])[n_files gt 1]+'.',/error,/center,dialog_parent=event.handler)
                                 break
                              endif else begin
                                 open_file = jumpy_file[ind]
                                 sht_count = sht_count[ind]
                                 n_files   = count 
                              endelse
                           endif else if state.dcsoptn[4,0] then begin
                              if ~finite(state.p_a2) then errmsg = 'Please enter A2 value.' $
                              else if n_files gt 1 then begin
                                 ok = dialog_message('The same A2 offset value of '+dm_to_string(state.a2offset)+' will be applied to all files.',/center,dialog_parent=event.handler)
                              endif
                           endif else if state.dcsoptn[5,0] then begin
                              if state.p_ss eq -1 then ok = dialog_message('Shutter state value of -1 will be used for unknown state.',/center,dialog_parent=event.handler)
                           endif else begin
                              if (~state.dcsoptn[1,0]) then begin  
                                 if state.p_mspd eq 0 then errmsg = 'Incorrect master speed.'
                                 if ~finite(state.p_wvln) then begin
                                    if n_elements(errmsg) eq 0 then errmsg = 'Incorrect wavelength.' $
                                    else errmsg = [errmsg,'Incorrect wavelength.']
                                 endif
                                 if state.p_denom eq 0 then begin
                                    if n_elements(errmsg) eq 0 then errmsg = 'Incorrect speed ratio denominator.' $
                                    else errmsg = [errmsg,'Incorrect speed ratio denominator.']
                                 endif
                              endif
                              if state.p_tsdmin eq 0 then begin
                                 if n_elements(errmsg) eq 0 then errmsg = 'Incorrect tsdmin.' $
                                 else errmsg = [errmsg,'Incorrect tsdmin.']
                              endif
                           endelse
                           ;Define scalar quantities
                           if state.dcsoptn[1,0] then begin
                              squans   = ["tsdmin"]
                              v_squans = double([state.p_tsdmin])
                           endif else begin
                              squans   = ["ch_ms","ch_res","ch_srdenom","ch_srmode","ch_wl","tsdmin"]
                              v_squans = double([state.p_mspd,state.p_resl,state.p_denom,state.p_srmode,state.p_wvln,state.p_tsdmin])
                           endelse
                        endif else if state.ftype eq 'MACS' then begin
                           if (state.p_flip lt 0) and (state.p_cfx lt 0) and finite(state.p_temp,/nan) and finite(state.p_a3,/nan) then errmsg = 'No changes have been selected.'
                        endif else break
                        if n_elements(errmsg) gt 0 then begin
                           ok = dialog_message(errmsg,/error,/center,dialog_parent=event.handler)
                           break
                        endif
                        for i=0L,n_files-1 do begin
                            if i eq 0 then mesg = obj_new('dm_progress',title='Patching...',message='Patching '+open_file[i],group_leader=state.tlb) $
                            else begin
                               if (~ obj_valid(mesg)) then break
                               mesg->update,message='Patching '+open_file[i]+'...  '+dm_to_string(i+1)+'/'+dm_to_string(n_files)
                            endelse
                            infile  = open_path+([state.pathsep,'/'])[obj_valid(ftpobj)]+open_file[i]
                            outfile = save_path+state.pathsep+state.p_strtlett+(['','_'])[state.ftype eq 'MACS']+open_file[i]
                            if obj_valid(ftpobj) then begin
                               ok = ftpobj->GetFileContent(infile,localfilename=ftpbufferdir+state.pathsep+'ftptmp'+open_file[i])
                               infile = ftpbufferdir+state.pathsep+'ftptmp'+open_file[i]
                            endif
                            if state.ftype eq 'DCS' then begin
                               if state.dcsoptn[2,0] then begin ; load the .sht datas first
                                  sum = 0L & temp_ind = 0L
                                  tmp = strsplit(open_file[i],'.',/extract)
                                  ind = where(stregex(dfiles,'^'+tmp[0]+'_[0-9]{4}\.sht$',/boolean),count)
                                  max_ind = 1
                                  for j=0,count-1 do begin
                                      infile1  = open_path+([state.pathsep,'/'])[obj_valid(ftpobj)]+dfiles[ind[j]]
                                      if obj_valid(ftpobj) then begin
                                         ok = ftpobj->GetFileContent(infile1,localfilename=ftpbufferdir+state.pathsep+'ftptmp'+dfiles[ind[j]])
                                         infile1 = ftpbufferdir+state.pathsep+'ftptmp'+dfiles[ind[j]]
                                      endif
                                      openr,unit,infile1,/get_lun
                                      data = ulonarr(1024,934) 
                                      readu,unit,data & free_lun,unit
                                      if obj_valid(ftpobj) then file_delete,infile1,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET
                                      ;now check if it is jumpy
                                      qty  = total(data[*,0:912],1)
                                      qty_max = max(qty) & qty_mean = mean(qty)
                                      m_ind   = where(qty ge 10*qty_mean,count1)
                                      if count1 ne 0 then qty[m_ind] = qty_max
                                      tmp = a_correlate(qty,[31,248])
                                      tmp1 = dm_to_number(strmid(infile1,7,4,/reverse_offset))
                                      max_ind = max_ind>(tmp1)
                                      if min(tmp) gt 0.3 then continue $
                                      else begin
                                         sum = sum+data
                                         temp_ind = [temp_ind,tmp1-1]
                                      endelse
                                  endfor 
                                  if n_elements(sum) eq 1 then begin   ;all jumpy data
                                     if n_elements(ajfile) eq 0 then ajfile = open_file[i] else ajfile = [ajfile,open_file[i]]
                                     continue
                                  endif
                                  sum = double(sum)
                               endif else if state.dcsoptn[6,0] then begin ;check file range for 5/2019 histogram hardware upgrade patch
                                  if (strmid(open_file[i],0,12) gt optn6_range[2]) or (strmid(open_file[i],0,12) lt optn6_range[0]) then begin                                 
                                     if n_elements(ncfile) eq 0 then ncfile = open_file[i] else ncfile = [ncfile,open_file[i]]
                                     continue
                                  endif
                               endif
                               ;check if the file is compressed
                               compress = (strlowcase(strmid(infile,2,3,/reverse_offset)) eq '.gz')
                               ;Read header to determine byte ordering ("endianness")
                               openr,unit,infile,compress=compress,/get_lun
                               header = bytarr(11)
                               readu,unit,header
                               free_lun,unit
                               continued = 0b
                               case string(header[9]) of
                                    "B": begin
                                         openr,unitr,infile,/swap_if_little_endian,compress=compress,/get_lun
                                         openw,unitw,outfile,/swap_if_little_endian,compress=compress,/get_lun
                                         end
                                    "L": begin
                                         openr,unitr,infile,/swap_if_big_endian,compress=compress,/get_lun
                                         openw,unitw,outfile,/swap_if_big_endian,compress=compress,/get_lun
                                         end
                                    else:begin
                                         ok = dialog_message("Cannot determine byte ordering of file "+open_file[i]+'. This file is not patched.',/center,dialog_parent=state.tlb)
                                         continued = 1b
                                         end
                               endcase
                               if continued then begin
                                  if obj_valid(ftpobj) then file_delete,infile,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET
                                  state.save_filesel->set_path      ;update the save directory
                                  continue
                               endif
                               readu,unitr,header
                               writeu,unitw,header             
                               ; Define the types of some variables
                               nlgth   = 0l
                               dldocgf = bytarr(5)
                               dt      = 0b
                               padding = 0b
                               scalar  = 0.0d
                               n1      = 0l
                               n2      = 0l
                               nstrgs  = 0l
                               nchars  = 0l
                               while (~eof(unitr)) do begin
                                     readu,unitr,nlgth              ; read name_length
                                     name = bytarr(nlgth)
                                     readu,unitr,name               ; read name
                                     readu,unitr,dldocgf            ; read 5 bytes of doc_length, doc, and global flag
                                     readu,unitr,dt                 ; read data type
                                     writeu,unitw,nlgth
                                     writeu,unitw,name
                                     writeu,unitw,dldocgf
                                     writeu,unitw,dt
                                     name = strlowcase(string(name))
                                     case dt of
                                          1: begin                  ; read scalar
                                             readu,unitr,padding
                                             writeu,unitw,padding
                                             readu,unitr,scalar
                                             ; Patchable scalars
                                             if (total(state.dcsoptn[2:*,0]) eq 0) then begin
                                                if state.dcsoptn[1,0] then begin ;save masterspeed and speed denominator info
                                                   case name of
                                                        'ch_ms':      ch_ms0 = scalar
                                                        'ch_srdenom': ch_srdenom0 = scalar
                                                        'tsdmin':     tsdmin0 = scalar
                                                        else:
                                                   endcase
                                                endif
                                                index = where(squans eq name,cnt)
                                                if (cnt eq 1) then scalar = v_squans[index]
                                             endif else if state.dcsoptn[5,0] then begin
                                                if name eq 'shutter_stat' then scalar = fix(state.p_ss,type=size(scalar,/type))
                                             endif else if state.dcsoptn[2,0] then begin
                                                case name of 
                                                     'duration':  scalar = scalar/max_ind*(n_elements(temp_ind)-1)
                                                     'ncycles':   scalar = double(n_elements(temp_ind)-1)
                                                     else:
                                                endcase
                                             endif else if state.dcsoptn[3,0] then begin
                                                case name of 
                                                     'tsdmin':    tsdmin0 = scalar
                                                     'ch_wl':     ch_wl0  = scalar
                                                     else:
                                                endcase
                                             endif
                                             writeu,unitw,scalar
                                             end
                                          2: begin                  ; read matrix
                                             readu,unitr,n1,n2
                                             nhi = (n1>n2)
                                             nlo = (n1<n2)
                                             if state.dcsoptn[2,0] and ((name eq 'temp_control') or (name eq 'temp_sample')) then begin
                                                if n1 eq 1 then n2 = n_elements(temp_ind)-1 $
                                                else n1 = n_elements(temp_ind)-1
                                             endif
                                             writeu,unitw,n1,n2
                                             readu,unitr,padding
                                             writeu,unitw,padding
                                             if (nlo eq 1) then data=dblarr(nhi) else data=dblarr(n1,n2) 
                                             readu,unitr,data
                                             if state.dcsoptn[2,0] then begin
                                                case name of 
                                                     'temp_control': data = data[temp_ind[1:*]]
                                                     'temp_sample':  data = data[temp_ind[1:*]]
                                                     'histodata':    data = sum[*,0:912]
                                                     'histohigh':    data = sum[*,913:930]
                                                     'timsum':       data = total(sum[*,0:930],2)
                                                     'detsum':       data = total(sum[*,0:930],1)
                                                     else:
                                                endcase
                                             endif else if state.dcsoptn[4,0] then begin
                                                if name eq 'motor_pos' then data[1] = state.a2offset+data[1]
                                             endif else if state.dcsoptn[3,0] or state.dcsoptn[6,0] then begin
                                                case name of 
                                                     'tchanlook':    tchanlook = data
                                                     'histodata':    begin
                                                                     if state.dcsoptn[3,0] then begin ;patch a single time channel
                                                                        times = tsdmin0+[0,reform(total(tchanlook[0,*]*0.05,/cumulative))]
                                                                        conversion = !dcs_hsq2mn/(!dcs_hom/!dcs_dsd)^2
                                                                        if (ch_wl0 eq 0.0) then energies = -conversion/times^2 $
                                                                        else energies = !dcs_hsq2mn/ch_wl0^2-conversion/times^2
                                                                        tmptmp = min(abs(energies-state.p_energy),tmp_imin)
                                                                        i_prev = tmp_imin-1 
                                                                        i_next = tmp_imin+1
                                                                        if i_prev lt 0 then i_prev = 999
                                                                        if i_next eq 1000 then i_next = 0 
                                                                        data[tmp_imin,*] = (data[i_prev,*]+data[i_next,*])/(2d)
                                                                     endif else begin ;path 5/2019 histogram hardware upgrade
                                                                        if strmid(open_file[i],0,12) gt optn6_range[1] then tmp_ind = n_elements(optn_indx1)-1 else tmp_ind = 0
                                                                        tmp = data[*,optn6_indx1[tmp_ind:*]]
                                                                        data[*,optn6_indx1[tmp_ind:*]] = data[*,optn6_indx2[tmp_ind:*]]
                                                                        data[*,optn6_indx2[tmp_ind:*]] = tmp
                                                                     endelse
                                                                     end 
                                                     else:
                                                endcase
                                             endif else begin
                                                if state.dcsoptn[1,0] and ((name eq 'histohigh') or (name eq 'histodata') or (name eq 'timesum')) then begin ;needs to shift data
                                                   tchwid = 6e4/ch_ms0*ch_srdenom0               ;assuming equal width, in usec
                                                   nshift = round((tsdmin0-state.p_tsdmin)/tchwid)
                                                   if (name eq 'histodata') or (name eq 'histohigh') then $
                                                      data=[shift(data[0:999,*],nshift,0),data[1000:1023,*]] $
                                                   else $
                                                      data=[shift(data[0:999],nshift),data[1000:1023]] ;timesum
                                                endif
                                             endelse
                                             writeu,unitw,data
                                             end
                                          5: begin                  ; read string
                                             readu,unitr,nchars     ; read number of characters in string element
                                             if (nchars gt 0) then begin
                                                str = bytarr(nchars)
                                                readu,unitr,str     ; read string element
                                             endif
                                             if (name eq 'comments') and state.dcsoptn[0,0] then begin
                                                nchars = long(strlen(state.p_comment)) 
                                                str = byte(state.p_comment)
                                             endif 
                                             writeu,unitw,nchars
                                             if (nchars gt 0) then writeu,unitw,str
                                             end
                                          7: begin                  ; read string array
                                             readu,unitr,nstrgs     ; read number of strings in array
                                             writeu,unitw,nstrgs
                                             for j=0L,nstrgs-1 do begin
                                                 readu,unitr,nchars ; read number of characters in string element
                                                 writeu,unitw,nchars
                                                 str = bytarr(nchars)
                                                 readu,unitr,str    ; read string element
                                                 writeu,unitw,str
                                             endfor
                                             end
                                          else:
                                     endcase
                               endwhile
                               free_lun,unitr
                               free_lun,unitw
                            endif else if state.ftype eq 'MACS' then begin
                               changed = 0b  
                               dm_load_macs,infile,data,info,header=header,trailer=trailer,/string,str_histo=str_histo,is_histogram=is_histogram
                               n_info = n_elements(info)
                               if obj_valid(ftpobj) then file_delete,infile,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET
                               if n_elements(data) eq 0 then continue
                               if state.p_flip ge 0 then begin
                                  ind_flip = where(strmatch(info,'flip',/fold_case),count)
                                  if count eq 1 then begin
                                     if state.p_flip eq 0 then begin
                                        data[*,ind_flip] = (['A','B'])[strmatch(data[*,ind_flip],'a',/fold_case)]
                                        changed = 1b
                                     endif else begin
                                        tmp1 = where(~strmatch(data[*,ind_flip],(['a','b'])[state.p_flip-1],/fold_case),cnt1)
                                        if cnt1 ne 0 then begin
                                           data[*,ind_flip] = (['A','B'])[state.p_flip-1]
                                           changed = 1b
                                        endif
                                     endelse
                                  endif
                               endif
                               if state.p_cfx ge 0 then begin
                                  ind_cfxbe = where(strmatch(info,'cfxbe',/fold_case),count_1)
                                  ind_cfxpg = where(strmatch(info,'cfxhopg',/fold_case),count_2)
                                  ind_cfxmg = where(strmatch(info,'cfxmgf',/fold_case),count_3)
                                  if (count_1 eq 1) and (count_2 eq 1) and (count_3 eq 1) then begin
                                     tmp1 = where(~strmatch(data[*,ind_cfxbe],(['out','in'])[state.p_cfx eq 1],/fold_case),cnt1) 
                                     tmp2 = where(~strmatch(data[*,ind_cfxpg],(['out','in'])[state.p_cfx eq 2],/fold_case),cnt2)
                                     tmp3 = where(~strmatch(data[*,ind_cfxmg],(['out','in'])[state.p_cfx eq 3],/fold_case),cnt3)
                                     if cnt1+cnt2+cnt3 ne 0 then begin
                                        data[*,ind_cfxbe] = (['OUT','IN'])[state.p_cfx eq 1]
                                        data[*,ind_cfxpg] = (['OUT','IN'])[state.p_cfx eq 2]
                                        data[*,ind_cfxmg] = (['OUT','IN'])[state.p_cfx eq 3]
                                        changed = 1b
                                     endif
                                  endif else begin
                                     ind_vbav = where(strmatch(info,'vbav',/fold_case),count_4)
                                     if count_4 ne 0 then data = shift(data,0,n_info-1-ind_vbav[0])
                                     data = [[data],[strarr(n_elements(data[*,0]),3)]] 
                                     data[*,n_info] = (['OUT','IN'])[state.p_cfx eq 1]
                                     data[*,n_info+1] = (['OUT','IN'])[state.p_cfx eq 2]
                                     data[*,n_info+2] = (['OUT','IN'])[state.p_cfx eq 3]
                                     if count_4 ne 0 then begin
                                        data = shift(data,0,ind_vbav[0]-n_info+1)
                                        info = [info[0:ind_vbav[0]],'cfxbe','cfxhopg','cfxmgf',info[(ind_vbav[0]+1):*]]
                                     endif else begin
                                        info = [info,'cfxbe','cfxhopg','cfxmgf']
                                     endelse
                                     changed = 1b
                                  endelse
                               endif
                               if finite(state.p_temp) then begin
                                  ind_temp = where(strmatch(info,'temp',/fold_case),count1)
                                  if count1 eq 1 then begin
                                     data[*,ind_temp] = dm_to_string(state.p_temp)
                                  endif else begin
                                     ind_time = where(strmatch(info,'time',/fold_case),count2)
                                     if count2 ne 0 then begin
                                        data = shift(data,0,n_info-1-ind_time[0])
                                        data = [[data],[strarr(n_elements(data[*,0]))]]
                                        data[*,n_info] = dm_to_string(state.p_temp)
                                        data = shift(data,0,ind_time[0]-n_info+1)
                                        data = shift(data,0,1)
                                        data = [[data],[strarr(n_elements(data[*,0]),2)]]
                                        data[*,(n_info+1):(n_info+2)] = dm_to_string(state.p_temp)
                                        data = shift(data,0,-1)
                                        info = [info[0:ind_time[0]],'Temp',info[(ind_time[0]+1):(n_info-2)],'TemperatureControlReading','TemperatureSetpoint',info[n_info-1]]
                                     endif else begin
                                        data = shift(data,0,1)
                                        data = [[data],[strarr(n_elements(data[*,0]),3)]]
                                        data[*,n_info:(n_info+2)] = dm_to_string(state.p_temp)
                                        data = shift(data,0,-1)
                                        info = [info[0:(n_info-2)],'Temp','TemperatureControlReading','TemperatureSetpoint',info[n_info-1]]
                                     endelse
                                  endelse
                                  changed = 1b
                               endif
                               if finite(state.p_a3) then begin
                                  ind_a3 = where(strmatch(info,'a3',/fold_case),count1)
                                  if count1 eq 1 then begin
                                     if n_elements(a3offset) eq 0 then a3offset = state.p_a3-dm_to_number(data[0,ind_a3],/double)
                                     data[*,ind_a3] = dm_to_string(dm_to_number(data[*,ind_a3],/double)+a3offset)
                                     changed = 1b
                                  endif
                               endif
                               if ~keyword_set(changed) then begin  ;no change
                                  if n_elements(ncfile) eq 0 then ncfile = open_file[i] else ncfile = [ncfile,open_file[i]]
                                  continue  
                               endif
                               dm_write_macs,outfile,header=header,info=info,data=data,is_histogram=is_histogram,str_histo=str_histo,trailer=trailer,nexus=nexus,mesgobj=mesg
                            endif
                            if obj_valid(ftpobj) then file_delete,infile,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET
                            state.save_filesel->set_path      ;update the save directory
                        endfor
                        if n_elements(ajfile) ne 0 then begin
                           if n_elements(ajfile) eq n_files then ad_mesg = "There is no recoverable data in "+(["the selected file.","all the selected files."])[n_elements(ajfile) gt 1] $
                           else ad_mesg = ["There is no recoverable data in the following file"+(['','s'])[n_elements(ajfile) gt 1]+":",ajfile]
                           ad_mesg = [ad_mesg,'', (["This file is","These files are"])[n_elements(ajfile) gt 1]+" not patched."]
                        endif
                        if n_elements(ncfile) ne 0 then begin
                           if n_elements(ncfile) eq n_files then ad_mesg = "There is no change to "+(["the selected file.","all the selected files."])[n_elements(ncfile) gt 1] $
                           else ad_mesg = ["There is no change to the following file"+(['','s'])[n_elements(ncfile) gt 1]+":",ncfile]
                           ad_mesg = [ad_mesg,'', (["This file is","These files are"])[n_elements(ncfile) gt 1]+" not patched."]
                        endif
                        if n_elements(ad_mesg) ne 0 then ok = dialog_message(ad_mesg,/center,dialog_parent=state.tlb)
                        state.open_filesel->set_path,ftp=obj_valid(ftpobj)
                        state.open_filesel->clear_file,fileloc=open_file[0]
                        if obj_valid(mesg) then obj_destroy,mesg
                        end  
         else:          begin
                        if strlowcase(tag_names(event,/structure)) eq 'dm_filesel_select' then begin  ;left file selector click event
                           if event.object eq state.open_filesel then begin
                              file = open_path+([state.pathsep,'/'])[obj_valid(ftpobj)]+open_file[0]
                              if obj_valid(ftpobj) then begin
                                 ok = ftpobj->GetFileContent(file,localfilename=ftpbufferdir+state.pathsep+'ftptmp'+open_file[0])
                                 file = ftpbufferdir+state.pathsep+'ftptmp'+open_file[0]
                              endif
                              case state.ftype of 
                                  'DCS':  begin
                                          davePtr = dm_define_pointer() & omithisto=2 & printinfo=0 & filetype='binary'     
                                          dcs_readdatafile,file,omithisto,printinfo,filetype,davePtr=davePtr
                                          this = dm_locate_datastrptr(davePtr)
                                          widget_control,state.mspd[0],set_value=dm_to_string((*(*this).specificPtr).ch_ms)
                                          widget_control,state.wvln[0],set_value=dm_to_string((*(*this).specificPtr).ch_wl,resolution=3)
                                          widget_control,state.tsdmin[0],set_value=dm_to_string((*(*this).specificPtr).tsdmin)
                                          dm_set_droplist,state.resl[0],select=(*(*this).specificPtr).ch_res-1
                                          dm_set_droplist,state.srmode[0],select=(*(*this).specificPtr).ch_srmode-1
                                          widget_control,state.denom[0],set_value=dm_to_string((*(*this).specificPtr).ch_srdenom)
                                          if state.dcsoptn[0,0] then begin
                                             state.p_comment = strtrim((*(*this).specificPtr).comments,2) 
                                             widget_control,state.comment,set_value=state.p_comment
                                          endif else if state.dcsoptn[4,0] then begin
                                             a2 = round(((*(*this).specificPtr).motor_pos)[1]*1000.0d)/(1000.0d) ;accurate to 0.001 degree
                                             widget_control,state.a2[0],set_value=dm_to_string(a2)
                                             if finite(state.p_a2) then begin
                                                if finite(state.a2offset) then begin  
                                                   state.p_a2 = a2+state.a2offset
                                                   widget_control,state.a2[1],set_value=dm_to_string(state.p_a2)
                                                endif else begin
                                                   state.a2offset = state.p_a2-a2
                                                endelse
                                             endif
                                          endif else if state.dcsoptn[5,0] then begin
                                             dm_set_droplist,state.a2[0],select=(*(*this).specificPtr).shutter_stat+1
                                          endif    
                                          heap_free,davePtr
                                          end
                                  'MACS': begin
                                          dm_load_macs,file,data,info,header=header,trailer=trailer,/string
                                          if n_elements(data) eq 0 then begin
                                             flip = 0 & cfx = 0 & temp = !values.f_nan & a3 = !values.f_nan
                                          endif else begin
                                             ndata = n_elements(data[*,0])
                                             ind_flip  = where(strmatch(info,'flip',/fold_case),cnt_flip)
                                             ind_cfxbe = where(strmatch(info,'cfxbe',/fold_case),cnt_cfxbe)
                                             ind_cfxpg = where(strmatch(info,'cfxhopg',/fold_case),cnt_cfxpg)
                                             ind_cfxmg = where(strmatch(info,'cfxmgf',/fold_case),cnt_cfxmg)
                                             ind_temp  = where(strmatch(info,'temp',/fold_case),cnt_temp)
                                             ind_a3    = where(strmatch(info,'a3',/fold_case),cnt_a3)
                                             if cnt_flip eq 0 then flip = 0 else begin
                                                tmp1 = where(strmatch(data[*,ind_flip],'a',/fold_case),cnt1)
                                                tmp2 = where(strmatch(data[*,ind_flip],'b',/fold_case),cnt2)
                                                if cnt1 eq ndata then flip = 1 $
                                                else if cnt2 eq ndata then flip = 2 $
                                                else if cnt1+cnt2 eq ndata then flip = 3 $
                                                else flip = 4
                                             endelse
                                             if (cnt_cfxbe eq 0) and (cnt_cfxpg eq 0) and (cnt_cfxmg eq 0) then cfx = 0 else begin
                                                if cnt_cfxbe ne 0 then tmp1 = (strmatch(data[0,ind_cfxbe],'in',/fold_case))[0] else tmp1 = 0
                                                if cnt_cfxpg ne 0 then tmp2 = (strmatch(data[0,ind_cfxpg],'in',/fold_case))[0] else tmp2 = 0
                                                if cnt_cfxmg ne 0 then tmp3 = (strmatch(data[0,ind_cfxmg],'in',/fold_case))[0] else tmp3 = 0
                                                cfx = 1+tmp1+2*tmp2+4*tmp3
                                             endelse
                                             if cnt_temp eq 0 then temp = !values.f_nan else temp = mean(dm_to_number(data[*,ind_temp]))
                                             if cnt_a3   eq 0 then a3   = !values.f_nan else a3   = dm_to_number(data[0,ind_a3])
                                          endelse
                                          dm_set_droplist,state.flip[0],select=flip
                                          dm_set_droplist,state.cfx[0],select=cfx
                                          widget_control,state.temp[0],set_value=dm_to_string(temp)
                                          widget_control,state.a3[0],set_value=dm_to_string(a3)
                                          end
                                  else:
                              endcase
                              if obj_valid(ftpobj) then file_delete,file,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET    
                           endif
                        endif else begin  ;file name starting letter menu event
                           ind = where(state.lett_ids eq event.id,cnt)
                           if cnt eq 1 then begin
                              dm_toggle_menubut,uncheck=state.lett_ids[fix(byte(state.p_strtlett))-fix(byte('A'))],check=event.id
                              state.p_strtlett = string(byte('A')+byte(ind))
                              widget_control,state.patchBut,set_value='Patch --> ('+state.p_strtlett+')'
                              if float(!version.release) ge 5.6 then widget_control,state.patchBut,tooltip=state.tooltip[state.dcsoptn[2,0]+2*state.dcsoptn[6,0]]+state.p_strtlett+'.'
                           endif
                        endelse
                        end
    endcase
    if event_cnt ne 0 then begin
       widget_control,state.cmdCol,/map
       widget_control,state.tlb,/update
    endif
    widget_control,event.handler,set_uvalue=state,/no_copy
end

;main program
pro dm_patch,event,dataDir=dataDir,workDir=workDir,macs=macs,ncnrftp=ncnrftp
    state = {group_leader:       0L, $   ;group leader
             tlb:                0L, $   ;top level base
             ftype:              '', $   ;file type
             dcsBut:             0L, $   ;DCS type
             macsBut:            0L, $   ;MACS type
             optnmenu:           0L, $   ;option menu base
             cmdCol:             0L, $   ;center column base
             openlabel:          0L, $   ;open file selector label
             patchBut:           0L, $   ;convert button
             doneBut:            0L, $   ;done button
             helpBut:            0L, $   ;Help button
             open_filesel:obj_new(), $   ;open file selector
             save_filesel:obj_new(), $   ;save file selector
             ncnrftp:            0b, $   ;flag for allowing access to NCNR ftp server
             ftpBut:             0L, $   ;allow NCNR ftp option button
             dataDir:            '', $   ;save initial data directory
             strtlett:           0L, $   ;starting letter menu
             lett_ids:   lonarr(26), $
             p_strtlett:        'P', $   ;starting letter
             s_strtlett:        'R', $   ;buffer
             pathsep:            '', $   ;path seperator
             tooltip:    ['','',''], $   ;tool tip for the patch button
             ;dcs
             optnname:    strarr(7), $   ;dcs options name. To add an option: 1. add optnname in dm_patch_changeftype 2. handle the optnBut and patch event in dm_patch_event.  
             optnBut:     lonarr(7), $   ;dcs options buttons
             dcsoptn:   bytarr(7,2), $   ;dcs options, point to a byte array: [[allowcomment, tsdminonly,jumpyonly,tconly,a2only,ssonly,6/2019 patch],[flag for showing the corresponding info]]             
             mspd:          [0L,0L], $   ;[raw master speed text box,patch master speed text box]
             p_mspd:         20000s, $   ;master speed 
             wvln:          [0L,0L], $   ;[raw wavelength text box,patch wavelength text box]
             p_wvln:  !values.f_nan, $   ;wavelength
             tsdmin:        [0L,0L], $   ;[raw tsdmin text box,patch tsdmin text box]
             p_tsdmin:           0s, $   ;tsdmin
             resl:          [0L,0L], $   ;[raw resolution mode selection droplist,patch resolution mode selection droplist]
             p_resl:             1s, $   ;resolution mode    1-low 2-medium
             srmode:        [0L,0L], $   ;[raw speed ratio mode selection droplist,patch speed ratio mode selection droplist]
             p_srmode:           1s, $   ;speed ratio mode   1-1/m 2-(m-1)/m
             denom:         [0L,0L], $   ;[raw speed ratio denominator text box,patch speed ratio denominator text box]
             p_denom:            0s, $   ;speed ratio denominator
             clabel:             0L, $   ;comment/a2 label
             aline:              0L, $   ;a2 line
             a2:            [0L,0L], $   ;[raw a2 text box,patch a2 text box]
             p_a2:    !values.f_nan, $   ;a2 value
             a2offset:!values.f_nan, $   ;save a2 offset value         
             p_ss:              -1s, $   ;shutter state value -1:unknown, 0:close, 1:open
             comment:            0L, $   ;comment text box, or energy text box
             p_comment:          '', $   ;comment string
             p_energy:!values.f_nan, $   ;energy value
             ;macs
             flip:          [0L,0L], $   ;[raw data flip state droplist id,patched flip state droplist id]
             p_flip:            -1s, $   ;patched flip choice
             cfx:           [0L,0L], $   ;[raw data CFX filter droplist id,patched CFX filter droplist id]
             p_cfx:             -1s, $   ;patched CFX filter choice
             temp:          [0L,0L], $   ;[raw data average sample temperature text box id, patched temperature text box id]
             p_temp:  !values.f_nan, $   ;patched temperature
             a3:            [0L,0L], $   ;[raw data first a3 text box id, patched a3 text box id]
             p_a3:    !values.f_nan, $   ;patched first a3 value        
             geom:    [0L,0L,0L,0L]  $   ;xsize,ysize,xoffset,yoffset
    }
    if keyword_set(macs) then state.ftype = 'MACS' else state.ftype = 'DCS'
    state.ncnrftp = keyword_set(ncnrftp)
    registerName  = 'dm_patch'
    if xregistered(registerName) then begin   ;only allow one copy to be running at one time
       FORWARD_FUNCTION LookupManagedWidget
       id = LookupManagedWidget(registername)
       widget_control,id,/show,/realize,iconify=0,get_uvalue=state1
       if state1.ftype ne state.ftype then begin  ;switch type
          dm_patch_changeftype,state1,ftype=state.ftype
          widget_control,id,set_uvalue=state1
       endif
       return   
    endif
    
    if n_elements(event) ne 0 then begin
       state.group_leader = event.top
       state.tlb = widget_base(title=title,/col,kill_notify='dm_patch_Exit',group_leader=event.top,/tlb_size_event,mbar=mbar,xpad=0,ypad=1,map=0)
    endif else $
       state.tlb = widget_base(title=title,/col,kill_notify='dm_patch_Exit',/tlb_size_event,mbar=mbar,xpad=0,ypad=1,map=0)
    
    ;menu bar
    filemenu       = widget_button(mbar,value='File',/menu)
    state.optnmenu = widget_button(mbar,value='Options',/menu)
    helpmenu       = widget_button(mbar,value='Help',/menu)
    ftypmenu       = widget_button(filemenu,value='File Type',/menu)
    state.dcsBut   = dm_widget_button(ftypmenu,value='DCS')
    state.macsBut  = dm_widget_button(ftypmenu,value='MACS')
    state.doneBut  = widget_button(filemenu,value='Exit',/separator)
    state.helpBut  = widget_button(helpmenu,value='Help')
    
    ;menu bar separator for windows system
    if !version.os_family eq 'Windows' then begin
       mbar_sep    = widget_label(state.tlb,sensitive=0,/dynamic_resize,value=' ',scr_ysize=5)
       tmp         = 0
    endif else tmp = 4
    
    row            = widget_base(state.tlb,/row,ypad=0)
    openCol        = widget_base(row,/col,ypad=0)
    state.cmdCol   = widget_base(row,/col,/align_center,ypad=0,xpad=0)
    saveCol        = widget_base(row,/col,ypad=0)
    state.openlabel = widget_label(openCol,value=state.ftype+' data file directory')
    label          = widget_label(saveCol,value='Patched file directory')
    if n_elements(workDir) eq 0 then cd,current=workDir
    if n_elements(dataDir) ne 0 then state.dataDir = dataDir
    state.open_filesel = obj_new('dm_filesel',parent=openCol,ysize=18+tmp,/frame,filter=fext,path=dataDir,/selectevent,group_leader=state.tlb,ncnrftp=state.ncnrftp)
    state.save_filesel = obj_new('dm_filesel',parent=saveCol,ysize=18+tmp,/frame,filter=fext,path=workDir)
    state.pathsep = dm_define_pointer(/getpathsep)
    state.tooltip = ['Select data files, set patch values in the center right column, then click this button to patch files.','Select raw .dcs.gz files, then click this button to patch. Sht files need to be present.',$
                     'Select data files, then click this button to patch files.']+' Patched file name starts with '
    if state.ncnrftp then begin ;ftp connection might not work
       state.open_filesel->getproperty,ncnrftp=ncnrftp
       state.ncnrftp = ncnrftp
    endif                 
    dm_patch_changeftype,state,/center
    geom = widget_info(state.tlb,/geometry)
    state.geom = [geom.scr_xsize,geom.scr_ysize,geom.xoffset,geom.yoffset]
    widget_control,state.tlb,set_uvalue=state
    xmanager,registerName,state.tlb,cleanup='dm_patch_Exit',event_handler='dm_patch_event',/no_block
end