; $Id: $
;#######################################################################
; 
; NAME:
;  dm_load_spe
;
; PURPOSE:
;  load spe file returns zdat[nx,ny], err[nx,ny], xdat[nx or nx+1], ydat[ny or ny+1]
;
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  August, 2023
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################
@dm_load_dcsdata  ;for dm_findbetween

;header = [ndet,nen,emin,emax]
pro dm_load_spe,file,zdat=zdat,zerr=zerr,xdat=xdat,ydat=ydat,xtit=xtit,ytit=ytit,ztit=ztit,$
    error=error,group_leader=group_leader,xhist=xhist,yhist=yhist,header=header

    ;check if the file is compressed
    if strlowcase(strmid(file,2,3,/reverse_offset)) eq '.gz' then compress=1 else compress=0

    openr,unit,file,/get_lun,error=error,compress=compress
    if error ne 0 then begin  
       ok = dialog_message("Can't open "+file,/error,dialog_parent=group_leader)
       return
    endif

    ;catch and clear possible io errors
    catch, myerror
    if myerror ne 0 then begin
       error = 1b
       catch,/cancel
       if n_elements(unit) ne 0 then free_lun,unit
       ok = dialog_message(!error_state.msg,/error,dialog_parent=group_leader)
       return
    end
   
    nx = 0L & ny = 0L & junk=' '
    
    ;firgure out the string formats
    readf,unit,junk       ; line 1
    extra = strlen(junk) mod 2
    width = strlen(junk)/2
    if width eq 0 then width = 5 ;default
    if extra ne 0 then $
       format2 = '((i'+dm_to_string(width+extra)+',i'+dm_to_string(width)+'))' $
    else  $
       format2 = '(2(i'+dm_to_string(width)+'))'
    readf,unit,junk
    readf,unit,junk       ; line 3
    point_lun,unit,0      ;reset the pointer to start
    extra = strlen(junk) mod 8
    width = strlen(junk)/8
    if width eq 0 then width = 10 ;default
    if extra ne 0 then $
       format1 = '(((g'+dm_to_string(width+extra)+'),7(g'+dm_to_string(width)+')))' $
    else $
       format1 = '(8(g'+dm_to_string(width)+'))'  
    
    readf,unit,junk
    tmp = strtrim(junk,2)
    if (strpos(tmp,' '))[0] eq -1 then begin
       reads,junk,nx,ny,format=format2
    endif else begin
       reads,junk,nx,ny
    endelse
    
    if arg_present(header) then header = [nx,ny]

    xdat = fltarr(nx+1)
    ydat = fltarr(ny+1)
    readf,unit,junk    ;read ### xdat
    xtit = strtrim((strsplit(junk,'###',/extract))[0],2)
    
    ;old code
    ;readf,unit,xdat,format=format1
    ;readf,unit,junk    ;read ### ydat
    if nx/8 ne 0 then begin
       tmp = fltarr((nx/8)*8)
       readf,unit,tmp,format=format1
       xdat[0:((nx/8)*8-1)]=tmp
    endif
    readf,unit,junk & junk=junk+' '
    if finite(dm_to_number(junk)) then begin
       tmp = fltarr(nx-(nx/8)*8+1)
       reads,junk,tmp,format=format1
       xdat[(nx/8)*8:nx]=tmp
       readf,unit,junk    ;read ### ydat
    endif 

    ytit = strtrim((strsplit(junk,'###',/extract))[0],2)
    readf,unit,ydat,format=format1

    ;determine whether the data is point type or histogram type
    if (xdat[nx] eq 0) and (nx ge 2) then begin
       if xdat[nx-1]-xdat[0] ge 0 then $
          tmp = xdat[1:nx]-xdat[0:nx-1] $
       else $
          tmp = xdat[0:nx-1]-xdat[1:nx]
       tmp1 = where(tmp ge 0,count)
       if count ne nx then $    ;not monotonic
          xdat = xdat[0:nx-1] $
       else if tmp[nx-1] gt 5.*tmp[nx-2] then $    ;arbitrary criterion
          xdat = xdat[0:nx-1]
    endif
    xhist = n_elements(xdat) eq (nx+1)
    if (ydat[ny] eq 0) and (ny gt 2) then begin
       if ydat[ny-1]-ydat[0] ge 0 then $
          tmp = ydat[1:ny]-ydat[0:ny-1] $
       else $
          tmp = ydat[0:ny-1]-ydat[1:ny]
       tmp1 = where(tmp ge 0,count)
       if count ne ny then $    ;not monotonic
          ydat = ydat[0:ny-1] $
       else if tmp[ny-1] gt 5.*tmp[ny-2] then $    ;arbitrary criterion
          ydat = ydat[0:ny-1]
    endif
    
    if arg_present(header) then begin
       if n_elements(ydat) eq (ny+1) then $
          min_eng = min((ydat[0:ny-1]+ydat[1:ny])/2.,max=max_eng) $
       else $ 
          min_eng = min(ydat,max=max_eng)
       header = [header,min_eng,max_eng]
    endif
    
    if ~arg_present(zdat) then begin  ;header only
       free_lun,unit
       return
    endif
    
    yhist = n_elements(ydat) eq (ny+1)
    tmp   = fltarr(ny)
    zdat  = fltarr(nx,ny)
    zerr  = fltarr(nx,ny)
    for i=0L,nx-1L do begin
        readf,unit,junk   ;read ### S(phi,w)
        if i eq 0 then begin  ;get ztit
           ztit = strtrim((strsplit(junk,'###',/extract))[0],2)
        endif
        readf,unit,tmp,format=format1
        zdat[i,*] = tmp
        readf,unit,junk   ;read ### Errors
        readf,unit,tmp,format=format1
        zerr[i,*] = tmp
    endfor
    free_lun,unit
end

;this procedure reads a spe file for dcs_mslice program
;Parameters:
;   open_path:  file path, string
;   open_file:  file name(s), string or string array
;Keywords:
; input:
;   avgsum:     0:arithmetic average 1:sum, effective when diffuse keyword is sest, default is self.bin_avgsum
;   background: if set, the loaded files are regarded as background files and will be summed
;   diffetyp:   only needed when diffuse keyword is set 0:single energy  1:multiple energy
;   diffuse:    if set, diffuse scattering mode
;   parent:     message widget parent,  default is self.tlb
;   title:      message title
;   eint_yn:    only applicable for diffuse mode, if set, I=sum(I[i,j]*dE[i,j]) over the specified time channels
;   e_range:    energy range - diffuse keyword set, default is self.e_range
;   t_chan:     0:all 1:elastic 2:specify erange  - efffective when diffuse keyword set, default is self.e_bin[self.samp_typ]
;   phxfile:    string containing the phx file name
;   eief:       if present and given a value, the procedure will not prompt for Ei or Ef value from user
;   masklimit:  a number indicating the threshhold below which the detector is masked, default:-1e20
; output:
;   error:      error flag 1 or 0
;   qty:        [ne,ndet] or [ne,ndet,nfiles]-diffuse keywords set [ne,ndet] if nfiles=1
;   dqty:       [ne,ndet] or [ne,ndet,nfiles]-diffuse keywords set [ne,ndet] if nfiles=1
;   ang:        [nfiles] - diffuse keyword set, not in spe file, need to be typed in by the user
;   eief:       Ei or Ef(depends on the instrument geometry), not in the spe file, need to be typed in by the user
;   emean:      [ne]
;   ewid:       diffuse:ewid[tchans] ; powder,single:[ntchan], the corresponding energy width for the time channels
;   info:       {ne:ne, ndet:ndet}  used for checking file compatibility
;   weight:     weight or duration info , not in the file, need to be typed in by the user
;   kfactor:    [ntchan]-(ki/kf)  or a scalor
pro dcs_mslice::dm_load_spe,open_path,open_file,ang=ang,avgsum=avgsum,background=background,$
    diffetyp=diffetyp,diffuse=diffuse,eief=eief,e_range=e_range,eint_yn=eint_yn,emean=emean,$
    error=error,ewid=ewid,info=info,kfactor=kfactor,parent=parent,phxfile=phxfile,qty=qty,dqty=dqty,$
    scriptmode=scriptmode,t_chan=t_chan,title=title,masklimit=masklimit,adddata=adddata,weight=weight
    ;   eadjust:    0: none 1:automatic 2:specified
    ;   epadjust:   for eadjust=2, epadjust specified the apparent elastic peak position
    ;   epadcheckfirst: if set, only check the first file for the shifting of elastic peak, and use it for all files
    eadjust        = self.eadjust
    epadjust       = self.eadjust_spec
    epadcheckfirst = self.eadjust_checkfirst
    if n_elements(masklimit)   eq 0 then masklimit = -1e20
    if (eadjust eq 2) and (~ finite(epadjust)) then eadjust=0
    sch_ead        = (eadjust gt 0)
    cd,current=current
    if file_test(self.dirs[1],/directory) then cd,self.dirs[1]
    if keyword_set(diffuse) then begin
       if n_elements(avgsum)   eq 0 then avgsum   = self.bin_avgsum
       if n_elements(t_chan)   eq 0 then t_chan   = self.e_bin[self.samp_typ]
       if n_elements(e_range)  ne 2 then e_range  = self.e_range
       if n_elements(diffetyp) eq 0 then diffetyp = (self.samp_typ eq 3)
       if diffetyp eq 0 then sch_ead = 0b
       if keyword_set(adddata) and t_chan ne 0 then t_chan = 2 ;forced to specify
    endif else begin
       if n_elements(diffetyp) eq 0 then diffetyp = 1     ;default is multiple energy
    endelse
    if n_elements(parent) eq 0 then parent = self.tlb
    if n_elements(title)  eq 0 then title  = 'Loading data ...'
    error   = 0b  ;initialize no error
    n_files = n_elements(open_file)
    n_read  = 0

    ;catch and clear possible io errors
    catch, myerror
    if myerror ne 0 then begin
       error = 1b
       catch,/cancel
       if obj_valid(mesg) then obj_destroy,mesg
       ok = dialog_message(!error_state.msg,/error)
       if n_elements(unit) ne 0 then free_lun,unit
       return
    end

    ;ask for ei/ef and phx file
    if arg_present(eief) and (~keyword_set(scriptmode)) then begin
       notok  = 1b
       prompt = [(['Ei (meV):','Ef (meV):'])[0>(self.instrgeom)<1],'phx file:']
       while notok do begin
          tmp   = dm_dialog_input(prompt,dialog_parent=parent,default=[dm_to_string(self.eief),self.phxfile],/e2lambda,$
                   info='Skip phx file to use the 2theta info in the SPE file.',is_file=[0,1],xsize=170,filter='phx')
          eief  = dm_to_number(tmp[0])
          notok = ~finite(eief)
       endwhile
       phxfile = tmp[1]
    endif else begin
       if n_elements(eief) eq 0 then eief = (finite(self.eief)?(self.eief):1e3)      ;arbitrary number
       if n_elements(phxfile) eq 0 then phxfile = ''
    endelse
    
    ;ask for weight info
    myweight = 1.0
    if (arg_present(weight) or (n_files ne 1)) and (~keyword_set(scriptmode)) then begin
       myweight = !values.f_nan & default = '1'
       while(total(finite(myweight)) ne n_files) do begin
            myweight = dm_dialog_listinput(open_file,title='Please input weight info:',default=default,$
                 label='weight',dialog_parent=parent,reset=1,info=['The weight is for summing files and weighted mean average binning, not for normalizing the intensity.',$
                 "Press OK if you don't know the information."],xvalue=open_file)
            default = myweight & myweight = dm_to_number(myweight)
       endwhile
    endif

    ;ask for rotation angles
    if keyword_set(diffuse) and (~ keyword_set(background)) then begin
       if ~keyword_set(scriptmode) then begin
          default = (n_elements(ang) ne 0)?(dm_to_string(ang)):'' & ang = !values.f_nan  
          while(total(finite(ang)) ne n_files) do begin
             ang = dm_dialog_listinput(open_file,title='Please input rotation angles:',default=default,$
                   label='angle',dialog_parent=parent,reset=0,xvalue=open_file)
             default = ang & ang = dm_to_number(ang)
          endwhile
       endif else if n_elements(ang) ne n_files then ang = fltarr(n_files)
    endif

    for i=0L,n_files-1L do begin
        file = open_path+self.pathsep+open_file[i]
        if i eq 0 then $
           mesg =  obj_new('dm_progress',title=title,message='Loading '+file,group_leader=parent) $
        else $
           mesg->update,message='loading '+file,title='Loading...  '+strtrim(string(i+1),2)+'/'+strtrim(string(n_files),2)

        ;read the spe file
        dm_load_spe,file,zdat=zdat,zerr=dzdat,xdat=xdat,ydat=ydat,xtit=xtit,ytit=ytit,ztit=ztit,error=error,group_leader=parent    ;xdat:tth, ydat:en
        if error ne 0 then begin
           if obj_valid(mesg) then obj_destroy,mesg
           return
        endif
        
        ;check dets masking
        ii = 0L & ncheck = n_elements(zdat[*,0]) & mskcount = ncheck
        while(mskcount eq ncheck) do begin
             mask = where(zdat[*,ii] le masklimit,mskcount,complement=tmpnomask)
             ii = ii+1    ;in case the first time channel is masked
        endwhile
        if mskcount gt 0 then begin
           if n_elements(nomask) eq 0 then begin
              nomask = tmpnomask 
           endif else begin
              nomask = dm_common(nomask,tmpnomask,count)
              if count eq 0 then tmp = temporary(nomask)
           endelse
           zdat[mask,*]  = !values.f_nan
           dzdat[mask,*] = !values.f_nan
        endif
        ;check time channels masking
        ii = 0L & ncheck = n_elements(zdat[0,*]) & mskcount = ncheck
        while(mskcount eq ncheck) do begin
             mask = where(zdat[ii,*] le masklimit,mskcount)
             ii = ii+1    ;in case the first detector is masked
        endwhile
        if mskcount gt 0 then begin
           zdat[*,mask]  = !values.f_nan
           dzdat[*,mask] = !values.f_nan
        endif
     
        ndet = n_elements(xdat)
        nen  = n_elements(ydat)
        zdim = size(zdat,/dimension)

        if n_elements(info) eq 0 then begin
           info={ndet:ndet,nen:nen,nx:zdim[0],ny:zdim[1]}
        endif else begin
           if (info.ndet ne ndet) or (info.nen ne nen) or (info.nx ne zdim[0]) or (info.ny ne zdim[1]) then begin
              ok = dialog_message('Selected data files are of different types.',/error,dialog_parent=parent)
              error = 1b                          ;error occurs
              if obj_valid(mesg) then obj_destroy,mesg
              return
           endif
        endelse

        if i eq 0 then begin
           emean = temporary(ydat)
           tth   = temporary(xdat)
           dtth  = fltarr(zdim[0])
           psi   = fltarr(zdim[0])
           dpsi  = fltarr(zdim[0])
           ewid  = emean[1:nen-1]-emean[0:nen-2]
           if n_elements(ewid) ne zdim[1] then ewid = [ewid,2*ewid[nen-2]-ewid[nen-3]]
           if ndet eq zdim[0]+1 then tth   = (tth[0:(ndet-2)]+tth[1:(ndet-1)])/2.
           if nen  eq zdim[1]+1 then emean = (emean[0:(nen-2)]+emean[1:(nen-1)])/2.

           if keyword_set(diffuse) then begin
              weight  = myweight
              all_tch = 1
              kfactor = 1.0
              ;search the proper time channels to be summed or averaged
              case t_chan of
                   0:          ;all time channels
                   1:  begin   ;elastic peak
                       if n_elements(nomask) ne 0 then $
                          tmp_y    = total(zdat[nomask,*],1,/double,/nan) $
                       else $
                          tmp_y    = total(zdat,1,/double,/nan)    
                       dm_gaussfit,tmp_y,params=params,ny=ny,fitnotgood=fitnotgood,debug=debug
                       if fitnotgood then begin
                          ok = dialog_message(["Can't locate the elastic peak from the first file.","All time channels are used instead."],$
                               dialog_parent=parent)
                       endif else begin
                          all_tch = 0     ;fitting is good
                          tchans  = params[1]-fix(2.5*params[2])+indgen(fix(5.0*params[2])+1)
                          index   = where(tchans lt 0, count)
                          if count ne 0 then tchans[index] = tchans[index]+ny
                          index   = where(tchans ge ny,count)
                          if count ne 0 then tchans[index] = tchans[index]-ny
                          emin    = min(emean[tchans],max=emax)
                          e_range = [emin,emax]
                       endelse
                       end
                   2:  begin   ;specified energy range
                       if n_elements(e_range) eq 2 then $
                          if total(finite(e_range)) eq 2 then begin
                             tchans = where((emean ge e_range[0]) and (emean le e_range[1]),count)
                             if count ne 0 then begin
                                all_tch = 0
                                if arg_present(kfactor) then begin
                                   if self.instrgeom eq 0 then begin   ;direct geometry
                                      kfactor = sqrt(eief/(eief-mean(e_range))) ;ki/kf_avg
                                   endif else begin   ;inverse geometry
                                      kfactor = sqrt((eief+mean(e_range))/eief) ;ki_avg/kf
                                   endelse
                                endif
                                break
                             endif
                          endif
                       ok = dialog_message(['The energy range specified is invalid.','All time channels are used instead.'],$
                                dialog_parent=parent)
                       end
                   else:
              endcase
              if all_tch eq 1 then begin
                 tchans = indgen(zdim[1])
                 emin   = min(emean,max=emax)
                 e_range = [emin,emax]
              endif
              ewid = ewid[tchans]
              nitem = (avgsum eq 1)?1.0:(1.0>(n_elements(ewid)))
              if keyword_set(eint_yn) then factor=1.0 else factor = 1.0/nitem
              if (diffetyp eq 1) and arg_present(kfactor) then begin
                 if self.instrgeom eq 0 then begin   ;direct geometry
                    kfactor = (sqrt(eief/(eief-emean)))[tchans]
                 endif else begin   ;inverse geometry
                    kfactor = (sqrt((eief+emean)/eief))[tchans]
                 endelse
              endif
           endif else begin
              qty    = 0.0
              dqty   = 0.0
              weight = 0.0
           endelse
        endif

        if sch_ead then begin          ;elastic peak adjustment
           erotate = 0
           i_el = dm_findbetween(emean,0)
           if (~ finite(i_el)) then $
              ok = dialog_message('No E=0 in the original data. Elastic peak position adjustment is skipped.',dialog_parent=parent,title='Warning: '+open_file[i]) 
           if eadjust eq 1 then begin
              if n_elements(nomask) ne 0 then $
                 tmp_y    = total(zdat[nomask,*],1,/double,/nan) $
              else $
                 tmp_y    = total(zdat,1,/double,/nan)
              dm_gaussfit,tmp_y,params=params,fitnotgood=fitnotgood,debug=debug
              if fitnotgood then begin;somehow the fitting is not right use all time channels
                 ok = dialog_message(["Can't locate the elastic peak.","Elastic peak adjustment is skipped."],$
                      dialog_parent=parent,title='Warning: '+open_file[i])
                 i_eadj = !values.f_nan
              endif else begin
                 i_eadj = fix(params[1]+0.5)
              endelse
           endif else if eadjust eq 2 then begin
              i_eadj = dm_findbetween(emean,epadjust)
              if (~ finite(i_eadj)) then $
                 ok = dialog_message('No E='+dm_to_string(epadjust)+' in the original data. Elastic peak position adjustment is skipped.',$
                      dialog_parent=parent,title='Warning: '+open_file[i])
           endif
           if finite(i_el) and finite(i_eadj) then erotate = fix(i_el-i_eadj) $
           else tmp= temporary(erotate)  ;destroy erotate
           if keyword_set(epadcheckfirst) and n_elements(erotate) ne 0 then sch_ead = 0b   
        endif
        if n_elements(erotate) ne 0 then begin
           zdat  = shift(zdat,0,erotate)
           dzdat = shift(dzdat,0,erotate)
        endif

        ;add or append data
        if keyword_set(diffuse) then begin
           zdat  = zdat[*,tchans]
           dzdat = dzdat[*,tchans] 
           if diffetyp eq 0 then begin
              if keyword_set(eint_yn) then begin
                 for j=0L,n_elements(zdat[*,0])-1 do begin
                     zdat[j,*] = zdat[j,*]*ewid
                     dzdat[j,*] = dzdat[j,*]*ewid
                 endfor
              endif
              if i eq 0 then begin
                 qty  = total(zdat,2,/nan)*factor
                 dqty = sqrt(total(dzdat^2,2,/nan))*factor
              endif else begin
                 qty  = [[qty],[total(zdat,2,/nan)*factor]]
                 dqty = [[dqty],[sqrt(total(dzdat^2,2,/nan))*factor]]
              endelse
           endif else begin
              if i eq 0 then begin
                 qty  = transpose(zdat)
                 dqty = transpose(dzdat)
              endif else begin
                 qty  = [[[qty]],[[transpose(zdat)]]]
                 dqty = [[[dqty]],[[transpose(dzdat)]]]
              endelse
           endelse
        endif else begin
           dzdat = dzdat^2  ;error bar
           ;handle maskings
           ii = 0L
           while(total(finite(qty[*,ii])) eq 0) do ii=ii+1    ;in case the first time channel is masked
           ind = where(~finite(qty[*,ii]),count)
           if count ne 0 then begin
              qty[ind,*]  = zdat[ind,*]*weight
              dqty[ind,*] = dzdat[ind,*]*((weight+myweight[i])^2-myweight[i]^2)
           endif
           ii = 0L
           while(total(finite(zdat[*,ii])) eq 0) do ii=ii+1   ;in case the first time channel is masked
           mask = where(~finite(zdat[*,ii]),mskcount)
           if (mskcount gt 0) and (i gt 0) then begin
              zdat[mask,*]  = qty[mask,*]/weight
              dzdat[mask,*] = dqty[mask,*]*((weight+myweight[i])^2-weight^2)/(weight^2*myweight[i]^2) 
           endif
           qty    = temporary(qty)+temporary(zdat)*myweight[i]
           dqty   = temporary(dqty)+temporary(dzdat)*(myweight[i]^2)
           weight = temporary(weight)+myweight[i]
        endelse
        n_read = n_read+1
        if (~ obj_valid(mesg)) then break
    endfor
 
    if (n_read ne n_files) and (n_elements(ang) ne 0) then ang = ang[0:n_read-1]

    if (~ keyword_set(diffuse)) then begin
       if arg_present(kfactor) then begin
          if self.instrgeom eq 0 then begin   ;direct geometry
             kfactor = sqrt(eief/(eief-emean))
          endif else begin   ;inverse geometry
             kfactor = sqrt((eief+emean)/eief)
          endelse
       endif
       if weight eq 0 then weight = 1.0
       qty  = qty/weight
       dqty = sqrt(dqty)/weight
       qty  = transpose(qty)   ;->[nen,ndet]
       dqty = transpose(dqty)  ;->[nen,ndet]
    endif else begin
       if diffetyp eq 1 then emean=emean[tchans] $
       else begin
          if t_chan eq 2 then emean=mean(emean[tchans]) $
          else emean = 0.0
       endelse
       if keyword_set(background) and (n_elements(weight) gt 1) then begin    ;sum the files
          ind  = (size(qty))[0]
          totw = total(weight)
          if totw le 0 then totw = 1.0
          weight = weight/totw
          for i=0,n_elements(weight)-1 do begin
              qty[*,i]  = qty[*,i]*weight[i]
              dqty[*,i] = (dqty[*,i]*weight[i])^2
          endfor
          qty    = total(qty,ind,/nan)          ;not bullet_proof for different masking
          dqty   = sqrt(total(dqty,ind,/nan))   
          weight = totw
       endif
    endelse
 
    ;read the phx file
    if n_elements(phxfile) ne 0 then begin
       if file_test(phxfile,/regular) then begin
          if obj_valid(mesg) then mesg->update,message='Loading detector file...' $
          else mesg=obj_new('dm_progress',title=title,message='Loading detector file...',group_leader=parent)
          self->dm_load_phx,phxfile=phxfile,tth=tth,psi=psi,dtth=dtth,dpsi=dpsi
       endif else begin
          if phxfile ne '' then ok = dialog_message("Can't open "+phxfile+'. Please check again.',/error,dialog_parent=parent)
       endelse
    endif
    if obj_valid(mesg) then obj_destroy,mesg
    cd,current
end

;the following procedure is for loading detector dependent background file for dcs_mslice
pro dcs_mslice::dm_load_spe_detbackground,open_path,open_file,error=error,qty=qty,dqty=dqty,parent=parent
    self->dm_load_spe,open_path,open_file,qty=qty,dqty=dqty,t_chan=2,e_range=self.bgeran,avgsum=0,$
        title='loading background file...',parent=parent,error=error,/diffuse,/background,diffetyp=0
end

;the following procedure is for loading time channel background file for dcs_mslice
pro dcs_mslice::dm_load_spe_tchanbackground,open_path,open_file,error=error,qty=qty,dqty=dqty,$
    parent=parent,emean=emean
    self->dm_load_spe,open_path,open_file,qty=qty,dqty=dqty,t_chan=2,e_range=self.dcor_eqran[0:1],$
        avgsum=0,title='loading background file...',parent=parent,error=error,/diffuse,/background,$
        diffetyp=1,emean=emean,eief=eief,phxfile=phxfile
end