; $Id$
;#######################################################################
;
; NAME:
;  dm_load_phx
;
; PURPOSE:
;  load phx file
;
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  June, 2010
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

; parameter:
;   file:   input file name
; keywords:
;   error:  error flag
;   tth:    two theta [ndet]
;   psi:    psi       [ndet]
;   dtth:
;   dpsi:
pro dm_load_phx,file,tth=tth,psi=psi,dtth=dtth,dpsi=dpsi,error=error,group_leader=group_leader
    ;check if the file is compressed
    if strlowcase(strmid(file,2,3,/reverse_offset)) eq '.gz' then compress=1 else compress=0

    openr,unit,file,/get_lun,error=error,compress=compress
    if error ne 0 then begin   ;read phx file
       ok = dialog_message("Can't open "+file,/error,dialog_parent=group_leader)
       return
    endif

    ;catch and clear possible io errors
    catch, myerror
    if myerror ne 0 then begin
       if keyword_set(error) then begin
          catch,/cancel
          if n_elements(unit) ne 0 then free_lun,unit
          ok = dialog_message(!error_state.msg,/error)
          return
       endif else begin
          ;try one more time, using fixed width format
          error = 1b
          if n_elements(unit) eq 0 then return $
          else point_lun,unit,0 ;reset the pointer to start
          tmp1 = ' '
          readf,unit,tmp1       ;first line 
          readf,unit,tmp1       ;second line
          format = '(7f'+dm_to_string(strlen(tmp1)/7)+',"/")'
          point_lun,unit,0      ;reset the pointer to start
       endelse
    end

    ndet = 0L
    readf,unit,ndet

    tmp  = fltarr(7,ndet)
    if keyword_set(error) then $    
       readf,unit,tmp,format=format $
    else $
       readf,unit,tmp
    tth  = reform(tmp[2,*])
    psi  = reform(tmp[3,*])
    dtth = reform(tmp[4,*])
    dpsi = reform(tmp[5,*])

    free_lun,unit
    error = 0b
end

;this procedure reads a phx file for dcs_mslice program and store the info into self.detPosPtr
;keywords:
;   phxfile:            the phx file name, if arg_present(phxfile) then the program will ask for one if not specified
;   tth,psi,dtth,dpsi:  if set, used as backup data, and check if phx file is valid by comparing
;                       number of detectors
pro dcs_mslice::dm_load_phx,phxfile=phxfile,tth=tth,psi=psi,dtth=dtth,dpsi=dpsi
    if arg_present(phxfile) then begin
       if n_elements(phxfile) eq 0 then begin
          phxfile = dm_choose_file('phx',dialog_parent=self.tlb,/read,title=$
               'Please choose a detector PHX file. Press cancel to skip.',path=self.dirs[0])
       endif
       error = 1b
       if file_test(phxfile,/regular) ne 0 then begin
          dm_load_phx,phxfile,tth=newtth,psi=newpsi,dtth=newdtth,dpsi=newdpsi,error=error,group_leader=self.tlb
       endif
       if (~ keyword_set(error)) then begin    ;newtth etc. are available
          replace = 1b
          if n_elements(tth) ne 0 then begin
             if n_elements(tth) ne n_elements(newtth) then begin
                replace = 0b
                ok = dialog_message([phxfile+' is incompatible with the SPE file.',$
                      "The PHX file is discarded."],/error,dialog_parent=self.tlb)
             endif
          endif
          if replace then begin
             tth  = temporary(newtth)
             psi  = temporary(newpsi)
             dtth = temporary(newdtth)
             dpsi = temporary(newdpsi)
          endif
       endif
    endif

    if n_elements(tth) eq 0 then return ;no data available
    ptr_free,self.detPosPtr
    self.detPosPtr = ptr_new({two_theta:temporary(tth),dtwo_theta:temporary(dtth),$
                     psi:temporary(psi),dpsi:temporary(dpsi)})
end