; $Id: $
;#######################################################################
;
; NAME:
;  dm_filesel
;
; PURPOSE:
;  This program defines a file selction object tailored for dcs_mslice program.                                                         
;
; CATEGORY:
;  general
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  July, 2025
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

; Usage:  filesel=obj_new('dm_filesel',parent=parent,path=path,$   
;           multi=multi,write=write,filter=filter,frame=frame,$    
;           ysize=ysize,ypad=ypad,xsize=xsize,group_leader=group_leader)           
;           
pro dm_filesel_Exit,tlb
    ;leave the job to obj_destroy
end

;widget event handler
pro dm_filesel_event,event
    widget_control,event.handler,get_uvalue=self
    if obj_valid(self) then self->event,event
end

;workaround of serveral reading problems with old icp data files, returns a fixedfile name in keyword if actions taken, remember to delete the fixed file afterward
pro dm_fix_tasfile,filename,fixedfile=fixedfile,group_leader=group_leader,bufferdir=bufferdir
    if n_elements(fixedfile) ne 0 then tmp = temporary(fixedfile)
    nlines = file_lines(filename)
    text_data = strarr(nlines)
    openr,lun,filename,/get_lun
    readf,lun,text_data
    free_lun,lun,/force
    is_asd = stregex(filename,'\.(bt4|ng5)',/boolean,/fold_case)  ;could have area sensitive detector data
    trim_text = strtrim(text_data,2)
    ind0 = where(strlen(trim_text) gt 0,cnt00)
    if cnt00 gt 0 then text_data = text_data[ind0] else return
    if is_asd then trim_text = trim_text[ind0] else trim_text = 0
    if stregex(text_data[0],'^ *# *(ice|nice)',/boolean,/fold_case) then return   ;not icp file
    remstrs = '^ *(fit|#asd)'  ;fit line, #ASD line
    ind0 = where(stregex(text_data,remstrs,/boolean,/fold_case),cnt0,complement=ind1,ncomplement=cnt1)  ;check and remove the lines that need to be removed
    if cnt0 gt 0 then begin
       if cnt1 gt 0 then text_data = text_data[ind1] else return
       if is_asd then trim_text = trim_text[ind1]
       fixed = 1b
    endif
    ind0 = where(stregex(text_data,'intensity *$',/boolean,/fold_case),cnt0)     ;check and add the missing date string after 'Intensity'
    if cnt0 eq 1 then begin
       text_data[ind0] = text_data[ind0]+'   '+systime()
       fixed = 1b
    endif
    ind0 = where(stregex(text_data,'# +det',/boolean,/fold_case),cnt0)           ;check and replace '# Det' with '#Det'
    if cnt0 eq 1 then begin
       tmp = stregex(text_data[ind0],'(.*)# +det(.*)',/fold_case,/extract,/subexpr)
       text_data[ind0] = tmp[1]+'#Det'+tmp[2]
       fixed = 1b
    endif
    if stregex(text_data[0],"'T'",/boolean,/fold_case) then begin                ;check if 'T' scan type, change it to 'Q'
       tmp = text_data[0]
       strput,tmp,"'Q'",strpos(strlowcase(tmp),"'t'")
       text_data[0] = tmp
       fixed = 1b
    endif
    ind0 = where(stregex(text_data,'xstal +orientation',/boolean,/fold_case),cnt0)
    if cnt0 gt 0 then begin                                                      ;check for conflicting crystal orientation
       tmp = stregex(text_data[ind0[0]-1]+' ','^(.* +)'+strjoin(replicate('([^ ]+ +)',7))+'$',/subexpr,/extract)
       if (strlen(tmp[0]) gt 0) && (total(abs(dm_to_number(tmp[2:4])-dm_to_number(tmp[6:8]))) eq 0) then begin
          text_data[ind0[0]-1] = tmp[1]+strjoin(['1','0','0'],' ')+'   '+tmp[5]+strjoin(['0','0','1'],' ')
          fixed = 1b
       endif
    endif
    ind0 = where(finite(dm_to_number(text_data)),complement=ind1,ncomplement=cnt1,cnt0)
    if (cnt0 gt 0) and (cnt1 gt 0) then begin
       ncol = n_elements(strsplit(text_data[ind1[cnt1-1]],' ',/extract))
       pos0 = stregex(text_data[ind1[cnt1-1]]+' ',strjoin(replicate('([^ ]+ +)',ncol)),/subexpr,length=len0)
       ind2 = where(ind0 gt ind1[cnt1-1],cnt0)
       ind0 = ind0[ind2]  ;actual data block
       bad_indx = -1
       for i=0,cnt0-1 do begin  ;check and fix no empty space between numbers issue
           tmp = strsplit(text_data[ind0[i]],' ',/extract,count=ncol1)
           if abs(ncol1-ncol) gt 1 then break   ;To prevent unnecessary check. Can't happen at more than 1 place? 
           if ncol-ncol1 eq 1 then bad_indx = [bad_indx,ind0[i]]
       endfor
       for i=1,n_elements(bad_indx)-1 do begin
           pos1 = stregex(text_data[bad_indx[i]],strjoin(replicate('( *[^ ]+)',ncol-1)),/subexpr,length=len1)
           if len1[0] gt 0 then begin
              for j=1,n_elements(len1)-1 do begin                       
                  if (pos1[j]+len1[j] gt pos0[j+1]) and (strlen(strtrim(strmid(text_data[bad_indx[i]],pos1[j],len1[j]),2)) gt 10) then begin ;avoid over correction
                     tmpstr = strmid(text_data[bad_indx[i]],(j-1)*10,10)+' '+strmid(text_data[bad_indx[i]],j*10,10)+' '  ;10-character width, might not hold
                     text_data[bad_indx[i]] = ((j eq 1)?'':strmid(text_data[bad_indx[i]],pos1[1],total(len1[1:(j-1)])))+tmpstr+strmid(text_data[bad_indx[i]],pos1[j+1])
                     spaceadded = 1b & fixed = 1b
                     break 
                  end
              endfor
           endif
       endfor
       if is_asd then begin  ;check and remove extra data block
          ind1 = where(stregex(trim_text[ind0],'( |'+string(9b)+')',/boolean),cnt1,ncomplement=cnt2)
          if (cnt2 gt 0) and (cnt1 gt 0) then begin
             text_data = [text_data[0:ind0[0]-1],text_data[ind0[ind1]]]
             fixed = 1b
          endif
          trim_text = ''
       endif
       if keyword_set(spaceadded) and (n_elements(group_leader) ne 0) then $
          ok = dialog_message('Spaces have been inserted to separate numbers. The data might be incorrect.',dialog_parent=group_leader,/center)
    endif else if cnt1 eq 0 then begin  ;missing header, add a header line
       ncol = n_elements(strsplit(text_data[0],' ',/extract))
       if (n_elements(group_leader) ne 0) and (ncol le 3) then begin
          id = dm_dialog_input('column '+dm_to_string(indgen(ncol)+1)+':',title='Enter Column Names',default=(['','','Intensity'])[(ncol eq 2):ncol-1],$
               info='The header is missing in the file. Please enter the column names.',cancel=cancel,dialog_parent=group_leader)
          if ~keyword_set(cancel) && (strlen(id[0]) gt 0) then begin
             if strlen(id[ncol-1]) eq 0 then id[ncol-1] = 'Intensity'
             if strlen(id[ncol-2]) eq 0 then id[ncol-2] = 'min'
          endif else tmp = temporary(id)
       endif
       text_data = [strjoin([(n_elements(id) gt 0)?id:['tmp_'+dm_to_string(indgen(ncol-1)+1),'Intensity'],systime()],'   '),text_data]
       fixed = 1b
    endif
    if keyword_set(fixed) then begin
       if n_elements(bufferdir) eq 0 then bufferdir = dm_define_pointer(/gettempdir)
       fixedfile = bufferdir+dm_define_pointer(/getpathsep)+'fixed_'+file_basename(filename)
       openw,lun,fixedfile,/get_lun,error=openerr
       if openerr eq 0 then begin
          for i=0,n_elements(text_data)-1 do printf,lun,text_data[i]
          free_lun,lun
       endif else tmp = temporary(fixedfile)
    endif
end

; Check if at top of directory, copied from cw_filesel.pro
function dm_filesel::atTop, path, ftp=ftp
    compile_opt hidden, strictarr

    if keyword_set(ftp) then return, (path eq '/pub/ncnrdata/') or (path eq '/') 
    
    CASE !version.os_family OF
        'Windows': atTop = (STRMID(path, STRLEN(path)-2) EQ ':\')
        else:  atTop = (path EQ '/')
    ENDCASE
    RETURN, atTop
end

pro dm_filesel::changesort
    self.sortfile = ~self.sortfile
    self->refresh
    if (self.group_leader ne 0) and self.sortevent then widget_control,self.group_leader,send_event={dm_filesel_sortfile,ID:self.tlb,TOP:self.group_leader,HANDLER:self.tlb,object:self},/no_copy
end

;*******************************************************************
;clear the file selection
;syntax:
;   filesel->clear_file
;arguments:
;   None
;keywords:
;   bottom: if set, go to the bottom of the file after clearing the selection
;   fileloc: if present, will go to the specified file name location
;*******************************************************************
pro dm_filesel::clear_file,bottom=bottom,fileloc=fileloc,highlight=highlight
    if ptr_valid(self.file) then begin
       ptr_free,self.file
       widget_control,self.fileList,set_list_select=-1
       if self.write then $
          self->my_widget_control,'saveFile',set_value=''
    endif
    if keyword_set(bottom) or (n_elements(fileloc) eq 1) then begin
       if ptr_valid(self.dFiles) then nFiles=n_elements(*self.dFiles)    $
       else nFiles=0
       visn = widget_info(self.fileList,/list_num_visible)
       if (n_elements(fileloc) eq 1) and (nFiles gt 0) then begin
          top = where(strlowcase(*self.dFiles) eq strlowcase(fileloc)) 
          if keyword_set(highlight) then begin
             highlight_extra = {set_list_select:top}
             top = top-visn+1
          endif
       endif else $
          top = nFiles-visn+1
       widget_control,self.fileList,set_list_top=0>(top),_extra=highlight_extra
    endif
    cd,self.root
end

;returns the file basename without the extension, allows for extension with different name but same length, pos and extra keyword are for strpos.
function dm_filesel::filebasename,filename,extension=extension,pos=pos,_extra=extra
    if n_elements(extension) eq 0 then begin
       if n_elements(pos) ne 0 then extension = strmid(filename[0],strpos(filename[0],'.',pos,_extra=extra)) $
       else extension = strmid(filename[0],strpos(filename[0],'.',_extra=extra))
    endif
    n_f = n_elements(filename)
    l_e = strlen(extension[0])
    l_f = strlen(filename) 
    tmp = strarr(n_f)
    for i=0,n_f-1 do tmp[i] = strmid(filename[i],0,l_f[i]-l_e)
    return,(n_f eq 1)?tmp[0]:tmp
end

;read dcs file header info
pro dm_filesel::dcsfileheader,name
    catch, anyerror
    if anyerror ne 0 then begin
       catch,/cancel
       ok = dialog_message(!error_state.msg ,/error,dialog_parent=self.tlb,/center)
       if ptr_valid(davePtr) then heap_free,davePtr
       return
    end
    files   = self.path
    if strmid(files,0,1,/reverse_offset) ne self.pathsep[obj_valid(self.ftpobj)] then files = files+self.pathsep[obj_valid(self.ftpobj)]
    files   = files+strtrim(*self.file,2)
    davePtr = dm_define_pointer() & omithisto=2 & printinfo=0 & filetype='binary'
    shutter = ['unknown','closed','open']
    info    = ''
    lines   = ''
    winfo   = fltarr(6)
    if name eq 'duration' then totduration = 0d
    for i=0,n_elements(files)-1 do begin
        line = self->filebasename((*self.file)[i])+' :   '
        if obj_valid(self.ftpobj) then begin
           ok = self.ftpobj->GetFileContent(files[i],localfilename=self.bufferdir+self.pathsep[0]+'ftptmp'+(*self.file)[i])
           files[i] = self.bufferdir+self.pathsep[0]+'ftptmp'+(*self.file)[i]
        endif
        dcs_readdatafile,files[i],omithisto,printinfo,filetype,davePtr=davePtr
        if obj_valid(self.ftpobj) then file_delete,files[i],/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET
        this = dm_locate_datastrptr(davePtr)
        if name eq 'temperature' then begin
           temp_sample  = (*(*this).specificPtr).temp_sample
           temp_control = (*(*this).specificPtr).temp_control
           postemp = where(temp_sample gt 0,count)
           if count ne 0 then begin
              temp_sample  = temp_sample[postemp]
              temp_control = temp_control[postemp]
           endif
        endif
        case name of
            'comment':        begin
                              line = line+strtrim((*(*this).specificPtr).comments,2)
                              if stregex(line,'T ?= *$',/boolean) then $      ;patch the comment ending with "T=" with the temerature set point
                                 line = self->formatlinestring(line,(*(*this).specificPtr).temp_setpoint,/temperature)
                              end
            'wavelength':     begin
                              lines = [lines,line]
                              winfo = [[winfo],[float((*(*this).specificPtr).ch_wl),(*(*this).specificPtr).ch_res,float((*(*this).specificPtr).ch_ms),$
                                       (*(*this).specificPtr).ch_srmode,(*(*this).specificPtr).ch_srdenom,float((*(*this).specificPtr).tsdmin)]]
                              end
            'duration':       begin
                              duration1 = (*(*this).specificPtr).duration
                              duration2 = dm_to_number((*(*this).specificPtr).stop_date,/epoch)-dm_to_number((*(*this).specificPtr).start_date,/epoch)
                              if (*(*this).specificPtr).polanal then duration = duration1 else duration = (abs(duration1-duration2) gt 0.1*duration1)?(duration2):(duration1)
                              totduration = totduration+duration
                              line = self->formatlinestring(line,duration,/time)
                              if (abs(duration1-duration2) gt 0.1*duration1) then line = self->formatlinestring(line+'  (duration field=',duration1,/time)+')'
                              end
            'temperature':    begin
                              line = self->formatlinestring(line+' [set]  ',(*(*this).specificPtr).temp_setpoint,/temperature)
                              line = self->formatlinestring(line+'  [ctrl]  ',temp_control,/temperature)
                              line = self->formatlinestring(line+'  [sample]  ',temp_sample,/temperature)
                              end
            'magfield':       begin
                              field = strupcase(tag_names(*(*this).specificPtr))
                              loc = where(field eq 'FIELD_SAMPLE',count)
                              if count eq 0 then begin
                                 line =  line+' N/A'
                              endif else begin
                                 if finite((*(*this).specificPtr).field_sample,/nan) then line =  line+' N/A' $
                                 else begin
                                    line = self->formatlinestring(line+' [sample]  ',(*(*this).specificPtr).field_sample,'magnetic field',res=4,unit='T')
                                    line = self->formatlinestring(line+'   [setpoint]  ',(*(*this).specificPtr).field_setpoint,'magnetic field',res=4,unit='T')
                                 endelse
                              endelse
                              end
            'shutter state':  line = line+shutter[(*(*this).specificPtr).shutter_stat+1]
            'sample position':begin
                              line = line+'A2='+dm_to_string((*(*this).specificPtr).motor_pos[1],resolution=2)+string('b0'XB)+'  x='+$
                                 dm_to_string((*(*this).specificPtr).motor_pos[3],resolution=2)+'  y='+$
                                 dm_to_string((*(*this).specificPtr).motor_pos[4],resolution=2)+'  z='+$
                                 dm_to_string((*(*this).specificPtr).motor_pos[2],resolution=2)+'  xtilt='+$
                                 dm_to_string((*(*this).specificPtr).motor_pos[5],resolution=2)+string('b0'xb)+'  ytilt='+$
                                 dm_to_string((*(*this).specificPtr).motor_pos[6],resolution=2)+string('b0'xb)
                              end
            'starting time':  line = line+(*(*this).specificPtr).start_date
            'polarization reduced data': line = line+(['No','Yes'])[(*(*this).specificPtr).polanal]
            else:
        endcase
        info = [info,line]
    endfor
    if name eq 'wavelength' then begin
       maxlength = max(self->wavlstrlen(winfo[0,1:*],/nozeroend))
       info = self->formatlinestring(lines[1:*],winfo[*,1:*],/wavelength,maxlength=maxlength)
    endif else info = info[1:*]
    if (name eq 'duration') and (n_elements(info) gt 1) then begin
       line = ''
       for i = 0, max(strlen(strtrim(*self.file,2)))-6 do line = line+' '
       line = line+'total :  '
       line = self->formatlinestring(line,totduration,/time)
       info = [info,'',line]
    endif
    heap_free,davePtr
    if n_elements(info) gt self.maxline then begin
       if widget_info(self.displayid,/valid_id) then widget_control,self.displayid,/destroy
       xdisplayfile,text=info,/editable,group=self.group_leader,title='DCS file info: ['+name+']',done_button='Exit',/grow_to_screen,return_id=wid
       dm_center_kid,wid,self.group_leader
       self.displayid = wid 
    endif else begin
       ok = dialog_message(info,/info,dialog_parent=self.group_leader,title='DCS file info: ['+name+']',/center)
    endelse
end

;read file header info of INX,NXSPE,SPE,WAND files
pro dm_filesel::fileheader,instr,name,field=field,tmpfile=tmpfile
    WIDGET_CONTROL,/HOURGLASS
    if n_params() ne 2 then return
    instr = strlowcase(instr[0]) & name = strlowcase(name[0])
    files = self.path
    if strmid(files,0,1,/reverse_offset) ne self.pathsep[obj_valid(self.ftpobj)] then files = files+self.pathsep[obj_valid(self.ftpobj)]
    files = files+strtrim(*self.file,2)
    if name eq 'duration' then totduration = [0d,0d]
    tmp = where(['comment','filter&collimator','starting time','histogram','focus'] eq name,count) & firstlineonly = (count ne 0)  ;no need to read the whole file
    tmp = where(['macs sample position','macs histogram'] eq instr+' '+name,count)  ;check if macs rg0 file exists
    if (count gt 0) and ~obj_valid(self.ftpobj) then begin
       tmp = file_search(self.path,'*.rg0',count=count)
       no_rg0 = (count eq 0)
       if obj_isa(self.pobj,'dcs_mslice') then begin
          self.pobj->getproperty,macshistomode=macshistomode
       endif
    endif else no_rg0 = 1b
    for i=0, n_elements(files)-1 do begin
        if stregex((*self.file)[i],'^.+\.[a-z0-9]{2,4}\.[a-z0-9]{2,4}$',/fold_case,/boolean) then extension=strmid((*self.file)[i],strpos((*self.file)[i],'.',5,/reverse_search,/reverse_offset)) $
        else extension = strmid((*self.file)[i],strpos((*self.file)[i],'.',/reverse_search))
        line = self->filebasename((*self.file)[i],extension=extension)+' :   '
        if obj_valid(self.ftpobj) then begin
           ok = self.ftpobj->GetFileContent(files[i],localfilename=self.bufferdir+self.pathsep[0]+'ftptmp_'+(*self.file)[i])
           files[i] = self.bufferdir+self.pathsep[0]+'ftptmp_'+(*self.file)[i]
        endif
        case instr of 
             'macs':  dm_load_macs,files[i],data,colinfo,header=header,maspacing=maspacing,comment=comment,error=error,all_neg=all_neg,group_leader=self.tlb,temperatureunit=temperatureunit,$
                      flipstring=flipstring,histodata=(name eq 'histogram'),is_histogram=is_histogram,n_chan=n_chan,firstlineonly=firstlineonly,t_mcfx=t_mcfx,niceinfo=niceinfo,hfocus=hfocus,$
                      vfocus=vfocus,/nomessage,noadjust=(name eq 'specify')
             'nxspe': dm_load_nxspe,files[i],header=header,error=error,angi=psi,group_leader=self.tlb
             'spe':   dm_load_spe,files[i],header=header,error=error,group_leader=self.tlb
             'wand':  dm_load_wand,files[i],data=data,psi=psi,temperature=temperature,error=error,group_leader=self.tlb,/headeronly
             'inx':   dm_load_inx,files[i],ei=ei,comment=comment,temperature=temperature,tchannel=tchannel,ydat=ydat,/headeronly,error=error,group_leader=self.tlb,inxerev=(~self.inx_erev)
             'tas':   begin
                      tmpobj = obj_new('dtas_data',filename=files[i])
                      ;fix several know bugs
                      if ~obj_valid(tmpobj) then begin
                         dm_fix_tasfile,files[i],fixedfile=tmpfile,bufferdir=self.bufferdir,group_leader=(self.group_leader eq 0)?self.tlb:self.group_leader
                         if n_elements(tmpfile) ne 0 then begin
                            tmpobj = obj_new('dtas_data',filename=tmpfile)
                            file_delete,tmpfile,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET
                         endif
                      endif
                      if obj_valid(tmpobj) then ret = tmpobj->get_property(isIce=isIce,var_names=var_names,data=data,header=header) else error = 1b
                      end   
             else:    error = 1b
        endcase
        if name eq 'viewfile' then begin
           if (!VERSION.OS_FAMILY ne 'Windows') then begin 
              xdisplayfile,files[i],group=self.group_leader,title=(*self.file)[i],done_button='Exit',/grow_to_screen,wtext=tid
              if self.idlversion ge 5.6 then scr_size = get_screen_size() else device,get_screen_size=scr_size
              widget_control,tid,scr_xsize=scr_size[0]-50
           endif else begin ;use wordpad in windows since widget_text in xdisplayfile has a 1024-character line limit
              spawn,'write.exe "'+files[i]+'"',/noshell
              if obj_valid(self.ftpobj) then wait,2  ;allow time for the file to be open before it is deleted
           endelse
        endif
        if obj_valid(self.ftpobj) then file_delete,files[i],/allow_nonexistent,/noexpand_path,/quiet
        if keyword_set(temporary(error)) then continue
        if (instr eq 'macs') and (n_elements(data) eq 0) then begin
           if keyword_set(all_neg) then $
              line = line+' (All det counts are -1. Do not use.)' $
           else $
              line = line+' (empty data)'
        endif else begin
           case instr+' '+name of
            'macs comment':             begin
                                        if n_elements(comment) eq 0 then begin
                                           tmp = stregex(header,'# *Comment(.*)',/fold_case,/subexpr,length=len)
                                           ind = (where(len[1,*] gt 0,count))[0]
                                           if count gt 0 then begin
                                              comment = strmid(header[ind],tmp[1,ind],len[1,ind])
                                           endif else begin
                                              comment=''
                                           endelse
                                        endif
                                        line = line+comment
                                        if stregex(line,'T ?= *$',/boolean) then begin      ;patch the comment ending with "T=" with the temerature set point
                                           indts = (where(colinfo eq 'temperaturesetpoint',count))[0] 
                                           if count ne 0 then line = self->formatlinestring(line,data[0,indts],/temperature)
                                        endif
                                        if stregex(line,'B ?= *$',/boolean) then begin      ;patch the comment ending with "B=" with the magnetic field
                                           indhs = (where(colinfo eq 'magfield',count))[0]
                                           if count eq 0 then indhs = (where(colinfo eq 'hfield',count))[0]
                                           if count ne 0 then line = self->formatlinestring(line,data[0,indhs],res=4,unit='T')
                                        endif
                                        end
            'macs energy':              begin
                                        Eiexist = 0b & Efexist = 0b
                                        indA2 = (where(colinfo eq 'a2',count))[0]
                                        if count ne 0 then begin
                                           if total(finite(data[*,indA2],/nan)) eq 0 then begin
                                              Ei = 2*maspacing[0]*sin(!dtor*data[*,indA2]/2)
                                              Ei = 81.8042/(Ei^2) & Eiexist = 1b
                                              line = self->formatlinestring(line+' Ei=',Ei,'Ei',/energy)
                                           endif else line = line+' Ei=N/A'
                                        endif else line = line+' Ei=N/A'
                                        indA5 = (where(colinfo eq 'a5',count))[0]
                                        if count ne 0 then begin
                                           ind_ana = lonarr(20)
                                           for j=1,20 do ind_ana[j-1] = (where(colinfo eq 'analyzertheta'+string(j,format='(i02)')))[0]
                                           if finite(data[0,indA5],/nan) then begin
                                              tmp_x = reform(data[0,ind_ana]) & tmp_dy = fltarr(20)+1
                                              dm_step_bin,0.5,tmp_x,yerr=tmp_dy
                                              tmp = min(tmp_dy,ind_tmp)
                                              tmp = min(abs(data[0,ind_ana]-tmp_x[ind_tmp]),ind_tmp)
                                              indA5 = ind_ana[ind_tmp]
                                           endif
                                           ;some analyzers may be off, needs to check
                                           for j=0L,n_elements(data[*,0])-1 do begin
                                               min_da5 = min(abs(data[j,indA5]-data[j,ind_ana]))
                                               bad_id  = where(abs(data[j,indA5]-data[j,ind_ana]) gt min_da5+0.5,badcount,complement=good_id) ;more than 0.5 degree off
                                               if badcount ne 0 then data[j,indA5] = mean(data[j,ind_ana[good_id]])
                                           endfor
                                           if total(finite(data[*,indA5],/nan)) eq 0 then begin
                                              Ef = 2*maspacing[1]*sin(!dtor*data[*,indA5])
                                              Ef = 81.8042/(Ef^2) & Efexist = 1b
                                              line = self->formatlinestring(line+', Ef=',Ef,'Ef',/energy)
                                              if badcount ne 0 then begin
                                                 tmp_Ef = 2*maspacing[1]*sin(!dtor*data[*,ind_ana[bad_id]])
                                                 tmp_Ef = 81.8042/(tmp_Ef^2)
                                                 tmp_Ef = self->formatlinestring(' (Ef=',tmp_Ef,'Ef',/energy)+')'
                                                 if n_elements(baddetinfo) eq 0 then begin
                                                    baddetinfo = ['* except for detector '+dm_to_string(bad_id,sep=',')+tmp_Ef]
                                                    line       = line+'*'
                                                 endif else begin
                                                    tmp_ind    = where(stregex(baddetinfo,'except for detector '+dm_to_string(bad_id,sep=',')+' \(',/boolean),count)
                                                    if count eq 1 then for j=0,tmp_ind[0] do line = line +'*' $
                                                    else begin
                                                       tmpst = '*' & for j=0,n_elements(baddetinfo)-1 do tmpst = tmpst+'*'
                                                       baddetinfo = [baddetinfo,tmpst+' except for detector '+dm_to_string(bad_id,sep=',')+tmp_Ef]
                                                       line  = line+tmpst
                                                    endelse
                                                 endelse
                                              endif
                                           endif else line = line+', Ef=N/A'
                                        endif else line = line+', Ef=N/A'
                                        if Eiexist and Efexist then begin
                                           if ~stregex(line,'E[if]=.+to.+',/fold_case,/boolean) then begin  ;to be consistent
                                              Ei = mean(Ei) & Ef = mean(Ef)
                                           endif
                                           line = self->formatlinestring(line+', E=',Ei-Ef,'E',/energy)
                                        endif
                                        end
            'macs bt7 energy':          begin
                                        Eiexist = 0b & Efexist = 0b
                                        indEi = (where(colinfo eq 'ei',count))[0]
                                        if count ne 0 then begin
                                           if total(finite(data[*,indEi],/nan)) eq 0 then begin
                                              Ei = data[*,indEi] & Eiexist = 1b
                                              line = self->formatlinestring(line+' Ei=',Ei,'Ei',/energy)
                                           endif else begin
                                              indA2 = (where(colinfo eq 'a2',count))[0]
                                              if count ne 0 then begin
                                                 if total(finite(data[*,indA2],/nan)) eq 0 then begin
                                                    Ei = 2*maspacing[0]*sin(!dtor*data[*,indA2]/2)
                                                    Ei = 81.8042/(Ei^2) & Eiexist = 1b
                                                    line = self->formatlinestring(line+' Ei=',Ei,'Ei',/energy)
                                                 endif else line = line+' Ei=N/A'
                                              endif else line = line+' Ei=N/A'
                                           endelse
                                        endif else line = line+' Ei=N/A'
                                        indEf = (where(colinfo eq 'ef',count))[0]
                                        if count ne 0 then begin
                                           if total(finite(data[*,indEf],/nan)) eq 0 then begin
                                              Ef = data[*,indEf] & Efexist = 1b
                                              line = self->formatlinestring(line+', Ef=',Ef,'Ef',/energy)
                                           endif else begin
                                              indA6 = (where(colinfo eq 'a6',count))[0]
                                              if count ne 0 then begin
                                                 if total(finite(data[*,indA6],/nan)) eq 0 then begin
                                                    Ef = 2*maspacing[0]*sin(!dtor*data[*,indA6]/2)
                                                    Ef = 81.8042/(Ef^2) & Efexist = 1b
                                                    line = self->formatlinestring(line+', Ef=',Ef,'Ef',/energy)
                                                 endif else line = line+', Ef=N/A'
                                              endif else line = line+', Ef=N/A'
                                           endelse
                                        endif else line = line+', Ef=N/A'
                                        if Eiexist and Efexist then begin
                                           if ~stregex(line,'E[if]=.+to.+',/fold_case,/boolean) then begin  ;to be consistent
                                              Ei = mean(Ei) & Ef = mean(Ef)
                                           endif
                                           line = self->formatlinestring(line+', E=',Ei-Ef,'E',/energy)
                                        endif
                                        end                            
            'macs kidney angle':        begin
                                        index = (where(colinfo eq 'kidney',count))[0]
                                        if count gt 0 then line = self->formatlinestring(line,data[*,index] ,'Kidney',/angle,tolerance=0.1)
                                        end
            'macs sample position':     begin
                                        index = (where(colinfo eq 'a3',count))[0]
                                        if count gt 0 then ng0a3 = data[*,index]
                                        if ~keyword_set(is_histogram) and ~keyword_set(no_rg0) then begin
                                           index = (where(colinfo eq 'kidney',count))[0]
                                           if count gt 0 then begin
                                              rg0a3 = 0.0
                                              for j=0,n_elements(data[*,index])-1 do begin
                                                  rg0_file = strmid(files[i],0,strpos(files[i],'.',/reverse_search))+'_'+dm_to_string(j)+'.rg0'
                                                  if (file_info(rg0_file)).exists then begin
                                                     dm_load_macshisto2d,rg0_file,theta_start=theta_start,n_theta=n_theta,theta_bin=theta_bin,a3_offset=a3_offset,/headeronly
                                                     rg0a3 = [rg0a3, theta_start+theta_bin*(0.5+indgen(n_theta))+(data[j,index])[0]+a3_offset]
                                                  endif else break
                                              endfor
                                              if n_elements(rg0a3) gt 1 then rg0a3 = rg0a3[1:*] else tmp = temporary(rg0a3)
                                           endif
                                        endif
                                        if keyword_set(macshistomode) and (n_elements(rg0a3) ne 0) then line = self->formatlinestring(line+' A3=',temporary(rg0a3),'A3',/angle,resolution=3) $
                                        else if n_elements(ng0a3) ne 0 then line = self->formatlinestring(line+' A3=',temporary(ng0a3),'A3',/angle)
                                        index = (where(colinfo eq 'smplx',count))[0]
                                        if count gt 0 then line = self->formatlinestring(line+'  x=',data[*,index],'x',tolerance=0.01)
                                        index = (where(colinfo eq 'smply',count))[0]
                                        if count gt 0 then line = self->formatlinestring(line+'  y=',data[*,index],'y',tolerance=0.01)
                                        index = (where(colinfo eq 'smplz',count))[0]
                                        if count gt 0 then line = self->formatlinestring(line+'  z=',data[*,index],'z',tolerance=0.01)
                                        index = (where(colinfo eq 'smplltilt',count))[0]
                                        if count gt 0 then line = self->formatlinestring(line+'  ltilt=',data[*,index],'',/angle)
                                        index = (where(colinfo eq 'smplutilt',count))[0]
                                        if count gt 0 then line = self->formatlinestring(line+'  utilt=',data[*,index],'',/angle)
                                        if n_elements(rg0a3) ne 0 then line = self->formatlinestring(line+' [ rg0 file: A3=',temporary(rg0a3),'A3',/angle,resolution=3)+' ]' $
                                        else if n_elements(ng0a3) ne 0 then line = self->formatlinestring(line+' [ ng0 file: A3=',temporary(ng0a3),'A3',/angle)+' ]'
                                        end
            'macs bt7 sample position': begin
                                        index = (where(colinfo eq 'a3',count))[0]
                                        if count gt 0 then line = self->formatlinestring(line+' A3=',data[*,index],'A3',/angle)
                                        index = (where(colinfo eq 'smplutrn',count))[0]
                                        if count gt 0 then line = self->formatlinestring(line+'  utrn=',data[*,index],'utrn',tolerance=0.01)
                                        index = (where(colinfo eq 'smplltrn',count))[0]
                                        if count gt 0 then line = self->formatlinestring(line+'  ltrn=',data[*,index],'ltrn',tolerance=0.01)
                                        index = (where(colinfo eq 'smplelev',count))[0]
                                        if count gt 0 then line = self->formatlinestring(line+'  elev=',data[*,index],'elev',tolerance=0.01)
                                        index = (where(colinfo eq 'smplltilt',count))[0]
                                        if count gt 0 then line = self->formatlinestring(line+'  ltilt=',data[*,index],'',/angle)
                                        index = (where(colinfo eq 'smplutilt',count))[0]
                                        if count gt 0 then line = self->formatlinestring(line+'  utilt=',data[*,index],'',/angle)
                                        end
            'macs temperature':         begin
                                        ind_samp = (where(colinfo eq 'temp',count0))[0]
                                        ind_cont = (where(colinfo eq 'temperaturecontrolreading',count1))[0]
                                        ind_setp = (where(colinfo eq 'temperaturesetpoint',count2))[0]
                                        unit = 'K'
                                        if n_elements(temperatureunit) ne 0 then begin
                                           case temperatureunit of 
                                                'celsius':     unit = string('b0'XB)+'C'
                                                'fahrenheit':  unit = string('b0'XB)+'F'
                                                else:
                                           endcase
                                        endif
                                        if count0 ne 0 then begin
                                           if count2 ne 0 then line = self->formatlinestring(line+' [set]  ',data[*,ind_setp],/temperature,unit=unit)
                                           if count1 ne 0 then line = self->formatlinestring(line+'  [ctrl]  ',data[*,ind_cont],/temperature,unit=unit)
                                           line = self->formatlinestring(line+'  [sample]  ',data[*,ind_samp],/temperature,unit=unit)
                                        endif else line = line+' N/A'
                                        end
            'macs magfield':            begin
                                        ind_field = (where(colinfo eq 'magfield',count))[0]
                                        if count eq 0 then ind_field = (where(colinfo eq 'hfield',count))[0]
                                        if count eq 0 then line = line+' N/A' $
                                        else line = self->formatlinestring(line+'B=',data[*,ind_field],'magnetic field',res=4,unit='T')
                                        end
            'macs monitor':             begin
                                        ind_time = (where(colinfo eq 'time',cnt_time))[0]
                                        ind_mon  = (where(colinfo eq 'monitor',cnt_mon))[0]
                                        index    = (where(stregex(header, '#reference',/boolean,/fold_case),count))[0]
                                        if count eq 0 then begin
                                           tmp = ''
                                           if (cnt_time eq 1) and (cnt_mon eq 1) then begin
                                              max_mon  = max(data[*,ind_mon],min=min_mon)
                                              max_time = max(data[*,ind_time],min=min_time)
                                              tmp = (['time','monitor'])[abs(max_mon-min_mon) lt abs(max_time-min_time)]
                                           endif
                                        endif else $
                                           tmp = strlowcase(strsplit(header[index],' '+string(9b),/extract))
                                        index = (where(colinfo eq tmp[n_elements(tmp)-1],count))[0]
                                        if count eq 0 then break
                                        line = line+dm_to_string(data[0,index],exponent=(tmp[n_elements(tmp)-1] eq 'monitor'))
                                        if tmp[n_elements(tmp)-1] eq 'time' then begin
                                           line = line+' Sec'
                                           if cnt_mon eq 1 then begin
                                              tmp  = mean(data[*,ind_mon])
                                              line = line+' ('+((tmp eq 0)?'':'~')+dm_to_string(tmp,/int)+')'
                                           endif
                                        endif else begin
                                           if cnt_time eq 1 then begin
                                              tmp  = mean(data[*,ind_time])
                                              line = self->formatlinestring(line+' ('+((tmp eq 0)?'':'~'),tmp,/time)+')'
                                           endif
                                        endelse
                                        end
            'macs filter&collimator':   begin
                                        tmp = stregex(header,'#Date(.*)',/fold_case,/subexpr,length=len)
                                        ind = (where(len[1,*] gt 0,count))[0]
                                        if count gt 0 then begin
                                           date = dm_to_number(strmid(header[ind],tmp[1,ind],len[1,ind]),/date)
                                           if date lt self.dates[0] then no_cfx = 1b
                                           if date lt self.dates[1] then no_mgf = 1b
                                        endif
                                        nstr = ['cfx'+['be','hopg','mgf']] 
                                        fstr = ['Be','HOPG','MgF2'] 
                                        if keyword_set(no_mgf) then fstr[2] = 'MgF2(empty)'
                                        stat = ''
                                        for j=0,n_elements(nstr)-1 do begin
                                            indx = (where(colinfo eq nstr[j],count))[0]
                                            if count ne 0 then begin
                                               if finite(data[0,indx]) then begin
                                                  if round(data[0,indx]) then begin
                                                     if stat eq '' then stat = fstr[j] else stat = stat+'&'+fstr[j]
                                                  endif
                                               endif
                                            endif
                                        endfor
                                        if stat eq '' then stat='None'
                                        line = line+'  CFX-'+stat
                                        stat = 'None'
                                        indx = (where(colinfo eq 'mcfx',count))[0]
                                        if count ne 0 then begin
                                           if finite(data[0,indx]) then begin
                                              tmp = round(data[0,indx])
                                              if (tmp ge 0) and (tmp le 2) then stat = (['BeO','Be','HOPG'])[tmp]
                                           endif
                                        endif
                                        line = line+',  MCFX-'+stat
                                        if n_elements(t_mcfx) ne 0 then line = self->formatlinestring(line+'  T_MCFX=',t_mcfx,/temperature,unit='K')
                                        nstr = [['a','b']+'colmon']
                                        fstr = ['RC_'+['60','40']+"'"]
                                        for j=0,n_elements(nstr)-1 do begin
                                            indx = (where(colinfo eq nstr[j],count))[0]
                                            stat = 0
                                            if count ne 0 then begin
                                               if finite(data[0,indx]) then stat = 0>(round(data[0,indx]))<1
                                            endif
                                            line = line+',  '+fstr[j]+'-'+(['out','in'])[stat[0]]
                                        endfor
                                        end
            'macs duration':            begin
                                        ind0 = (where(colinfo eq 'time',count0))[0]
                                        ind1 = (where(colinfo eq 'timestamp',count1))[0]
                                        if count1 ne 0 then datas = '[data]  ' else datas = ' '
                                        if count0 eq 0 then line = line+datas+'N/A' $
                                        else begin
                                           duration = total(0>(data[*,ind0]))
                                           totduration[0] = totduration[0]+duration
                                           line = self->formatlinestring(line+datas,duration,/time)
                                        endelse
                                        if count1 ne 0 then begin
                                           tmp = stregex(header,'# *Date(.*)',/fold_case,/subexpr,length=len)
                                           ind = (where(len[1,*] gt 0,count))[0]
                                           if count gt 0 then begin
                                              tstart = dm_to_number(strmid(header[ind],tmp[1,ind],len[1,ind]),/epoch,/double)
                                              if i eq 0 then tstart0 = tstart
                                              tend = data[n_elements(data[*,0])-1,ind1]
                                              line = self->formatlinestring(line+'   [actual]  ',tend-tstart,/time)
                                              totduration[1] = totduration[1]+tend-tstart
                                           endif
                                        endif
                                        end
            'macs starting time':       begin
                                        tmp = stregex(header,'# *Date(.*)',/fold_case,/subexpr,length=len)
                                        ind = (where(len[1,*] gt 0,count))[0]
                                        if count gt 0 then line = line+strtrim(strmid(header[ind],tmp[1,ind],len[1,ind]),2) else line = line+'N/A'
                                        end   
            'macs ptai':                begin 
                                        ind_ptai = (where(colinfo eq 'ptai',count))[0]
                                        if count eq 0 then line = line+'PTAI=N/A' $
                                        else begin
                                           ptai = data[*,ind_ptai]
                                           ptai = ptai[uniq(ptai,sort(ptai))]
                                           line = line+'PTAI='+dm_to_string(ptai,/int,separator=',')
                                        endelse
                                        ind_beta = (where(colinfo eq 'beta1',count))[0]
                                        if count eq 0 then line = line+',  beta1=N/A' $
                                        else begin
                                           beta = data[*,ind_beta]
                                           beta = beta[uniq(beta,sort(beta))]
                                           line = line+',  beta1='+dm_to_string(beta,separator=',',res=4)
                                        endelse
                                        ind_beta = (where(colinfo eq 'beta2',count))[0]
                                        if count eq 0 then line = line+',  beta2=N/A' $
                                        else begin
                                           beta = data[*,ind_beta]
                                           beta = beta[uniq(beta,sort(beta))]
                                           line = line+',  beta2='+dm_to_string(beta,separator=',',res=4)
                                        endelse
                                        ind_dmbt = (where(colinfo eq 'dmbt',count))[0]
                                        if count eq 0 then line = line+',  dmbt=N/A' $
                                        else line = line+',  dmbt='+dm_to_string(data[0,ind_dmbt],res=2)
                                        end  
            'macs bt7 focus':           begin
                                        count1 = 0 & count2 = 0
                                        if (n_elements(hfocus) ne 0) and (n_elements(vfocus) ne 0) then begin
                                           hf = ~strmatch(hfocus,'flat',/fold_case)
                                           vf = (~strmatch(vfocus,'flat',/fold_case))*2
                                           count1 = 1 & count2 = 1
                                        endif
                                        if (count1 eq 0) or (count2 eq 0) then line = line+'focus=N/A' $
                                        else line = line+(['flat','horizontal focusing','vertical focusing','doubly focusing'])[hf+vf]
                                        ind_vbah = (where((colinfo eq 'vbah') or (colinfo eq 'aperthori'),count))[0]
                                        if count eq 0 then line = line+',  AperHori=N/A' $
                                        else line = line+',  AperHori='+dm_to_string(data[0,ind_vbah],res=0)
                                        ind_vbav = (where((colinfo eq 'vbav') or (colinfo eq 'apertvert'),count))[0]
                                        if count eq 0 then line = line+',  AperVert=N/A' $
                                        else line = line+',  AperVert='+dm_to_string(data[0,ind_vbav],res=0)
                                        end
            'macs focus':               begin
                                        if (n_elements(hfocus) ne 0) and (n_elements(vfocus) ne 0) then begin
                                           hf = ~strmatch(hfocus,'flat',/fold_case)
                                           vf = (~strmatch(vfocus,'flat',/fold_case))*2
                                           count1 = 1 & count2 = 1
                                        endif else begin
                                           ind_mbd11 = (where(colinfo eq 'monblade11',count1))[0]
                                           ind_focus = (where(colinfo eq 'focus',count2))[0]
                                           if (count1 gt 0) and (count2 gt 0) then begin
                                              hf = (abs(data[0,ind_mbd11]) gt 0.01)
                                              vf = (data[0,ind_focus] le 5e3)*2
                                           endif
                                        endelse
                                        if (count1 eq 0) or (count2 eq 0) then line = line+'focus=N/A' $
                                        else begin
                                           line = line+(['flat','horizontal focusing','vertical focusing','doubly focusing'])[hf+vf]
                                           if ~hf then begin
                                              tmp_cnt = 0
                                              for j=1,21 do begin
                                                  ind_mbd = (where(colinfo eq 'monblade'+string(j,format='(i02)'),count))[0]
                                                  if count ne 0 then tmp_cnt = tmp_cnt+(abs(data[0,ind_mbd]) lt 0.01)
                                              endfor
                                              if tmp_cnt ne 21 then line = line+', '+dm_to_string(tmp_cnt)+'-blade only'
                                           endif
                                        endelse
                                        ind_vbah = (where(colinfo eq 'vbah',count))[0]
                                        if count eq 0 then line = line+',  VHAH=N/A' $
                                        else line = line+',  VBAH='+dm_to_string(data[0,ind_vbah],res=0)
                                        ind_vbav = (where(colinfo eq 'vbav',count))[0]
                                        if count eq 0 then line = line+',  VHAV=N/A' $
                                        else line = line+',  VBAV='+dm_to_string(data[0,ind_vbav],res=0)
                                        end                            
            'macs flip':                begin
                                        ind_flip = (where(colinfo eq 'flip',count))[0]
                                        if count gt 0  then begin
                                           flip = round(data[*,ind_flip])
                                           ind  = where(flip ne 0 and flip ne 1,cnt)
                                           if cnt ne 0 then flip[ind] = 2
                                           flip = flip[uniq(flip,sort(flip))]
                                           line = line+'flip='+strjoin(flipstring[flip],',')
                                        endif else begin
                                           ind_fpol = (where(colinfo eq 'frontpolarization',count))[0]
                                           if count eq 0 then line = line+'flip=N/A' else begin
                                              tmp = round(data[*,ind_fpol])
                                              tmp = tmp[uniq(tmp,sort(tmp))]
                                              line = line+'Polarizer='+strjoin((['up','down'])[tmp],',')
                                              ind_bpol = (where(colinfo eq 'backpolarization',count))[0]
                                              if count gt 0 then begin
                                                 tmp = round(data[*,ind_bpol])
                                                 tmp = tmp[uniq(tmp,sort(tmp))]
                                                 line = line+', Analyzer='+strjoin((['up','down'])[tmp],',')
                                              endif
                                           endelse
                                        endelse
                                        if total(stregex(colinfo,'spec[0-2][0-9]err',/boolean)) eq 20 then line = line+',  Error Bar=Yes' $
                                        else line = line+',  Error Bar=N/A'
                                        end  
            'macs bt7 flip':            begin
                                        ind_flip = (where(colinfo eq 'flip',count))[0]
                                        if count gt 0  then begin
                                           flip = round(data[*,ind_flip])
                                           ind  = where(flip ne 0 and flip ne 1,cnt)
                                           if cnt ne 0 then flip[ind] = 2
                                           flip = flip[uniq(flip,sort(flip))]
                                           line = line+'flip='+strjoin(flipstring[flip],',')
                                        endif else begin
                                           ind_fpol = (where(colinfo eq 'frontpolarization',count))[0]
                                           if count eq 0 then line = line+'flip=N/A' else begin
                                              tmp = round(data[*,ind_fpol])
                                              tmp = tmp[uniq(tmp,sort(tmp))]
                                              line = line+'polarizer='+strjoin((['up','down'])[tmp],',')
                                              ind_bpol = (where(colinfo eq 'backpolarization',count))[0]
                                              if count gt 0 then begin
                                                 tmp = round(data[*,ind_bpol])
                                                 tmp = tmp[uniq(tmp,sort(tmp))]
                                                 line = line+', analyzer='+strjoin((['up','down'])[tmp],',')
                                              endif
                                           endelse
                                        endelse
                                        end 
            'macs histogram':           begin
                                        if n_elements(is_histogram) eq 0 then begin
                                           line = line+' No'
                                        endif else begin
                                           if ~keyword_set(is_histogram) and ~keyword_set(no_rg0) then begin
                                              rg0_file = strmid(files[i],0,strpos(files[i],'.',/reverse_search))+'_0.rg0'
                                              is_histogram = (file_info(rg0_file)).exists*2
                                              if keyword_set(is_histogram) then dm_load_macshisto2d,rg0_file,n_tbin=n_chan,n_theta=n_theta,a3_offset=a3_offset,n_chan=tot_chan,/headeronly
                                           endif
                                           line = line+([' No',' Yes',' Event mode'])[is_histogram]
                                           if n_elements(n_chan) ne 0 then line = line+',  '+dm_to_string(n_chan)+' time channel'+(['','s'])[temporary(n_chan) gt 1]
                                           if n_elements(n_theta) ne 0 then line = line+',  '+dm_to_string(n_theta)+' A3 channel'+(['','s'])[temporary(n_theta) gt 1]
                                           if n_elements(tot_chan) ne 0 then line = line+',  '+(['w/o','w/'])[temporary(tot_chan) gt 20]+' coverage map
                                        endelse
                                        end                                             
            'macs specify':             begin
                                        if n_elements(niceinfo) ne 0 then tmp_info = niceinfo else tmp_info = colinfo
                                        for j=0,n_elements(field)-1 do begin
                                            ind_field = (where(tmp_info eq strlowcase(field[j]),count))[0]
                                            if j ne 0 then line = line+'  '
                                            if count eq 0 then line = line+field[j]+'=N/A' $
                                            else begin
                                               value = data[*,ind_field]
                                               if stregex(field[j],'(theta|^a[1-6]|mon(rot|blade)|^beta[1-2])',/fold_case,/boolean) then line = self->formatlinestring(line+' '+field[j]+'=',value,field[j],/angle) $
                                               else begin
                                                  value = value[uniq(value,sort(value))]
                                                  line  = line+field[j]+'='+dm_to_string(value,separator=',')
                                               endelse
                                            endelse
                                        endfor
                                        end
            'nxspe number of detectors':line = line+dm_to_string(header[0])
            'nxspe number of energies': line = line+dm_to_string(header[1])
            'nxspe energy range':       line = line+'['+dm_to_string(header[2])+', '+dm_to_string(header[3])+'] meV'
            'nxspe ki/kf correction flag':begin
                                        line = line+dm_to_string(header[4])
                                        if (header[4] ge 0) and (header[4] le 3) then $
                                        line = line+'  ('+(['no, const t bins','yes, const t bins','no, const E bins','yes, const E bins'])[header[4]]+')'
                                        end
            'nxspe fixed eief':         line = line+((n_elements(header) eq 6)?(dm_to_string(header[5])+' meV'):'unspecified')
            'nxspe psi':                line = self->formatlinestring(line,psi,'psi',/angle)
            'spe number of detectors':  line = line+dm_to_string(header[0])
            'spe number of energies':   line = line+dm_to_string(header[1])
            'spe energy range':         line = line+'['+dm_to_string(header[2])+', '+dm_to_string(header[3])+'] meV'
            'wand psi':                 line = self->formatlinestring(line,psi,'psi',/angle)
            'wand temperature':         line = self->formatlinestring(line,temperature,/temperature)
            'wand mask':                begin
                                        masklimit = -1e20
                                        ind = where(data[*,0] le masklimit,count)
                                        if count eq 0 then begin
                                           line = line+'no mask' 
                                        endif else begin
                                           if count lt 20 then $
                                              line = line+dm_to_string(ind,separator=', ') $
                                           else $
                                              line = line+ 'mask '+dm_to_number(count)+' detectors'
                                        endelse
                                        end            
            'inx comment':              line = line+strtrim(comment,2)
            'inx fixed eief':           line = self->formatlinestring(line,ei,/energy)
            'inx energy range':         line = self->formatlinestring(line,ydat,'energies',/energy)
            'inx mean temperature':     line = self->formatlinestring(line,temperature,/temperature)
            'inx time channel':         line = line+' [total]  '+dm_to_string(tchannel[0])+'  [actual]  '+dm_to_string(tchannel[1])
            'tas monitor':              begin
                                        tmp = temporary(wt) & tmp = temporary(te)
                                        if n_elements(header) ne 0 then isNeut = stregex(header[0],"'neut'",/boolean,/fold_case) else isNeut = 0
                                        if (where(strlowcase(var_names) eq 'mon'))[0] ne -1  then wt = *(data.mon)
                                        if (where(strlowcase(var_names) eq 'min'))[0] ne -1  then te = (*(data.min))*60
                                        if (where(strlowcase(var_names) eq 'time'))[0] ne -1 then te = *(data.time)
                                        if n_elements(wt) ne 0 then begin
                                           if isNeut then begin 
                                              line = line+dm_to_string(wt[0],/exponent)
                                              if n_elements(te) ne 0 then begin
                                                 tmp  = mean(te)
                                                 line = self->formatlinestring(line+' ('+((tmp eq 0)?'':'~'),tmp,/time)+')'
                                              endif
                                           endif else line = line+dm_to_string(wt[0])+' Sec'
                                        endif else line = line+' N/A'
                                        end
            'tas duration':             begin
                                        if (where(strlowcase(var_names) eq 'min'))[0] ne -1  then te = (*(data.min))*60
                                        if (where(strlowcase(var_names) eq 'time'))[0] ne -1 then te = *(data.time)
                                        if n_elements(te) ne 0 then begin
                                           duration = total(temporary(te))
                                           totduration[0] = totduration[0]+duration
                                           line = self->formatlinestring(line,duration,/time)
                                        endif else line = line+' N/A'
                                        end 
            'tas energy':               begin
                                        tmp = temporary(Ei) & tmp = temporary(Ef)
                                        if n_elements(header) ne 0 then begin
                                           ind = (where(stregex(header,'e[a-z] fixed',/boolean,/fold_case),count))[0]
                                           if ind gt 0 then begin
                                              Efixed = dm_to_number((strsplit(header[ind-1],' ',/extract))[2])
                                              isEf = stregex(header[ind],'e(a|f) fixed',/boolean,/fold_case)
                                              if isEf then Ef = Efixed else Ei = Efixed
                                              if (where(strlowcase(var_names) eq 'e'))[0] ne -1  then begin
                                                 if isEf then Ei = Ef+*(data.e) else Ef = Ei-*(data.e)
                                              endif
                                              if n_elements(Ei) ne 0 then line = self->formatlinestring(line+' Ei=',Ei,'Ei',/energy)
                                              if n_elements(Ef) ne 0 then line = self->formatlinestring(line+', Ef=',Ef,'Ef',/energy)
                                              if (n_elements(Ei) ne 0) and (n_elements(Ef) ne 0) then begin
                                                 if ~stregex(line,'E[if]=.+to.+',/fold_case,/boolean) then begin  ;to be consistent
                                                    Ei = mean(Ei) & Ef = mean(Ef)
                                                 endif
                                                 line = self->formatlinestring(line+', E=',Ei-Ef,'E',/energy)
                                              endif
                                           endif else line = line+' N/A' 
                                        endif else line = line+' N/A'
                                        end 
            'tas temperature':          begin
                                        tmp = temporary(t1)
                                        if (where(strlowcase(var_names) eq 'temp'))[0] ne -1  then t1 = *(data.temp)
                                        if (where(strlowcase(var_names) eq 'tact'))[0] ne -1  then t1 = *(data.tact)
                                        if n_elements(t1) ne 0 then line = self->formatlinestring(line,t1,/temperature,unit='K') else line = line+' N/A'
                                        end 
            'tas magfield':             begin
                                        tmp = temporary(hf)                                        
                                        if (where(strlowcase(var_names) eq 'hfield'))[0] ne -1  then hf = *(data.hfield)
                                        if (where(strlowcase(var_names) eq 'magfield'))[0] ne -1  then hf = *(data.magfield)
                                        if n_elements(hf) ne 0 then line = self->formatlinestring(line,hf,'magnetic field',res=4,unit='T') else line = line+' N/A'
                                        end
            'tas starttime':            if n_elements(header) ne 0 then begin
                                        tmp = stregex(header[0],"'([^']*[0-9]{2}\:[0-9]{2})'",/extract,/subexpr)
                                        if strlen(tmp[0]) ne 0 then line = line+tmp[1] else line = line+' N/A'
                                        end            
            else:
           endcase
        endelse
        if n_elements(info) eq 0 then info=line else info=[info,line]
        if obj_valid(tmpobj) then obj_destroy,tmpobj
    endfor
    if n_elements(baddetinfo) ne 0 then info=[info,'',baddetinfo,'(Detector number starts from 0.)']
    n_info = n_elements(info)
    if n_info gt 0 then begin
       if (name eq 'duration') and (n_info gt 1) then begin
          line0 = ''
          for i = 0, max(strlen(strtrim(*self.file,2)))-6 do line0 = line0+' '
          line = line0+'total :  '
          if totduration[1] gt 0 then begin
             line = self->formatlinestring(line+' [data]  ',totduration[0],/time)
             line = self->formatlinestring(line+'   [actual]  ',totduration[1],/time)      
             if n_elements(tstart0) ne 0 then line = [line,self->formatlinestring(line0+'time lapsed :  ',tend-tstart0,/time)]
          endif else $
             line = self->formatlinestring(line,totduration[0],/time)
          info = [info,'',line]
       endif
       if (instr+' '+name) eq 'wand mask'         then info=[info,'','Detector number starts from 0.']
       if (instr+' '+name) eq 'macs kidney angle' then info=[info,'','A4=kidney+[-76, -68, -60, ... 60, 68, 76]'+string('b0'XB)]
       if (name eq 'filter&collimator') and keyword_set(no_cfx) then info=[info,'','CFX info may not reflect the actual setting.']
    endif
    if obj_isa(self.pobj,'dcs_mslice') and (name eq 'fixed eief') then begin
       self.pobj->getproperty,instrgeom=instrgeom
       name = 'fixed '+(['Ei','Ef'])[instrgeom]
    endif
    if name eq 'specify' and n_elements(field) ne 0 then begin
       name = field[0]
       for i=1,n_elements(field)-1 do name=name+', '+field[i]
    endif
    title = strupcase(instr)+' file info: ['+name+']'
    if n_info gt self.maxline then begin
       if widget_info(self.displayid,/valid_id) then widget_control,self.displayid,/destroy
       xdisplayfile,text=info,/editable,group=self.group_leader,title=title,done_button='Exit',/grow_to_screen,return_id=wid
       dm_center_kid,wid,self.group_leader
       self.displayid = wid
    endif else if n_info gt 0 then begin
       ok = dialog_message(info,/info,dialog_parent=self.group_leader,title=title,/center)
    endif
end

function dm_filesel::ftplisting,i_dirs=i_dirs,i_file=i_file,n_dir=n_dir,n_file=n_file,count=count
    n_dir = 0 & n_file = 0 & count = 0
    tmp = self.ftpobj->getlisting()
    if obj_isa(self.ftpobj,'DaveHTTPRequest') then begin
       n_dir = tmp["subdirs"]->count()
       n_file = tmp["files_metadata"]->count()
       if n_dir gt 0 then begin
          tmp1 = (tmp["subdirs"]).toarray()+'/'
          i_dirs = lindgen(n_dir)
       endif
       if n_file gt 0 then begin
          if n_elements(tmp1) eq 0 then tmp1 = (tmp["files_metadata"]->keys()).toarray() else tmp1 = [tmp1,(tmp["files_metadata"]->keys()).toarray()]
          i_file = n_dir+lindgen(n_file)
       endif
       count = n_elements(tmp1)
       if count eq 0 then tmp = '' else tmp = temporary(tmp1)
    endif else if n_elements(tmp) gt 0 then begin ;legacy daveftpurl call 
       inds = stregex(tmp,'(d*)[rwxs-]{9}.+[A-Z][a-z][a-z] +[0-9]+ +[0-9:]+ +([^ ]+)',/subexpr,length=lens)
       index = where(lens[2,*] gt 0,count)
       if count gt 0 then begin
          inds = inds[*,index]
          lens = lens[*,index]
          tmp  = tmp[index]
          dirs = reform((lens[1,*] gt 0))
          for i=0L,counts-1 do tmp[i] = strmid(tmp[i],inds[2,i],lens[2,i])
          i_dirs = where(dirs,n_dir,complement=i_file,ncomplement=n_file)
          if n_dir gt 0 then tmp[i_dirs] = tmp[i_dirs]+'/'
       endif
    endif
    return,tmp
end

;create the ftp object, first try the latest davehttprequest (idl9.0), if not working, try the old daveftpurl anyway
function dm_filesel::ftpobj,ftpworking=ftpworking,DAVEftpURL=DAVEftpURL
    if self.idlversion lt 9.0 then DAVEftpURL = 1
    ftpobj = obj_new((['DaveHTTPRequest','DAVEftpURL'])[keyword_set(DAVEftpURL)],basedir=self.basedir)
    if arg_present(ftpworking) then begin
       ftpworking = obj_valid(ftpobj)
       catch, ftperror
       if ftperror ne 0 then begin
          catch,/cancel
          ftpworking = 0b
          if ~keyword_set(DAVEftpURL) then return,self->ftpobj(ftpworking=ftpworking,/DAVEftpURL)
       end
       if keyword_set(ftpworking) then begin
          tmp = ftpobj->getlisting()
          self.ftpname[0] = (['https://ncnr.nist.gov','ftp://ftp.ncnr.nist.gov'])[keyword_set(DAVEftpURL)]
       endif
    endif
    return,ftpobj
end

;*******************************************************************
;object event handler
;*******************************************************************
pro dm_filesel::event,event
    compile_opt IDL2    ;,strictarrsubs
                        ;idl2=defint32,strictarr
                        ;strictarr:   [] to index array
                        ;strictarrsubs: error when out-of-range indices,IDL5.6 or after
    ;error handling
    catch, anyerror
    if anyerror ne 0 then begin
       catch,/cancel
       ok = dialog_message(!error_state.msg ,/error,dialog_parent=self.tlb,/center)
       if widget_info(self.tlb,/update) eq 0 then widget_control,self.tlb,/update
       if n_elements(tastmpfile) ne 0 then file_delete,tastmpfile,/ALLOW_NONEXISTENT,/NOEXPAND_PATH,/QUIET       
       return
    end
    eventname = widget_info(event.id,/uname)
    type = tag_names(event,/structure)
    case strlowcase(type) of
        'widget_context':     eventname='fileinfo'
        'widget_kbrd_focus':  eventname='text_focus'
        else:
    endcase
    if eventname ne 'text_focus' then WIDGET_CONTROL,/HOURGLASS
    case strlowcase(eventname) of
       'text_focus': begin
            if event.enter then begin
               if !version.os_family eq 'Windows' then begin
                  widget_control,event.id,get_value=tmp
                  widget_control,event.id,set_text_select=[0,total(strlen(tmp))] 
               endif
            endif else $
               widget_control,event.id,set_text_select=[0,0]
            end
       'dripull': begin
            drives = get_drive_list(count=count)
            if self.ncnrftp then begin
               drives = [drives,self.ftpname[0]]
               count = count+1
            endif
            currdir = (*self.drivedir)[event.index<(0>(count-1)),1]
            currftp = obj_valid(self.ftpobj)
            if event.index ne count-1 then obj_destroy,self.ftpobj
            widget_control,event.id,set_value=drives,set_droplist_select=event.index<(0>(count-1))
            isdir = self->isDir(drives[event.index<(0>(count-1))],/showmesg)
            if isdir then begin
               if drives[event.index<(0>(count-1))] eq self.ftpname[0] then begin
                  if ~obj_valid(self.ftpobj) then self.ftpobj = self->ftpobj()
                  if ~obj_valid(self.ftpobj) then begin
                     ok = dialog_message("Can't access "+self.ftpname[0]+'.',dialog_parent=self.tlb,/center,/error) 
                     self->set_path,self.path
                  endif else begin  
                     self->set_path,currdir,/ftp
                  endelse
               endif else begin
                  if (strlen(currdir) eq 0) or ~self->isDir(currdir) then begin
                     cd,drives[event.index<(0>(count-1))]
                     cd,current=currdir
                  endif
                  self->set_path,currdir
                  self->clear_file
               endelse
            endif else begin
               if currftp and ~obj_valid(self.ftpobj) then self.ftpobj = self->ftpobj()
               self->set_path,self.path,ftp=obj_valid(self.ftpobj)
            endelse
          end
       'upbut': begin
            tmp = strsplit(self.path,self.pathsep[obj_valid(self.ftpobj)],/extract)
            tmp = tmp[n_elements(tmp)-1]+self.pathsep[obj_valid(self.ftpobj)] 
            if obj_valid(self.ftpobj) then begin
               current = strtrim(self.path,2)
               while(strmid(current,0,1,/reverse_offset) eq  self.pathsep[1]) do  current = strmid(current,0,strlen(current)-1)
               while(strmid(current,0,1,/reverse_offset) ne self.pathsep[1]) do begin
                  current = strmid(current,0,strlen(current)-1)
                  if strlen(current) eq 0 then begin
                     current='/'
                     break
                  endif
               endwhile
            endif else begin
               while ~file_test(self.path,/directory) do begin
                  self.path = strmid(self.path,0,strpos(self.path,self.pathsep[0],/reverse_search))
                  if strlen(self.path) eq 0 then break
               endwhile
               cd,self.path
               cd,self.upDir
               cd,current=current
            endelse
            self->set_path,current,ftp=obj_valid(self.ftpobj)
            self->clear_file,fileloc=tmp,/highlight
          end
       'refreshbut': self->refresh
       'pathlab':begin
            widget_control,event.id,get_value=newPath
            self->set_path,strtrim(newPath[0],2),ftp=obj_valid(self.ftpobj)
            self->clear_file
          end
       'limitfrom':begin
            widget_control,event.id,get_value=tmp
            tmp = strtrim(tmp,2)
            if self.limitfrom ne tmp then begin
               self.limitfrom = tmp
               self->set_path,ftp=obj_valid(self.ftpobj)
            endif
          end
       'limitto':begin
            widget_control,event.id,get_value=tmp
            tmp = strtrim(tmp,2)
            if self.limitto ne tmp then begin
               self.limitto = tmp
               self->set_path,ftp=obj_valid(self.ftpobj)
            endif
          end
       'filelist':begin
            if ~ptr_valid(self.dFiles) then break
            dFiles   = *self.dFiles
            selected = widget_info(event.id,/list_select) & tmp=where(selected ge 0,n_item)
            if n_item eq 0 then break  ;nothing selected
            isDir = self->isDir(dFiles[selected],/fast)
            case(event.clicks) of
                1:  begin ;single click
                    self->set_file  ;clear file pointer
                    index = where(isDir eq 0,count)
                    if count ne 0 then self->set_file,selected[index]
                    end
                2:  begin ;double click
                    self->getproperty,path=path
                    if isDir[0] then begin
                       if obj_valid(self.ftpobj) then begin
                          currDir = path+dFiles[event.index]
                       endif else begin
                          if dFiles[event.index] eq self.ftpname[1] then begin
                             currDir = self.ftpname[1]
                          endif else begin
                             cd,path
                             cd,dFiles[event.index]
                             cd,current=currDir
                          endelse
                       endelse
                       self->set_path,currDir,ftp=obj_valid(self.ftpobj)
                       self->clear_file
                    endif else self->set_file,event.index
                    end
                else:
            endcase
            if ptr_valid(self.file) and (self.group_leader ne 0) and self.selectevent then $
               widget_control,self.group_leader,send_event={dm_filesel_select,ID:self.tlb,TOP:self.group_leader,HANDLER:self.tlb,object:self},/no_copy
          end
       'fileinfo':begin
            self->my_widget_control,'contextMenu',event.id,/destroy
            ;context sensitive menu
            menuid = widget_base(event.id,/CONTEXT_MENU,UNAME='contextMenu')
            self->getproperty,file=file,eol=eol
            if n_elements(file) eq 0 then begin
               void = widget_button(menuid,value='Change File Type',uname='newfilter')
               if file_test(self.path,/directory,/write) then void = widget_button(menuid,value='Create New Directory',uname='newdirectory')
               if obj_valid(self.ftpobj) and ~xregistered('dm_copyftp',/noshow) then void = widget_button(menuid,value='Copy Files from NCNR Data Repository',uname='copyftp')
            endif else begin
               if obj_isa(self.pobj,'dcs_mslice') then begin
                  self.pobj->getproperty,instrgeom=instrgeom
                  fixeief = 'Fixed '+(['Ei','Ef'])[instrgeom]
               endif else fixeief = 'Fixed Ei or Ef'
               filter = ['.dcs','.ng0','.ng5','.spe','.nxspe','.inx','.wand','.b[ta-d]2','.bt4','.b[ta-d]7','.b[ta-d]9','.phd']  ;recognized filter type
               ind = bytarr(n_elements(filter))
               for i=0,n_elements(filter)-1 do ind[i] = stregex(file[0],'\'+filter[i]+'(\.|$)',/boolean,/fold_case)
               filter = (['',filter])[(where(ind))[0]+1]
               case filter of
                    '.dcs': begin
                            void = widget_button(menuid,value='Comment',              uname='dcs_comment')
                            void = widget_button(menuid,value='Wavelength',           uname='dcs_wavelength')
                            void = widget_button(menuid,value='Duration',             uname='dcs_duration')
                            void = widget_button(menuid,value='Temperature',          uname='dcs_temperature')
                            void = widget_button(menuid,value='Magnetic Field',       uname='dcs_magfield')
                            void = widget_button(menuid,value='Shutter State',        uname='dcs_shutterstate')
                            void = widget_button(menuid,value='Sample Position',      uname='dcs_sample')
                            void = widget_button(menuid,value='Starting Time',        uname='dcs_starttime')
                            if dm_to_number(systime(),/date) gt self.dates[2] then $
                            void = widget_button(menuid,value='Polarization Reduced Data',uname='dcs_polanal')
                            end
                    '.ng0': begin
                            void = widget_button(menuid,value='Comment',              uname='macs_comment')
                            void = widget_button(menuid,value='Monitor',              uname='macs_monitor')
                            void = widget_button(menuid,value='Duration',             uname='macs_duration')
                            void = widget_button(menuid,value='Energy',               uname='macs_energy')
                            void = widget_button(menuid,value='Kidney Angle',         uname='macs_kidney')
                            void = widget_button(menuid,value='Sample Position',      uname='macs_sample')
                            void = widget_button(menuid,value='Temperature',          uname='macs_temperature')
                            void = widget_button(menuid,value='Magnetic Field',       uname='macs_magfield')
                            void = widget_button(menuid,value='Filter and Collimator',uname='macs_filter')
                            void = widget_button(menuid,value='PTAI/Beta1/Beta2/dmbt',uname='macs_ptai')
                            void = widget_button(menuid,value='Focus/VBAH/VBAV',      uname='macs_focus')
                            void = widget_button(menuid,value='Polarization/Error Bar',uname='macs_flip')
                            void = widget_button(menuid,value='Starting Time',        uname='macs_starttime')
                            void = widget_button(menuid,value='Histogram Data',       uname='macs_histogram')
                            view = ~stregex(file[0],'\.nxs',/boolean,/fold_case)
                            if view then void = widget_button(menuid,value='Specify a Field...',uname='macs_specify')
                            end
                    '.ng5': begin
                            void = widget_button(menuid,value='Monitor',              uname='tas_monitor')
                            void = widget_button(menuid,value='Duration',             uname='tas_duration')
                            void = widget_button(menuid,value='Energy',               uname='tas_energy')
                            void = widget_button(menuid,value='Temperature',          uname='tas_temperature')
                            void = widget_button(menuid,value='Magnetic Field',       uname='tas_magfield')
                            void = widget_button(menuid,value='Starting Time',        uname='tas_starttime')
                            view = 1b
                            end               
                    '.spe': begin
                            void = widget_button(menuid,value='Number of Detectors',  uname='spe_ndet')
                            void = widget_button(menuid,value='Number of Energies',   uname='spe_ne')
                            void = widget_button(menuid,value='Energy Range',         uname='spe_eran')
                            view = 1b
                            end
                    '.nxspe': begin
                            void = widget_button(menuid,value='Number of Detectors',  uname='nxspe_ndet')
                            void = widget_button(menuid,value='Number of Energies',   uname='nxspe_ne')
                            void = widget_button(menuid,value=fixeief,                uname='nxspe_eief')
                            void = widget_button(menuid,value='Energy Range',         uname='nxspe_eran')
                            void = widget_button(menuid,value='ki/kf Correction Flag',uname='nxspe_lamcorr')
                            void = widget_button(menuid,value='Psi',                  uname='nxspe_psi')
                            end        
                    '.inx': begin
                            void = widget_button(menuid,value='Comment',              uname='inx_comment')
                            void = widget_button(menuid,value=fixeief,                uname='inx_eief')
                            void = widget_button(menuid,value='Energy Range',         uname='inx_eran')
                            void = widget_button(menuid,value='Mean Temperature',     uname='inx_temperature')
                            void = widget_button(menuid,value='Time Channel',         uname='inx_tchannel')
                            view = 1b
                            end
                    '.wand':begin
                            void = widget_button(menuid,value='Psi',                  uname='wand_psi')
                            void = widget_button(menuid,value='Temperature',          uname='wand_temperature')
                            void = widget_button(menuid,value='Detector Mask',        uname='wand_mask')
                            view = 1b
                            end
                    '.b[ta-d]2': begin
                            void = widget_button(menuid,value='Monitor',              uname='tas_monitor')
                            void = widget_button(menuid,value='Duration',             uname='tas_duration')
                            void = widget_button(menuid,value='Energy',               uname='tas_energy')
                            void = widget_button(menuid,value='Temperature',          uname='tas_temperature')
                            void = widget_button(menuid,value='Magnetic Field',       uname='tas_magfield')
                            void = widget_button(menuid,value='Starting Time',        uname='tas_starttime')
                            view = 1b
                            end
                    '.bt4': begin
                            void = widget_button(menuid,value='Monitor',              uname='tas_monitor')
                            void = widget_button(menuid,value='Duration',             uname='tas_duration')
                            void = widget_button(menuid,value='Energy',               uname='tas_energy')
                            void = widget_button(menuid,value='Temperature',          uname='tas_temperature')
                            void = widget_button(menuid,value='Magnetic Field',       uname='tas_magfield')
                            void = widget_button(menuid,value='Starting Time',        uname='tas_starttime')
                            view = 1b
                            end
                    '.b[ta-d]7': begin
                            void = widget_button(menuid,value='Comment',              uname='macs_comment')
                            void = widget_button(menuid,value='Monitor',              uname='macs_monitor')
                            void = widget_button(menuid,value='Duration',             uname='macs_duration')
                            void = widget_button(menuid,value='Energy',               uname='macs_bt7energy')
                            void = widget_button(menuid,value='Sample Position',      uname='macs_bt7sample')
                            void = widget_button(menuid,value='Temperature',          uname='macs_temperature')
                            void = widget_button(menuid,value='Magnetic Field',       uname='macs_magfield')
                            void = widget_button(menuid,value='Focus/Aperture',       uname='macs_bt7focus')
                            void = widget_button(menuid,value='Polarization',         uname='macs_bt7flip')
                            void = widget_button(menuid,value='Starting Time',        uname='macs_starttime')
                            void = widget_button(menuid,value='Specify a Field...',   uname='macs_specify')
                            view = ~stregex(file[0],'\.nxs',/boolean,/fold_case)
                            end
                    '.b[ta-d]9': begin
                            void = widget_button(menuid,value='Monitor',              uname='tas_monitor')
                            void = widget_button(menuid,value='Duration',             uname='tas_duration')
                            void = widget_button(menuid,value='Energy',               uname='tas_energy')
                            void = widget_button(menuid,value='Temperature',          uname='tas_temperature')
                            void = widget_button(menuid,value='Magnetic Field',       uname='tas_magfield')
                            void = widget_button(menuid,value='Starting Time',        uname='tas_starttime')
                            view = 1b
                            end        
                    '.phd': begin
                            void = widget_button(menuid,value='Comment',              uname='macs_comment')
                            void = widget_button(menuid,value='Monitor',              uname='macs_monitor')
                            void = widget_button(menuid,value='Duration',             uname='macs_duration')
                            void = widget_button(menuid,value='Energy',               uname='macs_bt7energy')
                            void = widget_button(menuid,value='Sample Position',      uname='macs_bt7sample')
                            view = ~stregex(file[0],'\.nxs',/boolean,/fold_case)
                            end        
                    else:   view = stregex(file[0],'\.(log|prm|txt|phx|b[ta-d][2-9]|ng[0-7])$',/boolean,/fold_case)
               endcase
               if (filter eq '.dcs') or (filter eq '.ng0') or (filter eq '.nxspe') then begin
                  rangeid = widget_button(menuid,value='Select File Range',/menu,separator=(n_elements(void) ne 0))
                  void = widget_button(rangeid,value='To the End of the List',uname='fselrange_end',sensitive=~keyword_set(eol))
                  void = widget_button(rangeid,value='To File Number...',     uname='fselrange_number')
               endif
               if keyword_set(view) and (n_elements(file) eq 1) then void1 = widget_button(menuid,value='View File Contents',uname='text_viewfile',separator=(n_elements(void) ne 0))
               if obj_valid(self.ftpobj) and ~xregistered('dm_copyftp',/noshow) then void = widget_button(menuid,value='Copy Files from NCNR Data Repository',uname='copyftp',separator=(n_elements(void) gt n_elements(void1)))
               if self.write then void = widget_button(menuid,value='Delete',uname='file_delete',separator=n_elements(void) ne 0)
               if (n_elements(void) eq 0) and (n_elements(void1) eq 0) then break  ;no context sensitive menu entry
            endelse
            if ~strmatch(self.filter,'.dcs',/fold_case) then void = widget_button(menuid,value=('Sort Files by '+['File Name','File Number'])[self.sortfile],uname='sort_file',/separator)
            widget_displaycontextmenu,event.id,event.x,event.y,menuid
          end
       'newfilter': begin
            newfilter = dm_dialog_input('New Filter:',default=self.filter,title='New File Type',info='e.g., .dcs and .*',cancel=cancel,dialog_parent=self.group_leader)
            if keyword_set(cancel)  or (newfilter eq '') then break
            if strupcase(newfilter) ne strupcase(self.filter) then begin
               self->set_filter,newfilter,/keepcurrentdir
               if self.group_leader ne 0 then widget_control,self.group_leader,send_event={dm_filesel_newfilter,ID:self.tlb,TOP:self.group_leader,HANDLER:self.tlb,object:self,filter:self.filter,keepcurrentdir:1},/no_copy
            endif
          end
       'newdirectory': begin
            newdir = dm_dialog_input('Directory Name:',dialog_parent=self.group_leader,cancel=cancel,xsize=120) 
            if (strlen(newdir) eq 0) or keyword_set(cancel) then break
            file_mkdir,self.path+self.pathsep[0]+newdir
            self->set_path,ftp=obj_valid(self.ftpobj)    ;update the directory
            self->set_file,filename=' '
          end
       'copyftp': dm_copyftp,event,filter=self.filter,workDir=self.workdir,dataDir=self.ftpname[0]+self.path 
       'file_delete': begin
            self->getproperty,file=file,path=path,sep=sep
            if n_elements(file) eq 0 then break
            info = 'Are you sure you want to delete '+(n_elements(file) gt 1?'the selected files':file)+'?'
            ans  = dialog_message(info,/question,dialog_parent=self.group_leader,/center)
            if strlowcase(ans) eq 'yes' then begin
               file_delete,path+sep+file,/allow_nonexistent
               self->set_path,ftp=obj_valid(self.ftpobj)    ;update the directory
               self->set_file,filename=' '
            endif
          end   
       'dcs_comment':       self->dcsfileheader, 'comment'
       'dcs_wavelength':    self->dcsfileheader, 'wavelength'
       'dcs_duration':      self->dcsfileheader, 'duration'
       'dcs_temperature':   self->dcsfileheader, 'temperature'
       'dcs_magfield':      self->dcsfileheader, 'magfield'
       'dcs_shutterstate':  self->dcsfileheader, 'shutter state'
       'dcs_sample':        self->dcsfileheader, 'sample position'
       'dcs_starttime':     self->dcsfileheader, 'starting time'
       'dcs_polanal':       self->dcsfileheader, 'polarization reduced data'
       'macs_comment':      self->fileheader,    'macs','comment'
       'macs_energy':       self->fileheader,    'macs','energy'
       'macs_bt7energy':    self->fileheader,    'macs','bt7 energy'
       'macs_kidney':       self->fileheader,    'macs','kidney angle'
       'macs_sample':       self->fileheader,    'macs','sample position'
       'macs_bt7sample':    self->fileheader,    'macs','bt7 sample position'
       'macs_temperature':  self->fileheader,    'macs','temperature'  
       'macs_magfield':     self->fileheader,    'macs','magfield'  
       'macs_monitor':      self->fileheader,    'macs','monitor'
       'macs_focus':        self->fileheader,    'macs','focus'
       'macs_bt7focus':     self->fileheader,    'macs','bt7 focus'
       'macs_filter':       self->fileheader,    'macs','filter&collimator'
       'macs_duration':     self->fileheader,    'macs','duration'
       'macs_ptai':         self->fileheader,    'macs','ptai'
       'macs_flip':         self->fileheader,    'macs','flip'
       'macs_bt7flip':      self->fileheader,    'macs','bt7 flip'
       'macs_starttime':    self->fileheader,    'macs','starting time'
       'macs_histogram':    self->fileheader,    'macs','histogram'
       'macs_specify':      begin
                            self.macs_specify = dm_dialog_input('field name:',title='Please specify:',default=self.macs_specify,info='Multiple fields should be separated by space or comma.',xsize=110,$
                                    cancel=cancel,dialog_parent=self.group_leader)
                            field = strsplit(self.macs_specify,' ,'+string(9b),/extract)
                            if (strlen(field[0]) eq 0) or keyword_set(cancel) then return
                            self->fileheader,    'macs','specify',field=field
                            end
       'tas_monitor':       self->fileheader,    'tas','monitor',    tmpfile=tastmpfile
       'tas_duration':      self->fileheader,    'tas','duration',   tmpfile=tastmpfile
       'tas_energy':        self->fileheader,    'tas','energy',     tmpfile=tastmpfile
       'tas_temperature':   self->fileheader,    'tas','temperature',tmpfile=tastmpfile
       'tas_magfield':      self->fileheader,    'tas','magfield',   tmpfile=tastmpfile
       'tas_starttime':     self->fileheader,    'tas','starttime',  tmpfile=tastmpfile
       'text_viewfile':     self->fileheader,    '','viewfile'
       'spe_ndet':          self->fileheader,    'spe','number of detectors'
       'spe_ne':            self->fileheader,    'spe','number of energies'
       'spe_eran':          self->fileheader,    'spe','energy range'
       'nxspe_ndet':        self->fileheader,    'nxspe','number of detectors'
       'nxspe_ne':          self->fileheader,    'nxspe','number of energies'
       'nxspe_eief':        self->fileheader,    'nxspe','fixed eief'
       'nxspe_eran':        self->fileheader,    'nxspe','energy range'
       'nxspe_lamcorr':     self->fileheader,    'nxspe','ki/kf correction flag'
       'nxspe_psi':         self->fileheader,    'nxspe','psi'
       'inx_comment':       self->fileheader,    'inx','comment'
       'inx_eief':          self->fileheader,    'inx','fixed eief'
       'inx_eran':          self->fileheader,    'inx','energy range'
       'inx_temperature':   self->fileheader,    'inx','mean temperature'
       'inx_tchannel':      self->fileheader,    'inx','time channel'
       'wand_psi':          self->fileheader,    'wand','psi'
       'wand_temperature':  self->fileheader,    'wand','temperature'
       'wand_mask':         self->fileheader,    'wand','mask'
       'sort_file':         self->changesort
       'fselrange_end':begin
            self->getproperty,file=file,dFiles=dFiles
            if n_elements(file) eq 0 then break
            ind = 0L
            for i=0,n_elements(file)-1 do ind = [ind,where(dFiles eq file[i])]
            if n_elements(dFiles) gt ind[-1] then ind = [ind,lindgen(n_elements(dFiles)-ind[-1]-1)+ind[-1]+1]
            self->set_file,ind[1:*],/highlight
          end
       'fselrange_number':begin
            self->getproperty,file=file,dFiles=dFiles
            if n_elements(file) eq 0 then break
            number = dm_dialog_input('To file number:',default=self.fnum_default,title='Please specify:',cancel=cancel,xsize=80,dialog_parent=self.group_leader)
            if keyword_set(cancel) or (strlen(number) eq 0) then break
            ind = 0L
            for i=0,n_elements(file)-1 do ind = [ind,where(dFiles eq file[i])]
            ind0 = where(stregex(dFiles,number,/boolean),cnt)
            if cnt ne 0 then begin
               self.fnum_default = number
               if ind0[-1] gt ind[-1] then ind = [ind,lindgen(ind0[-1]-ind[-1])+ind[-1]+1] $
               else if ind0[0] lt ind[1] then ind = [ind,lindgen(ind[1]-ind0[0])+ind0[0]]
            endif
            self->set_file,ind[1:*],/highlight
          end   
       'savefile':begin
            widget_control,event.id,get_value=newfile
            self->set_file,filename=newfile[0],/noupdate
          end
       'okbut':begin
            if ptr_valid(self.file) then $
               widget_control,self.tlb,/destroy
          end
       'cancelbut':begin
            self->clear_file
            widget_control,self.tlb,/destroy
          end
       else:
    endcase
    if obj_valid(self.ftpobj) then self.ftpobj->CloseConnections
end

;returns the string length of a vlue, converted to string first, 2 for each regular letter, 1 for dot
function dm_filesel::wavlstrlen,value,string=string,_extra=extra
    string = dm_to_string(value,_extra=extra)
    return, 2*strlen(string)-stregex(string,'\.',/boolean)
end

;format a line string
;keywords:
;  angle:       if set, compile an angle string
;  energy:      if set, compile an energy string
;  temperature: if set, compile a temperature string
;  time:        if set, compile a time string
;  tolerance:   tolerance for judging step
;  unit:        unit string
;  wavelength:  if set, compile a DCS wavelength string, invalue = [ch_wl,ch_res,ch_ms,ch_srmode,ch_srdenom,tsdmin]
;  maxlength:   in conjuntion with wavelength keyword, to achieve same display length
function dm_filesel::formatlinestring,line,invalue,name,time=time,temperature=temperature,angle=angle,energy=energy,resolution=resolution,tolerance=tolerance,unit=unit,$
    wavelength=wavelength,maxlength=maxlength
    if n_elements(line) eq 0 then line = ''
    if n_elements(name) eq 0 then name = ''
    if n_elements(invalue) eq 0 then return,line
    ind = where(finite(invalue),n_val)
    if n_val eq 0 then return,line=line+'NAN' else value = invalue[ind]
    if keyword_set(angle) or keyword_set(energy) then begin
       if keyword_set(energy) then begin
          if n_elements(resolution) eq 0 then resolution = 3
          if n_elements(unit) eq 0 then unit = ' meV'
       endif else begin
          if n_elements(resolution) eq 0 then resolution = 2
          if n_elements(unit) eq 0 then unit = string('b0'XB)  
       endelse
       val = round(value*((10d)^resolution)+(0.0001d),/L64)/((10d)^resolution)
       if n_val gt 1 then begin
          if val[0] gt val[1] then val = val[uniq(val,reverse(sort(val)))] $
          else val = val[uniq(val,sort(val))] 
          n_val = n_elements(val)
       endif 
       avg = mean(val)
       if n_elements(tolerance) eq 0 then tolerance = 2.0/((10d)^resolution)
       if max(abs(val-avg)) gt tolerance then begin
          line = line+dm_to_string(val[0],resolution=resolution)+(keyword_set(angle)?unit:'')+' to '+dm_to_string(val[n_val-1],resolution=resolution)+unit+$
                 ' ('+dm_to_string(n_elements(value))+(strlen(name) eq 0?'':' ')+name+(keyword_set(energy)?'':' angles') 
          if (n_val gt 2) and (n_val eq n_elements(value)) then begin
             step  = (val[n_val-1]-val[0])/(n_val-1.0)
             tmpan = val[0]+findgen(n_val)*step
             if max(abs(tmpan-val)) lt tolerance then $
                line = line+', step='+dm_to_string(step,resolution=resolution)+unit
          endif
          line = line+')'
       endif else line = line+dm_to_string(avg,resolution=resolution)+unit 
    endif else if keyword_set(temperature) then begin
       if n_elements(resolution) eq 0 then resolution = 2
       if n_elements(unit) eq 0 then unit = 'K'
       temp = round(value*((10d)^(resolution+1))+(0.0001d),/L64)/((10d)^(resolution+1))
       tavg = mean(temp)
       if max(abs(temp-tavg)) gt 2.0/((10d)^resolution) then begin
          mintemp = min(temp,imin,max=maxtemp,subscript_max=imax)
          if value[0] le value[n_elements(value)-1]+0.2 then $
             line = line+dm_to_string(mintemp,resolution=resolution)+' to '+dm_to_string(maxtemp,resolution=resolution)+' '+unit+', average=' $
          else $
             line = line+dm_to_string(maxtemp,resolution=resolution)+' to '+dm_to_string(mintemp,resolution=resolution)+' '+unit+', average='
       endif
       line = line+dm_to_string(tavg,resolution=resolution)+' '+unit
    endif else if keyword_set(time) then begin
       if n_elements(resolution) eq 0 then resolution = 2
       if value lt 60 then begin
          resolution = 1
          line = line+dm_to_string(value,resolution=resolution)+' second'
       endif else begin
          if value lt 1800 then begin
             value = value/60.
             line = line+dm_to_string(value,resolution=resolution)+' minute'
          endif else begin
             value = value/3600.
             if value gt 24. then begin
                value = value/24.
                line = line+dm_to_string(value,resolution=resolution)+' day'
             endif else begin
                line = line+dm_to_string(value,resolution=resolution)+' hour'
             endelse
          endelse
       endelse
       if round(value*((10.)^resolution),/L64)/((10.)^resolution) gt 1. then line = line+'s'
    endif else if keyword_set(wavelength) then begin
       if n_val ne 6*n_elements(line) then return,line
       value = reform(value,size(invalue,/dim))
       resl  = ['',' low','med','high']    
       for i=0,n_elements(line)-1 do begin
           wlen = self->wavlstrlen(value[0,i],string=wavl)
           if n_elements(maxlength) eq 1 then wavl = ((maxlength gt wlen)?string(bytarr(maxlength-wlen)+32b):'')+wavl
           slen = self->wavlstrlen(value[2,i],string=speed)
           if slen lt 10 then speed = string(bytarr(10-slen)+32b)+speed
           ch_srnumer = ([1,1>(value[4,i]-1)])[value[3,i]-1]
           line[i] = line[i]+wavl+' '+string('c5'XB)+'  '+resl[value[1,i]]+' res  '+speed+' rpm  '+dm_to_string(ch_srnumer)+'/'+dm_to_string(value[4,i])+'  tsdmin='+dm_to_string(value[5,i])
           if (value[0,i] eq 1) and (value[4,i] eq 1) and (value[5,i] eq 0) then line[i] = line[i]+' ???'
       endfor
    endif else begin
       if n_elements(resolution) eq 0 then resolution = 2
       if n_elements(tolerance)  eq 0 then tolerance = 2.0/((10d)^resolution)
       if n_elements(unit) eq 0 then unit = ''
       val = round(value*((10d)^resolution)+(0.0001d),/L64)/((10d)^resolution)
       if n_val gt 1 then begin
          if val[0] gt val[1] then val = val[uniq(val,reverse(sort(val)))] $
          else val = val[uniq(val,sort(val))] 
          n_val = n_elements(val)
       endif 
       avg = mean(val)
       if max(abs(val-avg)) gt tolerance then begin
          line = line+dm_to_string(val[0],resolution=resolution)+' to '+dm_to_string(val[n_val-1],resolution=resolution)+(strlen(unit) eq 0?'':' '+unit)+' ('+dm_to_string(n_elements(value))+' '+name
          if (n_val gt 2) and (n_val eq n_elements(value)) then begin
             step  = (val[n_val-1]-val[0])/(n_val-1.0)
             tmpan = val[0]+findgen(n_val)*step
             if max(abs(tmpan-val)) lt tolerance then $
                line = line+', step='+dm_to_string(step,resolution=resolution)
          endif
          line = line+')'
       endif else line = line+dm_to_string(avg,resolution=resolution)+(strlen(unit) eq 0?'':' '+unit)
    endelse
    return,line
end

;modified from IDL6.0 cw_filesel.pro
function dm_filesel::GetDirs,filter=filter,ftp=ftp
    compile_opt hidden, strictarr
    WIDGET_CONTROL,/HOURGLASS
    if keyword_set(ftp) then begin
       if ~obj_valid(self.ftpobj) then begin
          self.ftpobj = self->ftpobj()
          self.ftpobj->getproperty,currentdir=currentdir
          self.path = currentdir
          self->my_widget_control,'pathLab',set_value=self.path
       endif
       allFiles = self->ftplisting(i_dirs=i_dirs,i_file=i_file,n_dir=n_dir,n_file=n_file,count=count)
    endif else begin
       if !version.os_family eq 'unix' then $
          count = 0 $
       else $
          tmp = where((routine_info(/system,/function) eq 'FINDFILE'),count)
       if count eq 0 then $
          allFiles = call_function('File_SEARCH',COUNT=cnt,/mark_directory,/match_initial_dot,/nosort) $
       else $
          allFiles = call_function('FINDFILE',COUNT=cnt)    ;much faster, though obslete after IDL6.1
       dirs   = self->isDir(allFiles,/fast)
       i_dirs = where(dirs,n_dir,complement=i_file,ncomplement=n_file)   
    endelse
    dFiles = -1
    if n_dir gt 0 then begin
       dFiles = allFiles[i_dirs]
       if count ne 0 then begin    ;get rid of the . and .. directory returned by findfile
          dotdir = ['.','..']+self.pathsep[obj_valid(self.ftpobj)]
          notdotDir = where((dFiles ne dotdir[0]) and (dFiles ne dotdir[1]),nnotdot)
          if nnotdot gt 0 then dFiles = dFiles[notdotDir] else dFiles = -1
       endif
    endif
    if n_file gt 0 then begin
       files = allFiles[i_file]
       if n_elements(filter) eq 1 then begin
          if filter ne 'All Files' then begin
             if (filter ne 'Image Files') then begin
                mfilter = (STRPOS(filter, ',') ne -1) ? STRSPLIT(filter, ',', /EXTRACT) : filter
                oldFiles = files
                files = ''
                mfilter = STRUPCASE(STRTRIM(mfilter, 2))
                for i=0,N_ELEMENTS(mfilter)-1 do begin
                    test = WHERE(STRPOS(STRUPCASE(oldFiles), mfilter[i]) ge 0)
                    if (test[0] ne -1) then $
                       files = [files, oldFiles[test]]
                endfor
                files = (N_ELEMENTS(files) gt 1) ? files[1:*] : -1
             endif else begin
                ; This needs to query the image types
                for i = 0,N_ELEMENTS(files)-1 do begin
                    CATCH, errorStatus
                    if (errorStatus ne 0) then continue
                    if (~QUERY_IMAGE(files[i])) then $
                       files[i] = ''
                endfor
                CATCH, /CANCEL
                test = WHERE(files ne '')
                files = (test[0] ne -1) ? files[test] : -1
             endelse
          endif
       endif
       if size(files,/type) eq 7 then begin
          if size(dFiles,/type) eq 7 then dFiles=[dFiles,files] $
          else dFiles = files
       endif
    endif
    return,dFiles
end

;*******************************************************************
;get the vairous properties
;syntax:
;   filesel->getproperty,file=file,path=path,dFiles=dFiles,sep=sep
;arguments:
;   None
;keywords:
;   dates:        return date number [MACS cfx installed date, MACS mgf2 installed date, DCS 3He polarization implemented date]
;   dFiles:       return list of files in the selection window
;   eol:          flag for selected file reach end of list
;   file:         return selected files
;   ftpbufferdir: return ftp buffer directory
;   ftpobj:       ftp object
;   ftpserver:    ftp server string: 'https://ncnr.nist.gov' or 'ftp://ftp.ncnr.nist.gov'
;   limitfrom:    return from limit
;   limitto:      return to limit
;   ncnrftp:      return ncnrftp flag
;   path:         return path of selected files
;   sep:          return path separator
;*******************************************************************
pro dm_filesel::getproperty,dates=dates,dFiles=dFiles,eol=eol,file=file,ftpobj=ftpobj,ftpbufferdir=ftpbufferdir,ftpserver=ftpserver,limitfrom=limitfrom,limitto=limitto,$
    ncnrftp=ncnrftp,path=path,sep=sep,selected=selected,sortfile=sortfile
    if arg_present(file) or arg_present(selected) or arg_present(eol) then begin
       if widget_info(self.tlb,/valid_id) then begin
          ;the following codes is to prevent the misbehaviour of widget_list when selection is not in focus
          selected = widget_info(self.fileList,/list_select) & tmp=where(selected ge 0,n_item)
          if n_item gt 0 then begin
             isDir = self->isDir((*self.dFiles)[selected],/fast)
             self->set_file   ;clear file pointer
             index = where(isDir eq 0,count)
             if count ne 0 then self->set_file,selected[index]
          endif else tmp = temporary(selected)
       endif
       if ptr_valid(self.file) then begin
          file = strtrim(*self.file,2)
          if n_elements(file) eq 1 then file = file[0]
          tmp = where(file eq (*self.dFiles)[-1],eol)
       endif else eol = 0b
    endif
    if arg_present(ftpbufferdir) then ftpbufferdir = self.bufferdir
    if arg_present(ftpobj) then     ftpobj    = self.ftpobj
    if arg_present(ftpserver) then  ftpserver = self.ftpname[0]
    if arg_present(path) then       path      = strtrim(self.path,2)
    if arg_present(dFiles) then $
       if ptr_valid(self.dFiles) then begin
          dFiles = strtrim(*self.dFiles,2)
          if n_elements(dFiles) eq 1 then dFiles=dFiles[0]
       endif
    if arg_present(sep) then        sep       = self.pathsep[obj_valid(self.ftpobj)]
    if arg_present(limitfrom) then  limitfrom = self.limitfrom
    if arg_present(limitto) then    limitto   = self.limitto
    if arg_present(ncnrftp) then    ncnrftp   = self.ncnrftp
    if arg_present(sortfile) then   sortfile  = self.sortfile
    if arg_present(dates) then      dates     = self.dates
end

;*******************************************************************
;check whether a path is a valid directory
;syntax:
;   result=filesel->isDir(path,/showmesg)
;arguments:
;   path: a string of the path to be checked
;keywords:
;   nomesg: if set the error message is not displayed
;*******************************************************************
function dm_filesel::isDir,path,showmesg=showmesg,fast=fast
    nitem  = n_elements(path)
    if nitem eq 0 then return,0b
    if keyword_set(fast) then return,strpos(path,self.pathsep[obj_valid(self.ftpobj)]) ne -1
    if obj_valid(self.ftpobj) then begin
       result = bytarr(nitem)
       for i=0,nitem-1 do begin
           if path[i] ne self.ftpname[0] then begin
              self.ftpobj->setproperty,currentdir=path[i]
              catch, myerror
              if myerror ne 0 then begin
                 catch,/cancel
                 result[i] = 0b
                 continue
              endif
              tmp = self->ftplisting()
              result[i] = 1b
           endif 
       endfor
    endif else begin
       result = file_test(path,/directory)
    endelse
    index = where(path eq self.ftpname[0],count)
    if count eq 1 then begin
       if ~obj_valid(self.ftpobj) then self.ftpobj = self->ftpobj()
       if obj_valid(self.ftpobj) then result[index] = 1b
    endif
    if keyword_set(showmesg) then begin
       index = where(result eq 0,count)
       if count ne 0 then begin
          mesg = path[index]+' is not a valid directory.'
          if widget_info(self.tlb,/valid_id) then $
             ok = dialog_message(mesg,dialog_parent=self.tlb,/center) $
          else $
             ok = dialog_message(mesg,/center)
       endif
    endif
    return,(nitem eq 1)?result[0]:result
end

;a widget_control use uname instead of id as identifier
pro dm_filesel::my_widget_control,uname,base,_extra=extra
    if n_elements(base) eq 0 then base=self.tlb
    if ~widget_info(base,/valid_id) then base=self.tlb
    if n_elements(extra) ne 0 then begin
       for i=0,n_elements(uname)-1 do begin
           wid = widget_info(base,find_by_uname=uname[i])
           if wid ne 0 then widget_control,wid,_extra=extra
       endfor
    endif
end

pro dm_filesel::refresh
    self->set_path,ftp=obj_valid(self.ftpobj)
    self->getproperty,file=file,dFiles=dFiles
    if n_elements(file) eq 0 then self->clear_file,/bottom $
    else begin
       for i=0,n_elements(file)-1 do begin
           ind = where(dFiles eq file[i],count)
           if count ne 0 then begin
              if n_elements(selected) eq 0 then selected=ind $
              else selected=[selected,ind]
           endif
       endfor
       if n_elements(selected) gt 0 then self->set_file,selected,/highlight else self->clear_file,fileloc=file[0]
    endelse
end

;*******************************************************************
;setting self.file , not supposed to be called by outside user
;syntax:
;   filesel->set_file,[selected],filename=filename
;arguments:
;   selected:   an array of index
;keywords:
;   filename:   a string
;   noupdate:   if not set and if self.write eq 1 then (*self.file) will be shown in saveas textbox
;*******************************************************************
pro dm_filesel::set_file,selected,filename=filename,noupdate=noupdate,highlight=highlight
    ptr_free,self.file
    if n_elements(selected) eq 0 then begin
       if n_elements(filename) eq 1 then begin
          filename = strtrim(filename,2)
          if strlen(filename) ne 0 then begin
             tmp = where(['all files','*.*','.*'] eq strlowcase(self.filter),count)
             if count eq 0 then begin
                tmp1 = strsplit(self.filter,'.',/extract)
                tmp1 = strlowcase(tmp1[n_elements(tmp1)-1])    ;file type
                tmp2 = strsplit(filename,'.',/extract) & count = n_elements(tmp2)
                if (count le 1) and (strlen(tmp2[0]) ne 0) then filename = tmp2[0]+'.'+tmp1
             endif
             self.file = ptr_new(filename)
             widget_control,self.fileList,set_list_select=-1
          endif
       endif
    endif else begin
       if ptr_valid(self.dFiles) then begin
          self.file = ptr_new((*self.dFiles)[selected])
          if keyword_set(highlight) then begin
             widget_control,self.fileList,set_list_select=selected,set_list_top=selected[0]
          endif
       endif
    endelse
    if self.write and (~ keyword_set(noupdate)) then begin
       if ptr_valid(self.file) then tmp=*self.file else tmp=''
       self->my_widget_control,'saveFile',set_value=tmp
    endif
end

;set filter
pro dm_filesel::set_filter,filter,path=path,sortbynumber=sortbynumber,sortbyname=sortbyname,keepcurrentdir=keepcurrentdir
    if keyword_set(sortbynumber) then self.sortfile = 0
    if keyword_set(sortbyname)   then self.sortfile = 1
    if size(filter,/type) ne 7 then return
    self.filter = strtrim(filter[0],2)
    if (self.filter eq '.*') or (self.filter eq '*.*') or (self.filter eq '*') or (strlen(self.filter) eq 0) then begin
       self.filter = 'All Files'
    endif
    if strupcase(self.filter) ne 'ALL FILES' and strupcase(self.filter) ne 'IMAGE FILES' then begin
       if strmid(self.filter,0,1) ne '.' then self.filter = '.'+self.filter
       if strlowcase(self.filter) eq '.dcs.gz' then self.filter = '.dcs'
       if (strlowcase(self.filter) eq '.ng0') or (strlowcase(self.filter) eq '.bt9') then self.filter='.ng0,.bt9'  ;special case of MACS files
       tmp = stregex(self.filter,'([, ] *)[^.]+',length=len,/subexpr)
       while len[1] ne -1 do begin
             self.filter = strmid(self.filter,0,tmp[0])+',.'+strmid(self.filter,tmp[1]+len[1])         
             tmp = stregex(self.filter,'([, ] *)[^.]+',length=len,/subexpr)
       endwhile
    endif
    if self.ncnrftp and (self.filter ne 'All Files') then resetbuffer = ~stregex(self.filter,'dcs|b[ta-d][0-9]|cg[d0-9]|ng[0-9]|hfbs',/fold_case,/boolean) ;disable ncnrftp for non-NIST file types
    case strlowcase(self.filter) of
        '.dcs':           self.basedir = '/pub/ncnrdata/dcs/'
        '.ng0,.bt9':      self.basedir = '/pub/ncnrdata/macs/'
        '.ng0,.bt9,.log': self.basedir = '/pub/ncnrdata/macs/'
        else:             self.basedir = '/pub/ncnrdata/'
    endcase
    if self.ncnrftp and ~keyword_set(resetbuffer) and (self.idlversion ge 6.4) and ~obj_valid(self.ftpobj) then begin ;check if ftp server is working
       if ~file_test(self.bufferdir,/directory,/write) then resetbuffer = 1b else begin
          tmp1 = self->ftpobj(ftpworking=ftpworking)
          resetbuffer = ~ftpworking
          obj_destroy,tmp1
       endelse
    endif
    if obj_valid(self.ftpobj) and ~keyword_set(keepcurrentdir) then self.ftpobj->setproperty,currentdir=self.basedir
    self->set_path,path,ftp=obj_valid(self.ftpobj)
    self->clear_file
    if keyword_set(resetbuffer) then self->set_ftpbuffer,''
end

;*******************************************************************
;set ftp buffer directory
;syntax:
;   filesel->set_ftpbuffer,bufferdir
;arguments:
;   bufferdir: string of the new path to be set, if not a valid writable path, disable ftp
;*******************************************************************
pro dm_filesel::set_ftpbuffer,bufferdir
    if n_elements(bufferdir) eq 0 then return
    if ~file_test(bufferdir[0],/directory,/write) then begin
       if self.ncnrftp then begin ;disable ncnr ftp
          self.ncnrftp = 0b
          driPull = widget_info(self.tlb,find_by_uname='driPull')
          if driPull ne 0 then begin
             widget_control,self.tlb,update=0
             select = widget_info(driPull,/DROPLIST_SELECT)
             drives = get_drive_list(count=count)
             if select eq n_elements(drives) then self.path = ''
             widget_control,driPull,set_value=drives,set_droplist_select=select<(n_elements(drives)-1)
             olddrivedir = (*self.drivedir)
             ptr_free,self.drivedir
             self.drivedir = ptr_new([[drives],[strarr(n_elements(drives))]])
             for i=0,n_elements(drives)-1 do begin
                 ind = where(olddrivedir[*,0] eq drives[i],count)
                 if count ne 0 then (*self.drivedir)[i,1] = olddrivedir[ind[0],1]
             endfor
             self->set_path
             widget_control,self.tlb,/update
          endif else begin
             if self->atTop(self.path) or stregex(self.path,self.ftpname[1],/boolean) then self->set_path
          endelse
       endif
       return
    endif
    self.bufferdir = bufferdir[0]
    if ~self.ncnrftp then begin
       tmp = self->ftpobj(ftpworking=ftpworking)   
       self.ncnrftp = ftpworking
       obj_destroy,tmp
       if self.ncnrftp then begin
          driPull = widget_info(self.tlb,find_by_uname='driPull')
          if driPull ne 0 then begin
             widget_control,self.tlb,update=0
             select = widget_info(driPull,/DROPLIST_SELECT)
             drives = get_drive_list(count=count)
             drives = [drives,self.ftpname[0]]
             widget_control,driPull,set_value=drives,set_droplist_select=select
             olddrivedir = (*self.drivedir)
             ptr_free,self.drivedir
             self.drivedir = ptr_new([[drives],[strarr(n_elements(drives))]])
             for i=0,n_elements(drives)-1 do begin
                 ind = where(olddrivedir[*,0] eq drives[i],count)
                 if count ne 0 then (*self.drivedir)[i,1] = olddrivedir[ind[0],1] 
             endfor
             widget_control,self.tlb,/update
          endif else begin
             if self->atTop(self.path) then self->set_path
          endelse
       endif
    endif
end

;*******************************************************************
;go to a specified path
;syntax:
;   filesel->set_path,newpath
;arguments:
;   newapth: string of the new path to be set
;keywords:
;   ftp:          if set, newpath is a ftp directory
;   setpathevent: set the internal setpathevent parameter
;*******************************************************************
pro dm_filesel::set_path,newpath,ftp=ftp,setpathevent=setpathevent
    driPull = widget_info(self.tlb,find_by_uname='driPull')
    if n_elements(newpath) eq 0 then newpath=''
    if n_elements(setpathevent) ne 0 then self.setpathevent = keyword_set(setpathevent)
    if stregex(newpath,'^(https://|ftp://ftp\.)ncnr.nist.gov',/boolean,/fold_case) then begin
       newpath = strmid(newpath,strlen((['https://ncnr.nist.gov','ftp://ftp.ncnr.nist.gov'])[stregex(newpath,'^ftp',/boolean,/fold_case)]))
       ftp = 1b
    endif
    sendevent = ((newpath ne '') and self.setpathevent and (self.group_leader ne 0))
    if (driPull eq 0) and ((newpath eq self.ftpname[1]) or (newpath eq self.pathsep[0]+self.ftpname[1])) then begin
       newpath = self.basedir
       ftp = 1b
    endif
    if keyword_set(ftp) then begin
       if ~self.ncnrftp then return
       if ~obj_valid(self.ftpobj) then self.ftpobj = self->ftpobj()
       if strlen(newpath) eq 0 then self.ftpobj->getproperty,currentdir=newpath
    endif else begin
       obj_destroy,self.ftpobj
    endelse
    widget_control,self.tlb,update=0
    if (strlen(newpath[0]) ne 0) and (newpath[0] ne self.path) then begin
       if keyword_set(ftp) and strmid(newpath,0,1,/reverse_offset) ne '/' then newpath[0] = newpath[0]+'/'
       if (self->isDir(newPath[0],/showmesg)) then self.path=newpath[0]
    endif
    self->my_widget_control,'pathLab',set_value=self.path
    atTop = self->atTop(self.path,ftp=ftp)
    if atTop and (driPull eq 0) then ftp=0b
    self->my_widget_control,'upBut',sensitive=1-atTop
    if keyword_set(ftp) then self.ftpobj->setproperty,currentdir=self.path else cd,self.path
    ptr_free,self.dFiles
    dFiles = self->GetDirs(filter=self.filter,ftp=ftp)
    if (size(dFiles,/type) ne 7) then $  ;empty directory
       widget_control,self.fileList,set_value='' $ 
    else begin
       dFiles   = dFiles[sort(dFiles)]
       isDir    = self->isDir(dFiles,/fast)
       ind_file = where(isDir eq 0,nfile,complement=ind_dir,ncomplement=ndir)
       if ndir ne 0 then    dir = dFiles[ind_dir]   ;directories
       if nfile ne 0 then files = dFiles[ind_file]
       if self.limitfile then begin
          if nfile ne 0 then begin
             if self.limitfrom ne '' then begin
                ind = where(files ge self.limitfrom,count)
                if count ne 0 then files=files[ind] else tmp=temporary(files)
             endif
             if (self.limitto ne '') and (n_elements(files) ne 0) then begin
                ind = where(files le self.limitto+'~',count)
                if count ne 0 then files=files[ind] else tmp=temporary(files)
             endif
          endif
       endif
       if (driPull eq 0) and self.ncnrftp and atTop then begin
          index = where(dir eq self.ftpname[1],count)
          if count gt 0 then self.ftpname[1] = 'NCNR-ftp/'  ;make sure that the directory has not existed already
          dir = [dir,self.ftpname[1]]
          obj_destroy,self.ftpobj
       endif
       n_files = n_elements(files)
       if n_files ne 0 then begin
          ;sort files according the the ending number IDs
          if (strlowcase(self.filter) ne '.dcs') and (~self.sortfile) then begin
             for i=0L,n_files-1 do begin
                 len = 0 & tmp = files[i]
                 while((len lt 3) and (strpos(tmp,'.',/reverse_search) ge 0)) do begin
                      if tmp eq strmid(tmp,strpos(tmp,'.',/reverse_search)) then tmp = ''
                      tmp  = self->filebasename(tmp,/reverse_search)
                      tmp1 = stregex(tmp,'[0-9]+$',len=len)
                 endwhile
                 if len ge 3 then begin
                    tmp1 = strmid(files[i],strlen(tmp))
                    break
                 endif
             endfor
             if len ge 3 then begin
                tmpfiles = self->filebasename(files,extension=tmp1)
                ;firgure out the ID length
                minlen = 0
                for i=0L,n_files-1 do begin
                    tmp = stregex(tmpfiles[i],'[0-9]+$',len=len)
                    if len gt 1 then minlen = (minlen eq 0?100:minlen)<len
                endfor
                tmpzero = strmid('0000000000',0,minlen)
                for i=0L,n_files-1 do begin
                    tmp = stregex(tmpfiles[i],'[0-9]+$',len=len)
                    if len ge minlen then tmpfiles[i] = strmid(tmpfiles[i],tmp+len-minlen,minlen)+tmpfiles[i] $
                    else tmpfiles[i] = tmpzero+tmpfiles[i]
                endfor
                files = files[sort(tmpfiles)]
             endif
          endif
          if n_elements(dir) ne 0 then dir=[dir,files] else dir=files
       endif
       if n_elements(dir) ne 0 then dFiles=dir else tmp=temporary(dFiles)
       if n_elements(dFiles) ne 0 then begin
          self.dFiles = ptr_new(dFiles)
          widget_control,self.fileList,set_value=dFiles
       endif else begin
          widget_control,self.fileList,set_value=''
       endelse
    endelse
    if driPull ne 0 then begin
       drives = get_drive_list()
       if self.ncnrftp then drives = [drives,self.ftpname[0]]
       old_drive = widget_info(driPull,/droplist_select)
       widget_control,driPull,set_value=drives
       olddrivedir = (*self.drivedir)
       ptr_free,self.drivedir
       self.drivedir = ptr_new([[drives],[strarr(n_elements(drives))]])
       for i=0,n_elements(drives)-1 do begin
           ind = where(olddrivedir[*,0] eq drives[i],count)
           if count ne 0 then (*self.drivedir)[i,1] = olddrivedir[ind[0],1] 
       endfor
       if keyword_set(ftp) then begin
          widget_control,driPull,set_droplist_select=n_elements(drives)-1 
          (*self.drivedir)[n_elements(drives)-1,1] = self.path
       endif else begin
          case !version.os_family of
              'Windows':drive_loc = (WHERE(STRUPCASE(STRMID(self.path,0,STRPOS(self.path,'\')+1)) eq STRUPCASE(drives)))[0]
              'MacOS'  :drive_loc = (WHERE(STRMID(self.path,0,STRPOS(self.path,':')+1) eq drives))[0]
          endcase
          if drive_loc eq -1 then begin
             drive_loc = old_drive
             if self->isDir((*self.drivedir)[drive_loc,strlen((*self.drivedir)[drive_loc,1]) ne 0]) then $
                self->set_path,(*self.drivedir)[drive_loc,strlen((*self.drivedir)[drive_loc,1]) ne 0]
             widget_control,self.tlb,/update
             return
          endif else begin
             widget_control,driPull,set_droplist_select=drive_loc
             (*self.drivedir)[drive_loc,1] = self.path
          endelse
       endelse
    endif
    widget_control,self.tlb,/update
    if sendevent then widget_control,self.group_leader,send_event={dm_filesel_setpath,ID:self.tlb,TOP:self.group_leader,HANDLER:self.tlb,object:self},/no_copy
end

;*******************************************************************
;set internal parameters
;syntax:
;   filesel->setproperty,inxerev=inxerev,workdir=workdir
;keywords:
;   inxerev: keyword for INX data file energy transfer convention 0-neutron energy gain/loss 1-sample energy gain/loss
;   workdir: string for the new writable workdir
;*******************************************************************
pro dm_filesel::setproperty,inxerev=inxerev,workdir=workdir
    if n_elements(inxerev) ne 0 then self.inx_erev = keyword_set(inxerev)
    if (n_elements(workdir) ne 0) && file_test(workdir[0],/directory,/write) then self.workdir = workdir[0]
end

;*******************************************************************
;destroy the object and widget hierarchy, life cycle method
;syntax:
;   filesel->destroy
;arguments:
;   None
;keywords:
;   None
;*******************************************************************
pro dm_filesel::Cleanup
    cd,self.root
    if widget_info(self.extbase,/valid_id) then widget_control,self.extbase,/destroy
    if widget_info(self.tlb,/valid_id) then widget_control,self.tlb,/destroy
    ptr_free,self.dFiles,self.file,self.drivedir
    obj_destroy,self.ftpobj
end

;*******************************************************************
;class constructor, life cycle method
;
;syntax:object constructor,called automatically
;parameter:
;   None
;keywords:
;   groupleader:    groupleader id
;   parent:         parent ID of this object, if present, it looks like a widget component
;   pobj:           parent object, used for certain file header infos
;   path:           a string of starting path, default is current directory
;   multi:          if set, allow multiple file choice, only in opening files, default is 1
;   write:          if set, the fileselction is for saving files, defalut is 0
;   filter:         file selection filter, default is 'All Files'
;   frame:          if set, there is a frame around the widget
;   xsize:          xsize of the list widget
;   ysize:          row number of the filelist widget
;   limitfile:      if set, allow restrictions on start and end of file
;   limitfrom:      limit from
;   limitto:        limit to
;   ncnrftp:        if set, allow access to ncnr ftp site https://ncnr.nist.gov/pub/ncnrdata/ (previoulsy ftp://ftp.ncnr.nist.gov/pub/ncnrdata/)
;   bufferdir:      a string of the buffer directory to save the ftp files
;   workdir:        a writable working direcotory, only used for dm_copyftp
;   selectevent:    if set, an event will be sent back to group_leader when a file is selected
;   setpathevent:   if set, an event will be sent back to group_leader when a new path is select from the GUI
;*******************************************************************
function dm_filesel::Init,parent=parent,pobj=pobj,group_leader=group_leader,path=path,multi=multi,write=write,$
    filter=filter,frame=frame,ysize=ysize,ypad=ypad,xpad=xpad,xsize=xsize,limitfile=limitfile,inxerev=inxerev,$
    limitfrom=limitfrom,limitto=limitto,selectevent=selectevent,setpathevent=setpathevent,sortevent=sortevent,ncnrftp=ncnrftp,bufferdir=bufferdir,workdir=workdir 
    self.idlversion = float(!version.release)
    if n_elements(multi) eq 0 then self.multi=(1b) else self.multi=keyword_set(multi)
    self.inx_erev  = keyword_set(inxerev)
    self.write     = keyword_set(write)
    self.limitfile = keyword_set(limitfile)
    if self.write then begin
       self.multi = 0b
       self.limitfile = 0b
    endif
    self.selectevent = keyword_set(selectevent)
    self.sortevent = keyword_set(sortevent)
    if n_elements(filter)    eq 0 then self.filter='All Files' else self.filter=strtrim(filter[0],2)
    if n_elements(ysize)     eq 0 then ysize=8
    if n_elements(xsize)     eq 0 then xsize=220
    if n_elements(ypad)      eq 0 then ypad=1
    if n_elements(xpad)      eq 0 then xpad=3
    if n_elements(space)     eq 0 then space=2
    if n_elements(limitfrom) ne 0 then self.limitfrom=strtrim(limitfrom[0],2)
    if n_elements(limitto)   ne 0 then self.limitto=strtrim(limitto[0],2)
    if (n_elements(workdir)  ne 0) && file_test(workdir[0],/directory,/write) then self.workdir = workdir[0]
    ;[MACS cfx installed date, MACS mgf2 installed date, DCS 3He polarization implemented date]
    self.dates = [dm_to_number('9/14/2015',/date),dm_to_number('9/14/2115',/date),dm_to_number('12/14/2022',/date)] 
    cd,current = currdir
    self.path  = currdir
    self.root  = currdir
  
    ;check for ftp server
    self.ftpname = ['https://ncnr.nist.gov','NCNR_ftp/']
    self.basedir = '/pub/ncnrdata/'
    self.bufferdir = dm_define_pointer(/gettempdir)
    if keyword_set(ncnrftp) and (self.idlversion ge 6.4) then begin
       if (n_elements(bufferdir) eq 0) || ~file_test(bufferdir[0],/directory,/write) then bufferdir = self.bufferdir
       if file_test(bufferdir,/directory,/write) then begin
          self.bufferdir = bufferdir
          tmp1 = self->ftpobj(ftpworking=ftpworking)
          obj_destroy,tmp1
          self.ncnrftp = ftpworking
       endif
    endif

    ;error handling
    catch, anyerror
    if anyerror ne 0 then begin
       catch,/cancel
       ok = dialog_message(!error_state.msg ,/error,dialog_parent=group_leader,/center)
       return,0
    end
    
    self.maxline = 30
    if self.idlversion ge 5.6 then begin
       scr_size = get_screen_size()
       self.maxline = self.maxline>((scr_size[1]-100)/18)
    endif

    if n_elements(frame) eq 0 then frame=0
    if n_elements(pobj)  ne 0 then self.pobj = pobj
    if n_elements(parent) eq 0 then begin
       self.extbase = widget_base(group_leader=group_leader,map=0)
       self.tlb = widget_base(group_leader=self.extbase,/column,frame=frame,event_pro='dm_filesel_event',uname='filesel',$
            title='Please Select '+((self.multi eq 1)?'Files':'a File'),/modal,xpad=xpad,ypad=ypad,space=space)
    endif else begin
       if n_elements(group_leader) ne 0 then self.group_leader = group_leader else self.group_leader = parent
       self.tlb = widget_base(parent,/column,frame=frame,event_pro='dm_filesel_event',uname='filesel',xpad=xpad,ypad=ypad,space=space)
    endelse
    
    self.upDir = '..'
    ;check operation systems
    self.pathsep = [dm_define_pointer(/getpathsep),'/']
    case !VERSION.OS_FAMILY of
         'Windows': begin
                    driveVol  = 'Drive:'
                    drives    = get_drive_list()
                    if self.ncnrftp then drives = [drives,self.ftpname[0]]
                    end
         'MacOS':   begin
                    self.upDir = '::'
                    driveVol   = 'Volume:'
                    drives     = get_drive_list()
                    if self.ncnrftp then drives = [drives,self.ftpname[0]]
                    end
         'vms':     begin
                    self.upDir = '[-]'
                    driveVol   = ''
                    end
         else:      driveVol = ''
    endcase
    space = (widget_info(self.tlb,/geometry)).space
    if (driveVol ne '') then begin
       row0  = widget_base(self.tlb,/row,xpad=xpad,ypad=ypad,space=space)
       label = widget_label(row0,value=driveVol,/align_center)
       dlist = widget_droplist(row0,value=self.ftpname[0],uname='driPull',/dynamic_resize)
       self.drivedir = ptr_new([[drives],[strarr(n_elements(drives))]])
    endif
    row = widget_base(self.tlb,/row,xpad=xpad,ypad=0,space=space)
    defsysv,'!DAVE_AUXILIARY_DIR',exists=exists
    if exists then bitmap_filename = !DAVE_AUXILIARY_DIR+'refresh.bmp' $
    else bitmap_filename = file_which('refresh.bmp',/include_current_dir)
    refreshBut = widget_button(row,value=bitmap_filename,uname='refreshBut',/bitmap,/dynamic_resize)
    geom1 = widget_info(refreshBut,/geometry)
    if exists then bitmap_filename = !DAVE_AUXILIARY_DIR+'up1lvl.bmp' $
    else bitmap_filename = file_which('up1lvl.bmp',/include_current_dir)
    upBut = widget_button(row,value=bitmap_filename,uname='upBut',/bitmap,/dynamic_resize)
    geom  = widget_info(upBut,/geometry)
    if self.idlversion ge 5.6 then begin
       widget_control,refreshBut,tooltip='Refresh'
       widget_control,upBut,tooltip='Up One Directory'
    endif
    tmp  = geom.scr_xsize+geom1.scr_xsize+2*geom.margin+3*(xpad+space)+space
    if n_elements(label) ne 0 then begin
       widget_control,label,scr_xsize=geom.scr_xsize+geom1.scr_xsize+geom.margin+xpad+space
       geom  = widget_info(row0,/geom)
       xsize = (xsize > geom.scr_xsize)
       widget_control,dlist,set_value=drives
    endif
    void = widget_text(row,value=self.path,scr_xsize=xsize-tmp,ysize=1,/editable,/kbrd_focus_events,uname='pathLab')
    if self.limitfile then begin
       row   = widget_base(self.tlb,/row,xpad=xpad,ypad=ypad,space=space)
       label = widget_label(row,value='from:  ')
       geom  = widget_info(label,/geometry)
       tmp   = geom.scr_xsize+2.*geom.margin+(xpad+space)+space
       void  = widget_text(row,value=self.limitfrom,scr_xsize=xsize-tmp,ysize=1,/editable,uname='limitFrom',/all_events,/kbrd_focus_events)
       row   = widget_base(self.tlb,/row,xpad=xpad,ypad=ypad,space=space)
       label = widget_label(row,value=' to:   ',scr_xsize=geom.scr_xsize)
       void  = widget_text(row,value=self.limitto,scr_xsize=xsize-tmp,ysize=1,/editable,uname='limitTo',/all_events,/kbrd_focus_events)
    end
    self.fileList = widget_list(self.tlb,uname='fileList',multiple=self.multi,scr_xsize=xsize,/context_events)

    self->set_filter,self.filter,path=path
    self.setpathevent = keyword_set(setpathevent)
    if self.write then begin
       row   = widget_base(self.tlb,/row,xpad=xpad,ypad=ypad,space=space)
       label = widget_label(row,value='save as:  ')
       geom  = widget_info(label,/geometry)
       tmp   = geom.scr_xsize+xpad+2*space+2*geom.margin
       void  = widget_text(row,scr_xsize=xsize-tmp,ysize=1,/editable,uname='saveFile',/all_events,/kbrd_focus_events)
    endif
    widget_control,self.fileList,ysize=ysize
    geom = widget_info(self.fileList,/geometry)
    if n_elements(parent) eq 0 then begin
       row   = widget_base(self.tlb,/row,/align_center,xpad=xpad,ypad=ypad,space=space)
       void  = widget_button(row,value=(self.write eq 1)?'Save':'Open',uname='okBut')
       void  = widget_label(row,value='',xsize=15)
       void  = widget_button(row,value='Cancel',uname='cancelBut')
       geom  = widget_info(row,/geom)
       dm_center_kid,self.tlb,group_leader
       widget_control,self.tlb,/realize
    end
    widget_control,self.tlb,set_uvalue=self
    if n_elements(parent) eq 0 then xmanager,'dm_filesel',self.tlb,cleanup='dm_filesel_Exit'
    return,1
end

;*******************************************************************
;filesel class definition
;*******************************************************************
pro dm_filesel__define
    void={dm_filesel          ,$    ;the object class name
       idlversion:          0e,$    ;idl version number
       group_leader:        0L,$    ;group_leader or 0L for the modal widget
       pobj:         obj_new(),$
       extbase:             0L,$    ;only used in modal widget
       tlb:                 0L,$    ;identifier of top level base
       fileList:            0L,$    ;save the base ID for the context sensitive menus
       dFiles:       PTR_NEW(),$    ;diretory listing
       file:         PTR_NEW(),$    ;selected files
       drivedir:     PTR_NEW(),$    ;save windows drive and starting directory list
       selectevent:         0b,$    ;if set,send select event {dm_filesel_select,ID:self.tlb,TOP:self.group_leader,HANDLER:self.tlb,object:self}
       setpathevent:        0b,$    ;if set,send set path event {dm_filesel_setpath,ID:self.tlb,TOP:self.group_leader,HANDLER:self.tlb,object:self}
       sortevent:           0b,$    ;if set,send sort file event {dm_filesel_sortfile,ID:self.tlb,TOP:self.group_leader,HANDLER:self.tlb,object:self}
       path:                '',$    ;current directory
       root:                '',$    ;starting directory
       pathsep:        ['',''],$    ;path separator, [0] for regular files, [1] for ftp
       filter:              '',$    ;filter
       upDir:               '',$    ;up directory
       sortfile:            0b,$    ;0-sort by file number 1-sort by file name
       multi:               0b,$    ;multiple file selection flag,not allowed if write is set
       write:               0b,$    ;for saving files
       limitfile:           0b,$    ;for limitting file selection range
       limitfrom:           '',$    ;limit file selection
       limitto:             '',$    ;limit file selection
       ncnrftp:             0b,$    ;flag for ncnr ftp site
       workdir:             '',$    ;working directory
       bufferdir:           '',$    ;ftp buffer directory
       basedir:             '',$    ;ftp base directory
       ftpname:        ['',''],$    ;ftp name
       ftpobj:       obj_new(),$    ;daveftpurl object
       macs_specify:        '',$    ;save macs specify field
       fnum_default:        '',$    ;save default file range number
       dates:      lon64arr(3),$    ;special date number [MACS cfx installed date, MACS mgf2 installed date, DCS 3He polarization implemented date]
       displayid:           0L,$    ;save the xdisplay wid
       inx_erev:            0b,$    ;flag for reverse the energey transfer definition in INX files 0:neutron energy gain/loss 1:sample energy gain/loss, for INX type only
       maxline:             0s $    ;maximum number of lines in display
    }
end