; $Id: dm_calc_shielding.pro,v 1.11 2017/07/11 14:32:13 ymqiu Exp $
;#######################################################################
;
; NAME:
;  dm_calc_shielding
;
; PURPOSE:
;  this program calculates the absorption correction
;  annular, cylindrical, spherical, and flat plate geometry are implemented
;
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  June, 2022
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

; calculate the traveling path in an annular or cylindrical or speherical geometry
; equatorial scattering only if det_psi is not specified, 
; the cylinder and annulus sample is much taller than the beam height, and the sphere is smaller than the beam height
; input:
;   type:    0-cylinder
;            1-annulus
;            2-sphere
;   tth:     scalar or array of scattered angle in degree, tth=0 for north in DCS   (+:cw, equatorial value only)
;   r:       radius of the sampling point
;   phi:     angle of the sampling point in cylindrical or spherical coordinate system,
;            phi=0 for north in DCS (+:ccw)  range:[0,360)
;   psi:     angle of the sampling point in spherical coordinate range:[0,180)
;   rin:     inner radius of the annulus
;   rout:    outer radius of the annulus or the radius of the cylinder or the radius of the sphere
;   det_psi: detector psi angle in radian (dcs definition), if specified and equal to the number of tth, calculate all direction scattering
;   is_dcs:  if set, consider the beam width profile(3 cm wide, 1.5 cm center)
;   orc:     if set, calculate the effect of DCS oscillating radial collimator, results are saved in passrc
; output:
;   l1:      incoming path, same number of elements as that of tth
;   l2:      outgoing path, same number of elements as that of tth
;   cb:      1: incoming beam is in the 1.5 central region
;            0: outside that region
;   passrc:  flag indicating whether the neutron will pass the DCS radial collimator,1-pass 0-stopped
;            same number of elements as that of tth
;            assuming the sample and radial collimator are concentric and the center is at the beam center
pro dm_shielding_calcl1l2,tth,type,r,phi,psi,rin=rin,rout=rout,l1=l1,l2=l2,cb=cb,is_dcs=is_dcs,det_psi=det_psi,orc=orc,passrc=passrc
    ntth   = n_elements(tth)
    l2     = dblarr(ntth)
    l1     = dblarr(ntth)
    passrc = intarr(ntth)+1
    if ntth eq 1 then begin
       l2 = l2[0] & l1 = l1[0] & passrc = passrc[0]
    endif
    cb = 0b

    if r gt rout then return  ;outside the sample region
    
    tmp  = phi*!ddtor
    if type eq 2 then begin   ;sphere
       tmp1 = psi*!ddtor
       rp = r*sin(tmp1)
       x  = rp*cos(tmp)       ;sampling point in cartesian coordinates
       y  = rp*sin(tmp)
       z  = r*cos(tmp1)
       rp = sqrt(rout^2-z^2)
    endif else begin          ;cylinder & annulus
       x  = r*cos(tmp)        ;sampling point in cartesian coordinates
       y  = r*sin(tmp)
       rp = rout
    endelse
    x0 = -sqrt(rp^2-y^2)      ;entry intercept point in cartesian coordinates

    if (~ keyword_set(is_dcs)) or (abs(y) le 0.75) then cb=1b
    
    l1[*] = x-x0

    if type eq 1 then $
       if (x gt 0) and (abs(y) lt rin) then $  ;intercept the inner circle
          l1[*]  =   l1[*]-2*sqrt(rin^2-y^2)

    ;rotate the cartesian coordinate system by tth degree cw
    tmp    = tth*!ddtor
    sintth = sin(tmp)
    costth = cos(tmp)
    ;new coordinates for the sampling point
    xp = x*costth-y*sintth
    yp = x*sintth+y*costth
    
    if keyword_set(orc) and keyword_set(is_dcs) then begin
       delta     = 0.0349066       ;=2*!dtor; DCS radial collimator 2-degree separation
       rrc       = 20.0            ;inside radius of the radial collimator
       tmp       = asin(abs(yp)/rrc)
       passrc[*] = (tmp le delta)
    endif
    
    if (n_elements(det_psi) eq ntth) and (type eq 2) then begin ;sphere with det_psi
       ;rotate one more time by det_psi radian cw 
       rp     = sqrt(rout^2-yp^2)
       yp1    = -z
       sinpsi = sin(det_psi)
       cospsi = cos(det_psi)
       yp     = xp*sinpsi+yp1*cospsi
       xp     = xp*cospsi-yp1*sinpsi
    endif
    ;exit intercept point in this new coordinate system
    l2 = sqrt(rp^2-yp^2)-xp
    if type eq 1 then begin
       ind = where((xp lt 0) and (abs(yp) lt rin),count)        ;intercept the inner circle
       if count gt 0 then $
          l2[ind] = l2[ind]-2*sqrt(rin^2-yp[ind]^2)
    endif
    
    if (n_elements(det_psi) eq ntth) and (type lt 2) then l2 = abs(l2/cos(det_psi)) ;cylinder & annulus
end

; calculate the traveling path in a flat plate geometry
; equatorial scattering only if det_psi is not specified, the flat plate is much taller than the beam height
; input:
;   tth:     scalar or array of scattered angle in degree, tth=0 for north in DCS   (+:cw, equatorial value only)
;   sw:      sampling width position
;   st:      sampling thickness position
;   angle:   direction of the flat plate normal with respect to ki, counterclockwise=positive 
;   width:   width of the plate
;   thick:   thickness of the plate
;   det_psi: detector psi angle in radian (dcs definition), if specified and equal to the number of tth, calculate all direction
;   is_dcs:  if set, consider the beam width profile(3 cm wide, 1.5 cm center)
;   orc:     if set, calculate the effect of DCS oscillating radial collimator, results are saved in passrc
; output:
;   l1:      incoming path, same number of elements as that of tth
;   l2:      outgoing path, same number of elements as that of tth
;   cb:      1: incoming beam is in the 1.5 central region
;            0: outside that region
;   passrc:  flag indicating whether the neutron will pass the DCS radial collimator,1-pass 0-stopped
;            same number of elements as that of tth
;            assuming the sample and radial collimator are concentric and the center is at the beam center
pro dm_shielding_calcl1l2_fp,tth,sw,st,angle,width,thick,l1=l1,l2=l2,cb=cb,is_dcs=is_dcs,det_psi=det_psi,orc=orc,passrc=passrc
    ntth   = n_elements(tth)
    l2     = dblarr(ntth)
    l1     = dblarr(ntth)
    passrc = intarr(ntth)+1
    if ntth eq 1 then begin
       l2 = l2[0] & l1 = l1[0] & passrc = passrc[0]
    endif
    cb = 0b
    
    ;beam center coordinate system
    cosa = cos(angle*!ddtor)
    sina = sin(angle*!ddtor)
    rotm = [[cosa,sina],[-sina,cosa]]
    w1   = width-sw
    t1   = thick-st
    w2   = width/2d ;half width
    t2   = thick/2d ;half thickness
    x0   = [(sw-w2),(st-t2)]  ;sampling point
    x1   = rotm#x0  ;in rotated crystal coordinate system
    
    if (~ keyword_set(is_dcs)) or (abs(x1[0]) le 0.75) then cb=1b
      
    angle1 = double(angle)
    while (angle1 lt -180) do angle1 = angle1+360
    while (angle1 gt  180) do angle1 = angle1-360
    if abs(abs(angle1)-90) lt 1e-10 then begin
       l1[*] = (angle1 gt 0)?sw:w1
    endif else begin
       if abs(angle1) lt 90 then tmp_t = st else tmp_t = -t1
       xp = x0[0]-tmp_t*tan(angle*!ddtor)
       if abs(xp) gt w2 then $
          l1[*] = (xp gt 0)?(w1/abs(sina)):(sw/abs(sina))$
       else $
          l1[*] = abs(tmp_t/cosa)
    endelse
    
    tmp  = double(tth+angle)
    tth1 = tmp*!ddtor
    while (total(tmp lt -180) ne 0)  do tmp  = tmp+360*(tmp lt -180)
    while (total(tmp gt  180) ne 0)  do tmp  = tmp-360*(tmp gt  180)

    for i=0L,ntth-1 do begin
        if abs(abs(tmp[i])-90) lt 1e-10 then begin
           l2[i] = (tmp[i] gt 0)?w1:sw
        endif else begin
           if abs(tmp[i]) gt 90 then tmp_t = st else tmp_t = -t1
           xp = x0[0]-tmp_t*tan(tth1[i])
           if abs(xp) gt w2 then $
              l2[i] = (xp gt 0)?(w1/abs(sin(tth1[i]))):(sw/abs(sin(tth1[i])))$
           else $
              l2[i] = abs(tmp_t/cos(tth1[i]))
        endelse
    endfor
    
    if n_elements(det_psi) eq ntth then l2 = abs(l2/cos(det_psi))
    
    if keyword_set(orc) and keyword_set(is_dcs) then begin
       tmp0 = 20.*sin(0.0349066/2)       
       tmp  = abs(x0[0]*sin(tth1)-x0[1]*cos(tth1))
       passrc[*] = (tmp le tmp0)
    endif
end

; usage:
;   ei=5 & en=findgen(1000)/1000.*10-5 & tth=findgen(900)/10.+10 & info=[1,5,6,1.5,1.25]
;   factor=dm_calc_shielding(info,ei,en,tth,npoints=npoints,/interp,/debug)
; input:
;   info:           an array of the sample geometry and absorption information
;                   [type,li,la,rout/width,(rin/thickness),(angle)...]
;                   type:   0-cylinder
;                           1-annulus
;                           2-sphere
;                           3-flat plate
;   eief:           fixed Ei or Ef based on instrument geometry
;   en:             [nen] energy transfer corresponds to the time channels
;   det_tth:        [ndet] two-theta angle of the detectors in degree, DCS definition
;   det_psi:        [ndet] psi angle for detectors in radian, DCS definition, if present, consider out-of-plane scattering, no interpolation on det_tth unless uniq_psi keyword is present
;   uniq_psi:       an array of uniq value of det_psi, if present, still allow det_tth interpolation
;   group_leader:   group leader widget id
;   interp:         if set,calculate at most 100 en and 100 tth, the others are interpolated
;   npoints:        [nr,nphi,(npsi)] or [nthick,nwidth], number of points used in MC-calculation default[20,36,(18)]
;   is_dcs:         if set, consider the beam width profile(3 cm wide, 1.5 cm center)
;   orc:            if set, consider the effect of DCS oscillating radial collimator
;   instrgeom:      0 for direct geometry, 1 for inverse geometry
;   noquestion:     if set,no questions asked
;   timefactor:     A factor to be multiplied to the estimated calculation time
;   message:        progress bar message content   
;   mesgobj:        progress bar object
;   transmission:   if set, return transmission coefficient
;   l1factor:       incident beam path factor, multiplied to the calculated incident beam path, default 1.0
; return:
;   factor:         [nen,ndet] 1/A or A if transmission keyword is set, A is the transmission coefficient equ (6.3.3.1)
;   stopped:        1 if stopped, 0 otherwise
function dm_calc_shielding,info,eief,en,det_tth,group_leader=group_leader,interp=interp,npoints=npoints,det_psi=det_psi,uniq_psi=uniq_psi,is_dcs=is_dcs,orc=orc,instrgeom=instrgeom,$
    stopped=stopped,noquestion=noquestion,message=message,timefactor=timefactor,mesgobj=mesgobj,transmission=transmission,l1factor=l1factor,debug=debug

    if info[0] lt 0 or info[0] gt 3 then begin
       stopped = 1b
       return,1.0
    endif else $
       stopped = 0b
    if n_elements(npoints) lt 2 then npoints=[20,36]
    if info[0] eq 2 and n_elements(npoints) lt 3 then npoints=[npoints,18]
    if n_elements(timefactor) eq 0 then timefactor = 1.0
    if n_elements(l1factor) eq 0 then l1factor = 1.0d else l1factor = double(l1factor[0])
    en1  = double(en)  & intp_en  = 0 & nuniqpsi = 1
    tth1 = double(det_tth) & intp_tth = 0
    if n_elements(det_psi) ne 0 then det_psi1 = det_psi
    defsysv,'!ddtor',exists=exists
    if (~exists) then defsysv,'!ddtor',!dpi/(180d) ;double version of !dtor

    if keyword_set(interp) then begin
       nint = 100
       if n_elements(en) gt nint then begin
          emax     = max(en,min=emin)
          en1      = emin+(dindgen(nint)+0.5)/nint*(emax-emin)
          intp_en  = 1
       endif
       if (n_elements(det_tth) gt nint) and ((n_elements(det_psi) eq 0) or (n_elements(uniq_psi) gt 0))then begin  ;interpolate det_tth only when det_psi is not present
          tmax     = max(det_tth,min=tmin)
          tth1     = tmin+(dindgen(nint)+0.5)/nint*(tmax-tmin)
          nuniqpsi = 1>n_elements(uniq_psi) 
          intp_tth = 1
       endif
    endif

    nen1     = n_elements(en1)
    ndet1    = n_elements(tth1)
    li       = info[1] & la=info[2] & rout=info[3]
    e0       = 25.298862d                    ;meV, thermal neutron energy 2200m/s
    
    factor   = reform(dblarr(nen1,ndet1,nuniqpsi))
    if keyword_set(instrgeom) then begin
       efei  = eief+en1
       muin  = 1d/li+1d/(la*sqrt(efei/e0))   ;1/(incoming mean free path)
       muout = 1d/li+1d/(la*sqrt(eief/e0))   ;1/(outgoing mean free path)
    endif else begin
       efei  = eief-en1
       muin  = 1d/li+1d/(la*sqrt(eief/e0))   ;1/(incoming mean free path)
       muout = 1d/li+1d/(la*sqrt(efei/e0))   ;1/(outgoing mean free path)
    endelse
    case info[0] of
        0:  begin ;cylinder
            r   = rout*(dindgen(npoints[0])+0.5)/npoints[0]
            phi = dindgen(npoints[1])*360./npoints[1]
            end
        1:  begin ;annulus
            rin = info[4]
            r   = rin+(rout-rin)*(dindgen(npoints[0])+0.5)/(npoints[0])
            phi = dindgen(npoints[1])*360./npoints[1]
            end
        2:  begin ;sphere
            r   = rout*(dindgen(npoints[0])+0.5)/npoints[0]
            phi = dindgen(npoints[1])*360./npoints[1]
            psi = dindgen(npoints[2])*180./npoints[2]
            end
        3:  begin ;flat plate
            width = info[3] & thick = info[4] & ang = info[5]
            sw  = width*(dindgen(npoints[0])+0.5)/npoints[0]
            st  = thick*(dindgen(npoints[1])+0.5)/npoints[1]
            end
        else:
    endcase

    vol = 0.0d

    ;message bar
    if n_elements(message) eq 0 then message = 'Calculating absorption correction. Please wait...'
    if ~obj_valid(mesgobj) then mesgobj = obj_new('dm_progress',title='Absorption correction',message=message,group_leader=group_leader,/nostop) $
    else mesgobj->update,message=message
    current = systime(/sec) & firstpt = 1b  & nrep = 1.0  ;to check calculation time
    tmp_str = 'Try reducing the number of sampling points and using interpolation'+(['.',' and equatorial scattering only.'])[n_elements(det_psi) eq n_elements(det_tth)]
     
    ;pseudo MC calculation
    if info[0] eq 3 then begin               ;flat plate geometry
       for iu=0,nuniqpsi-1 do begin
           if nuniqpsi gt 1 then det_psi1 = replicate(uniq_psi[iu],ndet1)
           for i=0L,npoints[0]-1L do begin
               for j=0L,npoints[1]-1L do begin    
                   dm_shielding_calcl1l2_fp,tth1,sw[i],st[j],ang,width,thick,l1=l1,l2=l2,cb=cb,is_dcs=is_dcs,det_psi=det_psi1,orc=orc,passrc=passrc
                   if l1factor ne 1 then l1 = temporary(l1)*l1factor
                   weight = (cb)?(1.0d):(0.2d)    ;1 for central beam, 0.2 for outer beam
                   vol    = vol+weight
                   weight = weight*passrc
                   for k=0L,nen1-1L do begin
                       if keyword_set(instrgeom) then begin
                          factor[k,*,iu] = factor[k,*,iu]+exp(-muin[k]*l1-muout*l2)*weight
                       endif else begin
                          factor[k,*,iu] = factor[k,*,iu]+exp(-muin*l1-muout[k]*l2)*weight
                       endelse
                   endfor
                   if firstpt and (~keyword_set(noquestion)) then begin
                      time = (systime(/sec)-current)*npoints[0]*npoints[1]*nuniqpsi/60./nrep*timefactor
                      if time eq 0.0 then nrep = nrep+1.0 $
                      else firstpt = 0b
                      if time gt 1 then begin
                         ok = dialog_message(['The absorption calculation will take about '+dm_to_string(time,resolution=1)+' mintues. Do you still want to continue?',$
                              tmp_str],/question,dialog_parent=group_leader)
                         if ok eq 'No' then begin
                            stopped = 1b
                            obj_destroy,mesgobj
                            return,1.0
                         endif
                      endif
                   endif
               endfor
           endfor
       endfor
    endif else if info[0] eq 2 then begin    ;sphere
       for iu=0,nuniqpsi-1 do begin
           if nuniqpsi gt 1 then det_psi1 = replicate(uniq_psi[iu],ndet1)
           for m=0L,npoints[2]-1L do begin
               sinpsi = sin(psi[m]*!ddtor)
               for i=0L,npoints[0]-1L do begin
                   rsq = r[i]*r[i]
                   for j=0L,npoints[1]-1L do begin
                       dm_shielding_calcl1l2,tth1,info[0],r[i],phi[j],psi[m],rout=rout,l1=l1,l2=l2,cb=cb,is_dcs=is_dcs,det_psi=det_psi1,orc=orc,passrc=passrc
                       if l1factor ne 1 then l1 = temporary(l1)*l1factor
                       weight = ((cb)?(1.0d):(0.2d))*rsq*sinpsi   ;cb : 1 for central beam, 0.2 for outer beam
                       vol    = vol+weight
                       weight = weight*passrc
                       for k=0L,nen1-1L do begin
                           if keyword_set(instrgeom) then begin
                              factor[k,*,iu] = factor[k,*,iu]+exp(-muin[k]*l1-muout*l2)*weight
                           endif else begin
                              factor[k,*,iu] = factor[k,*,iu]+exp(-muin*l1-muout[k]*l2)*weight
                           endelse
                       endfor
                       if firstpt and (~keyword_set(noquestion)) then begin
                          time = (systime(/sec)-current)*npoints[0]*npoints[1]*npoints[2]/60./nrep*timefactor
                          if time eq 0.0 then nrep = nrep+1.0 $
                          else firstpt = 0b
                          if time gt 1 then begin
                             ok = dialog_message(['The absorption calculation will take about '+dm_to_string(time,resolution=1)+' mintue. Do you still want to continue?',$
                                  tmp_str],/question,dialog_parent=group_leader)
                             if ok eq 'No' then begin
                                stopped = 1b
                                obj_destroy,mesgobj
                                return,1.0
                             endif
                          endif
                       endif
                   endfor    
               endfor
           endfor
       endfor
    endif else begin                         ;cylinder and annulus
       for iu=0,nuniqpsi-1 do begin
           if nuniqpsi gt 1 then det_psi1 = replicate(uniq_psi[iu],ndet1)
           for i=0L,npoints[0]-1L do begin
               for j=0L,npoints[1]-1L do begin
                   dm_shielding_calcl1l2,tth1,info[0],r[i],phi[j],rin=rin,rout=rout,l1=l1,l2=l2,cb=cb,is_dcs=is_dcs,det_psi=det_psi1,orc=orc,passrc=passrc
                   if l1factor ne 1 then l1 = temporary(l1)*l1factor
                   weight = ((cb)?(1.0d):(0.2d))*r[i]   ;cb : 1 for central beam, 0.2 for outer beam
                   vol    = vol+weight
                   weight = weight*passrc
                   for k=0L,nen1-1L do begin
                       if keyword_set(instrgeom) then begin
                          factor[k,*,iu] = factor[k,*,iu]+exp(-muin[k]*l1-muout*l2)*weight
                       endif else begin
                          factor[k,*,iu] = factor[k,*,iu]+exp(-muin*l1-muout[k]*l2)*weight
                       endelse
                   endfor
                   if firstpt and (~keyword_set(noquestion)) then begin
                      time = (systime(/sec)-current)*npoints[0]*npoints[1]/60./nrep*timefactor
                      if time eq 0.0 then nrep = nrep+1.0 $
                      else firstpt = 0b
                      if time gt 1 then begin
                         ok = dialog_message(['The absorption calculation will take about '+dm_to_string(time,resolution=1)+' mintues. Do you still want to continue?',$
                              tmp_str],/question,dialog_parent=group_leader)
                         if ok eq 'No' then begin
                            stopped = 1b
                            obj_destroy,mesgobj
                            return,1.0
                         endif
                      endif
                   endif
               endfor
           endfor
       endfor
    endelse
    if keyword_set(interp) then begin
       nen  = n_elements(en)
       ndet = n_elements(det_tth)
       ;first interploate over all en
       if intp_en eq 1 then begin
          factor1 = reform(fltarr(nen,ndet1,nuniqpsi))
          for iu=0L,nuniqpsi-1 do $
              for i=0L,ndet1-1 do $
                  factor1[*,i,iu] = spline(en1,factor[*,i,iu],en)
       endif else factor1 = temporary(factor)
       ;then interpolate over all det_tth
       if intp_tth eq 1 then begin
          ind    = sort(det_tth)
          tth2   = det_tth[ind]
          factor = fltarr(nen,ndet)
          if (n_elements(det_psi) eq ndet) and (nuniqpsi gt 1) then begin
             detpsi2 = det_psi[ind]
             for iu=0L,nuniqpsi-1 do begin
                 ind_psi = where(detpsi2 eq uniq_psi[iu],cnt_psi)
                 if cnt_psi gt 0 then begin
                    for i=0L,nen-1 do $
                        factor[i,ind_psi] = spline(tth1,factor1[i,*,iu],tth2[ind_psi])
                 endif
             endfor
          endif else begin 
             for i=0L,nen-1 do $
                 factor[i,*] = spline(tth1,factor1[i,*],tth2)
          endelse
          tmp = factor
          factor[*,ind] = temporary(tmp)
       endif else factor = temporary(factor1)
    endif
    vol = vol/nuniqpsi  ;volume is counted nuniqpsi times
    if ~arg_present(mesgobj) then obj_destroy,mesgobj
    if keyword_set(debug) then  print,'calculation finished in ',systime(/sec)-current,' secs.'
    if keyword_set(transmission) then return,reform(float(factor)/float(vol)) else return,reform(float(vol)/float(factor))
end