; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEwriteASCIIGRP
;
; PURPOSE:
;   File writer for saving datasets as ASCII (Column) files
;
; CATEGORY:
;   Input Output
;
; SUPERCLASSES:
;   DAVEwriteASCII
;
; SUBCLASSES:
;
; CREATION:
;   See DAVEwriteASCIIGRP::Init
;
; METHODS:
;   WriteData
;
; INTERFACES:
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; October 2006
;-
;###############################################################################


;===============================================================================
; DAVEwriteASCIIGRP::writeData
; 
; PURPOSE:
;   Implements DAVEwriteASCII::writeData. Responsible for writing the
;   appropriate contents of the dataset to the ASCII file in col format.
;
; PARAMETERS:
;   oData [in] - The data object to be written
;
; KEYWORDS:
;   none
;
; RETURN VALUE:
;    1 - Valid
;    0 - Invalid.
function DAVEwriteASCIIGRP::writeData, oData
compile_opt idl2

catch, theError
if (theError ne 0) then begin
   catch, /cancel
   if (n_elements(lun) gt 0) then begin ; if IO error then close the file before exiting
      sStat = fStat(lun)
      if (sStat.open ne 0) then free_lun, lun, /force
   endif
   self->SignalError, !error_state.msg, severity=2
   return, 0
endif

retVal = 1
myTitle = 'Save as ASCII column text file'
msg = 'Could not write data'
oData->GetProperty, name=datasetname


;; Validate data object
;if (~self->isValid(oData,'DAVE1DATASET')) then  begin
;    self->SignalError,['Invalid dataset!','Cannot write selected dataset'] $
;                        ,severity=2
;    return, 0
;endif

;switch self->datasetType(oData) of
;   'DAVE1DATASET': begin
;
;      ;; Get the top level contents of dataset
;      cont = oData->Get(/all,count=cnt)
;      if (cnt lt 1) then  begin
;         self->SignalError,['Empty dataset!','Cannot write selected dataset'] $
;                            ,severity=2
;         return, 0
;      endif
;      
;      for i=0,cnt-1 do begin
;         if (retVal eq 0) then continue
;         
;         cont[i]->getProperty, type=type
;         case type of
;            'DAVE1COMMONSTR': begin
;               ;; 
;               obj = cont[i]->getByName('2') ; the dependent axis
;               if ((retVal = obj_valid(obj)) eq 0) then break
;               void = obj->getData(qty)
;               if (n_elements(qty) le 0) then begin
;                  retVal = 0
;                  break
;               endif
;               ndim = size(qty,/n_dimensions)
;               if (ndim ne 2) then begin
;                  retVal = 0
;                  msg = ['"'+datasetname+'" is not 2D','Cannot write it out using ASCII GRP format'] 
;                  break
;               endif
;               if ((void = obj->GetMetaData('Long_name',qtyLabel)) eq 0 ) then qtyLabel=''
;               if ((void = obj->GetMetaData('Units',qtyUnits)) eq 0 ) then qtyUnits=''
;               if ((void = obj->GetMetaData('Axes',axes)) eq 0 ) then axes=''
;               
;               errPresent = 0
;               obj = cont[i]->getByName('3') ; the error in the dependent axis
;               if (obj_valid(obj)) then $
;                  void = obj->getData(err)
;               if (n_elements(err) gt 1) then errPresent = 1
;               
;               obj = cont[i]->getByName('0') ; first independent axis
;               if (obj_valid(obj)) then begin
;                  void = obj->getData(xData)
;                  if (n_elements(xData) le 0) then begin
;                     retVal = 0
;                     break
;                  endif
;                  if ((void = obj->GetMetaData('Long_name',xLabel)) eq 0 ) then $
;                     xLabel='X Values'
;                  if ((void = obj->GetMetaData('Units',xUnits)) eq 0 ) then xUnits=''
;                  if ((void = obj->GetMetaData('Distribution',xType)) eq 0 ) then xType='POINTS'
;               endif               
;               obj = cont[i]->getByName('1') ; second independent axis
;               if (obj_valid(obj)) then begin
;                  void = obj->getData(yData)
;                  if (n_elements(yData) le 0) then begin
;                     retVal = 0
;                     break
;                  endif
;                  if ((void = obj->GetMetaData('Long_name',yLabel)) eq 0 ) then $
;                     yLabel='Y values'
;                  if ((void = obj->GetMetaData('Units',yUnits)) eq 0 ) then yUnits=''
;                  if ((void = obj->GetMetaData('Distribution',yType)) eq 0 ) then yType='POINTS'
;               endif               
;            end
;                        
;            else:
;         endcase
;      endfor
;
;      break
;   end
;   
;   'ASCIISPE':
;   'ASCIIGRP': begin
;
;      ;; Get the top level contents of dataset
;      void = oData->Get(/all,count=cnt)
;      if (cnt lt 1) then  begin
;         self->SignalError,['Empty dataset!','Cannot write selected dataset'] $
;                            ,severity=2
;         return, 0
;      endif
;
;      obj = oData->getByName('2') ; the dependent axis
;      if ((retVal = obj_valid(obj)) eq 0) then break
;      void = obj->getData(qty)
;      if (n_elements(qty) le 0) then retVal = 0
;      if ((void = obj->GetMetaData('Long_name',qtyLabel)) eq 0 ) then qtyLabel=''
;      if ((void = obj->GetMetaData('Units',qtyUnits)) eq 0 ) then qtyUnits=''
;      if ((void = obj->GetMetaData('Axes',axes)) eq 0 ) then axes=''
;      
;      errPresent = 0
;      obj = oData->getByName('3') ; the error in the dependent axis
;      if (obj_valid(obj)) then begin
;         void = obj->getData(err)
;         errPresent = 1
;      endif
;      
;      obj = oData->getByName('0') ; first independent axis
;      if (obj_valid(obj)) then begin
;         void = obj->getData(xdata)
;         if (n_elements(xData) le 0) then retVal = 0
;         if ((void = obj->GetMetaData('Long_name',xLabel)) eq 0 ) then $
;            xLabel='X Values'
;         if ((void = obj->GetMetaData('Units',xUnits)) eq 0 ) then xUnits=''
;         if ((void = obj->GetMetaData('Distribution',xType)) eq 0 ) then xType='POINTS'
;      endif
;      
;      obj = oData->getByName('1') ; second independent axis
;      if (obj_valid(obj)) then begin
;         void = obj->getData(yData)
;         if (n_elements(yData) le 0) then retVal = 0
;         if ((void = obj->GetMetaData('Long_name',yLabel)) eq 0 ) then $
;            yLabel='Y values'
;         if ((void = obj->GetMetaData('Units',yUnits)) eq 0 ) then yUnits=''
;         if ((void = obj->GetMetaData('Distribution',yType)) eq 0 ) then yType='POINTS'
;      endif
;
;      break
;   end
;
;   else: begin
;      self->SignalError,['Invalid dataset!','Cannot write selected dataset in ASCII Group format'] $
;                         ,severity=2
;      return, 0
;   end
;
;endswitch


errormsg = ''
if (obj_hasmethod(oData,'toASCIIGrp')) then begin
   if (oData->toASCIIGrp(dataStr, errormsg)) then begin
      xData = datastr.xData
      yData = dataStr.yData
      nx = n_elements(xData)
      ny = n_elements(yData)

      openw, lun, self->GetFilename(), /get_lun
   
      ;; write to disk and inform user of ouput destination
      ;; the nos of x values
      printf, lun, '# Number of '+dataStr.xTitle
      printf, lun, strtrim(string(nx),2)
      ;; the nos of y values
      printf, lun, '# Number of '+dataStr.yTitle
      printf, lun, strtrim(string(ny),2)
      ;; The x values
      printf, lun, '# '+dataStr.xTitle
      for i = 0,nx-1 do printf, lun, format='(E13.5)',xData[i]
      ;; The y values
      printf, lun, '# '+dataStr.yTitle
      for i = 0,ny-1 do printf, lun, format='(E13.5)',yData[i]

      ;; Deal with masked data.
      ; Replace NaNs with a filler value of -1e20
      index = where(finite(dataStr.data,/NAN),nMask)
      if (nMask gt 0) then begin
         dataStr.data[index] = -1e20
         dataStr.error[index] = 1.0
      endif
      
      ;; The data values
      for i=0,ny-1 do begin
         printf, lun, '# Group ',strtrim(string(i+1),2),' of ',strtrim(string(ny),2)
         for j=0,nx-1 do printf, lun, format='(2(E13.5,2X))', dataStr.data[j,i], dataStr.error[j,i]
      endfor
   
      free_lun, lun, /force
   
      return, 1
   endif else begin
      msg = (errormsg eq '')? ["Failure to convert to ASCII GRP format"] : errormsg
      Self->SignalError, msg, severity=2
      return, 0
   endelse
endif else begin
   msg = ['Unknown dataset type!',"Cannot convert to ASCII GRP format"]
   Self->SignalError, msg, severity=2
   return, 0
endelse

end


;===============================================================================
; DAVEwriteASCIIGRP::Init
; 
; PURPOSE:
;   Initialization method for objects of DAVEwriteASCIIGRP class
;
; PARAMETERS
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - Successful
;    0 - Failure
;
function DAVEwriteASCIIGRP::init, _REF_EXTRA=etc
compile_opt idl2

; Init superclass
validTypes = ['ASCIIGRP','ASCIISPE','DAVE1DATASET']
if (self->IDLitWriter::Init('grp' $
                            ,TYPES='ASCIIGRP' $
                            ,NAME="ASCII Group" $
                            ,DESCRIPTION="ASCII Group text file (grp)" $
                            ,_EXTRA=etc) eq 0) then return, 0


; call setproperty method, if necessary.
if (n_elements(etc) gt 0) then $
  self->SetProperty, _EXTRA=etc

self.asciiTypes = ['ASCIICOL','ASCIIGRP','ASCIISPE']

self->setProperty, asciiType='ASCIIGRP'

return, 1

end



;===============================================================================
pro DAVEwriteASCIIGRP__Define
compile_opt idl2

void = {DAVEwriteASCIIGRP, $
        inherits DAVEwriteASCII $
       }
end
