; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEreadOsiris
;
; PURPOSE:
;   File reader for OSIRIS Nexus raw data files
;
; CATEGORY:
;   Input Output
;
; SUPERCLASSES:
;   IDLitReader
;
; SUBCLASSES:
;
; CREATION:
;   See DAVEreadOsiris::Init
;
; METHODS:
;   GetData
;   IsA
;
; INTERFACES:
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; July 2004
;-
;###############################################################################


;===============================================================================
; DAVEreadOsiris::GetData
; 
; PURPOSE:
;   Responsible for actually reading the data from the file
;
; PARAMETERS:
;   oData [out] - An IDLitData (or subclass) object which will contain
;                 the data read from the file.
;
; KEYWORDS:
;   none
;
; RETURN VALUE:
;    1 - Successful Read
;    0 - Failure reading from the file.
function DAVEreadOsiris::GetData, oData
compile_opt idl2

catch, iErr
if(iErr ne 0)then begin
    catch, /cancel

    goto, ioerr  ;; do any cleanup needed
endif
on_ioerror, ioerr

oTool = self->GetTool()
if (not OBJ_VALID(oTool)) then $
  return, 0


filename = self->GetFilename()


handle = 0L

status = nxopen(filename,'NXACC_READ',handle)

status = nxopengroup(handle,'raw_data_1','NXentry')


status = nxopendata(handle,'title')
status = nxgetdata(handle,title)
status = nxclosedata(handle) 

status = nxopendata(handle,'beamline')
status = nxgetdata(handle,instrument)
status = nxclosedata(handle)

status = nxopendata(handle,'run_number')
status = nxgetdata(handle, run_number) & run_number = run_number[0]
status = nxclosedata(handle) 

status = nxopendata(handle,'start_time')
status = nxgetdata(handle, start_time)
status = nxclosedata(handle) 

status = nxopendata(handle,'end_time')
status = nxgetdata(handle, end_time)
status = nxclosedata(handle) 

status = nxopendata(handle,'duration')
status = nxgetdata(handle, duration) & duration = duration[0]
status = nxclosedata(handle) 

status = nxopendata(handle,'run_cycle')
status = nxgetdata(handle, run_cycle)
status = nxclosedata(handle) 

status = nxopendata(handle,'proton_charge')
status = nxgetdata(handle, proton_charge) & proton_charge = proton_charge[0]
status = nxclosedata(handle) 

;------ User
status = nxopengroup(handle,'user_1','NXuser')

status = nxopendata(handle,'name')
status = nxgetdata(handle, username)
status = nxclosedata(handle)

status = nxopendata(handle,'affiliation')
status = nxgetdata(handle, useraffiliation)
status = nxclosedata(handle)

status = nxclosegroup(handle)   ; User

;------ Instrument / Detector
status = nxopengroup(handle,'instrument','NXinstrument')
status = nxopengroup(handle,'detector_1','NXdata')

status = nxopendata(handle,'counts')
status = nxgetdata(handle, data)
status = nxgetattr(handle,'units',data_units,attType)
status = nxclosedata(handle) 

status = nxopendata(handle,'time_of_flight')
status = nxgetdata(handle, tof)
status = nxgetattr(handle,'units',tof_units,attType)
status = nxclosedata(handle) 

status = nxopendata(handle,'spectrum_index')
status = nxgetdata(handle, spec)
status = nxclosedata(handle)

status = nxopendata(handle,'polar_angle')
status = nxgetdata(handle, polar_angle)
status = nxclosedata(handle)

status = nxclosegroup(handle)   ; Detector
status = nxclosegroup(handle)   ; Instrument


;------ Monitor_1
status = nxopengroup(handle,'monitor_1','NXmonitor')

status = nxopendata(handle,'data')
status = nxgetdata(handle, mon1_data)
status = nxclosedata(handle) 

status = nxopendata(handle,'time_of_flight')
status = nxgetdata(handle, mon1_tof)
status = nxclosedata(handle) 

status = nxclosegroup(handle)   ; Monitor 1

;------ Monitor_2
status = nxopengroup(handle,'monitor_2','NXmonitor')

status = nxopendata(handle,'data')
status = nxgetdata(handle, mon2_data)
status = nxclosedata(handle) 

status = nxopendata(handle,'time_of_flight')
status = nxgetdata(handle, mon2_tof)
status = nxclosedata(handle) 

status = nxclosegroup(handle)   ; Monitor 2


;status = nxopendata(handle,'')
;status = nxgetdata(handle, )
;status = nxclosedata(handle) 


; 
data = float(temporary(data[*,960:1001]))
polar_angle = polar_angle[960:1001]
spec = spec[960:1001]

n = n_elements(mon1_tof)
mon1_tof = temporary(0.5*(mon1_tof[0:n-2]+mon1_tof[1:n-1]))   ; convert histogram to points
n = n_elements(mon2_tof)
mon2_tof = temporary(0.5*(mon2_tof[0:n-2]+mon2_tof[1:n-1]))   ; convert histogram to points

mon1Str = {tof:temporary(mon1_tof), data:float(temporary(mon1_data))}
mon2Str = {tof:temporary(mon2_tof), data:float(temporary(mon2_data))}

specificStr = {title:title $
                 ,run_number:run_number $
                 ,start_time:start_time $
                 ,end_time:end_time $
                 ,duration:duration $
                 ,users:username $
                 ,user_affiliation:useraffiliation $
                 ,run_cycle:run_cycle $
                 ,proton_charge:proton_charge $
                 ,monitor1:mon1Str $
                 ,monitor2:mon2Str $
                 }
                 
nx = n_elements(tof)
dims = size(data,/dimensions)
xType = (nx eq (dims[0]+1))? 'HISTOGRAM' : 'POINTS'

status = create_dave_pointer(davePtr $
                                  ,instrument = instrument $
                                  ,qty=data $
                                  ,qtUnits = '' $
                                  ,qtLabel = data_units $
                                  ,err = sqrt(data) $
                                  ,xvals = tof $
                                  ,xType = xType $
                                  ,xUnits = tof_units $
                                  ,xLabel = 'Time-of-flight' $
                                  ,yVals = polar_angle $
                                  ,yLabel = 'Scattering angle' $
                                  ,yUnits = '' $
                                  ,yType = 'POINTS' $
                                  ,specificStr = specificStr $
                                  ,treatment = 'Read data from '+filename $
                                  )

nameTag = file_basename(filename,'.'+Self->GetFileExtensions(),/fold_case)
if (status) then $
   oData = obj_new('DAVEDataset',davePtr=davePtr, nameTag=nameTag) $
else $
   oData = obj_new()

return, obj_valid(oData)


ioerr: ;; IO Error handler
if (strcmp(vartype(handle),'NXSTRUCT',/fold_case)) then $
   void = nxclose(handle)

self->SignalError, !error_state.msg, severity=2
return, 0

end



;===============================================================================
; DAVEreadOsiris::IsA
; 
; PURPOSE:
;   Check the given file to see whether it is the proper type for this
;   reader.
;
; PARAMETERS
;   strFilename - The file to check
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - strFilename is a proper file type
;    0 - strFilename is cannot be handled by this reader
;
function DAVEreadOsiris::IsA, strFilename
compile_opt idl2

; Basic error handler
catch, iErr
if(iErr ne 0)then begin
    catch, /cancel

   if (strcmp(vartype(handle),'NXSTRUCT',/fold_case)) then $
      void = nxclose(handle)

    return, 0
endif

; Call the base class method to check for proper extension for this reader
if (~self->IDLitReader::IsA(strFilename)) then return, 0

; Check file contents to make sure this is a valid Osiris Nexus file.
; Open the file and check that the instrument name is 'OSIRIS'
handle = 0L
inst = ''
if ((status = nxopen(strFilename,'NXACC_READ',handle))) then begin
   status = nxopengroup(handle,'raw_data_1','NXentry')
   if (status) then begin 
      status = nxopendata(handle,'beamline')
      if (status) then status = nxgetdata(handle,inst)
      
      status = strcmp(inst,'OSIRIS',/fold_case)
   endif
endif

if (strcmp(vartype(handle),'NXSTRUCT',/fold_case)) then $
   void = nxclose(handle)

return, status
end

;===============================================================================
; DAVEreadOsiris::Init
; 
; PURPOSE:
;   Initialization method for objects of DAVEreadOsiris class
;
; PARAMETERS
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - Successful
;    0 - Failure
;
function DAVEreadOsiris::Init, _REF_EXTRA=etc
compile_opt idl2

; Init superclass
if (self->IDLitReader::Init('nxs'  $            ; <-- acceptable file extension(s)
                            ,NAME='RAWOSIRIS' $
                            ,DESCRIPTION="OSIRIS Nexus raw data file (nxs)" $
                            ,_EXTRA=etc) eq 0) then return, 0

; call setproperty method, if necessary.
if (n_elements(etc) gt 0) then $
  self->SetProperty, _EXTRA=etc

return, 1

end




;===============================================================================
pro DAVEreadOsiris__Define
compile_opt idl2

void = {DAVEreadOsiris, inherits IDLitReader}

end
