; $Id$
;###############################################################################
;+
; NAME:
;   MODIFY_DAVE_POINTER
;
; PURPOSE:
;   This function modifies a DAVE pointer. If successful, the function returns
;   a 1 else it returns 0.  A valid DAVEPTR must be specified. If you wish to 
;   create a new davePtr structure, use CREATE_DAVE_POINTER()
;
; CATEGORY:
;   DAVE Utilities
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Jan 2009
;-
;###############################################################################
;
;
;	ret_val = MODIFY_DAVE_POINTER(	$
;								DAVEPTR ,					$
;								INSTRUMENT = instrument,	$
;								QTY = qty,					$
;								QTUNITS = qtunits,			$
;								QTLABEL = qtlabel,			$
;								ERR = err,					$
;								XVALS = xvals,				$
;								XTYPE = xtype,				$
;								XUNITS = xunits,			$
;								XLABEL = xlabel,			$
;								YVALS = yvals,				$
;								YTYPE = ytype,				$
;								YLABEL = ylabel,			$
;								YUNITS = yunits,			$
;								SPECIFICSTR = specificstr,	$
;								TREATMENT = treatment,		$
;								DNAME = dname,				$
;								DUNITS = dunits,			$
;								DLEGEND = dlegend,			$
;								DQTY = dqty,				$
;								DERR = derr,				$
;								ERMSG = errmsg				)
;
; PARAMETERS (REQUIRED)
;
;	DAVEPTR:	A valid DAVE pointer as defined in DAVE.PRO for instance. Use
;           create_dave_pointer() to create a davePtr structure.
;
;
; INPUT KEYWORDS
;
;	QTY:		Double of float array containing the data variable.  Can be one or two
;				dimensional where the size can be one of the following
;				(nx,ny), (nx+1,ny), (nx,ny+1),	(nx+1,ny+1),(nx),(nx+1) where
;				nx is the length of the first independent variable and ny is the
;				length of the second independent variable.
;	ERR:		Double of float array containing the error associated with the measured
;				quantity, QTY.  ERR must have the same dimension and size as QTY.
;	INSTRUMENT:	String variable describing instrument.
;	XVALS:		Double or float array containing the first independent variable for
;				QTY.  If not provided then a simple index array is determined based
;				on the size of QTY.
;	YVALS:		Double or float array containing the second independent variable for
;				QTY.  If not provided then a simple index array is determined based
;				on the size of QTY.
;	QTUNITS:	string variable specifying the units of QTY
;	QTLABEL:	string variable specifying the plot label for QTY
;	XTYPE:		String variable specifying if the first independent variable, XVALS,
;				is "POINTS" or "HISTO".
;	YTYPE:		String variable specifying if the second independent variable, YVALS,
;				is "POINTS" or "HISTO".
;	XLABEL:		string variable specifying the plot label for the first independent
;				variable, XVALS.
;	YLABEL:		string variable specifying the plot label for the second independent
;				variable, YVALS.
;	XUNITS:		string variable specifying the units of XVALS
;	YUNITS:		string variable specifying the units of YVALS
;	SPECIFICSTR:	structure containing any instrument specific information the
;					user wishes to include in the DAVE pointer.  Can contain any
;					variable type in its fields.
;	TREATMENT:	string array of any length containing the treatment of the data.
;
;	DNAME:		string name of the variable/parameter to be described in the
;				descriptive pointer in the DAVEPTR called DESCRIPTR.
;	DUNITS:		string variable specifying the units of the descriptive variable.
;	DLEGEND:	string variable specifying a description of the variable
;	DQTY:		value of the descriptive variable.
;	DERR:		value specifying the uncertainty of the descriptive variable.
;
;
; OPTIONAL OUTPUT KEYWORDS
;
;	ERMSG:		string variable specifying the cause of an error in which the function
;				returns 0.
;
;###############################################################################
;
; LICENSE:
;  The software in this file is written by an employee of 
;  National Institute of Standards and Technology 
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;
function modify_dave_pointer,	$
							daveptr,					$
							instrument = instrument,	$
							qty = qty,					$
							qtunits = qtunits,			$
							qtlabel = qtlabel,			$
							err = err,					$
							xvals = xvals,				$
							xtype = xtype,				$
							xunits = xunits,			$
							xlabel = xlabel,			$
							yvals = yvals,				$
							ytype = ytype,				$
							yunits = yunits,			$
							ylabel = ylabel,			$
							specificstr = specificstr,	$
							treatment = treatment,		$
							dname = dname,				$
							dunits = dunits,			$
							dlegend = dlegend,			$
							dqty = dqty,				$
							derr = derr,				$
							ermsg = errmsg

errmsg = ''
; Was a DAVE pointer set as a parameter?
if n_params() eq 0 then begin
	errmsg = 'DAVEPTR is a required'
	return,0
endif

; basic error handler
catch, iError
if (iError ne 0) then begin
   catch, /cancel
   errMsg = !ERROR_STATE.MSG
   return, 0
endif

if (n_elements(xvals) gt 0 || n_elements(yvals) gt 0 || $
    n_elements(qty) gt 0 || n_elements(err) gt 0 ) then begin
     
   histPtr =  (*(*daveptr).datastrptr).commonstr.histptr
   x = (n_elements(xvals) gt 0)? xvals : (*histPtr).x
   y = (n_elements(yvals) gt 0)? yvals : (*histPtr).y
   qty = (n_elements(qty) gt 0)? qty : (*histPtr).qty
   err = (n_elements(err) gt 0)? err : (*histPtr).err
   (*(*(*daveptr).datastrptr).commonstr.histptr) = $
      {x:x,y:y,qty:qty,err:err}
endif

if (n_elements(treatment) gt 0) then begin
   (*(*(*daveptr).datastrptr).commonstr.treatmentptr) = treatment
endif
if (n_elements(qtunits) gt 0) then begin
   (*(*daveptr).datastrptr).commonstr.histunits = qtunits
endif
if (n_elements(xunits) gt 0) then begin
   (*(*daveptr).datastrptr).commonstr.xunits = xunits
endif
if (n_elements(yunits) gt 0) then begin
   (*(*daveptr).datastrptr).commonstr.yunits = yunits
endif
if (n_elements(xlabel) gt 0) then begin
   (*(*daveptr).datastrptr).commonstr.xlabel = xlabel
endif
if (n_elements(ylabel) gt 0) then begin
   (*(*daveptr).datastrptr).commonstr.ylabel = ylabel
endif
if (n_elements(xtype) gt 0) then begin
   (*(*daveptr).datastrptr).commonstr.xtype = xtype
endif
if (n_elements(ytype) gt 0) then begin
   (*(*daveptr).datastrptr).commonstr.ytype = ytype
endif
if (n_elements(qtlabel) gt 0) then begin
   (*(*daveptr).datastrptr).commonstr.histlabel = qtlabel
endif
if (n_elements(instrument) gt 0) then begin
   (*(*daveptr).datastrptr).commonstr.instrument = instrument
endif
if (n_elements(specificstr) gt 0) then begin
   (*(*(*daveptr).datastrptr).specificptr) = specificstr
endif
if (n_elements(dname) gt 0) then begin
   (*(*davePtr).descriPtr).name = dname
endif
if (n_elements(dunits) gt 0) then begin
   (*(*davePtr).descriPtr).units = dunits
endif
if (n_elements(dlegend) gt 0) then begin
   (*(*davePtr).descriPtr).legend = dlegend
endif
if (n_elements(dqty) gt 0) then begin
   (*(*davePtr).descriPtr).qty = dqty
endif
if (n_elements(derr) gt 0) then begin
   (*(*davePtr).descriPtr).err = derr
endif

; Return a 1 if successful
return,1
end
