;==============================================================================
pro Dialog_NCNRpublicData_filterEvent, event
compile_opt idl2

widget_control, event.top, get_uvalue = sPtr

if (event.index eq -1 && ~strcmp(strtrim(event.str),'')) then begin
  ; combobox edited. Add new entry to list, if not already included
  widget_control, (*sPtr).wFilter, get_value=list ; get existing list
  void = where(strmatch(list,event.str,/fold),found)  ; is new entry in the list?
  if (~found) then widget_control, (*sPtr).wFilter, combobox_additem=strtrim(event.str) ; if not, add it
endif

; Refresh the display
Dialog_NCNRpublicData_refresh, sPtr


end

;==============================================================================
pro Dialog_NCNRpublicData_acceptFiles, sPtr
compile_opt idl2

;Retrieve the selected file(s) 
index = widget_info((*sPtr).wList,/list_select)
if (index[0] eq -1) then begin
  msg = 'No files were selected!'
  ;title = 'Dialog_NCNRpublicData'
  void = dialog_message(msg, title=(*sPtr).title, dialog_parent=(*sPtr).wTLB, /info)
  return
endif

; Elliminate diectory selections
nSel = n_elements(index)
nValid = 0
nDir = 0
iValid = []
for i=0,nSel-1 do begin
  if ((*(*sPtr).filetypePtr)[index[i]] ne 'd') then begin
    nValid++ 
    iValid = [iValid,index[i]]
  endif else nDir++
endfor

if ((nValid eq 0) && (nDir gt 0)) then begin
  msg = ['Invalid selection(s)!','Please make valid non-directory selection(s)']
  ;title = 'Dialog_NCNRpublicData'
  void = dialog_message(msg, title=(*sPtr).title, dialog_parent=(*sPtr).wTLB, /info)
  return  
endif

; Retrieve selected files and kill dialog
if (nValid gt 0) then begin
  (*sPtr).oDaveFtpUrl->GetProperty, currentDir = curDir
  selectedFiles = curDir[0] + (*(*sPtr).fileNamePtr)[iValid]
  if (ptr_valid((*sPtr).selectedFilesPtr)) then $
    (*(*sPtr).selectedFilesPtr) = selectedFiles else $
    (*sPtr).selectedFilesPtr = ptr_new(selectedFiles)
    
  ; Okay, we are done!
  widget_control, (*sPtr).wTLB, /destroy
endif

end


;==============================================================================
pro Dialog_NCNRpublicData_listEvent, event
compile_opt idl2
widget_control, event.top, get_uvalue = sPtr

; must be a widget_list event
if (~strcmp(tag_names(event,/struct),'WIDGET_LIST')) then return

; Only proceed if it was a double-click
if (event.clicks ne 2) then return

index = event.index

fileType = (*(*sPtr).fileTypePtr)[index]
case fileType of
  'd': begin
    ; selection is a directory
    ; Make the selection the current directory on the server
    (*sPtr).oDaveFtpUrl->GetProperty, currentDir = curDir    
    curDir = curDir + (*(*sPtr).fileNamePtr)[index] + '/'
    (*sPtr).oDaveFtpUrl->SetProperty, currentDir = curDir
    
    ; refresh the listing
    Dialog_NCNRpublicData_refresh, sPtr  
  end
  
  else: begin
    ; selection is a file, so retrieve the file and display it

    if (float(!version.release) ge 9.0) then begin
      (*sPtr).odaveftpurl->Getproperty, currentDir = curDir, baseFullPath=bfpath
      filename = bfpath+curDir+(*(*sPtr).filenameptr)[index]
      response = Httprequest.Get(filename)
      fileContents = response.content   ; these are raw bytes, thus requiring further processing
      
      ; first write the raw bytes to a temporary file on disk
      tmpFile = !home_dir+'ncnrdataTmpfile.dat'
      openw,lun,tmpfile,/get_lun
      writeu, lun, fileContents
      free_lun,lun,/force
      
      ; check if this is an hdf dataset
      isHDFDataset = H5f_is_hdf5(tmpFile)
      if (isHDFDataset) then begin
        void = H5_browser(tmpFile,/dialog_read)
        File_delete, tmpFile
        return
      endif else begin
        status = dave_read_filecontents(tmpFile,dstring = fileContents,errmsg = errmsg)
        File_delete, tmpFile
      endelse
    endif else begin
      (*sPtr).odaveftpurl->Getproperty, currentDir = curDir
      fname = curDir + (*(*sPtr).filenameptr)[index]

      fileContents = (*sPtr).odaveftpurl->Getfilecontent(fname, /string)
    endelse

    if (n_elements(fileContents) eq 0) then return
    
    doneTxt = "Close "+(*(*sPtr).fileNamePtr)[index]
    font = (!version.os_family eq 'Windows')? 'Courier New*24*Bold' : $
           '-adobe-courier-bold-o-normal--24-240-75-75-m-150-iso8859-1'
    font1 = (!version.os_family eq 'Windows')? 'Courier New*18*Bold' : $
           '-adobe-courier-bold-o-normal--18-180-75-75-m-110-iso8859-1'
    font2 = (!version.os_family eq 'Windows')? 'Courier New*16' : $
           '-adobe-courier-medium-o-normal--14-140-75-75-m-90-iso8859-1'
           
    xdisplayfile, text=fileContents, done_button=doneTxt, /grow_to_screen $
                ,group=event.top, font=font2 $
                ,title=fname
  end
endcase

end


;==============================================================================
pro Dialog_NCNRPublicDara_topDir, sPtr
compile_opt idl2

; make the baseDir the currentDir
(*sPtr).oDaveFtpUrl->GetProperty, baseDir = baseDir
(*sPtr).oDaveFtpUrl->SetProperty, currentDir = baseDir

; refresh the dir listing
Dialog_NCNRpublicData_refresh, sPtr

end


;==============================================================================
pro Dialog_NCNRPublicDara_UpDir, sPtr
compile_opt idl2

(*sPtr).oDaveFtpUrl->GetProperty, currentDir = curDir, baseDir = baseDir
baseLen = strlen(baseDir)

; strip the last directory from the currentdir name
toks = strsplit(curDir,'/',/extract,count=ntoks)
newDir = '/'+strjoin(toks[0:ntoks-2],'/')+'/'
newDirLen = strlen(newDir)

if (newDirLen lt baseLen) then newDir = baseDir

; make newDir the currentDir
(*sPtr).oDaveFtpUrl->SetProperty, currentDir = newdir

; refresh the dir listing
Dialog_NCNRpublicData_refresh, sPtr

end



;;==============================================================================
;pro Dialog_NCNRpublicData_listEvent, event
;compile_opt idl2
;widget_control, event.top, get_uvalue = sPtr
;
;; Only need to act for double clicks
;if (event.clicks ne 2) then return
;
;index = event.index
;ftype = (*(*sPtr).fileTypePtr)[index]
;case ftype of
;  'd': begin
;    ; selected file was a directory so make it the current directory
;    newDir = (*sPtr).currentDir + (*(*sPtr).fileNamePtr)[index] + '/'
;    (*sPtr).currentDir = newDir
;    
;    Dialog_NCNRpublicData_refresh, sPtr
;  end
;
;  else: begin
;    ; view contents of the selected file
;    sFileName = (*(*sPtr).fileNamePtr)[index]
;    fileName = (*sPtr).currentDir + sFileName
;    
;    ; Retrieve the file contents from the ftp server
;    (*sPtr).oURL->SetProperty, URL_PATH=filename
;    contents = (*sPtr).oURL->Get(/string)
;    
;    if (n_elements(contents) gt 0) then begin
;        doneTxt = "Close "+title
;        font = (!version.os_family eq 'Windows')? 'Courier New*24*Bold' : $
;            '-adobe-courier-bold-o-normal--24-240-75-75-m-150-iso8859-1'
;        font1 = (!version.os_family eq 'Windows')? 'Courier New*18*Bold' : $
;            '-adobe-courier-bold-o-normal--18-180-75-75-m-110-iso8859-1'
;        font2 = (!version.os_family eq 'Windows')? 'Courier New*16' : $
;            '-adobe-courier-medium-o-normal--14-140-75-75-m-90-iso8859-1'
;
;        xdisplayfile, sfilename, done_button=doneTxt, /grow_to_screen $
;                    ,group=event.top, font=font2, text=contents, title=filename
;    endif
;  end
;endcase
;
;end


;==============================================================================
; Reformat date from 'Wed Aug 09 13:51:23 2023' to '08/09/23 13:51:23'
function Dialog_NCNRpublicData_reformatdata, indate
compile_opt idl2

months = ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov']

toks = strsplit(indate,' ',/extract)
mm = string(where(months eq toks[1])+1,format='(I02)')
dd = toks[2]
yy = toks[4].Substring(-2)

outdate = strjoin([mm,dd,yy],'/')+' '+toks[3].Substring(0,4)

return, outdate
end


;==============================================================================
pro Dialog_ncnrpublicdata_refresh, sPtr
  compile_opt idl2

  ; get dirctory contents
dirList = (*sPtr).oDaveFtpUrl->GetListing()

filter = widget_info((*sPtr).wFilter,/combobox_gettext)
help,filter
if (strcmp(filter,'')) then filter='*'

; Process and reformat the listing so it is more concise 
;n = n_elements(dirList)
filetype = []
fileName = []
listing = []


if (float(!version.release) ge 9.0) then begin
  if (isa(dirList,/NULL)) then begin
    ; a null object => current directory is empty or something else went wrong
    date = [date,'']
    fileName = [fileName,'']
    listing = [listing,'']
  endif else begin
    ;; oDaveFtpUrl is a DaveHttpRequest object
    ; dirlist is an OrderedHash()
    ; contains 3 entries
    ; "subdirs", "files_metadata", "pathlist"
    dirs = dirList["subdirs"]    ; A list()
    nDir = dirs.Count()
    filesAndMetadata = dirList["files_metadata"]
    nFiles = filesAndMetadata.Count()
    dates = []
    if (nFiles gt 0) then begin
      files = filesAndMetadata.Keys()
      foreach value, filesAndMetadata do dates = [dates,value["mtime"]]
    endif
    ; filesAndMetadata structure example
    ;{
    ;"fp_monoBlade01_244.nxs.bt7": {
    ;"mtime": 1691589083,
    ;"sha256": "0d4d2db43b8c10d37b7ffa521a1983c91aef0b75bd5a6bf9130b5e856b8c0921"
    ;  },
    ;}
    for i=0,nDir-1 do begin
      filetype = [fileType,'d']
      fileName = [fileName,dirs[i]]
      line = String("Directory",dirs[i],format='(A,T20,A)')
      listing = [listing,line]
    endfor
    for i=0,nFiles-1 do begin
      filetype = [fileType,'f']
      if (~Strmatch(files[i],filter,/fold)) then continue ; use use-defined filer to filter out filenames that do not match the filter
      fileName = [fileName,files[i]]
      dateString = Systime(elapsed=dates[i],/utc)
      line = String(Dialog_ncnrpublicdata_reformatdata(dateString),files[i],format='(A,T20,A)')
      listing = [listing,line]
    endfor
  endelse
endif else begin
  ;; oDaveFtpUrl is a DaveFtpUrl object
  if (Strcmp(dirList[0],'')) then begin
    ; an empty string => current directory is empty
    date = [date,'']
    fileName = [fileName,'']
    listing = [listing,'']
  endif else begin
    for i=0,n-1 do begin
      toks = Strsplit(dirList[i],' ',/extract,count=ntoks)
      date = Strjoin(toks[5:7],' ')
      ;    date = [date,strjoin(toks[5:7],' ')]
      fname = toks[8]
      ft = Strmid(toks[0],0,1)
      fileSize = (Strcmp(ft,'d'))? 'Directory' : toks[4]
      if (Strcmp(ft,'d')) then begin
        ; this is a directory
        fileSize = 'Directory'
      endif else begin
        ; this is a file. If the file does  not match the filter string then
        ; don't include it in the listing by proceeding to the next loop iteration
        fileSize = toks[4]
        if (~Strmatch(fname,filter,/fold)) then continue
      endelse
      line = String(date,fileSize,fname,format='(A,T25,A,T40,A)')
      listing = [listing,line]
      filetype = [fileType,ft]
      fileName = [fileName,fname]
    endfor
  endelse
endelse

if (~isa(listing)) then listing = ''
if (~isa(filename)) then filename = ''
if (~isa(filetype)) then filetype = ''

if (ptr_valid((*sPtr).fileNamePtr)) then begin
  (*(*sPtr).fileNamePtr) = fileName
;  (*(*sPtr).datePtr) = date
  (*(*sPtr).listingPtr) = listing
  (*(*sPtr).fileTypePtr) = fileType
endif else begin
  (*sPtr).fileNamePtr = ptr_new(fileName)
;  (*sPtr).datePtr = ptr_new(date)
  (*sPtr).listingPtr = ptr_new(listing)
  (*sPtr).fileTypePtr = ptr_new(fileType)
endelse

(*sPtr).oDaveFtpUrl->GetProperty, currentDir=curDir
widget_control, (*sPtr).wList, set_value=listing
widget_control, (*sPtr).wStatusLabel, set_value=curDir

end

;==============================================================================
function Dialog_NCNRpublicData_callback, StatusInfo, ProgressInfo, CallbackData
compile_opt idl2

;print,'Start__________________________________________________________'
;print,'Status: ', StatusInfo
;print,'Progress[0]: ', ProgressInfo[0]
;;help, CallbackData
;print,'End============================================================'

vCancelFlag = 1 ; set to 0 to cancel the ftp request

if (~ptr_valid(CallbackData)) then return, vCancelFlag
 
; Expose statusinfo in the statustext field
widget_control, (*CallbackData).wStatusText, set_value=StatusInfo

; while we are receiving info from server (progressinfo[0] eq 1) then 
; make the interupt button sensitive
widget_control, (*CallbackData).wInterupt, sensitive=ProgressInfo[0] 

if ((*CallbackData).interupt eq 1) then begin
  ; this means the interupt button was pressed hence we need to cancel the 
  ; ftp request by returning a 0 to the caller of this callback function
  vCancelFlag = 0
  (*CallbackData).interupt = 0  ; reset the interupt
endif

return, vCancelFlag
end


;;==============================================================================
;pro Dialog_NCNRpublicData_connect, sPtr
;compile_opt idl2
;
;if (obj_valid((*sPtr).oURL)) then return
;
;  ; create a new url object
;  oUrl = OBJ_NEW('IDLnetUrl', callback_function='Dialog_NCNRpublicData_callback', callback_data=sPtr)
;
;  ; This is an ftp transaction
;  oUrl->SetProperty, URL_SCHEME = 'ftp'
;
;  ; Set the server name
;  oUrl->SetProperty, URL_HOSTNAME = (*sPtr).URL
;
;  ; Set the appropriate username and password
;  oUrl->SetProperty, URL_USERNAME = ''
;  oUrl->SetProperty, URL_PASSWORD = ''
;
;  ; the default server directory
;  oUrl->SetProperty, URL_PATH = (*sPtr).baseDir
;
;  (*sPtr).oURL = oURL
;end

;==============================================================================
pro Dialog_NCNRpublicData_events, event
compile_opt idl2
widget_control, event.top, get_uvalue = sPtr

catch, errorStatus
if (errorStatus ne 0) then begin
  catch,/cancel
  r = dialog_message(!error_state.msg, title=(*sPtr).title, dialog_parent=(*sPtr).wTLB, /error)
  return
endif

uname = widget_info(event.id,/uname)

case uname of 
  'DISMISS': begin
    ; clear the selected files ptr and exit
    if (ptr_valid((*sPtr).selectedFilesPtr)) then (*(*sPtr).selectedFilesPtr) = ''
    widget_control, event.top, /destroy
  end
  
  'REFRESH': Dialog_NCNRpublicData_refresh, sPtr
  
  'TOPDIR': Dialog_NCNRPublicDara_topDir, sPtr
  
  'UPDIR': Dialog_NCNRPublicDara_upDir, sPtr
  
  'ACCEPT': Dialog_NCNRPublicData_acceptFiles, sPtr
  
  'DIRLIST': Dialog_NCNRpublicData_listEvent, event
  
  'INTERUPT': (*sPtr).interupt = 1
  
  'FILTER': Dialog_NCNRPublicData_filterEvent, event

  else:
endcase

end


;==============================================================================
pro Dialog_NCNRpublicData_cleanup, wTLB
compile_opt idl2
widget_control, wTLB, get_uvalue = sPtr

;if ((*sPtr).closeconnections) then (*sPtr).oDaveFtpUrl->CloseConnections
(*sPtr).oDaveFtpUrl->CloseConnections
ptr_free, sPtr

end


;==============================================================================
; Dialog_NCNRpublicData
;+
;
; Purpose:
;   Simple dialog that can be used to browse the contents of a remote server defined by
;   the specified DAVEftpURL() object. It's functionality is designed to be similar
;   to dialog_pickfile() and as such can be used to select files of interest that can be 
;   used by the calling program.
;
; Parameters:
;   oDAVEFtpUrl -  An input parameter specifying an object reference to the remote server 
;                  to be browsed
; Keywords:
;   group_leader - Set this to the widget ID to be used as the parent of this dialog.
;   
;   title - set this to the title string of the dialog. Default is 'NCNR Public Data'
;   
;   filter - set this to an string array specifying file types to be displayed. 
;            Default is ['*.*']
;   
;   count - a named variable that will contained the number of files selected on exit.
;      
; Return:
;   The returned value is a string array specifing the files that were selected or an empty string if
;   either no files were selected or the dialog was dismissed.
;
; Example:
;   oFtp = DAVEftpURL()
;   files = Dialog_NCNRpublicData(oFtp,count=cnt) ; browse and select at least one file
;   if (cnt gt 0) then $
;     fileContents = oFtp->GetFileContents(files[0],/string)  ; retrieve contents of first file and save in a string array
;
;-
function Dialog_NCNRpublicData, oDaveFtpUrl,group_leader=gl, count=count, title=title, filter=filter
compile_opt idl2
on_error, 2

IDLge90 = (float(!version.release) ge 9.0)? 1 : 0
IDLge91 = (float(!version.release) ge 9.1)? 1 : 0
count = 0
if (n_elements(title) eq 0) then title = 'NCNR Public Data'
; Error handler for entry procedure
catch, errorStatus
if (errorStatus ne 0) then begin
  catch,/cancel
  r = dialog_message(!error_state.msg, title=title, dialog_parent=rwBase, /error)
  if (ptr_valid(sPtr)) then ptr_free, sPtr
  return, ''
endif

; if first parameter is not defined then define it
; if IDL version >= 9.0 then use an HttpRequest object to access NCNR Data Repository
; otherwise use a IDLnetURL based object.
; Basically because FTP is unsecured 
; However, HttpRequest class only introduced in IDL 9.0

if (IDLge90) then begin
  if (isa(oDaveFtpUrl,'DAVEFtpUrl')) then begin
    ;; if a DAVEFtpUrl object was passed in then do some cleanup
    oDaveFtpUrl->getProperty, currentDir=currentDir ; retrieve the current directory setting
;    currentDir = currentDir.replace("/pub","",/fold); remove the "/pub" portion of the path because
;                                                    ; it is assumed when using the HttpRequest object so "/pub/ncnrdata/bt7/" becomes "/ncnrdata/bt7/"
;                                                    
    oldDaveFtpUrl = oDaveFtpUrl
    oDaveFtpUrl = DAVEHttpRequest()                 ; use a custom HttpRequest object instead
    oDaveFtpUrl->SetProperty, currentDir=currentDir ; assign the retrieved current directory
  endif else begin
    if (~Obj_valid(oDaveFtpUrl)) then oDaveFtpUrl = DAVEHttpRequest()   
  endelse
endif else begin
  closeconnections = 0
  if (~Obj_valid(oDaveFtpUrl)) then begin
    closeconnections = 1
    oDaveFtpUrl = Daveftpurl()
  endif  
endelse

textWidth = 65
textHeight = 20
if (~isa(filter)) then filter = ['*.*'] ; if not defined, specify a default filter

gl = (n_elements(gl) gt 0)? gl : 0L
hasGL = widget_info(gl,/valid)

wTLB = widget_base(/col, /ALIGN_LEFT, TITLE=title, uname='WTLB' $
                  ,tlb_frame_attr=1 $
                  ,group_leader=gl $
                  ,floating=hasGL $
                  ,modal=hasGL $
                  )

wCB0 = widget_base(wTLB, /row)
wCB1 = widget_base(wTLB, /row)

wCB2 = widget_base(wCB0, /col)
wCB3 = widget_base(wCB0, /col)

wCB4 = widget_base(wCB1, /col)
wCB5 = widget_base(wCB1, /col)

font2 = (!version.os_family eq 'Windows')? 'Courier New*16' : $
       '-adobe-courier-medium-o-normal--14-140-75-75-m-90-iso8859-1'

; Label to display current directory on server
if (IDLge91) then begin
;  wStatusLabel = cw_coloredlabel(wCB2, value='NCNRData:', uname='STATUSLABEL'$
;                                ,font=font2,foreground_color=[255,0,0],/dynamic_resize)
  wStatusLabel = widget_label(   wCB2, value='NCNRData:', uname='STATUSLABEL',foreground_color=[255,0,0],/dynamic_resize,font=font2)
endif else begin
  wStatusLabel = cw_coloredlabel(wCB2, value='NCNRData:', uname='STATUSLABEL',foreground_color=[255,0,0],/dynamic_resize)
endelse


; List to display listing of current directory in server
wList = widget_list(wCB2, uname='DIRLIST', /multiple, font=font2, $
                    xsize = textWidth, ysize = textHeight)

; Instrictions label
label = 'Double click to open directory or view file contents'
if (IDLge91) then begin
;  wInsLabel = cw_coloredlabel(wCB2, value=label, uname='INSLABEL', font=font2 $
;                             ,foreground_color=[0,0,255],/align_center)
  wInsLabel = widget_label( wCB2, value=label, uname='INSLABEL',foreground_color=[0,0,255],/align_center, font=font2)
endif else begin
  wInsLabel = cw_coloredlabel(wCB2,value=label,uname='INSLABEL',foreground_color=[0,0,255],/align_center)
endelse
;help,'wstatusLabel,winslabel:',wstatusLabel,winslabel

void = widget_label(wCB3, value=' ')
wFilter = widget_combobox(wCB3, value=filter,uname='FILTER',/editable)
void = widget_label(wCB3, value=' ')
wTopDir = widget_button(wCB3, value='To Top Dir',uname='TOPDIR')
wUpDir = widget_button(wCB3, value='Up One Dir',uname='UPDIR')
void = widget_label(wCB3, value=' ')
wRefresh = widget_button(wCB3, value='Refresh Listing',uname='REFRESH')
void = widget_label(wCB3, value=' ')
wLoad = widget_button(wCB3, value='Accept Selected Files',uname='ACCEPT')
void = widget_label(wCB3, value=' ')
wDismiss = widget_button(wCB3, value='Dismiss Dialog',uname='DISMISS')

geom = widget_info(wList,/geom)
wStatusText = widget_text(wCB4, value='', uname='STATUSTEXT', scr_xsize=geom.scr_xsize)
geom = widget_info(wLoad,/geom)
wInterupt = widget_button(wCB5, value='Interupt Transfer', uname='Interupt',scr_xsize=geom.scr_xsize,sensitive=0)

; make combobox width the same as the other buttons
geom = widget_info(wTopDir,/geom)
widget_control, wFilter, scr_xsize=geom.scr_xsize
; 
; Create a state structure that holds all the widget IDs
; This structure will be stored in the top-level widget
sPtr = ptr_new({wTLB:wTLB $
  ,wStatusLabel:wStatusLabel $
  ,wStatusText:wStatusText $
  ,wInterupt:wInterupt $
  ,wList:wList $
  ,wUpDir:wUpDir $
  ,wLoad:wLoad $
  ,wFilter:wFilter $
;  ,datePtr:ptr_new() $
  ,fileTypePtr:ptr_new() $
  ,fileNamePtr:ptr_new() $
;  ,TopDir:'' $
  ,listingPtr:ptr_new() $  
;  ,URL:URL $
  ,interupt:0 $
  ,selectedFilesPtr:ptr_new() $
  ,title:title $
;  ,baseDir:baseDir $
;  ,currentDir:baseDir $
  ,oDaveFtpUrl:oDaveFtpUrl $
  })
  
; Add the state structure to the top-level widget
widget_control, wTLB, set_uvalue = sPtr

; Register a Callback fuction with the ftp object
oDaveFtpUrl->SetProperty, callback_function = 'Dialog_NCNRpublicData_callback'
; Register sPtr as callback data for the ftp object
oDaveFtpUrl->SetProperty, callback_data = sPtr

; Connect to the ftp server
;Dialog_NCNRpublicData_connect, sPtr


; draw the ui
widget_control, wTLB, /real

; Refresh the directory listing
Dialog_NCNRpublicData_refresh, sPtr  

;; NO BLOCK needs to be set to 0 in order for the build query events to fire
XMANAGER,'Dialog_NCNRpublicData', wTLB, event_handler='Dialog_NCNRpublicData_events', no_block=0

filenames = ptr_valid((*sPtr).selectedFilesPtr)?  (*(*sPtr).selectedfilesPtr) : ''
count = n_elements(filenames)
if (filenames[0] eq '') then count=0

if (isa(oldDaveFtpUrl)) then begin
  ; before exiting, update the cuurentdir of the old ftp object
  (*sPtr).oDaveFtpUrl->GetProperty, currentDir=curDir
  oldDaveFtpUrl->SetProperty, currentDir=curDir
endif

(*sPtr).oDaveFtpUrl->CloseConnections
ptr_free, sPtr

return, filenames
end