; $Id$
;###############################################################################
;+
; NAME:
;   cw_bt7psdDataManager
;
; PURPOSE:
;   Construct the widgets representing the data manager browser section of the
;   BT7 PSD Reduction user interface.
;
; CATEGORY:
;   DAVE Main Tool
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; September 2011
;-
;###############################################################################

;===============================================================================
; cw_bt7psdDataManager_ViewDetail 
;+
; Purpose:
;   View the contents of a data element in the DAVE Data Manager
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;-
pro cw_bt7psdDataManager_ViewDetail, sPtr, wTLB
compile_opt idl2

;; Get the item
;id = cw_itTreeView_getSelect((*sPtr).wDM, count=count)
;if(count gt 0)then begin
id = (*sPtr).identifier
if (~strcmp(id,'')) then begin
    oTool = (*sPtr).oUI->Gettool()
    oItem = oTool->GetByIdentifier(id)              
    if(obj_valid(oItem))then begin
        ;idParent = cw_itTreeView_getParent((*sPtr).wDM, id[0])
        font = (!version.os_family eq 'Windows')? 'Courier New*24*Bold' : $
               '-adobe-courier-bold-o-normal--24-240-75-75-m-150-iso8859-1'
        font1 = (!version.os_family eq 'Windows')? 'Courier New*18*Bold' : $
               '-adobe-courier-bold-o-normal--18-180-75-75-m-110-iso8859-1'
        font2 = (!version.os_family eq 'Windows')? 'Courier New*16' : $
               '-adobe-courier-medium-o-normal--14-140-75-75-m-90-iso8859-1'

        oItem->GetProperty, name=title, description=filename
        status = file_test(filename,/read)
        doneTxt = "Close "+title

        if (status) then begin
           xdisplayfile, filename, done_button=doneTxt, /grow_to_screen, group=wTLB, font=font2
        endif else begin
           fromFtp = (stregex(filename,'/pub/ncnrdata/',/fold,/bool))? 1 : 0
           if (fromFTP) then begin
              oTool->GetProperty, ftpObject=oFTP
              buffer = oFTP->GetFileContent(filename,/string)
              if (strcmp(buffer[0],'')) then return
              nlines = n_elements(buffer)
              xdisplayfile, filename, text=buffer, title=filename, done_button=doneTxt, /grow_to_screen, group=wTLB, font=font2
           endif else begin
              void = dialog_message("Could not retrieve contents",/error, $
                                  dialog_parent=wTLB, $
                                  title="No Data")        
           endelse
        endelse
        ;wd_viewDetail, oItem, group_leader=wTLB
    endif else $
      void = dialog_message("Could not retrieve contents",/error, $
                            dialog_parent=wTLB, $
                            title="No Data")
endif else $
  void = dialog_message("No data item selected",/error, $
                        dialog_parent=wTLB, $
                        title="No Selection")
end


;===============================================================================
; cw_bt7psdDataManager_ViewDetail 
;+
; Purpose:
;   Treatment history viewer
;
; Parameters
;    sPtr - The state pointer for the widget from which all relevant information is stored
;
;    wTLB - The top of this widget tree.
;-
pro cw_bt7psdDataManager_ViewHist, sPtr, wTLB
compile_opt idl2

oTool = (*sPtr).oUI->Gettool()
if(obj_valid(oTool))then begin
    oTool->GetProperty, history=history
    if (n_elements(history) gt 0) then begin
       font = (!version.os_family eq 'Windows')? 'Courier New*24*Bold' : $
              '-adobe-courier-bold-o-normal--24-240-75-75-m-150-iso8859-1'
       font1 = (!version.os_family eq 'Windows')? 'Courier New*18*Bold' : $
              '-adobe-courier-bold-o-normal--18-180-75-75-m-110-iso8859-1'
       font2 = (!version.os_family eq 'Windows')? 'Courier New*16' : $
              '-adobe-courier-medium-o-normal--14-140-75-75-m-90-iso8859-1'

        xdisplayfile, text=history, done_button=doneTxt, /grow_to_screen, group=wTLB, font=font2
    endif else begin
       void = dialog_message("Could not retrieve history",/error, $
                           dialog_parent=wTLB, $
                           title="No data")
    endelse
    ;wd_viewDetail, oItem, group_leader=wTLB
endif 

end


;===============================================================================
; cw_bt7psdDataManager_DeleteData 
;-
; Purpose:
;   Encapsulates the functionality used to delete data. Based on
;   _cw_itDataManager_DeleteData.  The basic difference results fron
;   the fact that the DAVEDATA_MANAGER service is used instead of
;   DATA_MANAGER.
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;-
pro cw_bt7psdDataManager_DeleteData, sPtr, wTLB, deltree=deltree, delall=delall, noPrompt=noPrompt
compile_opt idl2

oTool = (*sPtr).oUI->Gettool()

if (~keyword_set(delall)) then begin
   ;; Which data tree are we deleting from?
   ;; use the most recent selected item to find out
   if (stregex((*sPtr).identifier,'SAMP',/fold_case) ge 0) then begin
      dataType='samp'
      wDM = (*sPtr).wDM
      oTool->GetProperty, nSamp=nData, sampContRef=oCont, sumSampFlag=sumFlag
      sumLabel = 'SUMSAMPFLAG'
   endif
   if (stregex((*sPtr).identifier,'BKGD',/fold_case) ge 0) then begin
      dataType='bkgd'
      wDM = (*sPtr).wDM2
      oTool->GetProperty, nBkgd=nData, bkgdContRef=oCont, sumBkgdFlag=sumFlag
      sumLabel = 'SUMBKGDFLAG'
   endif
   if (n_elements(dataType) eq 0) then return
endif

if (keyword_set(deltree)) then begin
   ;; Delete all data in samp or bkgd or fast branches
   msg = ["Deleting all "+dataType+" dataset(s):",'']
   oArr = oCont->Get(/all, count=count)
   if(count lt 1)then return
   for i=0,count-1 do begin
      id = oArr[i]->GetFullIdentifier()
      oArr[i]->GetProperty, name=name
      iArr = (n_elements(iArr) eq 0)? id : [iArr,id]
      msg = [msg,name]
   endfor
endif else if (keyword_set(delall)) then begin
   ;; Deleting all data in samp, bkgd and fast branches
   msg = ["Deleting all dataset(s):",'']
   oTool->GetProperty, sampContRef=oSampCont
   oTool->GetProperty, bkgdContRef=oBkgdCont
   ;oTool->GetProperty, fastContRef=oFastCont
   oArr1 = oSampCont->Get(/all, count=scount)
   oArr2 = oBkgdCont->Get(/all, count=bcount)
   ;oArr3 = oFastCont->Get(/all, count=fcount)
   if (scount gt 0) then $
      oArr = (n_elements(oArr) gt 0)? [oArr,oArr1] : oArr1
   if (bcount gt 0) then $
      oArr = (n_elements(oArr) gt 0)? [oArr,oArr2] : oArr2
   ;if (fcount gt 0) then $
   ;   oArr = (n_elements(oArr) gt 0)? [oArr,oArr3] : oArr3
      count = n_elements(oArr)
   if(count lt 1)then return
   for i=0,count-1 do begin
      oArr[i]->GetProperty, name=name
      msg = [msg,name]
   endfor
endif else begin
   ;; Deleting only selected datasets

   ; Retrieve the detasets to be deleted
   msg = ["Deleting selected dataset(s):",'']
   if (n_elements(wDM) eq 0) then return
   id = cw_itTreeView_getSelect(wDM, count=count)
   if(count lt 1)then return
   for i=0,count-1 do begin
      oItem = oTool->GetByIdentifier(id[i])
      oItem->GetProperty, name=name
      oArr = (n_elements(oArr) eq 0)? oItem : [oArr,oItem]
      iArr = (n_elements(iArr) eq 0)? id[i] : [iArr,id[i]]
      msg = [msg,name]
   endfor
endelse


if (~keyword_set(noPrompt)) then begin
   title = "Delete Data"
   msg = [msg,' ','Are you sure?']
   status = dialog_message(title=title, msg, $
                           /question, /default_no, dialog_parent=wTLB)
   if (status eq 'No') then return
endif

oDM = oTool->GetService("DATA_MANAGER")

if (keyword_set(delTree)) then begin
   ;; Delete all datasets from specified tree
   oTool->DisableUpdates
   for i=0,count-1 do begin
      void = oArr[i]->GetMetaData('DATAPTRREF',dataPtr) ; the real data is in this dataPtr
      if (ptr_valid(dataPtr)) then heap_free, dataPtr   ; so must free it!
      oCont->Remove, oArr[i]   ; remove obj ref from container
      ; sum data don't have a full identifier so don't decrement nData and can't use DM service to delete
      if (~strcmp(oArr[i]->GetFullIdentifier(),'')) then begin
         nData--
         void = oDM->DeleteData(iArr[i])   ; finally delete obj ref from data manager
      endif else $
         if (obj_valid(oArr[i])) then obj_destroy, oArr[i]   ; just a double check
   endfor
   oTool->EnableUpdates

   case dataType of
      'samp': begin 
         oTool->SetProperty, nSamp=nData
         if (nData lt 1) then oTool->SetPropertyAttribute, 'SUMSAMPFLAG', sensitive=0
       end 
      'bkgd': begin
         oTool->SetProperty, nBkgd=nData
         if (nData lt 1) then begin
           oTool->SetPropertyAttribute, 'SUMBKGDFLAG', sensitive=0
           oTool->SetPropertyAttribute, 'BKGDFLAG', sensitive=0
           oTool->SetPropertyAttribute, 'bkgdSubVarFlag', sensitive=0
           oTool->SetPropertyAttribute, 'bkgdSubVarTol', sensitive=0
         endif
       end
      'fast': begin
         oTool->SetProperty, nFast=nData
         if (nData lt 1) then oTool->SetPropertyAttribute, 'SUMFASTFLAG', sensitive=0
       end
      else: return
   endcase

   oTool->refreshPropertySheet   ; force prop sheet to update

   ;; update context menu
   widget_control, (*sPtr).wView, sensitive=0
   widget_control, (*sPtr).wDel, sensitive=0
   widget_control, (*sPtr).wDelSamp, sensitive=0
   widget_control, (*sPtr).wDelAll, sensitive=0
;   widget_control, (*sPtr).wHist, sensitive=0
   widget_control, (*sPtr).wReload, sensitive=0
   widget_control, (*sPtr).wMSLICE, sensitive=0
   widget_control, (*sPtr).wDAVE, sensitive=0
   widget_control, (*sPtr).wASCII, sensitive=0
   widget_control, (*sPtr).wDAVEDM, sensitive=0
   Widget_control, (*sPtr).wCOMBDAVEDM, sensitive=0
   Widget_control, (*sPtr).wCOMBDAVE, sensitive=0
   Widget_control, (*sPtr).wCOMBPAN, sensitive=0
   widget_control, (*sPtr).wPAN, sensitive=0
;   widget_control, (*sPtr).wFFast, sensitive=0

   oTool->SetProperty, modifiedStatus = 1
   return
endif

if (keyword_set(delAll)) then begin
   ;; Delete all datasets from all three trees
   if (scount gt 0) then begin
      oTool->DisableUpdates
      oCont = oSampCont
      for i=0,scount-1 do begin
         void = oArr1[i]->GetMetaData('DATAPTRREF',dataPtr) ; the real data is in this dataPtr
         if (ptr_valid(dataPtr)) then heap_free, dataPtr   ; so must free it!
         oCont->Remove, oArr1[i]   ; remove obj ref from container
         void = oDM->DeleteData(oArr1[i]->GetfullIdentifier())   ; finally delete obj ref from data manager
         if (obj_valid(oArr1[i])) then obj_destroy, oArr1[i]
      endfor
      oTool->SetProperty, nSamp=0
      oTool->SetPropertyAttribute, 'SUMSAMPFLAG', sensitive=0
      oTool->EnableUpdates
   endif
   
   if (bcount gt 0) then begin
      oTool->DisableUpdates
      oCont = oBkgdCont
      for i=0,bcount-1 do begin
         void = oArr2[i]->GetMetaData('DATAPTRREF',dataPtr) ; the real data is in this dataPtr
         if (ptr_valid(dataPtr)) then heap_free, dataPtr   ; so must free it!
         oCont->Remove, oArr2[i]   ; remove obj ref from container
         void = oDM->DeleteData(oArr2[i]->GetfullIdentifier())   ; finally delete obj ref from data manager
         if (obj_valid(oArr2[i])) then obj_destroy, oArr2[i]   ; just a double check
      endfor
      oTool->SetProperty, nBkgd=0
      oTool->SetPropertyAttribute, 'SUMBKGDFLAG', sensitive=0
      oTool->SetPropertyAttribute, 'BKGDFLAG', sensitive=0
      oTool->SetPropertyAttribute, 'bkgdSubVarFlag', sensitive=0
      oTool->SetPropertyAttribute, 'bkgdSubVarTol', sensitive=0
      oTool->EnableUpdates
   endif

   oTool->refreshPropertySheet   ; force prop sheet to update

   ;; update context menu
   widget_control, (*sPtr).wView, sensitive=0
   widget_control, (*sPtr).wDel, sensitive=0
   widget_control, (*sPtr).wDelSamp, sensitive=0
   widget_control, (*sPtr).wDelAll, sensitive=0
;   widget_control, (*sPtr).wHist, sensitive=0
   widget_control, (*sPtr).wReload, sensitive=0
   widget_control, (*sPtr).wMSLICE, sensitive=0
   widget_control, (*sPtr).wDAVE, sensitive=0
   widget_control, (*sPtr).wASCII, sensitive=0
   widget_control, (*sPtr).wDAVEDM, sensitive=0
   widget_control, (*sPtr).wPAN, sensitive=0
   Widget_control, (*sPtr).wcombdavedm, sensitive=0
   Widget_control, (*sPtr).wcombdave, sensitive=0
   Widget_control, (*sPtr).wcombpan, sensitive=0
;   widget_control, (*sPtr).wFFast, sensitive=0

   oTool->SetProperty, modifiedStatus = 1
   return
endif



; Delete selected datasets
isSummed = 0
;; check whether deleting datasets that are part of a sum
;; if so then the sum will have to be deleted and recalculated
isSummed = sumFlag && (oCont->Count() eq nData + 1)
if (isSummed) then sumData = oCont->get(position=nData)

;; proceed with the deleting
oTool->DisableUpdates
for i=0,count-1 do begin
   void = oArr[i]->GetMetaData('DATAPTRREF',dataPtr) ; the real data is in this dataPtr
   if (ptr_valid(dataPtr)) then heap_free, dataPtr   ; so must free it!
   oCont->Remove, oArr[i]   ; remove obj ref from container
   nData--
   case dataType of
      'samp': begin 
         oTool->SetProperty, nSamp=nData
         if (nData lt 1) then oTool->SetPropertyAttribute, 'SUMSAMPFLAG', sensitive=0
       end 
      'bkgd': begin
         oTool->SetProperty, nBkgd=nData
         if (nData lt 1) then oTool->SetPropertyAttribute, 'SUMBKGDFLAG', sensitive=0
       end
      'fast': begin
         oTool->SetProperty, nFast=nData
         if (nData lt 1) then oTool->SetPropertyAttribute, 'SUMFASTFLAG', sensitive=0
       end
      else: return
   endcase
   
   void = oDM->DeleteData(iArr[i])   ; finally delete obj ref from data manager
   if (obj_valid(oArr[i])) then obj_destroy, oArr[i]   ; just a double check
endfor

;;; Update the sum if necessary
;if (isSummed) then begin
;   if (obj_valid(sumData)) then begin
;      void = sumData->GetMetaData('DATAPTRREF',dataPtr) ; the real data is in this dataPtr
;      if (ptr_valid(dataPtr)) then heap_free, dataPtr   ; so must free it!
;
;      oCont->remove, sumData
;      obj_destroy, sumData
;   endif
;   if (nData gt 0) then oTool->MergeData, dataType 
;endif
;
;oTool->refreshPropertySheet   ; force prop sheet to update

oTool->Updatecounts                ; update counts in remaining dataset because bkgd data could have been deleted, for example
if (nData gt 0) then begin
   ;; Update plot
   oItem = oCont->Get(position = 0)   ; Get the first dataset
   itemID = oItem->GetFullIdentifier()
   (*sPtr).identifier = itemID
   cw_bt7psdDataManager_setselect, wDM, itemID, /clear ; select it
   (*(*sPtr).pDataIDs) = itemID  ; also update internal selection record

   wd_BT7PSDtool_sendPlotEvent, (*sPtr).oUI, oItem   ; plot it
endif

oTool->UpdatePropertysheet
oTool->RefreshPropertySheet
oTool->EnableUpdates

oTool->SetProperty, modifiedStatus = 1

end


;===============================================================================
; cw_bt7psdDataManager_COMBTODAVEDM
;-
; Purpose:
;   Combine selected data and the save the results to the Main DAVE Data Manager
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;-
pro cw_bt7psdDataManager_COMBTODAVEDM, sPtr, wTLB 
compile_opt idl2

oTool = (*sPtr).oUI->Gettool()

status = oTool->CombineToDM()

end


;===============================================================================
; cw_bt7psdDataManager_exportToDAVEDM
;-
; Purpose:
;   Export selected data to the Main DAVE Data Manager
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;-
pro Cw_bt7psddatamanager_exporttodavedm, sPtr, wTLB
  compile_opt idl2

  oTool = (*sPtr).oui->Gettool()

  status = oTool->Exporttodavedm()

end


;===============================================================================
; Cw_bt7psddatamanager_COMBTOPAN
;+
; Purpose:
;   Combine selected data to PAN for fitting
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;-
pro Cw_bt7psddatamanager_COMBTOPAN, sPtr, wTLB 
compile_opt idl2

;; Use the most recent selected item to find out dataset(s) to be saved
if (stregex((*sPtr).identifier,'SAMP',/fold_case) lt 0) then return
dataType='samp'
wDM = (*sPtr).wDM

;
oTool = (*sPtr).oUI->Gettool()

status = oTool->CombineToPAN()

end


;===============================================================================
; cw_bt7psdDataManager_exportToPAN
;+
; Purpose:
;   Send selected data to PAN for fitting
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;-
pro Cw_bt7psddatamanager_exporttopan, sPtr, wTLB
  compile_opt idl2

  ;; Use the most recent selected item to find out dataset(s) to be saved
  if (Stregex((*sPtr).identifier,'SAMP',/fold_case) lt 0) then Return
  dataType='samp'
  wDM = (*sPtr).wdm

  ;
  oTool = (*sPtr).oui->Gettool()

  status = oTool->Exporttopan()

end


;===============================================================================
; cw_bt7psdDataManager_exportToMSLICE
;+
; Purpose:
;   Send selected data to Mslice for analysis
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;-
pro cw_bt7psdDataManager_exportToMslice, sPtr, wTLB 
compile_opt idl2

;; Use the most recent selected item to find out dataset(s) to be saved
if (stregex((*sPtr).identifier,'SAMP',/fold_case) lt 0) then return
dataType='samp'
wDM = (*sPtr).wDM

;
oTool = (*sPtr).oUI->Gettool()
void = oTool->Mslice()


end


;===============================================================================
; cw_bt7psdDataManager_saveToDisk
;+
; Purpose:
;   Combine and/or Save selected data to file in DAVE|ASCII format
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;-
pro cw_bt7psdDataManager_saveToDisk, sPtr, wTLB, format=format 
compile_opt idl2

;; Use the most recent selected item to find out dataset(s) to be saved
if (stregex((*sPtr).identifier,'SAMP',/fold_case) lt 0) then return
dataType='samp'
wDM = (*sPtr).wDM

;
oTool = (*sPtr).oUI->Gettool()

case format of
   'dave': status = oTool->SaveAsDave()
   'ascii': status = oTool->SaveAsASCII()
   'comb': status = oTool->CombToDAVE()
   else:
endcase

end


;===============================================================================
; cw_bt7psdDataManager_ReloadData 
;+
; Purpose:
;   Reload the selected dataset i.e. essentially update its contents
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;
pro cw_bt7psdDataManager_ReloadData, sPtr, wTLB
compile_opt idl2

;; Which data tree are we deleting from?
;; use the most recent selected item to find out
if (stregex((*sPtr).identifier,'SAMP',/fold_case) ge 0) then begin
   dataType='samp'
   wDM = (*sPtr).wDM
endif
if (stregex((*sPtr).identifier,'BKGD',/fold_case) ge 0) then begin
   dataType='bkgd'
   wDM = (*sPtr).wDM2
endif

; Retrieve the detasets to be deleted
if (n_elements(wDM) eq 0) then return
id = cw_itTreeView_getSelect(wDM, count=count)
if(count lt 1)then return

oTool = (*sPtr).oUI->Gettool()

for i=0,count-1 do begin
   oItem = oTool->GetByIdentifier(id[i])
   case dataType of
      'samp': status = oTool->LoadSamp(dataObject=oItem, /reload)
      'bkgd': status = oTool->LoadBkgd(dataObject=oItem, /reload)
      'fast': status = oTool->LoadFast(dataObject=oItem, /reload)
      else: return
   endcase
endfor

if (status) then begin
   ;; Update plot
   itemID = oItem->GetFullIdentifier()
   (*sPtr).identifier = itemID
   cw_bt7psdDataManager_setselect, wDM, itemID, /clear ; select it
   (*(*sPtr).pDataIDs) = itemID  ; also update internal selection record
   

   wd_BT7PSDtool_sendPlotEvent, (*sPtr).oUI, oItem   ; plot it
endif

end



;===============================================================================
; cw_bt7psdDataManager_RenameData 
;+
; Purpose:
;   Rename name tag of selected item/dataset
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;-
pro cw_bt7psdDataManager_RenameData, sPtr, wTLB
compile_opt idl2

;; Which data tree are we renaming from?
;; use the most recent selected item to find out
if (stregex((*sPtr).identifier,'SAMP',/fold_case) ge 0) then wDM = (*sPtr).wDM
if (stregex((*sPtr).identifier,'BKGD',/fold_case) ge 0) then wDM = (*sPtr).wDM2


;; Get the item
id = cw_itTreeView_getSelect(wDM, count=count)
if(count gt 0)then begin
    oTool = (*sPtr).oUI->Gettool()
    oItem = oTool->GetByIdentifier(id[0])              
    if(obj_valid(oItem))then begin
        oItem->GetProperty, name=name
        status = idlitwdprompttext(wTLB, $
                                   "Enter New Data Name:", $
                                   newName, $
                                   value=name, $
                                   title="Rename Data")
        if(status eq 1 and keyword_set(newName))then begin
            oItem->SetProperty, name=newName
            ;; Tell the tree to rebild
            cw_itTreeView_Callback, (*sPtr).wDM, id[0], "UPDATEITEM",id[0]

        endif
    endif else $
      void = dialog_message("Invalid Data Item.",/warning, $
                            dialog_parent=wTLB, $
                            title="Invalid Data")
endif else $
  void = dialog_message("No data item selected",/warning, $
                        dialog_parent=wTLB, $
                        title="No Data")

end


;===============================================================================
; cw_bt7psdDataManager_Cleanup
;+ 
; PURPOSE:
;   Widget cleanup procedure.
;
; PARAMETERS
;   widID (in)  - The id of the base widget to be destroyed
;
; KEYWORDS:
;-
pro cw_bt7psdDataManager_Cleanup, widID
compile_opt idl2

widget_control, widID, get_uvalue=sPtr
ptr_free,(*sPtr).pDataIDs,sPtr

end


;===============================================================================
; cw_bt7psdDataManager_Realize
;+ 
; PURPOSE:
;   Called when the widget is realized - checks widget dimensions.
;
; PARAMETERS
;   widID (in)  - The id of the base widget to be realized
;
; KEYWORDS:
;-
pro cw_bt7psdDataManager_Realize, widID
compile_opt idl2

widget_control, widID, get_uvalue=sPtr

;; Make sure the geometry is all right.
sGeom = widget_info(widID, /geometry)
;cw_bt7psdDataManager_Resize, widID, sGeom.xsize, (*sPtr).ySize0

end


;===============================================================================
; cw_bt7psdDataManager_GetWidgetID
;+ 
; PURPOSE:
;   Called to retrieve the widget ID of the specified item identifier
;
; PARAMETERS
;   widID (in)  - The id of the widget to be altered
;
;   idItem - The item whose widget ID is required
;
; Keywords:
;-
function cw_bt7psdDataManager_GetWidgetID, widID, idItem
compile_opt idl2

return, widget_info(widID, find_by_uname=strupcase(idItem))

end


;===============================================================================
; cw_bt7psdDataManager_SetSelect
;+ 
; PURPOSE:
;   Called to set the selection state of the tree based on the
;   identifiers of the items passed in.
;
; PARAMETERS
;   widID (in)  - The id of the widget to be altered
;
;   idItem - The item to select. Optional if CLEAR is set.
;
; Keywords:
;   CLEAR: If set, clear all selected items first. If this keyword
;       is specified then idItem need not be supplied.
;
;   UNSELECT: Will unselect, not select;
;-
pro cw_bt7psdDataManager_Setselect, wDM, idItem, UNSELECT=unselect, CLEAR=clear $
                                , dataType=dataType
compile_opt idl2

; this is a fudge to deal with multiple trees in the DM!!!
wDMC = wDM
if (keyword_set(dataType)) then begin
   ;==> appropriate wDM to pass on is a child of the input wDM
   widget_control, wDM, get_uvalue=sPtr
   case dataType of
      'samp': wDMC = (*sPtr).wDM
      'bkgd': wDMC = (*sPtr).wDM2
      else: return
   endcase
   (*(*sPtr).pDataIDs) = idItem
endif

cw_itTreeView_SetSelect,  wDMC, idItem, UNSELECT=unselect, CLEAR=clear

end


;===============================================================================
; cw_bt7psdDataManager_Setvalue
;+ 
; PURPOSE:
;   Called by widget_control when the value of a widget is to be set.
;
; PARAMETERS
;   widID (in)  - The id of the widget to be altered
;
; KEYWORDS:
;-
pro cw_bt7psdDataManager_Setvalue, wDM, value, dataType=dataType
compile_opt idl2

; this is a fudge to deal with multiple trees in the DM!!!
wDMC = wDM
if (keyword_set(dataType)) then begin
   ;==> appropriate wDM to pass on is a child of the input wDM
   widget_control, wDM, get_uvalue=sPtr
   case dataType of
      'samp': wDMC = (*sPtr).wDM
      'bkgd': wDMC = (*sPtr).wDM2
      else: return
   endcase
endif

cw_itTreeView_SetSelect,  wDMC, value

end


;===============================================================================
; cw_bt7psdDataManager_Getvalue
;+ 
; PURPOSE:
;   Called by widget_control when the value of a widget is required.
;
; PARAMETERS
;   widID (in)  - The id of the widget whose value is required
;
; KEYWORDS:
;
; RETURN VALUE:
;   The required value of the widget
;-
function cw_bt7psdDataManager_Getvalue, wDM, dataType=dataType
compile_opt idl2

; this is a fudge to deal with multiple trees in the DM!!!
wDMC = wDM
if (keyword_set(dataType)) then begin
   ;==> appropriate wDM to pass on is a child of the input wDM
   widget_control, wDM, get_uvalue=sPtr
   case dataType of
      'samp': wDMC = (*sPtr).wDM
      'bkgd': wDMC = (*sPtr).wDM2
      else: return, 0L
   endcase
endif

return, cw_itTreeView_GetSelect(wDMC)

end


;===============================================================================
; cw_bt7psdDataManager_Event
;+ 
; PURPOSE:
;   Main event handler for widget created by cw_bt7psdDataManager.
;
; PARAMETERS
;   event (in)  - The event to be handled
;
; KEYWORDS:
;
; RETURN VALUE:
;   0, if the event was completely handled here
;   Event Structure, if further processing is required.
;-
function cw_bt7psdDataManager_Event, event
compile_opt idl2


; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'cw_bt7psdDataManager_Event: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return, 0
    endif
endif

;; Get our state
widget_control, event.handler, get_uvalue=sPtr

sname = tag_names(event,/structure_name)
uname = widget_info(event.id, /uname)
;help, event, /struct

oUI = (*sPtr).oUI
oTool = (*sPtr).oUI->Gettool()

;******** Why was this (next line) here before!!!!!!!!!!!
;(*(*sPtr).pDataIDs) = ''

if (strcmp(sname,'WIDGET_TAB',/fold_case)) then begin
   ;; A new tab was selected so do the following
   ;; - check for any selected datasets
   ;; - if there are selected datasets, pick the first one and update the display with it
   case event.tab of
      0: wDM = (*sPtr).wDM   ; sample tab
      1: wDM = (*sPtr).wDM2   ; bkgd tab
      else:
   endcase
   if (n_elements(wDM) eq 0) then return, 0
   id = cw_itTreeView_getSelect(wDM, count=count)
   if (count lt 1) then return, 0

   oItem = oTool->GetByIdentifier(id[0])
   isData = obj_isa(oItem, "IDLitParameterSet") ;obj_isa(oItem, "IDLitData") && ~obj_isa(oItem,'IDL_CONTAINER')
   if (isData) then begin
      (*sPtr).identifier = id[0]   ; register data as currenty selected data
      wd_BT7PSDtool_sendPlotEvent, oUI, oItem   ; plot it
   endif

   sTypeFlag = -1
   if (obj_isa(oItem, "IDLitParameterSet")) then $
      status = oItem->GetMetaData('sampTypeFlag',sTypeFlag)

   ;; Update the context menu depending on 'selected' dataset 
   widget_control, (*sPtr).wView, sensitive=isData
   widget_control, (*sPtr).wDel, sensitive=isData
   widget_control, (*sPtr).wDelSamp, sensitive=isData
   widget_control, (*sPtr).wDelAll, sensitive=isData
;   widget_control, (*sPtr).wHist, sensitive=isData
   widget_control, (*sPtr).wReload, sensitive=isData
   dataType = ''
   if (isData) then oItem->GetProperty, type=dataType
   isSamp = strcmp(dataType,'samp',/fold_case)
   isBkgd = strcmp(dataType,'bkgd',/fold_case)
   isFast = strcmp(dataType,'fast',/fold_case)
   widget_control, (*sPtr).wMSLICE, sensitive=(isData && isSamp && (sTypeFlag eq 1))
   widget_control, (*sPtr).wDAVE, sensitive=(isData && isSamp &&(sTypeFlag eq 0 || sTypeFlag eq 2))
   widget_control, (*sPtr).wASCII, sensitive=(isData && isSamp &&(sTypeFlag eq 0 || sTypeFlag eq 2))
   widget_control, (*sPtr).wDAVEDM, sensitive=(isData && isSamp &&(sTypeFlag eq 0 || sTypeFlag eq 2))
   widget_control, (*sPtr).wPAN, sensitive=(isData && isSamp &&(sTypeFlag eq 0 || sTypeFlag eq 2))
   Widget_control, (*sPtr).wCOMBDAVEDM, sensitive=(isData && isSamp &&(sTypeFlag eq 0) && (count gt 1))
   Widget_control, (*sPtr).wCOMBDAVE, sensitive=(isData && isSamp &&(sTypeFlag eq 0) && (count gt 1))
   Widget_control, (*sPtr).wCOMBPAN, sensitive=(isData && isSamp &&(sTypeFlag eq 0) && (count gt 1))
;   widget_control, (*sPtr).wMSLICE, sensitive=(isData && (isSamp || isBkgd) && (sTypeFlag eq 1))
;   widget_control, (*sPtr).wDAVE, sensitive=(isData && (isSamp || isBkgd) &&(sTypeFlag eq 0 || sTypeFlag eq 2))
;   widget_control, (*sPtr).wASCII, sensitive=(isData && (isSamp || isBkgd) &&(sTypeFlag eq 0 || sTypeFlag eq 2))
;   widget_control, (*sPtr).wDAVEDM, sensitive=(isData && (isSamp || isBkgd) &&(sTypeFlag eq 0 || sTypeFlag eq 2))
;   widget_control, (*sPtr).wPAN, sensitive=(isData && (isSamp || isBkgd) &&(sTypeFlag eq 0 || sTypeFlag eq 2))
;
;   widget_control, (*sPtr).wFFast, sensitive=(isData && isFast)
   if (isSamp) then widget_control, (*sPtr).wDelSamp, set_value='Delete All Sample Data'
   if (isBkgd) then widget_control, (*sPtr).wDelSamp, set_value='Delete All Background Data'
   if (isFast) then widget_control, (*sPtr).wDelSamp, set_value='Delete All Fast Bkgd Data'

   return, 0
endif

switch strupcase(uname) of
    "VARIABLE": begin
        ;; Lets import a variable or two
        iStatus = IDLitwdCommandLineImport(oUI,group_leader=event.top)
        break
    end
    
    "FILE": begin
        ;; Lets import a file...
        iStatus = IDLitwdFileImport(oUI,group_leader=event.top)
        break
    end
    
    "BKGDDATAMANAGER":
    "FASTDATAMANAGER":
    "SAMPDATAMANAGER": begin
        ;; Context Menu?
        if(tag_names(/structure, event) eq 'WIDGET_CONTEXT')then begin
            WIDGET_DISPLAYCONTEXTMENU, event.id, $
              event.x, event.y, (*sPtr).wContext 
        endif else begin
            id= (*event.value)[0]
            oItem = oTool->GetByIdentifier(id)
            isData = obj_isa(oItem, "IDLitParameterSet") ;obj_isa(oItem, "IDLitData") && ~obj_isa(oItem,'IDL_CONTAINER')
            count = n_elements(*event.value)
            sTypeFlag = -1
            if (obj_isa(oItem, "IDLitParameterSet")) then $
               status = oItem->GetMetaData('sampTypeFlag',sTypeFlag)
            widget_control, (*sPtr).wView, sensitive=isData
            widget_control, (*sPtr).wDel, sensitive=isData
            widget_control, (*sPtr).wDelSamp, sensitive=isData
            widget_control, (*sPtr).wDelAll, sensitive=isData
;            widget_control, (*sPtr).wHist, sensitive=isData
            widget_control, (*sPtr).wReload, sensitive=isData
            dataType = ''
            if (isData) then oItem->GetProperty, type=dataType
            isSamp = strcmp(dataType,'samp',/fold_case)
            isBkgd = strcmp(dataType,'bkgd',/fold_case)
            isFast = strcmp(dataType,'fast',/fold_case)
            widget_control, (*sPtr).wMSLICE, sensitive=(isData && isSamp && (sTypeFlag eq 1))
            widget_control, (*sPtr).wDAVE, sensitive=(isData && isSamp && (sTypeFlag eq 0 || sTypeFlag eq 2))
            widget_control, (*sPtr).wASCII, sensitive=(isData && isSamp && (sTypeFlag eq 0 || sTypeFlag eq 2))
            widget_control, (*sPtr).wDAVEDM, sensitive=(isData && isSamp && (sTypeFlag eq 0 || sTypeFlag eq 2))
            widget_control, (*sPtr).wPAN, sensitive=(isData && isSamp && (sTypeFlag eq 0 || sTypeFlag eq 2))
            Widget_control, (*sPtr).wcombdavedm, sensitive=(isData && isSamp &&(sTypeFlag eq 0) && (count gt 1))
            Widget_control, (*sPtr).wcombdave, sensitive=(isData && isSamp &&(sTypeFlag eq 0) && (count gt 1))
            Widget_control, (*sPtr).wcombpan, sensitive=(isData && isSamp &&(sTypeFlag eq 0) && (count gt 1))
;            widget_control, (*sPtr).wMSLICE, sensitive=(isData && (isSamp || isBkgd) && (sTypeFlag eq 1))
;            widget_control, (*sPtr).wDAVE, sensitive=(isData && (isSamp || isBkgd) && (sTypeFlag eq 0 || sTypeFlag eq 2))
;            widget_control, (*sPtr).wASCII, sensitive=(isData && (isSamp || isBkgd) && (sTypeFlag eq 0 || sTypeFlag eq 2))
;            widget_control, (*sPtr).wDAVEDM, sensitive=(isData && (isSamp || isBkgd) && (sTypeFlag eq 0 || sTypeFlag eq 2))
;            widget_control, (*sPtr).wPAN, sensitive=(isData && (isSamp || isBkgd) && (sTypeFlag eq 0 || sTypeFlag eq 2))
;
;            widget_control, (*sPtr).wFFast, sensitive=(isData && isFast)
            if (isSamp) then widget_control, (*sPtr).wDelSamp, set_value='Delete All Sample Data';, set_uvalue='samp'
            if (isBkgd) then widget_control, (*sPtr).wDelSamp, set_value='Delete All Background Data';, set_uvalue='bkgd'
            if (isFast) then widget_control, (*sPtr).wDelSamp, set_value='Delete All Fast Bkgd Data';, set_uvalue='fast'

            (*sPtr).identifier = id
            (*(*sPtr).pDataIDs) = (*event.value)
            heap_free, event

            oTool->UpdatePropertysheet ; some propertysheet elements depend on the number of selected datasets
            
            ;;
            ; (*(*sPtr).pDataIDs) contains a list of the selected ids in the data manager.
            ; However, this includes ids for child nodes not just the parents. Parent nodes
            ; represent datasets and we need to identify these and then send for plotting.
            ; Parent nodes are ParameterSet objects and have datatypes that are either 'samp' or 'bkgd'.
            oTool->DisableUpdates

            iArr = (*(*sPtr).pDataIDs)
            count = n_elements(iArr)
            overplot = 0  ; first plot should not be an overplot
            (*sPtr).oUI->GetProperty, group_leader=wHandler ; the handler widget id for main widget
            iValid = []
            for i=0,count-1 do begin
               oItem = oTool->GetByIdentifier(iArr[i])
               if (~obj_isa(oItem, "IDLitParameterSet")) then continue
               oItem->GetProperty, type = dataType
               isSamp = strcmp(dataType,'samp',/fold_case)
               isBkgd = strcmp(dataType,'bkgd',/fold_case)
               if (isSamp || isBkgd) then iValid = [iValid,i]
;               if (isSamp || isBkgd) then begin
;                  plotEvent = {cw_bt7psdDataManager, $
;                               ID:event.id, $
;                               TOP:event.top, HANDLER:wHandler, $
;                               clicks:event.clicks, $
;                               overplot:overplot, $
;                               lastplot:(i eq count-1), $
;                               identifier:iArr[i], $
;                               position:i}
;                  wd_BT7PSDTool_event, plotEvent
;                  overplot = 1  ; subsequent plots should be overplotted!
;               endif
            endfor
            ; extract plot command (wd_BT7PSDTool_event, plotEvent) from loop and instead
            ; issue a accumulate all data to be plotted and issue a single plot command
            if (n_elements(iValid) gt 0) then begin
              plotEvent = {Cw_bt7psddatamanager, $
                id:Event.id, $
                top:Event.top, handler:wHandler, $
                clicks:Event.clicks, $
                overplot:0, $
                lastplot:0, $
                identifier:ptr_new(iArr[iValid]), $
                position:0}
              Wd_bt7psdtool_event, plotEvent
            endif
            
            oTool->EnableUpdates          

        endelse
        break
    end

    'DELETE': begin
      cw_bt7psdDataManager_DeleteData, sPtr, event.top
      break
    end

    'DELETESAMP': begin
      cw_bt7psdDataManager_DeleteData, sPtr, event.top, /deltree
      break
    end

    'DELETEALL': begin
      cw_bt7psdDataManager_DeleteData, sPtr, event.top, /delall
      break
    end

    'RELOAD': begin
      cw_bt7psdDataManager_ReloadData, sPtr, event.handler
      break
    end

    'TOMSLICE': begin
      cw_bt7psdDataManager_ExportToMSLICE, sPtr, event.handler
      break
    end

    'TODAVE': begin
      cw_bt7psdDataManager_saveToDisk, sPtr, event.handler, format='dave'
      break
    end

    'TOASCII': begin
      cw_bt7psdDataManager_saveToDisk, sPtr, event.handler, format='ascii'
      break
    end

    'TODAVEDM': begin
      cw_bt7psdDataManager_ExportToDAVEDM, sPtr, event.handler
      break
    end

    'COMBTODAVE': begin
      cw_bt7psdDataManager_saveToDisk, sPtr, event.handler, format='comb'
      break
    end

    'COMBTODAVEDM': begin
      Cw_bt7psddatamanager_COMBTODAVEDM, sPtr, Event.handler
      break
    end

    'COMBTOPAN': begin
      Cw_bt7psddatamanager_COMBTOPAN, sPtr, Event.handler
      break
    end

    'TOPAN': begin
      cw_bt7psdDataManager_ExportToPAN, sPtr, event.handler
      break
    end

    'FITFAST': begin
      if (~oTool->EditUserDefProperty(oTool, 'FASTPARAMS')) then break
      
      oData = oTool->GetByIdentifier((*sPtr).identifier)  ; get the last selected dataset
      if (~obj_valid(oData)) then break
      oData->GetProperty, type=dataType
      if (strcmp(dataType,'fast',/fold_case)) then $
        wd_BT7PSDtool_sendPlotEvent, oUI, oData
      break
    end

    'VIEWHIST': begin
      cw_bt7psdDataManager_viewHist, sPtr, event.handler
      break
    end

    'VIEWDETAILS': begin
      cw_bt7psdDataManager_viewDetail, sPtr, event.handler
      break
    end

    else:
endswitch

return, 0

end


;===============================================================================
; cw_bt7psdDataManager_Resize
;+ 
; PURPOSE:
;   Resize the specified widget to the specified dimension.
;
; PARAMETERS
;   widID (in)  - The id of the widget to be resized
;
;   newX (in)   - The new x size
;
;   newY (in)   - the new y size
;
; KEYWORDS:
;-
pro cw_bt7psdDataManager_Resize, widID, newX, newY

compile_opt idl2

WIDGET_CONTROL, widID, get_uvalue=sPtr


;geomCW = Widget_Info(widID, /geometry)

;; Change height of dm
;widget_control, (*sPtr).wDM, SCR_XSIZE=newX-12, $ ;; why -12? It just works
;                SCR_YSIZE= newY - geomButs.scr_ysize-16
;widget_control, (*sPtr).wDM, scr_xsize=newX - 12 $ ;; why -12? It just works
;                ,scr_ysize= newY - 12

; there are three DM tabs so deal with each separately
wDM = (*sPtr).wDM
oldGeom = widget_info((*sPtr).wDM,/geometry)

;widget_control, (*sPtr).wDM, scr_xsize = oldGeom.scr_xsize + newX $
;                           , scr_ysize = oldGeom.scr_ysize + newY
;
;widget_control, (*sPtr).wDM2, scr_xsize = oldGeom.scr_xsize + newX $
;                           , scr_ysize = oldGeom.scr_ysize + newY
;
widget_control, (*sPtr).wDM, scr_xsize = newX, scr_ysize = newY

widget_control, (*sPtr).wDM2, scr_xsize = newX, scr_ysize = newY

end


;===============================================================================
; cw_bt7psdDataManager
;+ 
; PURPOSE:
;   Construct a compound widget representing the data manager browser section of the
;   BT7 PSD Reduction user interface.
;
; PARAMETERS
;   wParent (in) - id of the parent widget for the widget to be
;                  built. The parent widget embeds the new one.
;
;   oUI (in)     - object reference of the tool UI
;
; KEYWORDS:
;   uName (in)   - the uname to be assigned to the new widget
;
;   xSize (in)   - the width of the new widget
;
;   ySize (in)   - the height of the new widget
;
; RETURN VALUE:
;   the id of the base widget of created widget hierarchy.
;-
function cw_bt7psdDataManager, wParent,oUI,uname=uName,xsize=xSize,ysize=ySize,_extra=_extra

compile_opt idl2

;; Check arguments.
if (N_PARAMS() lt 2) then $
  MESSAGE, 'Incorrect number of arguments.'

if (not WIDGET_INFO(wParent, /VALID)) then $
  MESSAGE, 'Invalid widget identifier.'

if (not keyword_set(ySize)) then ySize=300
if (not keyword_set(xSize)) then xSize=300

myname = 'cw_bt7psdDataManager'
wTabBase = widget_tab(wParent, tab_mode=0 $   ; mode eq 0 for Xplatform compt. since it is always 0 for motif
                       ,EVENT_FUNC= myname+'_event' $
                       ,PRO_SET_VALUE= myname+'_setvalue' $
                       ,FUNC_GET_VALUE= myname+'_getvalue' $
;                       ,uname=uName $
                       ,NOTIFY_REALIZE=myname+"_realize" $
                       ,KILL_NOTIFY=myname+"_cleanup" $
                      )
;w = widget_base(wParent, EVENT_FUNC= myname+'_event' $
;                       ,PRO_SET_VALUE= myname+'_setvalue' $
;                       ,FUNC_GET_VALUE= myname+'_getvalue' $
;                       ,uname=uName $
;                       ,NOTIFY_REALIZE=myname+"_realize" $
;                       ,KILL_NOTIFY=myname+"_cleanup" $
;                       ,/base_align_center, /row)

wSampTab = widget_base(wTabBase,/row,title='Sample',uname='SAMPTAB',tab_mode=2)
wBkgdTab = widget_base(wTabBase,/row,title='Background',uname='BKGDTAB',tab_mode=2)

;; Now create the tree widgets
wDM1 = cw_ittreeview(wSampTab, oUI $
                    ,IDENTIFIER="Sample Data Manager" $
                    ,uname="SAMPDATAMANAGER" $
                    ,xsize = xsize, ysize=ysize, /multiple);,/multiple)
wDM2 = cw_ittreeview(wBkgdTab, oUI $
                    ,IDENTIFIER="Bkgd Data Manager" $
                    ,uname="BKGDDATAMANAGER" $
                    ,xsize = xsize, ysize=ysize, /multiple);,/multiple)

;; Build context menu to be applied to the tree widget
widget_control, wDM1, /context_events
widget_control, wDM2, /context_events

;wContext = WIDGET_BASE(wTabBase,/CONTEXT_MENU,event_func=myname+'_event')  
wContext = WIDGET_BASE(wDM1,/CONTEXT_MENU,event_func=myname+'_event')  

tt1 = "Delete dataset"
tt2 = "Reload this dataset from disk"
tt3 = "View full contents of selected item"
tt4 = "View history of reduction steps applied"
tt5 = "Save reduced data in DAVE format file"
tt6 = "Save reduced data in ASCII format file"
tt7 = "Export selected data to DAVE Data Manager"
tt8 = "Fit fast background data"
tt9 = "Fit selected data in PAN"
tt10 = "Export to Mslice"
tt11 = "Combine Selected data and export to DAVE Data Manager"
tt12 = "Combine selected data and fit in PAN"
wDel = widget_button(wContext, value="Delete", uname="DELETE",tooltip=tt1,sensitive=0)
wReload = widget_button(wContext, value="Reload", uname="RELOAD",tooltip=tt2,sensitive=0)
wDelSamp = widget_button(wContext, value="Delete All Samp Data", uname="DELETESAMP",tooltip=tt1,sensitive=0)
wDelAll = widget_button(wContext, value="Delete All Data", uname="DELETEALL",tooltip=tt1,sensitive=0)
wView = widget_button(wContext, value="View Raw Contents", uname="VIEWDETAILS",tooltip=tt3,/separator,sensitive=0)
;wHist = widget_button(wContext, value="View Reduction History", uname="VIEWHIST",tooltip=tt4,sensitive=0)
;wFFast = widget_button(wContext, value="Fit Fast Bkgd", uname="FITFAST",tooltip=tt8,/separator,sensitive=0)
wDAVE = widget_button(wContext, value="Save as DAVE file", uname="TODAVE",tooltip=tt5,/separator,sensitive=0)
wCOMBDAVE = Widget_button(wContext, value="Combine as DAVE file", uname="COMBTODAVE",tooltip=tt13,sensitive=0)
wASCII = widget_button(wContext, value="Save as ASCII file", uname="TOASCII",tooltip=tt6,sensitive=0)
wDAVEDM = widget_button(wContext, value="Export to DAVE Data Manager", uname="TODAVEDM",tooltip=tt7,sensitive=0,/separator)
wCOMBDAVEDM = Widget_button(wContext, value="Combine to DAVE Data Manager", uname="COMBTODAVEDM",tooltip=tt11,sensitive=0)
wPAN = widget_button(wContext, value="Fit in PAN", uname="TOPAN",tooltip=tt9,sensitive=0)
wCOMBPAN = Widget_button(wContext, value="Combine to PAN", uname="COMBTOPAN",tooltip=tt12,sensitive=0)
wMSLICE = widget_button(wContext, value="Export to Mslice", uname="TOMSLICE",tooltip=tt10,sensitive=0)

state = { oUI:oUI $
          ,wDM:wDM1 $
          ,wDM2:wDM2 $
          ,ysize0:ysize/3 $       ;initial Y size used in realize notify
          ,wContext: wContext $
          ,wDel:wDel $
          ,wDelSamp:wDelSamp $
          ,wDelAll:wDelAll $
          ,wReload:wReload $
          ,wMSLICE:wMSLICE $
;          ,wFFast:wFFast $
          ,wDAVE:wDAVE $
          ,wASCII:wASCII $
          ,wDAVEDM:wDAVEDM $
          ,wCOMBDAVE:wCOMBDAVE $
          ,wCOMBDAVEDM:wCOMBDAVEDM $
          ,wCOMBPAN:wCOMBPAN $
          ,wPAN:wPAN $
          ,wView:wView $
;          ,wHist:wHist $
          ,identifier:'' $      ; used to store identifier of selected item
          ,pDataIDs:ptr_new('') $ ; store identifiers of multiple seleted items
        }

;; Normally, this would be the first child on the widget, but the
;; tree view widget is just a tree, and not wrapped and changes
;; could cause issues.
sPtr = ptr_new(State,/no_copy)
widget_control, wTabBase, set_uvalue=sPtr

;; Due to how the context menu is created, we place the state in the
;; context menu also
widget_control, wContext, set_uvalue=sPtr

return, wTabBase

end
