; $Id$
;###############################################################################
;+
; NAME:
;   CW_DAVEDataManager
;
; PURPOSE:
;   Construct the widgets representing the data manager browser section of the
;   main DAVE user interface.
;
; CATEGORY:
;   DAVE Main Tool
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; May 2004
;-
;###############################################################################


;===============================================================================
; cw_dataDataManager_NewVis 
;
; Purpose:
;   Create a visualization from the selected data container
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    event - The widget event.
;
pro cw_daveDataManager_NewVis, sPtr, event
compile_opt idl2

oTool = (*sPtr).oUI->Gettool()

;; Get the item
selectedItem = oTool->GetSelectedData(count=cnt)

if (cnt le 0) then begin
   oTool->StatusMessage, 'No data item selected - new visualization could not be created'
   return
endif

if (cnt gt 1) then selectedItem = selectedItem[0]

if (~obj_valid(selectedItem)) then begin
   oTool->StatusMessage, 'Invalid Data Selection - new visualization could not be created'
   return
endif

widget_control, event.id, get_value=visType   ; visType=widget_info(event.id,/uname)
if (strcmp(visType,'2d plot',/fold_case)) then visType = 'Plot3D'
;if (strcmp(visType,'Multi Plot',/fold_case)) then visType = 'Plot'
;if (strcmp(visType,'Multi Plot2D',/fold_case)) then visType = 'Plot3D'


;wd_DAVETool_autoCreateVis, oTool, visType
wd_DAVETool_autoVis, oTool, visType, 'New'

end


;===============================================================================
; cw_dataDataManager_AddVis 
;
; Purpose:
;   Visualize the selected dataset using an existing visualization window
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    event - The widget event.
;
pro cw_daveDataManager_AddVis, sPtr, event
compile_opt idl2

oTool = (*sPtr).oUI->Gettool()

;; Get the item
selectedItem = oTool->GetSelectedData(count=cnt)

if (cnt le 0) then begin
   oTool->StatusMessage, 'No data item selected - visualization could not be added'
   return
endif

if (cnt gt 1) then selectedItem = selectedItem[0]

if (~obj_valid(selectedItem)) then begin
   oTool->StatusMessage, 'Invalid Data Selection - visualization could not be added'
   return
endif

widget_control, event.id, get_value=visType   ; visType=widget_info(event.id,/uname)
if (strcmp(visType,'2d plot',/fold_case)) then visType = 'Plot3D'

;wd_DAVETool_autoAddVis, oTool, visType
wd_DAVETool_autoVis, oTool, visType, 'Add'

end


;===============================================================================
; cw_dataDataManager_ViewDetail 
;
; Purpose:
;   View the contents of a data element in the DAVE Data Manager
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;
pro cw_daveDataManager_ViewDetail, sPtr, wTLB
compile_opt idl2

;; Get the item
id = cw_itTreeView_getSelect((*sPtr).wDM, count=count)
if(count gt 0)then begin
    oTool = (*sPtr).oUI->Gettool()
    oItem = oTool->GetByIdentifier(id[0])              
    if(obj_valid(oItem))then begin
        idParent = cw_itTreeView_getParent((*sPtr).wDM, id[0])
;        oItem->GetProperty, name=name
        wd_viewDetail, oItem, group_leader=wTLB
    endif else $
      void = dialog_message("Invalid Data Item.",/warning, $
                            dialog_parent=wTLB, $
                            title="Invalid Data")
endif else $
  void = dialog_message("No data item selected",/warning, $
                        dialog_parent=wTLB, $
                        title="No Data")
end


;===============================================================================
; cw_dataDataManager_DeleteData 
;
; Purpose:
;   Encapsulates the functionality used to delete data. Based on
;   _cw_itDataManager_DeleteData.  The basic difference results fron
;   the fact that the DAVEDATA_MANAGER service is used instead of
;   DATA_MANAGER.
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;
pro cw_daveDataManager_DeleteData, sPtr, wTLB
compile_opt idl2

;; Get the item
id = cw_itTreeView_getSelect((*sPtr).wDM, count=count)
if(count lt 1)then begin
  void = dialog_message("No data item selected",/warning, $
                        dialog_parent=wTLB, $
                        title="No Data")
endif
oTool = (*sPtr).oUI->Gettool()

; Retrieve top-level detasets to be deleted
msg = "Delete Data item(s):"
for i=0,count-1 do begin
   idParent = file_basename(cw_itTreeView_getParent((*sPtr).wDM, id[i]))
   if (idParent ne 'DAVE DATA MANAGER') then continue
   oItem = oTool->GetByIdentifier(id[i])
   oItem->GetProperty, name=name
   oArr = (n_elements(oArr) eq 0)? oItem : [oArr,oItem]
   iArr = (n_elements(iArr) eq 0)? id[i] : [iArr,id[i]]
   msg = [msg,name]
endfor

count = n_elements(iArr)
if (count lt 1) then return

if (count gt 1) then begin
   ; Get final confirmation from user
   title = "Delete Data"
   status = dialog_message(title="Delete Data", msg, $
                           /question, /default_no, dialog_parent=wTLB)
endif else begin
   ; Don't ask for confrmation for a single dataset
   status = 'Yes'
endelse

; Delete the data
if(status eq 'Yes')then begin
   ;oDM = oTool->GetService("DAVEDATA_MANAGER")
   
   for i=0,count-1 do begin
      status = 'Yes'
      if (obj_isa(oArr[i],'GENERICDATASET')) then begin
         ; Ask for confirmation, if the dataset is being used.
         oArr[i]->GetProperty, usedCount=usedCnt, name=name
         if (usedCnt gt 0) then begin
            msg = [name,'is being used!','Do you really want to delete?']
            status = dialog_message(title=title, msg, $
                        /question, /default_no, dialog_parent=wTLB)
         endif
      endif
      ;if (status eq 'Yes') then void = oDM->DeleteData(iArr[i])
      ;; rather than calling the data manager's deleteData() method,
      ;; do the following instead so we can use the no_notify keyword
      ;; to prevent the data manager tree from being refreshed
      ;; repeatedly on UNIX when multiple datasets are being deleted.
      if (strcmp(status,'Yes',/fold)) then begin
         oData = oTool->GetByIdentifier(iArr[i])
         ;; Get the parent of this data item:
         oData->GetProperty, _PARENT=oparent
         if(obj_valid(oparent))then $
             oParent->Remove, oData, /no_notify
      
         oData->RemoveDataObserver, oParent  ; necessary otherwise the next line will trigger a refresh
         obj_destroy, oData      
      endif

   endfor
   
   ;; Fake a callback to rebuild the tree.
   idParent = cw_itTreeView_getParent((*sPtr).wDM, iArr[0])
   cw_itTreeView_Callback, (*sPtr).wDM, idParent,"UPDATEITEM", idParent
endif

end

;===============================================================================
; cw_dataDataManager_DuplicateData 
;
; Purpose:
;   Encapsulates the functionality used to duplicate data. Based on
;   _cw_itDataManager_DuplicateData.  The basic difference results fron
;   the fact that the DAVEDATA_MANAGER service is used instead of
;   DATA_MANAGER.
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;
pro cw_daveDataManager_DuplicateData, sPtr, wTLB
compile_opt idl2

;; Get the item
id = cw_itTreeView_getSelect((*sPtr).wDM, count=count)
if(count gt 0)then begin
    oTool = (*sPtr).oUI->Gettool()
    oItem = oTool->GetByIdentifier(id[0])              
    if(obj_valid(oItem))then begin
        idParent = cw_ittreeView_GetParent((*sPtr).wDM, id[0])

        oDM = oTool->GetService("DAVEDATA_MANAGER")	;; get DM service
        status = oDM->CopyData(id, parent=idparent)
        if(status eq 1)then begin
            ;; Success
            ;; Fake a callback to rebuild the tree.
            cw_itTreeView_Callback, (*sPtr).wDM, id[0], "UPDATEITEM",id[0]

        endif else $ ;; error
          void = dialog_message("Unknow error during data duplication.",/error, $
                                dialog_parent=wTLB, $
                                title="Error")                   
    endif else $
      void = dialog_message("Invalid Data Item.",/warning, $
                            dialog_parent=wTLB, $
                            title="Invalid Data")
endif else $ ;; nothing selected
  void = dialog_message("No data item selected",/warning, $
                        dialog_parent=wTLB, $
                        title="No Data")

end


;===============================================================================
; cw_dataDataManager_RenameData 
;
; Purpose:
;   Encapsulates the functionality used to rename data. Based on
;   _cw_itDataManager_RenameData.  The basic difference results fron
;   the fact that the DAVEDATA_MANAGER service is used instead of
;   DATA_MANAGER.
;
; Parameters
;    sPtr - The state pointer for this widget
;
;    wTLB - The top of this widget tree.
;
pro cw_daveDataManager_RenameData, sPtr, wTLB
compile_opt idl2

;; Get the item
id = cw_itTreeView_getSelect((*sPtr).wDM, count=count)
if(count gt 0)then begin
    oTool = (*sPtr).oUI->Gettool()
    oItem = oTool->GetByIdentifier(id[0])              
    if(obj_valid(oItem))then begin
        oItem->GetProperty, name=name
        status = idlitwdprompttext(wTLB, $
                                   "Enter New Data Name:", $
                                   newName, $
                                   value=name, $
                                   title="Rename Data")
        if(status eq 1 and keyword_set(newName))then begin
            oItem->SetProperty, name=newName
            ;; Tell the tree to rebild
            cw_itTreeView_Callback, (*sPtr).wDM, id[0], "UPDATEITEM",id[0]

        endif
    endif else $
      void = dialog_message("Invalid Data Item.",/warning, $
                            dialog_parent=wTLB, $
                            title="Invalid Data")
endif else $
  void = dialog_message("No data item selected",/warning, $
                        dialog_parent=wTLB, $
                        title="No Data")

end


;===============================================================================
; cw_daveDataManager_Cleanup
; 
; PURPOSE:
;   Widget cleanup procedure.
;
; PARAMETERS
;   widID (in)  - The id of the base widget to be destroyed
;
; KEYWORDS:
;
pro cw_daveDataManager_Cleanup, widID
compile_opt idl2

widget_control, widID, get_uvalue=sPtr
ptr_free,(*sPtr).pDataIDs,sPtr

end


;===============================================================================
; cw_daveDataManager_Realize
; 
; PURPOSE:
;   Called when the widget is realized - checks widget dimensions.
;
; PARAMETERS
;   widID (in)  - The id of the base widget to be realized
;
; KEYWORDS:
;
pro cw_daveDataManager_Realize, widID
compile_opt idl2

widget_control, widID, get_uvalue=sPtr

;; Make sure the geometry is all right.
sGeom = widget_info(widID, /geometry)
cw_daveDataManager_Resize, widID, sGeom.xsize, (*sPtr).ySize0

end


;===============================================================================
; cw_daveDataManager_Setvalue
; 
; PURPOSE:
;   Called by widget_control when the value of a widget is to be set.
;
; PARAMETERS
;   widID (in)  - The id of the widget to be altered
;
; KEYWORDS:
;
pro cw_daveDataManager_Setvalue, widID, value
compile_opt idl2

WIDGET_CONTROL, widID, GET_UVALUE=sPtr
cw_itTreeView_SetSelect,  (*sPtr).wDM, value

end


;===============================================================================
; cw_daveDataManager_Getvalue
; 
; PURPOSE:
;   Called by widget_control when the value of a widget is required.
;
; PARAMETERS
;   widID (in)  - The id of the widget whose value is required
;
; KEYWORDS:
;
; RETURN VALUE:
;   The required value of the widget
;
function cw_daveDataManager_Getvalue, widID
compile_opt idl2

WIDGET_CONTROL, widID, GET_UVALUE=sPtr

return, cw_itTreeView_GetSelect((*sPtr).wDM)

end


;===============================================================================
; cw_daveDataManager_Event
; 
; PURPOSE:
;   Main event handler for widget created by cw_daveDataManager.
;
; PARAMETERS
;   event (in)  - The event to be handled
;
; KEYWORDS:
;
; RETURN VALUE:
;   0, if the event was completely handled here
;   Event Structure, if further processing is required.
;
function cw_daveDataManager_Event, event
compile_opt idl2


; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'CW_DAVEDATAMANAGER: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return, 0
    endif
endif

;; Get our state
widget_control, event.handler, get_uvalue=sPtr

uname = widget_info(event.id, /uname)
case strupcase(uname) of
    "VARIABLE": begin
        ;; Lets import a variable or two
        iStatus = IDLitwdCommandLineImport((*sPtr).oUI,group_leader=event.top)
    end
    "FILE": begin
        ;; Lets import a file...
        iStatus = IDLitwdFileImport((*sPtr).oUI,group_leader=event.top)
    end
    "DAVEDATAMANAGER": begin
        ;; Context Menu?
        if(tag_names(/structure, event) eq 'WIDGET_CONTEXT')then begin
            WIDGET_DISPLAYCONTEXTMENU, event.id, $
              event.x, event.y, (*sPtr).wContext 
        endif else begin
            id= (*event.value)[0]
            oTool = (*sPtr).oUI->Gettool()
            oItem = oTool->GetByIdentifier(id)
            isData = obj_isa(oItem, "IDLitData")
            widget_control, (*sPtr).wView, sensitive=isData
            widget_control, (*sPtr).wDel, sensitive=isData
            widget_control, (*sPtr).wDup, sensitive=isData
            ;widget_control, (*sPtr).wRename, sensitive=isData

            if (isData) then begin
                ;; don't allow view for containers
                if (obj_isa(oItem,'IDL_CONTAINER')) then begin
                    widget_control, (*sPtr).wView, sensitive=0
                endif

                ;; allow del,dup for top-level entries only
                oItem->GetProperty, _parent=oParent ; NB: using private keyword!!
                if (obj_valid(oParent)) then begin
                    oParent->GetProperty, identifier=parentID
                    sensitive = (strcmp(parentID,'DAVE Data Manager',/fold_case))[0]
                    widget_control, (*sPtr).wDel, sensitive=sensitive
                    widget_control, (*sPtr).wDup, sensitive=sensitive
                endif
                
                ;; Only activate new vis for valid data containers
                widget_control, (*sPTr).wNewVis, sensitive=0 ; default setting
                validDataTypes = (*sPtr).newVisValidDataTypes
                oItem->GetProperty, type=dataType
                void = where(dataType eq validDataTypes, validType)
                if (validType) then begin
                  widget_control, (*sPtr).wNewVis, sensitive=1 ; activate the full new vis menu

                  ;; Deal with sub-entries in context menu
                  oTop = GetMainDataObjectParent(oItem[0])
                  oTop->GetProperty, nDimensions=ndim
                  unames2D = ['newPlot2D','newContour','newImage','newSurface']
                  n2D = n_elements(unames2D)
                  case ndim of
                     1: begin
                        wID = widget_info((*sPtr).wNewVis,find_by_uname='newPlot')
                        if (widget_info(wID,/valid)) then widget_control, wID, sensitive=1
                        for i=0,n2D-1 do begin
                           wID = widget_info((*sPtr).wNewVis,find_by_uname=unames2D[i])
                           if (widget_info(wID,/valid)) then widget_control, wID, sensitive=0                       
                        endfor
                     end
                     
                     2: begin
                        wID = widget_info((*sPtr).wNewVis,find_by_uname='newPlot')
                        if (widget_info(wID,/valid)) then widget_control, wID, sensitive=0
                        for i=0,n2D-1 do begin
                           wID = widget_info((*sPtr).wNewVis,find_by_uname=unames2D[i])
                           if (widget_info(wID,/valid)) then widget_control, wID, sensitive=1                       
                        endfor                     
                     end
                     
                     else:
                  endcase
                endif
                
                ;; Only activate add vis for valid data containers
                widget_control, (*sPTr).wAddVis, sensitive=0 ; default setting
                validDataTypes = (*sPtr).addVisValidDataTypes
                oItem->GetProperty, type=dataType
                void = where(dataType eq validDataTypes, validType)
                if (validType) then begin
                  widget_control, (*sPtr).wAddVis, sensitive=1
                  
                  ;; Deal with sub-entries in context menu
                  oTop = GetMainDataObjectParent(oItem[0])
                  oTop->GetProperty, nDimensions=ndim
                  case ndim of
                     1: begin
                        wID = widget_info((*sPtr).wAddVis,find_by_uname='addPlot')
                        if (widget_info(wID,/valid)) then widget_control, wID, sensitive=1
                        wID = widget_info((*sPtr).wAddVis,find_by_uname='addPlot2D')
                        if (widget_info(wID,/valid)) then widget_control, wID, sensitive=0
                     end
                     
                     2: begin
                        wID = widget_info((*sPtr).wAddVis,find_by_uname='addPlot')
                        if (widget_info(wID,/valid)) then widget_control, wID, sensitive=0
                        wID = widget_info((*sPtr).wAddVis,find_by_uname='addPlot2D')
                        if (widget_info(wID,/valid)) then widget_control, wID, sensitive=1
                     end
                     
                     else:
                  endcase
                endif
            endif

            (*sPtr).identifier = id
            (*(*sPtr).pDataIDs) = (*event.value)
            heap_free, event

            return, {CW_DAVEDATAMANAGER, $
                     ID:event.handler, $
                     TOP:event.top, HANDLER:0, $
                     clicks:event.clicks, $
                     identifier:id}
        endelse
    end
    'DELETE': cw_daveDataManager_DeleteData, sPtr, event.top
    'DUPLICATE': cw_daveDataManager_DuplicateData, sPtr, event.top
    'VIEWDETAILS': cw_daveDataManager_viewDetail, sPtr, event.top
    'NEWPLOT': cw_daveDataManager_newVis, sPtr, event
    'NEWPLOT2D': cw_daveDataManager_newVis, sPtr, event
    'NEWCONTOUR': cw_daveDataManager_newVis, sPtr, event
    'NEWIMAGE': cw_daveDataManager_newVis, sPtr, event
    'NEWSURFACE': cw_daveDataManager_newVis, sPtr, event
    'ADDPLOT': cw_daveDataManager_addVis, sPtr, event
    'ADDPLOT2D': cw_daveDataManager_addVis, sPtr, event
    ;'RENAME': cw_daveDataManager_RenameData, sPtr, event.top

    else:
endcase

return, 0

end


;===============================================================================
; cw_daveDataManager_Resize
; 
; PURPOSE:
;   Resize the specified widget to the specified dimension.
;
; PARAMETERS
;   widID (in)  - The id of the widget to be resized
;
;   newX (in)   - The new x size
;
;   newY (in)   - the new y size
;
; KEYWORDS:
;
pro cw_daveDataManager_Resize, widID, newX, newY

compile_opt idl2

WIDGET_CONTROL, widID, get_uvalue=sPtr

;; Retrieve the button size
;geomButs = Widget_Info((*sPtr).wButtons, /geometry)

geomCW = Widget_Info(widID, /geometry)

;; Change height of dm
;widget_control, (*sPtr).wDM, SCR_XSIZE=newX-12, $ ;; why -12? It just works
;                SCR_YSIZE= newY - geomButs.scr_ysize-16
widget_control, (*sPtr).wDM, scr_xsize=newX - 12 $ ;; why -12? It just works
                ,scr_ysize= newY - 12

end


;===============================================================================
; cw_daveDataManager
; 
; PURPOSE:
;   Construct a compound widget representing the data manager browser section of the
;   main DAVE user interface.
;
; PARAMETERS
;   wParent (in) - id of the parent widget for the widget to be
;                  built. The parent widget embeds the new one.
;
;   oUI (in)     - object reference of the tool UI
;
; KEYWORDS:
;   uName (in)   - the uname to be assigned to the new widget
;
;   xSize (in)   - the width of the new widget
;
;   ySize (in)   - the height of the new widget
;
; RETURN VALUE:
;   the id of the base widget of created widget hierarchy.
;
function cw_daveDataManager, wParent,oUI,uname=uName,xsize=xSize,ysize=ySize,_extra=_extra

compile_opt idl2

;; Check arguments.
if (N_PARAMS() lt 2) then $
  MESSAGE, 'Incorrect number of arguments.'

if (not WIDGET_INFO(wParent, /VALID)) then $
  MESSAGE, 'Invalid widget identifier.'

if (not keyword_set(ySize)) then ySize=300
if (not keyword_set(xSize)) then xSize=300

myname = 'cw_daveDataManager'

wWrapper = widget_base(wParent, EVENT_FUNC= myname+'_event' $
                       ,PRO_SET_VALUE= myname+'_setvalue' $
                       ,FUNC_GET_VALUE= myname+'_getvalue' $
                       ,uname=uName $
                       ,NOTIFY_REALIZE=myname+"_realize" $
                       ,KILL_NOTIFY=myname+"_cleanup" $
                       ,/base_align_center, /column)


;; Now create the tree widget
wDM = cw_ittreeview(wWrapper, oUI $
                    ,IDENTIFIER="DAVE Data Manager" $
                    ,uname="DAVEDATAMANAGER" $
                    ,xsize = xsize,ysize=ysize,/multiple);,/multiple)

;; Build context menu to be applied to the tree widget
widget_control, wDM, /context_events
wContext = WIDGET_BASE(wWrapper,/CONTEXT_MENU,event_func=myname+'_event')  

tt1 = "Delete dataset"
tt2 = "Create a dulicate copy of this dataset"
tt3 = "View detailed contents of selected item"
tt4 = "Create a new visualization from this dataset"
tt5 = "Add this dataset to an existing visualization"
wDel = widget_button(wContext, value="Delete", uname="DELETE",tooltip=tt1)
wDup = widget_button(wContext, value="Duplicate", uname="DUPLICATE",tooltip=tt2)
wView = widget_button(wContext, value="View Details", uname="VIEWDETAILS",tooltip=tt3)
wNewVis = widget_button(wContext, value="Create New Visualization",uname="NewVis",tooltip=tt4,/menu,sensitive=0)
wPlot = widget_button(wNewVis,value="Plot",uname="newPlot")
wPlot2D = widget_button(wNewVis,value='2D Plot',uname="newPlot2D")
wContour = widget_button(wNewVis,value="Contour",uname="newContour")
wImage = widget_button(wNewVis,value="Image",uname="newImage")
wSurface = widget_button(wNewVis,value="Surface",uname="newSurface")
wAddVis = widget_button(wContext, value="Add to Existing Visualization",uname="AddVis",tooltip=tt5,/menu,sensitive=0)
wPlot = widget_button(wAddVis,value="Plot",uname="addPlot")
wPlot = widget_button(wAddVis,value='2D Plot',uname="addPlot2D")
;wRename = widget_button(wContext, value="Rename...", uname="RENAME")

;; Buttons
;wButtons = widget_base(wWrapper, /row, space=10, /align_left)
;wVoid = widget_button(wbuttons, value="Import Variable...", uname="VARIABLE")
;wVoid = widget_button(wbuttons, value="Import File...", uname="FILE")

; Valid data types from which the context menu "New Visualization" can be launched
newVisValidDataTypes = ['DAVE1DATASET','DAVE1COMMONSTR','ASCIICOL','ASCIIGRP','ASCIISPE']
addVisValidDataTypes = ['DAVE1DATASET','DAVE1COMMONSTR','ASCIICOL','ASCIIGRP','ASCIISPE']
state = { oUI:oUI $
          ,wDM:wDM $
          ,ysize0:ysize $       ;initial Y size used in realize notify
          ,wContext: wContext $
          ,wDel:wDel $
          ,wDup:wDup $
          ,wView:wView $
          ,wNewVis:wNewVis $
          ,wAddVis:wAddVis $
          ,newVisValidDataTypes:newVisValidDataTypes $
          ,addVisValidDataTypes:addVisValidDataTypes $
          ,identifier:'' $      ; used to store identifier of selected item
          ,pDataIDs:ptr_new('') $ ; store identifiers of multiple seleted items
;          ,wRename:wRename $
;          ,wButtons:wButtons $
        }

;; Normally, this would be the first child on the widget, but the
;; tree view widget is just a tree, and not wrapped and changes
;; could cause issues.
sPtr = ptr_new(State,/no_copy)
widget_control, wWrapper, set_uvalue=sPtr

;; Due to how the context menu is created, we place the state in the
;; context menu also
widget_control, wContext, set_uvalue=sPtr

return, wWrapper

end
