; $Id$

;===============================================================================
; DAVEvisPlot3D::_UpdateErrorBars
; 
; PURPOSE:
;   Overide IDLitVisPlot3D::_UpdateErrorBars. Updates the error bar
;   geometry based on the plot data. Take histogram data into account.
;
; PARAMETERS:
;   dim [in] -  0 for X, 1 for Y, 2 for Z
;
; KEYWORDS:
;
pro DAVEvisPlot3d::_UpdateErrorBars, dim

    compile_opt idl2, hidden

    dimName = (['X','Y','Z'])[dim]
    oErrData = self->GetParameter(dimName + ' ERROR')
    if (~OBJ_VALID(oErrData)) then $
        return
    if (~oErrData->GetData(errdata)) then $
        return

    ; Retrieve plot data.
    oDataX = self->GetParameter('X')
    if (~OBJ_VALID(oDataX) || ~oDataX->GetData(xdata)) then $
        return
    if (~oDataX->GetMetaData('Distribution',distribution)) then distribution=''
    histogram = (strcmp(distribution,'HISTOGRAM',4,/FOLD_CASE) gt 0)? 1 : 0
    if (histogram) then begin
        ;; convert histograms to points
        nx = n_elements(xdata)
        index = lindgen(nx-1)
        points = 0.5*(xdata[index]+xdata[index+1]) ; ==> mid values of the bin boundaries
        xdata = temporary(points)
    endif

    oDataY = self->GetParameter('Y')
    if (~OBJ_VALID(oDataY) || ~oDataY->GetData(ydata)) then $
        return
    if (~oDataY->GetMetaData('Distribution',distribution)) then distribution=''
    histogram = (strcmp(distribution,'HISTOGRAM',4,/FOLD_CASE) gt 0)? 1 : 0
    if (histogram) then begin
        ;; convert histograms to points
        ny = n_elements(ydata)
        index = lindgen(ny-1)
        points = 0.5*(ydata[index]+ydata[index+1]) ; ==> mid values of the bin boundaries
        ydata = temporary(points)
    endif

    oDataZ = self->GetParameter('Z')
    if (~OBJ_VALID(oDataZ) || ~oDataZ->GetData(zdata)) then $
        return

    ndata = N_ELEMENTS(xdata)


    self._oPolyline->GetProperty, DATA=plotData

    ErrorData = fltarr(3, 2*ndata)
    ; set the other dimension's coordinates of polyline data
    ; same coordinate for both values of the specified dimension
    case dim of
    0: begin
        oError = self._oXerror
        plotdata = xdata
        ErrorData[1,0:*:2] = ydata
        ErrorData[1,1:*:2] = ydata
        ErrorData[2,0:*:2] = zdata
        ErrorData[2,1:*:2] = zdata
       end
    1: begin
        oError = self._oYerror
        plotdata = ydata
        ErrorData[0,0:*:2] = xdata
        ErrorData[0,1:*:2] = xdata
        ErrorData[2,0:*:2] = zdata
        ErrorData[2,1:*:2] = zdata
       end
    2: begin
        oError = self._oZerror
        plotdata = zdata
        ErrorData[0,0:*:2] = xdata
        ErrorData[0,1:*:2] = xdata
        ErrorData[1,0:*:2] = ydata
        ErrorData[1,1:*:2] = ydata
       end
    endcase

    case size(errdata, /n_dimensions) of
    1: begin
        ; vector of error values applied as both + and - error
        ErrorData[dim, 0:*:2] = plotdata - errdata
        ErrorData[dim, 1:*:2] = plotdata + errdata
    end
    2: begin
        ; 2xN array, [0,*] is negative error, [1,*] is positive error
        ErrorData[dim, 0:*:2] = plotdata - errdata[0,*]
        ErrorData[dim, 1:*:2] = plotdata + errdata[1,*]
    end
    else:
    endcase

    ; Handle logarithmic axes.
    if (self._xVisLog) then begin
        ErrorData[0, *] = alog10(ErrorData[0, *])
    endif
    if (self._yVisLog) then begin
        ErrorData[1, *] = alog10(ErrorData[1, *])
    endif
    if (self._zVisLog) then begin
        ErrorData[2, *] = alog10(ErrorData[2, *])
    endif


    polylineDescript = LONARR(3*ndata)
    polylineDescript[0:*:3]=2
    polylineDescript[1:*:3]=lindgen(ndata)*2
    polylineDescript[2:*:3]=lindgen(ndata)*2+1


    ; display the properties, even if the error bars themselves are hidden
    self->SetPropertyAttribute, [dimName + '_ERRORBARS', $
        'ERRORBAR_CAPSIZE', 'ERRORBAR_COLOR'], HIDE=0

    ; Retrieve HIDE property - it may be specified on command line
    ; and set prior to processing of the parameter
    oError->GetProperty, HIDE=hide
    oError->SetProperty, DATA=temporary(ErrorData), $
        HIDE=hide, $   ; may be hid from dataDisconnect
        POLYLINES=polylineDescript

    self->_UpdateCapSize

end



;===============================================================================
; DAVEvisPlot3D::OnDataChangeUpdate
; 
; PURPOSE:
;   Overide IDLitVisPlot::OnDataChangeUpdate. This procedure method is
;   called by a Subject via a Notifier when its data has changed.
;   This method obtains the data from the subject and updates the
;   internal IDLgrPlot object. 
;
; PARAMETERS:
;   oSubject [in] - The data item for the parameter that has changed.
;
;   parmName [in] - The name of the parameter that has changed.
;
; KEYWORDS:
;
pro DAVEvisPlot3d::OnDataChangeUpdate, oSubject, parmName

    compile_opt idl2, hidden


    case strupcase(parmName) of
    '<PARAMETER SET>':begin
        ;; Get our data
        position = oSubject->Get(/ALL, count=nCount, NAME=names)
        for i=0, nCount-1 do begin
            oData = oSubject->GetByName(names[i],count=nCount)
            IF nCount NE 0 THEN self->OnDataChangeUpdate,oData,names[i]
        endfor
    END

    'X': BEGIN
        success = oSubject->GetData(data)
        if (success) then begin
        
            oDataSpace = self->GetDataSpace(/UNNORMALIZED)
            if (OBJ_VALID(oDataSpace)) then begin
               ;; check for metadata and process them
               histogram = 0
               if (obj_isa(oSubject,'IDLitData') && (oSubject->GetMetadataCount() gt 0)) then begin
                   
                   ;; Axis Label
                   xLabel = ''
                   if (oSubject->GetMetaData('LONG_NAME',label)) then xLabel += label
                   if (oSubject->GetMetaData('UNITS',units)) then $
                     xLabel = (strlen(units) gt 0)? xLabel + ' ('+units+')' : xLabel
                   
                   oAxes = oDataSpace->GetAxes(/container)
                   
                   if (obj_valid(oAxes) && obj_isa(oAxes,'IDLitVisDataAxes')) then $
                     oAxes->SetProperty,xtitle=strtrim(xLabel)
                   
                   ;; Set plot mode: points / histograms
                   if (oSubject->GetMetaData('Distribution',distribution)) then $
                     histogram = (strcmp(distribution,'HISTOGRAM',4,/FOLD_CASE) gt 0)? 1 : 0
                   
                   self->SetProperty, histogram=histogram
                   
               endif
               
               if (histogram) then begin
                   ;; convert to point mode
                   nx = n_elements(data)
                   index = lindgen(nx-1)
                   points = 0.5*(data[index]+data[index+1]) ; ==> mid values of the bin boundaries
                   data = temporary(points)
               endif

            endif

            dim=0   ; X
            self->_SetPolylineOneVector, dim, data

            ; We might get an X by itself through the import dialog.
            ; Add dummy Y and Z parameters so that a valid plot is
            ; produced.  If this is normal processing of a
            ; parameter_set these will get overwritten by the desired
            ; data.  We will not set these as parameters so as not to
            ; confuse the user who imported only a single vector and
            ; does not expect to see dummy parameters in the parameter
            ; editor.
            lenX = n_elements(data)
            oDataY = self->GetParameter('Y')
            if ~OBJ_VALID(oDataY) then begin
                self->_SetPolylineOneVector, 1, indgen(lenX)
            endif
            oDataZ = self->GetParameter('Z')
            if ~OBJ_VALID(oDataZ) then begin
                self->_SetPolylineOneVector, 2, indgen(lenX)
            endif

            ; must process all dims at least once in order for ranges to
            ; be valid before adding shadows
            if (self._dimsProcessed GE 3) then self->_UpdateShadows
        endif
    END

    'Y': BEGIN
        success = oSubject->GetData(data)
        if (success) then begin

            oDataSpace = self->GetDataSpace(/UNNORMALIZED)
            if (OBJ_VALID(oDataSpace)) then begin        
               ;; check for metadata and process them
               histogram = 0
               if (obj_isa(oSubject,'IDLitData') && (oSubject->GetMetadataCount() gt 0)) then begin
                   
                   ;; Axis Label
                   yLabel = ''
                   if (oSubject->GetMetaData('LONG_NAME',label)) then yLabel += label
                   if (oSubject->GetMetaData('UNITS',units)) then $
                     yLabel = (strlen(units) gt 0)? yLabel + ' ('+units+')' : yLabel
                   
                   oAxes = oDataSpace->GetAxes(/container)
                   
                   if (obj_valid(oAxes) && obj_isa(oAxes,'IDLitVisDataAxes')) then $
                     oAxes->SetProperty,ytitle=strtrim(yLabel)
                   
                   ;; Set plot mode: points / histograms
                   if (oSubject->GetMetaData('Distribution',distribution)) then $
                     histogram = (strcmp(distribution,'HISTOGRAM',4,/FOLD_CASE) gt 0)? 1 : 0
                   
                   self->SetProperty, histogram=histogram
                   
               endif
               
               if (histogram) then begin
                   ;; convert to point mode
                   ny = n_elements(data)
                   index = lindgen(ny-1)
                   points = 0.5*(data[index]+data[index+1]) ; ==> mid values of the bin boundaries
                   data = temporary(points)
               endif
            endif

            dim=1   ; Y
            self->_SetPolylineOneVector, dim, data
            if (self._dimsProcessed GE 3) then  self->_UpdateShadows
        endif
    END

    'Z': BEGIN
        success = oSubject->GetData(data)
        if (success) then begin

            oDataSpace = self->GetDataSpace(/UNNORMALIZED)
            if (OBJ_VALID(oDataSpace)) then begin
 
               ;; check for metadata and process them
               if (obj_isa(oSubject,'IDLitData') && (oSubject->GetMetadataCount() gt 0)) then begin
                   ;; Z label
                   zLabel = ''
                   if (oSubject->GetMetaData('LONG_NAME',label)) then zLabel += label
                   if (oSubject->GetMetaData('UNITS',units)) then $
                     zLabel = (strlen(units) gt 0)? zLabel + ' ('+units+')' : zLabel
                   
                   oDataSpace = self->GetDataSpace()
                   oAxes = oDataSpace->GetAxes(/container)
                   
                   if (obj_valid(oAxes) && obj_isa(oAxes,'IDLitVisDataAxes')) then $
                     oAxes->SetProperty,ztitle=strtrim(zLabel)                   
               endif
            endif

            dim=2   ; Z
            self->_SetPolylineOneVector, dim, data
            if (self._dimsProcessed GE 3) then self->_UpdateShadows
            self->_UpdateSymSize        ; handle the case for an overplot with no range change
        endif
    END

    'VERTICES': BEGIN
        if OBJ_ISA(oSubject, 'IDLitDataContainer') then begin
            oData = oSubject->Get(/ALL)
            success = oData[0]->GetData(dataX)
            type=SIZE(dataX, /TYPE)
            data = MAKE_ARRAY(3, N_ELEMENTS(dataX), TYPE=type)
            data[0,*]=dataX
            success = oData[1]->GetData(dataY)
            data[1,*]=dataY
            success = oData[2]->GetData(dataZ)
            data[2,*]=dataZ
        endif else success = oSubject->GetData(data)
        if (success) then begin
            ;; Check for valid vertex data
            dims = SIZE(data, /DIMENSIONS)
            nDims = SIZE(data, /N_DIMENSIONS)
            if nDims ne 2 or (dims[0] ne 2 and dims[0] ne 3) then begin
                self->ErrorMessage, $
                  [IDLitLangCatQuery('Message:InvalidVertex:Text')], $
                    severity=0, $
                    TITLE=IDLitLangCatQuery('Message:InvalidVertex:Title')
            endif else begin
                oDataSpace = self->GetDataSpace(/UNNORMALIZED)
                if (OBJ_VALID(oDataSpace)) then begin
                    oDataSpace->GetProperty, XLOG=xLog, YLOG=yLog, ZLOG=zLog
                    if (n_elements(xLog) gt 0) && (xLog gt 0) then data[0,*] = alog10(data[0,*])
                    if (n_elements(yLog) gt 0) && (yLog gt 0) then data[1,*] = alog10(data[1,*])
                    if (n_elements(zLog) gt 0) && (zLog gt 0) then data[2,*] = alog10(data[2,*])
                endif
                self._oPolyline->SetProperty, DATA=data
                self->_UpdateShadows
                self->_UpdateSymSize        ; handle the case for an overplot with no range change
            endelse
        endif
    END

    'X ERROR': BEGIN
        success = oSubject->GetData(data)
        if (success) then begin
            self->_UpdateErrorBars, 0
            self->SetPropertyAttribute, 'ERRORBAR_COLOR', /SENSITIVE
            self->SetPropertyAttribute, 'ERRORBAR_CAPSIZE', /SENSITIVE
            self->SetPropertyAttribute, 'X_ERRORBARS', /SENSITIVE
            self._oItXErrorBarContainer->SetProperty, /IMPACTS_RANGE

        endif
    END

    'Y ERROR': BEGIN
        success = oSubject->GetData(data)
        if (success) then begin
            self->_UpdateErrorBars, 1
            self->SetPropertyAttribute, 'ERRORBAR_COLOR', /SENSITIVE
            self->SetPropertyAttribute, 'ERRORBAR_CAPSIZE', /SENSITIVE
            self->SetPropertyAttribute, 'Y_ERRORBARS', /SENSITIVE
            self._oItYErrorBarContainer->SetProperty, /IMPACTS_RANGE

        endif
    END

    'Z ERROR': BEGIN
        success = oSubject->GetData(data)
        if (success) then begin
            self->_UpdateErrorBars, 2
            self->SetPropertyAttribute, 'ERRORBAR_COLOR', /SENSITIVE
            self->SetPropertyAttribute, 'ERRORBAR_CAPSIZE', /SENSITIVE
            self->SetPropertyAttribute, 'Z_ERRORBARS', /SENSITIVE
            self._oItZErrorBarContainer->SetProperty, /IMPACTS_RANGE

        endif
    END

    'PALETTE': begin
        if(oSubject->GetData(data)) then begin
            if (size(data, /TYPE) ne 1) then data=bytscl(temporary(data))
            self._oPalette->SetProperty, $
                RED_VALUES=data[0,*], $
                GREEN_VALUES=data[1,*], $
                BLUE_VALUES=data[2,*]
            oVertColors = self->GetParameter('VERTEX_COLORS')
            self->SetPropertyAttribute, 'VISUALIZATION_PALETTE', /SENSITIVE
            if OBJ_VALID(oVertColors) then begin
                success = oVertColors->GetData(colors)
            endif
            if ~OBJ_VALID(oVertColors) || $
                (size(colors, /n_dimensions) gt 1) then begin
                ; define default indices
                oVertColorsDefault = OBJ_NEW('IDLitDataIDLVector', BINDGEN(256), $
                    NAME='<DEFAULT INDICES>')
                result = self->SetData(oVertColorsDefault, $
                    PARAMETER_NAME='VERTEX_COLORS',/by_value)
            endif
        endif
     end

    'VERTEX_COLORS': begin
        if(oSubject->GetData(data)) then begin
            if (size(data, /TYPE) ne 1) then data=bytscl(temporary(data))
            self._oPolyline->SetProperty, VERT_COLORS=data

            oRgbTable = self->GetParameter('PALETTE')
            if ~OBJ_VALID(oRgbTable) && $
                (size(data, /n_dimensions) eq 1) then begin
                ; define default palette, allows editing colors
                ; only used if vertex colors parameter is supplied
                ; and vertex colors are indices not colors.
                ramp = BINDGEN(256)
                colors = transpose([[ramp],[ramp],[ramp]])
                oColorTable = OBJ_NEW('IDLitDataIDLPalette', colors, NAME='RGB Table')

                ;; Set the data as by_value, so the parameter interface
                ;; will manage it.
                result = self->SetData(oColorTable, PARAMETER_NAME='PALETTE',/by_value)
            endif
        endif
    end

    ELSE: ; ignore unknown parameters
    ENDCASE

end


;===============================================================================
; DAVEvisPlot3D::Init
;
; PURPOSE:
;   Constructor
;
; RETURN VALUE:
;   1 = success
;   0 = failure
;
function DAVEvisPlot3D::Init, _REF_EXTRA=etc
compile_opt idl2

; Call base class
if (~self->IDLitVisPlot3D::Init(_EXTRA=etc)) then return, 0


; Set some sensible defults
colors = [[0,0,0] $     ; Black
         ,[255,0,0] $   ; Red
         ,[0,255,0] $   ; Green
         ,[0,0,255] $   ; Blue
         ,[100,0,0] $   ; dark red
         ,[0,100,0] $   ; dark Green
         ,[0,0,100] $   ; dark Blue
         ]
icol = (size(colors,/dimensions))[1]
cindex = fix(icol * randomu(long(SYSTIME(/seconds))));randomu(seed))   ; values 0 --> icol-1
color = colors[*,cindex]                ; one from colors array above
sym_index = fix(23*randomu(seed))+2      ; symbols 2 --> 24
if (sym_index eq 3 || sym_index eq 8 || sym_index eq 9) then sym_index = 24  ; ignore symbol 3 (Dot), 8 (>) and 9 (<)

self->SetProperty, linestyle=6 $                ; no lines
                  ,sym_index=sym_index $        ; randomly selected symbol
                  ,use_default_color=1 $        ; don't use seperate color for symbol
                  ,errorbar_color=color $       ; errorbar color
                  ,sym_color=color $                ; foreground color
                  ,color=color                 ; foreground color
;print,'color=',color

return, 1

end


;===============================================================================
; DAVEvisPlot3D__define
; 
; PURPOSE:
;   DAVEvisPlot class structure definition
;
pro DAVEvisPlot3D__define

    compile_opt idl2

    struc = {DAVEvisPlot3D, $
        inherits IDLitVisPlot3D }

end
