; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEvisContour
;
; PURPOSE:
;   This file simply inherits and extends IDL's IDLitVisContour visualization class
;
; CATEGORY:
;   DAVE Visualization
;
; SUPERCLASSES:
;   IDLitVisContour
;
; SUBCLASSES:
;
; CREATION:
;
; METHODS:
;
; INTERFACES:
;
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; March 2005
;-
;###############################################################################



;===============================================================================
; DAVEvisContour::EnsureXYParameters
; 
; PURPOSE:
;   Ensure that X and Y parameters exist, based on the contour data 
;   dimensions. Overides IDLitVisContour::EnsureXYParameters and is
;   essentially the same except that any data that are automatically
;   created are stored in the system 'Data Manager' and not underneath
;   the parent of the 'Z' parameter.
;
; PARAMETERS:
;
; KEYWORDS:
;   none
;
pro DAVEvisContour::EnsureXYParameters

; Retrieve Z parameter first to get data dimensions.
oZParam = self->GetParameter('Z')
if (~OBJ_VALID(oZParam)) then return
if (~(oZParam->GetData(pData, /POINTER))) then return
dims = SIZE(*pData, /DIMENSIONS)

paramNames = ['X', 'Y']
for i=0,1 do begin
    ;; Check if parameter already exists.
    oParam = self->GetParameter(paramNames[i])
    if (~obj_valid(oParam)) then begin
        ;; Create and set the parameter.
        data = DINDGEN(dims[i])
        oData = OBJ_NEW('IDLitDataIDLVector', data, NAME=paramNames[i])
        oData->SetProperty, /AUTO_DELETE
        self->SetParameter, paramNames[i], oData, /NO_UPDATE
        
        ;; Add to data manager.
        oData = self->GetParameter(paramNames[i])

        self->AddByIdentifier,'/DATA MANAGER',oData
    endif
endfor

; Send a notification message to update UI
self->DoOnNotify, self->GetFullIdentifier(),"ADDITEMS", ''

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEvisContour::_GetXYdata
; 
; PURPOSE:
;   Overide IDLitVisContour::_GetXYdata. Retrieve the data associated
;   with the indicated parameter. DAVE extension copes with histogram
;   data.
;
; PARAMETERS:
;   param [in] - 'X' or 'Y'
;
; KEYWORDS:
;    Returns the data associated with the input parameter if
;    successul. Note that an index will be generated if no data is
;    currently attached to the parameter.
;
function DAVEvisContour::_GetXYdata, param

    compile_opt idl2, hidden

    oParamZ = self->GetParameter('Z')
    if (~OBJ_VALID(oParamZ) || ~oParamZ->GetData(pData, /POINTER)) then $
        return, 0

    dims = SIZE(*pData, /DIMENSIONS)
    dim = (param eq 'X') ? dims[0] : dims[1]

    oParamXY = self->GetParameter(param)
    if (~OBJ_VALID(oParamXY) || ~oParamXY->GetData(xyData)) then begin
        xyData = FINDGEN(dim)
    endif

    ;; if parameter contains histogram data, convert it into point mode
    if (~obj_valid(oParamXY)  ||  ~oParamXY->GetMetaData('Distribution',distribution)) then distribution=''
    histogram = (strcmp(distribution,'HISTOGRAM',4,/FOLD_CASE) gt 0)? 1 : 0
    if (histogram) then begin
        ;; convert histograms to points
        nx = n_elements(xyData)
        index = lindgen(nx-1)
        xydata = 0.5*(xyData[index]+xyData[index+1]) ; ==> mid values of the bin boundaries
    endif

   ; Modified this to support 2D x and y parameters
    minn = MIN(xyData, MAX=maxx)
    ;if ((N_ELEMENTS(xyData) ne dim) || (minn eq maxx)) then begin
    ;    xyData = FINDGEN(dim)
    ;endif
    ndimXY = SIZE(xyData, /N_DIMENSIONS)
    if ((ndimXY eq 1 && N_ELEMENTS(xyData) ne dim) || $
        (ndimXY eq 2 && ~ARRAY_EQUAL(SIZE(xyData, /DIM), dims)) || $
        (minn eq maxx)) then begin
        xyData = FINDGEN(dim)
    endif



    ; We have our X or Y data. Now check for logarithmic.
    isLog = (param eq 'X') ? self._xVisLog : self._yVisLog

    ; Turn off log if values are zero or negative.
    if (minn le 0) then $
        isLog = 0

    ; Take log if necessary.
    if (isLog) then $
        xyData = ALOG10(TEMPORARY(xyData))

    ; May need to disable log.
    if (param eq 'X') then begin
        self._xVisLog = isLog
    endif else begin
        self._yVisLog = isLog
    endelse

    return, xyData

end



;===============================================================================
; DAVEvisContour::OnDataChangeUpdate
; 
; PURPOSE:
;   Extend IDLitVisContour::OnDataChangeUpdate.
;   This method is called when the data associated with a parameter
;   has changed. When called, the visualization performs the
;   desired updates to reflect the change.
;
; PARAMETERS:
;   oSubject - The data item for the parameter that has changed.
;
;   parmName - The name of the parameter that has changed.
;
; KEYWORDS:
;   none
;
pro DAVEvisContour::OnDataChangeUpdate, oSubject, parmName
compile_opt idl2

; Call base class method
self->IDLitVisContour::OnDataChangeUpdate, oSubject, parmName

;; Perform additional tasks not handled by the base class method
case strupcase(parmName) of
    'Z': begin
        ;; check for metadata and process them
        oData = oSubject
        if (obj_isa(oData,'IDLitData') && (oData->GetMetadataCount() gt 0)) then begin
            ;; Y label
            zLabel = ''
            if (oData->GetMetaData('LONG_NAME',label)) then zLabel += label
            if (oData->GetMetaData('UNITS',units)) then $
              zLabel = (strlen(units) gt 0)? zLabel + ' ('+units+')' : zLabel
            
            oDataSpace = self->GetDataSpace()
            oAxes = oDataSpace->GetAxes(/container)
            
            if (obj_valid(oAxes) && obj_isa(oAxes,'IDLitVisDataAxes')) then $
              oAxes->SetProperty,ztitle=strtrim(zLabel)
        endif
        
        self->SetProperty, /pal_color ; Use palette
    end
    
    'Y': begin
        ;; check for metadata and process them
        oData = oSubject
        if (obj_isa(oData,'IDLitData') && (oData->GetMetadataCount() gt 0)) then begin
            ;; Y label and units
            yLabel = ''
            if (oData->GetMetaData('LONG_NAME',label)) then yLabel += label
            if (oData->GetMetaData('UNITS',units)) then $
              yLabel = (strlen(units) gt 0)? yLabel + ' ('+units+')' : yLabel
            
            oDataSpace = self->GetDataSpace()
            oAxes = oDataSpace->GetAxes(/container)
            
            if (obj_valid(oAxes) && obj_isa(oAxes,'IDLitVisDataAxes')) then $
              oAxes->SetProperty,ytitle=strtrim(yLabel)
            
            ;; If histogram data, convert to point mode
;             if (oData->GetMetaData('Distribution',dist)) then begin
;                if (strmid(strupcase(dist),0,4) eq 'HIST') then begin
;                   yData = self->_GetXYdata('Y')
;                   len = n_elements(yData)
;                   index = lindgen(len-1)
;                   yData = 0.5*(yData[index]+yData[index+1]) ; ==> mid values of the bin boundaries

;                   self._oContour->SetProperty, GEOMY=temporary(yData)
;                   ;; Force a recalculation of the tick interval/length.
;                   self->IDLitVisContour::SetProperty,TICKINTERVAL=self._tickinterval,TICKLEN=self._ticklen
;                   self->UpdateSelectionVisual
;                endif
;             endif
        endif
    end
    
    'X': begin
        ;; check for metadata and process them
        oData = oSubject
        if (obj_isa(oData,'IDLitData') && (oData->GetMetadataCount() gt 0)) then begin
            ;; X label and units
            xLabel = ''
            if (oData->GetMetaData('LONG_NAME',label)) then xLabel += label
            if (oData->GetMetaData('UNITS',units)) then $
              xLabel = (strlen(units) gt 0)? xLabel + ' ('+units+')' : xLabel

            oDataSpace = self->GetDataSpace()
            oAxes = oDataSpace->GetAxes(/container)

            if (obj_valid(oAxes) && obj_isa(oAxes,'IDLitVisDataAxes')) then $
              oAxes->SetProperty,xtitle=strtrim(xLabel)
            
            ;; If histogram data, convert to point mode
;             if (oData->GetMetaData('Distribution',dist)) then begin
;                if (strmid(strupcase(dist),0,4) eq 'HIST') then begin
;                   xData = self->_GetXYdata('X')
;                   len = n_elements(xData)
;                   index = lindgen(len-1)
;                   xData = 0.5*(xData[index]+xData[index+1]) ; ==> mid values of the bin boundaries

;                   self._oContour->SetProperty, GEOMX=temporary(xData)
;                   ;; Force a recalculation of the tick interval/length.
;                   self->IDLitVisContour::SetProperty,TICKINTERVAL=self._tickinterval,TICKLEN=self._ticklen
;                   self->UpdateSelectionVisual
;                endif
;             endif
         endif                    
     end
    
    else:
endcase

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEvisContour::_LoadPalette
;
; Purpose:
;   Overide IDLitVisContour::_LoadPalette
;   This procedure method is used to load color values into the
;   color table. RGB values may come from a parameter defined
;   by the user or default values from a rainbow palette.
;
; Arguments:
;   None.
;
; Keywords:
;   None.
;
pro DAVEvisContour::_LoadPalette

compile_opt idl2

oDataRGB = self->GetParameter('PALETTE')
if ~OBJ_VALID(oDataRGB) then begin
    ;; Define default palette, allows editing color table

    ;; Create a temporal IDLgrPalette from IDL color table
    ;; 13 (rainbow). Extract rgb values from it.
    oVoidPalette = obj_new('IDLgrPalette')
    oVoidPalette->LoadCT,13
    oVoidPalette->GetProperty, red_values=red,green_values=green,blue_values=blue
    obj_destroy,oVoidPalette
    colors = transpose([[red],[green],[blue]])
    oDataRGB = OBJ_NEW('IDLitDataIDLPalette', colors, NAME='RGB Table')
endif else begin
    ;; reset even if parameter already existed in order to reconnect palette
    ;; and update parameter for export
    success = oDataRGB->GetData(colors)
    oDataRGB = OBJ_NEW('IDLitDataIDLPalette', colors, NAME='RGB Table')
endelse
; connect palette
result = self->SetData(oDataRGB, PARAMETER_NAME='PALETTE',/by_value)

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEvisContour::Init
; 
; PURPOSE:
;   Init
;
; PARAMETERS:
;
; KEYWORDS:
;   none
;
function DAVEvisContour::Init, _REF_EXTRA=etc
compile_opt idl2

print,'DAVEvisContour::Init called...'

; Call base class init method
if (~self->IDLitVisContour::Init(_EXTRA=etc)) then return, 0

; Set sensible default properties
self->SetProperty, /fill;, n_levels=21; filled contours
;self->setPropertyAttribute,'PAL_COLOR',SENSITIVE=1,
self._zVisLog = 1

return, 1
end


;===============================================================================
; DAVEvisContour__define
; 
; PURPOSE:
;   DAVEvisContour class structure definition
;
pro DAVEvisContour__define

compile_opt idl2

struct = {DAVEvisContour, $
         inherits IDLitVisContour }

end
