
; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEopDataSlice
;
; PURPOSE:
;   A simple operation that rebins the plottable data of an
;   IDLitData object. The plottable data must consist of an dependent
;   component with at least one independent component.
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOperation
;
; METHODS:
;   DoAction
;   DoExecuteUI
;   GetProperty
;   RecordInitialValues
;   RecordFinalValues
;   RedoOperation
;   SetProperty
;   UndoOperation
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Mar 2005
;-
;###############################################################################


;===============================================================================
; DAVEopDataSlice::GetProperty
; 
; PURPOSE:
;   Accessor
;
; PARAMETERS:
;
; KEYWORDS:
;   offset [out] - The offset that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDataSlice::GetProperty, rangemin=rangemin,rangemax=rangemax $
  ,slicecenter=slicecenter,slicewidth=slicewidth,axis=axis, norm=norm $
  ,xValue=xValue, yValue=yValue, dataValue=dataValue, xtitle=xtitle,ytitle=ytitle $
  , _REF_EXTRA=etc
compile_opt idl2

; rangemin
if (arg_present(rangemin)) then rangemin = (Self.axis eq 0)? Self.range[0,0] : Self.range[1,0]

;rangemax
if (arg_present(rangemax)) then  rangemax = (Self.axis eq 0)? Self.range[0,1] : Self.range[1,1]

;slicecenter
if (arg_present(slicecenter)) then slicecenter = (Self.axis eq 0)? $
                                                          Self.slicecenter[0] : Self.sliceCenter[1]

;slicewidth
if (arg_present(slicewidth)) then slicewidth = (Self.axis eq 0)? $
                                                        Self.slicewidth[0] : Self.sliceWidth[1]
;
if (arg_present(axis)) then axis = self.axis

;
if (arg_present(norm)) then norm = self.norm

if (obj_valid(Self.oTarget)) then begin
   ;
   if (arg_present(xValue)) then (Self.oTarget)->GetProperty, axis1Value=xValue   
   ;
   if (arg_present(yValue)) then (Self.oTarget)->GetProperty, axis2Value=yValue   
   ;
   if (arg_present(dataValue)) then (Self.oTarget)->GetProperty, dataValue=dataValue
   ;
   if (arg_present(xTitle)) then (Self.oTarget)->GetProperty, axis1Label=xTitle
   ;
   if (arg_present(yTitle)) then (Self.oTarget)->GetProperty, axis2Label=yTitle
endif

; call base class accessor
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::GetProperty, _EXTRA=etc

end


;===============================================================================
; DAVEopDataSlice::SetProperty
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;
; KEYWORDS:
;   offset [in] - The offset that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDataSlice::SetProperty, rangemin=rangemin,rangemax=rangemax, norm=norm $
  ,slicecenter=slicecenter,slicewidth=slicewidth,axis=axis, no_update=no_update $ ;,types=types
  , _EXTRA=etc
compile_opt idl2

; rangemin
if (n_elements(rangemin)) then begin
   if (Self.axis eq 0) then self.range[0,0]=rangemin else self.range[1,0]=rangemin
endif

; rangemax
if (n_elements(rangemax)) then begin
   if (Self.axis eq 0) then self.range[0,1]=rangemax else self.range[1,1]=rangemax
endif

rmin = (Self.axis eq 0)? Self.range[0,0] : Self.range[1,0]
rmax = (Self.axis eq 0)? Self.range[0,1] : Self.range[1,1]
sw = (Self.axis eq 0)? Self.sliceWidth[0] : Self.sliceWidth[1]
sc = (Self.axis eq 0)? Self.sliceCenter[0] : Self.sliceCenter[1]
; slicecenter
if (n_elements(sliceCenter)) then begin
   ;slicecenter must be between rangemin and rangemax
   sliceCenter = rmin > sliceCenter < rmax
   
   ; slicecenter - 0.5*slicewidth should be greater than rangemin
   value = sliceCenter - 0.5*sw
   if (value lt rmin) then sliceCenter = rmin + 0.5*sw
   
   ; slicecenter + 0.5*slicewidth should be less than rangemax
   value = sliceCenter + 0.5*sw
   if (value gt rmax) then sliceCenter = rmax - 0.5*sw
   
   self.sliceCenter[Self.axis] = sliceCenter
endif

; slicewidth
if (n_elements(sliceWidth)) then begin   
   ; slicecenter - 0.5*slicewidth should be greater than rangemin
   value = sc - 0.5*sliceWidth
   if (value lt rmin) then sliceWidth = 2 * (sc - rmin)
   
   ; slicecenter + 0.5*slicewidth should be less than rangemax
   value = sc + 0.5*sliceWidth
   if (value gt rmax) then sliceWidth = 2* (rmax - sc)
   
   self.slicewidth[Self.axis] = sliceWidth
endif

; axis
if (n_elements(axis)) then self.axis = axis

; norm
if (n_elements(norm)) then self.norm = norm

; Update the slice parameters as they depend on the ind axis data
;Self->RefreshSliceSettings, /noreset

; Call base class mutator
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::SetProperty, _EXTRA=etc 

end


;===============================================================================
; DAVEopDataSlice::RefreshSliceSettings
; 
; PURPOSE:
;   
;
; PARAMETERS:
;
; KEYWORDS:
;
;
function DAVEopDataSlice::RefreshSliceSettings, noreset=noreset
compile_opt idl2

reset = ~keyword_set(noreset)

(Self.oTarget)->GetProperty, axis1Value=x, axis2Value=y;, dataValues=dat
x0 = min(x,max=x1)
y0 = min(y,max=y1)
Self.range[0,*] = [x0,x1]
Self.range[1,*] = [y0,y1]
xr = x1 - x0
yr = y1 - y0

; current operation's slice width / center
cur_sc = Self.sliceCenter
cur_sw = Self.sliceWidth

; if these are outside the data window, then reset them
xc = 0.5*(x0 + x1)
yc = 0.5*(y0 + y1)
if (cur_sc[0] le x0 || cur_sc[0] ge x1) then Self.sliceCenter[0] = xc
if (cur_sw[0] gt xr || cur_sw[0] le 0.0) then Self.sliceWidth[0] = 0.05*xr
if (cur_sc[1] le y0 || cur_sc[1] ge y1) then Self.sliceCenter[1] = yc
if (cur_sw[1] gt yr || cur_sw[1] le 0.0) then Self.sliceWidth[1] = 0.05*yr

;if (reset) then begin
;   x0 = 0.5*(x0 + x1)
;   y0 = 0.5*(y0 + y1)
;   Self.sliceCenter = [xc,yc]
;   Self.sliceWidth = [0.0,0.0] 
;endif

; reset operation properties with the new values
range = Self.range[Self.axis,*]
Self->SetProperty, rangemin=range[0], rangemax=range[1] $
                    ,sliceCenter=Self.sliceCenter[Self.axis] $
                    ,sliceWidth=Self.sliceWidth[Self.axis]

return, 1
end



;===============================================================================
; DAVEopDataSlice::RecordInitialValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataSlice::RecordInitialValues, oCmdSet, oTarget, idProp
compile_opt idl2

; retrieve the first and only command object created earlier
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
self->GetProperty,slicecenter=center,slicewidth=width,axis=axis,norm=norm
void = oCmd->AddItem('OLD_AXIS',axis)
void = oCmd->AddItem('OLD_CENTER',center)
void = oCmd->AddItem('OLD_WIDTH',width)
void = oCmd->AddItem('OLD_NORM',norm)

return, 1

end


;===============================================================================
; DAVEopDataSlice::RecordFinalValues
; 
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataSlice::RecordFinalValues, oCmdSet, oTarget, idProp
compile_opt idl2

; retrieve the first and only command object created earlier
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then return, 0

; Get the value to be stored and add to command obj
self->GetProperty,slicecenter=center,slicewidth=width,axis=axis,norm=norm
void = oCmd->AddItem('NEW_AXIS',axis)
void = oCmd->AddItem('NEW_CENTER',center)
void = oCmd->AddItem('NEW_WIDTH',width)
void = oCmd->AddItem('NEW_NORM',norm)

return, 1

end


;===============================================================================
; DAVEopDataSlice::DoExecuteUI
; 
; PURPOSE:
;   Launch the UI dialog to collect appropriate user information for
;   this operation.
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataSlice::DoExecuteUI
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Use the build-in 'PropertySheet' UI service to let the user
; customize the operation's property.
return, oTool->DoUIService('DataSlice',self)

end


;===============================================================================
; DAVEopDataSlice::UndoOperation
; 
; PURPOSE:
;   Provides the 'undo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataSlice::UndoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects.
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then begin
   obj_destroy, oCmdSet
   return, 0
endif

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Restore operation attributes
void = oCmd->GetItem('OLD_AXIS',axis)
void = oCmd->GetItem('OLD_CENTER',center)
void = oCmd->GetItem('OLD_WIDTH',width)
void = oCmd->GetItem('OLD_NORM',norm)
self->SetProperty, axis=axis,norm=norm,sliceCenter=center,sliceWidth=width

; Undo previous slice by deleting the previously generated output from the Data Manager
void = oCmd->GetItem('OUTPUTID',idDataset)
oDataset = oTool->GetByIdentifier(idDataset)
oDM = oTool->GetService("DAVEDATA_MANAGER")  
status = oDM->DeleteData(oDataset->GetFullIdentifier())

; Indicate that the tool has been modified and force it to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1

end


;===============================================================================
; DAVEopDataSlice::RedoOperation
; 
; PURPOSE:
;   Provides the 'redo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDataSlice::RedoOperation, oCmdSet
compile_opt idl2

; Retrieve the command objects.
oCmd = oCmdSet->Get(position=0)
if (~obj_valid(oCmd)) then begin
   obj_destroy, oCmdSet
   return, 0
endif

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

oCmd->GetProperty, target_identifier=idTarget
oTarget = oTool->GetByIdentifier(idTarget)
if (~obj_valid(oTarget)) then begin
   obj_destroy, oCmdSet
   return, 0
endif


; Restore operation attributes
void = oCmd->GetItem('NEW_AXIS',axis)
void = oCmd->GetItem('NEW_CENTER',center)
void = oCmd->GetItem('NEW_WIDTH',width)
void = oCmd->GetItem('NEW_NORM',norm)
self->SetProperty, axis=axis,norm=norm,sliceCenter=center,sliceWidth=width

void = oCmd->GetItem('LOLIM',lolim)
void = oCmd->GetItem('UPLIM',uplim)
void = oCmd->GetItem('WIDTH',width)
void = oCmd->GetItem('TRMT',trmt)
void = oCmd->GetItem('NAMETAG',nameTag)

; Get data from the target object
case axis of
   0:oTarget->GetProperty, axis1Value=vin,dataValue=zin, errorValue=zein, axis1Distribution=dist
   1:oTarget->GetProperty, axis2Value=vin,dataValue=zin, errorValue=zein, axis2Distribution=dist
   else: return, 0
endcase

; create errors if missing
if (n_elements(zein) le 0) then zein = zin*0.0 + 1.0

if (axis eq 1) then begin
    zin = transpose(temporary(zin))  ; transpose data so that y-axis data become the rows to be rebinned
    zein = transpose(temporary(zein))
endif


; Slice data by performing a rebin where the output bins correspond to the region to be sliced
case strupcase(strmid(dist,0,5)) of   ; first 4 letters of distribution type
   'HISTO': begin
      vout = findgen(2)*width + lolim
      drebin,vin,zin,zein,vout,zout,zeout,/hist,/to_hist,err=err,emsg=emsg
   end
   
   'POINT': begin
      vout = findgen(3)*width + lolim - 0.5 * width
      drebin,vin,zin,zein,vout,zout,zeout,/points,/to_points,err=err,emsg=emsg
      zout = temporary(zout[1,*])
      zeout = temporary(zeout[1,*])   
   end
   
   else: return, 0
endcase

if (err ne 0) then begin
   !error_state.code=err
   !error_state.msg = emsg
   oTool->StatusMessage,emsg
   obj_destroy, oCmdSet   ; no longer valid so remove!
   return, 0
endif

; reform the output to 1D case it isn't (ie 1xN)
; and include normalization (by default, Drebin normalises to bin/slice width)
zout = norm * reform(temporary(zout))
zeout = norm * reform(temporary(zeout))

; - Convert the contents of the target dataset into a davePtr structure
; - replace appropriate sections of the davePtr with the cut results
; - create a new dataset from the davePtr structure
; - add the new dataset to the Data Manager
if (~oTarget->toDavePtr(davePtr, errMsg)) then begin
   oTool->StatusMessage, errMsg
   return, 0
endif

; Since the dataset is now 1D, the cut axis has to be reset to a scalar and 
; changed into the y-axis while the uncut axis now becomes the x-axis. This is 
; because for a 1D davePtr, the x-axis is expected to hold the independent data.
; And, of course, the dependent and independent data must be set to the sliced 
; data and error.
etc = {yvals:0.5*(lolim+uplim)}
if (axis eq 0) then begin   ; no need to do this if slicing the y-axis
   oTarget->GetProperty, axis2Value=xv,axis2Distribution=xd,axis2Units=xu,axis2Label=xl
   etc = create_struct(etc,'xVals',xv,'xUnits',xu,'xLabel',xl,'xType',xd)
endif

status = modify_dave_pointer(davePtr, qty = zout, err = zeout $ ; modify the davePtr
                                 ,treatment = trmt $
                                 ,ermsg = errMsg $
                                 ,_EXTRA=etc $
                                 )
if (~status) then begin
   oTool->StatusMessage, errMsg
   return, 0
endif

; Create a DaveDataset object
oResult = obj_new('DAVEDataset',davePtr=davePtr, nameTag=nameTag)

; And add it to the Data Manager
oTool->AddDatasetToDataManager, oResult

;  Replace previous output ref with current one for undo/redo
void = oCmd->AddItem('OUTPUTID',oResult->GetFullIdentifier(),/overwrite)

; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow


return, 1

end


;===============================================================================
; DAVEopDataSlice::DoAction
; 
; PURPOSE:
;   Implements the main function for this operation. Apply the
;   operation's offset to the data being operated on.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an IDLitCommandSet object
;    If unsuccessful, a NULL object.
;
function DAVEopDataSlice::DoAction, oTool
compile_opt idl2

; oTool should be valid and the DAVE Main Tool
if (~obj_valid(oTool) || ~obj_isa(oTool,'DAVETOOL')) then return, obj_new()

; Get the selected dataset(s)
oSelections = oTool->GetSelectedData()
void = where(obj_valid(oSelections),cnt)
if (cnt eq 0) then begin
    oTool->StatusMessage, 'No valid data to operate on! Select a dataset from the Data Browser tree.'
    return, obj_new()
endif

; Locate first dataset containing experiment data
; Only top-level entries in the Data Manager Browser can be valid datasets
self->GetProperty, types=validTypes ; should be ['DAVE1DATASET','ASCIISPE','ASCIICOL','ASCIIGRP']
nValid = n_elements(validTypes)
found = 0
for i = 0,n_elements(oSelections)-1 do begin
    if (found) then continue   ; need to locate one dataset only

    ;; Only top-level dataset selections from the data browser can be processed
    ;; ie the this operation is design to deal with complete datasets
    oTop = getmaindataobjectparent(oSelections[i])
    if (~obj_valid(oTop)) then continue
    ;if (oTop ne oSelections[i]) then continue ; ie selection must be a main entry in the tree

    ;; Search for one of the valid data types from this selection.
    ;; Essentially, the valid types refer to the experimental or plottable data containing
    ;; an independent, dependent and error component!
    j = 0
    repeat begin
        oRes = oSelections[i]->GetByType(validTypes[j],count=found)
    endrep until (found || ++j ge nValid)

    if (found) then oTarget=oTop
endfor

if (~obj_valid(oTarget)) then begin
    oTool->StatusMessage, 'Selected dataset could not be found!'
    return, obj_new()
endif

; if target does not contain 2D data then bail out
oTarget->GetProperty, nDimensions=nDims
if (nDims ne 2) then begin
    oTool->StatusMessage, 'Slice operation must be performed on a 2D dataset!'
    return, obj_new()
endif

; Make the first valid target the operation's target dataset
self.oTarget = oTarget

; Create a command set obj by calling the base class DoAction
oCmdSet = self->IDLitOperation::DoAction(oTool)

;; Set operation properties based on the selected data
if (~self->RefreshSliceSettings()) then begin
    obj_destroy, oCmdSet
    return, obj_new()
endif

; create a command object to store operation details, 
; This is the first and only one that will be created for this operation
oCmd = obj_new('IDLitCommand',target_identifier=oTarget->getfullidentifier())
if (~obj_valid(oCmd)) then return, 0
oCmdSet->Add, oCmd

; Is some UI needed prior to execution?
self->GetProperty, show_execution_ui=doUI
hasPropSet = 0b
if (doUI) then begin
    ;; Record initial properties for this operation.
    if (~self->RecordInitialValues(oCmdSet,oTarget,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; Perform our UI.
    if (~self->DoExecuteUI()) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif

    ;; The hourglass will have been cleared by the dialog.
    ;; So turn it back on.
    ;void = oTool->DoUIService("HourGlassCursor", self)
    
    ;; Record final properties for this operation.
    if (~self->RecordFinalValues(oCmdSet,oTarget,'')) then begin
        obj_destroy, oCmdSet
        return, obj_new()
    endif
endif

;; Slice the data, creating a new dataset in the process
if (~self->Slice(oCmdSet)) then begin
    obj_destroy, oCmdSet
    return, obj_new()
endif

; return the command set obj
return, oCmdSet

end


;===============================================================================
; DAVEopDataSlice::Slice
; 
; PURPOSE:
;   Perform the data cut using the specified slice parameters. Create a new dataset
;   and add it to the Data Browser.
;
; PARAMETERS:
;   oCmdSet [in|out] - an IDLitCommandSet object which stores the
;                      undo/redo buffer info.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, 1
;    If unsuccessful, 0.
;
function DAVEopDataSlice::Slice, oCmdSet
compile_opt idl2

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Get current slice operation settings
self->GetProperty,slicecenter=center,slicewidth=width,axis=axis

if (width le 0.0) then begin
   msg = 'Slice width cannot be 0.0'
   oTool->StatusMessage, msg
   return, 0
endif

lolim = center - 0.5 * width
uplim = lolim + width

oTarget = Self.oTarget
if (~obj_hasmethod(oTarget,'Clone')) then begin
   oTarget->GetProperty, name=name
   msg = 'Cannot handle unknown dataset object - '+name
   oTool->StatusMessage, msg
   return, 0
endif


; Get data from the target object
case axis of
   0:oTarget->GetProperty, axis1Value=vin,dataValue=zin, errorValue=zein, axis1Distribution=dist
   1:oTarget->GetProperty, axis2Value=vin,dataValue=zin, errorValue=zein, axis2Distribution=dist
   else: return, 0
endcase

; create errors if missing
if (n_elements(zein) le 0) then zein = zin*0.0 + 1.0

if (axis eq 1) then begin
    zin = transpose(temporary(zin))  ; transpose data so that y-axis data become the rows to be rebinned
    zein = transpose(temporary(zein))
endif

; If vin is not strictly increasing, then reverse it and the data
if (~strincrease(vin)) then begin
   rverse = 1
   vin = reverse(vin,/overwrite)
   zin = reverse(zin,/overwrite)
   zein = reverse(zein,/overwrite)
endif else rverse = 0

; Limit output range to within the data
nin = n_elements(vin)
lolim = lolim > vin[0]
uplim = uplim < vin[nin-1]
width = uplim - lolim

; Slice data by performing a rebin where the output bins correspond to the region to be sliced
case strupcase(strmid(dist,0,5)) of   ; first 4 letters of distribution type
   'HISTO': begin
      vout = findgen(2)*width + lolim
      drebin,vin,zin,zein,vout,zout,zeout,/hist,/to_hist,err=err,emsg=emsg
   end
   
   'POINT': begin
      vout = findgen(3)*width + lolim - 0.5 * width
      drebin,vin,zin,zein,vout,zout,zeout,/points,/to_points,err=err,emsg=emsg
      zout = temporary(zout[1,*])
      zeout = temporary(zeout[1,*])   
   end
   
   else: return, 0
endcase

if (err ne 0) then begin
   !error_state.code=err
   !error_state.msg = emsg
   oTool->StatusMessage,emsg
   return, 0
endif

; reform the output to 1D case it isn't (ie 1xN)
; and include normalization (by default, Drebin normalises to bin/slice width)
norm = (Self.norm eq 0)? width : 1.0
zout = norm * reform(temporary(zout))
zeout = norm * reform(temporary(zeout))

; - Convert the contents of the target dataset into a davePtr structure
; - replace appropriate sections of the davePtr with the cut results
; - create a new dataset from the davePtr structure
; - add the new dataset to the Data Manager
if (~oTarget->toDavePtr(davePtr, errMsg)) then begin
   oTool->StatusMessage, errMsg
   return, 0
endif
oTarget->GetProperty, name=name
fmt = '(G10.3)'
axisname = (axis eq 0)? 'X-' : 'Y-'
nameTag = name+'_'+axisname+'slice_'+strtrim(string(lolim,format=fmt),2)+'_' $
                                     +strtrim(string(uplim,format=fmt),2)

line = '=================================================='
trmt = [line, $
        'Timestamp: '+systime(), $
        'Data Slice taken from '+name, $
        'Slice taken along the '+axisname+'axis from ' $
                +strtrim(string(lolim,format=fmt),2)+' to ' $
                +strtrim(string(uplim,format=fmt),2) $
        ]


; Since the dataset is now 1D, the cut axis has to be reset to a scalar and 
; changed into the y-axis while the uncut axis now becomes the x-axis. This is 
; because for a 1D davePtr, the x-axis is expected to hold the independent data.
; And, of course, the dependent and independent data must be set to the sliced 
; data and error.
etc = {yvals:0.5*(lolim+uplim)}
if (axis eq 0) then begin   ; no need to do this if slicing the y-axis
   oTarget->GetProperty, axis2Value=xv,axis2Distribution=xd,axis2Units=xu,axis2Label=xl
   etc = create_struct(etc,'xVals',xv,'xUnits',xu,'xLabel',xl,'xType',xd)
endif

status = modify_dave_pointer(davePtr, qty = zout, err = zeout $ ; modify the davePtr
                                 ,treatment = trmt $
                                 ,ermsg = errMsg $
                                 ,_EXTRA=etc $
                                 )
if (~status) then begin
   oTool->StatusMessage, errMsg
   return, 0
endif

; Create a DaveDataset object
oResult = obj_new('DAVEDataset',davePtr=davePtr, nameTag=nameTag)

; And add it to the Data Manager
oTool->AddDatasetToDataManager, oResult

; Save some useful details for undo/redo
oCmd = oCmdSet->Get(position=0)   ; retrieve the first and only command object created earlier
if (~obj_valid(oCmd)) then return, 0
void = oCmd->AddItem('LOLIM',lolim)
void = oCmd->AddItem('UPLIM',uplim)
void = oCmd->AddItem('WIDTH',width)
void = oCmd->AddItem('TRMT',trmt)
void = oCmd->AddItem('NAMETAG',nameTag)
void = oCmd->AddItem('OUTPUTID',oResult->GetFullIdentifier())

; Force the tool that generated this operation to refresh 
oTool->_SetDirty, 1
oTool->RefreshCurrentWindow

return, 1
end


;===============================================================================
; DAVEopDataSlice::Cleanup
; 
; PURPOSE:
;   DAVEopDataSlice class cleanup
;
pro DAVEopDataSlice::Cleanup
compile_opt idl2

; call base class cleanup
self->IDLitOperation::Cleanup

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataSlice::Init
; 
; PURPOSE:
;   Initialize an object of this class
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - if successful
;    0 - otherwise
;
function DAVEopDataSlice::Init, _REF_EXTRA=etc
compile_opt idl2

; call superclass init
; This operation is not reversible
; This operation is expensive
if (~self->IDLitOperation::Init(NAME='Data Slicer' $
;                                ,types=['DAVE1COMMONSTR','ASCIISPE','ASCIICOL','ASCIIGRP'] $
                                ,reversible_operation=0,expensive_computation=1 $
                                ,_EXTRA=etc)) then return, 0

;self->SetProperty, reversible_operation=0, expensive_computation=1

; Unhide the SHOW_EXECUTION_UI property
self->SetPropertyAttribute, 'SHOW_EXECUTION_UI', hide=0

self->RegisterProperty, 'rangemin', /float, description='Lower limit of data' $
  ,name='Slice Axis Lower Limit',sensitive=0
self->RegisterProperty, 'rangemax', /float, description='Upper limit of data' $
  ,name='Slice Axis Upper Limit',sensitive=0
self->RegisterProperty, 'norm', enumlist=['None','Per unit slicewidth'], description='Normalization' $
  ,name='Normalization',sensitive=1
self->RegisterProperty, 'axis', enumlist=['x-axis','y-axis'], description='Slice Axis' $
  ,name='Slice Axis',sensitive=1
self->RegisterProperty, 'slicecenter', /float, description='Desired Slice Center' $
  ,name='Slice Center',sensitive=1
self->RegisterProperty, 'slicewidth', /float, description='Desired Slice Width' $
  ,name='Slice Width',sensitive=1

self.norm = 0
self.axis = 1

; return success
return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDataSlice__define
; 
; PURPOSE:
;   DAVEopDataSlice class structure definition
;
pro DAVEopDataSlice__define

compile_opt idl2

struc = {DAVEopDataSlice $
         ,inherits IDLitOperation $
         ,norm:0 $               ; normalization flag
         ,axis:0 $               ; slice axis
         ,range:[[0.0,0.0],[0.0,0.0]] $      ; range of data in both axes
         ,sliceCenter:[0.0,0.0] $      ; where to make the slice for both axes
         ,sliceWidth:[0.0,0.0] $       ; thickness of the slice for both axes
         ,oTarget:obj_new() $    ; the current target object.
        }

end
