; $Id$
;###############################################################################
;+
; CLASS_NAME:
;   DAVEopDatasetScale
;
; PURPOSE:
;   A simple operation that multiplies the data of the selected
;   IDLitData object(s) by a scalefactor.
;
; CATEGORY:
;   DAVE Main Tool
;
; SUPERCLASSES:
;   IDLitOperation
;
; METHODS:
;   DoAction
;   DoExecuteUI
;   GetProperty
;   RecordInitialValues
;   RecordFinalValues
;   RedoOperation
;   SetProperty
;   UndoOperation
;
; Richard Tumanjong Azuah
; NIST Center for Neutron Research
; azuah@nist.gov; (301) 9755604
; Mar 2005
;-
;###############################################################################


;===============================================================================
; DAVEopDatasetScale::GetProperty
;
; PURPOSE:
;   Accessor
;
; PARAMETERS:
;
; KEYWORDS:
;   scalefactor [out] - The scalefactor that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDatasetScale::GetProperty, _scalefactor=_scalefactor, _comments=_comments $
  , _ngroups=_ngroups, _scaleFlag=_scaleFlag, _grpscalefactors=_grpscalefactors $
  , _REF_EXTRA=etc
  compile_opt idl2

  if (arg_present(_scalefactor)) then _scalefactor =  self._scalefactor
  if (arg_present(_comments)) then _comments =  self._comments
  if (arg_present(_ngroups)) then _ngroups =  self._ngroups
  if (arg_present(_scaleFlag)) then _scaleFlag =  self._scaleFlag
  if (arg_present(_grpscalefactors)) then _grpscalefactors =  self._grpscalefactors

  ; call base class accessor
  if(n_elements(etc) gt 0) then $
    self->IDLitOperation::GetProperty, _EXTRA=etc

end


;===============================================================================
; DAVEopDatasetScale::SetProperty
;
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;
; KEYWORDS:
;   scalefactor [in] - The scalefactor that should be applied to the data
;
; RETURN VALUE:
;
pro DAVEopDatasetScale::SetProperty, _scalefactor=_scalefactor, _comments=_comments $
  , _ngroups=_ngroups, _scaleFlag=_scaleFlag, _grpscalefactors=_grpscalefactors, noWarn=noWarn $
  , _EXTRA=etc
compile_opt idl2

; The UI dialog should be resposible for checking the integrity of the
; value of scalefactor so will not be repeated here
if (n_elements(_scalefactor) gt 0) then begin
  if (_scalefactor eq 0.0) then begin
    status = Self->PromptUserYesNo('Really set scalefactor to 0?', answer, Title='Scalefactor Value')
    if (status ne 0 && answer eq 1) then self._scalefactor = _scalefactor
  endif else self._scalefactor = _scalefactor
endif

if (n_elements(_ngroups) gt 0) then self._ngroups = _ngroups
if (n_elements(_comments) gt 0) then self._comments = _comments
if (n_elements(_scaleFlag) gt 0) then begin
  Self._scaleFlag = _scaleFlag
  ;Self->SetPropertyAttribute, '_ngroups', hide=(nDim eq 1), sensitive=0
  Self->SetPropertyAttribute, '_scalefactor', hide=(Self._scaleFlag eq 1)
  Self->SetPropertyAttribute, '_grpscalefactors', hide=(Self._scaleFlag eq 0)
  Self->SetPropertyAttribute, '_comments', hide=(Self._scaleFlag eq 0), sensitive=0
  Self._comments = (Self._scaleFlag eq 0)? 'Specify scalefactor for dataset' $
    : 'Specify scalefactors for each group separated by a comma'
endif

if (n_elements(_grpscalefactors) gt 0) then begin
  if (~keyword_set(noWarn)) then noWarn=0
  ; check the entry first
  ng = Self._ngroups
  sf_in = float(strsplit(_grpscalefactors,/extract,count=ns))
  
  if (ns ne ng) then begin
    if (ns lt ng) then begin
      msg = 'You have specified scalefactors for '+strtrim(string(ns),2)+' groups'
      msg = [msg,'But there are '+strtrim(string(Self._ngroups),2)+' groups present in dataset']
      msg = [msg,'Missing scalefactors have been set to a value of 1.0!']
      msg = [msg,'Please correct as necessary - use space to delimit values for each group']
      if (noWarn eq 0) then Self->ErrorMessage,msg,severity=0,title='Group Scalefactors'
      sf = replicate(1.0,ng)
      sf[0:ns-1] = sf_in
    endif else if (ns gt ng) then begin
      sf = sf_in[0:ng-1]
    endif
  endif else begin
    sf = sf_in    
  endelse
  
  Self._grpscalefactors = strjoin(strtrim(string(float(sf)),2),' ',/single)
endif

; Call base class mutator
if(n_elements(etc) gt 0) then $
  self->IDLitOperation::SetProperty, _EXTRA=etc

end



;===============================================================================
; DAVEopDatasetScale::RecordInitialValues
;
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDatasetScale::RecordInitialValues, oCmdSet, oTarget, idProp
  compile_opt idl2

  ; create a command object to store the values
  oCmd = obj_new('IDLitCommand',target_identifier=oTarget->getfullidentifier())
  if (~obj_valid(oCmd)) then return, 0

;  ; Get the value to be stored and add to command obj
;  oTarget->GetProperty, scalefactor=scalefactor
;  void = oCmd->AddItem('OLD_SCALEFACTOR',scalefactor)

  ; Add the command to command set
  oCmdSet->Add, oCmd

  return, 1

end


;===============================================================================
; DAVEopDatasetScale::RecordFinalValues
;
; PURPOSE:
;   Mutator
;
; PARAMETERS:
;   oCmdSet [in|out] - The command set obj in which to make recordings
;
;   oTarget [in] - The object whose props are being altered (self in
;                  this case)
;
;   idProp - not used
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDatasetScale::RecordFinalValues, oCmdSet, oTarget, idProp
  compile_opt idl2

  ; Retrieve the first command object from the command set
  oCmd = oCmdSet->Get(position=0)
  if (~obj_valid(oCmd)) then return, 0

;  ; Get the value to be stored and add to command obj
;  oTarget->GetProperty, scalefactor=scalefactor
;  void = oCmd->AddItem('NEW_SCALEFACTOR',scalefactor)

  return, 1

end


;===============================================================================
; DAVEopDatasetScale::DoExecuteUI
;
; PURPOSE:
;   Launch the UI dialog to collect appropriate user information for
;   this operation.
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDatasetScale::DoExecuteUI
  compile_opt idl2

  ; Get the tool
  oTool = self->GetTool()
  if (~obj_valid(oTool)) then return, 0

  ; Use the build-in 'PropertySheet' UI service to let the user
  ; customize the operation's property.
  return, oTool->DoUIService('PropertySheet',self)

end


;===============================================================================
; DAVEopDatasetScale::UndoOperation
;
; PURPOSE:
;   Provides the 'undo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDatasetScale::UndoOperation, oCmdSet
compile_opt idl2

; Retrieve the command object (there is only one for this operation)
; from the command set
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0

; Loop through rest of commands and undo the changes that were made
for i=0,nCmds-1 do begin
  status = oCmds[i]->GetItem('ID_DATA', idData)
  oData = oTool->GetByIdentifier(idData)
  if (~obj_valid(oData)) then begin
    oCmdSet->Remove, oCmds[i] ; This command obviously has a problem so delete it!
    oTool->StatusMessage,'Missing dataset! Undo could not be completed'
    continue
  endif
  status = oCmds[i]->GetItem('SCALEFLAG', scaleFlag)
  status = oCmds[i]->GetItem('SCALEFACTOR', scaleFactor)
  nsf = n_elements(scalefactor)
  void = oData->GetData(data)
  if (nsf gt 1) then begin
    unitx = replicate(1.0,(size(data))[1])
    scalefactor = unitx#scalefactor         ; convert to 2D to avoid for loops in group-dependent scaling
  endif
  void = oData->setData(data/scaleFactor,/no_copy,/no_notify)   ; modify but do not notify observers
  status = oCmds[i]->GetItem('ERREXIST', errExist)
  if (errExist) then begin
    status = oCmds[i]->GetItem('ID_ERR', idErr)
    oErr = oTool->GetByIdentifier(idErr)
    if (obj_valid(oErr)) then  begin
      void = oErr->GetData(data)
      void = oErr->setData(data/scaleFactor,/no_copy,/no_notify)
    endif
  endif
  status = oCmds[i]->GetItem('TRMTEXIST', trmtExist)
  if (trmtExist) then begin
    status = oCmds[i]->GetItem('ID_TREATMENT', idTrmt)
    oTrmt = oTool->GetByIdentifier(idTrmt)
    if (obj_valid(oTrmt)) then begin
      status = oCmds[i]->GetItem('OLD_TREATMENT', oldTrmt)
      void = oTrmt->SetData(oldTrmt)
    endif
  endif
  ; Notify Observers
  oData->notifyDataChange
  oData->notifyDataComplete
  if (errExist) then begin
    oErr->notifyDataChange
    oErr->notifyDataComplete    
  endif
  
endfor

return, 1

end


;===============================================================================
; DAVEopDatasetScale::RedoOperation
;
; PURPOSE:
;   Provides the 'redo' functionality for this operation
;
; PARAMETERS:
;   oCmdSet [in] - The command set obj in which to make recordings
;
; KEYWORDS:
;
; RETURN VALUE:
;   1 - success
;   0 - failure
;
function DAVEopDatasetScale::RedoOperation, oCmdSet
compile_opt idl2

; Retrieve the command object (there is only one for this operation)
; from the command set
oCmds = oCmdSet->Get(/all,count=nCmds)
if (nCmds lt 1) then return, 0

; Get the tool
oTool = self->GetTool()
if (~obj_valid(oTool)) then return, 0
  
; Loop through the commands and redo the changes that were made
for i=0,nCmds-1 do begin
  status = oCmds[i]->GetItem('ID_DATA', idData)
  oData = oTool->GetByIdentifier(idData)
  if (~obj_valid(oData)) then begin
    oCmdSet->Remove, oCmds[i] ; This command obviously has a problem so delete it!
    oTool->StatusMessage,'Missing dataset! Redo could not be completed'
    continue
  endif
  status = oCmds[i]->GetItem('SCALEFLAG', scaleFlag)
  status = oCmds[i]->GetItem('SCALEFACTOR', scaleFactor)
  nsf = n_elements(scalefactor)
  void = oData->GetData(data)
  if (nsf gt 1) then begin
    unitx = replicate(1.0,(size(data))[1])
    scalefactor = unitx#scalefactor         ; convert to 2D to avoid for loops in group-dependent scaling
  endif
  void = oData->setData(data * scaleFactor,/no_copy,/no_notify)   ; modify but do not notify obervers
  status = oCmds[i]->GetItem('ERREXIST', errExist)
  if (errExist) then begin
    status = oCmds[i]->GetItem('ID_ERR', idErr)
    oErr = oTool->GetByIdentifier(idErr)
    if (obj_valid(oErr)) then  begin
      void = oErr->GetData(data)
      void = oErr->setData(data * scaleFactor,/no_copy,/no_notify)
    endif
  endif
  status = oCmds[i]->GetItem('TRMTEXIST', trmtExist)
  if (trmtExist) then begin
    status = oCmds[i]->GetItem('ID_TREATMENT', idTrmt)
    oTrmt = oTool->GetByIdentifier(idTrmt)
    if (obj_valid(oTrmt)) then begin
      status = oCmds[i]->GetItem('NEW_TREATMENT', newTrmt)
      void = oTrmt->SetData(newTrmt)
    endif
  endif
  ; Notify Observers
  oData->notifyDataChange
  oData->notifyDataComplete
  if (errExist) then begin
    oErr->notifyDataChange
    oErr->notifyDataComplete
  endif
endfor

return, 1

end


;===============================================================================
; DAVEopDatasetScale::DoAction
;
; PURPOSE:
;   Implements the main function for this operation. Apply the
;   operation's scalefactor to the data being operated on.
;
; PARAMETERS:
;   oTool [in] - the object reference of the tool from which the
;                operation was launched.
;
; KEYWORDS:
;
; RETURN VALUE:
;    If successful, an IDLitCommandSet object
;    If unsuccessful, a NULL object.
;
function DAVEopDatasetScale::DoAction, oTool
compile_opt idl2

; oTool should be valid and the DAVE Main Tool
if (~obj_valid(oTool) || ~obj_isa(oTool,'DAVETOOL')) then return, obj_new()

; Get the selected dataset(s)
oSelections = oTool->GetSelectedData()
void = where(obj_valid(oSelections),cnt)
if (cnt eq 0) then begin
    oTool->StatusMessage, 'No valid data to operate on! Select a dataset from the Data Browser tree.'
    return, obj_new()
endif

; Locate valid dataset containers that can be scaled
self->GetProperty, types=validTypes ; should be ['DAVE1COMMONSTR','ASCIISPE','ASCIIGRP','ASCIICOL']
nValid = n_elements(validTypes)
oSel = []
for i = 0,n_elements(oSelections)-1 do begin
   ;; search for one of the valid types from this selction
   ;; Valid types are containers with at least one independent and dependent data components
   j = 0
   repeat begin
      oRes = oSelections[i]->GetByType(validTypes[j],count=found)
   endrep until (found || ++j ge nValid) 
   
   if (~obj_valid(oRes)) then continue
   oSel = (n_elements(oSel) gt 0)? [oSel,oSelections[i]] : oSelections[i]
endfor
nSel = n_elements(oSel)
if (nSel eq 0) then begin
    oTool->StatusMessage, 'Invalid data selection(s)! Please select dataset folders for scaling'
    return, obj_new()
endif

; Create a command set obj by calling the base class DoAction
oCmdSet = self->IDLitOperation::DoAction(oTool)

; Is some UI needed prior to execution?
self->GetProperty, show_execution_ui=doUI
hasPropSet = 0b

;; Record initial value for the scalefactor property of this operation.
if (~self->RecordInitialValues(oCmdSet,self,'')) then begin
  obj_destroy, oCmdSet
  return, obj_new()
endif

; loop through selected datasets and perform scaling
for i = 0,nSel-1 do begin
  oTarget = oSel[i]
  
  ;; Update properties based on selected dataset
  oTarget->GetProperty, nDimensions=nDim, axis2Value=yValues
  Self._nGroups = (nDim eq 1)? 1 : n_elements(yValues)
  Self._scaleFlag = (nDim eq 1)? 0 : Self._scaleFlag
  Self._comments = (Self._scaleFlag eq 0)? 'Specify scalefactor for dataset' $
    : 'Specify scalefactors (space delimited) for each group'
  Self->SetPropertyAttribute, '_scaleFlag', hide=(nDim eq 1), sensitive=1
  Self->SetPropertyAttribute, '_ngroups', hide=(nDim eq 1), sensitive=0
  Self->SetPropertyAttribute, '_scalefactor', hide=(Self._scaleFlag eq 1), sensitive=1
  Self->SetPropertyAttribute, '_grpscalefactors', hide=(Self._scaleFlag eq 0), sensitive=1
  Self->SetPropertyAttribute, '_comments', hide=(Self._scaleFlag eq 0), sensitive=0
  
  if (nDim eq 2) then begin
    sf = Self._grpscalefactors
    Self->SetProperty, _grpscalefactors=sf, /noWarn  ; simply using SetProperty to perform checks for current target dataset
  endif
     
  if (doUI) then begin
    ;; Perform our UI.
    if (~self->DoExecuteUI()) then begin
      ;obj_destroy, oCmdSet
      ;return, obj_new()
      continue
    endif
  endif

  ;; Perform the scalling
  if (i eq 0) then begin
    ; Retrieve the first command object from the command set (created by RecordInitialValues())
    oCmd = oCmdSet->Get(position=0)
    if (~obj_valid(oCmd)) then return, 0
  endif else begin
    ; create a command to store info about the scaling
    oCmd = obj_new("IDLitCommand", TARGET_IDENTIFIER=oTarget->GetFullIdentifier())
    oCmdSet->Add, oCmd    ; Add the command to command set
  endelse
  void = oCmd->AddItem('SCALEFLAG',Self._scaleFlag)

  oTarget->GetProperty, dataRef=oData, dataValue=data, errorRef=oErr, errorValue=error, trmtRef = oTrmt
  errExist = (n_elements(error) gt 0)
  trmtExist = obj_valid(oTrmt)
  void = oCmd->AddItem('TRMTEXIST',trmtExist)
  scalefactor = (Self._scaleFlag eq 1)? float(strsplit(Self._grpscalefactors,/extract,count=nsf)) $
    : Self._scalefactor
  sfString = (Self._scaleFlag eq 1)? Self._grpscalefactors : string(scalefactor)
  trmtText = (Self._scaleFlag eq 0)? 'Scalefactor applied: ' : 'Group-dependent Scalefactors applied: '

  void = oCmd->AddItem('ID_DATA',oData->GetFullIdentifier())
  void = oCmd->AddItem('ERREXIST',errExist)
  errExist = (n_elements(error) gt 0)
  if (errExist) then void = oCmd->AddItem('ID_ERR',oErr->GetFullIdentifier())
  void = oCmd->AddItem('SCALEFACTOR',scaleFactor)
  nsf = n_elements(scalefactor)
  if (nsf gt 1) then begin
    unitx = replicate(1.0,(size(data))[1])
    scalefactor = unitx#scalefactor         ; convert to 2D to avoid for loops in group-dependent scaling
  endif
  void = oData->setData(data * scaleFactor,/no_copy,/no_notify)   ; modify but do not notify obervers
  if (errExist) then void = oErr->SetData(error * scaleFactor,/no_copy,/no_notify)    


  if (trmtExist) then begin
    void = oTrmt->GetData(trmt)
    line = '____________________________________________________'
    ;; modify treatment info accordingly
    newTrmt = [trmt,line,'Timestamp: '+systime(), 'Perform a scaling of the dataset']
    newTrmt = [newTrmt,trmtText+sfString]

    void = oCmd->AddItem('ID_TREATMENT',oTrmt->GetFullIdentifier())
    void = oCmd->AddItem('OLD_TREATMENT',trmt) ; treatment info
    void = oCmd->AddItem('NEW_TREATMENT',newTrmt)
  endif
  
  ; Notify Observers
  oData->notifyDataChange
  oData->notifyDataComplete
  if (errExist) then begin
    oErr->notifyDataChange
    oErr->notifyDataComplete    
  endif
endfor

;; Record final value for the scalefactor property of this operation.
if (~self->RecordFinalValues(oCmdSet,self,'')) then begin
  obj_destroy, oCmdSet
  return, obj_new()
endif

; return the command set obj
return, oCmdSet

end


;===============================================================================
; DAVEopDatasetScale::Cleanup
;
; PURPOSE:
;   DAVEopDatasetScale class cleanup
;
pro DAVEopDatasetScale::Cleanup

  compile_opt idl2

  ; call base class cleanup
  self->IDLitOperation::Cleanup

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDatasetScale::Init
;
; PURPOSE:
;   Initialize an object of this class
;
; PARAMETERS:
;
; KEYWORDS:
;
; RETURN VALUE:
;    1 - if successful
;    0 - otherwise
;
function DAVEopDatasetScale::Init, _REF_EXTRA=etc
compile_opt idl2

; call superclass init
if (~self->IDLitOperation::Init(NAME='Scalefactor' $
  ,_EXTRA=etc)) then return, 0
;types=['IDLVECTOR','IDLARRAY2D','IDLARRAY3D','IDLIMAGE'] $


; Unhide the SHOW_EXECUTION_UI property
self->SetPropertyAttribute, 'SHOW_EXECUTION_UI', hide=0

; This operation is reversible
; This operation is not expensive
self->SetProperty, reversible_operation=1, expensive_operation=0

; Register an scalefactor property for this operation
Self->RegisterProperty, '_ngroups', /integer, description='Nos of Groups in Dataset' $
  ,name='Nos of Groups in dataset',sensitive=0, hide=1

Self->RegisterProperty, '_scaleFlag', enum=['No','Yes'], description='Dataset Scalefactor' $
  ,name='Group-Dependent Scalefactors?',sensitive=1, hide=1

Self->RegisterProperty, '_scalefactor', /float, description='Dataset Scalefactor' $
  ,name='Scalefactor',sensitive=1, hide=1

Self->RegisterProperty, '_grpscalefactors', /string, description='Dataset Scalefactors' $
  ,name='Scalefactors per Group ',sensitive=1, hide=1

Self->RegisterProperty, '_comments', /string, description='Notes' $
  ,name='Comments',sensitive=0, hide=0

; init scalefactor to 1.0
Self._scalefactor = 1.0
Self._grpscalefactors = '1.00'
Self._scaleFlag = 0
Self._comments = 'Specify scalefactors (space delimited) for each group'

; return success
return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
; DAVEopDatasetScale__define
;
; PURPOSE:
;   DAVEopDatasetScale class structure definition
;
pro DAVEopDatasetScale__define

  compile_opt idl2

  struc = {DAVEopDatasetScale   $
    ,inherits IDLitOperation    $
    ,_comments:''                  $ ; useful help notes
    ,_scaleFlag:0               $ ; 0 => single scalefactor, 1 => group-dependent scalefactors
    ,_ngroups:0                 $ ; number of groups present in dataset
    ,_scalefactor:1.0           $ ; scale factor to be applied to 1D dataset
    ,_grpscalefactors:''        $ ; scale factors to be applied to 2D dataset
  }

end
