;$Id$
;###############################################################################
;
; NAME:
;  SCAT_CONVOLVE
;
; PURPOSE:
;  Convolution function for use in fitting data in RAINS.
;
; CATEGORY:
;  DAVE, Data Analysis, RAINS, surface fitting
;
; AUTHOR:
;   Robert M. Dimeo, Ph.D.
;   NIST Center for Neutron Research
;   100 Bureau Drive
;   Gaithersburg, MD 20899
;   Phone: (301) 975-8135
;   E-mail: robert.dimeo@nist.gov
;   http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;+
; NAME:
;       SCAT_CONVOLVE
;
; PURPOSE:
;
;		This function performs the numerical convolution of an image (model)
;		with a point spread (resolution) function.  The x and y parameters
;		passed into the function are used for integration scaling and the
;		functions IMAGE and PSF must have the same size and be equally spaced.
;
; AUTHOR:
;
;       Robert M. Dimeo, Ph.D.
;		NIST Center for Neutron Research
;       100 Bureau Drive
;		Gaithersburg, MD 20899
;       Phone: (301) 975-8135
;       E-mail: robert.dimeo@nist.gov
;       http://www.ncnr.nist.gov/staff/dimeo
;
; CATEGORY:
;
;       Mathematics, DAVE_REFINE application, DAVE project
;
; CALLING SEQUENCE:
;
;		ycon = SCAT_CONVOLVE(x,y,image,psf)
;
; INPUT PARAMETERS (REQUIRED)
;
;		X:	array (float) of values describing the first dimension of
;			PSF and IMAGE
;		Y:	array (float) of values describing the second dimension of
;			PSF and IMAGE
;		IMAGE:	2-d array of floats describing the model function
;		PSF:	2-d array of floats describing the resolution or
;				'point-spread' function.  Assumed to be normalized.
;
; COMMON BLOCKS:
;
;		None
;
; DEPENDENCIES:
;
;		None
;
; REQUIREMENTS:
;
;	Written using IDL 5.6.  I cannot ensure that it will work with previous
;	releases.
;
;

; MODIFICATION HISTORY:
;
;       Written by Rob Dimeo, April 14, 2003
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function scat_convolve,x,y,image,psf
; This function returns the convolution integral of image and psf, both
; assumed to be rectangular, equally-spaced arrays.  The x-dimension is
; blurred but the y dimension is not (y-resolution is assumed to be a
; Dirac delta function).

sim = size(image)
nx = sim[1] & ny = sim[2]
dx = (moment(x[1:nx-1]-x[0:nx-2]))[0]
xhi = max(x,min = xlo)
nshift = fix(0.5*(abs(xhi)-abs(xlo))/dx)
shifted_image = shift(image,nshift,0)
ycon = fltarr(nx,ny)
for i = 0,ny-1 do begin
	fun = reform(shifted_image[*,i])
	res = reform(psf[*,i])
	sfun = size(fun) & sc = sfun/2 & npix = n_elements(fun)
	funFt = fft(fun,-1) & resFt = fft(res,-1)
	ycon_local = npix*float(fft(funFt*resFt,1))
	sc = sc+(sfun mod 2)
	ycon[*,i] = shift(ycon_local,sc[1])
endfor
return,ycon*dx
end