; $Id$
;###############################################################################
;
; NAME:
;  Pan_voigt
;
; PURPOSE:
;  Voigt function. Convolution of a Gaussian function and a Lorentzian function
;  See ref in http://www.originlab.com/doc/Origin-Help/Voigt-FitFunc
;
; CATEGORY:
;  DAVE, Data Analysis, PAN
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;  March 2017
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function pan_voigt, x, parms,$
                    parmnames = parmnames, $
                    canDraw = canDraw, $
                    drawMessage = drawMessage, $
                    initParms = initParms, $
                    func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, $
                    changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, xrange=xrange, $
                    _Extra = extra
canDraw = 1
twoDimFlag = 0

if N_params() eq 0 then begin
  parmnames = ['Area','Center','Lorentzian FWHM','Gaussian FWHM']
  initParms = [1.0,0.0,1.0,1.0]
  Return,-1
endif

drawMessage = Strarr(4)
drawMessage[0:1] = ['Hold left mouse button down','and drag function centroid']
drawMessage[2:3] = ['Hold left mouse button down','and drag to change width']

if (Keyword_set(changefirst)) then begin
  ; user is initializing the function graphically
  ; this block of code is executed when the mouse is first release at the location for the function centroid
  ; translate the position into suitable parameters.
  ;Print,'###Changefirst called within function...'
  func_dataHash['ymax'] = yMouseClick
  fwhm = 0.05*(xrange[1]-xrange[0])
  ;sig = fwhm/Sqrt(8.0*Alog(2.0))
  wL = 0.5*fwhm
  wG = 0.5*fwhm
  parms[0] = yMouseClick * (!pi^(1.5) * wG^2) / (2 * Alog(2) *wL)
  parms[1] = xMouseClick
  parms[2] = wL
  parms[3] = wG
endif

if (Keyword_set(changesecond)) then begin
  ; user is initializing the function graphically
  ; this block of code is executed when the mouse is move to define the function width
  ; translate the position into suitable parameters.
  ;Print,'###Changesecond called within function...'
  fwhm = 2.0*Abs(xMouseClick-parms[1])
  wL = 0.5*fwhm
  wG = 0.5*fwhm
  ;sig = fwhm/Sqrt(8.0*Alog(2.0))
  ;amp = parms[0]/Sqrt(2.0*!dpi*sig^2)
  parms[0] = func_dataHash['ymax']* (!pi^(1.5) * wG^2) / (2 * Alog(2) *wL)
  parms[2] = wL
  parms[3] = wG
endif

wG = parms[3]
wL = parms[2]
area = parms[0]
cen = parms[1]

ln2 = Alog(2)
sqrtln2 = Sqrt(ln2)

if (wL le 0.0) then begin
  ;; it is a Gaussian
  yout = area * 2.0 * Sqrt(ln2 / !pi)/wG * Exp(-(2*sqrtln2*(x - cen)/wG)^2)
endif else if (wG le 0.0) then begin
  ;; it is a Lorentzian
  yout = area * 2.0 / !pi * wL / (4*(x - cen)^2 + wL^2)
endif else begin
  ;; the convolution integral
  n = N_elements(x)
  yout=Fltarr(n)
  sf = (area * 2 * ln2 *wL) / (!pi^(1.5) * wG^2)
  inf = !values.d_infinity
  for i=0,n-1 do begin
    x_xc = x[i] - cen
    etc = {x_xc:x_xc,wl:wL,wg:wG,sqrtln2:sqrtln2}
    yout[i] = sf*Qpint1d('TheVoigtIntegrand',-inf,+inf,functargs=etc,status=status,nfev=nfev)
  endfor
endelse

Return,yout

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;###############################################################################
function TheVoigtIntegrand, y, x_xc=x_xc, wL=wL, wG=wG, sqrtln2=sqrtln2
if (n_elements(sqrtln2) eq 0) then sqrtln2 = sqrt(alog(2))
a = sqrtln2*wL/wG
w = 2.0 * sqrtln2 * x_xc/wG
return, exp(-y^2)/(a^2 + (w - y)^2)

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;###############################################################################
pro test_pan_voigt

p1 = [1.0,0.0,0.0,2*Sqrt(2*Alog(2))*1.53]
p2 = [1.0,0.0,1.0,2*Sqrt(2*Alog(2))*1.30]
p3 = [1.0,0.0,2.0,2*Sqrt(2*Alog(2))*1.0]
p4 = [1.0,0.0,3.6,2*Sqrt(2*Alog(2))*0.0]

x = findgen(201)*0.1 - 10.0

y1 = Pan_voigt(x,p1)
y2 = Pan_voigt(x,p2)
y3 = Pan_voigt(x,p3)
y4 = Pan_voigt(x,p4)

p1 = Plot(x,y1,thick=5.0)
p1 = Plot(x,y2,color='b',thick=5.0,/overplot)
p1 = Plot(x,y3,color='g',thick=5.0,/overplot)
p1 = Plot(x,y4,color='r',thick=5.0,/overplot)

end