; $Id$
;###############################################################################
;
; NAME:
;  PAN_DiffusionInASphere_2D
;
; PURPOSE:
;  Implementation of a diffusion-in-a-sphere model for use in fitting
;  data in PAN. Ref: F. Volino and A.J. Dianoux, Molecular Physics 41, 271-279 (1980)
;
; CATEGORY:
;  DAVE, Data Analysis, surface fitting
;
; AUTHOR:
;  Richard Tumanjong Azuah
;  NIST Center for Neutron Research
;  azuah@nist.gov; (301) 9755604
;
;  Robert M. Dimeo, Ph.D.
;  NIST Center for Neutron Research
;  Phone: (301) 975-8135
;  E-mail: robert.dimeo@nist.gov
;  http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
function PAN_DiffusionInASphere_2D,x,parms,                       $
  parmnames=parmnames, single_parmnames=single_parmnames, multi_parmnames=multi_parmnames,    $
  canDraw=canDraw, drawMessage=drawMessage, xrange=xrange, xunits=xunits, $
  func_dataHash=func_dataHash,twoDimFlag=twoDimFlag, initParms=initParms, $
  qvals=qvals,xVals=xVals,yVals=yVals,groupNumber=groupNumber,qGroup=qGroup, $
  changesecond=changesecond, changefirst=changefirst, xMouseClick=xMouseClick, yMouseClick=yMouseClick, $
  resPtr=resPtr, resLimit=rLimit,resolutionRequiredFlag=resolutionRequiredFlag, extConvolFlag=extConvolFlag, $
  _Extra = extra

nx = isa(x)? n_elements(x) : n_elements(xvals)
nq = n_elements(qvals)
ux = 1+bytarr(nx)
uq = 1+bytarr(nq)

canDraw = 0
twoDimFlag = 1
resolutionRequiredFlag = 1  ; does function require a resolution function to be defined 0=>No, 1=>Yes

extConvolFlag = 0           ; does function require external convolution? 0=>No, 1=>Yes
                            ; if set to No, it implies a convolution is performed within this function when a resolution function is present
                            ; Also set to No, if you DO NOT wish anq external convolution to be performed at all 


if (n_params() eq 0) or (n_params() eq 1) then begin

  ; retrieve values for Xnl and l.
  ; These are stored in the disk file 'xnl.txt' as described in this 
  ; Ref: F. Volino and A.J. Dianoux, Molecular Physics 41, 271-279 (1980)
  xnlFilename = !DAVE_AUXILIARY_DIR+'xnl.txt'
  if (~file_test(xnlFilename,/read)) then return, 0
  nlines = file_lines(xnlFilename)
  buf = dblarr(2,nlines)
  openr,lun,xnlFilename,/get_lun
  readf,lun,buf
  free_lun,lun,/force
  l = reform(fix(buf[0,*]))
  xnl = reform(buf[1,*])
  if (Isa(func_dataHash,'hash')) then begin
    func_dataHash['xnl'] = xnl
    func_dataHash['l'] = l
  endif
  
  ; Define hbar according to the energy units
  eUnits = xunits.Toupper()
  eUnitsKnown = eUnits.Equals('MEV') || eUnits.Equals('UEV')
  if (eUnitsKnown) then begin
    if (eUnits.Equals('UEV')) then hbar = 658.21        ; ueV.ps
    if (eUnits.Equals('MEV')) then hbar = 0.65821       ; meV.ns
  endif else begin
    ; Unknown energy units so let user specify
    msg = 'The unit of energy for the data could not be determined!'
    msg = [msg,'Is the unit of the energy axis in ueV or meV?',' ']
    msg = [msg,'Click Yes if energy unit is ueV',' ']
    msg = [msg,'Or No if energy unit is meV']
    title = 'Diffussion in A Sphere 2D -- Specify Energy Unit'
    if (N_elements(wTLB) eq 0) then wTLB=0L
    decision = Dialog_message(msg,title=title,/question,dialog_parent=wTLB)
    xunits = (decision.Equals('Yes'))? 'ueV' : 'meV'
    hbar = (decision.Equals('Yes'))? 658.21 : 0.65821
    eUnits = xunits.Toupper()
  endelse
  ;if strupcase(xunits) eq 'MEV' then hbar = hbar*1.d4 else hbar = hbar*1.d7
  if (Isa(func_dataHash,'hash')) then func_dataHash['hbar'] = hbar

  tUnits = eUnits.Equals('UEV')? 'ps' : 'ns'     ; time unit depends on energy unit of data

  parmnames = ['Sphere radius (Ang)','Diffusivity (Ang^2/'+tUnits+')', $
    'Intensity'+Strtrim(String(Indgen(nq)),2), $
    'Center'+Strtrim(String(Indgen(nq)),2)]

  single_parmnames = parmnames[0:1]
  multi_parmnames = ['Intensity','Center']

	if arg_present(initParms) then begin
    intensity = max(yVals,dimension = 1,index)
    index = index - indgen(nq)*nx ; index contains the 1D subscripts so need to reform
    center = xvals[index]

    ; assuming a single main peak then estimate HWHM for the _largest_ Q value
    hm = 0.5*intensity[nq-1]
    res = Where(yVals[*,nq-1] ge hm, found)
    if (found gt 0) then begin
      Gamma = 0.5*xvals[res[found-1]] - xvals[res[0]]
    endif

		D = Gamma/(hbar*qVals[nq-1]^2)        ; from defn of Gamma in units of Ang^2/ps or Ang^2/ns
		a = 2.0                               ; in Ang units
		initParms = [a,D,intensity,center]
	endif
	return,-1
endif

if (~Isa(resPtr,'Pointer')) then Return, Fltarr(nx,nq)*0.0 ; can't proceed without a pointer to the resolution function
yout = Fltarr(nx,nq)
yres = (*resPtr).data
if (yres.ndim ne 2) then Return, yout   ; bail if resolution is not 2D

hbar = 658.0  ; ueV.ps; assumes energy units in microvolts! Should have been defined already
if (Isa(func_dataHash,'hash')) then begin
  hbar = func_dataHash['hbar']
  xnl  = func_dataHash['xnl']
  l    = func_dataHash['l']
endif

a = parms[0]
D = parms[1]
ntable = n_elements(xnl)
index = indgen(nq) + 2
intensity = ux#(parms[index])
center = ux#(parms[index+nq])

; Now calculate the EISF and delta term
A0 = ((3.0*sph_bessel(qVals*a,1))/qVals*a)^2
pseudo_delta = Fltarr(2,nq)             ; delta-function parameters
pseudo_delta[0,*] = A0                  ; delta area
pseudo_delta[1,*] = parms[index+nq]     ; delta position

; Now determine the second term which is an accumulated sum
; over xnl,l of the product of the Anl form factor and a Lorentzian
sumAnlLor = 0.0
xuq = x#uq
for i = 1,ntable-1 do begin

  ; Anl is defined differently when Q*a == Xnl than when it is not
  ; The default case is when Q*a != Xnl
  ; then re-evaluate calc whenever Q*a == Xnl
  first_term = 6.0*xnl[i]^2/((xnl[i])^2-l[i]*(l[i]+1.0))
  second_term_num = qVals*a*Sph_bessel(qVals*a,l[i]+1) - l[i]*Sph_bessel(qVals*a,l[i])
  second_term_den = (qVals*a)^2 - (xnl[i])^2
  Anli = first_term * (second_term_num/second_term_den)
  indexEqual = where(qVals*a eq xnl[i], nEqual)

  if (nEqual gt 0) then begin
    Anli[indexEqual] = 1.5*Sph_bessel(xnl[i],l[i])*((xnl[i])^2-l[i]*(l[i]+1.0))/(xnl[i])^2
  endif

  prefactor = (2*l[i] + 1)/!dpi

  gama = (xnl[i]/a)^2*hbar*D
  Lori = gama/(gama^2 + (xuq - center)^2)
  
  ; accumulate the sum
  sumAnlLor = sumAnlLor + prefactor*(ux#Anli)*Lori


;	Anl = 1.5*((sph_bessel(xnl[i],l[i])*((xnl[i])^2-l[i]*(l[i]+1.0))/(xnl[i])^2
;	ok = where(qVals*a ne xnl[i],count_ok)
;
;	if count_ok gt 0 then begin
;		first_term = (6.0*xnl[i]^2)/((xnl[i])^2-l[i]*(l[i]+1.0))
;		second_term_num = (y[ok]*a*sph_bessel(y[ok]*a,l[i]+1)-l[i]*sph_bessel(y[ok]*a,l[i]))
;		second_term_den = (y[ok]*a)^2-(xnl[i])^2
;		second_term = second_term_num/second_term_den
;		Anl[ok] = (second_term)
;	endif
;
;	num_arg = ux#((xnl[i]^2)*hbar*D/a^2)#uq
;	sum = sum + (2.0*l[i]+1)*(ux#Anl)*num_arg/$
;		(num_arg^2+(x#uq-center)^2)
endfor

; Perform convolution with resolution
; both the first (delta) and 2nd terms are convoluted and summed
yout = intensity*Pan_convolute(x,resPtr,sumAnlLor,pseudo_delta = pseudo_delta)

return,yout
end
