;===============================================================================
;+
; SPINSCalib_EPSDE::Plot
;
; PURPOSE:
;   Plot current results
;
; PARAMETERS:
;
; KEYWORDS:
; 
;   
;-
pro SPINSCalib_EPSDE::Plot, _REF_EXTRA=etc
compile_opt idl2

Self->GetProperty, xvalues=x, yvalues=y, evalues=e, initparam=xp0,bestparam=xp1
status = Self->Model(xp0,calc=c0)
status = Self->Model(xp1,calc=c1)

p = errorplot(x,y,e,sym_color="green",symbol="square",linestyle=6)
p0 = plot(x,c0,"r-2",/overplot)
p1 = plot(x,c1,"b-3",/overplot)

Self->GetProperty, bestfvalue=bestfValue, generationCnt=genCnt, idealparam=xpi
print,'Initial parameters: ',xp0
print,'Best fit parameters: ',xp1
print,'Ideal fit parameters: ',xpi
print,'%tage dfference: ', 100.0*abs((xp1-xpi)/xpi)
print,'Nof of Generations reached = ',genCnt,'   Chisq = ',bestfValue
end
;-------------------------------------------------------------------------------


;===============================================================================
;+
; SPINSCalib_EPSDE::Evaluate 
;
; PURPOSE:
;   Get Property
;
; PARAMETERS:
;
; KEYWORDS:
; 
;   params - function parameters
;   
;   modelFlag  - specify which model to use
;   
;-
function SPINSCalib_EPSDE::Model, params, modelFlag=modelFlag, calc=calc, _REF_EXTRA=etc
compile_opt idl2
returnValue = 0

if (n_elements(modelFlag) eq 0) then modelFlag=Self.modelFlag

switch modelFlag of
   1: begin
      ; Single Gaussian + bkgd constant

      ; 
      ;               y = bc + A*exp( -0.5((x - B)/C)^2) 
      ;
      ; params[0] = Imax, params[1] = Center, params[2] = sigma, params[4] = bkgd constant
      ;

      xData = (*Self.xDataPtr)
      
      (*Self.calcPtr) =  params[0]*Exp(-0.5*((xData - params[1])/params[2])^2) + params[3]

      returnValue = 1
      Self.nFE++  ; update the function evaluation count counter
      if (arg_present(calc)) then calc = (*Self.calcPtr)

   end


   else:
endswitch


return, returnValue

end
;-------------------------------------------------------------------------------


;===============================================================================
;+
; SPINSCalib_EPSDE::Evaluate 
;
; PURPOSE:
;   Get Property
;
; PARAMETERS:
;
; KEYWORDS:
; 
;   params - function parameters
;   
;   errFuncFlag  - specify which error function calculation to use
;   
;-
function SPINSCalib_EPSDE::Evaluate, params, errFuncFlag=errFuncFlag, _REF_EXTRA=etc
compile_opt idl2


if (n_elements(errFuncFlag) eq 0) then errFuncFlag=Self.errFuncFlag

errorFunc = 1.0e4
status = Self->Model(params)
yData = (*Self.yDataPtr)
yError = (*Self.yErrorPtr)
calc = (*Self.calcPtr)
nd = n_elements(yData)
np = n_elements(params)

case errFuncFlag of
   ; mean square error (MSE)
   0: errorFunc = 1.0/(nd-np-1.0) * total(((yData - calc)/yError)^2, /NAN)
   
   ; Mean-abosolute error (MAE)
   1: errorFunc = 1.0/(nd-np-1.0) * total(abs((yData - calc)/yError), /NAN)
   
   ; Mean-square error of the log transformed data (MSElog)
   2: errorFunc = 1.0/(nd-np-1.0) * total(((alog10(yData) - alog10(calc))/alog10(yError))^2, /NAN)
   
   ; Mean-absolute error of the log transformed data (MAElog)
   3: errorFunc = 1.0/(nd-np-1.0) * total(abs((alog10(yData) - alog10(calc))/alog10(yError)), /NAN)
   
   else:

endcase


return, errorFunc
end
;-------------------------------------------------------------------------------


;===============================================================================
;+
; SPINSCalib_EPSDE::GetProperty 
;
; PURPOSE:
;   Get Property
;
; PARAMETERS:
;
; KEYWORDS:
; 
;   errFuncFlag - the error function choice
;   
;   nPop  - set if the data specified by the dataObject keyword is to be reloaded
;   
;   nMS   - the number of mutation strategies available
;   
;   model - an object reference to the model function to be fitted
;   
;   epsilon  - another terminating condition: Stop fit when change in the objective function is less than epsilon.
;   
;   nFEMax - maximum nos of function evaluations before terminating fit
;-
pro SPINSCalib_EPSDE::GetProperty, errFuncFlag=errFuncFlag, nPop=nPop, maxnFE=maxnFE, epsilon=epsilon $
                      , model=model, modelFlag=modelFlag, nMS=nMS $
                      , generationCnt=generationCnt $
                      , bestParam=bestParam, initParam=initParam $
                      , idealParam=idealParam, bestCalc=bestCalc $
                      , nFE=nFE, bestFValue=bestFValue $
                      , xvalues=xvalues,yvalues=yvalues, evalues=evalues $
                      , _REF_EXTRA=etc
compile_opt idl2

if (arg_present(nPop)) then nPop = Self.nPop
if (arg_present(maxnFE)) then maxnFE = Self.nFEMax
if (arg_present(epsilon)) then epsilon = Self.epsilon
if (arg_present(model)) then model = Self.model
if (arg_present(nMS)) then nMS = Self.nMS
if (arg_present(errFuncFlag)) then errFuncFlag = Self.errFuncFlag
if (arg_present(modelFlag)) then modelFlag = Self.modelFlag
if (arg_present(generationCnt)) then generationCnt = Self.generationCnt
if (arg_present(nFE)) then nFE = Self.nFE
if (arg_present(bestFValue)) then bestFValue = Self.xpbestFValue
if (arg_present(bestParam)) then begin
   if (ptr_valid(Self.xpBestPtr)) then bestParam = (*Self.xpBestPtr)
endif
if (arg_present(InitParam)) then begin
   if (ptr_valid(Self.xpInitPtr)) then initParam = (*Self.xpInitPtr)
endif
if (arg_present(idealParam)) then begin
   if (ptr_valid(Self.pIdealPtr)) then idealParam = (*Self.pIdealPtr)
endif
if (arg_present(xvalues)) then begin
   if (ptr_valid(Self.xDataPtr)) then xvalues = (*Self.xDataPtr)
endif
if (arg_present(yvalues)) then begin
   if (ptr_valid(Self.yDataPtr)) then yvalues = (*Self.yDataPtr)
endif
if (arg_present(evalues)) then begin
   if (ptr_valid(Self.yErrorPtr)) then evalues = (*Self.yErrorPtr)
endif




end
;-------------------------------------------------------------------------------


;===============================================================================
;+
; SPINSCalib_EPSDE::SetProperty 
;
; PURPOSE:
;   Set Property
;
; PARAMETERS:
;
; KEYWORDS:
; 
;   errFuncFlag - the error function calculation flag
;   
;   nPop  - set if the data specified by the dataObject keyword is to be reloaded
;   
;   nMS   - the number of mutation strategies available
;   
;   model - an object reference to the model function to be fitted
;   
;   epsilon  - another terminating condition: Stop fit when change in the objective function is less than epsilon.
;   
;   maxnFE - maximum nos of function evaluations before terminating fit
;-
pro SPINSCalib_EPSDE::SetProperty, errFuncFlag=errFuncFlag, nPop=nPop, maxnFE=maxnFE, epsilon=epsilon $
                      , model=model,modelFlag=modelFlag, nMS=nMS $
                      , xData=xData, yData=yData, yError=yError, pMin=pMin, pMax=pMax $
                      , startingFlag=startingFlag, _EXTRA=etc
compile_opt idl2

if (n_elements(pMin) gt 0) then begin
  *Self.pMinPtr = pMin
  Self.nPar = n_elements(pMin)
endif
if (n_elements(pMax) gt 0) then *Self.pMaxPtr = pMax
if (n_elements(xData) gt 0) then *Self.xDataPtr = xData
if (n_elements(ydata) gt 0) then begin
  *Self.yDataPtr = ydata
  *Self.calcPtr = yData
endif
if (n_elements(yError) gt 0) then begin
  index = where(yError le 0.0, cnt)
  if (cnt gt 0) then yError[index] = 1
  *Self.yErrorPtr = yError
endif


if (keyword_set(nPop)) then Self.nPop = nPop
if (keyword_set(maxnFE)) then Self.nFEMax = maxnFE
if (keyword_set(epsilon)) then Self.epsilon = epsilon
if (keyword_set(model) && obj_valid(model)) then Self.model = model
if (keyword_set(nMS)) then Self.nMS = nMS
if (keyword_set(errFuncFlag)) then Self.errFuncFlag = errFuncFlag
if (keyword_set(startingFlag)) then Self.startingFlag = startingFlag
if (keyword_set(modelFlag)) then begin
   if (modelFlag le 0) then begin
      print, 'Improper model selection!'
      return
   endif
   Self.modelFlag = modelFlag
   Self.startingFlag = 1
   void = Self->Model()
endif

;if (keyword_set()) then Self. = 

end

;===============================================================================
;+
; SPINSCalib_EPSDE::Fit
;
; PURPOSE:
;   Fit a model function using a Differential evolution algorithm with an 
;   Ensemble of Parameters and Mutation Strategies
;
; PARAMETERS:
;
; KEYWORDS:
;  
;-
function SPINSCalib_EPSDE::Fit, modelFlag=modelFlag, _EXTRA=etc
compile_opt idl2


; Set and Retrieve info about the model
if (keyword_set(modelFlag)) then Self->SetProperty, modelflag=modelFlag
nPar = Self.nPar    ; nos of parameters in the model
pMin = (*Self.pMinPtr)    ; 1D array of size nPar containing the minimum range for each parameter
pMax = (*Self.pMaxPtr)    ; 1D array of size nPar containing the maximum range for each parameter
nPop = Self.nPop     ; the population size; ie each parameter is sampled nPop times in each iteration/generation

; Create initial population of size (nPar x nPop)
; Made up of nPop vectors, each containing nPar elements
; Each element is randomly generated in the range [pmin,pmax]
uVec = dblarr(nPop) + 1.0
xp = (pMin#uVec) + randomu(systime(/seconds),nPar,nPop)*(pMax#uVec - pMin#uVec)

; Randomly assign nPop
; - mutation strategies out of a choice of 3
; - scale factors for scaling the difference vector and
; - cross-overs values: determine how trial vector chosen from a pair of target/mutant vectors 
CR = (*Self.crPtr)
SF = (*Self.sfPtr)
nCR = n_elements(CR)
nSF = n_elements(SF)
scsPool = fltarr(3,nPop)
scsPool[0,*] = fix(3*randomu(systime(/seconds),nPop))+1
scsPool[1,*] = CR[fix(nCR*randomu(systime(/seconds),nPop))]
scsPool[2,*] = SF[fix(nSF*randomu(systime(/seconds),nPop))]


; Find best vector and objective value from initial population
Self.nFE = 0   ; reset to 0
bestIndex = 0
bestfValue = Self->Evaluate(xp[*,0])
fValues = dblarr(nPop)
fValues[0] = bestfValue
for i = 1,nPop-1 do begin
   fValues[i] = Self->Evaluate(xp[*,i])
   if (fValues[i] lt bestfValue) then begin
      bestfValue = fValues[i]
      bestIndex = i
   endif
endfor
bestXp = xp[*,bestIndex]
if (ptr_valid(Self.xpInitPtr)) then $
   (*Self.xpInitPtr) = bestXp else $
   Self.xpInitPtr = ptr_new(bestXp)
if (ptr_valid(Self.xpBestPtr)) then $
   (*Self.xpBestPtr) = bestXp  else $
   Self.xpBestPtr = ptr_new(bestXp)

if (ptr_valid(Self.xpBest_gen)) then ptr_free, Self.xpBest_gen
Self.xpBest_gen = ptr_new([])
if (ptr_valid(Self.xpBestfValue_gen)) then ptr_free, Self.xpBestfValue_gen
Self.xpBestfValue_gen = ptr_new([])
if (ptr_valid(Self.xpPool)) then ptr_free, Self.xpPool
Self.xpPool = ptr_new([])
if (ptr_valid(Self.xpfValuePool)) then ptr_free, Self.xpfValuePool
Self.xpfValuePool = ptr_new([])

Self.generationCnt = 1
scsPoolBreed = []
RR = intarr(nPop) - 1
rate = []
prevGenfValue = bestfValue
deltafValue = bestfValue
while ((Self.nFE lt Self.nFEMax) && (deltafValue gt Self.epsilon)) do begin  
   nscsPoolBreed = (size(scsPoolBreed))[2]
   if (Self.generationCnt gt 1) then begin
      ; Specify mutation strategies, CR and SF pool for this generation
      
      randk = randomu(systime(/seconds), nPop)
      for k=0,nPop-1 do begin
         if (isa(scsPoolBreed) && (randk[k] le rateMean)) then begin
            ; select a random integer index in range [0:n] where n is the 2nd dim length of scsPoolBreed 
            index = fix(randomu(rrSeed)*nscsPoolBreed)
            RR[k] = index ;fix(randomu(systime(/seconds))*nscsPoolBreed)
            
            ; randomly select from the successful (S,CR,SF) pool and 
            ; replace the current k of the (S,CR,SF) pool
            scsPool[*,k] = scsPoolBreed[*,index]
;;TODO       should the 'index' variable be different rands for RR and scsPool? 
         endif else begin
            RR[k] = -1
            ; make a new (S,CR,SF) set to replace the current k
            scsPool[0,k] = fix(3*randomu(sSeed))+1
            scsPool[1,k] = CR[fix(nCR*randomu(crSeed))]
            scsPool[2,k] = SF[fix(nSF*randomu(sfSeed))]
         endelse
      
      endfor
   endif

   RRR = []
   cntBadTrial = 0
   void = where(RR ge 0, RRisDefined)
   xpOld = xp  ; xpOld is the current pop while xp is the newly 
               ; emerging population for the next generation

   ; loop through the population
   for i=0,nPop-1 do begin
      ; the CR previously allocated randomly
      CRi = scsPool[1,i]    
      
      ; for SF, get the SF previously allocated. Generate a normal
      ; distribution of nPar SF values with a mean set to the given SF and
      ; a std dev of 0.001
      SFdist = randomn(sfdSeed,/normal,nPar)*0.001 + scsPool[2,i]
      
      
      ; Each vector consists of nPar individuals. In generating the trial vector, 
      ; CR controls whether individuals are selected from the current vector or the
      ; mutant vector based on the cross over rules. Individuals are selected based on
      ; the value of a random nos relative to the value of CR.
      ; Generate nPar rand nos between [0,1] and determine those that are less
      ; than or equal to the current CR. Where this is the case, select individuals
      ; from the mutant vector else select from the current vector.
      fromMutant = randomu(mSeed,nPar) le CRi   ; 1's and 0's; 1 ==> mutant individual to be included in next trial
      if (total(fromMutant) eq 0) then begin
         ;==> all the random nos were gt CR (thus the trial will equal the current vector!)
         ; in this case, simply generate a random nos between 1 and nPar and force the
         ; trial individual matching the random nos to come from the mutant
         index = fix(randomu(jSeed)*nPar)
         fromMutant[index] = 1
      endif
      fromXp = fromMutant ne 1   ; a 1 ==> current individual to be included in next trial vector
      
      jrnd = fix(randomu(iSeed,nPop)*nPop) ; randomly select nPop indices from 0 to nPop-1
      case scsPool[0,i] of    ; the mutant strategy
         1: begin ; DE/best/2
            ; Vi = Xbest + SF*(Xr1 - Xr2) + SF*(Xr3 - Xr4)
            ; Ui = Vi if rand[0,1] < CR for j=1,nPar || when randj eq j
            ; Ui = Xi otherwise
            mutantXp = bestXp + SFdist*( xpOld[*,jrnd[0]] - xpOld[*,jrnd[1]] $
                                       + xpOld[*,jrnd[2]] - xpOld[*,jrnd[3]] ) 
            trialXp = xpOld[*,i] * fromXp  +  mutantXp * fromMutant
         end
         
         2: begin ; DE/rand/1
            ; Vi = Xr1 + SF*(Xr2 - Xr3)
            ; Ui = Vi if rand[0,1] < CR for j=1,nPar || when randj eq j
            ; Ui = Xi otherwise
            mutantXp = xpOld[*,jrnd[0]] + SFdist*( xpOld[*,jrnd[1]] - xpOld[*,jrnd[2]])
            trialXp = xpOld[*,i] * fromXp  +  mutantXp * fromMutant
         end
         
         3: begin ; DE/current-to-rand/1
            ; Ui = Xi + K*(Xr1 - Xi) + SF*(Xr2 - Xr3) where k=rand[0,1]
            kk = randomu(kkSeed)
            trialXp = xpOld[*,i] + kk*(xpOld[*,jrnd[0]] - xpOld[*,i]) $
                             + SFdist*(xpOld[*,jrnd[1]] - xpOld[*,jrnd[2]])
         end
         
         else:
      endcase
      
      
      ; Ensure the trial vector is within parameter bounds
      if (total((trialXp lt pMin) or (trialXp gt pMax)) gt 0.0) then begin
         ; => at least one individual is outside bounds 
         ;    so generate a new trial vector using random individuals within bounds
         trialxp = pMin + randomu(pSeed,nPar)*(pMax - pMin)
      endif
      
      ; evaluate trial vector
      fValue = Self->Evaluate(trialXp)
      if (fValue lt fValues[i]) then begin
         xp[*,i] = trialXp
         fValues[i] = fValue
         if (fValue lt bestfValue) then begin
            bestfValue = fValue
            bestIndex = i
            bestXp = trialXp
         endif
         ; RR contains nPop random indices that were selected whenever an (S,CR,SF) set
         ; was carried over from a previous generation. Whenever a successful trial
         ; vector is found, the current RR index is recorded in RRR
         if (RRisDefined gt 0) then begin
            if (RR[i] ge 0) then RRR = [RRR,RR[i]]
         endif
         ; Store (S,CR,SF) sets that lead to a successful trial vector into scsPoolBreed
         if (isa(scsPoolBreed)) then $
            scsPoolBreed = transpose([transpose(scsPool[*,i]),transpose(scsPoolBreed)]) else $
            scsPoolBreed = scsPool[*,i]
      endif else $
         cntBadTrial++    
   endfor
   
   ; delete entries from scsPoolBreed whose indices match RRR
   if (isa(RRR)) then begin
      scsPoolBreed[0,RRR] = -1
      index = where(scsPoolBreed[0,*] gt 0,cnt)
      if (cnt gt 0) then scsPoolBreed = scsPoolBreed[*,index]
   endif
   
   ; record ratio of unchanged vectors in population in this generation
   rate = [rate,float(cntBadTrial)/nPop]
   ; Calculate mean rate for last 10 records
   n = n_elements(rate)
   rateMean = (n gt 10)? mean(rate[n-10:n-1]) : mean(rate)
   
   (*Self.xpBestPtr) = bestXp
   Self.xpbestfValue = bestfValue
   if (bestfValue ne prevGenfValue) then  deltafValue = abs(bestfValue - prevGenfValue)
   prevGenfValue = bestfValue
   
   (*Self.xpBest_gen) = [(*Self.xpBest_gen),transpose(bestXp)]
   (*Self.xpBestfValue_gen) = [(*Self.xpBestfValue_gen),bestfValue]
   
   ;; require a minimum of 20 generations to begin evaluating deltafValue condition
   if (Self.generationCnt ge 20) then begin
      ng = Self.generationCnt
      deltafValue = abs((*Self.xpBestfValue_gen)[ng-20] - (*Self.xpBestfValue_gen)[ng-19])
      for j = ng-19, ng-2 do $
         deltafValue = deltafValue > abs((*Self.xpBestfValue_gen)[j] - (*Self.xpBestfValue_gen)[j+1])     
   endif else begin
      deltafValue = 1.0
   endelse

   Self.generationCnt = Self.generationCnt + 1

endwhile

;            ,xpBest_gen:ptr_new()    $
;            ,xpBestfValue_gen:ptr_new() $
;            ,xpPool:ptr_new()        $
;            ,xpfValuePool:ptr_new()  $


return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
;+
; SPINSCalib_EPSDE::Init 
;
; PURPOSE:
;   Initialization method
;
; PARAMETERS:
;
; KEYWORDS:
; 
;   errFuncFlag - the error function calculation to use
;   
;   nPop  - set if the data specified by the dataObject keyword is to be reloaded
;   
;   nMS   - the number of mutation strategies available
;   
;   model - an object reference to the model function to be fitted
;   
;   epsilon  - another terminating condition: Stop fit when change in the objective function is less than epsilon.
;   
;   nFEMax - maximum nos of function evaluations before terminating fit
;-
function SPINSCalib_EPSDE::Init, errFuncFlag=errFuncFlag, nPop=nPop, nFEMax=nFEMax, epsilon=epsilon $
                    , model=model, modelFlag=modelFlag, nMS=nMS $
                    , _REF_EXTRA=etc
                    
compile_opt idl2

Self.pMinPtr = ptr_new(0)
Self.nPar = n_elements(0)
Self.pMaxPtr = ptr_new(0)
Self.xDataPtr = ptr_new(0)
Self.yDataPtr = ptr_new(0)
Self.calcPtr = ptr_new(0)
Self.yErrorPtr = ptr_new(0)


;; Call our super classes
if (~self->IDLitComponent::Init(_EXTRA=etc)) then $
  return, 0
if (~self->IDL_Container::Init()) then $
  return, 0
if (~self->IDL_Object::Init()) then $
  return, 0


if (~keyword_set(errFuncFlag)) then errFuncFlag = 0
Self.errFuncFlag = errFuncFlag

if (~keyword_set(modelFlag)) then modelFlag = 1
Self.modelFlag = modelFlag

if (~keyword_set(nPop)) then nPop = 100
Self.nPop = nPop

if (~keyword_set(nFEMax)) then nFEMax = 10000
Self.nFEMax = nFEMax

if (~keyword_set(epsilon)) then epsilon = 1.0D-5
Self.epsilon = epsilon

if (~keyword_set(model)) then model = obj_new()
Self.model = model

Self.crPtr = ptr_new([0.1,0.2,0.3,0.4,0.5,0.6,0.7,0.9,0.9])

Self.sfPtr = ptr_new([0.4,0.5,0.6,0.7,0.9,0.9])

if (~keyword_set(nMS)) then nMS = 3
Self.nMS = nMS

Self.startingFlag = 1
Self.generationCnt = 0
Self.name = "Diff. Evolution"
Self->SetPropertyAttribute, 'NAME', /hide
Self->SetPropertyAttribute, 'DESCRIPTION', /hide

Self->SetProperty, _EXTRA=etc

return, 1

end
;-------------------------------------------------------------------------------


;===============================================================================
pro SPINSCalib_EPSDE__define
   struct = {SPINSCalib_EPSDE $
            ,inherits IDLitComponent $
            ,inherits IDL_Container  $
            ,inherits IDL_Object $
             
            ,model:obj_new()         $    ; the model function to be fitted
            ,modelFlag:0             $    ; which build-in function to use
            ,errFuncFlag:0           $    ; specify which error function calculation to use

            ,startingFlag:0          $    ; a control flag
            ,nPar:0                  $    ; nos of parameters
            ,pMinPtr:ptr_new()       $    ; parameters bounds - minimum
            ,pMaxPtr:ptr_new()       $    ; parameters bounds - maximum
            ,pIdealPtr:ptr_new()     $    ; ideal best fir parameters (where available for test model/data)
            ,pIdealErrorPtr:ptr_new()$    ; ideal best fit parameter errors (where available for test model/data)
            ,xDataPtr:ptr_new()      $    ; x data values to be fitted
            ,yDataPtr:ptr_new()      $    ; y data values (dependent data) to be fitted
            ,yErrorPtr:ptr_new()     $    ; error in y values 
            ,calcPtr:ptr_new()       $    ; latest calculated dependent data values
            
            ,nPop:0L                 $    ; population size
            ,nFEMax:0L               $    ; maximum nos of function evaluations before terminating
            ,epsilon:0.0D            $    ; another terminating condition: improvements in the cost function is less than epsilon.
            ,crPtr:ptr_new()         $    ; cross over pool of values
            ,sfPtr:ptr_new()         $    ; scale factor pool of values
            ,nMS:0L                  $    ; nos of mutation strategies in use
            ,msFlagPtr:ptr_new()     $    ; flags indicating mutation strategy pool
            ,generationCnt:0L        $    ; nos of generations to date
            
;            ,xpNewGenPtr:ptr_new()   $    ; the target vectors (new generation population)
;            ,xpOldGenPtr:ptr_new()   $    ; the target vectors (old/current generation population)
;            ,vpPtr:ptr_new()         $    ; the mutation vectors used to determine the trial vectors
;            ,upPtr:ptr_new()         $    ; the trial vectors used to calculate the next generation (xpNewGenPtr)

            ,xpInitPtr:ptr_new()     $    ; the initial vector used at the start
            ,xpBestPtr:ptr_new()     $    ; the best vector found so far
            ,xpbestfValue:0.0D       $    ; value of cost function evaluated using xpBest, the best parameter
            ,nFE:0L                  $    ; current count of function evaluations
            
            ,xpBest_gen:ptr_new()    $
            ,xpBestfValue_gen:ptr_new() $
            ,xpPool:ptr_new()        $
            ,xpfValuePool:ptr_new()  $
           }
end
