; $Id$
; ************************************ ;
; NAME:
;  IS_NICE
;
; PURPOSE:
;  This function determines if the file specified
;  by the argument FILENAME is a valid NICE hdf/nexus file
;  as produced by the NICE data acquisition system
;  at the NCNR.
;
; CATEGORY:
;  DAVE, Data Reduction, TAS.
;
; AUTHOR:
;   Richard Tumanjong Azuah
;   NIST Center for Neutron Research
;   100 Bureau Drive
;   Gaithersburg, MD 20899
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
; ************************************ ;
function is_hdf_nice,filename,errmsg = errmsg,oFTP=oFTP, nEntries=nEntries

errmsg = 'Unsupported file format'
catch,the_error
if the_error ne 0 then begin
   catch,/cancel
   if n_elements(lun) ne 0 then free_lun,lun,/force
   errmsg = !error_state.msg
   return,0B
endif

fromFtp = (stregex(filename,'/pub/ncnrdata/',/fold,/bool))? 1 : 0
if (fromFTP) then begin
  ; if fromFTP, get contents of remote file and save in a local file on disk
  if (~Obj_valid(oFTP)) then begin
    Self->Statusmessage, 'The FTP service object is invalid. Cannot proceed'
    Return, 0
  endif
  ; retrieve the file contents from ftp server and save in temporal local file
  tmpFile = !home_dir+'BT7TmpDatasetFromFTP.dat'
  filename=oFTP->Getfilecontent(filename,localfilename=tmpFile)
  if (~File_test(filename,/read)) then begin
    Self->Statusmessage, 'Unable to create local file '+tmpFile
    Return, 0  ; if local file was not created then bail
  endif
endif else begin
  if ~file_test(filename) then begin
     errmsg = 'File '+filename+' not found'
     return,0B
  endif
endelse

retVal = H5f_is_hdf5(filename)
if (~retVal) then return, 0

if arg_present(nEntries) then begin
  ; file handle must be a long integer and must be set before calling Nxopen()
  handle=0L
  hdlerror=0

  ; Open the hdf file
  ; The file handle will subsequently be used just like a file unit
  status = Nxopen(filename,'NXACC_READ',handle)
  if (~status) then begin
    errmsg = 'Could not open nexus file: '+filename
    void = Dialog_message([filename,' ',errmsg],/error)
    Return, 0
  endif

  ; retrieve the number of top-level entries in the dataset
  nEntries = H5g_get_nmembers(handle.ivid, '/')
  
  status = Nxclose(handle)          ; close Nexus/hdf file
endif

; if necessary, delete tmpfile that was created above
if (fromFTP && file_test(tmpFile)) then File_delete, tmpFile
  
return, retVal

end


;************************************* ;
pro test_is_hdf_nice, filename
if (~isa(filename)) then filename = dialog_pickfile()
ret = is_hdf_nice(filename,errmsg = errmsg, nEntries=nEntries)
if ret then begin
  print,filename+' is a valid NICE hdf/nexus file'
  print,'There are ',strtrim(string(nEntries),2),' top-level entries available in the dataset'
endif else print,errmsg

end