; $Id$
;###############################################################################
;
;CLASS NAME:
;    cwo_displayIQT
;
;PURPOSE:
;       Object compound widget to interact with I(Q,t) data from the
;       Neutron Spin Echo data reduction software.
;       This is meant to be a Singleton method, so only one
;       instance of this should exist.
;
;           AN OBJECT COMPOUND WIDGET TO PLOT I(Q,t)
;           DATA FOR THE NSE DATA REDUCTION MODULE
;           IN DAVE.
;CATEGORY:
;       Data visualization and analysis.
;SUPERCLASSES:
;       myproperties
;
;METHODS:
;    cwo_displayIQT_class::doNothing
;    cwo_displayIQT
;    cwo_displayIQT_class::draw
;    cwo_displayIQT_class::normalize
;    cwo_displayIQT_class::event
;    cwo_displayIQT_event_func
;    nse_expdecay
;    nse_stretchexpdecay
;    cwo_displayIQT_class::removeData
;    cwo_displayIQT_class::addData
;    cwo_displayIQT_class::resize
;    cwo_displayIQT_realize
;    cwo_displayIQT_class::cleanup
;    cwo_displayIQT_class::init
;    cwo_displayIQT_class__define
;
;
; AUTHOR:
; Larry Kneller
; NIST Center for Neutron Research
; 100 Bureau Drive, Gaithersburg, MD 20899
; United States
; kneller@nist.gov  301-975-8839
; Jan 24,2005
;
; LICENSE:
; The software in this file is written by an employee of
; National Institute of Standards and Technology
; as part of the DAVE software project.
;
; The DAVE software package is not subject to copyright protection
; and is in the public domain. It should be considered as an
; experimental neutron scattering data reduction, visualization, and
; analysis system. As such, the authors assume no responsibility
; whatsoever for its use, and make no guarantees, expressed or
; implied, about its quality, reliability, or any other
; characteristic. The use of certain trade names or commercial
; products does not imply any endorsement of a particular product,
; nor does it imply that the named product is necessarily the best
; product for the stated purpose. We would appreciate acknowledgment
; if the DAVE software is used of if the code in this file is
; included in another product.
;
;###############################################################################




function nse_expdecay,X,P;INDEPENDENT VAR IS t*Q^2
;
;NAME:
;        nse_expdecay
;
;PURPOSE:
;       function for fitting the data.
;PARAMETERS:
;       X   data values
;       P   parameter values.
;KEYWORDS:
;       none
;RETURN VALUE:
;       function values at the X avlues.
    ;DECAYING EXPONENTIAL FOR FITTING I(Q,t) CURVES

    ;A=P[0]
    ;D=P[1]
    ;X=t*Q^2

    return,P[0]*exp(-1.0*P[1]*X)

;020805
;STEVE SENT THE FOLLOWING:
;   THIS CAN BE ACCOUNTED FOR BY SIMPLY CHOOSING WHICH t VALUES ARE
;   SUBMITTED TO THE FIT, I.E. t vs t*Q^2
;
;
;> Function ApparentDiffusion(w,tval) : FitFunc
;>     Wave w
;>     Variable tval
;>
;>     Variable result,Dapp,beta_exp,amp,qval,bkg
;>     Dapp = w[0]
;>     qval = w[1]
;>
;>
;>     result = exp( -1 * (Dapp*tval) * qval^2 )
;>
;>     return(result)
;> End





end;nse_expdecay
function nse_stretchexpdecay,t,P;INDEPENDENT VAR IS t*Q^2
;
;NAME:
;        nse_stretchexpdecay
;
;PURPOSE:
;       function for fitting the data to a stretched exp.
;PARAMETERS:
;       t   the data values
;       P   the function parameters
;KEYWORDS:
;       none
;RETURN VALUE:
;       the values of the function at t.
;
    ;DECAYING EXPONENTIAL

    ;A    =P[0]
    ;gamma=P[1]
    ;beta =P[2]
    ;t    =t


    return,P[0]*exp(-1.0*(P[1]*t)^P[2])

end;nse_expdecay




;
function cwo_displayIQT_class::doNothing
;
;NAME:
;        cwo_displayIQT_class::doNothing
;
;PURPOSE:
;       ;A do_nothing METHOD.
;PARAMETERS:
;       None.
;KEYWORDS:
;       None.
;
;RETURN VALUE:
;       0.
    return,0
end;doNothing

;;############################################################
;function cwo_displayIQT,parent,datRef,$
;                            obj_ref=obj_ref,$
;                            uvalue=uvalue,$
;                            _Extra=extra
;;
;;NAME:
;;        cwo_displayIQT
;;
;;PURPOSE:
;;       Function to create the cwo.
;;PARAMETERS:
;;       parent  The parent widget id.
;;       datRef  The ooEcho object with the I(Q,t) data.
;;KEYWORDS:
;;       obj_ref[out]    The object reference for this cwo.
;;       uvalue[in]      The uvalue for this cwo.
;;
;;_EXTRA - is used to pass keywords meant for the superclass.
;
;;RETURN VALUE:
;;       tlb:    THE TOP LEVEL BASE ID FOR THIS cwo.
;;
;
;    ;FUNCTION TO RETURN THE BASE ID FROM THE OBJECT
;    o = obj_new('cwo_displayIQT_class',parent,datRef,uvalue=uvalue);,_Extra=extra)
;
;    if obj_valid(o) gt 0 then begin
;
;        obj_ref=o   ;SETS VALUE OF obj_ref IN FUNCTION CALL
;                    ;TO ALLOW USER OF PROGRAM ACCESS TO THE
;                    ;OBJECT REFERENCE.
;
;
;        x = o->getProperty(/tlb)
;
;        return,o->getProperty(/tlb)
;    endif else begin
;        return,0L
;    endelse
;
;end;cwo_displayIQT
;############################################################
pro cwo_displayIQT_class::draw,xrange=xrange,$
                         yrange=yrange,$
                         zoom=zoom
;
;NAME:
;        cwo_displayIQT_class::draw
;
;PURPOSE:
;       draw method for the cwo.
;PARAMETERS:
;
;KEYWORDS:
;        xrange     xrange for plot
;        yrange     yrange for plot
;        zoom       zoom flag.
;;


;print,'cwo_displayIQT_class DRAW begin'
self.stepIndex = 0
device,get_decomposed = dc
device,decomposed = 1

    ;LIFT THE PLOTTING CODE FROM THE displayIQT ROUTINE IN
    ;ooDisplayEcho AND MODIFY IT HERE.

    iqt = *(self.data)
    siqt = *(self.ddata)
    sz = size(iqt)
;    print,'size(iqt)=',sz
    iqtfit = *(self.datafit)
    qvals = *((self.datacontainer->get(position=(self.datacontainer->count()-1)))->getProperty(tag='qarcs'))
    qformat = '(f8.4)'

    fitparms = *(self.datafitparms)
    ;print,fitparms
    szfitparms = size(fitparms)

    if widget_info(self.displayD,/valid_id) gt 0 then begin
        ;help,fitparms
        if obj_valid(self.displaydobj) then $
            (self.displaydobj)->changedata,qvals,fitparms[*,1]

    endif



    ;GET THE FOURIER TIMES AND I(Q,t)
    tauvals = *(self.xvals)
    if n_elements(xrange) eq 0 then begin
        xrange = dblarr(2)
        xrange[0] = min(tauvals)
        xrange[1] = max(tauvals)
    endif ;else print,'cwo_displayIQT_class::draw xrange=',xrange
    if n_elements(yrange) eq 0 then begin
        yrange = dblarr(2)
        if min(iqt) eq double('Infinity') or min(iqt) eq double('Nan') then begin
            yrange[0] = 0.1
        endif else begin
            yrange[0] = min(iqt)
        endelse
        if min(iqt) eq double('Infinity') or min(iqt) eq double('Nan') then begin
            yrange[1] = 2.0
        endif else begin
            yrange[1] = max(iqt)
        endelse
    endif ;else print,'cwo_displayIQT_class::draw yrange=',yrange
    if n_elements(zoom) eq 0 then zoom = 0


    ;PLOT I(Q,t)
    ;print,'cwo_displayIQT_class::draw self.winvis=',self.winvis
;    wset,self.winvis
    ;print,'cwo_displayIQT_class::draw self.winpix=',self.winpix
    wset,self.winpix

    ;CREATE AXES
    if self.xlog eq 1 then xtitle = 'log(t (ns))' else xtitle = 't (ns)'
    if self.ylog eq 1 then ytitle = 'log(I(Q,t))' else ytitle = 'I(Q,t)'

    infindex = where(iqt eq double('Infinity'))
    if infindex[0] ne -1 then begin
        iqt[infindex] = double('Nan')
        siqt[infindex] = double('Nan')
    endif

;print,iqt
;print,*self.yrange
;for i = 0,sz[1]-1 do begin
;    print,min(iqt[i,*]),max(iqt[i,*])
;endfor;i
;print,!y.crange
;help,*(self.xrange)
;help,*(self.yrange)
;print,*(self.yrange)

;042705
;SOME NEW GLITCH IN plot OCCURRED THAT FORCED ME TO ADD THE
;xrange AND yrange EXPLICITLY.
;EVERYTHING HAD BEEN WORKING FINE AND SUDDENLY THAT
;CHANGED.

red = 255L
green = 256L*red
blue = 256L*green
black = 0L
white = red+green+blue
;100705
;ADDING THE NEXT TEST IN CASE DATA CONTAIN ONLY NaN's.

;print,(*(self.xrange))
;print,(*(self.yrange))
stestrange = strtrim(string([(*(self.xrange)),(*(self.yrange))]),2)
testwhere = where(stestrange eq '-NaN' or stestrange eq 'NaN' $
                  or stestrange eq '-Infinity' or stestrange eq 'Infinity')
;print,stestrange
;print,testwhere

if (*(self.xrange))[0] eq (*(self.xrange))[1] or (*(self.yrange))[0] eq (*(self.yrange))[1] $
    or testwhere[0] ne -1 then begin
    plot,[0,1],[0,1],/nodata,$
            title=self.plottitle,$
            xtitle=xtitle,$
            ytitle=ytitle,$
            background = white, $
            color=black,xstyle=1,ystyle=1,$
            xticks = 5, yticks = 5;,$

endif else begin
    plot,(*(self.xrange)),(*(self.yrange)),/nodata,$
            xrange = (*(self.xrange)),$
            yrange = (*(self.yrange)),$
            xlog=self.xlog,$
            title=self.plottitle,$
            xtitle=xtitle,$
            ylog=self.ylog,$
            ytitle=ytitle,$
            background = white, $
            color=black,xstyle=1,ystyle=1;,$
            ;xticks = 5, yticks = 5;,$
            ;xticklayout=2,yticklayout=2
endelse


;plot,dindgen(10),10.0*sin(dindgen(10)),yrange=*self.yrange;[-100,100]
;print,!y.crange

    r=150L & g=0L & b=0L
    color = r + 256L*(g + 256L*b)
    symsize = 2
    oplot,tauvals,reform(iqt[0,*],sz[2]),$
            color = color,psym=8,symsize=symsize
    oplot,tauvals,reform(iqtfit[0,*],sz[2]),$
            color = color,psym=0,symsize=symsize
;print,!y.crange


    for jj = 0,n_elements(tauvals)-1 do $
            plots,[tauvals[jj],tauvals[jj]], $
                    [iqt[0,jj]-siqt[0,jj], iqt[0,jj]+siqt[0,jj]],$
                    color=color,/data
;    xyouts, 0.8,0.9,$
;            'Q'+strtrim(string(0),2), /normal,$
;            color = color

    ;CREATE THE LEGEND
    plotxrange = !x.crange & plotyrange = !y.crange
    ;xnorm = 0.8 & ynorm = 0.91

    xnorm = self.xlegend & ynorm = self.ylegend

    ;NEW USAGE OF nse_legend    01/27/05
    FORMAT = '(", P",I1,"=",E9.2)'
    legendText = 'Q='+strtrim(string(qvals[0],format=qformat),2)+'/'+STRING("305B)
    for ifp = 0,szfitparms[2]-1 do begin
        ls = string(ifp,fitparms[0,ifp],format=FORMAT)
        legendText = legendText+ls
    endfor;ifp

    nse_legend,self.winpix,xnorm,ynorm,$
                legendText,$;'Q='+strtrim(string(qvals[0],format=qformat),2)+'/'+STRING("305B),$
                color=color,psym=8,$
                symsize=symsize

;    nse_legend,self.winpix,xnorm,ynorm,plotxrange,plotyrange, $
;                    'Q'+strtrim(string(0),2),$
;                    xlog=self.xlog,ylog=self.ylog,$
;                    color=color,psym=8,symsize=symsize



    for i=1,sz[1]-1 do begin

        ;STORE COLORS
        ;tvlct,r,g,b,/get

;        rnew = bindgen(256)
;        gnew = bindgen(256)
;        bnew = bindgen(256)

        case i mod 6 of
        0:begin
;            gnew *= 0 & bnew *= 0
            rnew = 255 & gnew = 0 & bnew = 0
        end
        1:begin
;            rnew *= 0 & bnew *= 0
            rnew = 0 & gnew = 255 & bnew = 0
        end
        2:begin
;            gnew *= 0
            rnew = 0 & gnew = 0 & bnew = 255
        end
        3:begin
;            rnew *= 0 & gnew *= 0
            rnew = 155 & gnew = 155 & bnew = 0;255 & bnew = 0
        end
        4:begin
;            rnew *= 0
            rnew = 255 & gnew = 0 & bnew = 255
        end
        5:begin
;            bnew *= 0
            rnew = 0 & gnew = 255 & bnew = 255
        end
        else:begin
        end
        endcase

device,decomposed=0
        tvlct,rnew,gnew,bnew
device,decomposed = 1

        psym = i mod 6
        if i gt 2 then psym = psym + 2
        color = 50 + 100*(psym mod 3);-1*color - 75*psym

        ;r=150L & g=0L & b=0L
        color = rnew + 256L*(gnew + 256L*bnew)

        oplot,tauvals,reform(iqt[i,*],sz[2]),$
                color=color,$
                psym=psym,symsize=symsize
        oplot,tauvals,reform(iqtfit[i,*],sz[2]),$
                color = color,psym=0,symsize=symsize

        for jj = 0,n_elements(tauvals)-1 do $
            plots,[tauvals[jj],tauvals[jj]], $
                    [iqt[i,jj]-siqt[i,jj], iqt[i,jj]+siqt[i,jj]],$
                    color=color,/data


        ;111904
        ;NORMALIZE THESE COORDINATES SO oplot CAN BE USED
        ;TO ADD SYMBOLS TO THE LEGENDS.

;        xnorm = 0.8 & ynorm = 0.91-0.05*i
        xnorm = self.xlegend & ynorm = self.ylegend - 0.04*i
        ;NEW USAGE OF nse_legend    01/27/05
        legendText = 'Q='+strtrim(string(qvals[i],format=qformat),2)+'/'+STRING("305B)
        for ifp = 0,szfitparms[2]-1 do begin
            ls = string(ifp,fitparms[i,ifp],format=FORMAT)
            legendText = legendText+ls
        endfor;ifp
                nse_legend,self.winpix,xnorm,ynorm,$
                    legendText,$;'Q='+strtrim(string(qvals[i],format=qformat),2)+'/'+STRING("305B),$
                    color=color,psym=psym,$
                    symsize=symsize
;        nse_legend,self.winpix,xnorm,ynorm,plotxrange,plotyrange, $
;                        'Q'+strtrim(string(i),2),$
;                        xlog=self.xlog,ylog=self.ylog,$
;                        color=color,psym=psym,symsize=symsize

;        ;RESTORE COLORS
;        ;print,'Restore CT'
;        tvlct,r,g,b


    endfor;i

    if zoom ne 0 then begin
        plots,[self.rpress[0],self.rpress[0]],$
                [self.rpress[1],self.rmotion[1]],$
                color=0L,/data
        plots,[self.rpress[0],self.rmotion[0]],$
                [self.rmotion[1],self.rmotion[1]],$
                color=0L,/data
        plots,[self.rmotion[0],self.rmotion[0]],$
                [self.rmotion[1],self.rpress[1]],$
                color=0L,/data
        plots,[self.rmotion[0],self.rpress[0]],$
                [self.rpress[1],self.rpress[1]],$
                color=0L,/data
    endif;zoombox

    wset,self.winvis
    ;COPY IMAGE IN VIRTUAL WINDOW TO CURRENT WINDOW
    device,copy = [0,0,!d.x_size,!d.y_size,0,0,self.winpix]

;GET AXIS POINTERS
if ptr_valid(self.xptr) ne 0 then ptr_free,self.xptr
if ptr_valid(self.yptr) ne 0 then ptr_free,self.yptr
self.xptr = ptr_new(!x)
self.yptr = ptr_new(!y)

device,decomposed = dc
;print,'cwo_displayIQT DRAW end'

end;draw

;#############################################################
pro cwo_displayIQT_class::drawOne,xrange=xrange,$
                            yrange=yrange,$
                            zoom=zoom
;
;NAME:
;        cwo_displayIQT_class::drawOne
;
;PURPOSE:
;       draw IQT curve for one Q at a time.  Useful for stepping through curves.
;PARAMETERS:
;
;KEYWORDS:
;        xrange     xrange for plot
;        yrange     yrange for plot
;        zoom       zoom flag.
;;


device,get_decomposed = dc
device,decomposed = 1

    ;LIFT THE PLOTTING CODE FROM THE displayIQT ROUTINE IN
    ;ooDisplayEcho AND MODIFY IT HERE.

    iqt = *(self.data)
    siqt = *(self.ddata)
    sz = size(iqt)
    if self.stepIndex ge sz[1] then self.stepIndex = 0
    if self.stepIndex lt 0 then self.stepIndex = sz[1]-1


;    print,'size(iqt)=',sz
    iqtfit = *(self.datafit)
    qvals = *((self.datacontainer->get(position=(self.datacontainer->count()-1)))->getProperty(tag='qarcs'))
    qformat = '(f8.4)'

    fitparms = *(self.datafitparms)
    ;print,fitparms
    szfitparms = size(fitparms)



    ;GET THE FOURIER TIMES AND I(Q,t)
    tauvals = *(self.xvals)
    if n_elements(xrange) eq 0 then begin
        xrange = dblarr(2)
        xrange[0] = min(tauvals)
        xrange[1] = max(tauvals)
    endif ;else print,'cwo_displayIQT_class::draw xrange=',xrange
    if n_elements(yrange) eq 0 then begin
        yrange = dblarr(2)
        yrange[0] = min(iqt)
        yrange[1] = max(iqt)
    endif ;else print,'cwo_displayIQT_class::draw yrange=',yrange
    if n_elements(zoom) eq 0 then zoom = 0

    ;PLOT I(Q,t)
    ;print,'cwo_displayIQT_class::draw self.winvis=',self.winvis
;    wset,self.winvis
    ;print,'cwo_displayIQT_class::draw self.winpix=',self.winpix
    wset,self.winpix

    ;CREATE AXES
    if self.xlog eq 1 then xtitle = 'log(t (ns))' else xtitle = 't (ns)'
    if self.ylog eq 1 then ytitle = 'log(I(Q,t))' else ytitle = 'I(Q,t)'


;042705
;SOME NEW GLITCH IN plot OCCURRED THAT FORCED ME TO ADD THE
;xrange AND yrange EXPLICITLY.
;EVERYTHING HAD BEEN WORKING FINE AND SUDDENLY THAT
;CHANGED.

;CHECK PROBLEM WITH xrange,yrange AS DATA?
;PLOT: Warning: Infinite plot range.
;PLOT: Data range for axis has zero length.
;Execution halted at: cwo_displayIQT_class::DRAWONE  584 C:\Documents and Settings\kneller\My Documents\DAVE_SOURCEFORGE\dave\programs\NSE\cwo_displayiqt__define.pro
;                     CWO_DISPLAYIQT_EVENT_FUNC 1225 C:\Documents and Settings\kneller\My Documents\DAVE_SOURCEFORGE\dave\programs\NSE\cwo_displayiqt__define.pro
;                     WIDGET_PROCESS_EVENTS
;                     $MAIN$

    plot,(*(self.xrange)),(*(self.yrange)),/nodata,$
            xrange = (*(self.xrange)),$
            yrange = (*(self.yrange)),$
            xlog=self.xlog,$
            title=self.plottitle,$
            xtitle=xtitle,$
            ylog=self.ylog,$
            ytitle=ytitle,$
            background = -1, $
            color=0L,xstyle=1,ystyle=1,$
            xticks = 5, yticks = 5;,$
            ;xticklayout=2,yticklayout=2

    FORMAT = '(", P",I2,"=",E9.2)'
    if self.stepindex eq 0 then begin

        r=150L & g=0L & b=0L
        color = r + 256L*(g + 256L*b)
        symsize = 2
        oplot,tauvals,reform(iqt[0,*],sz[2]),$
                color = color,psym=8,symsize=symsize
        oplot,tauvals,reform(iqtfit[0,*],sz[2]),$
                color = color,psym=0,symsize=symsize


        for jj = 0,n_elements(tauvals)-1 do $
                plots,[tauvals[jj],tauvals[jj]], $
                        [iqt[0,jj]-siqt[0,jj], iqt[0,jj]+siqt[0,jj]],$
                        color=color,/data
        ;    xyouts, 0.8,0.9,$
        ;            'Q'+strtrim(string(0),2), /normal,$
        ;            color = color

        ;CREATE THE LEGEND
        plotxrange = !x.crange & plotyrange = !y.crange
        ;xnorm = 0.8 & ynorm = 0.91

        xnorm = self.xlegend & ynorm = self.ylegend

        ;NEW USAGE OF nse_legend    01/27/05
        legendText = 'Q='+strtrim(string(qvals[0],format=qformat),2)+'/'+STRING("305B)
        for ifp = 0,szfitparms[2]-1 do begin
            ls = string(ifp,fitparms[0,ifp],format=FORMAT)
            legendText = legendText+ls
        endfor;ifp

        nse_legend,self.winpix,xnorm,ynorm,$
                    legendText,$;'Q='+strtrim(string(qvals[0],format=qformat),2)+'/'+STRING("305B),$
                    color=color,psym=8,$
                    symsize=symsize

        ;    nse_legend,self.winpix,xnorm,ynorm,plotxrange,plotyrange, $
        ;                    'Q'+strtrim(string(0),2),$
        ;                    xlog=self.xlog,ylog=self.ylog,$
        ;                    color=color,psym=8,symsize=symsize



    endif else begin
        for i=self.stepindex,self.stepindex do begin

            ;STORE COLORS
            ;tvlct,r,g,b,/get

        ;        rnew = bindgen(256)
        ;        gnew = bindgen(256)
        ;        bnew = bindgen(256)

            case i mod 6 of
            0:begin
        ;            gnew *= 0 & bnew *= 0
                rnew = 255 & gnew = 0 & bnew = 0
            end
            1:begin
        ;            rnew *= 0 & bnew *= 0
                rnew = 0 & gnew = 255 & bnew = 0
            end
            2:begin
        ;            gnew *= 0
                rnew = 0 & gnew = 0 & bnew = 255
            end
            3:begin
        ;            rnew *= 0 & gnew *= 0
                rnew = 155 & gnew = 155 & bnew = 0;255 & bnew = 0
            end
            4:begin
        ;            rnew *= 0
                rnew = 255 & gnew = 0 & bnew = 255
            end
            5:begin
        ;            bnew *= 0
                rnew = 0 & gnew = 255 & bnew = 255
            end
            else:begin
            end
            endcase

        device,decomposed=0
            tvlct,rnew,gnew,bnew
        device,decomposed = 1

            psym = i mod 6
            if i gt 2 then psym = psym + 2
            color = 50 + 100*(psym mod 3);-1*color - 75*psym

            ;r=150L & g=0L & b=0L
            color = rnew + 256L*(gnew + 256L*bnew)

            oplot,tauvals,reform(iqt[i,*],sz[2]),$
                    color=color,$
                    psym=psym,symsize=symsize
            oplot,tauvals,reform(iqtfit[i,*],sz[2]),$
                    color = color,psym=0,symsize=symsize

            for jj = 0,n_elements(tauvals)-1 do $
                plots,[tauvals[jj],tauvals[jj]], $
                        [iqt[i,jj]-siqt[i,jj], iqt[i,jj]+siqt[i,jj]],$
                        color=color,/data


            ;111904
            ;NORMALIZE THESE COORDINATES SO oplot CAN BE USED
            ;TO ADD SYMBOLS TO THE LEGENDS.

        ;        xnorm = 0.8 & ynorm = 0.91-0.05*i
            xnorm = self.xlegend & ynorm = self.ylegend - 0.04*i
            ;NEW USAGE OF nse_legend    01/27/05
            legendText = 'Q='+strtrim(string(qvals[i],format=qformat),2)+'/'+STRING("305B)
            for ifp = 0,szfitparms[2]-1 do begin
                ls = string(ifp,fitparms[i,ifp],format=FORMAT)
                legendText = legendText+ls
            endfor;ifp
                    nse_legend,self.winpix,xnorm,ynorm,$
                        legendText,$;'Q='+strtrim(string(qvals[i],format=qformat),2)+'/'+STRING("305B),$
                        color=color,psym=psym,$
                        symsize=symsize
        ;        nse_legend,self.winpix,xnorm,ynorm,plotxrange,plotyrange, $
        ;                        'Q'+strtrim(string(i),2),$
        ;                        xlog=self.xlog,ylog=self.ylog,$
        ;                        color=color,psym=psym,symsize=symsize

        ;        ;RESTORE COLORS
        ;        ;print,'Restore CT'
        ;        tvlct,r,g,b


        endfor;i
    endelse
    if zoom ne 0 then begin
        plots,[self.rpress[0],self.rpress[0]],$
                [self.rpress[1],self.rmotion[1]],$
                color=0L,/data
        plots,[self.rpress[0],self.rmotion[0]],$
                [self.rmotion[1],self.rmotion[1]],$
                color=0L,/data
        plots,[self.rmotion[0],self.rmotion[0]],$
                [self.rmotion[1],self.rpress[1]],$
                color=0L,/data
        plots,[self.rmotion[0],self.rpress[0]],$
                [self.rpress[1],self.rpress[1]],$
                color=0L,/data
    endif;zoombox

    wset,self.winvis
    ;COPY IMAGE IN VIRTUAL WINDOW TO CURRENT WINDOW
    device,copy = [0,0,!d.x_size,!d.y_size,0,0,self.winpix]

;GET AXIS POINTERS
if ptr_valid(self.xptr) ne 0 then ptr_free,self.xptr
if ptr_valid(self.yptr) ne 0 then ptr_free,self.yptr
self.xptr = ptr_new(!x)
self.yptr = ptr_new(!y)

device,decomposed = dc

end;drawOne

;#############################################################
function cwo_displayIQT_class::normalize,x,y,xrange,yrange,xlog,ylog
;
;NAME:
;        cwo_displayIQT_class::normalize
;
;PURPOSE:
;       CREATE NORMALIZED (i.e. 0-1) COORDINATES FOR THE INPUT x,y VALUES
;PARAMETERS:
;       x       x coord
;       y       y coord
;       xrange  xrange for the window
;       yrange  yrange for the window
;       xlog    log or lin
;       ylog    log or lin
;KEYWORDS:
;       none
;RETURN VALUE:
;       [xnorm,ynorm]   THE NORMALIZED COORDINATE PAIR.

    ;PRODUCE A NORMALIZED COORDINATE BASED ON THE x,y
    ;POSITIONS AND xrange,yrange.
    ;
    ;THE PURPOSE OF THIS IS TO GIVE THE COORDINATE OF THE
    ;NORMALIZED POSITINO ON THE PLOT.
    ;
    ;e.g. x = 0.5, y=0.5
    ;

    ;xlog = self.xlog & ylog = self.ylog

;    if xlog eq 1 then xrange = 10.0^xrange
;    if ylog eq 1 then yrange = 10.0^yrange

    ;print,'cwo_displayIQT_class::normalize xrange=',xrange
    ;print,'cwo_displayIQT_class::normalize yrange=',yrange


    x1 = xrange[1] & x0 = xrange[0]
    y1 = yrange[1] & y0 = yrange[0]

    xnorm = x*(x1-x0) + x0
    ynorm = y*(y1-y0) + y0

    if xlog eq 1 then xnorm = 10.0^xnorm
    if ylog eq 1 then ynorm = 10.0^ynorm

    return,[xnorm,ynorm]
end;normalize


;#############################################################
pro cwo_displayIQT_class::makefit,expdecay=expdecay,stretchexp=stretchexp,apparentdiffusion=apparentdiffusion


;
;NAME:
;        cwo_displayeIQT::makefit
;
;PURPOSE:
;       Get the fit parameters for the selected fit function.
;PARAMETERS:
;       none
;KEYWORDS:
;       none


    fitchoice = 0
    if n_elements(expdecay) eq 0 then expdecay = 0
    if n_elements(stretchexp) eq 0 then stretchexp = 0
    if n_elements(apparentdiffusion) eq 0 then apparentdiffusion = 0

    ;SET UP A DEFAULT
    if expdecay eq 1 then fitchoice = 0
    if stretchexp eq 1 then fitchoice = 1
    if apparentdiffusion eq 1 then begin
            fitchoice = 2
            ;print,'cwo_displayecho::makefit fitchoice=',fitchoice
    endif;apparentdiffusion

    ;if expdecay eq 0 and stretchexp eq 0 then fitchoice = 0



    n = self.datacontainer->count()-1
    ref = self.datacontainer->get(position=n)

    ;NOW USE REF TO CALCULATE THE I(Q,t) VALUES AND
    ;ADD THE CALCULATED VALUES TO THE DATA ARRAY OR
    ;HAVE A SEPARATE LOCAL datafit ARRAY

    ;help,*(self->getproperty(/datafit))


    ;NOW USE THE DATA OBJECT TO
    ;CALCULATE ITS OWN FITS BASED ON THE FIT OPTION PASSED TO
    ;THE ooEcho::fitIQT METHOD
    ;
    ;   e.g. dumfit = ooEchoRef->fitIQT(stretchExp=0,expDecay=0,custom='P[0]+P[1]*exp(-1.0*P[2]*t)')
    ;
    ;           THEN ooEcho::fitIQT SAVES THE PARMS FOR EACH Q IN A POINTER VARIABLE IN ooEcho.
    ;           THAT INFORMATION IS QUERIED EACH TIME cwo_displayIQT NEEDS THE INFO.
    ;           ooEcho SHOULD STORE VALUES OF FIT POINTS AND THE FIT PARAMETERS.
    ;           PERHAPS cwo_displayIQT DOESN'T EVENT NEED LOCAL COPIES SINCE IT
    ;           HAS THE OBJECT REFERENCES!!!!!
    ;
    ;           TOMORROW, 1) WRITE THE FIT METHOD.
    ;                     2) UPDATE cwo_displayIQT TO EXTRACT DATA INFO DIRECTLY FROM ooEcho REFS.
    ;                     3) ADD A widget_combobox TO SELECT THE CURRENT DATA OBJECT'S I(Q,t)
    ;                        TO DISPLAY.
    ;                     4) MULTIPLE ooEcho OBJECT SELECTION POSSIBLE???
    ;
    ;


    ;GET PARAMETERS FOR ALL CURVES.
; EXP DECAY
    case fitchoice of
    0:begin;'expdecay'
        dumparms = ref->fitIQT(stretchExp=0,expDecay=1,dumparms_error=dumparms_error,$
                                                    custom='P[0]+P[1]*exp(-1.0*P[2]*t)')
        if ptr_valid(self.datafitparms) gt 0 then ptr_free,self.datafitparms
        self.datafitparms = ptr_new(dumparms)
        sz = size(dumparms)
        ;t IS KEPT TRACK OF ELSEWHERE, AND IT IS
        ;THE FOURIER TIMES MULTIPLIED BY 10^9
        t = (*(ref->getProperty(tag='fourierTime')))
        datafitvals = dblarr(sz[1],n_elements(t))
        if ptr_valid(self.datafit) then ptr_free,self.datafit
        for i=0,sz[1]-1 do begin
            datafitvals[i,*] = nse_expdecay(t,dumparms[i,*])
        endfor
        self.datafit = ptr_new(datafitvals)
    end;expdecay
    1:begin;stretch
        dumparms = ref->fitIQT(stretchExp=1,expDecay=0,dumparms_error=dumparms_error,$
                                                    custom='P[0]+P[1]*exp(-1.0*P[2]*t)')
        if ptr_valid(self.datafitparms) gt 0 then ptr_free,self.datafitparms
        self.datafitparms = ptr_new(dumparms)
        sz = size(dumparms)
        t = (*(ref->getProperty(tag='fourierTime')))
        Q = (*(ref->getProperty(tag='qarcs')))
        datafitvals = dblarr(sz[1],n_elements(t))
        if ptr_valid(self.datafit) then ptr_free,self.datafit
        for i=0,sz[1]-1 do begin
            X = t*((Q[i])^(2))
            datafitvals[i,*] = nse_stretchexpdecay(X,dumparms[i,*])
        endfor
        self.datafit = ptr_new(datafitvals)
    end;stretch
    2:begin;apparentdiffusion
        dumparms = ref->fitIQT(stretchExp=0,expDecay=0,apparentdiffusion=1,dumparms_error=dumparms_error,$
                                                    custom='P[0]+P[1]*exp(-1.0*P[2]*t)')
        if ptr_valid(self.datafitparms) gt 0 then ptr_free,self.datafitparms
        self.datafitparms = ptr_new(dumparms)
        sz = size(dumparms)
        ;t IS KEPT TRACK OF ELSEWHERE, AND IT IS
        ;THE FOURIER TIMES MULTIPLIED BY 10^9
        t = ((10.0d)^(9.0d))*(*(ref->getProperty(tag='fourierTime')))
        Q = (*(ref->getProperty(tag='qarcs')))
        datafitvals = dblarr(sz[1],n_elements(t))
        if ptr_valid(self.datafit) then ptr_free,self.datafit
        for i=0,sz[1]-1 do begin
            X = t*((Q[i])^(2))
            datafitvals[i,*] = nse_expdecay(((10.0d)^(-9.0d))*X,dumparms[i,*])
        endfor
        self.datafit = ptr_new(datafitvals)
    end;apparentdiffusion
    else:begin;defaults to exp decay
        dumparms = ref->fitIQT(stretchExp=0,expDecay=1,dumparms_error=dumparms_error,$
                                                    custom='P[0]+P[1]*exp(-1.0*P[2]*t)')
        if ptr_valid(self.datafitparms) gt 0 then ptr_free,self.datafitparms
        self.datafitparms = ptr_new(dumparms)

        sz = size(dumparms)


        t = (*(ref->getProperty(tag='fourierTime')))
        Q = (*(ref->getProperty(tag='qarcs')))

        datafitvals = dblarr(sz[1],n_elements(t))
        if ptr_valid(self.datafit) then ptr_free,self.datafit


        for i=0,sz[1]-1 do begin
            X = t*((Q[i])^(2))
            datafitvals[i,*] = nse_expdecay(X,dumparms[i,*])
        endfor
        self.datafit = ptr_new(datafitvals)
    end ;else
    endcase

    ;print,*(ref->getproperty(tag='IQT'))

    ;print,*(ref->getproperty(tag='IQT')) - *self.datafit



;INFORMATION FROM addData
;        iqt = *(ref->getProperty(tag='IQT'))
;        siqt = *(ref->getProperty(tag='sIQT'))
;        tauvals = *(ref->getProperty(tag='fourierTime'))
;
;
;
;        if ptr_valid(self.data) gt 0 then ptr_free,self.data
;        if ptr_valid(self.ddata) gt 0 then ptr_free,self.ddata
;        if ptr_valid(self.datafit) gt 0 then ptr_free,self.datafit
;        if ptr_valid(self.xvals) gt 0 then ptr_free,self.xvals
;
;        self.data=ptr_new(iqt)
;        self.ddata=ptr_new(siqt)
;        self.xvals=ptr_new((10.0^9)*tauvals)
;
;
;        ;INITIALIZE DATA FIT ARRAY.
;        dumfit = 0.0*iqt
;        self.datafit = ptr_new(dumfit)




    ;NOW WHEN THE MOUSE IS NEAR THE CORRESPONDING LEGEND,
    ;DISPLAY THE FIT PARAMETERS ASSOCIATED WITH THE DATA FIT.


end;cwo_displayIQT_class::makefit

;#############################################################
pro cwo_displayIQT_class::event,event
;
;NAME:
;        cwo_displayIQT_class::event
;
;PURPOSE:
;
;PARAMETERS:
;
;KEYWORDS:
;
;_EXTRA - is used to pass keywords meant for the superclass.

end;event




;#############################################################
function cwo_displayIQT_event_func,event
;
;NAME:
;        cwo_displayIQT_event_func
;
;PURPOSE:
;       event handler
;PARAMETERS:
;       event   EVENT PRODUCED BY THE cwo
;KEYWORDS:
;       none
;
;RETURN VALUE:
;       new_event   AN EVENT WITH MORE INFO PASSED UP THE WIDGET HIERARCHY.


    s = 'Event occurred: '
    closeIQT = 0
    writeIQT = 0
    writeDAVE = 0
    writeIgor = 0
    write3Col = 0

    etype = strlowcase(tag_names(event,/structure_name))

    stash = widget_info(event.handler,/child)

    widget_control,stash,get_uvalue=self

    the_c = self->getProperty(/datacontainer)
    ref = the_c->get(/all)
    rremove = where(obj_valid(ref) eq 0)
    ;print,rremove
    if rremove[0] ne -1 then begin
        if n_elements(rremove) eq the_c->count() then begin
            the_c->remove,/all
        endif else begin
            for i=0,n_elements(rremove)-1 do begin
                the_c->remove,position=max(rremove)
            endfor;i
        endelse
    endif

    if the_c->count() gt 0 then begin
        case event.id of
        self->getProperty(/fitbut):begin
    ;        print,tag_names(event,/structure_name)
    ;        print,tag_names(event)
    ;        print,event
    ;        print,'Fitopts index =',nse_comboindex(self->getProperty(/fitopts))
    ;        print,'Have data object do fit and produce fitfunc(xvals) for each data set.'

            index = nse_comboindex(self->getProperty(/fitopts))

            ;print,'index=',index

            if index eq 0 then self->makefit,expdecay=1
            if index eq 1 then self->makefit,stretchexp=1
            if index eq 2 then self->makefit,apparentdiffusion=1
            if index eq 3 then self->makefit,expdecay=1

            self->draw        ;
            ;NOW NEED TO MAKE THIS OBJECT FIT ITSELF, RETURN FIT VALUES AND
            ;SEND THE FIT POINTS TO THE PLOT WITH LINE AS THE SYMBOL AND
            ;THE SAME COLOR AS THE DATA.
            ;

        end
        self->getProperty(/displayDButton):begin
            if widget_info(self->getproperty(/displayd),/valid_id) eq 0 then begin
                base = widget_base( group_leader=self->getproperty(/tlb),$
                                    title = 'Plot of D v Q',xsize=300,ysize=300,/floating)
                id = cwo_displayd(base,xsize=300,ysize=300,obj=obj)
                self->setproperty,displaydbase=base
                self->setproperty,displayd=id
                if obj_valid(self->getproperty(/displaydobj)) then obj_destroy,self->getproperty(/displaydobj)
                self->setproperty,displaydobj=obj

                widget_control,self->getProperty(/displayDButton),set_value="Hide D v Q Plot"
                widget_control,self->getproperty(/displaydbase),/realize
                self->draw
            endif else begin
                widget_control,self->getproperty(/displaydbase),/destroy
                widget_control,self->getProperty(/displayDButton),set_value="Plot D v Q"
            endelse
        end
        self->getproperty(/fitopts):begin
            ;print,tag_names(event,/structure_name)
            ;print,tag_names(event)
            ;print,event
            ;print,'Save the ????'
        end
        self->getProperty(/x_loglinbutton):begin
;            print,tag_names(event)
;            print,event
            self->setProperty,xlog = event.select

            s=s+'x_loglinbutton'
;            if self->getProperty(/xlog) eq 1 then begin
;                self->setProperty,xlog = 0
;                widget_control,self->getProperty(/x_loglinbutton),set_value='x log'
;    ;            widget_control,self->getProperty(/loglinstatus),set_value='Linear'
;            endif else begin
;                self->setProperty,xlog = 1
;                widget_control,self->getProperty(/x_loglinbutton),set_value='x lin'
;    ;            widget_control,self->getProperty(/loglinstatus),set_value='Linear'
;            endelse
            self->draw
        end
        self->getProperty(/x_minfield):begin
            s=s+'x_minfield'
            widget_control,self->getProperty(/y_minfield),get_value=ymin
            widget_control,self->getProperty(/y_maxfield),get_value=ymax

            widget_control,self->getProperty(/x_minfield),get_value=minval
            widget_control,self->getProperty(/x_maxfield),get_value=maxval
            xmin = min([minval,maxval])
            xmax = max([minval,maxval])
            widget_control,self->getProperty(/x_minfield),set_value=xmin
            widget_control,self->getProperty(/x_maxfield),set_value=xmax
    ;        print,(*(self->getProperty(/xrange)))
            (*(self->getProperty(/xrange)))[0] = xmin
            (*(self->getProperty(/xrange)))[1] = xmax
            ;print,(*(self->getProperty(/xrange)))
            s=s+string(minval)
            self->draw,yrange=[ymin,ymax],xrange = [xmin,xmax]
        end
        self->getProperty(/x_maxfield):begin
            s=s+'x_maxfield'
            widget_control,self->getProperty(/y_minfield),get_value=ymin
            widget_control,self->getProperty(/y_maxfield),get_value=ymax

            widget_control,self->getProperty(/x_minfield),get_value=minval
            widget_control,self->getProperty(/x_maxfield),get_value=maxval
            xmin = min([minval,maxval])
            xmax = max([minval,maxval])
            widget_control,self->getProperty(/x_minfield),set_value=xmin
            widget_control,self->getProperty(/x_maxfield),set_value=xmax
    ;        print,(*(self->getProperty(/xrange)))
            (*(self->getProperty(/xrange)))[0] = xmin
            (*(self->getProperty(/xrange)))[1] = xmax
    ;        print,(*(self->getProperty(/xrange)))
            s=s+string(maxval)
            self->draw,yrange=[ymin,ymax],xrange = [xmin,xmax]
        end
        self->getProperty(/y_loglinbutton):begin
            self->setProperty,ylog = event.select
            s=s+'y_loglinbutton'
;            if self->getProperty(/ylog) eq 1 then begin
;                self->setProperty,ylog = 0
;                widget_control,self->getProperty(/y_loglinbutton),set_value='y log'
;            endif else begin
;                self->setProperty,ylog = 1
;                widget_control,self->getProperty(/y_loglinbutton),set_value='y lin'
;            endelse
            self->draw
        end
        self->getProperty(/y_minfield):begin
            s=s+'y_minfield'
            ;042705
            ;THE NEXT LINES SHOULD BE UNNECESSARY SINCE THEY ARE STORED IN
            ;MEMBER VARIABLES.
            widget_control,self->getProperty(/x_minfield),get_value=xmin
            widget_control,self->getProperty(/x_maxfield),get_value=xmax

            widget_control,self->getProperty(/y_minfield),get_value=minval
            widget_control,self->getProperty(/y_maxfield),get_value=maxval
            ymin = min([minval,maxval])
            ymax = max([minval,maxval])
            widget_control,self->getProperty(/y_minfield),set_value=ymin
            widget_control,self->getProperty(/y_maxfield),set_value=ymax
    ;        print,(*(self->getProperty(/yrange)))
            (*(self->getProperty(/yrange)))[0] = ymin
            (*(self->getProperty(/yrange)))[1] = ymax
    ;        print,(*(self->getProperty(/yrange)))
            s=s+string(minval)
            self->draw,yrange=[ymin,ymax],xrange = [xmin,xmax]
        end
        self->getProperty(/y_maxfield):begin
            s=s+'y_maxfield'
            widget_control,self->getProperty(/x_minfield),get_value=xmin
            widget_control,self->getProperty(/x_maxfield),get_value=xmax
            widget_control,self->getProperty(/y_minfield),get_value=minval
            widget_control,self->getProperty(/y_maxfield),get_value=maxval
            ymin = min([minval,maxval])
            ymax = max([minval,maxval])
            widget_control,self->getProperty(/y_minfield),set_value=ymin
            widget_control,self->getProperty(/y_maxfield),set_value=ymax
    ;        print,'(*(self->getProperty(/yrange)))=',(*(self->getProperty(/yrange)))
            (*(self->getProperty(/yrange)))[0] = ymin
            (*(self->getProperty(/yrange)))[1] = ymax
    ;        print,'(*(self->getProperty(/yrange)))=',(*(self->getProperty(/yrange)))
            s=s+string(maxval)
            self->draw,yrange=[ymin,ymax],xrange = [xmin,xmax]
        end
        self->getProperty(/stepbutton):begin
            s=s+'stepbutton'
            self->drawOne
            self->setproperty,stepIndex = self->getproperty(/stepIndex) + 1
            ;closeIQT = 1
        end
        self->getProperty(/printbutton):begin
            wset,self->getProperty(/winvis)
            result = tvrd(true=3)
            datcontainer = (self->getProperty(/datacontainer))
            ref = datcontainer->get(position=datcontainer->count()-1)
            work_dir = ref->getProperty(tag='workdir')
            print,'work_dir=',work_dir
            work_dir = ref->getProperty(tag='datadir')
            print,'work_dir=',work_dir
            nse_ps_plotter,result,directory=work_dir,filename='IQT_Plot.ps'
        end
        self->getProperty(/closebutton):begin
            s=s+'closebutton'
            closeIQT = 1
        end
        self->getProperty(/writeDAVEbutton):begin
           writeDAVE = 0
            datcontainer = (self->getProperty(/datacontainer))
            ref = datcontainer->get(position=datcontainer->count()-1)
            work_dir = ref->getProperty(tag='datadir')
            dumval = ref->writeDAVE(work_dir = work_dir);self.work_dir)
            ref->setproperty,'datadir',work_dir
            ref->setproperty,'workdir',work_dir
        end
        self->getProperty(/writeIgorbutton):begin
            writeIgor = 0
            datcontainer = (self->getProperty(/datacontainer))
            ref = datcontainer->get(position=datcontainer->count()-1)
            work_dir = ref->getProperty(tag='datadir')
            dumval = ref->writeIQTIgor(work_dir = work_dir);self.work_dir)
            ref->setproperty,'datadir',work_dir
            ref->setproperty,'workdir',work_dir
        end
        self->getProperty(/writebutton):begin
            writeIQT = 0
            datcontainer = (self->getProperty(/datacontainer))
            ref = datcontainer->get(position=datcontainer->count()-1)
            work_dir = ref->getProperty(tag='datadir')
            ref->writeIQTASCII,work_dir
            ref->setproperty,'datadir',work_dir
            ref->setproperty,'workdir',work_dir
        end
        self->getProperty(/write3colbutton):begin
           write3Col = 0
            datcontainer = (self->getProperty(/datacontainer))
            ref = datcontainer->get(position=datcontainer->count()-1)
           work_dir = ref->getProperty(tag='datadir')
           ref->writeIQT3ColASCII,work_dir
            ref->setproperty,'datadir',work_dir
            ref->setproperty,'workdir',work_dir

        end
        self->getProperty(/draw):begin
            case etype of
            'widget_draw':begin


                if event.key ne 0 or event.ch ne 0 then begin
                    ;HANDLE KEYBOARD EVENTS IF THEY OCCUR
                    ;BY MOVING THE LEGEND COORDINATES.
                    ;L,R,U,D,pageup,pagedown,home,end
                    ;5,6,7,8,9     ,10      ,11  ,12

                    if event.press eq 1 then begin
                        case event.key of
                        5b:begin
                            self->setproperty,xlegend = self->getProperty(/xlegend) - 0.01
                        end
                        6b:begin
                            self->setproperty,xlegend = self->getProperty(/xlegend) + 0.01
                        end
                        7b:begin
                            self->setproperty,ylegend = self->getProperty(/ylegend) + 0.01
                        end
                        8b:begin
                            self->setproperty,ylegend = self->getProperty(/ylegend) - 0.01
                        end
                        9b:begin
                            self->setproperty,xlegend = 0.8
                            self->setproperty,ylegend = 0.91
                        end
                        10b:begin
                            self->setproperty,xlegend = 0.8
                            self->setproperty,ylegend = 0.3
                        end
                        11b:begin
                            self->setproperty,xlegend = 0.2
                            self->setproperty,ylegend = 0.91
                        end
                        12b:begin
        ;                    print,tag_names(event,/structure_name)
        ;                    print,tag_names(event)
        ;                    print,event
                            self->setproperty,xlegend = 0.1
                            self->setproperty,ylegend = 0.3
                        end
                        else:begin
                        end
                        endcase
                        if self->getProperty(/xlegend) ge 0.9 then self->setProperty,xlegend = 0.9
                        if self->getProperty(/xlegend) le 0.1 then self->setProperty,xlegend = 0.1
                        if self->getProperty(/ylegend) ge 1.0 then self->setProperty,ylegend = 1.0
                        if self->getProperty(/ylegend) le 0.1 then self->setProperty,ylegend = 0.1

                        self->draw
                    endif;press

                endif else begin
                    if event.press eq 4 then begin
                        ;RIGHT BUTTON PRESS RESORES ORIGINAL SCALE.
                        xrange = *(self->getProperty(/origxrange))
                        yrange = *(self->getProperty(/origyrange))
                        if ptr_valid(self->getproperty(/xrange)) ne 0 then $
                                                ptr_free,self->getproperty(/xrange)
                        self->setproperty,xrange=ptr_new(xrange)
                        self->setproperty,xmin = min(xrange)
                        self->setproperty,xmax = max(xrange)
                        if ptr_valid(self->getproperty(/yrange)) ne 0 then $
                                                ptr_free,self->getproperty(/yrange)
                        self->setproperty,yrange=ptr_new(yrange)
                        self->setproperty,ymin = min(yrange)
                        self->setproperty,ymax = max(yrange)

                        xrange = (*(self->getproperty(/xrange)))
                        yrange = (*(self->getproperty(/yrange)))

                        widget_control,self->getproperty(/x_minfield),set_value=min(xrange)
                        widget_control,self->getproperty(/x_maxfield),set_value=max(xrange)
                        widget_control,self->getproperty(/y_minfield),set_value=min(yrange)
                        widget_control,self->getproperty(/y_maxfield),set_value=max(yrange)

                        self->draw;,xrange = xrange,yrange = yrange

                    endif;right press

                    if event.press eq 1 then begin
                        !x = *(self->getproperty(/xptr))
                        !y = *(self->getproperty(/yptr))
                        self->setProperty,press = 1
                        rpress = (convert_coord(event.x,event.y,/device,/to_data))[0:1]
                        print,'rpress=',rpress
                        self->setproperty,rpress = rpress
                    endif;mouse press
                    if event.release eq 1 then begin
                        !x = *(self->getproperty(/xptr))
                        !y = *(self->getproperty(/yptr))
                        self->setproperty,press = 0
                        rrelease = (convert_coord(event.x,event.y,/device,/to_data))[0:1]
                        self->setproperty,rrelease = rrelease
                        rpress = (self->getproperty(/rpress))

                        if rpress[0] eq rrelease[0] then begin
                            xrange = *(self->getProperty(/origxrange))
                        endif else begin
                            xrange = [min([rpress[0],rrelease[0]]),max([rpress[0],rrelease[0]])]
                        endelse
                        if rpress[1] eq rrelease[1] then begin
                            yrange = *(self->getProperty(/origyrange))
                        endif else begin
                        yrange = [min([rpress[1],rrelease[1]]),max([rpress[1],rrelease[1]])]
                        endelse


                        if ptr_valid(self->getproperty(/xrange)) ne 0 then $
                                                ptr_free,self->getproperty(/xrange)
                        self->setproperty,xrange=ptr_new(xrange)
                        self->setproperty,xmin = min(xrange)
                        self->setproperty,xmax = max(xrange)
                        if ptr_valid(self->getproperty(/yrange)) ne 0 then $
                                                ptr_free,self->getproperty(/yrange)
                        self->setproperty,yrange=ptr_new(yrange)
                        self->setproperty,ymin = min(yrange)
                        self->setproperty,ymax = max(yrange)

            ;            print,'(*(self->getproperty(/xrange))=',$
            ;                            (*(self->getproperty(/xrange)))
                        xrange = (*(self->getproperty(/xrange)))

            ;            print,'(*(self->getproperty(/yrange))=',$
            ;                            (*(self->getproperty(/yrange)))
                        yrange = (*(self->getproperty(/yrange)))

                        widget_control,self->getproperty(/x_minfield),set_value=min(xrange)
                        widget_control,self->getproperty(/x_maxfield),set_value=max(xrange)
                        widget_control,self->getproperty(/y_minfield),set_value=min(yrange)
                        widget_control,self->getproperty(/y_maxfield),set_value=max(yrange)
                    self->draw;,xrange = xrange,yrange = yrange

                    endif;mouse release
                    if event.type eq 2 then begin
                        if self->getproperty(/press) ne 0 then begin
                            !x = *(self->getproperty(/xptr))
                            !y = *(self->getproperty(/yptr))
                            rmotion = (convert_coord(event.x,event.y,/device,/to_data))[0:1]

                            self->setProperty,rmotion = rmotion
                            self->draw,zoom=1
                        endif;drag
                    endif;mouse motion

                   ;NEED TO HANDLE MOTION EVENTS (RUBBER-BAND BOX ZOOMING)
                   ;AND PLACE VALUES INTO TEXT FIELDS
                   ;
                   ;RIGHT CLICK TO RETURN TO ORIGINAL SCALE.
                   ;
                   ;AND OUTSIDE THIS NEED TO ADD SYMBOLS TO LEGENDS AND
                   ;PLACE LEGENDS AROUND DATA.
                endelse;keys v. mouse
            end;;widget_draw
            'widget_tracking':begin
                if event.enter eq 1 then begin
    ;                print,'IN'
                    ;SET INPUT_FOCUS ON ENTER/EXIT SO KEYBOARD EVENTS WORK.
                    widget_control,self->getProperty(/draw),input_focus=1
                endif else begin
    ;                print,'OUT'
                    widget_control,self->getProperty(/draw),input_focus=0
                endelse



            end;widget_tracking
            else:begin
            end
            endcase;etype

        end;self->getproperty(/draw)
        self->getProperty(/tlb):begin
            print,'cwo_displayiqt_event  NEVER GET HERE?'
            print,'tlb EVENT.'
    ;        child = widget_info(self->getProperty(/tlb),/child)
    ;        self->tlbevents,event

        end
        self->getproperty(/autographNSEbutton):begin
            a = checkforobjinstance('cwo_autographnse',ref=ref)
            ;print,a
            if a eq 0 then begin
                ;base = widget_base(group_leader=self->getproperty(/tlb),title='AutographNSE',/floating)
                w = cwo_autographnse(self->getproperty(/tlb))
                ;widget_control,base,/realize
                ;xmanager,'autograph',base,event_handler='cwo_autographnse_event_pro',/no_block
            endif

        end
        else:begin
    ;print,tag_names(event,/structure_name)
    ;print,tag_names(event)
    ;print,event
    ;print,'IQT else'
        end
        endcase;event.id
    endif else begin
        ;NO OBJECT IN DISPLAYIQT
        case event.id of
        self->getProperty(/closebutton):begin
            s=s+'closebutton'
            closeIQT = 1
        end
        else:begin
        end
        endcase

    endelse

;print,tag_names(event,/structure_name)
;print,tag_names(event)
;print,event
;print,s
    newEvent = {CWO_DISPLAYIQT_EVENT,$
                    id:self->getProperty(/tlb),$
                    top:event.top,$
                    handler:event.handler,$
                    closeIQT:closeIQT,$
                    writeIQT:writeIQT,$
                    writeDAVE:writeDAVE,$
                    writeIgor:writeIgor,$
                    write3Col:write3Col}

    return,newEvent
end;event

;pro cwo_autographnse_event_pro,event
;    ;do nothing
;
;    ;print,event.id
;
;end;cwo_autographnse_event_pro


;#############################################################
pro cwo_displayIQT_class::removeData,n
;
;NAME:
;        cwo_displayIQT_class::removeData
;
;PURPOSE:
;       REMOVE OBJECTS FROM THE DATA CONTAINER.
;PARAMETERS:
;       n   The object position to remove.
;KEYWORDS:
;       none

    self.datacontainer->remove,position=n

end;removeData
;#############################################################
pro cwo_displayIQT_class::addData,ref,init=init
;
;NAME:
;        cwo_displayIQT_class::addData
;
;PURPOSE:
;       Add a data object to this object's container.
;       Update the parameters of the object for display.
;PARAMETERS:
;       ref     The data reference to add.
;KEYWORDS:
;       init    Run the draw method if init is set.


        ;SIMPLY ADD ref TO THE DATACONTAINER AND UPDATE THE
        ;WIDGET TO DISPLAY DATA FROM ref.


        ;AS A FIRST PASS, SIMPLY REPLACE THE CURRENTLY DISPLAYED
        ;DATA WITH THE NEW DATA.

        self.datacontainer->add,ref,position=self.datacontainer->count()

        iqt = *(ref->getProperty(tag='IQT'))
        siqt = *(ref->getProperty(tag='sIQT'))
        tauvals = *(ref->getProperty(tag='fourierTime'))



        if ptr_valid(self.data) gt 0 then ptr_free,self.data
        if ptr_valid(self.ddata) gt 0 then ptr_free,self.ddata
        if ptr_valid(self.datafit) gt 0 then ptr_free,self.datafit
        if ptr_valid(self.xvals) gt 0 then ptr_free,self.xvals

        self.data=ptr_new(iqt)
        self.ddata=ptr_new(siqt)
        self.xvals=ptr_new((10.0^9)*tauvals)


        x_1 = min(*self.xvals)
        x_2 = max(*self.xvals)

;060605
        if ptr_valid(self.xrange) gt 0 then ptr_free,self.xrange
        self.xrange = ptr_new([double(x_1),double(x_2)])
        if ptr_valid(self.yrange) gt 0 then ptr_free,self.yrange



        yrange = [min(iqt-siqt),max(iqt+siqt)]
        self.yrange = ptr_new(yrange)

        if yrange[0] le 0 then begin
            widget_control,self.y_loglinbutton,set_value='y log'
            self.ylog = 0
        endif


        self.origxmin = double(x_1)
        self.origxmax = double(x_2)
        self.origymin = yrange[0]
        self.origymax = yrange[1]



        ;INITIALIZE DATA FIT ARRAY.
        dumfit = 0.0*iqt
        self.datafit = ptr_new(dumfit)


        sz = size(*self.data)
        ;print,'displayecho::addData sz[1],sz[2]=',sz[1],sz[2]
        if ptr_valid(self.datafitparms) gt 0 then ptr_free,self.datafitparms
        self.datafitparms = ptr_new(dblarr(sz[1],2))

        title = file_basename(ref->getProperty(tag='filename'))+': I(Q,t)'
        widget_control,self.parent,base_set_title=title
        self.plottitle = title


        if n_elements(init) eq 0 then begin
            self->draw
        endif



end;addData




;#############################################################
pro cwo_displayIQT_class::resize,event,xsize=xsize,ysize=ysize,lastresize=lastresize;
;NAME:
;        cwo_displayIQT_class::resize
;
;PURPOSE:
;       Resize method to handle resize events.
;PARAMETERS:
;       event   The tlbresize event that got us here.  Gives me event.x,y directly.
;KEYWORDS:
;       xsize   The new xsize of the cwo.
;       ysize   The new ysize of the cwo.
;       lastresize  Flag to call resize recursively if 0, or
;                   to not call it if it is 1


;021605
;
;I CAN'T FIGURE OUT WHAT TO DO HERE FOR THE MAC RESIZING.
;COME BACK TO THIS AT A LATER DATE.


;011305
;
;STEAL THIS RESIZE EVENT HANDLER FROM THE ooDisplayEcho CODE.
;
;UPDATE THE CODE BELOW FOR cwo_displayiqt
;
;THIS MUST BE CALLED FROM THE PARENT EVENT HANDLER, SINCE
;THE BASE CREATED THERE WILL BE RESPONSIBLE FOR RESIZING
;THE WIDGET BELOW IT IN THE HIERARCHY.
;
;THIS HANDLER SHOULD DETERMINE THE SIZE OF THE PARENT AND
;RESIZE THE PARTS OF THIS cwo APPROPRIATELY.


;TOMORROW!!!


;011405
;
;ACTUALLY, WHAT THIS SHOULD DO IS GET XSIZE, YSIZE FROM EITHER THE
;PARENT BASE OR FROM ARGUMENTS, AND SET THE SIZES OF ITS COMPONENTS
;APPROPRIATELY.


;021505
;JUST AS WITH oodisplayecho ON THE MAC, THE GEOMETRY IS
;NOT UPDATED UNTIL AFTER THE EVENT COMPLETES ITSELF,
;SO THESE SIZES MUST BE RECEIVED FROM THE event
;TO WORK PROPERLY ON THE MAC.

;
;FIRST TRY TO RUN THIS HANDLER TWICE.



    if !Version.OS eq 'Win32' then begin
        xpadding = 8
        ypadding = 30
        magicxpadding = 25
        magicypadding = 0
	xmacmagic = 35
	ymacmagic = 5
    endif else begin
        xpadding = 10
        ypadding = 10
        magicxpadding = 25
        magicypadding = 30
	xmacmagic = 53
	ymacmagic = 40
    endelse


;040505
;COPYING THIS FROM oodisplayecho
            if !version.os eq 'Win32' then begin
                xresizemagic = 0
                yresizemagic = 0
            endif else begin
                xresizemagic = 6;10
                yresizemagic = 6;40
            endelse

            r = get_screen_size()
                                            ;013105
            if event.x lt 350 then begin    ;DROP SIZES FROM 800x600 TO 770x500 FOR MAC X11
                parentxsize=350
            endif else begin
                if event.x ge (r[0]-50) then begin
                    parentxsize=r[0]-50
                endif else begin
                    parentxsize=event.x-xresizemagic
                endelse
            endelse
            if event.y lt 450 then begin
                parentysize=450
            endif else begin
                if event.y ge (r[1]-50) then begin
                    parentysize=r[1]-50
                endif else begin
                    parentysize=event.y-yresizemagic
                endelse
            endelse


;;031105
;;MINIMIZE THE DRAW WIDGET ON A TLB EVENT ON THE MAC SO
;;THAT IF THE USER SHRINKS THE WINDOW, IT DOESN'T BUMP
;;THE EDGES AND CAUSE ANOTHER RESIZE EVENT.
;;NOTE: WHEN ACTUALLY ON THE MAC, TLB RESIZE EVENTS ARE _DRAGS_,
;;	WHEREAS ON WINDOWS THE EVENT OCCURS AT THE END OF THE DRAG.
;    if !version.os eq 'Win32' then begin
;            ;widget_control,self.draw,xsize=self.xllb2,ysize=self.yllb2
;	endif else begin
;;            widget_control,self.draw,xsize=event.x-150-xmacmagic-1,$
;;                                ysize=event.y-ymacmagic-1
            widget_control,self.draw,xsize=parentxsize-150-xmacmagic-1,$
                                ysize=parentysize-ymacmagic-1
;	endelse




;
;
;    self.filetreeysize = min([500,self.atlbysize - self.ypadding])
;    self.tabsysize = min([500,self.atlbysize - self.ypadding])



;021605
;COMMENTING OUT
;    ;SET xsize,ysize
;    pgeom = widget_info(self.parent,/geometry)
;    pxsize = pgeom.xsize
;    pysize = pgeom.ysize


    ;THIS RESIZE COMMAND SEEMS NECESSARY, BUT I DON'T CURRENTLY SEE WHY.
    ;SEE IF IT WORKS ON A MAC NOW.
    widget_control,self.parent,xsize=parentxsize,ysize=parentysize


;021605
;COMMENTING OUT
;    if n_elements(xsize) eq 0 then xsize = pgeom.xsize
;    if n_elements(ysize) eq 0 then ysize = pgeom.ysize

;021605
;REPLACING WITH

;040705
    xsize = parentxsize;event.x
    ysize = parentysize;event.y


    ;SET SIZE MEMBER VARIABLES
    self.xllb2 = xsize - self.xllb1 - 2*xpadding
    if (xsize - self.xllb1) gt 0 then begin
        self.xllb2 = xsize - self.xllb1 - 2*xpadding
    endif else begin
        self.xllb2 = 10 ;WHAT IS A GOOD VALUE?
    endelse
    self.yllb1 = ysize - ypadding
    self.yllb2 = ysize - ypadding



;                parent:0L,              $  RESIZED OUTSIDE THIS OBJECT.
;                tlb:0L,                 $  RESIZE
;                llb1:0L,                $
;                llb2:0L,                $  RESIZE
;                xllb1:150,              $
;                yllb1:300,              $  RESIZE
;                xllb2:150,              $  RESIZE
;                yllb2:300,              $  RESIZE


;021505
;COMMENT OUT THE NEXT 3 LINES.
;    widget_control,self.tlb, xsize=xsize-xpadding, ysize=ysize-ypadding
;    widget_control,self.llb1,xsize=self.xllb1,ysize=self.yllb1
;    widget_control,self.llb2,xsize=self.xllb2,ysize=self.yllb2


    ;DOES THIS COMMAND WORK ON THE MAC?????
;    widget_control,self.draw,xsize=self.xllb2,ysize=self.yllb2

;;REPLACE IT WITH THE FOLLOWING:
;    widget_control,self.draw,/destroy
;    self.draw=widget_draw(self.llb2,xsize=self.xllb2,ysize=self.yllb2)




;WHY ISN'T self.winvis SET ELSEWHERE??????
;    widget_control,self.draw,value=id
;    self.winvis = id

    if n_elements(lastresize) eq 0 then lastresize=0
    if !Version.OS ne 'ANYTHING' and lastresize eq 0 then begin;'Win32' and lastresize eq 0 then begin
        ;self->resizeWidgets,event,lastresize=1
        self->resize,event,xsize=xsize,ysize=ysize,lastresize=1
    endif else begin

        widget_control,self.draw,xsize=self.xllb2,ysize=self.yllb2

        ;CREATE PIXMAP WINDOW FOR DOUBLE BUFFERING OF IMAGE WINDOW
        wdelete,self.winpix
	tlbgeom = widget_info(self.parent,/geom)
	dgeom = widget_info(self.draw,/geometry)
;        window,/free,/pixmap,$;xsize=tlbgeom.xsize-150-xmacmagic,$
;			dgeom.xsize-xmacmagic,ysize=dgeom.ysize-ymacmagic

;NOTE: WHEN ACTUALLY ON THE MAC, TLB RESIZE EVENTS ARE _DRAGS_,
;	WHEREAS ON WINDOWS THE EVENT OCCURS AT THE END OF THE DRAG.
    if !version.os eq 'Win32' then begin
;            window,/free,/pixmap,xsize=self.xllb2,ysize=self.yllb2
;            self.winpix = !d.window
;            widget_control,self.draw,xsize=self.xllb2,ysize=self.yllb2
            window,/free,/pixmap,xsize=tlbgeom.xsize-150-xmacmagic,$
				ysize=tlbgeom.ysize-ymacmagic
            self.winpix = !d.window
            widget_control,self.draw,xsize=tlbgeom.xsize-150-xmacmagic,$
                                ysize=tlbgeom.ysize-ymacmagic
	endif else begin
            window,/free,/pixmap,xsize=tlbgeom.xsize-150-xmacmagic,$
				ysize=tlbgeom.ysize-ymacmagic
            self.winpix = !d.window
            widget_control,self.draw,xsize=tlbgeom.xsize-150-xmacmagic,$
                                ysize=tlbgeom.ysize-ymacmagic
	endelse

        self->draw
    endelse


;I CAN'T FIGURE OUT WHAT TO DO HERE FOR THE MAC.
;COME BACK TO THIS AT A LATER DATE.


end;resize                  ;tlbEvents

;######################################################################################
;LIFE CYCLE METHODS
;######################################################################################
;#############################################################
pro cwo_displayiqt_killedme,id

;
;NAME:
;        cwo_displayIQT_killedme
;
;PURPOSE:
;       Catch X-out EVENTS FOR cwo_displayiqt AND DESTROY THE SINGLETON.
;PARAMETERS:
;       id  The widget id of the first child.
;KEYWORDS:
;       none


    ;print,"IQT WAS X'ED OUT. Destroying the object so a new one can be created later."
    print,'cwo_displayIQT cleanup: cwo_displayIQT_killedme'
    widget_control,id,get_uvalue=obj
    obj_destroy,obj

end;cwo_iqt_displayiqt_killedme,id

;#############################################################
pro cwo_displayIQT_realize,id
;
;NAME:
;        cwo_displayIQT_realize
;
;PURPOSE:
;       realize method.
;PARAMETERS:
;       id  the widget id.
;KEYWORDS:
;       none

    ;SET winvis ON REALIZATION OF cwo.

    widget_control,id,get_uvalue=obj

    ;help,obj

    widget_control,obj->getProperty(/draw),get_value=winvis
    ;print,'cwo_displayIQT_realize winvis=',winvis
    obj->setproperty,winvis=winvis

end;cwo_displayIQT_realize
;#############################################################
pro cwo_displayIQT_class::cleanup
;
;NAME:
;        cwo_displayIQT_class::cleanup
;
;PURPOSE:
;       cleanup method for cwo_displayIQT_class
;PARAMETERS:
;       none
;KEYWORDS:
;       none

    self.datacontainer->remove,/all
    if obj_valid(self.datacontainer) then obj_destroy,self.datacontainer
    if ptr_valid(self.data) ne 0 then ptr_free,self.data
    if ptr_valid(self.ddata) ne 0 then ptr_free,self.ddata
    if ptr_valid(self.xvals) ne 0 then ptr_free,self.xvals
    if ptr_valid(self.xrange) ne 0 then ptr_free,self.xrange
    if ptr_valid(self.yrange) ne 0 then ptr_free,self.yrange
    if ptr_valid(self.npts) ne 0 then ptr_free,self.npts
    if ptr_valid(self.filename) ne 0 then ptr_free,self.filename
    if ptr_valid(self.xptr) ne 0 then ptr_free,self.xptr
    if ptr_valid(self.yptr) ne 0 then ptr_free,self.yptr

end;cleanup

;#############################################################
function cwo_displayIQT_class::init, parent, datRef, $
                                uvalue=uvalue,$
                                xsize=xsize,$
                                ysize=ysize
;
;NAME:
;        cwo_displayIQT_class::init
;
;PURPOSE:
;       init method
;PARAMETERS:
;       parent  paren of the cwo
;       datRef  ooEcho object with I(Q,t) data.
;KEYWORDS:
;       uvalue  uvalue for the cwo
;       xsize   xsize for the cwo
;       ysize   ysize for the cwo
;
;RETURN VALUE:
;       tlb     id of the top level base.



    ;AT THIS POINT, THIS OBJECT EXISTS, EVEN ON THE FIRST INSTANTIATION.
    objcount = checkForObjInstance('cwo_displayIQT_class',ref=dispobj)
    if objcount gt 1 then begin
;        print,'ANOTHER cwo_displayIQT_class ALREADY EXISTS.'
;        print,'NEED TO BE CAREFUL WITH THIS SINCE '
;        print,'THE CALL TO CREATE THE cwo IS DONE AS A FUNCTION.'
;        print,' RETURN 0 FOR THE WIDGET ID. AND DON"T INSTANTIATE'
;        print,' THE OBJECT.  ADD NEW DATA TO ALREADY-EXISTING OBJECT.'



        ;AT THIS POINT I ADD THE DATA TO THE EXISTING OBJECT
        ;AND THEN RETURN 0.

        ;HOW DO I KNOW THAT THE EXISTING OBJECT IS THE ZEROTH OBJECT???
        ;IS IT JUST LUCKY THAT THIS WORKS EVERY TMIE SO FAR???

        if obj_valid(dispobj[0]->getProperty(/datacontainer)) gt 0 then begin
            dispobj[0]->addData,datRef
        endif else begin
            dispobj[1]->addData,datRef
        endelse


        return,0L
    endif

    self.xpadding = 25
    self.ypadding = 25


    ;DEFINE WIDGETS IN THIS OBJECT WIDGET
    ;print,self.tlb
    device,get_screen_size=sz
;052305
;COMMENT OUT THE NEXT LINE
;    print,'IQT Display: get_screen_size=',sz
    self.parent = parent
;    self.tlb = widget_base(parent,xsize = 400, ysize = 300, /row,$
    self.tlb = widget_base(parent,/row,$;xsize = sz[0]/2, ysize = sz[1]/2,$
                            event_func='cwo_displayIQT_event_func',$
                            uvalue=uvalue)

;    print,'self.tlb=',self.tlb
;
;    print,'self->getProperty(/tlb)=',self->getProperty(/tlb)
    xllb1 = 150
    yllb1 = sz[1]/2

    self.xllb1 = xllb1
    self.yllb1 = yllb1

    llb1 = widget_base(self.tlb, /column, $; xsize=xllb1, ysize=yllb1,$
                            kill_notify = 'cwo_displayIQT_killedme',$
                            uvalue = self)

    self.llb1 = llb1

    xllb2 = (sz[0]/2)-xllb1
    yllb2 = sz[1]/2
    self.xllb2 = xllb2
    self.yllb2 = yllb2
    llb2 = widget_base(self.tlb);, /column);xsize=xllb2, ysize=yllb2)
    self.llb2=llb2



    ;ADD LIST OF FUNCTIONS TO FIT ALL I(Q,t) CURVES.

    self.fitbut = widget_button(llb1,value='Fit I(Q,t)');Function')
    fitlist = ['A*exp(-D(Q)*t)','A*exp(-(D(Q)*(Q^2)*t)^B)','A*exp(-D(Q)*(Q^2)*t)','default']
    self.fitopts = widget_combobox(llb1,value = fitlist)


;120205
;    self.x_loglinbutton = widget_button(llb1,value='x log')
    self.x_loglinbutton = cw_bgroup(llb1,'log(t)',/nonexclusive,/col)



    self.xlog=0
    self.ylog=1

;CONVERT TIMES TO ns
;    x_1 = 10.0^(-11)
;    x_2 = 5.0*(10.0^(-8))
    x_1 = 0.01
    x_2 = 50.0

    xrange = [double(x_1),double(x_2)]
    yrange = [double(0.1),double(2.0)]

    self.origxmin = double(x_1)
    self.origxmax = double(x_2)
    self.origymin = yrange[0]
    self.origymax = yrange[1]

    self.xrange = ptr_new(xrange)
    self.yrange = ptr_new(yrange)

    self.origxrange = ptr_new(xrange)
    self.origyrange = ptr_new(yrange)

    self.x_minfield = cw_field(llb1,title='xmin',/floating,$
                                value=strtrim(string(x_1),2),$
                                /return_events)
    self.x_maxfield = cw_field(llb1,title='xmax',/floating,$
                                value=strtrim(string(x_2),2),$
                                /return_events)

;    self.y_loglinbutton = widget_button(llb1,value='y lin')
    self.y_loglinbutton = cw_bgroup(llb1,'log(I)',/nonexclusive,/col,set_value=[1])


    self.y_minfield = cw_field(llb1,title='ymin',/floating,$
                                value=strtrim(string(yrange[0]),2),$
                                /return_events)
    self.y_maxfield = cw_field(llb1,title='ymax',/floating,$
                                value=strtrim(string(yrange[1]),2),$
                                /return_events)

    self.stepbutton = widget_button(llb1,value='Step through curves')
    ;self.displaydbutton = widget_button(llb1,value='Plot D v Q')
    self.writeDAVEbutton = widget_button(llb1,value='write to DAVE')
    self.writeIgorbutton = widget_button(llb1,value='write ASCII files',$
                                            tooltip='One Q value per file. (USE FOR AutographNSE in DAVE or Igor.)')
    self.writebutton = widget_button(llb1,value='write ASCII file',$
                                            tooltip='All Q values in one file.')
    self.write3Colbutton = widget_button(llb1,value='write 3Col ASCII files',$
                                            tooltip='One Q value per file.')
                                            ;tooltip='All Q values in one file.')
    self.autographNSEbutton = widget_button(llb1,value='Plot and Fit D(Q)',$
                                                tooltip='AutographNSE')


    self.printbutton = widget_button(llb1,value='print')
    self.closebutton = widget_button(llb1,value='close')


;    drawtemp =    widget_draw(llb2,xsize=300, ysize=300,$

    self.xlegend = 0.8 & self.ylegend = 0.91

    xsize=xllb2
    ysize=yllb2
    drawtemp =    widget_draw(llb2,xsize=xsize-self.xpadding, ysize=ysize-self.ypadding,$
                        notify_realize='cwo_displayIQT_realize',$
                        /motion_events,/button_events,$
                        keyboard_events=1,tracking_events=1);,$
                        ;tooltip='THIS IS THE TOOL TIP'+string(13B)+ $
                        ;        'FOR THE DISPLAYITQ draw Widget.');,$
                        ;uvalue=self)

    dgeom = widget_info(drawtemp,/geometry)

    window,/free,/pixmap,xsize=xsize-self.xpadding,ysize=ysize-self.ypadding

;	magic = 50
;    window,/free,/pixmap,xsize=dgeom.xsize-magic,ysize=dgeom.ysize-magic
    self.winpix = !d.window

    ;print,'drawtemp=',drawtemp
    self.draw = drawtemp
    ;print,'self.draw=',self.draw
    ;self.winpix = winpix
;    help,self
;    help,self.data



    self.datacontainer = obj_new('IDL_container')
    self->addData,datRef,/init



    widget_control,self.draw,set_uvalue=self





    return,1
end;init

;#############################################################
pro cwo_displayIQT_class__define,class
;
;NAME:
;        cwo_displayIQT_class__define
;
;PURPOSE:
;       Define object structure and inherit 'myproperties'.
;
;       SUGGESTION:  WHY DID'T I INHERIT IDL_Container AND MAKE THIS AND isa INSTEAD OF A hasa?
;
;PARAMETERS:
;       class   Parameter to allow use of myproperties to get the
;               object's tag names.
;KEYWORDS:
;       none
;




    class = {cwo_displayIQT_class,               $
                inherits myproperties,  $
                datacontainer:obj_new(),$   ;CONTAINER OF ALL THE DATA OBJECTS.
                data:ptr_new(),         $   ;POINTER TO A 2D ARRAY
                ddata:ptr_new(),        $   ;POINTER TO THE 2D ARRAY ERROR BARS
                datafit:ptr_new(),      $   ;POINTER TO A 2D ARRAY OF FIT VALUES
                datafitparms:ptr_new(), $   ;POINTER TO THE DATA FIT PARAMETERS.
                xvals:ptr_new(),        $   ;POINTER TO THE X-VALUES
                npts:ptr_new(),         $   ;NUMBER OF POINTS IN EACH DATA OBJECT
                filename:ptr_new(),     $   ;ARRAY OF FILENAMES FOR LEGENDS
                plottitle:'',           $
                parent:0L,              $
                displayd:0L,            $   ;A WIDGET TO DISPLAY THE D VALUES OF THE FITS TO THE DATA.
                displaydobj:obj_new(),  $   ;THE OBJECT REFERENCE FOR displayd
                xptr:ptr_new(),         $
                yptr:ptr_new(),         $
                tlb:0L,                 $
                llb1:0L,                $
                llb2:0L,                $
                xllb1:150,               $   ;SIZES FOR THE SUBBASES
                yllb1:300,               $
                xllb2:150,               $
                yllb2:300,               $
                xpadding:25,$
                ypadding:25,$
                draw:0L,                $
                winvis:0L,              $
                winpix:0L,              $
                x_loglinbutton:0L,      $
                y_loglinbutton:0L,      $
                xlog:1,                 $
                ylog:0,                 $
                $;declaring arrays here messes up myproperties__getProperty!!! ---- THIS WAS FIXED!!!
                xrange:ptr_new(),       $;xrange:dblarr(2),       $
                origxrange:ptr_new(),   $
                xmin:0.0,               $
                xmax:100.0,             $
                origxmin:0.0,           $
                origxmax:100.0,         $
                logxmin:0.1,            $
                logxmax:10.0,           $
                yrange:ptr_new(),       $;dblarr(2),       $
                origyrange:ptr_new(),   $
                ymin:0.0,               $
                ymax:100.0,             $
                origymin:0.0,           $
                origymax:100.0,         $
                logymin:1e-11,          $
                logymax:1e-7,           $
                stepbutton:0L,          $
                stepIndex:0,            $
                fitbut:0L,              $
                fitopts:0L,             $
                x_minfield:0L,          $
                x_maxfield:0L,          $
                y_minfield:0L,          $
                y_maxfield:0L,          $
                xlegend:0.8,            $;X AND Y NORMALIZED COORDINATES
                ylegend:0.91,           $;FOR THE LEGENDS.
                press:0,                $
                release:0,              $
                rpress:[0.0,0.0],       $;MOUSE COORDINATES DURING EVENTS
                rrelease:[0.0,0.0],     $
                rmotion:[0.0,0.0],      $
                displayDButton:0L,      $
                displayDBase:0L,        $
                closebutton:0L,         $
                printbutton:0L,         $
                writebutton:0L,         $
                write3Colbutton:0L,     $
                writeDAVEbutton:0L,     $
                writeIgorbutton:0L,     $
                autographNSEbutton:0L   $
            }
end;cwo_displayIQT_class__define



;############################################################
function cwo_displayIQT,parent,datRef,$
                            obj_ref=obj_ref,$
                            uvalue=uvalue,$
                            _Extra=extra
;
;NAME:
;        cwo_displayIQT
;
;PURPOSE:
;       Function to create the cwo.
;PARAMETERS:
;       parent  The parent widget id.
;       datRef  The ooEcho object with the I(Q,t) data.
;KEYWORDS:
;       obj_ref[out]    The object reference for this cwo.
;       uvalue[in]      The uvalue for this cwo.
;
;_EXTRA - is used to pass keywords meant for the superclass.

;RETURN VALUE:
;       tlb:    THE TOP LEVEL BASE ID FOR THIS cwo.
;

    ;FUNCTION TO RETURN THE BASE ID FROM THE OBJECT
    o = obj_new('cwo_displayIQT_class',parent,datRef,uvalue=uvalue);,_Extra=extra)

    if obj_valid(o) gt 0 then begin

        obj_ref=o   ;SETS VALUE OF obj_ref IN FUNCTION CALL
                    ;TO ALLOW USER OF PROGRAM ACCESS TO THE
                    ;OBJECT REFERENCE.


        x = o->getProperty(/tlb)

        return,o->getProperty(/tlb)
    endif else begin
        return,0L
    endelse

end;cwo_displayIQT


