;$Id$
;###############################################################################
;
; NAME:
;  PLOT_GROUP
;
; PURPOSE:
;  Plot wrapper for fixed-window scan reduction tool.
;
; CATEGORY:
;  DAVE, HFBS, data reduction
;
; AUTHOR:
;   Robert M. Dimeo, Ph.D.
;   NIST Center for Neutron Research
;   100 Bureau Drive
;   Gaithersburg, MD 20899
;   Phone: (301) 975-8135
;   E-mail: robert.dimeo@nist.gov
;   http://www.ncnr.nist.gov/staff/dimeo
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used of if the code in this file is
;  included in another product.
;
;###############################################################################
;+
; NAME:
;       PLOT_GROUP
;
; PURPOSE:
;
;		Simple procedure wrapper for the PLOT command in IDL.  Suitable
;		for invocation at the command line.
;
; AUTHOR:
;
;       Robert M. Dimeo, Ph.D.
;		NIST Center for Neutron Research
;       100 Bureau Drive
;		Gaithersburg, MD 20899
;       Phone: (301) 975-8135
;       E-mail: robert.dimeo@nist.gov
;       http://www.ncnr.nist.gov/staff/dimeo
;
; CATEGORY:
;
;       Data reduction, DAVE project, HFBS
;
; CALLING SEQUENCE:
;
;		PLOT_GROUP,ODATA,DETECTOR
;
; PARAMETERS:
;
;		ODATA:		Object reference for the data to be plotted.  This could be an
;					array or a single object reference.  If it is an array then
;					the subsequent plot is an overplot of all of the data sets
;					that were specified.
;
;		DETECTOR:	Detector value to be plotted (def: 1)
;
;
; KEYWORDS:
;
;		This procedure accepts any of the keywords common to the PLOT procedure
;		in IDL.
;
;		AUTOSCALE:	If set then the program decides on a range for plotting data.
;
; COMMON BLOCKS:
;
;		None
;
; REQUIREMENTS:
;
;		None
;
;
; MODIFICATION HISTORY:
;
;       Written by Rob Dimeo, July 10, 2003
;-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
pro plot_group,oarray,detector,autoscale = autoscale,_Extra = extra
if n_params() eq 0 then return
if n_params() eq 1 then detector = 1
if n_elements(autoscale) eq 0 then autoscale = 1
ndata = n_elements(oarray)
symbol = [1,2,4,5,6,7,0,0,0,0]
linestyle = [0,0,0,0,0,0,0,1,2,3]
; Extract the autoscaled viewing range
for i = 0,ndata-1 do begin
	ret = oarray[i]->get(indep_var = indep_var)
	ret = oarray[i]->set(detector = detector)
	ret = oarray[i]->get(data = data,error = error)
	y = reform(data[detector,*]) & dy = reform(error[detector,*])
	qty = *indep_var.pqty
	min_qty = min(qty,max = max_qty)
	min_y = min(y,max = max_y)
	if i eq 0 then begin
		ylo = min_y
		yhi = max_y
		xlo = min_qty
		xhi = max_qty
	endif else begin
		xlo = [xlo,min_qty]
		xhi = [xhi,max_qty]
		ylo = [ylo,min_y]
		yhi = [yhi,max_y]
	endelse
endfor
xmax = max(xhi) & xmin = min(xlo)
ymax = max(yhi) & ymin = min(ylo)
deltax = 0.1*(xmax-xmin)
deltay = 0.1*(ymax-ymin)
if autoscale eq 1 then begin
	xrange = [xmin-deltax,xmax+deltax]
	yrange = [ymin-deltay,ymax+deltay]
endif else begin
	countx = 0 & county = 0
	if n_elements(extra) ne 0 then begin
		these_tags = tag_names(extra)
		xok = where(these_tags eq 'XRANGE',countx)
		yok = where(these_tags eq 'YRANGE',county)
	endif
	if countx eq 0 then xrange = [xmin-deltax,xmax+deltax] else $
		xrange = extra.xrange
	if county eq 0 then yrange = [ymin-deltay,ymax+deltay] else $
		yrange = extra.yrange
endelse
ret = oarray[0]->get(data = data,error = error,indep_var = indep_var)
y = reform(data[detector,*]) & dy = reform(error[detector,*])
x = *indep_var.pqty
xtitle = indep_var.label
strout = 'Detector: '+strtrim(string(detector),2)
plot,x,y,psym = symbol[0],linestyle = linestyle[0],xrange = xrange, $
	/xsty,yrange = yrange,/ysty, $
	xtitle = xtitle,title = strout,ytitle = 'Intensity (arb units)'
errplot,x,y-dy,y+dy,width = 0.0

if ndata gt 1 then begin
	for i = 1,ndata-1 do begin
		ret = oarray[i]->get(data = data,error = error,indep_var = indep_var)
		y = reform(data[detector,*]) & dy = reform(error[detector,*])
		x = *indep_var.pqty
		oplot,x,y,psym = symbol[i],linestyle = linestyle[i]
		errplot,x,y-dy,y+dy,width = 0.0
	endfor
endif

return
end