; $Id$
; Written by J.R.D. Copley.
;************************************************************************************************
;###############################################################################
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;###############################################################################
;
;************************************************************************************************
pro DCSToolsPlanner_cleanup,tlb
;************************************************************************************************
;
compile_opt strictarr
widget_control,tlb,get_uvalue = pState
obj_destroy,(*pState).plotWin
ptr_free,(*pState).npsptr
ptr_free,pState
;
end


;************************************************************************************************
pro DCSToolsPlanner_exitbutton,event
;************************************************************************************************
;
compile_opt strictarr
;
; Respond to the "EXIT" button.  Destroy the widget.
;
widget_control,event.top,/destroy
end

;************************************************************************************************
pro DCSToolsPlanner_printbutton,event
;************************************************************************************************
;
compile_opt strictarr
;
; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'DCSToolsPlanner_printbutton: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif


; Respond to the "Print" button.
;
widget_control,event.top,GET_UVALUE=pState
(*pState).plotWin->saveas,'printer'
;
end



;************************************************************************************************
pro DCSToolsPlanner_t2ebutton_handler,event
;************************************************************************************************
;
compile_opt strictarr
;
widget_control,event.top,/destroy
end


;************************************************************************************************
pro DCSToolsPlanner_t2ebutton,event
;************************************************************************************************
;
compile_opt strictarr
;
common modifiable_stuff,mspeed,rmode,israt,sdenom,tsdm,samperiod,nout,tchanmethod
;
; Respond to "Time to energy" button.
;
widget_control,event.top,get_uvalue=pState
wl=(*pState).wl
tchanmethod=(*pState).tchanmethod
nout=(*pState).nout
tsdmin=float(tsdm)
econvert=!dcs_hsq2mn/(!dcs_hom/!dcs_dsd)^2
tsd0=wl/!dcs_hom*!dcs_dsd
e0=econvert/tsd0^2
dt_VME=0.05
;nin=samperiod/dt_VME
;nin=round(nin)
;
tsdmax=tsdmin+samperiod
;
emax=econvert/tsdmin^2
emin=econvert/tsdmax^2
;
; This statement is included in order to force compilation of procedures in
; the file dcs_make_tlookup.pro
dcs_make_tlookup,-1.0
;
case tchanmethod of
    0: dcs_make_tlookup_fixed_dt,samperiod,nout,dt_VME,tsdmin,tstart,tstop
    1: dcs_make_tlookup_fixed_de,emin,emax,nout,econvert,dt_VME,tstart,tstop
    else: stop
endcase
;
minframe=-5
nframes=6
repeat begin
    tsdmaxframe=tsdmin+minframe*samperiod+samperiod
    minframe=minframe+1
endrep until tsdmaxframe gt 0.0
minframe=minframe-1
maxframe=minframe+nframes-1
nch=n_elements(tstart)
contents=strarr(nframes,nch+1)
for frame=minframe,maxframe do begin
    time=[tstart[0],tstop]+frame*samperiod
    contents[frame-minframe,*]=string(e0-econvert/time^2,format='(f8.3)')
    indices=where(time le 0.0,count)
    if (count gt 0) then contents[frame-minframe,indices]="---"
endfor
;
column_labels="Frame"+strcompress(string(indgen(nframes)+minframe))
row_labels=["Chn. 1 start","Chn."+strcompress(string(indgen(nch)+1))+" end"]
tlb=widget_base(title="Time channel to energy transfer",$
       /col,/modal,group_leader=event.top)
    rowb=widget_base(tlb,/row)
    leftb=widget_base(rowb,/col)
    void=widget_label(leftb,value="Wavelength ="+string(wl))
    void=widget_label(leftb,value="SR denominator ="+string((*pState).sdenom))
    rightb=widget_base(rowb,/col)
    void=widget_label(rightb,value="t_SDmin ="+string(tsdmin))
    void=widget_label(rightb,value="D_SD ="+string(!dcs_dsd))
    void=widget_label(tlb,value="Energy transfers in meV")
    void=widget_table(tlb,xsize=nframes,ysize=nch+1,$
       x_scroll_size=nframes,y_scroll_size=19,$
       column_widths=[intarr(nframes)+50,5],column_labels=column_labels,$
       alignment=1,row_labels=row_labels,value=contents)
    done=widget_button(tlb,value="DONE")
centertlb,tlb
geom=widget_info(tlb,/geometry)
widget_control,leftb,xsize=0.5*geom.xsize
widget_control,tlb,/realize
;
xmanager,"DCSToolsPlanner_t2ebutton",tlb,$
    event_handler="DCSToolsPlanner_t2ebutton_handler"
;
end


;************************************************************************************************
pro DCSToolsPlanner_tchanbutton,event
;************************************************************************************************
;
compile_opt strictarr
;
common modifiable_stuff,mspeed,rmode,israt,sdenom,tsdm,samperiod,nout,tchanmethod
;
; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'DCSToolsPlanner_tchanbutton: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif


; Respond to "Save tchan widths" button.
;
widget_control,event.top,get_uvalue=pState
wl=(*pState).wl
tchanmethod=(*pState).tchanmethod
nout=(*pState).nout
tsdmin=float(tsdm)
fname=dialog_pickfile(title='Name of file in which to save time channel widths',file='twid.dat',filter='*.dat', $
                      path=(*pState).workDir)
warnings=strarr(20)
warnings[0]="No file written."
if (fname ne "") then begin
    dcs_make_tlookup,wl,samperiod,tchanmethod,nout,tsdmin,fname,imsg,nch
    if (imsg eq 0) then begin
       warnings[0]="File has been written."
       warnings[1:2]=["Total valid VME channels =",strcompress(long(total(nch[0:nout-1])))]
       warnings[3:4]=["Total of all VME channels =",strcompress(long(total(nch)))]
    endif else begin
       case imsg of
         1: warnings[1]="Unacceptable period."
       2: warnings[1:2]=["Unacceptable minimum ","t-o-f from sample to detector."]
         else:
       endcase
    endelse
endif
widget_control,(*pState).warningstext,set_value=warnings
end


;************************************************************************************************
pro DCSToolsPlanner_textbuttons,event
;************************************************************************************************
;
compile_opt strictarr
;
; Respond to parameter changes, to the "Recalculate" button, to the "Save parameters" button, or
; to the "Restore parameters" button.
;
common modifiable_stuff,mspeed,rmode,israt,sdenom,tsdm,samperiod,nout,tchanmethod
;common zoom_related,x1,x2,x1r,x2r,drawzoom,dozoom,zoomed,zdrawn,xlim,ylim,bangx,bangy
;
; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'DCSToolsPlanner_textbuttons: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif


;Get information from widget.
widget_control,event.top,GET_UVALUE=pState
;
saving=event.id eq (*pState).savpb
restoring=event.id eq (*pState).respb
otherwise=not saving and not restoring
npsptr=(*pState).npsptr
;
if (restoring) then begin
    ;if (!operating_system eq "Windows") then $
    ;   directory="c:\rsi\dave\dcs\text files\" else directory=""
    filename=dialog_pickfile(path=(*pState).workDir,$
       file="plannerfile.pln",/read,filter="*.pln",/fix_filter)
    if (filename ne "") then begin
       restore,filename=filename
       widget_control,(*pState).lambdatext,set_value=string(planpar.wl)
       widget_control,(*pState).rattext,set_value=string(planpar.erat)
       widget_control,(*pState).phiminslider,set_value=string(planpar.phimin)
       widget_control,(*pState).dphislider,set_value=string(planpar.dphi)
       widget_control,(*pState).phimaxslider,set_value=string(planpar.phimax)
       widget_control,(*pState).speedtext,set_value=string(planpar.ms)
       widget_control,(*pState).sratdroplist,set_droplist_select=planpar.israt
       widget_control,(*pState).sratslider,set_value=string(planpar.sdenom)
       widget_control,(*pState).rmodeslider,set_value=string(planpar.rmode)
       widget_control,(*pState).tsdmtext1,set_value=string(planpar.tsdm)
       widget_control,(*pState).tsdmslider,set_value=string(planpar.tsdm)
       widget_control,(*pState).tsdmdroplist,set_droplist_select=planpar.dtmet
       widget_control,(*pState).tsdmtext3,set_value=string(planpar.nout)
    endif
endif
;
; Read wavelength from widget.
; Convert wavelength to Angstroms. Store in state variable.
widget_control,(*pState).lambdatext,GET_VALUE=temp
wl_read=float(temp[0])
wl=wl_read
(*pState).wl=wl
;
; Read ratio (E0-Efmax)/E0, required to determine plot range. Store in state variable.
widget_control,(*pState).rattext,GET_VALUE=temp
rat=float(temp[0])
(*pState).rat=rat
;
; Read min, step and max values of phi. Store in state variable.
widget_control,(*pState).phiminslider,GET_VALUE=temp
phimin=float(temp[0])
(*pState).phimin=phimin
widget_control,(*pState).dphislider,GET_VALUE=temp
dphi=float(temp[0])
(*pState).dphi=dphi
widget_control,(*pState).phimaxslider,GET_VALUE=temp
phimax=float(temp[0])
(*pState).phimax=phimax
;
; Obtain number of final energies for the plot.
nef=(*pState).nef
;
; Read chopper system master speed. Store in state variable.
widget_control,(*pState).speedtext,GET_VALUE=temp
mspeed=fix(temp[0])
(*pState).mspeed=mspeed
;
; Read resolution mode (should be 1, 2 or 3). Store in state variable.
widget_control,(*pState).rmodeslider,GET_VALUE=temp
rmode=fix(temp[0])
(*pState).rmode=rmode
;
; Read chopper system frame overlap speed ratio denominator.
; Also read choice between '(m-1)/m' and '1/m' method of computing numerator.
; Store this choice and the denominator in state variable.
widget_control,(*pState).sratslider,GET_VALUE=temp
sdenom=temp[0]
israt=widget_info((*pState).sratdroplist,/DROPLIST_SELECT)
(*pState).israt=israt
(*pState).sdenom=sdenom
;
if (event.id eq (*pState).lambdatext) then begin
    (*pState).tsdm_maxval=(*pState).wl*1000
    widget_control,(*pState).tsdmslider,set_slider_max=fix((*pState).tsdm_maxval)
endif
;
; Read minimum proposed time of flight from sample to detectors. Store in state variable.
if (event.id eq (*pState).tsdmslider) then begin
    widget_control,(*pState).tsdmslider,GET_VALUE=temp
    widget_control,(*pState).tsdmtext1,SET_VALUE=strcompress(temp,/remove_all)
endif else begin
    widget_control,(*pState).tsdmtext1,GET_VALUE=temp
    temp=long(temp[0])
    temp=temp > 0 < (*pState).tsdm_maxval
    widget_control,(*pState).tsdmslider,SET_VALUE=temp
endelse
temp=temp>1
tsdm=float(temp)
tsdmrat=tsdm/(6.0e7/mspeed)
if (abs(tsdmrat-round(tsdmrat)) lt 0.001) then begin
endif
(*pState).tsdm=tsdm
;
; Read choice between 'constant dt' and 'constant dE' methods of computing time channels.
; Store this choice in state variable.
tchanmethod=widget_info((*pState).tsdmdroplist,/DROPLIST_SELECT)
(*pState).tchanmethod=tchanmethod
;
; Read number of valid time channels. Store in state variable.
widget_control,(*pState).tsdmtext3,GET_VALUE=temp
nout=temp[0]
(*pState).nout=nout
;
if (saving) then begin
    planpar={wl:wl_read,erat:rat,phimin:phimin,dphi:dphi,phimax:phimax,$
    ms:mspeed,israt:israt,sdenom:sdenom,rmode:rmode,tsdm:tsdm,dtmet:tchanmethod,nout:nout}
    ;if (!operating_system eq "Windows") then $
    ;   directory="c:\rsi\dave\dcs\text files\" else directory=""
    filename=dialog_pickfile(path=(*pState).workDir, $
       file="plannerfile.pln",/write,filter="*.pln",/fix_filter)
    if (filename ne "") then begin
      if (strmid(filename,3,/reverse) ne ".pln") then filename=filename+".pln"
      save,filename=filename,planpar
    endif
endif
;
; Call DCSToolsPlanner_plot_kar to plot the kinematically allowed region, etc.
DCSToolsPlanner_plot_kar,pState
;
; Put information back into widget.
; (The DCSToolsPlanner_plot_kar routine [q.v.] may have modified some of the
; information from the widget.)
widget_control,event.top,SET_UVALUE=pState
;
end


;************************************************************************************************
pro DCSToolsPlanner_plot_kar,pState
;************************************************************************************************
;
compile_opt strictarr
;
; Plot kinematically allowed region, possibly modify some of the input parameters,
; generating warning messages, and compute and display useful derived quantities.
;

common constants,conversion
common energies,e0
common dependent_values,messagetext
common dependent_stuff,periodstext,elimitstext,warningstext
common modifiable_values,rattext,phiminslider,phimaxslider,$
    speedtext,rmodeslider,sratslider,tsdmtext1
common modifiable_stuff,mspeed,rmode,israt,sdenom,tsdm,samperiod,nout,tchanmethod
;common zoom_related,x1,x2,x1r,x2r,drawzoom,dozoom,zoomed,zdrawn,xlim,ylim,bangx,bangy
;

wl=(*pState).wl
phimin=(*pState).phimin
dphi=(*pState).dphi
phimax=(*pState).phimax
rat=(*pState).rat
nef=(*pState).nef
npsptr=(*pState).npsptr
;
e0=!dcs_hsq2mn/wl^2
;
; If necessary, redefine rat.
if (rat le 1.0) then rat=1.0
;
; If necessary swap phimin and phimax
if (phimin gt phimax) then begin
    phitemp=phimin
    phimin=phimax
    phimax=phitemp
endif
; Redefine phimax (its value may or may not change).
phimax=phimin+dphi*fix(round((phimax-phimin)/dphi))
if (phimax gt 180) then phimax=phimax-dphi
;
warnings=strarr(20)
iwarning=0
warnings[0]='WARNINGS/INFORMATION'
;
; Make sure master speed is between 1200 and 20000 rpm.
if (not (1200 le mspeed and mspeed le 20000)) then begin
    mspeed=1200>mspeed<20000
    iwarning=iwarning+1
    warnings[iwarning]='Illegal master speed --> corrected.'
endif
;
; Make sure resolution mode is 1, 2 or 3.
if (not (1 le rmode and rmode le 3)) then begin
    iwarning=iwarning+1
    warnings[iwarning]='Illegal resolution mode --> corrected.'
    rmode=1>rmode<3
endif
;
; Make sure speed ratio denominator is between 1 and 25.
; Compute speed ratio numerator.
if (not (1 le sdenom and sdenom le 25)) then begin
    iwarning=iwarning+1
    warnings[iwarning]='Illegal speed ratio denominator'
    iwarning=iwarning+1
    warnings[iwarning]='             --> corrected.'
    sdenom=1>sdenom<25
endif
if (sdenom lt 3) then snum=1 else begin
    if (israt eq 1) then snum=1 else snum=sdenom-1
endelse
;
; Compute master period.
mperiod=float(60*1e6/mspeed)
; Compute time between pulses at sample.
samperiod=mperiod*sdenom
; Compute period and speed of frame overlap chopper.
sc_period=samperiod/snum
sc_speed=float(60*1e6/sc_period)
if (not (1200 le sc_speed and sc_speed le 20000)) then begin
    iwarning=iwarning+1
    warnings[iwarning]='Illegal frame overlap chopper speed.'
    iwarning=iwarning+1
    warnings[iwarning]='*** Speed ratio has been reset to 1/1.'
    snum=1
    sdenom=1
    samperiod=mperiod
    sc_period=samperiod
    sc_speed=mspeed
endif
;
tsdm=fix(tsdm)
;
; Compute scattered neutron energies corresponding to
; first and last time channels.
ehigh=!dcs_hsq2mn/(!dcs_hom*tsdm/!dcs_dsd)^2
elow=!dcs_hsq2mn/(!dcs_hom*(tsdm+samperiod)/!dcs_dsd)^2
; Print warning message if incident neutron energy does not
; lie between "ehigh" and "elow".
if (not (elow lt e0 and e0 lt ehigh)) then begin
    iwarning=iwarning+1
    warnings[iwarning]='Unexpected minimum time sample-detector.'
    iwarning=iwarning+1
    warnings[iwarning]=' (Elastic time of flight is excluded).'
    iwarning=iwarning+1
    warnings[iwarning]=' Problem has NOT been corrected.'
endif
;
; Determine plot limits.
efmax=e0*rat
qmax=sqrt((e0+efmax-2*sqrt(e0*efmax)*cos(phimax*!dtor))*conversion)
xrange=[e0-efmax,e0]
yrange=[0,qmax]
;
; Plot y axis at x=0 (elastic energy)
(*pState).plotWin->setproperty,xdat=[0,0],ydat=[1e-2,1e2],linestyle=4,xtitle='E0-Ef (meV)',$
    ytitle='Q (A-1)',xrange=xrange,yrange=yrange,/nodraw;,/hidelegend
(*pState).plotWin->remove_text
; plot lines at ehigh and/or elow, corresponding to first and last time channels respectively.
(*pState).plotWin->add_plot,[e0-ehigh,e0-ehigh],[1e-2,1e2],linestyle=0,color='red'
(*pState).plotWin->add_plot,[e0-elow,e0-elow],[1e-2,1e2],linestyle=0,color='red'
; Plot more lines corresponding to later frames.
for k=2,4 do begin
    elowk=!dcs_hsq2mn/(!dcs_hom*(tsdm+k*samperiod)/!dcs_dsd)^2
    (*pState).plotWin->add_plot,[e0-elowk,e0-elowk],[1e-2,1e2],linestyle=2,color='red'
endfor
;
; Scattered energies and energy transfers.
ef=indgen(nef)*efmax/(nef-1)
et=e0-ef
;
; For each value of phi determine a set of Q values and plot locus.
for phi=phimin,phimax+dphi*0.01,dphi do begin
    cphi=cos(phi*!dtor)
    q=sqrt((e0+ef-2*sqrt(e0*ef)*cphi)*conversion)
    (*pState).plotWin->add_plot,et,q
endfor
; Write some stuff on the plot.
text="Wavelength ="+strcompress(wl)+'!C'+$
     "T_SD(min)   ="+strcompress(tsdm)+'!C'+$
     "Speed ratio = "+strcompress(string(snum)+'/'+string(sdenom),/remove_all)+'!C'+$
     (["Low","Medium","High"])[0>(rmode-1)<2]+' resolution'
(*pState).plotWin->add_text,text,0.58,0.82,fontsize=10
;
; If zoom rectangle is to be drawn, draw it.
;zdrawn=0
;if (state.zoom.drawzoom) then begin
;   xx=[x1[0],x1[0],x2[0],x2[0],x1[0]]
;   yy=[x1[1],x2[1],x2[1],x1[1],x1[1]]
;   plots,xx,yy,linestyle=7,color=1
;   state.zoom.zdrawn=1
;endif
;
; Update various quantities.
widget_control,rattext,set_value=string(rat,format='(f6.3)')
widget_control,phiminslider,set_value=phimin
widget_control,phimaxslider,set_value=phimax
;
; Minimum and maximum values of Q for elastic scattering.
qelmin=sqrt(2*e0*(1-cos(phimin*!dtor))*conversion)
qelmax=sqrt(2*e0*(1-cos(phimax*!dtor))*conversion)
;
; Determine flux at sample and elastic energy resolution.
flux=dcs_jcc_flux(wl,rmode,npsptr)
flux=flux/sdenom*(20000/mspeed)
elres=dcs_jcc_resn(wl,wl,rmode,mspeed)
;
; Update message section.
widget_control,messagetext,set_value=['lambda = '+string(wl,format='(f6.3)')+' A',$
    '  (i.e. '+string(wl*0.1,format='(f6.4)')+' nm)',$
    'k0 = '+string(2*!pi/wl,format='(f6.3)')+' A-1',$
    '  (i.e. '+string(20*!pi/wl,format='(f6.2)')+' nm-1)',$
    ' ',$
    'Flux at sample: ',$
    string(round(flux),format='(i6)')+' n/cm2/s',$
    'Elastic energy resolution',$
    string(elres,format='(f7.1)')+' ueV',$
    ' ',$
    'E0 = '+string(e0,format='(f7.3)')+' meV',$
    'Elastic scattering Q range:',$
    'Q_min = '+string(qelmin,format='(f6.3)')+' A-1',$
    'Q_max = '+string(qelmax,format='(f6.3)')+' A-1']
;
; Update information about speeds and periods etc.
widget_control,speedtext,set_value=strcompress(mspeed,/remove_all)
widget_control,rmodeslider,set_value=rmode
widget_control,sratslider,set_value=sdenom
widget_control,tsdmtext1,set_value=strcompress(tsdm,/remove_all)
widget_control,periodstext,set_value=[$
    'Master period = '+string(long(mperiod),format='(i7)')+' us',$
    'Period at sample = '+string(long(samperiod),format='(i7)')+' us',$
    'Slow chopper period = '+string(long(sc_period),format='(i7)')+' us',$
    'S.C. speed ratio = '+strcompress(string(snum)+'/'+string(sdenom)),$
    'S.C. speed = '+string(long(sc_speed),format='(i7)')+' rpm']
widget_control,elimitstext,set_value=['Range of energy transfers',$
    'min E0-Ef = '+string(e0-ehigh,format='(f8.2)')+' meV',$
    'max E0-Ef = '+string(e0-elow,format='(f8.2)')+' meV']
;
for k=0,5 do begin
    if (abs(float(tsdm-samperiod*k)) le 1.0) then $
       res=dialog_message("WARNING: T_SD(min) is a multiple of the period at the sample.")
endfor
;
; Write any warning messages.
widget_control,warningstext,set_value=warnings
;
ass=3.594   ; assumed lattice constant of stainless steel
hm=3.956    ; h/m
h22m=81.8   ; h^2/(2*m)
dsd=4010.0  ; distance sample to detectors

tau0=wl/hm
tel=tau0*dsd
e0=h22m/wl^2
; Draw Detector spurion lines
if (*pState).spurionflag[0] then begin
   h2k2l2=[3,4,8,11,12,16]
   nref=n_elements(h2k2l2)
   dhkl=ass/sqrt(h2k2l2)
   arg=wl/(2*dhkl)
   index=where(arg gt 1.0/sqrt(2.0) and arg lt 1.0,count)
   et=fltarr(nref)+!values.F_NaN
   delta=fltarr(nref)
   if (count gt 0) then begin
      ttheta=2*asin(arg[index])
      delta[index]=2*(ttheta-!pi/2)
     t=tel+tau0*2*dsd*sin(delta[index]/2.0)
     tauf=t/dsd
     wlf=tauf*hm
     et[index]=e0-h22m/wlf^2
   endif
   index=where(finite(et),count)
   if (count gt 0) then begin
      et=et[index]
      for i=0,count-1 do $
          (*pState).plotWin->add_plot,[et[i],et[i]],[1e-2,1e2],linestyle=2,$
                  color='blue',legend='Detector Spurion: '+dm_to_string(et[i])+' meV'  
   endif
endif
;
; Draw Chopper spurion lines
if (*pState).spurionflag[1] then begin
   period=6.0e7/mspeed
   integ=-3+indgen(7)
   tf=tel+integ*period
   et=e0-h22m/(hm/dsd*tf)^2
   index=where(tf gt 0.0,count)
   if (count gt 0) then begin
      et=et[index]
      for i=0,count-1 do $
          (*pState).plotWin->add_plot,[et[i],et[i]],[1e-2,1e2],linestyle=2,$
                  color='cyan',legend='Chopper Spurion: '+dm_to_string(et[i])+' meV'  
   endif
endif
;
(*pState).plotWin->draw
end


;************************************************************************************************
pro DCSToolsPlanner_handler,event
;************************************************************************************************
;
common constants,conversion
common energies,e0
common modifiable_stuff,mspeed,rmode,israt,sdenom,tsdm,samperiod,nout,tchanmethod
common cursor_related,postext
;
compile_opt strictarr
;
; Basic error Handler
if (n_elements(!debug) && (!debug eq 0)) then begin
    catch, catchError
    if (catchError ne 0) then begin
        ;;print, 'Error handled!'
        eTitle = 'DCSToolsPlanner_handler: Error encountered'
        eMsg = 'An error or unusual condition was encountered!'
        eMsg = [eMsg,'Please, report the following to the DAVE team:']
        eMsg = [eMsg,!error_state.msg]
;    eMsg = [eMsg,'Function/module name: calledfunc()',!error_state.msg]
        void = dialog_message(/error,eMsg,title=eTitle,dialog_parent=event.top)
        catch, /cancel
        return
    endif
endif


IF DAVE_SET_FOCUS(EVENT) THEN RETURN
;
widget_control,event.handler,get_uvalue=pState
;handle the plotting window events
if tag_names(event,/structure_name) eq 'DM_PLOT_MOTION' then begin
   result=[event.xval,event.yval]
   ptext=strarr(13)
   if total(finite(result)) eq 2 then begin
      ; Define display information for E0-Ef and Q.
      ptext[0]='E0-Ef = '+string(result[0],format='(f9.3)')+' meV'
      ptext[1]='Q = '+string(result[1],format='(f6.3)')+' A-1'
      ;
      ; If Ef>0 define display information for Ef.
      ef=e0-result[0]
      q=result[1]
      if (ef gt 0.0) then begin
         ptext[2]='Ef = '+string(ef,format='(f9.3)')+' meV'
         cphi=(-q^2/conversion+e0+ef)/(2*sqrt(e0*ef))
         ; If computed cos(phi) satisfies -1<cos(phi)<1, define
         ; display information for phi.
         if (abs(cphi) le 1.0) then begin
            phi=acos(cphi)*180/!pi
            ptext[3]='phi = '+string(phi,format='(f6.2)')+' deg'
         endif
         wl0=sqrt(!dcs_hsq2mn/e0)
         wlf=sqrt(!dcs_hsq2mn/ef)
         eres=dcs_jcc_resn(wl0,wlf,rmode,mspeed)
         tsd=long(sqrt(!dcs_hsq2mn/ef)/!dcs_hom*!dcs_dsd)
         ptext[4]='tsd = '+string(tsd,format='(i6)')+' us'
         fraction=(tsd-tsdm)/samperiod
         frame=0
         while (fraction lt 0.0) do begin fraction=fraction+1.0 & frame=frame-1 & endwhile
         while (fraction ge 1.0) do begin fraction=fraction-1.0 & frame=frame+1 & endwhile
         if (tchanmethod eq 1) then begin
            etmin=e0-!dcs_hsq2mn/(!dcs_hom*tsdm/!dcs_dsd)^2
            etmax=e0-!dcs_hsq2mn/(!dcs_hom*(tsdm+samperiod)/!dcs_dsd)^2
            et=e0-!dcs_hsq2mn/(!dcs_hom*(tsdm+fraction*samperiod)/!dcs_dsd)^2
            fraction=(et-etmin)/(etmax-etmin)
         endif
         ptext[5]='Time channel'+string(fraction*nout,format='(i5)')+' ['+string(frame,format='(i2)')+']'
         kks=fix(tsd/samperiod)
         for kk=0,4 do begin
             ptext[6+kk]='E0-Ef['+strcompress(kk-kks+frame,/remove_all)+'] ='+$
             string(e0-!dcs_hsq2mn/((!dcs_hom*(tsd+(kk-kks)*samperiod))/!dcs_dsd)^2,format='(f9.3)')+' meV'
         endfor
         ptext[11]='Energy Resolution'
         ptext[12]=string(eres,format='(f7.1)')+' ueV'
      endif
   endif
   ;
   ; Update the display.
   widget_control,postext,set_value=ptext
endif
uname = widget_info(event.id,/uname)
case strlowcase(uname) of
     'dcsplandetspu':begin
        (*pState).spurionflag[0] = 1b-(*pState).spurionflag[0]
        widget_control,event.id,set_button=(*pState).spurionflag[0]
        DCSToolsPlanner_plot_kar,pState
        end
     'dcsplanchopspu':begin
        (*pState).spurionflag[1] = 1b-(*pState).spurionflag[1]
        widget_control,event.id,set_button=(*pState).spurionflag[1]
        DCSToolsPlanner_plot_kar,pState
        end
     else:
endcase
end


;************************************************************************************************
pro DCSToolsPlanner,$
    group_leader = group_leader,workDir=workDir,dataDir=dataDir,$
;    notify_ids = notify_ids,$
    register_name = register_name, $
    _EXTRA=etc

;************************************************************************************************
;
compile_opt strictarr
;
;
;;;;;;;added following 4 lines for non-modal widget action
if n_elements(group_leader) eq 0 then group_leader = 0L
;if n_elements(notify_ids) eq 0 then notify_ids = [0L,0L]
if n_elements(register_name) eq 0 then register_name = 'DCSToolsPlanner'
if xregistered(register_name) then return
;
common constants,conversion
common energies,e0
common cursor_related,postext
common dependent_values,messagetext
common dependent_stuff,periodstext,elimitstext,warningstext
common modifiable_values,rattext,phiminslider,phimaxslider,$
    speedtext,rmodeslider,sratslider,tsdmtext1
common modifiable_stuff,mspeed,rmode,israt,sdenom,tsdm,samperiod,nout,tchanmethod
;common zoom_related,x1,x2,x1r,x2r,drawzoom,dozoom,zoomed,zdrawn,xlim,ylim,bangx,bangy

;
; Lines commented out.
;widget_control,group_leader,get_uvalue=statePtr
;davePtr=(*statePtr).davePtr
;
dcs_sysvars,davePtr=davePtr
;
hbarsq2mn=!dcs_hsq2mn/(2*!pi)^2
conversion=1/hbarsq2mn
;
;dcs_sysdep,fonts=fonts
;fonts=fonts[0]
;
screensize=!screensize
;
; Starting values of input parameters
wl=4.8
phimin=10
dphi=10
phimax=140
rat=3.0
nef=101

; Read in or define the overall size of the widget.
if (keyword_set(xsizekar)) then xsk=xsizekar else xsk=screensize[0]*0.9
if (keyword_set(ysizekar)) then ysk=ysizekar else ysk=screensize[1]*0.9
if (keyword_set(group_leader)) then group_leader=group_leader $
    else group_leader=""
;
srattyp=['(m-1)/m','1/m']
tsdmtyp=['Constant dt','Constant dE']
;
; Initial values.
wl=6.0
rat=4.0
phimin=5
dphi=5
phimax=140
mspeed=20000
israt=0
;snum=1
sdenom=round(wl/2)
rmode=1
tsdm=500
tsdm_maxval=fix(1000*wl)
tchanmethod=0
nout=1000
;
; Define base widget and other widgets.
tlb=widget_base(title="Kinematically Allowed Region",$
    /column,xoffset=screensize[0]*0.05,$
    group_leader=group_leader,ysize=ysk,mbar=bar)
    ;
    toprow=widget_base(tlb,/row,/align_left,frame=2);,ysize=0.41*ysk)
       ;
       inputtext=widget_base(toprow,frame=2,/column);,ysize=0.40*ysk)
         lambdabase=widget_base(inputtext,/row,frame=2)
          label=widget_label(lambdabase,value='lambda: ')
          lambdatext=widget_text(lambdabase,$
              value=string(wl,format='(f6.3)'),$
                 /editable,event_pro='DCSToolsPlanner_textbuttons',xsize=10)
         speedbase=widget_base(inputtext,/row,frame=2)
          speedlabel=widget_label(speedbase,$
              value='Master speed (rpm): ')
          speedtext=widget_text(speedbase,$
              value=strcompress(string(mspeed),/remove_all),$
              /editable,event_pro='DCSToolsPlanner_textbuttons',xsize=5)
         sratbase=widget_base(inputtext,/column,frame=2)
          sratbase1=widget_base(sratbase,/row)
              sratlabel1=widget_label(sratbase1,$
                 value='Speed ratio method')
              sratdroplist=widget_droplist(sratbase1,$
                 title='',value=srattyp,$
                 event_pro='DCSToolsPlanner_textbuttons')
          sratbase2=widget_base(sratbase,/row)
              sratslider=widget_slider(sratbase,$
                 value=sdenom,min=1,max=6,title='Speed ratio denominator',$
                 event_pro='DCSToolsPlanner_textbuttons')
         rmodebase=widget_base(inputtext,/row,frame=2)
          rmodeslider=widget_slider(rmodebase,$
              value=rmode,min=1,max=3,title='Res''l''n mode (low=1,high=3)',$
              event_pro='DCSToolsPlanner_textbuttons')
         tsdmbaseA=widget_base(inputtext,/col,frame=2)
          tsdmbase1=widget_base(tsdmbaseA,/row)
              tsdmlabel1=widget_label(tsdmbase1,$
                 value='Minimum time Sample-Detector: ')
              tsdmtext1=widget_text(tsdmbase1,value=strcompress(tsdm,/remove_all),$
                 /editable,event_pro='DCSToolsPlanner_textbuttons',xsize=6)
              tsdmslider=widget_slider(tsdmbaseA,value=strcompress(tsdm,/remove_all),$
                 event_pro='DCSToolsPlanner_textbuttons',$
                 /suppress_value,min=0,max=tsdm_maxval)
       ;
       messagebase=widget_base(toprow,frame=2,/column);,ysize=0.40*ysk)
       messagetext=widget_text(messagebase,value=strarr(20),ysize=20)
       ;
       outputtext=widget_base(toprow,frame=2,/column,xsize=200);,ysize=0.40*ysk)
         periodstext=widget_text(outputtext,value=strarr(5),ysize=5)
         elimitstext=widget_text(outputtext,value=strarr(3),ysize=3)
         warningstext=widget_text(outputtext,xsize=20,value=strarr(20),/scroll,ysize=8)
       ;
       buttonbase=widget_base(toprow,/row,frame=2);,ysize=0.40*ysk)
         otherbuttons=widget_base(buttonbase,/column)
          ratbase=widget_base(otherbuttons,/row,frame=2,xsize=140)
              label=widget_label(ratbase,value='Efmax/E0: ')
              rattext=widget_text(ratbase,$
                 value=string(rat,format='(f6.3)'),$
                 /editable,event_pro='DCSToolsPlanner_textbuttons',xsize=10)
          phiminbase=widget_base(otherbuttons,/row,frame=2,xsize=120)
              phiminslider=widget_slider(phiminbase,value=phimin,$
                 event_pro='DCSToolsPlanner_textbuttons',min=0,max=180,$
                 title='phimin (deg)')
          dphibase=widget_base(otherbuttons,/row,frame=2,xsize=120)
              dphislider=widget_slider(dphibase,value=dphi,$
                 event_pro='DCSToolsPlanner_textbuttons',min=1,max=30,$
                 title='dphi (deg)')
          phimaxbase=widget_base(otherbuttons,/row,frame=2,xsize=120)
              phimaxslider=widget_slider(phimaxbase,value=phimax,$
                 event_pro='DCSToolsPlanner_textbuttons',min=0,max=180,$
                 title='phimax (deg)')
          tsdmbaseB=widget_base(otherbuttons,/col,frame=2)
              tsdmbase2=widget_base(tsdmbaseB,/row)
                 tsdmlabel2=widget_label(tsdmbase2,value='Method')
                 tsdmdroplist=widget_droplist(tsdmbase2,$
                   title='',value=tsdmtyp,event_pro='DCSToolsPlanner_textbuttons')
              tsdmbase3=widget_base(tsdmbaseB,/row)
                 tsdmlabel3=widget_label(tsdmbase3,value='# valid time channels: ')
                 tsdmtext3=widget_text(tsdmbase3,value=strcompress(nout),$
                   /editable,event_pro='DCSToolsPlanner_textbuttons',xsize=6)
       ;
       cmbase=widget_base(toprow,/col);,ysize=0.40*ysk);,xsize=xsk*0.165
         cursorbase=widget_base(cmbase,/column,frame=2,xsize=xsk*0.16)
          postext=widget_text(cursorbase,value=strarr(13),ysize=13)
;     cmbase2=widget_base(toprow,/col);,ysize=0.40*ysk);,xsize=xsk*0.165
;
       mainbuttons=widget_button(bar,value='File',/menu)
;      mainbuttons=widget_base(cmbase2,/column,frame=2,xsize=xsk*0.16)
          gob=widget_button(mainbuttons,$
              value='Recalculate',event_pro='DCSToolsPlanner_textbuttons')
;          zoomb=widget_button(mainbuttons,$
;             value='Zoom in',event_pro='DCSToolsPlanner_zoombutton')
          psb=widget_button(mainbuttons,$
              value='Print',event_pro='DCSToolsPlanner_printbutton')
          tsdmb=widget_button(mainbuttons,$
              value='Save tchan widths',event_pro='DCSToolsPlanner_tchanbutton')
          t2eb=widget_button(mainbuttons,$
              value='Relate t-channel to E-transfer(s)',event_pro='DCSToolsPlanner_t2ebutton')
          savpb=widget_button(mainbuttons,$
              value='Save parameters',event_pro='DCSToolsPlanner_textbuttons')
          respb=widget_button(mainbuttons,$
              value='Restore parameters',event_pro='DCSToolsPlanner_textbuttons')
          doneb=widget_button(mainbuttons,$
              value='EXIT',event_pro='DCSToolsPlanner_exitbutton')
;    
       spurionflag=[1b,0b] ;default show detector spurion, but not chopper spurion
       optnmenu=widget_button(bar,value='Option',/menu)     
          void = widget_button(optnmenu,value='Show Detector Spurion',uname='dcsplandetspu',/checked_menu)
          widget_control,void,set_button=spurionflag[0]
          void = widget_button(optnmenu,value='Show Chopper Spurion',uname='dcsplanchopspu',/checked_menu)
          widget_control,void,set_button=spurionflag[1]
     
;
    bottomrow=widget_base(tlb,/row,/align_left,frame=2);,ysize=0.57*ysk)
       ;
       geomx=widget_info(tlb,/geom)
       geomy=widget_info(toprow,/geom)
       plotwin=obj_new('dm_plot',xsize=geomx.xsize,ysize=0.95*(ysk-geomy.ysize),$
          /draw_motion_events,widgetbase=bottomrow,group_leader=tlb,/compound,legdpos=[0.03,0.15])
;
geom=widget_info(bottomrow,/geom)
widget_control,toprow,xsize=geom.xsize

; Realize the widget
widget_control,tlb,/realize
;
; Initial settings of droplists.
widget_control,sratdroplist,set_droplist_select=israt
widget_control,tsdmdroplist,set_droplist_select=tchanmethod
;
; Plot the kinematically allowed region.
nef=101

dcs_getnps,npsptr

cd,current=current
if n_elements(workDir) eq 0 then workDir = current
if n_elements(dataDir) eq 0 then dataDir = current

; Define the state variable.
state={lambdatext:lambdatext,$
    wl:wl,$
    rattext:rattext,$
    rat:rat,$
    phiminslider:phiminslider,$
    phimin:phimin,$
    phimaxslider:phimaxslider,$
    phimax:phimax,$
    dphislider:dphislider,$
    dphi:dphi,$
    nef:nef,$
    psb:psb,$
    t2eb:t2eb,$
    savpb:savpb,$
    respb:respb,$
    speedtext:speedtext,$
    mspeed:mspeed,$
    rmodeslider:rmodeslider,$
    rmode:rmode,$
    sratslider:sratslider,$
    sratdroplist:sratdroplist,$
    israt:israt,$
    sdenom:sdenom,$
    tsdmtext1:tsdmtext1,$
    tsdmslider:tsdmslider,$
    tsdm:tsdm,$
    tsdm_maxval:tsdm_maxval,$
    tsdmdroplist:tsdmdroplist,$
    tchanmethod:tchanmethod,$
    tsdmtext3:tsdmtext3,$
    nout:nout,$
    periodstext:periodstext,$
    elimitstext:elimitstext,$
    warningstext:warningstext,$
;    notify_ids:notify_ids,  $              ; for non-modal widget
    workDir:workDir, $
    dataDir:dataDir, $
    npsptr:npsptr, $
    spurionflag:spurionflag, $        ;spurion flag [det,chopper]
    plotWin:plotWin}
;
pState=ptr_new(state)
widget_control,tlb,SET_UVALUE=pState

DCSToolsPlanner_plot_kar,pState

RET = DAVE_SET_FOCUS(TLB)
;
; Manage the widget
; added register_name variable to the call to xmanager.
xmanager,register_name,tlb,/no_block,cleanup='DCSToolsPlanner_cleanup',$
    group_leader=group_leader,event_handler='DCSToolsPlanner_handler'
;
; Lines commented out
;dave_makeDavePtr,davePtr=davePtr
;(*statePtr).davePtr=davePtr
;tvlct,(*rgbptr).r,(*rgbptr).g,(*rgbptr).b
;ptr_free,rgbptr
;
end
