; $Id: $
;#######################################################################
;
; NAME:
;  dm_volumegrid_bin
;
; PURPOSE:
;  bin volume data according to given xstep, ystep, and zstep
;
; CATEGORY:
;  general
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  April, 2023
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

; parameters:
;   in_x(y,z)step: x(y,z)step
;   xdat:          input xdat[ndat]    output xdat[xsize]
;   ydat:          input ydat[ndat]    output ydat[ysize]
;   zdat:          input zdat[ndat]    output zdat[zsize]
;   idat:          input idat[ndat]    output idat[xsize,ysize,zsize]
; keyword:
;   avgsum:        -1:weighted mean 0: arithmetic mean 1:sum, default 0
;   weight:        used as weight for avgsum = -1. if not present, use the error bar 
;   bintr:         binning threshold, bins with fewer data points than bintr*(max number of points in a bin) will be discarded
;   conststep:     0: use the average as the resulted binning values(default) 1: constant binning intervals
;   xstart:        binning starting point, if absent, use the min(xdat) as the starting point
;   ystart:        binning starting point, if absent, use the min(ydat) as the starting point
;   zstart:        binning starting point, if absent, use the min(zdat) as the starting point   
;   checkfinite:   flag for checking finite values of idat
;   leave_blank:   if set, leave empty column or rows as they are
;   ewid:          energy width, used when e-integrated intensity option is chose, input  dat=sum(I*ewid), output dat=sum(I*ewid)/sum(ewid)
;   extlib:        0:not using external library 1:use external library            
;   ierr:          input error bar of idat[ndat], ouput ierr[xsize,ysize,zsize]
;   zero_error:    if nonzero, use this value as error bar for zero intensities
;   bin_zeroerror: flag for how to treat zero intenisty error, default 0 (user specify)
;   zoerrestrange: used when bin_zeroerror=1, the zero intensity error bar statistical estimate range: 0-automatic search till non-zero counts are found; n(>=1)-specified search range
;   estapplyall:   used when bin_zeroerror=1, if set, zoerrestrange is applied to all intensity error bar statistical estimate
;   foundrange:    output the range when zoerrestrange=0
pro dm_volumegrid_bin,in_xstep,in_ystep,in_zstep,xdat,ydat,zdat,idat,ierr=ierr,xstart=xstart,ystart=ystart,zstart=zstart,avgsum=avgsum,bintr=bintr,checkfinite=checkfinite,$
    conststep=conststep,ewid=ewid,leave_blank=leave_blank,extlib=extlib,debug=debug,weight=weight,zero_error=zero_error,bin_zeroerror=bin_zeroerror,zoerrestrange=zoerrestrange,$
    estapplyall=estapplyall,foundrange=foundrange,group_leader=group_leader

    if n_elements(avgsum) eq 0 then avgsum = 0        ;default using arithmetic mean in binning
    if n_elements(bintr)  eq 0 then bintr = 0
    if n_elements(conststep) eq 0 then conststep = 0  ;default is using average as the binned value
    if keyword_set(conststep) then leave_blank = 1
    if n_elements(extlib) eq 0 then extlib = 0        ;default not using external binning library
    if n_elements(zero_error) eq 0 then zero_error = 0.0
    ndat    = n_elements(idat)
    ok_ierr = n_elements(ierr) eq ndat
    ok_eint = n_elements(ewid) eq ndat
    ok_wght = n_elements(weight) eq ndat
    x_type  = size(xdat,/type)
    y_type  = size(ydat,/type)
    z_type  = size(zdat,/type)
    i_type  = size(idat,/type)
    if ok_ierr then e_type = size(ierr,/type)
    if ok_wght then w_type = size(weight,/type)
    if ok_eint then avgsum = 1
    if keyword_set(bin_zeroerror) and ok_ierr then begin
       if n_elements(zoerrestrange) eq 0 then zoerrestrange = 0
       if zoerrestrange eq 0 then foundrange = 0
    endif else bin_zeroerror = 0
    if (avgsum eq -1) and (~ok_wght) and (~ok_ierr) then avgsum = 0    ;weighted average requires the weight or corresponding ierr, without it, switching to arithmetic mean
    if keyword_set(debug) then current = systime(/sec)
    if keyword_set(checkfinite) then begin
       ind = where(finite(idat),count) 
       if (count ne ndat) and (count ne 0) then begin
          xdat = xdat[ind]
          ydat = ydat[ind]
          zdat = zdat[ind]
          idat = idat[ind]
          if ok_ierr then ierr = ierr[ind]
          if ok_eint then ewid = ewid[ind]
          if ok_wght then weight = weight[ind]
          ind  = 0 & ndat = count
       endif
    endif
    xstep   = abs(in_xstep)
    ystep   = abs(in_ystep)
    zstep   = abs(in_zstep)
    xmax    = max(xdat,min=xmin)
    ymax    = max(ydat,min=ymin)
    zmax    = max(zdat,min=zmin)
    if n_elements(xstart) eq 1 then begin
       if finite(xstart) then begin
          xmin = xstart
          ind = where(xdat ge xstart,count)
          if count ne ndat then begin
             xdat = xdat[ind]
             ydat = ydat[ind]
             zdat = zdat[ind]
             idat = idat[ind]
             if ok_ierr then ierr = ierr[ind]
             if ok_eint then ewid = ewid[ind]
             if ok_wght then weight = weight[ind]
          endif
          ind = 0 & ndat = count
       endif
    endif
    if n_elements(ystart) eq 1 then begin
       if finite(ystart) then begin
          ymin = ystart
          ind = where(ydat ge ystart,count)
          if count ne ndat then begin
             xdat = xdat[ind]
             ydat = ydat[ind]
             zdat = zdat[ind]
             idat = idat[ind]
             if ok_ierr then ierr = ierr[ind]
             if ok_eint then ewid = ewid[ind]
             if ok_wght then weight = weight[ind]
          endif
          ind = 0 & ndat = count
       endif
    endif
    if n_elements(zstart) eq 1 then begin
       if finite(zstart) then begin
          zmin = zstart
          ind = where(zdat ge zstart,count)
          if count ne ndat then begin
             xdat = xdat[ind]
             ydat = ydat[ind]
             zdat = zdat[ind]
             idat = idat[ind]
             if ok_ierr then ierr = ierr[ind]
             if ok_eint then ewid = ewid[ind]
             if ok_wght then weight = weight[ind]
          endif
          ind = 0 & ndat = count
       endif
    endif
    if (avgsum eq -1) and (~ok_wght) then begin
       weight = ierr
       ind = where(ierr eq 0,complement=ind1,count)
       if count ne 0 then begin
          tmp = min(abs(ierr[ind1]))
          if zero_error ne 0 then tmp = (tmp)<(zero_error)
          weight[ind] = tmp        ;make sure no ierr=0 exists, otherwise replace them with smallest of nonzero value
       endif
       weight = 1./weight^2
       ok_wght = 1b
       ind = 0 & ind1 = 0
    endif
    if (xstep eq 0) then xstep = ((xmax eq xmin)?(0.1):((xmax-xmin)/100.))
    if (ystep eq 0) then ystep = ((ymax eq ymin)?(0.1):((ymax-ymin)/100.))
    if (zstep eq 0) then zstep = ((zmax eq zmin)?(0.1):((zmax-zmin)/100.))
    xmax    = xmax+xstep/2.
    xmin    = xmin-xstep/2.
    ymax    = ymax+ystep/2.
    ymin    = ymin-ystep/2.
    zmax    = zmax+zstep/2.
    zmin    = zmin-zstep/2.
    xsize   = ceil((xmax-xmin)/xstep)
    ysize   = ceil((ymax-ymin)/ystep)
    zsize   = ceil((zmax-zmin)/zstep)
    tmp_x   = dblarr(xsize) & num_x = lonarr(xsize)
    tmp_y   = dblarr(ysize) & num_y = lonarr(ysize)
    tmp_z   = dblarr(zsize) & num_z = lonarr(zsize)
    tmp_i   = dblarr(xsize,ysize,zsize) & num_i = lonarr(xsize,ysize,zsize)
    if ok_eint then tmp_w = dblarr(xsize,ysize,zsize)
    if ok_ierr then tmp_e = dblarr(xsize,ysize,zsize)
    if ok_wght then tmp_m = dblarr(xsize,ysize,zsize)
  
    defsysv,'!DAVE_AUXILIARY_DIR',exists=exists
    if exists and extlib then begin
       filename = 'rebinnd'+'_'+!version.os+'_'+strtrim(string(!VERSION.MEMORY_BITS),2)
       shlib = (FILE_SEARCH(!DAVE_AUXILIARY_DIR+'rebin_nd'+path_sep()+filename+'*'))[0]
    endif else shlib = ''
    if (strlen(shlib) eq 0) and extlib then begin
       mesg = 'The external binning library is not found. IDL binning code is used instead.'
       if n_elements(group_leader) ne 0 then begin
          if (group_leader eq LookupManagedWidget('dcs_mslice')) or (group_leader eq LookupManagedWidget('mslice')) then begin
             widget_control,group_leader,get_uvalue=obj_mslice
             if obj_valid(obj_mslice) then obj_mslice->setproperty,binextlib=0 $
             else mesg = [mesg,'','Please disable the external library in the menu Option->Binning Method->Use External Binning Library.']
          endif
       endif
       ok = dialog_message(mesg,dialog_parent=group_leader,/center,title='Please note:')
    endif
    
    if (strlen(shlib) gt 0) then begin
       if (ok_eint and ok_ierr) then begin
          r = CALL_EXTERNAL(shlib, 'rebin3d_iew', long(ndat),double(xdat),double(ydat),double(zdat),$
                double(idat),double(ierr),double(ewid),long(xsize),long(ysize),long(zsize),double(xmin),double(xstep),$
                double(ymin),double(ystep), double(zmin),double(zstep), num_x, tmp_x, num_y, tmp_y, num_z, tmp_z, $
                num_i,tmp_i,tmp_e, tmp_w, /CDECL) 
       endif       
       if (ok_eint and ~ok_ierr) then begin 
          r = CALL_EXTERNAL(shlib, 'rebin3d_iw', long(ndat),double(xdat),double(ydat),double(zdat),$
                double(idat),double(ewid),long(xsize),long(ysize),long(zsize),double(xmin),double(xstep),$
                double(ymin),double(ystep), double(zmin),double(zstep), num_x, tmp_x, num_y, tmp_y, num_z, tmp_z, $
                num_i,tmp_i, tmp_w, /CDECL)  
       endif       
       if (~ok_eint and ok_ierr) then begin 
          r = CALL_EXTERNAL(shlib, 'rebin3d_ie', long(ndat),double(xdat),double(ydat),double(zdat),$
                double(idat),double(ierr),long(xsize),long(ysize),long(zsize),double(xmin),double(xstep),$
                double(ymin),double(ystep), double(zmin),double(zstep), num_x, tmp_x, num_y, tmp_y, num_z, tmp_z, $
                num_i,tmp_i,tmp_e,  /CDECL) 
       endif       
       if (~ok_eint and ~ok_ierr) then begin
          r = CALL_EXTERNAL(shlib, 'rebin3d_i', long(ndat),double(xdat),double(ydat),double(zdat),$
                double(idat),long(xsize),long(ysize),long(zsize),double(xmin),double(xstep),$
                double(ymin),double(ystep), double(zmin),double(zstep), num_x, tmp_x, num_y, tmp_y, num_z, tmp_z, $
                num_i,tmp_i,  /CDECL)
       endif 
       if avgsum eq -1 then avgsum = 0  ;weighted mean not implemented yet
    endif else begin  ;use histogram function to do the 3d-binning
       ;first construct a 1-d array for histogram function
       use64 = (xsize*ysize*zsize) lt 0 
       tmp_zdat = long((zdat-zmin)/zstep)
       if ~keyword_set(conststep) then begin
          num_z = histogram(tmp_zdat,binsize=1L,min=0L,max=zsize-1L,reverse_indices=rindx)
          for i=0L,n_elements(num_z)-1L do begin
              if num_z[i] eq 0 then continue
              tmp_z[i] = total(zdat[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
          endfor
       endif
       rindx = 0 & zdat = 0
       if use64 then begin
          tmp = long64(ysize)*long64(temporary(tmp_zdat))
          tmpmax = long64(xsize)*long64(ysize)*long64(zsize)-(1LL) 
       endif else begin
          tmp = long(ysize)*long(temporary(tmp_zdat))
          tmpmax = long(xsize)*long(ysize)*long(zsize)-(1L) 
       endelse
       tmp_ydat = long((ydat-ymin)/ystep)
       if ~keyword_set(conststep) then begin
          num_y = histogram(tmp_ydat,binsize=1L,min=0L,max=ysize-1L,reverse_indices=rindx)
          for i=0L,n_elements(num_y)-1L do begin
              if num_y[i] eq 0 then continue
              tmp_y[i] = total(ydat[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
          endfor
       endif
       rindx = 0 & ydat = 0
       if use64 then tmp = long64(xsize)*(temporary(tmp)+long64(temporary(tmp_ydat))) $
       else tmp = long(xsize)*(temporary(tmp)+long(temporary(tmp_ydat)))
       tmp_xdat = long((xdat-xmin)/xstep)
       if ~keyword_set(conststep) then begin
          num_x = histogram(tmp_xdat,binsize=1L,min=0L,max=xsize-1L,reverse_indices=rindx)
          for i=0L,n_elements(num_x)-1L do begin
              if num_x[i] eq 0 then continue
              tmp_x[i] = total(xdat[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
          endfor
       endif
       rindx = 0 & xdat = 0
       if use64 then tmp = temporary(tmp)+long64(temporary(tmp_xdat)) $
       else tmp = temporary(tmp)+long(temporary(tmp_xdat))
       num_i = histogram(temporary(tmp),binsize=1,min=0,max=tmpmax,reverse_indices=rindx)
       num_i = reform(num_i,xsize,ysize,zsize,/overwrite)
       if ok_ierr then begin
          ierr = ierr^2
          if (zero_error ne 0) and (avgsum eq -1) then begin
             ind = where(ierr eq 0,cnt)
             if cnt ne 0 then zero_error = zero_error*min(weight[ind])
          endif
       endif
       n_rindx = n_elements(rindx)
       if avgsum eq -1 then begin  ;weighted mean
          if keyword_set(bin_zeroerror) then tmp_ierr = ierr  ;need another copy
          for i=0LL,n_elements(num_i)-1LL do begin
              if num_i[i] eq 0 then continue
              if keyword_set(bin_zeroerror) then begin  ;special care for zero intensities
                 tmp_rowx = long64(i mod xsize)
                 tmp_rowy = long64((i/xsize) mod ysize)
                 tmp_rowz = long64((i/xsize)/ysize)
                 tmp1     = 0d 
                 tmp2     = 0d
                 if keyword_set(estapplyall) and (zoerrestrange) gt 0 then begin
                    for rowx=(0LL)>(tmp_rowx-zoerrestrange),(tmp_rowx+zoerrestrange)<(xsize-1) do begin
                        for rowy=(0LL)>(tmp_rowy-zoerrestrange),(tmp_rowy+zoerrestrange)<(ysize-1) do begin
                            for rowz=(0LL)>(tmp_rowz-zoerrestrange),(tmp_rowz+zoerrestrange)<(zsize-1) do begin
                                tmp_in = (rowz*ysize+rowy)*xsize+rowx
                                if rindx[tmp_in] lt n_rindx then begin
                                   if rindx[tmp_in+1] gt rindx[tmp_in] then begin
                                      tmp1 = temporary(tmp1)+total(tmp_ierr[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                                      tmp2 = temporary(tmp2)+total(weight[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                                   endif else begin
                                      tmp1 = temporary(tmp1)+tmp_ierr[rindx[rindx[tmp_in]]]
                                      tmp2 = temporary(tmp2)+weight[rindx[rindx[tmp_in]]]
                                   endelse
                                endif
                            endfor
                        endfor
                    endfor
                    if tmp2 gt 0 then begin
                       tmp = tmp1/tmp2
                       ierr[rindx[rindx[i]:(rindx[i+1]-1)]] = weight[rindx[rindx[i]:(rindx[i+1]-1)]]*tmp
                    endif
                 endif else begin
                    ind = where(tmp_ierr[rindx[rindx[i]:(rindx[i+1]-1)]] eq 0,cnt)
                    if cnt ne 0 then begin
                       if zoerrestrange eq 0 then begin
                          ;expanding the range gradually until finding non-zero intensities
                          found   = 0b
                          tmp_ran = 0l
                          tmp_max = max([tmp_rowx,xsize-tmp_rowx,tmp_rowy,ysize-tmp_rowy,tmp_rowz,zsize-tmp_rowz])
                          while (~found) do begin
                                for rowx=(0LL)>(tmp_rowx-tmp_ran),(tmp_rowx+tmp_ran)<(xsize-1) do begin
                                    for rowy=(0LL)>(tmp_rowy-tmp_ran),(tmp_rowy+tmp_ran)<(ysize-1) do begin
                                        for rowz=(0LL)>(tmp_rowz-tmp_ran),(tmp_rowz+tmp_ran)<(zsize-1) do begin
                                            if max(abs([rowx-tmp_rowx,rowy-tmp_rowy,rowz-tmp_rowz])) eq tmp_ran then begin
                                               tmp_in = (rowz*ysize+rowy)*xsize+rowx
                                               if rindx[tmp_in] lt n_rindx then begin
                                                  if rindx[tmp_in+1] gt rindx[tmp_in] then begin
                                                     tmp1 = temporary(tmp1)+total(tmp_ierr[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                                                     tmp2 = temporary(tmp2)+total(weight[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                                                  endif else begin
                                                     tmp1 = temporary(tmp1)+tmp_ierr[rindx[rindx[tmp_in]]]
                                                     tmp2 = temporary(tmp2)+weight[rindx[rindx[tmp_in]]]
                                                  endelse
                                               endif
                                            endif
                                        endfor
                                    endfor
                                endfor
                                if tmp1 gt 0 then begin
                                   tmp = tmp1/tmp2
                                   if keyword_set(estapplyall) then ierr[rindx[rindx[i]:(rindx[i+1]-1)]] = weight[rindx[rindx[i]:(rindx[i+1]-1)]]*tmp $
                                   else ierr[(rindx[rindx[i]:(rindx[i+1]-1)])[ind]] = weight[(rindx[rindx[i]:(rindx[i+1]-1)])[ind]]*tmp
                                   found = 1b
                                endif
                                tmp_ran = tmp_ran+1
                                if tmp_ran gt tmp_max then found = 1b
                          endwhile
                          foundrange = foundrange>(tmp_ran-1)
                       endif else begin
                          for rowx=(0LL)>(tmp_rowx-zoerrestrange),(tmp_rowx+zoerrestrange)<(xsize-1) do begin
                              for rowy=(0LL)>(tmp_rowy-zoerrestrange),(tmp_rowy+zoerrestrange)<(ysize-1) do begin
                                  for rowz=(0LL)>(tmp_rowz-zoerrestrange),(tmp_rowz+zoerrestrange)<(zsize-1) do begin
                                      tmp_in = (rowz*ysize+rowy)*xsize+rowx
                                      if rindx[tmp_in] lt n_rindx then begin
                                         if rindx[tmp_in+1] gt rindx[tmp_in] then begin
                                            tmp1 = temporary(tmp1)+total(tmp_ierr[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                                            tmp2 = temporary(tmp2)+total(weight[rindx[rindx[tmp_in]:(rindx[tmp_in+1]-1)]],/double)
                                         endif else begin
                                            tmp1 = temporary(tmp1)+tmp_ierr[rindx[rindx[tmp_in]]]
                                            tmp2 = temporary(tmp2)+weight[rindx[rindx[tmp_in]]]
                                         endelse
                                      endif
                                  endfor
                              endfor
                          endfor
                          if tmp2 gt 0 then begin
                             tmp = tmp1/tmp2
                             ierr[(rindx[rindx[i]:(rindx[i+1]-1)])[ind]] = weight[(rindx[rindx[i]:(rindx[i+1]-1)])[ind]]*tmp
                          endif
                       endelse
                    endif
                 endelse
              endif
              tmp_i[i] = total(idat[rindx[rindx[i]:(rindx[i+1]-1)]]*weight[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
              tmp_m[i] = total(weight[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
              if ok_ierr then tmp_e[i] = total(ierr[rindx[rindx[i]:(rindx[i+1]-1)]]*(weight[rindx[rindx[i]:(rindx[i+1]-1)]])^2,/double)
          endfor
          if keyword_set(bin_zeroerror) then tmp_ierr = 0  ;release the memory
       endif else begin
          for i=0LL,n_elements(num_i)-1LL do begin
              if num_i[i] eq 0 then continue
              tmp_i[i] = total(idat[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
              if ok_ierr then tmp_e[i] = total(ierr[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
              if ok_eint then tmp_w[i] = total(ewid[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
              if ok_wght then tmp_m[i] = total(weight[rindx[rindx[i]:(rindx[i+1]-1)]],/double)
          endfor
       endelse
       rindx = 0
       if keyword_set(conststep) then begin
          if use64 then num_x = long64(total(reform(total(num_i,2),xsize,zsize),2)) else num_x = long(total(reform(total(num_i,2),xsize,zsize),2)) ;use reform to avoid error when xsize or ysize or zsize=1
          if use64 then num_y = long64(total(reform(total(num_i,3),xsize,ysize),1)) else num_y = long(total(reform(total(num_i,3),xsize,ysize),1))
          if use64 then num_z = long64(total(reform(total(num_i,1),ysize,zsize),1)) else num_z = long(total(reform(total(num_i,1),ysize,zsize),1))
       endif
    endelse
    idat = 0
    if ok_ierr then ierr = 0

    indx = where(num_x ne 0,countx,complement=indx1,ncomplement=countx1)
    indy = where(num_y ne 0,county,complement=indy1,ncomplement=county1)
    indz = where(num_z ne 0,countz,complement=indz1,ncomplement=countz1)
    if keyword_set(leave_blank) then begin
       if countx1 ne 0 then begin
          num_x[indx1]=1.0 & tmp_x[indx1]=xmin+indx1*xstep
       endif
       if county1 ne 0 then begin
          num_y[indy1]=1.0 & tmp_y[indy1]=ymin+indy1*ystep
       endif
       if countz1 ne 0 then begin
          num_z[indz1]=1.0 & tmp_z[indz1]=zmin+indz1*zstep
       endif
    endif else begin
       if (countx ne 0) and (countx1 ne 0) then begin
          tmp_x = tmp_x[indx]   & num_x = num_x[indx]
          tmp_i = tmp_i[indx,*,*] & num_i = num_i[indx,*,*]
          if ok_ierr then tmp_e = tmp_e[indx,*,*]
          if ok_eint then tmp_w = tmp_w[indx,*,*]
          if ok_wght then tmp_m = tmp_m[indx,*,*]
       endif
       if (county ne 0) and (county1 ne 0) then begin
          tmp_y = tmp_y[indy]   & num_y=num_y[indy]
          tmp_i = tmp_i[*,indy,*] & num_i=num_i[*,indy,*]
          if ok_ierr then tmp_e = tmp_e[*,indy,*]
          if ok_eint then tmp_w = tmp_w[*,indy,*]
          if ok_wght then tmp_m = tmp_m[*,indy,*]
       endif
       if (countz ne 0) and (countz1 ne 0) then begin
          tmp_z = tmp_z[indz]   & num_z=num_z[indz]
          tmp_i = tmp_i[*,*,indz] & num_i=num_i[*,*,indz]
          if ok_ierr then tmp_e = tmp_e[*,*,indz]
          if ok_eint then tmp_w = tmp_w[*,*,indz]
          if ok_wght then tmp_m = tmp_m[*,*,indz]
       endif
    endelse
    indx=-1 & indx1=-1 & indy=-1 & indy1=-1 & indz=-1 & indz1=-1
    if keyword_set(conststep) then begin
       xdat = xmin+(findgen(xsize)+0.5)*xstep
       ydat = ymin+(findgen(ysize)+0.5)*ystep
       zdat = zmin+(findgen(zsize)+0.5)*zstep
       tmp_i = tmp_i[0:(xsize-1),0:(ysize-1),0:(zsize-1)]
       num_i = num_i[0:(xsize-1),0:(ysize-1),0:(zsize-1)]
       if ok_ierr then tmp_e = tmp_e[0:(xsize-1),0:(ysize-1),0:(zsize-1)]
       if ok_eint then tmp_w = tmp_w[0:(xsize-1),0:(ysize-1),0:(zsize-1)]
       if ok_wght then tmp_m = tmp_m[0:(xsize-1),0:(ysize-1),0:(zsize-1)]
    endif else begin
       xdat = temporary(tmp_x)/temporary(num_x)
       ydat = temporary(tmp_y)/temporary(num_y)
       zdat = temporary(tmp_z)/temporary(num_z)
    endelse
    idat = temporary(tmp_i)
    if ok_wght then weight = temporary(tmp_m)
    if ok_ierr then begin
       if avgsum ne -1 then begin
          ierr = sqrt(temporary(tmp_e))
          if n_elements(zero_error) ne 0 then begin
             ind = where(ierr eq 0,count)
             if count ne 0 then ierr[ind] = zero_error
             ind = 0
          endif
       endif else $
          ierr = fltarr(size(idat,/dimension))
    endif
    index1 = where(num_i ne 0,count1,complement=index0,ncomplement=count0)
    if (count1 ne 0) then begin
       if ok_eint then begin
          idat[index1] = idat[index1]/tmp_w[index1]
          if ok_ierr then ierr[index1] = ierr[index1]/tmp_w[index1]
       endif else if avgsum eq 0 then begin     ;arithmetic mean   
          idat[index1] = idat[index1]/num_i[index1]
          if ok_ierr then ierr[index1] = ierr[index1]/num_i[index1]
       endif else if avgsum eq -1 then begin    ;weighted mean
          idat[index1] = idat[index1]/weight[index1]
          if ok_ierr then begin
             ierr[index1] = sqrt(tmp_e[index1])/weight[index1]
             if zero_error ne 0 then begin
                ind = where(ierr[index1] eq 0,count)
                if count ne 0 then ierr[index1[ind]] = zero_error/weight[index1[ind]]
             endif 
             tmp_e = 0
          endif
       endif
       if (bintr gt 0.0) and (bintr lt 1.0) then begin
          min_num_i = bintr*(max(num_i)>1)
          ind = where(num_i lt min_num_i,ct1)
          if ct1 gt 0 then begin
             idat[ind] = !values.f_nan
             if ok_ierr then ierr[ind] = !values.f_nan
          endif
       endif
    endif
    if (count0 ne 0) then begin
       idat[index0] = !values.f_nan
       if ok_ierr then ierr[index0] = !values.f_nan
    endif
    if size(xdat,/type) ne x_type then xdat = fix(xdat,type=x_type)
    if size(ydat,/type) ne y_type then ydat = fix(ydat,type=y_type)
    if size(zdat,/type) ne z_type then zdat = fix(zdat,type=z_type)
    if size(idat,/type) ne i_type then idat = fix(idat,type=i_type)
    if ok_ierr then begin
       if size(ierr,/type) ne e_type then ierr = fix(ierr,type=e_type)
    endif
    if ok_wght then begin
       if size(weight,/type) ne w_type then weight = fix(weight,type=w_type)
    endif
    if keyword_set(debug) then print,'volumegrid_bin finished in ',systime(/sec)-current,' secs.'
end