; $Id$
;#######################################################################
;
; NAME:
;  dm_proj_spec_samp
;
; PURPOSE:
;  transform data points in the form (detector_angles, energy)->(Qx,Qy,Qz)
;  in the spectrometer frame where [1,0,0] is along ki, [0,1,0] in the horizontal scattering plane
;  and [0,0,1] pointing upward
;  then into the sample reference frame by a rotation of the coordinate system
;
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  Mrach, 2014
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

; parameters:
;   eief:     Ei or Ef according to instrument geometry, Ei for macs
;   en:       energy transfer: [ne] for powder and single crystal type, scalor or [ne] for diffuse scattering type, Ef for macs
;   det_tt:   two_theata:[ndet]
;   det_psi:  psi:[ndet]
; keywords:
;   psi:      scalor for single crystal type, [nfile] array for diffuse scattering type; positive if ccw rotation of u from ki
;   type:     0:powder 1:single crystal 2:diffuse scattering
;   diffetyp: combined with type=2   0:single energy 1:multiple energy
;   kidney:   [ndat] array, for MACS instrument, if present, psi=[ndat], eief=Ei[ndat], en=Ef[ndat], instrgeom keyword doesn't matter
;   is_spe:   for matlab style spe file
;   instrgeom:0-direct geometry, 1-inverse geometry
function dm_proj_spec_samp,eief,en,det_tt,det_psi,psi=psi,type=type,debug=debug,kidney=kidney,is_spe=is_spe,diffetyp=diffetyp,instrgeom=instrgeom
    if n_elements(kidney) eq n_elements(eief) then ismacs = 1b $
    else ismacs = 0b
    if keyword_set(debug) then begin
       print,'##proj spec samp...'
       current=systime(/sec)
    endif
    E_ksq = 2.0721246  ;E_ksq=!dcs_hsq2mn/4./!dpi/!dpi;=2.0721246
    nen = n_elements(en)
    if nen eq 1 then nen = n_elements(eief)
    if ismacs then begin
       ki = sqrt(eief/E_ksq) 
       kf = sqrt(en/E_ksq)   
    endif else begin
       if keyword_set(instrgeom) then begin
          ki = sqrt((eief+en)/E_ksq)
          kf = sqrt(eief/E_ksq)*(fltarr(nen)+1.0)
       endif else begin
          ki = sqrt(eief/E_ksq)*(fltarr(nen)+1.0)
          kf = sqrt((eief-en)/E_ksq)
       endelse
    endelse
    if n_elements(ki) eq 1 then ki=ki[0]
    if n_elements(kf) eq 1 then kf=kf[0]
    ndet = n_elements(det_tt) & nfile = n_elements(psi)
    if n_elements(type) eq 0 then type=0      ;default-powder type
    case type of
       0:begin  ;powder type -> Q[nen,ndet], nen=ndat for macs
         if ismacs then begin
            Q = fltarr(nen,ndet,/nozero)
            for i=0L,nen-1L do begin
                Q[i,*] = sqrt(ki[i]*ki[i]+kf[i]*kf[i]-2.*(ki[i]*kf[i])*cos(!dtor*(kidney[i]-76.0+det_tt)))
            endfor     
         endif else begin
            Q = sqrt((ki*ki+kf*kf)#(fltarr(ndet)+1.0)-2.*(ki*kf)#cos(det_tt*!dtor))
         endelse
         end
       1:begin  ;single crystal type -> Q[3,nen,ndet], nen=ndat for macs
         Q      = fltarr(3,nen,ndet,/nozero)
         detpsi = det_psi*!dtor
         sinpsi = sin(detpsi)
         cospsi = cos(temporary(detpsi))
         if ismacs then begin
            for i=0L,nen-1L do begin
                dettt  = (kidney[i]-76.0+det_tt)*!dtor
                cos2th = cos(dettt) 
                sin2th = sin(temporary(dettt)) 
                Q[0,i,*] = ki[i]-kf[i]*cos2th 
                if keyword_set(is_spe) then begin ;spe type definition of det_psi
                   Q[1,i,*] = -kf[i]*sin2th*cospsi
                   Q[2,i,*] = -kf[i]*sin2th*sinpsi
                endif else begin  ;dcs type definition of det_psi
                   index = where(sin2th eq 0,count)
                   Q[1,i,*] = kf[i]*sin2th*sqrt(1.0-(sinpsi/sin2th)^2)
                   Q[2,i,*] = -kf[i]*sinpsi
                   if count gt 0 then begin
                      Q[1,i,index] = 0.0
                      Q[2,i,index] = 0.0
                   endif
                endelse
                rsinpsi  = sin(psi[i]*!dtor)
                rcospsi  = cos(psi[i]*!dtor)
                R        = [[rcospsi,-rsinpsi,0.0],[rsinpsi,rcospsi,0.0],[0.0,0.0,1.0]] 
                Q[*,i,*] = reform(Q[*,i,*])##R
            endfor
         endif else begin
            dettt    = det_tt*!dtor
            cos2th   = cos(dettt)
            sin2th   = sin(temporary(dettt))
            Q[0,*,*] = temporary(ki)#(fltarr(ndet)+1.0)-kf#temporary(cos2th)
            if keyword_set(is_spe) then begin  ;spe style definition of det_psi
               Q[1,*,*] = -kf#(sin2th*temporary(cospsi))
               Q[2,*,*] = -temporary(kf)#(temporary(sin2th)*temporary(sinpsi))
            endif else begin    ;dcs style defininiton of det_psi (7.6285 for upper bank, -7.6285 for lower bank) and 2th(+/-)
               index = where(sin2th eq 0,count)
               Q[1,*,*] = kf#(sin2th*sqrt(1.0-(sinpsi/sin2th)^2))
               Q[2,*,*] = -temporary(kf)#temporary(sinpsi)
               if count gt 0 then begin
                  Q[1,*,index] = 0.0
                  Q[2,*,index] = 0.0
               endif
            endelse
            sinpsi = sin(psi[0]*!dtor)
            cospsi = cos(psi[0]*!dtor)
            R = [[cospsi,-sinpsi,0.0],[sinpsi,cospsi,0.0],[0.0,0.0,1.0]]
            Q = reform(Q,3,nen*ndet,/overwrite)
            Q = temporary(Q)##temporary(R)
            Q = reform(Q,3,nen,ndet,/overwrite)
         endelse
         end
       2:begin  ;diffuse scattering type -> Q[3,ndet, nfile] or Q[3,nen,ndet,nfile]
         dettt  = det_tt*!dtor
         detpsi = det_psi*!dtor
         cos2th = cos(dettt)
         sin2th = sin(temporary(dettt))
         sinpsi = sin(detpsi)
         cospsi = cos(temporary(detpsi))
         if diffetyp eq 0 then begin    ;single energy
            Q0      = fltarr(3,ndet,/nozero)
            Q0[0,*] = temporary(ki)-kf*temporary(cos2th)
            if keyword_set(is_spe) then begin  ;spe style definition of psi
               Q0[1,*] = -kf*(sin2th*temporary(cospsi))
               Q0[2,*] = -temporary(kf)*(temporary(sin2th)*temporary(sinpsi))
            endif else begin                   ;dcs style definition of psi
               index = where(sin2th eq 0,count)
               Q0[1,*] = kf*(sin2th*sqrt(1-(sinpsi/sin2th)^2))
               Q0[2,*] = -temporary(kf)*temporary(sinpsi)
               if count gt 0 then begin
                  Q0[1,index] = 0.0
                  Q0[2,index] = 0.0
               endif
            endelse
         endif else begin               ;multiple energy
            Q0        = fltarr(3,nen,ndet,/nozero)
            Q0[0,*,*] = temporary(ki)#(fltarr(ndet)+1.0)-kf#temporary(cos2th)
            if keyword_set(is_spe) then begin  ;spe style definition of psi
               Q0[1,*,*] = -kf#(sin2th*temporary(cospsi))
               Q0[2,*,*] = -temporary(kf)#(temporary(sin2th)*temporary(sinpsi))
            endif else begin                   ;dcs style definition of psi
               index = where(sin2th eq 0,count)
               Q0[1,*,*] = kf#(sin2th*sqrt(1-(sinpsi/sin2th)^2))
               Q0[2,*,*] = -temporary(kf)#temporary(sinpsi)
               if count gt 0 then begin
                  Q0[1,*,index] = 0.0
                  Q0[2,*,index] = 0.0
               endif
            endelse
         endelse
         R        = fltarr(3,3,nfile)
         sinpsi   = sin(psi*!dtor)
         cospsi   = cos(psi*!dtor)
         R[0,0,*] = cospsi
         R[1,0,*] = -sinpsi
         R[0,1,*] = temporary(sinpsi)
         R[1,1,*] = temporary(cospsi)
         R[2,2,*] = 1.0
         if diffetyp eq 0 then begin    ;single energy
            Q = fltarr(3,ndet,nfile,/nozero)
            for i=0,nfile-1 do $
                Q[*,*,i] = Q0##R[*,*,i]
         endif else begin               ;multiple energy
            Q  = fltarr(3,nen,ndet,nfile,/nozero)
            Q0 = reform(Q0,3,nen*ndet,/overwrite)
            for i=0,nfile-1 do begin
                tmp = Q0##R[*,*,i]
                Q[*,*,*,i] = reform(tmp,3,nen,ndet,/overwrite)
            endfor
         endelse
         end
       else:
    endcase
    if keyword_set(debug) then begin
       print,'##proj spec samp finished in ',systime(/sec)-current,' sec.'
    endif
    return, Q
end