; $Id: dm_plot_trajectory.pro,v 1.52 2016/03/31 17:01:38 ymqiu Exp $
;#######################################################################
;
; NAME:
;  dm_plot_trajectory
;
; PURPOSE:
;  plot trajectory
;
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  March, 2025
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

pro dcs_mslice::dm_plot_trajectory,overplot=overplot,debug=debug
    WIDGET_CONTROL,/HOURGLASS
    if keyword_set(debug) then current=systime(/sec)
    if (~ ptr_valid(self.projStrPtr) ) or self.error then begin
       ok = dialog_message('Please calculate projection first.',/error,dialog_parent=self.tlb)
       return
    endif
    
    none_id = [4+(self.intn_typ eq (n_elements(self.intn_typ_str)-2)),3,2,3]
    if self.extravaxis_yn[0] and (self.instrname ne 'macs') and (self.instrname ne 'wand') then none_id[1:3] = none_id[1:3]+1
    
    if (self.traj_x[self.samp_typ] eq none_id[self.samp_typ]) or (self.traj_y[self.samp_typ] eq none_id[self.samp_typ]) then begin
       ok = dialog_message('Invalid axis.',/error,dialog_parent=self.tlb)
       return
    endif

    ;select data
    self->dm_choose_trajdata,self.traj_x[self.samp_typ],data=xdat,title=xtit,is_E=is_E_x,dadata=da_xdat,is_hkl=is_hkl_x,hkl_info=xinfo
    self->dm_choose_trajdata,self.traj_y[self.samp_typ],data=ydat,title=ytit,is_E=is_E_y,dadata=da_ydat,is_hkl=is_hkl_y,hkl_info=yinfo
    
    if is_hkl_x or is_hkl_y then qoffset = 0.
    
    if n_elements(da_xdat) ne n_elements(da_ydat) then begin  ;remove da_xdat and da_ydat if they don't agree with each other
       da_xdat = 0 & tmp = temporary(da_xdat)
       da_ydat = 0 & tmp = temporary(da_ydat)
    endif
    
    ndat = n_elements(xdat)

    if is_E_x then begin
       if self.samp_typ eq 3 then uniq_E=xdat[*,0,0] $
       else uniq_E=xdat[*,0]
    endif
    if is_E_y then begin
       if self.samp_typ eq 3 then uniq_E=ydat[*,0,0] $
       else uniq_E=ydat[*,0]
    endif
    if (self.instrname eq 'macs') and (n_elements(uniq_E) ne 0) then uniq_E = round(temporary(uniq_E)*(1000d),/L64)*0.001

    extra_legend = ''
    tran_invalid = bytarr(2)
    check_fold = bytarr(10) ;u1,u2,u3,E,T,H,...
    is_fold2 = ((self.samp_typ gt 0) and (self.fold_type eq 1))
    is_u3 = ((self.extravaxis_yn[0]) and (self.instrname ne 'macs') and (self.instrname ne 'wand'))
    check_fold[0] = (is_fold2 and self.view_u1fold and finite((*(self.view_u1foldcenter))[0]))
    check_fold[1] = (is_fold2 and self.view_u2fold and finite((*(self.view_u2foldcenter))[0]))
    check_fold[2] = (is_fold2 and is_u3 and self.view_u3fold and finite((*(self.view_u3foldcenter))[0]))
    fold_center = [(*(self.view_u1foldcenter))[0],(*(self.view_u2foldcenter))[0],(*(self.view_u3foldcenter))[0],fltarr(7)]
    
    ;thickness
    if self.disp_flag[0] and (self.traj_z1[self.samp_typ] ne none_id[self.samp_typ]) and (total(finite(self.traj_z1ran)) ne 0) then begin
       self->dm_check_thickness,self.traj_z1[self.samp_typ],range=self.traj_z1ran,ndat=ndat,index=index,daindex=daindex,is_E=is_E_z1,traninvalid=traninvalid,emptyitem=emptyitem,$
           extra_tit=extra_legend,checkfold=check_fold[self.traj_z1[self.samp_typ]],foldcenter=fold_center[self.traj_z1[self.samp_typ]],qoffset=qoffset,debug=debug
       tran_invalid[0] = traninvalid
    endif
    if self.disp_flag[1] and (self.traj_z2[self.samp_typ] ne none_id[self.samp_typ]) and (total(finite(self.traj_z2ran)) ne 0) then begin
       self->dm_check_thickness,self.traj_z2[self.samp_typ],range=self.traj_z2ran,ndat=ndat,index=index1,daindex=daindex1,is_E=is_E_z2,traninvalid=traninvalid,emptyitem=emptyitem,$
           extra_tit=extra_legend,checkfold=check_fold[self.traj_z2[self.samp_typ]],foldcenter=fold_center[self.traj_z2[self.samp_typ]],qoffset=qoffset,debug=debug
       tran_invalid[1] = traninvalid
       if n_elements(index1) ne 0 then begin   
          if n_elements(index) eq 0 then index = index1 $
          else index = dm_common(index,index1,/no_copy)
       endif 
       if n_elements(daindex1) ne 0 then begin
          if n_elements(daindex) eq 0 then daindex = daindex1 $
          else daindex = dm_common(daindex,daindex1,/no_copy)
       endif 
    endif
    
    if n_elements(qoffset) eq 3 then begin
       if is_hkl_x then xtit = self->dm_adjust_xyztit(qoffset,title=xtit,hkl_info=xinfo)
       if is_hkl_y then ytit = self->dm_adjust_xyztit(qoffset,title=ytit,hkl_info=yinfo)
    endif
    
    if n_elements(index) ne 0 then begin
       if index[0] eq -1 then begin
          emptyitem = 'combined thickness'
          tran_invalid[*] = 1b
          extra_legend = ''
       endif else begin
          xdat = xdat[index]
          ydat = ydat[index]
       endelse
       index = -1
    endif
    
    if (n_elements(da_xdat) ne n_elements(daindex)) and (n_elements(daindex) ne 0) then begin
       if (daindex[0] ne -1) and (n_elements(da_xdat) ne 0) then begin
          da_xdat = da_xdat[daindex]
          da_ydat = da_ydat[daindex]
          daindex = -1
       endif else begin ;no dark angle trajectory
          da_xdat = 0 & da_ydat = 0 & tmp = temporary(da_xdat) & tmp = temporary(da_ydat)
       endelse
    endif
    
    ;fold to both sides
    if check_fold[self.traj_x[self.samp_typ]] then begin
       xdat = [xdat,2.0*fold_center[self.traj_x[self.samp_typ]]-xdat]
       ydat = [ydat,ydat]
       if n_elements(da_xdat) ne 0 then begin
          da_xdat = [da_xdat,2.0*fold_center[self.traj_x[self.samp_typ]]-da_xdat]
          da_ydat = [da_ydat,da_ydat]
       endif
    endif
    if check_fold[self.traj_y[self.samp_typ]] then begin
       xdat = [xdat,xdat]
       ydat = [ydat,2.0*fold_center[self.traj_y[self.samp_typ]]-ydat]
       if n_elements(da_xdat) ne 0 then begin
          da_xdat = [da_xdat,da_xdat]
          da_ydat = [da_ydat,2.0*fold_center[self.traj_y[self.samp_typ]]-da_ydat]
       endif
    endif 
  
    if n_elements(emptyitem) ne 0 then $
       ok = dialog_message(['The specified '+emptyitem+' range covers no data.','The entire possible range is used instead.'],dialog_parent=self.tlb,/center)
  
    ;xrange and yrange
    n_xran = total(finite(self.traj_xran))
    n_yran = total(finite(self.traj_yran))
    if n_xran eq 2 then begin
       if self.traj_xran[1] gt self.traj_xran[0] then begin
          index = where((xdat le self.traj_xran[1]) and (xdat ge self.traj_xran[0]),count)
          if count gt 0 then begin
             xdat  = xdat[index]
             ydat  = ydat[index]
             index = -1
             if n_elements(da_xdat) gt 0 then begin
                index = where((da_xdat le self.traj_xran[1]) and (da_xdat ge self.traj_xran[0]),count)
                if count gt 0 then begin
                   da_xdat = da_xdat[index]
                   da_ydat = da_ydat[index]
                   index = -1
                endif else begin
                   da_xdat = 0 & da_ydat = 0 & tmp = temporary(da_xdat) & tmp = temporary(da_ydat)
                endelse
             endif
          endif
       endif
    endif
    if n_yran eq 2 then begin
       if self.traj_yran[1] gt self.traj_yran[0] then begin
          index = where((ydat le self.traj_yran[1]) and (ydat ge self.traj_yran[0]),count)
          if count gt 0 then begin
             xdat  = xdat[index]
             ydat  = ydat[index]
             index = -1
             if n_elements(da_xdat) gt 0 then begin
                index = where((da_ydat le self.traj_yran[1]) and (da_ydat ge self.traj_yran[0]),count)
                if count gt 0 then begin
                   da_xdat = da_xdat[index]
                   da_ydat = da_ydat[index]
                   index = -1
                endif else begin
                   da_xdat = 0 & da_ydat = 0 & tmp = temporary(da_xdat) & tmp = temporary(da_ydat)
                endelse
             endif 
          endif
       endif
    endif
    if n_xran lt 2 then begin
       if finite(self.traj_xran[0]) then begin
          index = where(xdat ge self.traj_xran[0],count)
          if count gt 0 then begin
             xdat  = xdat[index]
             ydat  = ydat[index]
             index = -1
             if n_elements(da_xdat) gt 0 then begin
                index = where(da_xdat ge self.traj_xran[0],count)
                if count gt 0 then begin
                   da_xdat = da_xdat[index]
                   da_ydat = da_ydat[index]
                   index = -1
                endif else begin
                   da_xdat = 0 & da_ydat = 0 & tmp = temporary(da_xdat) & tmp = temporary(da_ydat)
                endelse
             endif
          endif
       endif
       if finite(self.traj_xran[1]) then begin
          index = where(xdat le self.traj_xran[1],count)
          if count gt 0 then begin
             xdat  = xdat[index]
             ydat  = ydat[index]
             index = -1
             if n_elements(da_xdat) gt 0 then begin
                index = where(da_xdat le self.traj_xran[1],count)
                if count gt 0 then begin
                   da_xdat = da_xdat[index]
                   da_ydat = da_ydat[index]
                   index = -1
                endif else begin
                   da_xdat = 0 & da_ydat = 0 & tmp = temporary(da_xdat) & tmp = temporary(da_ydat)
                endelse
             endif
          endif
       endif
    endif
    if n_yran lt 2 then begin
       if finite(self.traj_yran[0]) then begin
          index = where(ydat ge self.traj_yran[0],count)
          if count gt 0 then begin
             xdat  = xdat[index]
             ydat  = ydat[index]
             index = -1
             if n_elements(da_xdat) gt 0 then begin
                index = where(da_ydat ge self.traj_yran[0],count)
                if count gt 0 then begin
                   da_xdat = da_xdat[index]
                   da_ydat = da_ydat[index]
                   index = -1
                endif else begin
                   da_xdat = 0 & da_ydat = 0 & tmp = temporary(da_xdat) & tmp = temporary(da_ydat)
                endelse
             endif
          endif
       endif
       if finite(self.traj_yran[1]) then begin
          index = where(ydat le self.traj_yran[1],count)
          if count gt 0 then begin
             xdat  = xdat[index]
             ydat  = ydat[index]
             index = -1
             if n_elements(da_xdat) gt 0 then begin
                index = where(da_ydat le self.traj_yran[1],count)
                if count gt 0 then begin
                   da_xdat = da_xdat[index]
                   da_ydat = da_ydat[index]
                   index = -1
                endif else begin
                   da_xdat = 0 & da_ydat = 0 & tmp = temporary(da_xdat) & tmp = temporary(da_ydat)
                endelse
             endif
          endif
       endif
    endif

    if n_elements(xdat) gt 1e5 then begin
       if (self.instrname eq 'macs') and (n_elements(uniq_E) ne 0) then begin
          if is_E_x then xdat = round(temporary(xdat)*(1000d),/L64)*0.001 
          if is_E_y then ydat = round(temporary(ydat)*(1000d),/L64)*0.001
       endif
       xmax = max(xdat,min=xmin)
       ymax = max(ydat,min=ymin)
       nogrid = 0
       if is_E_x then begin
          nogrid = 1
          index  = where((uniq_E ge xmin) and (uniq_E le xmax),count)
          if count ne 0 then uniq_E=uniq_E[index]
          index  = -1
       endif
       if is_E_y then begin
          nogrid = 2
          index  = where((uniq_E ge ymin) and (uniq_E le ymax),count)
          if count ne 0 then uniq_E=uniq_E[index]
          index  = -1
       endif
       xstep = (xmax-xmin)/300.
       ystep = (ymax-ymin)/300.
       if xstep eq 0 then xstep=0.1
       if ystep eq 0 then ystep=0.1
       zdat  = fltarr(n_elements(xdat))+1
       if nogrid eq 1 then extra={uniq_xval:uniq_E} $
       else if nogrid eq 2 then extra={uniq_yval:uniq_E}
       dm_stepgrid_bin,xstep,ystep,xdat,ydat,zdat,extlib=self.bin_extlib,debug=debug,_extra=extra,group_leader=self.tlb
       index = where(finite(temporary(zdat)),count)
       if count ne 0 then begin
          n_x   = n_elements(xdat)
          xdat  = xdat[index mod n_x]
          ydat  = ydat[fix(index/n_x,type=3)]
          index = -1
       endif
    endif
    if keyword_set(debug) then help,xdat

    legend = '' & title = ''
    ;extra info in the title and legend
    if self.instrname eq 'dcs' then begin
       title = '\lambda='+dm_to_string(sqrt(81.8042/self.eief),resolution=2)+' \AA' 
    endif else begin
       if self.instrname eq 'macs' then begin
          if self.macsintn_typ eq 0 then begin  ;'SPEC'
             title = (['Ei=','Ef='])[self.instrgeom]+dm_to_string(self.eief,resolution=3)+' meV, Detector=SPEC'
          endif else begin                      ;'DIFF' or 'DIFF+SPEC'
             if self.instrgeom eq 0 then Ei = [self.eief,self.eief] $
             else Ei = [self.eief,self.eief,0]+self.estartend 
             if Ei[0] eq Ei[1] then title = 'Ei='+dm_to_string(Ei[0],resolution=3)+' meV' $
             else title = 'Ei=['+dm_to_string(Ei,separator=':')+'] meV'
             title = title +', Detector=DIFF'
          endelse
       endif else if self.instrname ne 'wand' then begin
          title = (['Ei=','Ef='])[self.instrgeom]+dm_to_string(self.eief,resolution=3)+' meV'
       endif
    endelse
    if self.instrname eq 'macs' then begin
       kid_fnt = finite(dm_to_number(self.kidney[0:2]))
       if (~lmgr(/vm)) and stregex(self.kidney[0],'ul|ll|cn',/boolean,/fold_case) then kid_fnt[0] = 0  ;need to replace ul, ll, and cn
       if (~lmgr(/vm)) and stregex(self.kidney[1],'ul|ll|cn',/boolean,/fold_case) then kid_fnt[1] = 0
       kidran = self.kidney[0:2]+' \deg'
       if ~kid_fnt[0] then kidran[0] = dm_calckidney((strlen(self.kidney[0]) eq 0)?'ll':self.kidney[0])
       if ~kid_fnt[1] then kidran[1] = dm_calckidney((strlen(self.kidney[1]) eq 0)?'ul':self.kidney[1])
       step = dm_to_number(self.kidney[2])
       if kid_fnt[2] then begin
          if step eq 0 then legend = 'kidney='+kidran[0] $
          else legend = 'kidney=['+kidran[0]+':'+kidran[1]+':'+kidran[2]+']'
       endif else $
          legend = 'kidney=['+kidran[0]+':'+kidran[1]+']'
    endif
    switch self.samp_typ of
        1:  begin
            if self.instrname eq 'macs' then begin
               if (self.estartend[2] eq 0) and (~is_E_x) and (~is_E_y) and (~keyword_set(is_E_z1)) then title = title+', E='+dm_to_string(self.estartend[0],resolution=3)+' meV'
               if finite(self.psi[2]) then begin
                  if self.psi[2] eq 0 then legend = legend+', A3='+dm_to_string(self.psi[0])+' \deg' $
                  else legend = legend+',  A3=['+dm_to_string(self.psi[0])+' \deg:'+dm_to_string(self.psi[1])+' \deg:'+dm_to_string(self.psi[2])+' \deg]'
               endif else $
                  legend = legend+', A3=['+dm_to_string(self.psi[0])+' \deg:'+dm_to_string(self.psi[1])+' \deg]'
               if ptr_valid(self.a3mask) then legend = legend+', a3 mask applied'
            endif else $
               legend = 'psi='+dm_to_string(self.orie_psi)+' \deg'
            break
            end
        2:
        3:  begin
            if finite(self.psi[2]) then begin
               if self.psi[2] eq 0 then legend = 'psi='+dm_to_string(self.psi[0])+' \deg' $
               else legend = 'psi=['+dm_to_string(self.psi[0])+' \deg:'+dm_to_string(self.psi[1])+' \deg:'+dm_to_string(self.psi[2])+' \deg]'
            endif else $
               legend = 'psi=['+dm_to_string(self.psi[0])+' \deg:'+dm_to_string(self.psi[1])+' \deg]'
            break
            end
        else:    
    endswitch
    if ((self.samp_typ ge 2) or ((self.samp_typ eq 1) and (self.instrname eq 'macs'))) and self.datr_yn and finite(self.datr_psi) and (~ptr_valid(self.dataStrPtr)) $
       and (n_elements(da_xdat) eq 0) then begin
       legend = legend+', dark angle='+dm_to_string(self.datr_psi)+' \deg'  ;dark angle trajectory doesn't exist, but dark angle specified, show this info anyway
    endif
    if (strlen(legend) eq 0) or (strlen(extra_legend) eq 0) then sep='' else sep=', '
    legend = legend+sep+extra_legend
    if (self.instrname ne 'wand') and (self.samp_typ ge 2) and (~keyword_set(is_E_x)) and (~keyword_set(is_E_y)) and (~keyword_set(is_E_z1)) and (~keyword_set(is_E_z2)) then begin
       case self.e_bin[self.samp_typ] of
          0:  title = title
          1:  title = title+', Elastic'
          2:  title = self->range_string(title,'E',self.e_range, 'meV')
       endcase
    endif
    if (self.instrname eq 'dcs') and (total(self.dcsdetmask[1:3]) ne 0) then begin  ;add mask info for DCS
       ind = where(1-self.dcsdetmask[1:3],count)
       if count gt 0 then begin
          title = title+', '+(['Lower','Central','Upper'])[ind[0]]
          for i=1,count-1 do title = title+'+'+(['Lower','Central','Upper'])[ind[i]]
          title = title+' Bank'+(['','s'])[count gt 1]
       endif
    endif

    ;plot the data
    isnew = 0b
    if self.useitool then begin
       ok = dialog_message('Itools plotting is not implemented yet. Soooorry!',dialog_parent=self.tlb)
       return
    endif else begin
       self.current = self->searchcurrent(crossplot=crossplot,keep=keep)
       if (~obj_valid(self.current)) or (keyword_set(crossplot) and (~keyword_set(overplot))) or keyword_set(keep) then begin
          self.current = obj_new('dm_plot',xdat,ydat,xtit=xtit,ytit=ytit,/no_copy,/showxgrid,legdpos=[0.03,0.28],/showygrid,parentobj=self,linestyle='no line',psym='dot',color='green',$
                symsize=0.025,xran=self.traj_xran,yran=self.traj_yran,layer=5,title=title,legend=legend,path=self.dirs[1],background=(['black','white'])[self.pbgcol_bw],vt_col=self.ppvc_choice,$
                tickdir=self.tdir_choice,xsize=(*self.pdim_list)[0,self.pdim_choice],ysize=(*self.pdim_list)[1,self.pdim_choice],render_method=self.ptrm_choice,notooltip=self.pnotooltip,/isolatin1,$
                wtitle='Trajectory')
          isnew = 1b & crossplot = 0b & self.traj_layer = 0
       endif else begin
          if keyword_set(overplot) then begin
             if (~ isnew) then self.traj_layer=self.traj_layer+1
             self.current->add_plot,xdat,ydat+0.005*((self.instrname eq 'macs')?1:round(self.traj_layer/2.))*(-1)^(self.traj_layer),/no_copy,psym='dot',linestyle='no line',symsize=0.025,$
                color=(['green','red','magenta','cyan','blue'])[(self.traj_layer mod 5)],layer=5+self.traj_layer,legend=legend
          endif else begin
             self.traj_layer = 0
             self.current->setproperty,xdat=xdat,ydat=ydat,xtit=xtit,ytit=ytit,/no_copy,ylog=0,xran=self.traj_xran,yran=self.traj_yran,color='green',psym='dot',linestyle='no line',$
                symsize=0.025,title=title,/nodraw,layer=5,/showxgrid,/showygrid,gridontop=1,hidelegend=0,legend=legend,bgcolor=self.pbgcol_bw,vt_col=self.ppvc_choice,tickdir=self.tdir_choice,$
                wtitle='Trajectory'
          endelse
       endelse
       ;add powder lines in diffuse scattering mode and powder mode
       if ptr_valid(self.powd_line) and (isnew or (~ keyword_set(overplot))) then begin
          tmp = size(*self.powd_line) 
          if tmp[0] eq 2 then tmp=1 else tmp=tmp[tmp[0]] 
          if self.samp_typ eq 0 then begin    ;powder type
             if self.traj_y[self.samp_typ] eq 1 then ix = 1
             if self.traj_x[self.samp_typ] eq 1 then ix = 0
             if n_elements(ix) ne 0 then iy = 1-ix
          endif else begin
             ix = self.traj_x[self.samp_typ]
             iy = self.traj_y[self.samp_typ]
             if (self.instrname ne 'macs') and (self.instrname ne 'wand') then imax = 1+self.extravaxis_yn[0] else imax = 1
             if max([ix,iy]) gt imax then iy = temporary(ix)  ;no powder line
          endelse
          if n_elements(ix) ne 0 then begin
             for i=0L,tmp-1 do begin
                 self.current->add_plot,(*self.powd_line)[*,ix,i],(*self.powd_line)[*,iy,i],psym='no symbol',color='dark yellow',linestyle='solid',layer=i,$
                        legend=(['',(['Al','Cu','Stainless Steel','Specified'])[(self.powd_typ-1)>0]+' powder line'])[i eq 0]
             endfor
          endif
       endif
       ;add dark angle trajectory
       if n_elements(da_xdat) ne 0 then begin
          self.current->add_plot,da_xdat,da_ydat,psym='dot',color='dark grey',symsize=0.025,linestyle='no line',layer=0,legend='dark angle='+dm_to_string(self.datr_psi)+' \deg'
       endif
       if obj_valid(self.current) then self.current->draw else return
    endelse
    if ~keyword_set(crossplot) then begin
       if keyword_set(overplot) and (~isnew)then $
          self->dm_add_plotwindow,type=2,title=strtrim((strsplit(xtit,'()',/extract))[0],2)+' vs. '+strtrim((strsplit(ytit,'()',/extract))[0],2)+'[overplot]'  $
       else $
          self->dm_add_plotwindow,type=2,title=strtrim((strsplit(xtit,'()',/extract))[0],2)+' vs. '+strtrim((strsplit(ytit,'()',/extract))[0],2)
    endif
    if keyword_set(debug) then print,'plot_trajectory finished in ',systime(/sec)-current,' sec.'
end