; $Id: $
;#######################################################################
; 
; NAME:
;  dm_load_wand
;
; PURPOSE:
;  load wand file returns data[ndet,npsi], psi[npsi], temperature[npsi]
;
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  April, 2020
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

pro dm_load_wand,file,data=data,psi=psi,temperature=temperature,error=error,group_leader=group_leader,headeronly=headeronly
    ;check if the file is compressed
    if strlowcase(strmid(file,2,3,/reverse_offset)) eq '.gz' then compress=1 else compress=0

    openr,unit,file,/get_lun,error=error,compress=compress
    if error ne 0 then begin  
       ok = dialog_message("Can't open "+file,/error,dialog_parent=group_leader)
       return
    endif

    ;catch and clear possible io errors
    catch, myerror
    if myerror ne 0 then begin
       error = 1b
       catch,/cancel
       if n_elements(unit) ne 0 then free_lun,unit
       ok = dialog_message(!error_state.msg,/error)
       return
    end
    
    nline = file_lines(file,compress=compress)
    tmp = ''
    readf,unit,tmp
    tmp = dm_to_number(strsplit(tmp,' ,'+string(9b),/extract))
    if n_elements(tmp) ne 642 then begin
       ok = dialog_message([file_basename(file)+" doesn't have the right format.",$
              "Each line in the file should be: Psi Temperature I1 I2 I3 ... I640."],/error,dialog_parent=group_leader)
       free_lun,unit
       error = 1 
       return
    endif else begin
       data = tmp[2:641]
       psi  = tmp[0]
       temperature = tmp[1]
    endelse
       
    if nline gt 1 then begin
       if keyword_set(headeronly) then begin 
          tmp0 = 0.0 & tmp1 = 0.0 & tmp = ' '
          while (~ eof(unit)) do begin
             readf,unit,tmp0,tmp1,tmp
             psi = [psi,tmp0]
             temperature = [temperature,tmp1]
          endwhile
       endif else begin
          point_lun,unit,0
          data = fltarr(642,nline)
          readf,unit,data
          psi = reform(data[0,*])
          temperature = reform(data[1,*])
          data = data[2:641,*]
       endelse
    endif
    free_lun,unit
end

;this procedure reads a wand file for dcs_mslice program
;Parameters:
;   open_path:   file path, string
;   open_file:   file name(s), string or string array
;Keywords:
; input:
;   masklimit:   a number indicating the threshhold below which the detector is masked, default:-1e20
;   parent:      message widget parent,  default is self.tlb
;   sum:         if set, the loaded files are summed over all psi   
;   title:       message title
; output:
;   error:       error flag 1 or 0
;   eief:        incident energy
;   emean:       energy transfer
;   qty:         [ndet,npsi]
;   dqty:        [ndet,npsi] 
;   ang:         [npsi] 
;   temperature: [npsi]
;   weight:      weight or duration info , not in the file, need to be typed in by the user
;   zero_error:  return error bar for sqrt(intensity=1)
pro dcs_mslice::dm_load_wand,open_path,open_file,ang=ang,eief=eief,emean=emean,error=error,parent=parent,$
    qty=qty,dqty=dqty,scriptmode=scriptmode,sum=sum,title=title,masklimit=masklimit,temperature=temperature,weight=weight,zero_error=zero_error
    eief  = self.eief
    emean = 0.0                ;elastic
    if n_elements(masklimit)   eq 0 then masklimit = -1e20
    if n_elements(parent) eq 0 then parent = self.tlb
    if n_elements(title)  eq 0 then title  = 'Loading data ...'
    error   = 0b  ;initialize no error
    n_files = n_elements(open_file)
    n_read  = 0

    ;catch and clear possible io errors
    catch, myerror
    if myerror ne 0 then begin
       error = 1b
       catch,/cancel
       if obj_valid(mesg) then obj_destroy,mesg
       ok = dialog_message(!error_state.msg,/error)
       return
    end
    
    ;ask for weight info
    myweight = 1.0
    if (arg_present(weight) or (n_files ne 1)) and (~keyword_set(scriptmode)) then begin
       myweight = !values.f_nan & default = '1'
       while(total(finite(myweight)) ne n_files) do begin
            myweight = dm_dialog_listinput(open_file,title='Please input weight info:',default=default,$
                 label='weight',dialog_parent=parent,reset=1,info=['The weight is for summing files and weighted mean average binning, not for normalizing the intensity.',$
                 "Press OK if you don't know the information."],xvalue=open_file)
            default = myweight & myweight = dm_to_number(myweight)
       endwhile
    endif

    for i=0L,n_files-1L do begin
        file = open_path+self.pathsep+open_file[i]
        if i eq 0 then $
           mesg =  obj_new('dm_progress',title=title,message='Loading '+file,group_leader=parent) $
        else $
           mesg->update,message='loading '+file,title='Loading...  '+strtrim(string(i+1),2)+'/'+strtrim(string(n_files),2)

        ;read the wand file
        dm_load_wand,file,data=data,psi=psi,temperature=temp,error=error,group_leader=parent
        if error ne 0 then begin
           if obj_valid(mesg) then obj_destroy,mesg
           return
        endif 
        
        mask = where(data[*,0] le masklimit,mskcount)
        if mskcount gt 0 then begin
           data[mask,*] = !values.f_nan
        endif
     
        if self.tempavg_yn then begin ;averaging temperature of each file
           temp[*] = mean(temp)
        endif
           
        if i eq 0 then begin
           qty  = data
           ang  = psi
           temperature = temp
           weight = replicate(myweight[i],n_elements(psi))
        endif else begin
           qty  = [[qty],[data]]
           ang  = [ang,psi]
           temperature = [temperature,temp]
           weight = [weight,replicate(myweight[i],n_elements(psi))]
        endelse
        if (~ obj_valid(mesg)) then break
    endfor
    dqty = sqrt(qty)
    zero_error = 1.0
    if keyword_set(sum) then begin
       for i=0,n_elements(ang)-1 do begin
           qty[*,i] = qty[*,i]*weight[i]
       endfor
       qty  = total(qty,2,/NAN) & weight = total(weight)
       if weight eq 0 then weight = 1.0
       dqty = sqrt(qty)/weight
       qty  = qty/weight
       ang  = mean(ang)
       temperature = mean(temperature)
       zero_error = zero_error/weight
    endif
    
    if obj_valid(mesg) then obj_destroy,mesg
end