; $Id: $
;#######################################################################
;
; NAME:
;  dm_load_parameter
;
; PURPOSE:
;  load parameter file
;
; CATEGORY:
;  dcs_mslice
;
; AUTHOR:
;  Yiming Qiu
;  NIST Center for Neutron Research
;  100 Bureau Drive, Gaithersburg, MD 20899-6102
;  United States
;  yiming.qiu@nist.gov
;  March, 2025
;
; LICENSE:
;  The software in this file is written by an employee of
;  National Institute of Standards and Technology
;  as part of the DAVE software project.
;
;  The DAVE software package is not subject to copyright protection
;  and is in the public domain. It should be considered as an
;  experimental neutron scattering data reduction, visualization, and
;  analysis system. As such, the authors assume no responsibility
;  whatsoever for its use, and make no guarantees, expressed or
;  implied, about its quality, reliability, or any other
;  characteristic. The use of certain trade names or commercial
;  products does not imply any endorsement of a particular product,
;  nor does it imply that the named product is necessarily the best
;  product for the stated purpose. We would appreciate acknowledgment
;  if the DAVE software is used or if the code in this file is
;  included in another product.
;
;#######################################################################

pro dcs_mslice::dm_load_parameter,parmfile,init=init,session=session,debug=debug
    WIDGET_CONTROL,/HOURGLASS
    openr,unit,parmfile,/get_lun
    tmp  = ''
    mesg = ''
    if self.samp_typ eq -1 then begin
       n_item = n_elements((*self.dataPtr).label)
       while(~ eof(unit)) do begin
          readf,unit,tmp 
          tmp  = strtrim(tmp,2)
          head = strmid(tmp,0,8)
          tmp  = strmid(tmp,8,strlen(tmp)-8)
          case strlowcase(head) of
            'samptyp:':begin
                       tmpvalue = dm_to_number(tmp,/int)
                       if tmpvalue ne -1 then begin
                          free_lun,unit
                          if (~ keyword_set(init)) then $
                             ok = dialog_message('Wrong type of parameter file.',/error,dialog_parent=self.tlb)
                          return
                       endif
                       end
            'datadir:':self->setproperty,dataDir=tmp
            'workdir:':self->setproperty,workDir=tmp  
            'selfshd:':self.ssfac          = dm_to_number(tmp,/float)  
            'binmeth:':self.bin_avgsum     = dm_to_number(tmp,/int)
            'bincons:':self.bin_conststep  = dm_to_number(tmp,/int)
            'binextl:':self.bin_extlib     = dm_to_number(tmp,/int)
            'bgcolor:':self.pbgcol_bw      = dm_to_number(tmp,/int)
            'maskfil:':self.mask_filler    = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'toloptn:':self.empt_tol_optn  = dm_to_number(tmp,/int)
            'datatol:':begin
                       tmpvalue            = dm_to_number(strsplit(tmp,'&',/extract),/float)
                       n_tmp1              = n_elements(tmpvalue)
                       (*self.data_tolerance)[0:((n_tmp1<(n_item-2))-1)] = tmpvalue[0:((n_tmp1<(n_item-2))-1)] 
                       end
            'pvt_col:':self.ppvc_choice    = dm_to_number(tmp,/int)
            'tickdir:':self.tdir_choice    = dm_to_number(tmp,/int)
            'pxysize:':self.pdim_choice    = 0>(dm_to_number(tmp,/int))<(n_elements((*self.pdim_list)[0,*])-1)
            'volselx:':self.volsel_x[*]    = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'volsely:':self.volsel_y[*]    = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'volselz:':self.volsel_z[*]    = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'volxran:':self.vol_xran       = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'volx_st:':self.vol_xstep      = dm_to_number(tmp,/float)
            'volyran:':self.vol_yran       = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'voly_st:':self.vol_ystep      = dm_to_number(tmp,/float)
            'volzran:':self.vol_zran       = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'volz_st:':self.vol_zstep      = dm_to_number(tmp,/float)
            'voliran:':self.vol_iran       = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'vol_smo:':self.vol_smooth     = dm_to_number(tmp,/int)
            'volintp:':self.vol_interp     = dm_to_number(tmp,/int)
            'vol_col:':self.vol_color      = dm_to_number(tmp,/int)
            'vol_slt:':if n_item gt 5 then begin
                       tmpvalue            = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
                       n_tmp1              = n_elements(tmpvalue)
                       n_tmp2              = n_item-5
                       (*self.volsel_t)[0:((n_tmp1<n_tmp2)-1)] = tmpvalue[0:((n_tmp1<n_tmp2)-1)] 
                       endif
            'vol_tft:':if n_item gt 5 then begin
                       tmpvalue            = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
                       n_tmp1              = n_elements(tmpvalue)/2
                       n_tmp2              = n_item-5
                       for i=0,(n_tmp1<n_tmp2)-1 do (*self.volsel_tft)[i,*] = tmpvalue[2*i:2*i+1]
                       endif 
            'dispslh:':self.slice_x[*]     = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'dispslv:':self.slice_y[*]     = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'dispx_r:':self.slice_xran     = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'dispxst:':self.slice_xstep    = dm_to_number(tmp,/float)
            'dispy_r:':self.slice_yran     = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'dispyst:':self.slice_ystep    = dm_to_number(tmp,/float)
            'dispi_r:':self.slice_iran     = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'dispslt:':if n_item gt 4 then begin
                       tmpvalue            = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
                       n_tmp1              = n_elements(tmpvalue)
                       n_tmp2              = n_item-4
                       (*self.slice_t)[0:((n_tmp1<n_tmp2)-1)] = tmpvalue[0:((n_tmp1<n_tmp2)-1)] 
                       endif
            'disptft:':if n_item gt 4 then begin
                       tmpvalue            = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
                       n_tmp1              = n_elements(tmpvalue)/2
                       n_tmp2              = n_item-4
                       for i=0,(n_tmp1<n_tmp2)-1 do (*self.slice_tft)[i,*] = tmpvalue[2*i:2*i+1]
                       endif
            'dispsmo:':self.slice_smooth   = dm_to_number(tmp,/int)
            'dispker:':self.slice_sm_kernel[0:8] = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'dispsha:':self.slice_shading  = dm_to_number(tmp,/int)
            'dispcol:':self.slice_color    = dm_to_number(tmp,/int)
            'dispcut:':self.slice_cutcolor = dm_to_number(tmp,/int)
            'cut___x:':self.cut_x[*]       = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'cutxran:':self.cut_ran        = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'cut_xst:':self.cut_step       = dm_to_number(tmp,/float)    
            'cut___y:':self.cut_y[*]       = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'cutyran:':self.cut_yran       = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'cut_sym:':self.cut_symbol     = dm_to_number(tmp,/int)
            'cut_col:':self.cut_color      = dm_to_number(tmp,/int)
            'userdef:':self.coldefine[0:2] = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'cutline:':self.cut_line       = dm_to_number(tmp,/int)  
            'cut_slt:':if n_item gt 3 then begin
                       tmpvalue            = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
                       n_tmp1              = n_elements(tmpvalue)
                       n_tmp2              = n_item-3
                       (*self.cutsel_t)[0:((n_tmp1<n_tmp2)-1)] = tmpvalue[0:((n_tmp1<n_tmp2)-1)] 
                       endif
            'cut_tft:':if n_item gt 3 then begin
                       tmpvalue            = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
                       n_tmp1              = n_elements(tmpvalue)/2
                       n_tmp2              = n_item-3
                       for i=0,(n_tmp1<n_tmp2)-1 do (*self.cutsel_tft)[i,*] = tmpvalue[2*i:2*i+1] 
                       endif
            else:
          endcase
       endwhile 
       if (~ keyword_set(debug)) then begin
          if self.bin_extlib and (self.bin_avgsum eq -1) then self.bin_avgsum = 0 ;weighted mean not implemented in external library
          (*self.colors)[0] = (['white','black'])[self.pbgcol_bw]
          ;some warning messages
          if (~lmgr(/vm)) and (self.ssfac ne 1) then mesg = [mesg,'Background subtraction factor is '+dm_to_string(self.ssfac)+'.']
          if (~lmgr(/vm)) and (self.empt_tol_optn eq 1) then mesg = [mesg,'The background subtraction tolerance method is "Within Tolerance of the Closest".']
          if self.bin_avgsum ne 0 then mesg = [mesg,'Binning method is '+(['Weighted Mean','Arithmetic Mean','Sum'])[self.bin_avgsum+1]+'.']
          if finite(self.mask_filler[0]) then mask = 'less than or equal to '+dm_to_string(self.mask_filler[0]) else mask = ''
          if finite(self.mask_filler[1]) then mask = mask+((strlen(mask) eq 0)?'':' and ')+'greater than or equal to '+dm_to_string(self.mask_filler[1])
          if strlen(mask) ne 0 then mesg = [mesg,'Masking data '+mask+'.']
          if (~ keyword_set(init)) then begin
             self->reset_display,/reset
             self->reset_menu
             self->data_subbg
          endif
       endif               
    endif else begin
       ptr_free,self.absp_info         ;clear any existing self-shielding information
       clear_history = 1b
       newftype      = 0b
       dcsdetmask    = self.dcsdetmask
       while(~ eof(unit)) do begin
          readf,unit,tmp
          tmp  = strtrim(tmp,2)
          head = strmid(tmp,0,8)
          tmp  = strmid(tmp,8,strlen(tmp)-8)
          case strlowcase(head) of
            'samptyp:':begin
                       tmpvalue = dm_to_number(tmp,/int)
                       if tmpvalue eq -1 then begin
                          free_lun,unit
                          if (~ keyword_set(init)) then $
                             ok = dialog_message('Wrong type of parameter file.',/error,dialog_parent=self.tlb)
                          return
                       endif
                       self.samp_typ     = tmpvalue
                       end
            'ncnrftp:':self.ncnrftp      = dm_to_number(tmp,/int)
            'sshield:':begin
                       tmpvalue = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
                       if n_elements(tmpvalue) eq ([6,7,6,8])[tmpvalue[0]] then tmpvalue = [tmpvalue[0:2],1,tmpvalue[3:*]] 
                       self.absp_info    = ptr_new(tmpvalue)
                       end
            'l1_fact:':self.absp_l1factor= dm_to_number(tmp)
            'in_geom:':self.instrgeom    = dm_to_number(tmp,/int)
            'powd_la:':self.powd_latta   = dm_to_number(tmp)
            'powd_bc:':self.powd_bcc     = dm_to_number(tmp,/int)
            'viewwin:':self.view[0:1]    = (dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int))[0:1]
            'f__type:':begin
                       old               = self.ftype
                       self.ftype        = dm_to_number(tmp,/int)
                       newftype          = (old ne self.ftype)
                       end 
            'zeroerr:':self.zero_error[0] = dm_to_number(tmp)    
            'binzerr:':self.bin_zeroerror = dm_to_number(tmp,/int)  
            'statest:':begin
                       tmpvalue           = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
                       self.bin_statrange[0:(n_elements(tmpvalue)-1)] = tmpvalue
                       end      
            'ftlflag:':begin
                       tmpvalue          = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
                       self.ftoolflag[0] = tmpvalue[0]  ;only load the first flag
                       end
            'koffset:':self.kidney_offset = dm_to_number(tmp)  
            'psi_tol:':begin
                       tmpvalue = abs(dm_to_number(strsplit(tmp,'&',/extract,/preserve)))
                       self.psi_tolerance[0:(n_elements(tmpvalue)-1)] = tmpvalue
                       end
            'macstol:':begin
                       tmpvalue = abs(dm_to_number(strsplit(tmp,'&',/extract,/preserve)))
                       self.macs_tolerance[0:(n_elements(tmpvalue)-1)] = tmpvalue
                       end
            'macslm2:':self.macs_lamb2    = dm_to_number(tmp,/int)
            'macsmon:':begin
                       tmpvalue           = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
                       self.macs_mon[1,0:(n_elements(tmpvalue)-1)] = tmpvalue
                       end
            'macsttp:':self.macstemp_typ  = dm_to_number(tmp,/int)
            'macsseq:':self.macs_seq_geom = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'macshis:':self.macshistomode = dm_to_number(tmp,/int)
            'macswid:':self.macshistowidth= dm_to_number(tmp)
            'macsunt:':self.macshistounit = dm_to_number(tmp,/int)
            'macsnch:':self.macshistonchan= dm_to_number(tmp,/long)
            'macscut:':self.macshistocut  = dm_to_number(tmp,/int)
            'macsraw:':self.macs_rawintn  = dm_to_number(tmp,/int)
            'a3_mask:':begin
                       self->clear_mask,/a3
                       self.a3mask = ptr_new(tmp)
                       end
            'bga3msk:':begin
                       self->clear_mask,/bga3
                       self.bga3mask = ptr_new(tmp)
                       end
            'tempavg:':self.tempavg_yn    = dm_to_number(tmp,/int) 
            'toloptn:':self.empt_tol_optn = dm_to_number(tmp,/int)
            'dcsmask:':begin
                       tmpvalue = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
                       self.dcsdetmask[0:n_elements(tmpvalue)-1] = tmpvalue 
                       end
            'datrpsi:':self.datr_psi     = dm_to_number(tmp)
            'datrwid:':self.datr_width   = abs(dm_to_number(tmp))
            'dcor_eq:':self.dcor_eqran[0:3] = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'dcor_yn:':self.dcor_yn      = dm_to_number(tmp,/int)
            'evax_yn:':begin
                       tmpvalue = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
                       self.extravaxis_yn[0:n_elements(tmpvalue)-1] = tmpvalue 
                       end
            'evax_sp:':self.extravaxis_spec = strsplit(tmp,string('A4'xb),/extract,/preserve)      
            'eemp_yn:':self.extraempcan_yn = dm_to_number(tmp,/int)
            'emp_fac:':begin
                       tmpvalue = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
                       self.empcanfactor[0:n_elements(tmpvalue)-1] = tmpvalue 
                       end            
            'edep_yn:':self.edep_yn      = dm_to_number(tmp,/int)
            'edep_dt:':edep_det          = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'bitflip:':self.bitflip      = dm_to_number(tmp,/int)
            'inxerev:':self.inxerev      = dm_to_number(tmp,/int)
            'dcs2thc:':self.tthcorr      = dm_to_number(tmp,/int)
            'eadj_yn:':self.eadjust      = dm_to_number(tmp,/int)
            'eadj_sp:':self.eadjust_spec = dm_to_number(tmp,/float)
            'eadj_cf:':self.eadjust_checkfirst = dm_to_number(tmp,/int)
            'eint_yn:':self.eint_yn      = dm_to_number(tmp,/int)
            'selfshd:':self.ssfac        = dm_to_number(tmp,/float)
            'ssdfile:':begin
                       tmpvalue = strsplit(tmp,string('A4'xb),/extract,/preserve)
                       if n_elements(tmpvalue) gt 1 then det_or_en = dm_to_number(tmpvalue[0],/int)
                       tmp = tmpvalue[n_elements(tmpvalue)-1]
                       self->load_ssfac,tmp,ssfac=ssfac
                       if n_elements(ssfac) ne 0 then begin
                          ssfacptr_changed = 1b
                          ptr_free,self.ssfacPtr
                          self.ssfacPtr = ptr_new({file:tmp,ssfac:ssfac,det_or_en:keyword_set(det_or_en)})
                       endif
                       end
            'datadir:':dataDir = tmp     ;need to set file type first to work for ftp directory
            'workdir:':self->setproperty,workDir=tmp
            'intnlab:':self.intn_lab     = tmp
            'intntyp:':self.intn_typ     = dm_to_number(tmp,/int)
            'binmeth:':self.bin_avgsum   = dm_to_number(tmp,/int)
            'bincons:':self.bin_conststep= dm_to_number(tmp,/int)
            'binextl:':self.bin_extlib   = dm_to_number(tmp,/int)
            'bgcolor:':self.pbgcol_bw    = dm_to_number(tmp,/int)
            'pvt_col:':self.ppvc_choice  = dm_to_number(tmp,/int)
            'tickdir:':self.tdir_choice  = dm_to_number(tmp,/int)
            'pxysize:':self.pdim_choice  = 0>(dm_to_number(tmp,/int))<(n_elements((*self.pdim_list)[0,*])-1)
            'ebinsiz:':begin
                       tmpvalue          = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
                       if n_elements(tmpvalue) eq 3 then tmpvalue=[tmpvalue[1],tmpvalue[0],tmpvalue[2],1,1] $
                       else if n_elements(tmpvalue) eq 4 then tmpvalue=[tmpvalue,1]
                       self.e_bin        = tmpvalue
                       end
            'e_range:':self.e_range      = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'mon_fit:':begin
                       tmpvalue          = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
                       if n_elements(tmpvalue) eq 1 then tmpvalue = [tmpvalue,0]
                       self.mon_sumfit   = tmpvalue[0:1]
                       end
            'mon_typ:':begin
                       tmpvalue          = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
                       if n_elements(tmpvalue) eq 1 then tmpvalue = [tmpvalue,1]
                       self.mon_typ      = [0>(tmpvalue[0])<2,0>(tmpvalue[1])<2]
                       end
            'ortharb:':self.view_ortharbi= dm_to_number(tmp,/int)
            'latt__a:':self.latt_parm[0] = dm_to_number(tmp,/double)
            'latt__b:':self.latt_parm[1] = dm_to_number(tmp,/double)
            'latt__c:':self.latt_parm[2] = dm_to_number(tmp,/double)
            'latt_aa:':self.latt_parm[3] = dm_to_number(tmp,/double)
            'latt_bb:':self.latt_parm[4] = dm_to_number(tmp,/double)
            'latt_cc:':self.latt_parm[5] = dm_to_number(tmp,/double)
            'orie_ux:':self.orie_u[0]    = dm_to_number(tmp,/float)
            'orie_uy:':self.orie_u[1]    = dm_to_number(tmp,/float)
            'orie_uz:':self.orie_u[2]    = dm_to_number(tmp,/float)
            'orie_vx:':self.orie_v[0]    = dm_to_number(tmp,/float)
            'orie_vy:':self.orie_v[1]    = dm_to_number(tmp,/float)
            'orie_vz:':self.orie_v[2]    = dm_to_number(tmp,/float)
            'lockluv:':self.locklattuv   = dm_to_number(tmp,/int)
            'chckluv:':self.checklattuv  = dm_to_number(tmp,/int)
            'oriepsi:':begin
                       orie_psi          = dm_to_number(tmp,/float)
                       self.orie_psi     = orie_psi
                       end
            'orieoff:':self.orie_offset  = dm_to_number(tmp,/float)
            'foldtyp:':self.fold_type    = dm_to_number(tmp,/int)
            'symrtyp:':symrot_type       = dm_to_number(tmp,/int)       ;backward compatible for older parameter file
            'mrrftyp:':mrrf_type         = dm_to_number(tmp,/int)       ;backward compatible for older parameter file
            'num_rot:':self.numrot_optn  = dm_to_number(tmp,/int)
            'statvec:':self.symrot_startvec = dm_to_number(tmp,/float)
            'view1tp:':self.view_u1typ   = dm_to_number(tmp,/float)
            'view2tp:':self.view_u2typ   = dm_to_number(tmp,/float)
            'viewu1h:':self.view_u1[0]   = dm_to_number(tmp,/float)
            'viewu1k:':self.view_u1[1]   = dm_to_number(tmp,/float)
            'viewu1l:':self.view_u1[2]   = dm_to_number(tmp,/float)
            'view1lb:':begin
                       tmpvalue          = strsplit(tmp,'&',/extract,/preserve_null)
                       if n_elements(tmpvalue) eq 3 then begin ;old format parameter file
                          tmpvalue       = [tmpvalue[1],tmpvalue[0]]
                          if self.samp_typ lt 2 then self.samp_typ = 1-self.samp_typ
                       endif
                       self.view_u1lab   = tmpvalue
                       if (self.view_u1typ ge 4) and (self.intn_typ eq (n_elements(self.intn_typ_str)-2)) then begin
                          if self.instrname eq 'dcs' then begin
                             if self.view_u1typ eq 4 then id = 5 else id = 4
                          endif else id = self.view_u1typ+1 
                       endif else id = self.view_u1typ
                       self.view_label[id] = self.view_u1lab[0]
                       end
            'viewu1f:':self.view_u1fold  = dm_to_number(tmp,/int)
            'viewu1d:':begin
                       tmpvalue          = strsplit(tmp,'&',/extract,/preserve_null)
                       *(self.view_u1folddir) = dm_to_number(tmpvalue,/int)
                       end
            'viewu1c:':*(self.view_u1foldcenter) = dm_to_number(strsplit(tmp,'&',/extract,/preserve_null),/float)   
            'viewu2h:':self.view_u2[0]   = dm_to_number(tmp,/float)
            'viewu2k:':self.view_u2[1]   = dm_to_number(tmp,/float)
            'viewu2l:':self.view_u2[2]   = dm_to_number(tmp,/float)
            'view2lb:':begin
                       tmpvalue          = strsplit(tmp,'&',/extract,/preserve_null)
                       if n_elements(tmpvalue) eq 3 then tmpvalue = [tmpvalue[1],tmpvalue[0]]
                       self.view_u2lab   = tmpvalue
                       if (self.view_u2typ ge 4) and (self.intn_typ eq (n_elements(self.intn_typ_str)-2)) then begin
                          if self.instrname eq 'dcs' then begin
                             if self.view_u2typ eq 4 then id = 5 else id = 4
                          endif else id = self.view_u2typ+1 
                       endif else id = self.view_u2typ
                       self.view_label[id] = self.view_u2lab[0]
                       end
            'viewu2f:':self.view_u2fold  = dm_to_number(tmp,/int)
            'viewu2d:':begin
                       tmpvalue          = strsplit(tmp,'&',/extract,/preserve_null)
                       *(self.view_u2folddir) = dm_to_number(tmpvalue,/int)
                       end
            'viewu2c:':*(self.view_u2foldcenter) = dm_to_number(strsplit(tmp,'&',/extract,/preserve_null),/float)     
            'viewu3h:':self.view_u3[0]   = dm_to_number(tmp,/float)
            'viewu3k:':self.view_u3[1]   = dm_to_number(tmp,/float)
            'viewu3l:':self.view_u3[2]   = dm_to_number(tmp,/float)
            'view3lb:':self.view_u3lab   = tmp
            'viewu3f:':self.view_u3fold  = dm_to_number(tmp,/int)
            'viewu3d:':begin
                       tmpvalue          = strsplit(tmp,'&',/extract,/preserve_null)
                       *(self.view_u3folddir) = dm_to_number(tmpvalue,/int)
                       end
            'viewu3c:':*(self.view_u3foldcenter) = dm_to_number(strsplit(tmp,'&',/extract,/preserve_null),/float)        
            'd4selx1:':self.d4sel_x1     = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'd4selx2:':self.d4sel_x2     = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'd4selx3:':self.d4sel_x3     = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'd4selx4:':self.d4sel_x4     = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'd4selt1:':self.d4sel_t1     = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'd4selt2:':self.d4sel_t2     = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'd4s_t1r:':self.d4sel_tft1   = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'd4s_t2r:':self.d4sel_tft2   = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'd4s_x1r:':self.d4_x1ran     = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'd4s_x2r:':self.d4_x2ran     = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'd4s_x3r:':self.d4_x3ran     = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'd4s_x4r:':self.d4_x4ran     = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'd4s_x1s:':self.d4_x1step    = dm_to_number(tmp,/float)
            'd4s_x2s:':self.d4_x2step    = dm_to_number(tmp,/float)
            'd4s_x3s:':self.d4_x3step    = dm_to_number(tmp,/float)
            'd4s_x4s:':self.d4_x3step    = dm_to_number(tmp,/float)     
            'volselx:':self.volsel_x[*]  = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'volsely:':self.volsel_y[*]  = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'volselz:':self.volsel_z[*]  = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'volxran:':self.vol_xran     = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'volx_st:':self.vol_xstep    = dm_to_number(tmp,/float)
            'volyran:':self.vol_yran     = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'voly_st:':self.vol_ystep    = dm_to_number(tmp,/float)
            'volzran:':self.vol_zran     = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'volz_st:':self.vol_zstep    = dm_to_number(tmp,/float)
            'voliran:':self.vol_iran     = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'vol_smo:':self.vol_smooth   = dm_to_number(tmp,/int)
            'volintp:':self.vol_interp   = dm_to_number(tmp,/int)
            'vol_col:':self.vol_color    = dm_to_number(tmp,/int)
            'vol__t1:':self.volsel_t1[*] = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'vol__t2:':self.volsel_t2[*] = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'vol__t3:':self.volsel_t3[*] = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'vol_t1r:':self.volsel_tft1  = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'vol_t2r:':self.volsel_tft2  = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'vol_t3r:':self.volsel_tft3  = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'dispslh:':self.slice_x[*]   = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'dispslv:':self.slice_y[*]   = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'dispx_r:':self.slice_xran   = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'dispxst:':self.slice_xstep  = dm_to_number(tmp,/float)
            'dispy_r:':self.slice_yran   = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'dispyst:':self.slice_ystep  = dm_to_number(tmp,/float)
            'dispi_r:':self.slice_iran   = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'disp_t1:':self.slice_t1[*]  = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'disp_t2:':self.slice_t2[*]  = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'disp_t3:':self.slice_t3[*]  = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'disp_t4:':self.slice_t4[*]  = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'dispt1r:':self.slice_t1ran  = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'dispt2r:':self.slice_t2ran  = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'dispt3r:':self.slice_t3ran  = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'dispt4r:':self.slice_t4ran  = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/float)
            'dispsmo:':self.slice_smooth = dm_to_number(tmp,/int)
            'dispker:':self.slice_sm_kernel[0:8] = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'dispsha:':self.slice_shading  = dm_to_number(tmp,/int)
            'dispcol:':self.slice_color    = dm_to_number(tmp,/int)
            'dispcut:':self.slice_cutcolor = dm_to_number(tmp,/int)
            'cut___x:':begin
                       tmpvalue          = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
                       if n_elements(tmpvalue) eq 3 then tmpvalue=[tmpvalue[1],tmpvalue[0],tmpvalue[2],0]
                       self.cut_x[*]     = tmpvalue
                       end
            'cutxran:':self.cut_ran      = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'cut_xst:':self.cut_step     = dm_to_number(tmp,/float)
            'cut__t1:':self.cut_t1[*]    = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'cut__t2:':self.cut_t2[*]    = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'cut__t3:':self.cut_t3[*]    = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'cut__t4:':self.cut_t4[*]    = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'cut__t5:':self.cut_t5[*]    = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'cut_t1r:':self.cut_t1ran    = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'cut_t2r:':self.cut_t2ran    = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'cut_t3r:':self.cut_t3ran    = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'cut_t4r:':self.cut_t4ran    = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'cut_t5r:':self.cut_t5ran    = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'cut___y:':begin
                       tmpvalue          = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
                       if n_elements(tmpvalue) eq 3 then tmpvalue=[tmpvalue[1],tmpvalue[0],tmpvalue[2],0]
                       self.cut_y[*]     = tmpvalue
                       end
            'cutyran:':self.cut_yran     = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'cut_sym:':self.cut_symbol   = dm_to_number(tmp,/int)
            'cut_col:':self.cut_color    = dm_to_number(tmp,/int)
            'userdef:':self.coldefine[0:2] = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'cutline:':self.cut_line     = dm_to_number(tmp,/int)
            'traj__x:':begin
                       tmpvalue          = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
                       tmpid             = where(tmpvalue gt 4,count)
                       if count ne 0 then tmpvalue[tmpid] = 0
                       if n_elements(tmpvalue) eq 3 then tmpvalue=[tmpvalue[1],tmpvalue[0],tmpvalue[2],0]
                       self.traj_x[*]    = tmpvalue
                       end
            'traj_xr:':self.traj_xran    = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'traj__y:':begin
                       tmpvalue          = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
                       tmpid             = where(tmpvalue gt 4,count)
                       if count ne 0 then tmpvalue[tmpid] = 1
                       if n_elements(tmpvalue) eq 3 then tmpvalue=[tmpvalue[1],tmpvalue[0],tmpvalue[2],1]
                       self.traj_y[*]    = tmpvalue
                       end
            'traj_yr:':self.traj_yran    = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'traj_z1:':self.traj_z1[*]   = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'traj_z2:':self.traj_z2[*]   = dm_to_number(strsplit(tmp,'&',/extract,/preserve),/int)
            'trajz1r:':self.traj_z1ran   = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'trajz2r:':self.traj_z2ran   = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'm___jqy:':self.mjqy         = dm_to_number(tmp,/float)
            'eief___:':self.eief         = dm_to_number(tmp,/float)
            'estaend:':self.estartend    = dm_to_number(strsplit(tmp,'&',/extract,/preserve))
            'phxfile:':self.phxfile      = tmp
            'userfun:':self.userf_str    = tmp
            'userfbf:':self.userf_before = dm_to_number(tmp,/int)
            'history:':begin
                       if clear_history then begin
                          clear_history  = 0b
                          self.n_history = 0
                          totnum         = n_elements(self.file_hist)  ;make sure the capacity is not exceeded
                          ptr_free,self.file_hist
                       endif
                       self.file_hist[self.n_history mod totnum] = ptr_new(strsplit(tmp,string('A4'xb),/extract,/preserve))
                       if strmid((*(self.file_hist[self.n_history mod totnum]))[0],0,6) ne 'ftype_' then $
                          *(self.file_hist[self.n_history mod totnum]) = ['ftype_'+dm_to_string(self.ftype,/int),*(self.file_hist[self.n_history mod totnum])]
                       self.n_history = self.n_history+1
                       end
            'loadfil:':if keyword_set(session) then begin
                       tmpvalue = strsplit(tmp,string('A4'xb),/extract,/preserve)
                       self->script,'load_data',filename=tmpvalue[3:*],directory=tmpvalue[2],/scriptmode
                       end
            'addfile:':if keyword_set(session) then begin
                       tmpvalue = strsplit(tmp,string('A4'xb),/extract,/preserve)
                       self->script,'add_data',filename=tmpvalue[3:*],directory=tmpvalue[2],/scriptmode
                       end
            'empfil0:':if keyword_set(session) then begin
                       tmpvalue = strsplit(tmp,string('A4'xb),/extract,/preserve)
                       self->script,'load_emptycan',filename=tmpvalue[3:*],directory=tmpvalue[2],/scriptmode
                       end
            'empfil1:':if keyword_set(session) then begin
                       tmpvalue = strsplit(tmp,string('A4'xb),/extract,/preserve)
                       self->script,'load_emptycan2',filename=tmpvalue[3:*],directory=tmpvalue[2],/scriptmode
                       end
            'empfil2:':if keyword_set(session) then begin
                       tmpvalue = strsplit(tmp,string('A4'xb),/extract,/preserve)
                       self->script,'load_emptycan3',filename=tmpvalue[3:*],directory=tmpvalue[2],/scriptmode
                       end
            else:
          endcase
       endwhile
       if (n_elements(symrot_type) ne 0) or (n_elements(mrrf_type) ne 0) then begin
          if (self.fold_type eq 0) and (n_elements(mrrf_type) ne 0) then self.fold_type = mrrf_type $
          else if (self.fold_type eq 1) and (n_elements(symrot_type) ne 0) then  self.fold_type = symrot_type+2
       endif
       ptr_free,self.projStrPtr
       if (~ keyword_set(debug)) then begin
          bin_method = 'Binning method is '+(['Weighted Mean','Arithmetic Mean','Sum'])[self.bin_avgsum+1]+'.'
          if (strlowcase((*self.ftypename)[self.ftype]) eq 'macs') and (self.macshistomode eq 2) and (self.bin_avgsum ne ([-1,1])[self.macs_rawintn]) then begin
             ok = dialog_message(bin_method+' Do you want to switch to '+(['Weighted Mean for the event mode?','Sum for the raw intensity event mode?'])[self.macs_rawintn],/question,dialog_parent=self.tlb,/center)
             if strlowcase(ok) eq 'yes' then self.bin_avgsum = ([-1,1])[self.macs_rawintn]
          endif
          if self.bin_extlib and (self.bin_avgsum eq -1) then self.bin_avgsum = 0 ;weighted mean not implemented in external library
          if lmgr(/vm) then self.userf_str = '' ;not possible to run user function in virtual machine mode
          self.userfwarning = 1b
          (*self.colors)[0] = (['white','black'])[self.pbgcol_bw]
          self->set_filehistory
          self->reset_file
          if newftype then begin
             tmp = self.ftype & self.ftype = -1 & intn_typ = self.intn_typ
             ok  = self->dm_filetools('ftype_'+dm_to_string(tmp),init=init,pdt_reset=pdt_reset,/keep_extravaxis)
             if self.intn_typ ne intn_typ then self->setproperty,intntype=intn_typ
          endif else begin
             self->reset_menu,pdt_reset=pdt_reset
             if (self.instrname eq 'dcs') or (self.instrname eq 'wand') then begin
                ptr_free,self.detPosPtr
                case self.instrname of
                  'dcs' : self.detPosPtr = dm_load_detpos(/dcs, adjust2th=self.tthcorr) 
                  'wand': self.detPosPtr = dm_load_detpos(/wand,adjust2th=self.tthcorr)
                  else:
                endcase
             endif
             if self.instrname eq 'macs' then self->reset_intnlab
          endelse
          if n_elements(dataDir) ne 0 then begin
             self->setproperty,dataDir=dataDir
             if obj_valid(self.fileSel) then self.filesel->set_path,dataDir
          endif
          if n_elements(edep_det) ne 0 then self.edep_det = edep_det
          if (~keyword_set(init)) then begin
             if (~keyword_set(pdt_reset)) then self->reset_pdt
             if (self.n_history ne 0) and (~keyword_set(session)) then self->set_filehistory,choose=0  ;go to the latest files
          endif
          ;some warning messages   
          if self.instrgeom ne ([0,1])[self.instrname eq 'macs'] then mesg = [mesg,'The instrument geometry is '+(['direct','inverse'])[self.instrgeom]+'.']
          if keyword_set(ssfacptr_changed) then mesg = [mesg,'Detector dependent self-shielding factor is read from '+(*self.ssfacPtr).file+'.'] $
          else if self.ssfac ne 1 then mesg = [mesg,'Self-shielding factor is '+dm_to_string(self.ssfac)+'.']
          if ptr_valid(self.absp_info) then begin
             mesg = [mesg,(['Cylinder','Annulus','Sphere','Flat plate'])[(*self.absp_info)[0]]+' absorption correction is set.']
             if self.absp_l1factor ne 1 then mesg = [mesg,'Absorption correction incident beam path factor is '+dm_to_string(self.absp_l1factor)+'.']
          endif
          if ((self.samp_typ ge 2) or (self.instrname eq 'macs')) then begin
             if self.empt_tol_optn eq 1 then mesg = [mesg,'The empty can subtraction tolerance method is "Within Tolerance of the Closest".']
             if self.instrname eq 'macs' then begin
                if keyword_set(self.macshistomode) then begin
                   mesg = [mesg,'Data are in '+([dm_to_string(self.macshistonchan)+'-channel histogram  mode.',$
                           'event mode. Binary .rg0 2D histogram data must be present in the data directory.'])[self.macshistomode-1]]
                   if (self.macshistomode eq 2) and (self.macs_rawintn eq 1) then $
                      mesg = [mesg,'The intensity is not normalized. The rebinned intensity and weight are for output only.']  
                endif
                tolwarning = 0b
                if (abs(self.macs_tolerance[0,0]) gt 0.5) or (abs(self.macs_tolerance[1,0]) gt 1) then tolwarning = 1b
                if (self.samp_typ eq 1) and (abs(self.macs_tolerance[2,0]) gt 1) then tolwarning = 1b
                if keyword_set(tolwarning) then begin
                   tolwarning = dm_to_string(self.macs_tolerance[0:(1+self.samp_typ),0])+[' meV',string('b0'xb)+'(kidney)',string('b0'xb)+'(A3)']
                   for i=1,n_elements(tolwarning)-1 do tolwarning[0] = tolwarning[0]+([', ',', and '])[i ne 1]+tolwarning[i]
                   mesg = [mesg,'The empty can subtraction tolerance values are '+tolwarning[0]+'.']
                endif
             endif else begin
                if abs(self.psi_tolerance[0]) gt 2 then mesg = [mesg,'The empty can subtraction psi tolerance value is '+dm_to_string(self.psi_tolerance[0])+string('b0'xb)+'.']
            endelse
          endif
          if (self.intn_typ eq (n_elements(self.intn_typ_str)-2)) and finite(self.mjqy,/nan) then self->setproperty,intntype=([n_elements(self.intn_typ_str)-1,0])[self.instrname eq 'dcs']
          if self.instrname eq 'dcs' then begin
             if (self.mon_typ[0] ne 1)  then mesg = [mesg,(['FC0','BM1','Duration'])[self.mon_typ[0]]+' is used as the monitor.']
             if (self.mon_sumfit[0] eq 0) then mesg = [mesg,'Monitor counting method is sum.']
             if total(self.dcsdetmask) ne 0 then begin
                if self.dcsdetmask[0] then tmpstr = 'Negative angle'
                if self.dcsdetmask[4] then begin
                   if n_elements(tmpstr) ne 0 then tmpstr = tmpstr+' and upper and lower bank' else tmpstr = 'Upper and lower bank' 
                endif else begin
                   if self.dcsdetmask[1] then begin
                      if n_elements(tmpstr) ne 0 then tmpstr = tmpstr+' and lower bank' else tmpstr = 'Lower bank' 
                   endif
                   if self.dcsdetmask[3] then begin
                      if n_elements(tmpstr) ne 0 then tmpstr = tmpstr+' and upper bank' else tmpstr = 'Upper bank' 
                   endif
                endelse
                if self.dcsdetmask[2] then begin
                   if n_elements(tmpstr) ne 0 then tmpstr = tmpstr+' and central bank' else tmpstr = 'Central bank' 
                endif
                mesg = [mesg,tmpstr+' detectors are masked.']
             endif else begin
                if (total(abs(fix(dcsdetmask)-fix(self.dcsdetmask))) ne 0) then $
                   mesg = [mesg,'The detector masks have been changed.']
             endelse
             if self.intn_typ ne 0 then mesg = [mesg,'The intensity type is "'+self.intn_typ_str[self.intn_typ]+'".']
             if self.extravaxis_yn[1] then begin
                if self.macstemp_typ gt 0 then mesg = [mesg,'The temperature choice is "'+(['Control','Set Point'])[self.macstemp_typ-1]+'".']
             endif
          endif else if self.instrname eq 'macs' then begin
             if self.mon_typ[1] ne 1 then mesg = [mesg,(['Time','','Pulse number'])[self.mon_typ[1]]+' is used as the monitor.']     
             if dm_to_number(systime(),/date) ge self.macs_CFXdate[0] then begin
                if ~self.macs_lamb2 and ~keyword_set(init) then mesg = [mesg,'The monitor lambda/2 correction is disabled.']
             endif else begin
                if self.macs_lamb2 then mesg = [mesg,'The monitor lambda/2 correction is enabled.']
             endelse
             if self.locklattuv then mesg = [mesg,'The lattice parameters and crystal orientation axes have been locked.']
             if ~self.checklattuv then mesg = [mesg,'Loading files will not check for lattice parameters and crystal orientation consistency.']
             if self.extravaxis_yn[1] then begin
                if self.macstemp_typ gt 0 then mesg = [mesg,'The temperature choice is "'+(['Control','Set Point'])[self.macstemp_typ-1]+'".']
                if self.tempavg_yn        then mesg = [mesg,'The temperature of each file will be averaged.']
             endif
             if n_elements(orie_psi) ne 0 then self->script,'set_parameter',name='oriePsi',value=orie_psi,resolution=3 ;self.orie_psi has been reset to 0 in file_menu
             if self.intn_typ ne n_elements(self.intn_typ_str)-1 then mesg = [mesg,'The intensity type is "'+self.intn_typ_str[self.intn_typ]+'". Some factors will be applied to the intensity.']
             if ptr_valid(self.a3mask)   then mesg = [mesg,'A3 angle is masked.']
             if ptr_valid(self.bga3mask) then mesg = [mesg,'Background A3 angle is masked.']
          endif else if self.instrname eq 'wand' then begin
             if self.extravaxis_yn[0] and self.tempavg_yn then mesg = [mesg,'The temperature of each file will be averaged.']  
          endif else if self.instrname eq 'inx' then begin
             if self.inxerev then mesg = [mesg,'The data file energy transfer definition is sample energy gain/loss, instead of the traditional neutron energy gain/loss.']  
          endif else begin
             if self.intn_typ ne n_elements(self.intn_typ_str)-1 then mesg = [mesg,'The intensity type is "'+self.intn_typ_str[self.intn_typ]+'". Some factors will be applied to the intensity.']
          endelse 
          if (self.samp_typ ne 0) and (self.fold_type eq 1) then mesg = [mesg,'Folding is to both sides.']
          if (self.samp_typ ne 0) and (self.fold_type eq 4) then begin
              mesg = [mesg,'Symmetry rotation folding is 360'+string('b0'xb)+' expansion.']
              if self.numrot_optn eq 1 then mesg = [mesg,'The number of rotations is nfold-1.']
          endif
          if self.bin_avgsum eq 1 then mesg = [mesg,bin_method]
          if self.tthcorr and ((self.instrname eq 'dcs') or (self.instrname eq 'wand')) then $
             mesg = [mesg,(*self.ftypename)[self.ftype]+' 2theta Correction is On.']
          if self.eadjust ne 0 then mesg = [mesg,'Elastic Peak Adjustment is On.']
          if self.eint_yn then mesg = [mesg,'E-integrated intensity is enabled.']
          if strlen(self.userf_str) gt 0 then mesg = [mesg,'User macro is specified.']
       endif
    endelse
    free_lun,unit
    self.pnotooltip = 1b 
    if n_elements(mesg) gt 1 then ok = dialog_message(mesg[1:*],dialog_parent=self.tlb,title='Please note:')
end